--
-- This file is part of TALER
-- Copyright (C) 2014--2022 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--


CREATE FUNCTION create_table_contracts(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'contracts';
BEGIN
  PERFORM create_partitioned_table(
     'CREATE TABLE %I '
     '(contract_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
     ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
     ',pub_ckey BYTEA NOT NULL CHECK (LENGTH(pub_ckey)=32)'
     ',contract_sig BYTEA NOT NULL CHECK (LENGTH(contract_sig)=64)'
     ',e_contract BYTEA NOT NULL'
     ',purse_expiration INT8 NOT NULL'
     ',PRIMARY KEY (purse_pub)'
     ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'encrypted contracts associated with purses'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the purse that the contract is associated with'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'signature over the encrypted contract by the purse contract key'
    ,'contract_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public ECDH key used to encrypt the contract, to be used with the purse private key for decryption'
    ,'pub_ckey'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'AES-GCM encrypted contract terms (contains gzip compressed JSON after decryption)'
    ,'e_contract'
    ,table_name
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_contracts(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'contracts';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_contract_serial_id_key'
    ' UNIQUE (contract_serial_id) '
  );
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('contracts'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('contracts'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
