BEGIN;
SELECT _v.register_patch('exchange-0003', NULL, NULL);
SET search_path TO exchange;
CREATE OR REPLACE FUNCTION create_table_purse_actions(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_actions';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I'
      '(purse_pub BYTEA NOT NULL PRIMARY KEY CHECK(LENGTH(purse_pub)=32)'
      ',action_date INT8 NOT NULL'
      ',partner_serial_id INT8'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'purses awaiting some action by the router'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public (contract) key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'when is the purse ready for action'
    ,'action_date'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'wad target of an outgoing wire transfer, 0 for local, NULL if the purse is unmerged and thus the target is still unknown'
    ,'partner_serial_id'
    ,table_name
    ,partition_suffix
  );
END $$;
CREATE OR REPLACE FUNCTION purse_requests_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO
    purse_actions
    (purse_pub
    ,action_date)
  VALUES
    (NEW.purse_pub
    ,NEW.purse_expiration);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION purse_requests_insert_trigger()
  IS 'When a purse is created, insert it into the purse_action table to take action when the purse expires.';
CREATE OR REPLACE FUNCTION master_table_purse_actions()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_actions';
BEGIN
  CREATE INDEX IF NOT EXISTS purse_action_by_target
    ON purse_actions
    (partner_serial_id,action_date);
  CREATE TRIGGER purse_requests_on_insert
    AFTER INSERT
    ON purse_requests
    FOR EACH ROW EXECUTE FUNCTION purse_requests_insert_trigger();
  COMMENT ON TRIGGER purse_requests_on_insert
          ON purse_requests
    IS 'Here we install an entry for the purse expiration.';
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('purse_actions'
    ,'exchange-0003'
    ,'create'
    ,TRUE
    ,FALSE),
    ('purse_actions'
    ,'exchange-0003'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE OR REPLACE FUNCTION create_table_purse_deletion(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_deletion';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I'
      '(purse_deletion_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',purse_sig BYTEA CHECK (LENGTH(purse_sig)=64)'
      ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'signatures affirming explicit purse deletions'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'signature of type WALLET_PURSE_DELETE'
    ,'purse_sig'
    ,table_name
    ,partition_suffix
  );
END $$;
COMMENT ON FUNCTION create_table_purse_deletion
  IS 'Creates the purse_deletion table';
CREATE OR REPLACE FUNCTION constrain_table_purse_deletion(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_deletion';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
      ' ADD CONSTRAINT ' || table_name || '_delete_serial_key '
        'UNIQUE (purse_deletion_serial_id)'
  );
END $$;
CREATE OR REPLACE FUNCTION master_table_purse_requests_was_deleted (
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_requests';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE exchange.' || table_name ||
    ' ADD COLUMN'
    ' was_deleted BOOLEAN NOT NULL DEFAULT(FALSE)'
  );
  COMMENT ON COLUMN purse_requests.was_deleted
    IS 'TRUE if the purse was explicitly deleted (purse must have an entry in the purse_deletion table)';
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('purse_deletion'
    ,'exchange-0003'
    ,'create'
    ,TRUE
    ,FALSE),
    ('purse_deletion'
    ,'exchange-0003'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('purse_requests_was_deleted'
    ,'exchange-0003'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE OR REPLACE FUNCTION create_table_kyc_attributes(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'kyc_attributes';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I'
      '(kyc_attributes_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_payto BYTEA PRIMARY KEY CHECK (LENGTH(h_payto)=32)'
      ',kyc_prox BYTEA NOT NULL CHECK (LENGTH(kyc_prox)=32)'
      ',provider VARCHAR NOT NULL'
      ',birthdate VARCHAR'
      ',collection_time INT8 NOT NULL'
      ',expiration_time INT8 NOT NULL'
      ',encrypted_attributes VARCHAR NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (h_payto)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'KYC data about particular payment addresses'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash of payto://-URI the attributes are about'
    ,'h_payto'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'short hash of normalized full name and birthdate; used to efficiently find likely duplicate users'
    ,'kyc_prox'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'birth date of the user, in format YYYY-MM-DD where a value of 0 is used to indicate unknown (in official documents); NULL if the birth date was not collected by the provider; used for KYC-driven age restrictions'
    ,'birthdate'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'time when the attributes were collected by the provider'
    ,'collection_time'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'time when the attributes should no longer be considered validated'
    ,'expiration_time'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'configuration section name of the provider that affirmed the attributes'
    ,'provider'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     '(encrypted) JSON object (as string) with the attributes'
    ,'encrypted_attributes'
    ,table_name
    ,partition_suffix
  );
END $$;
COMMENT ON FUNCTION create_table_kyc_attributes
  IS 'Creates the kyc_attributes table';
CREATE OR REPLACE FUNCTION constrain_table_kyc_attributes(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'kyc_attributes';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
      ' ADD CONSTRAINT ' || table_name || '_serial_key '
        'UNIQUE (kyc_attributes_serial_id)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_similarity_index '
    'ON ' || table_name || ' '
    '(kyc_prox);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_expiration_time '
    'ON ' || table_name || ' '
    '(expiration_time ASC);'
  );
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('kyc_attributes'
    ,'exchange-0003'
    ,'create'
    ,TRUE
    ,FALSE),
    ('kyc_attributes'
    ,'exchange-0003'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE OR REPLACE FUNCTION create_table_aml_status(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'aml_status';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I'
      '(aml_status_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_payto BYTEA PRIMARY KEY CHECK (LENGTH(h_payto)=32)'
      ',threshold_val INT8 NOT NULL DEFAULT(0)'
      ',threshold_frac INT4 NOT NULL DEFAULT(0)'
      ',status INT4 NOT NULL DEFAULT(0)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (h_payto)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'AML status for a particular payment destination'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash of the payto://-URI this AML status is about'
    ,'h_payto'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'monthly inbound transaction limit below which we are OK (if status is 1)'
    ,'threshold_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     '0 for all OK, 1 for AML decision required, 2 for account is frozen (prevents further transactions)'
    ,'status'
    ,table_name
    ,partition_suffix
  );
END $$;
COMMENT ON FUNCTION create_table_aml_status
  IS 'Creates the aml_status table';
CREATE OR REPLACE FUNCTION constrain_table_aml_status(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'aml_status';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
      ' ADD CONSTRAINT ' || table_name || '_serial_key '
        'UNIQUE (aml_status_serial_id)'
  );
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('aml_status'
    ,'exchange-0003'
    ,'create'
    ,TRUE
    ,FALSE),
    ('aml_status'
    ,'exchange-0003'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE TABLE aml_staff
  (aml_staff_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,decider_pub BYTEA PRIMARY KEY CHECK (LENGTH(decider_pub)=32)
  ,master_sig BYTEA CHECK (LENGTH(master_sig)=64)
  ,decider_name VARCHAR NOT NULL
  ,is_active BOOLEAN NOT NULL
  ,read_only BOOLEAN NOT NULL
  ,last_change INT8 NOT NULL
  );
COMMENT ON TABLE aml_staff
  IS 'Table with AML staff members the exchange uses or has used in the past. Entries never expire as we need to remember the last_change column indefinitely.';
COMMENT ON COLUMN aml_staff.decider_pub
  IS 'Public key of the AML staff member.';
COMMENT ON COLUMN aml_staff.master_sig
  IS 'The master public key signature on the AML staff member status, of type TALER_SIGNATURE_MASTER_AML_KEY.';
COMMENT ON COLUMN aml_staff.decider_name
  IS 'Name of the staff member.';
COMMENT ON COLUMN aml_staff.is_active
  IS 'true if we are currently supporting the use of this AML staff member.';
COMMENT ON COLUMN aml_staff.is_active
  IS 'true if the member has read-only access.';
COMMENT ON COLUMN aml_staff.last_change
  IS 'Latest time when active status changed. Used to detect replays of old messages.';
CREATE OR REPLACE FUNCTION create_table_aml_history(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'aml_history';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I'
      '(aml_history_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_payto BYTEA PRIMARY KEY CHECK (LENGTH(h_payto)=32)'
      ',new_threshold_val INT8 NOT NULL DEFAULT(0)'
      ',new_threshold_frac INT4 NOT NULL DEFAULT(0)'
      ',new_status INT4 NOT NULL DEFAULT(0)'
      ',decision_time INT8 NOT NULL DEFAULT(0)'
      ',justification VARCHAR NOT NULL'
      ',decider_pub BYTEA CHECK (LENGTH(decider_pub)=32)'
      ',decider_sig BYTEA CHECK (LENGTH(decider_sig)=64)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (h_payto)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'AML decision history for a particular payment destination'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash of the payto://-URI this AML history is about'
    ,'h_payto'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'new monthly inbound transaction limit below which we are OK'
    ,'new_threshold_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     '0 for all OK, 1 for AML decision required, 2 for account is frozen (prevents further transactions)'
    ,'new_status'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'when was the status changed'
    ,'decision_time'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'human-readable justification for the status change'
    ,'justification'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the staff member who made the AML decision'
    ,'decider_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature key of the staff member affirming the AML decision; of type AML_DECISION'
    ,'decider_sig'
    ,table_name
    ,partition_suffix
  );
END $$;
COMMENT ON FUNCTION create_table_aml_history
  IS 'Creates the aml_history table';
CREATE OR REPLACE FUNCTION constrain_table_aml_history(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'aml_history';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
      ' ADD CONSTRAINT ' || table_name || '_serial_key '
        'UNIQUE (aml_history_serial_id)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_main_index '
    'ON ' || table_name || ' '
    '(h_payto ASC, decision_time ASC);'
  );
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('aml_history'
    ,'exchange-0003'
    ,'create'
    ,TRUE
    ,FALSE),
    ('aml_history'
    ,'exchange-0003'
    ,'constrain'
    ,TRUE
    ,FALSE);
COMMIT;
