BEGIN;
SELECT _v.register_patch('exchange-0002', NULL, NULL);
SET search_path TO exchange;
CREATE TABLE denominations
  (denominations_serial BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,denom_pub_hash BYTEA PRIMARY KEY CHECK (LENGTH(denom_pub_hash)=64)
  ,denom_type INT4 NOT NULL DEFAULT (1) 
  ,age_mask INT4 NOT NULL DEFAULT (0)
  ,denom_pub BYTEA NOT NULL
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  ,valid_from INT8 NOT NULL
  ,expire_withdraw INT8 NOT NULL
  ,expire_deposit INT8 NOT NULL
  ,expire_legal INT8 NOT NULL
  ,coin_val INT8 NOT NULL
  ,coin_frac INT4 NOT NULL
  ,fee_withdraw_val INT8 NOT NULL
  ,fee_withdraw_frac INT4 NOT NULL
  ,fee_deposit_val INT8 NOT NULL
  ,fee_deposit_frac INT4 NOT NULL
  ,fee_refresh_val INT8 NOT NULL
  ,fee_refresh_frac INT4 NOT NULL
  ,fee_refund_val INT8 NOT NULL
  ,fee_refund_frac INT4 NOT NULL
  );
COMMENT ON TABLE denominations
  IS 'Main denominations table. All the valid denominations the exchange knows about.';
COMMENT ON COLUMN denominations.denom_type
  IS 'determines cipher type for blind signatures used with this denomination; 0 is for RSA';
COMMENT ON COLUMN denominations.age_mask
  IS 'bitmask with the age restrictions that are being used for this denomination; 0 if denomination does not support the use of age restrictions';
COMMENT ON COLUMN denominations.denominations_serial
  IS 'needed for exchange-auditor replication logic';
CREATE INDEX denominations_by_expire_legal_index
  ON denominations
  (expire_legal);
CREATE TABLE IF NOT EXISTS denomination_revocations
  (denom_revocations_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,denominations_serial INT8 PRIMARY KEY REFERENCES denominations (denominations_serial) ON DELETE CASCADE
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  );
COMMENT ON TABLE denomination_revocations
  IS 'remembering which denomination keys have been revoked';
CREATE FUNCTION create_table_wire_targets(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(wire_target_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',wire_target_h_payto BYTEA PRIMARY KEY CHECK (LENGTH(wire_target_h_payto)=32)'
      ',payto_uri VARCHAR NOT NULL'
    ') %s ;'
    ,'wire_targets'
    ,'PARTITION BY HASH (wire_target_h_payto)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'All senders and recipients of money via the exchange'
    ,'wire_targets'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Can be a regular bank account, or also be a URI identifying a reserve-account (for P2P payments)'
    ,'payto_uri'
    ,'wire_targets'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Unsalted hash of payto_uri'
    ,'wire_target_h_payto'
    ,'wire_targets'
    ,partition_suffix
  );
END $$;
CREATE FUNCTION constrain_table_wire_targets(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'wire_targets';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wire_target_serial_id_key'
    ' UNIQUE (wire_target_serial_id)'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('wire_targets'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('wire_targets'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE TABLE kyc_alerts
  (h_payto BYTEA PRIMARY KEY CHECK (LENGTH(h_payto)=32)
  ,trigger_type INT4 NOT NULL
  ,UNIQUE(trigger_type,h_payto)
  );
COMMENT ON TABLE kyc_alerts
  IS 'alerts about completed KYC events reliably notifying other components (even if they are not running)';
COMMENT ON COLUMN kyc_alerts.h_payto
  IS 'hash of the payto://-URI for which the KYC status changed';
COMMENT ON COLUMN kyc_alerts.trigger_type
  IS 'identifies the receiver of the alert, as the same h_payto may require multiple components to be notified';
CREATE TABLE wire_fee
  (wire_fee_serial BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,wire_method VARCHAR NOT NULL
  ,start_date INT8 NOT NULL
  ,end_date INT8 NOT NULL
  ,wire_fee_val INT8 NOT NULL
  ,wire_fee_frac INT4 NOT NULL
  ,closing_fee_val INT8 NOT NULL
  ,closing_fee_frac INT4 NOT NULL
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  ,PRIMARY KEY (wire_method, start_date)
  );
COMMENT ON TABLE wire_fee
  IS 'list of the wire fees of this exchange, by date';
COMMENT ON COLUMN wire_fee.wire_fee_serial
  IS 'needed for exchange-auditor replication logic';
CREATE INDEX wire_fee_by_end_date_index
  ON wire_fee
  (end_date);
CREATE TABLE global_fee
  (global_fee_serial BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,start_date INT8 NOT NULL
  ,end_date INT8 NOT NULL
  ,history_fee_val INT8 NOT NULL
  ,history_fee_frac INT4 NOT NULL
  ,account_fee_val INT8 NOT NULL
  ,account_fee_frac INT4 NOT NULL
  ,purse_fee_val INT8 NOT NULL
  ,purse_fee_frac INT4 NOT NULL
  ,purse_timeout INT8 NOT NULL
  ,history_expiration INT8 NOT NULL
  ,purse_account_limit INT4 NOT NULL
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  ,PRIMARY KEY (start_date)
  );
COMMENT ON TABLE global_fee
  IS 'list of the global fees of this exchange, by date';
COMMENT ON COLUMN global_fee.global_fee_serial
  IS 'needed for exchange-auditor replication logic';
CREATE INDEX global_fee_by_end_date_index
  ON global_fee
  (end_date);
CREATE TABLE wire_accounts
  (payto_uri VARCHAR PRIMARY KEY
  ,master_sig BYTEA CHECK (LENGTH(master_sig)=64)
  ,is_active BOOLEAN NOT NULL
  ,last_change INT8 NOT NULL
  );
COMMENT ON TABLE wire_accounts
  IS 'Table with current and historic bank accounts of the exchange. Entries never expire as we need to remember the last_change column indefinitely.';
COMMENT ON COLUMN wire_accounts.payto_uri
  IS 'payto URI (RFC 8905) with the bank account of the exchange.';
COMMENT ON COLUMN wire_accounts.master_sig
  IS 'Signature of purpose TALER_SIGNATURE_MASTER_WIRE_DETAILS';
COMMENT ON COLUMN wire_accounts.is_active
  IS 'true if we are currently supporting the use of this account.';
COMMENT ON COLUMN wire_accounts.last_change
  IS 'Latest time when active status changed. Used to detect replays of old messages.';
CREATE TABLE auditors
  (auditor_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,auditor_pub BYTEA PRIMARY KEY CHECK (LENGTH(auditor_pub)=32)
  ,auditor_name VARCHAR NOT NULL
  ,auditor_url VARCHAR NOT NULL
  ,is_active BOOLEAN NOT NULL
  ,last_change INT8 NOT NULL
  );
COMMENT ON TABLE auditors
  IS 'Table with auditors the exchange uses or has used in the past. Entries never expire as we need to remember the last_change column indefinitely.';
COMMENT ON COLUMN auditors.auditor_pub
  IS 'Public key of the auditor.';
COMMENT ON COLUMN auditors.auditor_url
  IS 'The base URL of the auditor.';
COMMENT ON COLUMN auditors.is_active
  IS 'true if we are currently supporting the use of this auditor.';
COMMENT ON COLUMN auditors.last_change
  IS 'Latest time when active status changed. Used to detect replays of old messages.';
CREATE TABLE auditor_denom_sigs
  (auditor_denom_serial BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,auditor_uuid INT8 NOT NULL REFERENCES auditors (auditor_uuid) ON DELETE CASCADE
  ,denominations_serial INT8 NOT NULL REFERENCES denominations (denominations_serial) ON DELETE CASCADE
  ,auditor_sig BYTEA CHECK (LENGTH(auditor_sig)=64)
  ,PRIMARY KEY (denominations_serial, auditor_uuid)
  );
COMMENT ON TABLE auditor_denom_sigs
  IS 'Table with auditor signatures on exchange denomination keys.';
COMMENT ON COLUMN auditor_denom_sigs.auditor_uuid
  IS 'Identifies the auditor.';
COMMENT ON COLUMN auditor_denom_sigs.denominations_serial
  IS 'Denomination the signature is for.';
COMMENT ON COLUMN auditor_denom_sigs.auditor_sig
  IS 'Signature of the auditor, of purpose TALER_SIGNATURE_AUDITOR_EXCHANGE_KEYS.';
CREATE TABLE exchange_sign_keys
  (esk_serial BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,exchange_pub BYTEA PRIMARY KEY CHECK (LENGTH(exchange_pub)=32)
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  ,valid_from INT8 NOT NULL
  ,expire_sign INT8 NOT NULL
  ,expire_legal INT8 NOT NULL
  );
COMMENT ON TABLE exchange_sign_keys
  IS 'Table with master public key signatures on exchange online signing keys.';
COMMENT ON COLUMN exchange_sign_keys.exchange_pub
  IS 'Public online signing key of the exchange.';
COMMENT ON COLUMN exchange_sign_keys.master_sig
  IS 'Signature affirming the validity of the signing key of purpose TALER_SIGNATURE_MASTER_SIGNING_KEY_VALIDITY.';
COMMENT ON COLUMN exchange_sign_keys.valid_from
  IS 'Time when this online signing key will first be used to sign messages.';
COMMENT ON COLUMN exchange_sign_keys.expire_sign
  IS 'Time when this online signing key will no longer be used to sign.';
COMMENT ON COLUMN exchange_sign_keys.expire_legal
  IS 'Time when this online signing key legally expires.';
CREATE TABLE signkey_revocations
  (signkey_revocations_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,esk_serial INT8 PRIMARY KEY REFERENCES exchange_sign_keys (esk_serial) ON DELETE CASCADE
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  );
COMMENT ON TABLE signkey_revocations
  IS 'Table storing which online signing keys have been revoked';
CREATE TABLE extensions
  (extension_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,name VARCHAR NOT NULL UNIQUE
  ,manifest BYTEA
  );
COMMENT ON TABLE extensions
  IS 'Configurations of the activated extensions';
COMMENT ON COLUMN extensions.name
  IS 'Name of the extension';
COMMENT ON COLUMN extensions.manifest
  IS 'Manifest of the extension as JSON-blob, maybe NULL.  It contains common meta-information and extension-specific configuration.';
CREATE TABLE profit_drains
  (profit_drain_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,wtid BYTEA PRIMARY KEY CHECK (LENGTH(wtid)=32)
  ,account_section VARCHAR NOT NULL
  ,payto_uri VARCHAR NOT NULL
  ,trigger_date INT8 NOT NULL
  ,amount_val INT8 NOT NULL
  ,amount_frac INT4 NOT NULL
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  ,executed BOOLEAN NOT NULL DEFAULT FALSE
  );
COMMENT ON TABLE profit_drains
  IS 'transactions to be performed to move profits from the escrow account of the exchange to a regular account';
COMMENT ON COLUMN profit_drains.wtid
  IS 'randomly chosen nonce, unique to prevent double-submission';
COMMENT ON COLUMN profit_drains.account_section
  IS 'specifies the configuration section in the taler-exchange-drain configuration with the wire account to drain';
COMMENT ON COLUMN profit_drains.payto_uri
  IS 'specifies the account to be credited';
COMMENT ON COLUMN profit_drains.trigger_date
  IS 'set by taler-exchange-offline at the time of making the signature; not necessarily the exact date of execution of the wire transfer, just for orientation';
COMMENT ON COLUMN profit_drains.amount_val
  IS 'amount to be transferred';
COMMENT ON COLUMN profit_drains.master_sig
  IS 'EdDSA signature of type TALER_SIGNATURE_MASTER_DRAIN_PROFIT';
COMMENT ON COLUMN profit_drains.executed
  IS 'set to TRUE by taler-exchange-drain on execution of the transaction, not replicated to auditor';
CREATE FUNCTION create_table_legitimization_processes(
  IN shard_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(legitimization_process_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_payto BYTEA NOT NULL CHECK (LENGTH(h_payto)=32)'
      ',expiration_time INT8 NOT NULL DEFAULT (0)'
      ',provider_section VARCHAR NOT NULL'
      ',provider_user_id VARCHAR DEFAULT NULL'
      ',provider_legitimization_id VARCHAR DEFAULT NULL'
      ',UNIQUE (h_payto, provider_section)'
    ') %s ;'
    ,'legitimization_processes'
    ,'PARTITION BY HASH (h_payto)'
    ,shard_suffix
  );
  PERFORM comment_partitioned_table(
    'List of legitimization processes (ongoing and completed) by account and provider'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'unique ID for this legitimization process at the exchange'
    ,'legitimization_process_serial_id'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'foreign key linking the entry to the wire_targets table, NOT a primary key (multiple legitimizations are possible per wire target)'
    ,'h_payto'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'in the future if the respective KYC check was passed successfully'
    ,'expiration_time'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Configuration file section with details about this provider'
    ,'provider_section'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifier for the user at the provider that was used for the legitimization. NULL if provider is unaware.'
    ,'provider_user_id'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifier for the specific legitimization process at the provider. NULL if legitimization was not started.'
    ,'provider_legitimization_id'
    ,'legitimization_processes'
    ,shard_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_legitimization_processes(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  partition_name VARCHAR;
BEGIN
  partition_name = concat_ws('_', 'legitimization_processes', partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || partition_name
    || ' '
      'ADD CONSTRAINT ' || partition_name || '_serial_key '
        'UNIQUE (legitimization_process_serial_id)');
  EXECUTE FORMAT (
    'CREATE INDEX IF NOT EXISTS ' || partition_name || '_by_provider_and_legi_index '
        'ON '|| partition_name || ' '
        '(provider_section,provider_legitimization_id)'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || partition_name || '_by_provider_and_legi_index '
    'IS ' || quote_literal('used (rarely) in kyc_provider_account_lookup') || ';'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('legitimization_processes'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('legitimization_processes'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_legitimization_requirements(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(legitimization_requirement_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_payto BYTEA NOT NULL CHECK (LENGTH(h_payto)=32)'
      ',required_checks VARCHAR NOT NULL'
      ',UNIQUE (h_payto, required_checks)'
    ') %s ;'
    ,'legitimization_requirements'
    ,'PARTITION BY HASH (h_payto)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'List of required legitimizations by account'
    ,'legitimization_requirements'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'unique ID for this legitimization requirement at the exchange'
    ,'legitimization_requirement_serial_id'
    ,'legitimization_requirements'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'foreign key linking the entry to the wire_targets table, NOT a primary key (multiple legitimizations are possible per wire target)'
    ,'h_payto'
    ,'legitimization_requirements'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'space-separated list of required checks'
    ,'required_checks'
    ,'legitimization_requirements'
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_legitimization_requirements(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  partition_name VARCHAR;
BEGIN
  partition_name = concat_ws('_', 'legitimization_requirements', partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || partition_name || ' '
    'ADD CONSTRAINT ' || partition_name || '_serial_id_key '
    'UNIQUE (legitimization_requirement_serial_id)');
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('legitimization_requirements'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('legitimization_requirements'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_reserves(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'reserves';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(reserve_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA PRIMARY KEY CHECK(LENGTH(reserve_pub)=32)'
      ',current_balance_val INT8 NOT NULL DEFAULT(0)'
      ',current_balance_frac INT4 NOT NULL DEFAULT(0)'
      ',purses_active INT8 NOT NULL DEFAULT(0)'
      ',purses_allowed INT8 NOT NULL DEFAULT(0)'
      ',max_age INT4 NOT NULL DEFAULT(120)'
      ',expiration_date INT8 NOT NULL'
      ',gc_date INT8 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Summarizes the balance of a reserve. Updated when new funds are added or withdrawn.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'EdDSA public key of the reserve. Knowledge of the private key implies ownership over the balance.'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Current balance remaining with the reserve.'
    ,'current_balance_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Number of purses that were created by this reserve that are not expired and not fully paid.'
    ,'purses_active'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Number of purses that this reserve is allowed to have active at most.'
    ,'purses_allowed'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Used to trigger closing of reserves that have not been drained after some time'
    ,'expiration_date'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Used to forget all information about a reserve during garbage collection'
    ,'gc_date'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_reserves(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'reserves';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_unique_uuid'
    ' UNIQUE (reserve_uuid)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_expiration_index '
    'ON ' || table_name || ' '
    '(expiration_date'
    ',current_balance_val'
    ',current_balance_frac'
    ');'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_expiration_index '
    'IS ' || quote_literal('used in get_expired_reserves') || ';'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve_uuid_index '
    'ON ' || table_name || ' '
    '(reserve_uuid);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_gc_date_index '
    'ON ' || table_name || ' '
    '(gc_date);'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_gc_date_index '
    'IS ' || quote_literal('for reserve garbage collection') || ';'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_reserves_in(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_in';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(reserve_in_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA PRIMARY KEY'
      ',wire_reference INT8 NOT NULL'
      ',credit_val INT8 NOT NULL'
      ',credit_frac INT4 NOT NULL'
      ',wire_source_h_payto BYTEA CHECK (LENGTH(wire_source_h_payto)=32)'
      ',exchange_account_section TEXT NOT NULL'
      ',execution_date INT8 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'list of transfers of funds into the reserves, one per incoming wire transfer'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the debited bank account and KYC status'
    ,'wire_source_h_payto'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the reserve. Private key signifies ownership of the remaining balance.'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Amount that was transferred into the reserve'
    ,'credit_val'
    ,table_name
    ,partition_suffix
  );
END $$;
CREATE FUNCTION constrain_table_reserves_in(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_in';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_reserve_in_serial_id_key'
    ' UNIQUE (reserve_in_serial_id)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve_in_serial_id_index '
    'ON ' || table_name || ' '
    '(reserve_in_serial_id);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_exch_accnt_section_execution_date_idx '
    'ON ' || table_name || ' '
    '(exchange_account_section '
    ',execution_date'
    ');'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_exch_accnt_reserve_in_serial_id_idx '
    'ON ' || table_name || ' '
    '(exchange_account_section'
    ',reserve_in_serial_id DESC'
    ');'
  );
END
$$;
CREATE FUNCTION foreign_table_reserves_in()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'reserves_in';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub) '
    ' REFERENCES reserves(reserve_pub) ON DELETE CASCADE'
  );
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves_in'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_in'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserves_in'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_reserves_close(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_close';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(close_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL'
      ',execution_date INT8 NOT NULL'
      ',wtid BYTEA NOT NULL CHECK (LENGTH(wtid)=32)'
      ',wire_target_h_payto BYTEA CHECK (LENGTH(wire_target_h_payto)=32)'
      ',amount_val INT8 NOT NULL'
      ',amount_frac INT4 NOT NULL'
      ',closing_fee_val INT8 NOT NULL'
      ',closing_fee_frac INT4 NOT NULL'
      ',close_request_row INT8 NOT NULL DEFAULT(0)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'wire transfers executed by the reserve to close reserves'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the credited bank account (and KYC status). Note that closing does not depend on KYC.'
    ,'wire_target_h_payto'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_reserves_close(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_close';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_close_uuid_pkey'
    ' PRIMARY KEY (close_uuid)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve_pub_index '
    'ON ' || table_name || ' (reserve_pub);'
  );
END $$;
CREATE FUNCTION foreign_table_reserves_close()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_close';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub)'
    ' REFERENCES reserves (reserve_pub) ON DELETE CASCADE'
  );
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves_close'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_close'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserves_close'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_close_requests(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'close_requests';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(close_request_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL CHECK (LENGTH(reserve_pub)=32)'
      ',close_timestamp INT8 NOT NULL'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',close_val INT8 NOT NULL'
      ',close_frac INT4 NOT NULL'
      ',close_fee_val INT8 NOT NULL'
      ',close_fee_frac INT4 NOT NULL'
      ',payto_uri VARCHAR NOT NULL'
      ',done BOOL NOT NULL DEFAULT(FALSE)'
      ',PRIMARY KEY (reserve_pub,close_timestamp)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Explicit requests by a reserve owner to close a reserve immediately'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'When the request was created by the client'
    ,'close_timestamp'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature affirming that the reserve is to be closed'
    ,'reserve_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Balance of the reserve at the time of closing, to be wired to the associated bank account (minus the closing fee)'
    ,'close_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the credited bank account. Optional.'
    ,'payto_uri'
    ,table_name
    ,partition_suffix
  );
END $$;
CREATE FUNCTION constrain_table_close_requests(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'close_requests';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_close_request_uuid_index '
    'ON ' || table_name || ' '
    '(close_request_serial_id);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_close_request_done_index '
    'ON ' || table_name || ' '
    '(done);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_close_request_uuid_pkey'
    ' UNIQUE (close_request_serial_id)'
  );
END
$$;
CREATE FUNCTION foreign_table_close_requests()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'close_requests';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub) '
    ' REFERENCES reserves(reserve_pub) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('close_requests'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('close_requests'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('close_requests'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_reserves_open_deposits(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_open_deposits';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(reserve_open_deposit_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',reserve_pub BYTEA NOT NULL CHECK (LENGTH(reserve_pub)=32)'
      ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)'
      ',coin_sig BYTEA NOT NULL CHECK (LENGTH(coin_sig)=64)'
      ',contribution_val INT8 NOT NULL'
      ',contribution_frac INT4 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'coin contributions paying for a reserve to remain open'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the specific reserve being paid for (possibly together with reserve_sig).'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_reserves_open_deposits(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_open_deposits';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name || ' '
      'ADD CONSTRAINT ' || table_name || '_coin_unique '
        'PRIMARY KEY (coin_pub,coin_sig)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_uuid '
    'ON ' || table_name || ' '
    '(reserve_open_deposit_uuid);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve '
    'ON ' || table_name || ' '
    '(reserve_pub);'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves_open_deposits'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_open_deposits'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_reserves_open_requests(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_open_requests';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(open_request_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL'
      ',request_timestamp INT8 NOT NULL'
      ',expiration_date INT8 NOT NULL'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',reserve_payment_val INT8 NOT NULL'
      ',reserve_payment_frac INT4 NOT NULL'
      ',requested_purse_limit INT4 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table (
     'requests to keep a reserve open'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column (
     'Fee to pay for the request from the reserve balance itself.'
    ,'reserve_payment_val'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_reserves_open_requests(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_open_requests';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_by_uuid'
    ' PRIMARY KEY (open_request_uuid)'
    ',ADD CONSTRAINT ' || table_name || '_by_time'
    ' UNIQUE (reserve_pub,request_timestamp)'
  );
END
$$;
CREATE FUNCTION foreign_table_reserves_open_requests()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_open_requests';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub '
    ' FOREIGN KEY (reserve_pub)'
    ' REFERENCES reserves (reserve_pub) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves_open_requests'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_open_requests'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserves_open_requests'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_reserves_out(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_out';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(reserve_out_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_blind_ev BYTEA CHECK (LENGTH(h_blind_ev)=64) UNIQUE'
      ',denominations_serial INT8 NOT NULL'
      ',denom_sig BYTEA NOT NULL'
      ',reserve_uuid INT8 NOT NULL'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',execution_date INT8 NOT NULL'
      ',amount_with_fee_val INT8 NOT NULL'
      ',amount_with_fee_frac INT4 NOT NULL'
    ') %s ;'
    ,'reserves_out'
    ,'PARTITION BY HASH (h_blind_ev)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table (
     'Withdraw operations performed on reserves.'
    ,'reserves_out'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column (
     'Hash of the blinded coin, used as primary key here so that broken clients that use a non-random coin or blinding factor fail to withdraw (otherwise they would fail on deposit when the coin is not unique there).'
    ,'h_blind_ev'
    ,'reserves_out'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column (
     'We do not CASCADE ON DELETE for the foreign constrain here, as we may keep the denomination data alive'
    ,'denominations_serial'
    ,'reserves_out'
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_reserves_out(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_out';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_reserve_out_serial_id_key'
    ' UNIQUE (reserve_out_serial_id)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve_uuid_and_execution_date_index '
    'ON ' || table_name || ' '
    '(reserve_uuid, execution_date);'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_reserve_uuid_and_execution_date_index '
    'IS ' || quote_literal('for get_reserves_out and exchange_do_withdraw_limit_check') || ';'
  );
END
$$;
CREATE FUNCTION foreign_table_reserves_out()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_out';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_denom'
    ' FOREIGN KEY (denominations_serial)'
    ' REFERENCES denominations (denominations_serial)'
    ',ADD CONSTRAINT ' || table_name || '_foreign_reserve '
    ' FOREIGN KEY (reserve_uuid)'
    ' REFERENCES reserves (reserve_uuid) ON DELETE CASCADE'
  );
END
$$;
CREATE FUNCTION create_table_reserves_out_by_reserve(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'reserves_out_by_reserve';
BEGIN
  PERFORM create_partitioned_table(
  'CREATE TABLE %I'
    '(reserve_uuid INT8 NOT NULL' 
    ',h_blind_ev BYTEA CHECK (LENGTH(h_blind_ev)=64)'
    ') %s '
    ,table_name
    ,'PARTITION BY HASH (reserve_uuid)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table (
     'Information in this table is strictly redundant with that of reserves_out, but saved by a different primary key for fast lookups by reserve public key/uuid.'
    ,table_name
    ,partition_suffix
  );
END $$;
CREATE FUNCTION constrain_table_reserves_out_by_reserve(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'reserves_out_by_reserve';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_main_index '
    'ON ' || table_name || ' '
    '(reserve_uuid);'
  );
END $$;
CREATE FUNCTION reserves_out_by_reserve_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO exchange.reserves_out_by_reserve
    (reserve_uuid
    ,h_blind_ev)
  VALUES
    (NEW.reserve_uuid
    ,NEW.h_blind_ev);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION reserves_out_by_reserve_insert_trigger()
  IS 'Replicate reserve_out inserts into reserve_out_by_reserve table.';
CREATE FUNCTION reserves_out_by_reserve_delete_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  DELETE FROM exchange.reserves_out_by_reserve
   WHERE reserve_uuid = OLD.reserve_uuid;
  RETURN OLD;
END $$;
COMMENT ON FUNCTION reserves_out_by_reserve_delete_trigger()
  IS 'Replicate reserve_out deletions into reserve_out_by_reserve table.';
CREATE FUNCTION master_table_reserves_out()
RETURNS void
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER reserves_out_on_insert
  AFTER INSERT
   ON reserves_out
   FOR EACH ROW EXECUTE FUNCTION reserves_out_by_reserve_insert_trigger();
  CREATE TRIGGER reserves_out_on_delete
  AFTER DELETE
    ON reserves_out
   FOR EACH ROW EXECUTE FUNCTION reserves_out_by_reserve_delete_trigger();
END $$;
COMMENT ON FUNCTION master_table_reserves_out()
  IS 'Setup triggers to replicate reserve_out into reserve_out_by_reserve.';
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves_out'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_out'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserves_out'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('reserves_out_by_reserve'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_out_by_reserve'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserves_out'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_known_coins(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'known_coins';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(known_coin_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',denominations_serial INT8 NOT NULL'
      ',coin_pub BYTEA NOT NULL PRIMARY KEY CHECK (LENGTH(coin_pub)=32)'
      ',age_commitment_hash BYTEA CHECK (LENGTH(age_commitment_hash)=32)'
      ',denom_sig BYTEA NOT NULL'
      ',remaining_val INT8 NOT NULL DEFAULT(0)'
      ',remaining_frac INT4 NOT NULL DEFAULT(0)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'information about coins and their signatures, so we do not have to store the signatures more than once if a coin is involved in multiple operations'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Denomination of the coin, determines the value of the original coin and applicable fees for coin-specific operations.'
    ,'denominations_serial'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'EdDSA public key of the coin'
    ,'coin_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Value of the coin that remains to be spent'
    ,'remaining_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Optional hash of the age commitment for age restrictions as per DD 24 (active if denom_type has the respective bit set)'
    ,'age_commitment_hash'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'This is the signature of the exchange that affirms that the coin is a valid coin. The specific signature type depends on denom_type of the denomination.'
    ,'denom_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_known_coins(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'known_coins';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_known_coin_id_key'
    ' UNIQUE (known_coin_id)'
  );
END
$$;
CREATE FUNCTION foreign_table_known_coins()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'known_coins';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_denominations'
    ' FOREIGN KEY (denominations_serial) '
    ' REFERENCES denominations (denominations_serial) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('known_coins'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('known_coins'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('known_coins'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_refresh_commitments(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refresh_commitments';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(melt_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',rc BYTEA PRIMARY KEY CHECK (LENGTH(rc)=64)'
      ',old_coin_pub BYTEA NOT NULL'
      ',old_coin_sig BYTEA NOT NULL CHECK(LENGTH(old_coin_sig)=64)'
      ',amount_with_fee_val INT8 NOT NULL'
      ',amount_with_fee_frac INT4 NOT NULL'
      ',noreveal_index INT4 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (rc)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Commitments made when melting coins and the gamma value chosen by the exchange.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'The gamma value chosen by the exchange in the cut-and-choose protocol'
    ,'noreveal_index'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Commitment made by the client, hash over the various client inputs in the cut-and-choose protocol'
    ,'rc'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Coin being melted in the refresh process.'
    ,'old_coin_pub'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_refresh_commitments(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refresh_commitments';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_old_coin_pub_index '
    'ON ' || table_name || ' '
    '(old_coin_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_melt_serial_id_key'
    ' UNIQUE (melt_serial_id)'
  );
END
$$;
CREATE FUNCTION foreign_table_refresh_commitments()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refresh_commitments';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (old_coin_pub) '
    ' REFERENCES known_coins (coin_pub) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('refresh_commitments'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('refresh_commitments'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('refresh_commitments'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_refresh_revealed_coins(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refresh_revealed_coins';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(rrc_serial BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',melt_serial_id INT8 NOT NULL'
      ',freshcoin_index INT4 NOT NULL'
      ',link_sig BYTEA NOT NULL CHECK(LENGTH(link_sig)=64)'
      ',denominations_serial INT8 NOT NULL'
      ',coin_ev BYTEA NOT NULL'
      ',h_coin_ev BYTEA NOT NULL CHECK(LENGTH(h_coin_ev)=64)'
      ',ev_sig BYTEA NOT NULL'
      ',ewv BYTEA NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (melt_serial_id)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Revelations about the new coins that are to be created during a melting session.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'needed for exchange-auditor replication logic'
    ,'rrc_serial'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the refresh commitment (rc) of the melt operation.'
    ,'melt_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'index of the fresh coin being created (one melt operation may result in multiple fresh coins)'
    ,'freshcoin_index'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'envelope of the new coin to be signed'
    ,'coin_ev'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'exchange contributed values in the creation of the fresh coin (see /csr)'
    ,'ewv'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash of the envelope of the new coin to be signed (for lookups)'
    ,'h_coin_ev'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'exchange signature over the envelope'
    ,'ev_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_refresh_revealed_coins(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refresh_revealed_coins';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_coins_by_melt_serial_id_index '
    'ON ' || table_name || ' '
    '(melt_serial_id);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_rrc_serial_key'
    ' UNIQUE (rrc_serial) '
    ',ADD CONSTRAINT ' || table_name || '_coin_ev_key'
    ' UNIQUE (coin_ev) '
    ',ADD CONSTRAINT ' || table_name || '_h_coin_ev_key'
    ' UNIQUE (h_coin_ev) '
    ',ADD PRIMARY KEY (melt_serial_id, freshcoin_index)'
  );
END
$$;
CREATE FUNCTION foreign_table_refresh_revealed_coins()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refresh_revealed_coins';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_melt'
    ' FOREIGN KEY (melt_serial_id)'
    ' REFERENCES refresh_commitments (melt_serial_id) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_denom'
    ' FOREIGN KEY (denominations_serial)'
    ' REFERENCES denominations (denominations_serial) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('refresh_revealed_coins'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('refresh_revealed_coins'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('refresh_revealed_coins'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_refresh_transfer_keys(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refresh_transfer_keys';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(rtc_serial BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',melt_serial_id INT8 PRIMARY KEY'
      ',transfer_pub BYTEA NOT NULL CHECK(LENGTH(transfer_pub)=32)'
      ',transfer_privs BYTEA NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (melt_serial_id)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Transfer keys of a refresh operation (the data revealed to the exchange).'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'needed for exchange-auditor replication logic'
    ,'rtc_serial'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the refresh commitment (rc) of the operation.'
    ,'melt_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'transfer public key for the gamma index'
    ,'transfer_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'array of TALER_CNC_KAPPA-1 transfer private keys that have been revealed, with the gamma entry being skipped'
    ,'transfer_privs'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_refresh_transfer_keys(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refresh_transfer_keys';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_rtc_serial_key'
    ' UNIQUE (rtc_serial)'
  );
END
$$;
CREATE FUNCTION foreign_table_refresh_transfer_keys()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refresh_transfer_keys';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || 'foreign_melt_serial_id'
    ' FOREIGN KEY (melt_serial_id)'
    ' REFERENCES refresh_commitments (melt_serial_id) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('refresh_transfer_keys'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('refresh_transfer_keys'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('refresh_transfer_keys'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_deposits(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'deposits';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(deposit_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',shard INT8 NOT NULL'
      ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)'
      ',known_coin_id INT8 NOT NULL' 
      ',amount_with_fee_val INT8 NOT NULL'
      ',amount_with_fee_frac INT4 NOT NULL'
      ',wallet_timestamp INT8 NOT NULL'
      ',exchange_timestamp INT8 NOT NULL'
      ',refund_deadline INT8 NOT NULL'
      ',wire_deadline INT8 NOT NULL'
      ',merchant_pub BYTEA NOT NULL CHECK (LENGTH(merchant_pub)=32)'
      ',h_contract_terms BYTEA NOT NULL CHECK (LENGTH(h_contract_terms)=64)'
      ',coin_sig BYTEA NOT NULL CHECK (LENGTH(coin_sig)=64)'
      ',wire_salt BYTEA NOT NULL CHECK (LENGTH(wire_salt)=16)'
      ',wire_target_h_payto BYTEA CHECK (LENGTH(wire_target_h_payto)=32)'
      ',done BOOLEAN NOT NULL DEFAULT FALSE'
      ',policy_blocked BOOLEAN NOT NULL DEFAULT FALSE'
      ',policy_details_serial_id INT8'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
    'Deposits we have received and for which we need to make (aggregate) wire transfers (and manage refunds).'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Used for load sharding in the materialized indices. Should be set based on merchant_pub. 64-bit value because we need an *unsigned* 32-bit value.'
    ,'shard'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Used for garbage collection'
    ,'known_coin_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the target bank account and KYC status'
    ,'wire_target_h_payto'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Salt used when hashing the payto://-URI to get the h_wire'
    ,'wire_salt'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Set to TRUE once we have included this deposit in some aggregate wire transfer to the merchant'
    ,'done'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'True if the aggregation of the deposit is currently blocked by some policy extension mechanism. Used to filter out deposits that must not be processed by the canonical deposit logic.'
    ,'policy_blocked'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'References policy extensions table, NULL if extensions are not used'
    ,'policy_details_serial_id'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_deposits(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'deposits';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_deposit_serial_id_pkey'
    ' PRIMARY KEY (deposit_serial_id) '
    ',ADD CONSTRAINT ' || table_name || '_coin_pub_merchant_pub_h_contract_terms_key'
    ' UNIQUE (coin_pub, merchant_pub, h_contract_terms)'
  );
END
$$;
CREATE FUNCTION foreign_table_deposits()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'deposits';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (coin_pub) '
    ' REFERENCES known_coins (coin_pub) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_coin_id'
    ' FOREIGN KEY (known_coin_id) '
    ' REFERENCES known_coins (known_coin_id) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_policy_details'
    ' FOREIGN KEY (policy_details_serial_id) '
    ' REFERENCES policy_details (policy_details_serial_id) ON DELETE CASCADE'
  );
END
$$;
CREATE FUNCTION create_table_deposits_by_ready(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'deposits_by_ready';
BEGIN
  PERFORM create_partitioned_table(
  'CREATE TABLE %I'
    '(wire_deadline INT8 NOT NULL'
    ',shard INT8 NOT NULL'
    ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)'
    ',deposit_serial_id INT8'
    ') %s ;'
    ,table_name
    ,'PARTITION BY RANGE (wire_deadline)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
    'Enables fast lookups for deposits_get_ready, auto-populated via TRIGGER below'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_deposits_by_ready(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'deposits_by_ready';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_main_index '
    'ON ' || table_name || ' '
    '(wire_deadline ASC, shard ASC, coin_pub);'
  );
END
$$;
CREATE FUNCTION create_table_deposits_for_matching(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'deposits_for_matching';
BEGIN
  PERFORM create_partitioned_table(
  'CREATE TABLE %I'
    '(refund_deadline INT8 NOT NULL'
    ',merchant_pub BYTEA NOT NULL CHECK (LENGTH(merchant_pub)=32)'
    ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)' 
    ',deposit_serial_id INT8'
    ') %s ;'
    ,table_name
    ,'PARTITION BY RANGE (refund_deadline)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Enables fast lookups for deposits_iterate_matching, auto-populated via TRIGGER below'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_deposits_for_matching(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'deposits_for_matching';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_main_index'
    ' ON ' || table_name || ' '
    '(refund_deadline ASC, merchant_pub, coin_pub);'
  );
END
$$;
CREATE OR REPLACE FUNCTION deposits_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
DECLARE
  is_ready BOOLEAN;
BEGIN
  is_ready = NOT (NEW.done OR NEW.policy_blocked);
  IF (is_ready)
  THEN
    INSERT INTO exchange.deposits_by_ready
      (wire_deadline
      ,shard
      ,coin_pub
      ,deposit_serial_id)
    VALUES
      (NEW.wire_deadline
      ,NEW.shard
      ,NEW.coin_pub
      ,NEW.deposit_serial_id);
    INSERT INTO exchange.deposits_for_matching
      (refund_deadline
      ,merchant_pub
      ,coin_pub
      ,deposit_serial_id)
    VALUES
      (NEW.refund_deadline
      ,NEW.merchant_pub
      ,NEW.coin_pub
      ,NEW.deposit_serial_id);
  END IF;
  RETURN NEW;
END $$;
COMMENT ON FUNCTION deposits_insert_trigger()
  IS 'Replicate deposit inserts into materialized indices.';
CREATE OR REPLACE FUNCTION deposits_update_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
DECLARE
  was_ready BOOLEAN;
DECLARE
  is_ready BOOLEAN;
BEGIN
  was_ready = NOT (OLD.done OR OLD.policy_blocked);
  is_ready = NOT (NEW.done OR NEW.policy_blocked);
  IF (was_ready AND NOT is_ready)
  THEN
    DELETE FROM exchange.deposits_by_ready
     WHERE wire_deadline = OLD.wire_deadline
       AND shard = OLD.shard
       AND coin_pub = OLD.coin_pub
       AND deposit_serial_id = OLD.deposit_serial_id;
    DELETE FROM exchange.deposits_for_matching
     WHERE refund_deadline = OLD.refund_deadline
       AND merchant_pub = OLD.merchant_pub
       AND coin_pub = OLD.coin_pub
       AND deposit_serial_id = OLD.deposit_serial_id;
  END IF;
  IF (is_ready AND NOT was_ready)
  THEN
    INSERT INTO exchange.deposits_by_ready
      (wire_deadline
      ,shard
      ,coin_pub
      ,deposit_serial_id)
    VALUES
      (NEW.wire_deadline
      ,NEW.shard
      ,NEW.coin_pub
      ,NEW.deposit_serial_id);
    INSERT INTO exchange.deposits_for_matching
      (refund_deadline
      ,merchant_pub
      ,coin_pub
      ,deposit_serial_id)
    VALUES
      (NEW.refund_deadline
      ,NEW.merchant_pub
      ,NEW.coin_pub
      ,NEW.deposit_serial_id);
  END IF;
  RETURN NEW;
END $$;
COMMENT ON FUNCTION deposits_update_trigger()
  IS 'Replicate deposits changes into materialized indices.';
CREATE OR REPLACE FUNCTION deposits_delete_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
DECLARE
  was_ready BOOLEAN;
BEGIN
  was_ready = NOT (OLD.done OR OLD.policy_blocked);
  IF (was_ready)
  THEN
    DELETE FROM exchange.deposits_by_ready
     WHERE wire_deadline = OLD.wire_deadline
       AND shard = OLD.shard
       AND coin_pub = OLD.coin_pub
       AND deposit_serial_id = OLD.deposit_serial_id;
    DELETE FROM exchange.deposits_for_matching
     WHERE refund_deadline = OLD.refund_deadline
       AND merchant_pub = OLD.merchant_pub
       AND coin_pub = OLD.coin_pub
       AND deposit_serial_id = OLD.deposit_serial_id;
  END IF;
  RETURN NEW;
END $$;
COMMENT ON FUNCTION deposits_delete_trigger()
  IS 'Replicate deposit deletions into materialized indices.';
CREATE FUNCTION master_table_deposits()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER deposits_on_insert
    AFTER INSERT
     ON deposits
     FOR EACH ROW EXECUTE FUNCTION deposits_insert_trigger();
  CREATE TRIGGER deposits_on_update
    AFTER UPDATE
      ON deposits
     FOR EACH ROW EXECUTE FUNCTION deposits_update_trigger();
  CREATE TRIGGER deposits_on_delete
    AFTER DELETE
     ON deposits
     FOR EACH ROW EXECUTE FUNCTION deposits_delete_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('deposits'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('deposits'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('deposits'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('deposits_by_ready'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,TRUE),
    ('deposits_by_ready'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,TRUE),
    ('deposits_for_matching'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,TRUE),
    ('deposits_for_matching'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,TRUE),
    ('deposits'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_refunds(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refunds';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(refund_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)'
      ',deposit_serial_id INT8 NOT NULL'
      ',merchant_sig BYTEA NOT NULL CHECK(LENGTH(merchant_sig)=64)'
      ',rtransaction_id INT8 NOT NULL'
      ',amount_with_fee_val INT8 NOT NULL'
      ',amount_with_fee_frac INT4 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Data on coins that were refunded. Technically, refunds always apply against specific deposit operations involving a coin. The combination of coin_pub, merchant_pub, h_contract_terms and rtransaction_id MUST be unique, and we usually select by coin_pub so that one goes first.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies ONLY the merchant_pub, h_contract_terms and coin_pub. Multiple deposits may match a refund, this only identifies one of them.'
    ,'deposit_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'used by the merchant to make refunds unique in case the same coin for the same deposit gets a subsequent (higher) refund'
    ,'rtransaction_id'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_refunds (
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refunds';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_coin_pub_index '
    'ON ' || table_name || ' '
    '(coin_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_refund_serial_id_key'
    ' UNIQUE (refund_serial_id) '
    ',ADD PRIMARY KEY (deposit_serial_id, rtransaction_id) '
  );
END
$$;
CREATE FUNCTION foreign_table_refunds ()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refunds';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (coin_pub) '
    ' REFERENCES known_coins (coin_pub) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_deposit'
    ' FOREIGN KEY (deposit_serial_id) '
    ' REFERENCES deposits (deposit_serial_id) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('refunds'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('refunds'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('refunds'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_wire_out(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'wire_out';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I'
      '(wireout_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',execution_date INT8 NOT NULL'
      ',wtid_raw BYTEA UNIQUE NOT NULL CHECK (LENGTH(wtid_raw)=32)'
      ',wire_target_h_payto BYTEA CHECK (LENGTH(wire_target_h_payto)=32)'
      ',exchange_account_section TEXT NOT NULL'
      ',amount_val INT8 NOT NULL'
      ',amount_frac INT4 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (wtid_raw)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'wire transfers the exchange has executed'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'identifies the configuration section with the debit account of this payment'
    ,'exchange_account_section'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the credited bank account and KYC status'
    ,'wire_target_h_payto'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_wire_out(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'wire_out';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_wire_target_h_payto_index '
    'ON ' || table_name || ' '
    '(wire_target_h_payto);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wireout_uuid_pkey'
    ' PRIMARY KEY (wireout_uuid)'
  );
END
$$;
CREATE FUNCTION wire_out_delete_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  DELETE FROM exchange.aggregation_tracking
   WHERE wtid_raw = OLD.wtid_raw;
  RETURN OLD;
END $$;
COMMENT ON FUNCTION wire_out_delete_trigger()
  IS 'Replicate reserve_out deletions into aggregation_tracking. This replaces an earlier use of an ON DELETE CASCADE that required a DEFERRABLE constraint and conflicted with nice partitioning.';
CREATE FUNCTION master_table_wire_out()
RETURNS void
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER wire_out_on_delete
    AFTER DELETE
      ON wire_out
     FOR EACH ROW EXECUTE FUNCTION wire_out_delete_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('wire_out'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('wire_out'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('wire_out'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_aggregation_transient(
  IN shard_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'aggregation_transient';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
    '(amount_val INT8 NOT NULL'
    ',amount_frac INT4 NOT NULL'
    ',wire_target_h_payto BYTEA CHECK (LENGTH(wire_target_h_payto)=32)'
    ',merchant_pub BYTEA CHECK (LENGTH(merchant_pub)=32)'
    ',exchange_account_section TEXT NOT NULL'
    ',legitimization_requirement_serial_id INT8 NOT NULL DEFAULT(0)'
    ',wtid_raw BYTEA NOT NULL CHECK (LENGTH(wtid_raw)=32)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (wire_target_h_payto)'
    ,shard_suffix
  );
  PERFORM comment_partitioned_table(
    'aggregations currently happening (lacking wire_out, usually because the amount is too low); this table is not replicated'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
       'Sum of all of the aggregated deposits (without deposit fees)'
      ,'amount_val'
      ,table_name
      ,shard_suffix
  );
  PERFORM comment_partitioned_column(
       'identifier of the wire transfer'
      ,'wtid_raw'
      ,table_name
      ,shard_suffix
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('aggregation_transient'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_aggregation_tracking(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'aggregation_tracking';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(aggregation_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',deposit_serial_id INT8 PRIMARY KEY'
      ',wtid_raw BYTEA NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (deposit_serial_id)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'mapping from wire transfer identifiers (WTID) to deposits (and back)'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'identifier of the wire transfer'
    ,'wtid_raw'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_aggregation_tracking(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'aggregation_tracking';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_wtid_raw_index '
    'ON ' || table_name || ' '
    '(wtid_raw);'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_wtid_raw_index '
    'IS ' || quote_literal('for lookup_transactions') || ';'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_aggregation_serial_id_key'
    ' UNIQUE (aggregation_serial_id) '
  );
END
$$;
CREATE FUNCTION foreign_table_aggregation_tracking()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'aggregation_tracking';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_deposit'
    ' FOREIGN KEY (deposit_serial_id) '
    ' REFERENCES deposits (deposit_serial_id) ON DELETE CASCADE' 
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('aggregation_tracking'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('aggregation_tracking'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('aggregation_tracking'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_recoup(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'recoup';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(recoup_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)'
      ',coin_sig BYTEA NOT NULL CHECK(LENGTH(coin_sig)=64)'
      ',coin_blind BYTEA NOT NULL CHECK(LENGTH(coin_blind)=32)'
      ',amount_val INT8 NOT NULL'
      ',amount_frac INT4 NOT NULL'
      ',recoup_timestamp INT8 NOT NULL'
      ',reserve_out_serial_id INT8 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub);'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Information about recoups that were executed between a coin and a reserve. In this type of recoup, the amount is credited back to the reserve from which the coin originated.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Coin that is being debited in the recoup. Do not CASCADE ON DROP on the coin_pub, as we may keep the coin alive!'
    ,'coin_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the h_blind_ev of the recouped coin and provides the link to the credited reserve.'
    ,'reserve_out_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature by the coin affirming the recoup, of type TALER_SIGNATURE_WALLET_COIN_RECOUP'
    ,'coin_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Denomination blinding key used when creating the blinded coin from the planchet. Secret revealed during the recoup to provide the linkage between the coin and the withdraw operation.'
    ,'coin_blind'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_recoup(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'recoup';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_coin_pub_index '
    'ON ' || table_name || ' '
    '(coin_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_recoup_uuid_key'
    ' UNIQUE (recoup_uuid) '
  );
END
$$;
CREATE FUNCTION foreign_table_recoup()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'recoup';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserves_out'
    ' FOREIGN KEY (reserve_out_serial_id) '
    ' REFERENCES reserves_out (reserve_out_serial_id) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (coin_pub) '
    ' REFERENCES known_coins (coin_pub)'
  );
END
$$;
CREATE FUNCTION create_table_recoup_by_reserve(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'recoup_by_reserve';
BEGIN
  PERFORM create_partitioned_table(
  'CREATE TABLE %I'
    '(reserve_out_serial_id INT8 NOT NULL' 
    ',coin_pub BYTEA CHECK (LENGTH(coin_pub)=32)' 
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_out_serial_id)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Information in this table is strictly redundant with that of recoup, but saved by a different primary key for fast lookups by reserve_out_serial_id.'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_recoup_by_reserve(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'recoup_by_reserve';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_main_index '
    'ON ' || table_name || ' '
    '(reserve_out_serial_id);'
  );
END
$$;
CREATE FUNCTION recoup_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO exchange.recoup_by_reserve
    (reserve_out_serial_id
    ,coin_pub)
  VALUES
    (NEW.reserve_out_serial_id
    ,NEW.coin_pub);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION recoup_insert_trigger()
  IS 'Replicate recoup inserts into recoup_by_reserve table.';
CREATE FUNCTION recoup_delete_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  DELETE FROM exchange.recoup_by_reserve
   WHERE reserve_out_serial_id = OLD.reserve_out_serial_id
     AND coin_pub = OLD.coin_pub;
  RETURN OLD;
END $$;
COMMENT ON FUNCTION recoup_delete_trigger()
  IS 'Replicate recoup deletions into recoup_by_reserve table.';
CREATE FUNCTION master_table_recoup()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER recoup_on_insert
    AFTER INSERT
     ON recoup
     FOR EACH ROW EXECUTE FUNCTION recoup_insert_trigger();
  CREATE TRIGGER recoup_on_delete
    AFTER DELETE
      ON recoup
     FOR EACH ROW EXECUTE FUNCTION recoup_delete_trigger();
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('recoup'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('recoup'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('recoup'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('recoup_by_reserve'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('recoup_by_reserve'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('recoup'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_recoup_refresh(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'recoup_refresh';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
    '(recoup_refresh_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
    ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)'
    ',known_coin_id BIGINT NOT NULL'
    ',coin_sig BYTEA NOT NULL CHECK(LENGTH(coin_sig)=64)'
    ',coin_blind BYTEA NOT NULL CHECK(LENGTH(coin_blind)=32)'
    ',amount_val INT8 NOT NULL'
    ',amount_frac INT4 NOT NULL'
    ',recoup_timestamp INT8 NOT NULL'
    ',rrc_serial INT8 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Table of coins that originated from a refresh operation and that were recouped. Links the (fresh) coin to the melted operation (and thus the old coin). A recoup on a refreshed coin credits the old coin and debits the fresh coin.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Refreshed coin of a revoked denomination where the residual value is credited to the old coin. Do not CASCADE ON DROP on the coin_pub, as we may keep the coin alive!'
    ,'coin_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'FIXME: (To be) used for garbage collection (in the absence of foreign constraints, in the future)'
    ,'known_coin_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Link to the refresh operation. Also identifies the h_blind_ev of the recouped coin (as h_coin_ev).'
    ,'rrc_serial'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Denomination blinding key used when creating the blinded coin from the planchet. Secret revealed during the recoup to provide the linkage between the coin and the refresh operation.'
    ,'coin_blind'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_recoup_refresh(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'recoup_refresh';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_rrc_serial_index'
    ' ON ' || table_name || ' '
    '(rrc_serial);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_coin_pub_index'
    ' ON ' || table_name || ' '
    '(coin_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_recoup_refresh_uuid_key'
    ' UNIQUE (recoup_refresh_uuid) '
  );
END
$$;
CREATE FUNCTION foreign_table_recoup_refresh()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'recoup_refresh';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (coin_pub) '
    ' REFERENCES known_coins (coin_pub)'
    ',ADD CONSTRAINT ' || table_name || '_foreign_known_coin_id'
    ' FOREIGN KEY (known_coin_id) '
    ' REFERENCES known_coins (known_coin_id) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_rrc_serial'
    ' FOREIGN KEY (rrc_serial) '
    ' REFERENCES refresh_revealed_coins (rrc_serial) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('recoup_refresh'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('recoup_refresh'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('recoup_refresh'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_prewire(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'prewire';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(prewire_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY'
      ',wire_method TEXT NOT NULL'
      ',finished BOOLEAN NOT NULL DEFAULT false'
      ',failed BOOLEAN NOT NULL DEFAULT false'
      ',buf BYTEA NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (prewire_uuid)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'pre-commit data for wire transfers we are about to execute'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'set to TRUE if the bank responded with a non-transient failure to our transfer request'
    ,'failed'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'set to TRUE once bank confirmed receiving the wire transfer request'
    ,'finished'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'serialized data to send to the bank to execute the wire transfer'
    ,'buf'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_prewire(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'prewire';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_finished_index '
    'ON ' || table_name || ' '
    '(finished);'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_finished_index '
    'IS ' || quote_literal('for gc_prewire') || ';'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_failed_finished_index '
    'ON ' || table_name || ' '
    '(failed,finished);'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_failed_finished_index '
    'IS ' || quote_literal('for wire_prepare_data_get') || ';'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('prewire'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('prewire'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_cs_nonce_locks(
  partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(cs_nonce_lock_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',nonce BYTEA PRIMARY KEY CHECK (LENGTH(nonce)=32)'
      ',op_hash BYTEA NOT NULL CHECK (LENGTH(op_hash)=64)'
      ',max_denomination_serial INT8 NOT NULL'
    ') %s ;'
    ,'cs_nonce_locks'
    ,'PARTITION BY HASH (nonce)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'ensures a Clause Schnorr client nonce is locked for use with an operation identified by a hash'
    ,'cs_nonce_locks'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'actual nonce submitted by the client'
    ,'nonce'
    ,'cs_nonce_locks'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash (RC for refresh, blind coin hash for withdraw) the nonce may be used with'
    ,'op_hash'
    ,'cs_nonce_locks'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Maximum number of a CS denomination serial the nonce could be used with, for GC'
    ,'max_denomination_serial'
    ,'cs_nonce_locks'
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_cs_nonce_locks(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'cs_nonce_locks';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_cs_nonce_lock_serial_id_key'
    ' UNIQUE (cs_nonce_lock_serial_id)'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('cs_nonce_locks'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('cs_nonce_locks'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_purse_requests(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_requests';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(purse_requests_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
      ',merge_pub BYTEA NOT NULL CHECK (LENGTH(merge_pub)=32)'
      ',purse_creation INT8 NOT NULL'
      ',purse_expiration INT8 NOT NULL'
      ',h_contract_terms BYTEA NOT NULL CHECK (LENGTH(h_contract_terms)=64)'
      ',age_limit INT4 NOT NULL'
      ',flags INT4 NOT NULL'
      ',in_reserve_quota BOOLEAN NOT NULL DEFAULT(FALSE)'
      ',amount_with_fee_val INT8 NOT NULL'
      ',amount_with_fee_frac INT4 NOT NULL'
      ',purse_fee_val INT8 NOT NULL'
      ',purse_fee_frac INT4 NOT NULL'
      ',balance_val INT8 NOT NULL DEFAULT (0)'
      ',balance_frac INT4 NOT NULL DEFAULT (0)'
      ',purse_sig BYTEA NOT NULL CHECK(LENGTH(purse_sig)=64)'
      ',PRIMARY KEY (purse_pub)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Requests establishing purses, associating them with a contract but without a target reserve'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Local time when the purse was created. Determines applicable purse fees.'
    ,'purse_creation'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'When the purse is set to expire'
    ,'purse_expiration'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Hash of the contract the parties are to agree to'
    ,'h_contract_terms'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'see the enum TALER_WalletAccountMergeFlags'
    ,'flags'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'set to TRUE if this purse currently counts against the number of free purses in the respective reserve'
    ,'in_reserve_quota'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total amount expected to be in the purse'
    ,'amount_with_fee_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Purse fee the client agreed to pay from the reserve (accepted by the exchange at the time the purse was created). Zero if in_reserve_quota is TRUE.'
    ,'purse_fee_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'Total amount actually in the purse (updated)'
    ,'balance_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature of the purse affirming the purse parameters, of type TALER_SIGNATURE_PURSE_REQUEST'
    ,'purse_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_purse_requests(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_requests';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_merge_pub '
    'ON ' || table_name || ' '
    '(merge_pub);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_purse_expiration '
    'ON ' || table_name || ' '
    '(purse_expiration);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_purse_requests_serial_id_key'
    ' UNIQUE (purse_requests_serial_id) '
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('purse_requests'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('purse_requests'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_purse_merges(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_merges';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
    '(purse_merge_request_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
    ',partner_serial_id INT8'
    ',reserve_pub BYTEA NOT NULL CHECK(length(reserve_pub)=32)'
    ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
    ',merge_sig BYTEA NOT NULL CHECK (LENGTH(merge_sig)=64)'
    ',merge_timestamp INT8 NOT NULL'
    ',PRIMARY KEY (purse_pub)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Merge requests where a purse-owner requested merging the purse into the account'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'identifies the partner exchange, NULL in case the target reserve lives at this exchange'
    ,'partner_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the target reserve'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'signature by the purse private key affirming the merge, of type TALER_SIGNATURE_WALLET_PURSE_MERGE'
    ,'merge_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'when was the merge message signed'
    ,'merge_timestamp'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_purse_merges(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_merges';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_reserve_pub '
    'ON ' || table_name || ' '
    '(reserve_pub);'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_reserve_pub '
    'IS ' || quote_literal('needed in reserve history computation') || ';'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_purse_merge_request_serial_id_key'
    ' UNIQUE (purse_merge_request_serial_id) '
  );
END
$$;
CREATE FUNCTION foreign_table_purse_merges()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_merges';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_partner_serial_id'
    ' FOREIGN KEY (partner_serial_id) '
    ' REFERENCES partners(partner_serial_id) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_purse_pub'
    ' FOREIGN KEY (purse_pub) '
    ' REFERENCES purse_requests (purse_pub) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('purse_merges'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('purse_merges'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('purse_merges'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_account_merges(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'account_merges';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I '
      '(account_merge_request_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL CHECK (LENGTH(reserve_pub)=32)'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
      ',wallet_h_payto BYTEA NOT NULL CHECK (LENGTH(wallet_h_payto)=32)'
      ',PRIMARY KEY (purse_pub)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Merge requests where a purse- and account-owner requested merging the purse into the account'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the target reserve'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'signature by the reserve private key affirming the merge, of type TALER_SIGNATURE_WALLET_ACCOUNT_MERGE'
    ,'reserve_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_account_merges(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'account_merges';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve_pub '
    'ON ' || table_name || ' '
    '(reserve_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_account_merge_request_serial_id_key'
    ' UNIQUE (account_merge_request_serial_id) '
  );
END
$$;
CREATE FUNCTION foreign_table_account_merges()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'account_merges';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub) '
    ' REFERENCES reserves (reserve_pub) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_purse_pub'
    ' FOREIGN KEY (purse_pub) '
    ' REFERENCES purse_requests (purse_pub)'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('account_merges'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('account_merges'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('account_merges'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_purse_decision(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_decision';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(purse_decision_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
      ',action_timestamp INT8 NOT NULL'
      ',refunded BOOL NOT NULL'
      ',PRIMARY KEY (purse_pub)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Purses that were decided upon (refund or merge)'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_purse_decision(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_decision';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_purse_action_serial_id_key'
    ' UNIQUE (purse_decision_serial_id) '
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('purse_decision'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('purse_decision'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_contracts(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'contracts';
BEGIN
  PERFORM create_partitioned_table(
     'CREATE TABLE %I '
     '(contract_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
     ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
     ',pub_ckey BYTEA NOT NULL CHECK (LENGTH(pub_ckey)=32)'
     ',contract_sig BYTEA NOT NULL CHECK (LENGTH(contract_sig)=64)'
     ',e_contract BYTEA NOT NULL'
     ',purse_expiration INT8 NOT NULL'
     ',PRIMARY KEY (purse_pub)'
     ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'encrypted contracts associated with purses'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the purse that the contract is associated with'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'signature over the encrypted contract by the purse contract key'
    ,'contract_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public ECDH key used to encrypt the contract, to be used with the purse private key for decryption'
    ,'pub_ckey'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'AES-GCM encrypted contract terms (contains gzip compressed JSON after decryption)'
    ,'e_contract'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_contracts(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'contracts';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_contract_serial_id_key'
    ' UNIQUE (contract_serial_id) '
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('contracts'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('contracts'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_history_requests(
  IN shard_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'history_requests';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(history_request_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL CHECK (LENGTH(reserve_pub)=32)'
      ',request_timestamp INT8 NOT NULL'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',history_fee_val INT8 NOT NULL'
      ',history_fee_frac INT4 NOT NULL'
      ',PRIMARY KEY (reserve_pub,request_timestamp)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,shard_suffix
  );
  PERFORM comment_partitioned_table(
     'Paid history requests issued by a client against a reserve'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'When was the history request made'
    ,'request_timestamp'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature approving payment for the history request'
    ,'reserve_sig'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'History fee approved by the signature'
    ,'history_fee_val'
    ,table_name
    ,shard_suffix
  );
END $$;
CREATE FUNCTION constrain_table_history_requests(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  partition_name VARCHAR;
BEGIN
  partition_name = concat_ws('_', 'history_requests', partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || partition_name ||
    ' ADD CONSTRAINT ' || partition_name || '_serial_id'
    ' UNIQUE (history_request_serial_id)'
  );
END
$$;
CREATE FUNCTION foreign_table_history_requests()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'history_requests';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub) '
    ' REFERENCES reserves(reserve_pub) ON DELETE CASCADE'
  );
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('history_requests'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('history_requests'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('history_requests'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_purse_deposits(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_deposits';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(purse_deposit_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',partner_serial_id INT8'
      ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
      ',coin_pub BYTEA NOT NULL'
      ',amount_with_fee_val INT8 NOT NULL'
      ',amount_with_fee_frac INT4 NOT NULL'
      ',coin_sig BYTEA NOT NULL CHECK(LENGTH(coin_sig)=64)'
      ',PRIMARY KEY (purse_pub,coin_pub)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Requests depositing coins into a purse'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'identifies the partner exchange, NULL in case the target purse lives at this exchange'
    ,'partner_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the coin being deposited'
    ,'coin_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total amount being deposited'
    ,'amount_with_fee_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature of the coin affirming the deposit into the purse, of type TALER_SIGNATURE_PURSE_DEPOSIT'
    ,'coin_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_purse_deposits(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_deposits';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_coin_pub'
    ' ON ' || table_name || ' (coin_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_purse_deposit_serial_id_key'
    ' UNIQUE (purse_deposit_serial_id) '
  );
END
$$;
CREATE FUNCTION foreign_table_purse_deposits()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_deposits';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_partner'
    ' FOREIGN KEY (partner_serial_id) '
    ' REFERENCES partners(partner_serial_id) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (coin_pub) '
    ' REFERENCES known_coins (coin_pub) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('purse_deposits'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('purse_deposits'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('purse_deposits'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_wads_in(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'wads_in';
BEGIN
  PERFORM create_partitioned_table(
     'CREATE TABLE %I '
     '(wad_in_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
     ',wad_id BYTEA PRIMARY KEY CHECK (LENGTH(wad_id)=24)'
     ',origin_exchange_url TEXT NOT NULL'
     ',amount_val INT8 NOT NULL'
     ',amount_frac INT4 NOT NULL'
     ',arrival_time INT8 NOT NULL'
     ',UNIQUE (wad_id, origin_exchange_url)'
     ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (wad_id)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Incoming exchange-to-exchange wad wire transfers'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Unique identifier of the wad, part of the wire transfer subject'
    ,'wad_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Base URL of the originating URL, also part of the wire transfer subject'
    ,'origin_exchange_url'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Actual amount that was received by our exchange'
    ,'amount_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Time when the wad was received'
    ,'arrival_time'
    ,table_name
    ,partition_suffix
  );
END $$;
CREATE FUNCTION constrain_table_wads_in(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'wads_in';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wad_in_serial_id_key'
    ' UNIQUE (wad_in_serial_id) '
    ',ADD CONSTRAINT ' || table_name || '_wad_is_origin_exchange_url_key'
    ' UNIQUE (wad_id, origin_exchange_url) '
  );
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('wads_in'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('wads_in'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_wad_in_entries(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'wad_in_entries';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(wad_in_entry_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',wad_in_serial_id INT8'
      ',reserve_pub BYTEA NOT NULL CHECK(LENGTH(reserve_pub)=32)'
      ',purse_pub BYTEA PRIMARY KEY CHECK(LENGTH(purse_pub)=32)'
      ',h_contract BYTEA NOT NULL CHECK(LENGTH(h_contract)=64)'
      ',purse_expiration INT8 NOT NULL'
      ',merge_timestamp INT8 NOT NULL'
      ',amount_with_fee_val INT8 NOT NULL'
      ',amount_with_fee_frac INT4 NOT NULL'
      ',wad_fee_val INT8 NOT NULL'
      ',wad_fee_frac INT4 NOT NULL'
      ',deposit_fees_val INT8 NOT NULL'
      ',deposit_fees_frac INT4 NOT NULL'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',purse_sig BYTEA NOT NULL CHECK (LENGTH(purse_sig)=64)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'list of purses aggregated in a wad according to the sending exchange'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'wad for which the given purse was included in the aggregation'
    ,'wad_in_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'target account of the purse (must be at the local exchange)'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the purse that was merged'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'hash of the contract terms of the purse'
    ,'h_contract'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Time when the purse was set to expire'
    ,'purse_expiration'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Time when the merge was approved'
    ,'merge_timestamp'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total amount in the purse'
    ,'amount_with_fee_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total wad fees paid by the purse'
    ,'wad_fee_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total deposit fees paid when depositing coins into the purse'
    ,'deposit_fees_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature by the receiving reserve, of purpose TALER_SIGNATURE_ACCOUNT_MERGE'
    ,'reserve_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature by the purse of purpose TALER_SIGNATURE_PURSE_MERGE'
    ,'purse_sig'
    ,table_name
    ,partition_suffix
  );
END $$;
CREATE FUNCTION constrain_table_wad_in_entries(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'wad_in_entries';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_reserve_pub '
    'ON ' || table_name || ' '
    '(reserve_pub);'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_reserve_pub '
    'IS ' || quote_literal('needed in reserve history computation') || ';'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wad_in_entry_serial_id_key'
    ' UNIQUE (wad_in_entry_serial_id) '
  );
END $$;
CREATE FUNCTION foreign_table_wad_in_entries()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'wad_in_entries';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_wad_in'
    ' FOREIGN KEY(wad_in_serial_id)'
    ' REFERENCES wads_in (wad_in_serial_id) ON DELETE CASCADE'
  );
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('wad_in_entries'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('wad_in_entries'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('wad_in_entries'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_wads_out(
  IN shard_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'wads_out';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(wad_out_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',wad_id BYTEA PRIMARY KEY CHECK (LENGTH(wad_id)=24)'
      ',partner_serial_id INT8 NOT NULL'
      ',amount_val INT8 NOT NULL'
      ',amount_frac INT4 NOT NULL'
      ',execution_time INT8 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (wad_id)'
    ,shard_suffix
  );
  PERFORM comment_partitioned_table(
     'Wire transfers made to another exchange to transfer purse funds'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Unique identifier of the wad, part of the wire transfer subject'
    ,'wad_id'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'target exchange of the wad'
    ,'partner_serial_id'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Amount that was wired'
    ,'amount_val'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Time when the wire transfer was scheduled'
    ,'execution_time'
    ,table_name
    ,shard_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_wads_out(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'wads_out';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wad_out_serial_id_key'
    ' UNIQUE (wad_out_serial_id) '
  );
END
$$;
CREATE FUNCTION foreign_table_wads_out()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'wads_out';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_partner'
    ' FOREIGN KEY(partner_serial_id)'
    ' REFERENCES partners(partner_serial_id) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('wads_out'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('wads_out'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('wads_out'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_wad_out_entries(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'wad_out_entries';
BEGIN
  PERFORM create_partitioned_table(
     'CREATE TABLE %I '
     '(wad_out_entry_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
     ',wad_out_serial_id INT8'
     ',reserve_pub BYTEA NOT NULL CHECK(LENGTH(reserve_pub)=32)'
     ',purse_pub BYTEA PRIMARY KEY CHECK(LENGTH(purse_pub)=32)'
     ',h_contract BYTEA NOT NULL CHECK(LENGTH(h_contract)=64)'
     ',purse_expiration INT8 NOT NULL'
     ',merge_timestamp INT8 NOT NULL'
     ',amount_with_fee_val INT8 NOT NULL'
     ',amount_with_fee_frac INT4 NOT NULL'
     ',wad_fee_val INT8 NOT NULL'
     ',wad_fee_frac INT4 NOT NULL'
     ',deposit_fees_val INT8 NOT NULL'
     ',deposit_fees_frac INT4 NOT NULL'
     ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
     ',purse_sig BYTEA NOT NULL CHECK (LENGTH(purse_sig)=64)'
     ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
    'Purses combined into a wad'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Wad the purse was part of'
    ,'wad_out_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Target reserve for the purse'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Hash of the contract associated with the purse'
    ,'h_contract'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Time when the purse expires'
    ,'purse_expiration'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Time when the merge was approved'
    ,'merge_timestamp'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total amount in the purse'
    ,'amount_with_fee_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'Wad fee charged to the purse'
    ,'wad_fee_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total deposit fees charged to the purse'
    ,'deposit_fees_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature by the receiving reserve, of purpose TALER_SIGNATURE_ACCOUNT_MERGE'
    ,'reserve_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature by the purse of purpose TALER_SIGNATURE_PURSE_MERGE'
    ,'purse_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_wad_out_entries(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'wad_out_entries';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve_pub '
    'ON ' || table_name || ' '
    '(reserve_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wad_out_entry_serial_id_key'
    ' UNIQUE (wad_out_entry_serial_id) '
  );
END
$$;
CREATE FUNCTION foreign_table_wad_out_entries()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'wad_out_entries';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_wad_out'
    ' FOREIGN KEY(wad_out_serial_id)'
    ' REFERENCES wads_out (wad_out_serial_id) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('wad_out_entries'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('wad_out_entries'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('wad_out_entries'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE TABLE policy_fulfillments
  (fulfillment_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE PRIMARY KEY
  ,fulfillment_timestamp INT8 NOT NULL
  ,fulfillment_proof VARCHAR
  ,h_fulfillment_proof BYTEA NOT NULL CHECK(LENGTH(h_fulfillment_proof) = 64) UNIQUE
  ,policy_hash_codes BYTEA NOT NULL CHECK(0 = MOD(LENGTH(policy_hash_codes), 16))
  );
COMMENT ON TABLE policy_fulfillments
  IS 'Proofs of fulfillment of policies that were set in deposits';
COMMENT ON COLUMN policy_fulfillments.fulfillment_timestamp
  IS 'Timestamp of the arrival of a proof of fulfillment';
COMMENT ON COLUMN policy_fulfillments.fulfillment_proof
  IS 'JSON object with a proof of the fulfillment of a policy. Supported details depend on the policy extensions supported by the exchange.';
COMMENT ON COLUMN policy_fulfillments.h_fulfillment_proof
  IS 'Hash of the fulfillment_proof';
COMMENT ON COLUMN policy_fulfillments.policy_hash_codes
  IS 'Concatenation of the policy_hash_code of all policy_details that are fulfilled by this proof';
CREATE TABLE policy_details
  (policy_details_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,policy_hash_code BYTEA PRIMARY KEY CHECK(LENGTH(policy_hash_code)=16)
  ,policy_json VARCHAR
  ,deadline INT8 NOT NULL
  ,commitment_val INT8 NOT NULL
  ,commitment_frac INT4 NOT NULL
  ,accumulated_total_val INT8 NOT NULL
  ,accumulated_total_frac INT4 NOT NULL
  ,fee_val INT8 NOT NULL
  ,fee_frac INT4 NOT NULL
  ,transferable_val INT8 NOT NULL
  ,transferable_frac INT8 NOT NULL
  ,fulfillment_state smallint NOT NULL CHECK(fulfillment_state between 0 and 5)
  ,fulfillment_id BIGINT NULL REFERENCES policy_fulfillments (fulfillment_id) ON DELETE CASCADE
  );
COMMENT ON TABLE policy_details
  IS 'Policies that were provided with deposits via policy extensions.';
COMMENT ON COLUMN policy_details.policy_hash_code
  IS 'ID (GNUNET_HashCode) that identifies a policy.  Will be calculated by the policy extension based on the content';
COMMENT ON COLUMN policy_details.policy_json
  IS 'JSON object with options set that the exchange needs to consider when executing a deposit. Supported details depend on the policy extensions supported by the exchange.';
COMMENT ON COLUMN policy_details.deadline
  IS 'Deadline until the policy must be marked as fulfilled (maybe "forever")';
COMMENT ON COLUMN policy_details.commitment_val
  IS 'The amount that this policy commits to.  Invariant: commitment >= fee';
COMMENT ON COLUMN policy_details.accumulated_total_val
  IS 'The sum of all contributions of all deposit that reference this policy.  Invariant: The fulfilment_state must be Insufficient as long as accumulated_total < commitment';
COMMENT ON COLUMN policy_details.fee_val
  IS 'The fee for this policy, due when the policy is fulfilled or timed out';
COMMENT ON COLUMN policy_details.transferable_val
  IS 'The amount that on fulfillment or timeout will be transferred to the payto-URI''s of the corresponding deposit''s.  The policy fees must have been already deducted from it.  Invariant: fee+transferable <= accumulated_total.  The remaining amount (accumulated_total - fee - transferable) can be refreshed by the owner of the coins when the state is Timeout or Success.';
COMMENT ON COLUMN policy_details.fulfillment_state
  IS 'State of the fulfillment:
       - 0 (Failure)
       - 1 (Insufficient)
       - 2 (Ready)
       - 4 (Success)
       - 5 (Timeout)';
COMMENT ON COLUMN policy_details.fulfillment_id
  IS 'Reference to the proof of the fulfillment of this policy, if it exists.  Invariant: If not NULL, this entry''s .hash_code MUST be part of the corresponding policy_fulfillments.policy_hash_codes array.';
CREATE TABLE work_shards
  (shard_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,last_attempt INT8 NOT NULL
  ,start_row INT8 NOT NULL
  ,end_row INT8 NOT NULL
  ,completed BOOLEAN NOT NULL DEFAULT FALSE
  ,job_name VARCHAR NOT NULL
  ,PRIMARY KEY (job_name, start_row)
  );
COMMENT ON TABLE work_shards
  IS 'coordinates work between multiple processes working on the same job';
COMMENT ON COLUMN work_shards.shard_serial_id
  IS 'unique serial number identifying the shard';
COMMENT ON COLUMN work_shards.last_attempt
  IS 'last time a worker attempted to work on the shard';
COMMENT ON COLUMN work_shards.completed
  IS 'set to TRUE once the shard is finished by a worker';
COMMENT ON COLUMN work_shards.start_row
  IS 'row at which the shard scope starts, inclusive';
COMMENT ON COLUMN work_shards.end_row
  IS 'row at which the shard scope ends, exclusive';
COMMENT ON COLUMN work_shards.job_name
  IS 'unique name of the job the workers on this shard are performing';
CREATE INDEX work_shards_by_job_name_completed_last_attempt_index
  ON work_shards
  (job_name
  ,completed
  ,last_attempt ASC
  );
CREATE INDEX work_shards_by_end_row_index
  ON work_shards
  (end_row DESC);
CREATE INDEX work_shards_by_rows
  ON work_shards
  (job_name
  ,start_row
  ,end_row);
CREATE UNLOGGED TABLE revolving_work_shards
  (shard_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,last_attempt INT8 NOT NULL
  ,start_row INT4 NOT NULL
  ,end_row INT4 NOT NULL
  ,active BOOLEAN NOT NULL DEFAULT FALSE
  ,job_name VARCHAR NOT NULL
  ,PRIMARY KEY (job_name, start_row)
  );
COMMENT ON TABLE revolving_work_shards
  IS 'coordinates work between multiple processes working on the same job with partitions that need to be repeatedly processed; unlogged because on system crashes the locks represented by this table will have to be cleared anyway, typically using "taler-exchange-dbinit -s"';
COMMENT ON COLUMN revolving_work_shards.shard_serial_id
  IS 'unique serial number identifying the shard';
COMMENT ON COLUMN revolving_work_shards.last_attempt
  IS 'last time a worker attempted to work on the shard';
COMMENT ON COLUMN revolving_work_shards.active
  IS 'set to TRUE when a worker is active on the shard';
COMMENT ON COLUMN revolving_work_shards.start_row
  IS 'row at which the shard scope starts, inclusive';
COMMENT ON COLUMN revolving_work_shards.end_row
  IS 'row at which the shard scope ends, exclusive';
COMMENT ON COLUMN revolving_work_shards.job_name
  IS 'unique name of the job the workers on this shard are performing';
CREATE INDEX revolving_work_shards_by_job_name_active_last_attempt_index
  ON revolving_work_shards
  (job_name
  ,active
  ,last_attempt
  );
CREATE TABLE partners
  (partner_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,partner_master_pub BYTEA NOT NULL CHECK(LENGTH(partner_master_pub)=32)
  ,start_date INT8 NOT NULL
  ,end_date INT8 NOT NULL
  ,next_wad INT8 NOT NULL DEFAULT (0)
  ,wad_frequency INT8 NOT NULL
  ,wad_fee_val INT8 NOT NULL
  ,wad_fee_frac INT4 NOT NULL
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  ,partner_base_url TEXT NOT NULL
  );
COMMENT ON TABLE partners
  IS 'exchanges we do wad transfers to';
COMMENT ON COLUMN partners.partner_master_pub
  IS 'offline master public key of the partner';
COMMENT ON COLUMN partners.start_date
  IS 'starting date of the partnership';
COMMENT ON COLUMN partners.end_date
  IS 'end date of the partnership';
COMMENT ON COLUMN partners.next_wad
  IS 'at what time should we do the next wad transfer to this partner (frequently updated); set to forever after the end_date';
COMMENT ON COLUMN partners.wad_frequency
  IS 'how often do we promise to do wad transfers';
COMMENT ON COLUMN partners.wad_fee_val
  IS 'how high is the fee for a wallet to be added to a wad to this partner';
COMMENT ON COLUMN partners.partner_base_url
  IS 'base URL of the REST API for this partner';
COMMENT ON COLUMN partners.master_sig
  IS 'signature of our master public key affirming the partnership, of purpose TALER_SIGNATURE_MASTER_PARTNER_DETAILS';
CREATE INDEX IF NOT EXISTS partner_by_wad_time
  ON partners (next_wad ASC);
CREATE TABLE partner_accounts
  (payto_uri VARCHAR PRIMARY KEY
  ,partner_serial_id INT8 REFERENCES partners(partner_serial_id) ON DELETE CASCADE
  ,partner_master_sig BYTEA CHECK (LENGTH(partner_master_sig)=64)
  ,last_seen INT8 NOT NULL
  );
CREATE INDEX IF NOT EXISTS partner_accounts_index_by_partner_and_time
  ON partner_accounts (partner_serial_id,last_seen);
COMMENT ON TABLE partner_accounts
  IS 'Table with bank accounts of the partner exchange. Entries never expire as we need to remember the signature for the auditor.';
COMMENT ON COLUMN partner_accounts.payto_uri
  IS 'payto URI (RFC 8905) with the bank account of the partner exchange.';
COMMENT ON COLUMN partner_accounts.partner_master_sig
  IS 'Signature of purpose TALER_SIGNATURE_MASTER_WIRE_DETAILS by the partner master public key';
COMMENT ON COLUMN partner_accounts.last_seen
  IS 'Last time we saw this account as being active at the partner exchange. Used to select the most recent entry, and to detect when we should check again.';
COMMIT;
