--
-- This file is part of TALER
-- Copyright (C) 2022 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_withdraw_age_commitments(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'withdraw_age_commitments';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(withdraw_age_commitment_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_commitment BYTEA PRIMARY KEY CHECK (LENGTH(h_commitment)=64)'
      ',amount_with_fee_val INT8 NOT NULL'
      ',amount_with_fee_frac INT4 NOT NULL'
      ',max_age_group INT2 NOT NULL'
      ',reserve_pub BYTEA NOT NULL CHECK (LENGTH(reserve_pub)=32)'
      ',reserve_sig BYTEA CHECK (LENGTH(reserve_sig)=64)'
      ',noreveal_index INT4 NOT NULL'
      ',timestamp INT8 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Commitments made when withdrawing coins with age restriction and the gamma value chosen by the exchange.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'The gamma value chosen by the exchange in the cut-and-choose protocol'
    ,'noreveal_index'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'The maximum age group that the client commits to with this request'
    ,'max_age_group'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Commitment made by the client, hash over the various client inputs in the cut-and-choose protocol'
    ,'h_commitment'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Reference to the public key of the reserve from which the coins are going to be withdrawn'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature of the reserve''s private key over the withdraw-age request'
    ,'reserve_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Timestamp with the time when the withdraw-age request was received by the exchange'
    ,'timestamp'
    ,table_name
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_withdraw_age_commitments(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'withdraw_age_commitments';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);

  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD PRIMARY KEY  (h_commitment, reserve_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_withdraw_age_commitment_id_key'
    ' UNIQUE (withdraw_age_commitment_id);'
  );
END
$$;


CREATE FUNCTION foreign_table_withdraw_age_commitments()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'withdraw_age_commitments';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub)'
    ' REFERENCES reserves (reserve_pub) ON DELETE CASCADE;'
  );
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('withdraw_age_commitments'
    ,'exchange-0003'
    ,'create'
    ,TRUE
    ,FALSE),
    ('withdraw_age_commitments'
    ,'exchange-0003'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('withdraw_age_commitments'
    ,'exchange-0003'
    ,'foreign'
    ,TRUE
    ,FALSE);
