--
-- This file is part of TALER
-- Copyright (C) 2014--2022 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_reserves_open_requests(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_open_requests';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(open_request_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL'
      ',request_timestamp INT8 NOT NULL'
      ',expiration_date INT8 NOT NULL'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',reserve_payment_val INT8 NOT NULL'
      ',reserve_payment_frac INT4 NOT NULL'
      ',requested_purse_limit INT4 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table (
     'requests to keep a reserve open'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column (
     'Fee to pay for the request from the reserve balance itself.'
    ,'reserve_payment_val'
    ,table_name
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_reserves_open_requests(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_open_requests';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_by_uuid'
    ' PRIMARY KEY (open_request_uuid)'
    ',ADD CONSTRAINT ' || table_name || '_by_time'
    ' UNIQUE (reserve_pub,request_timestamp)'
  );
END
$$;


CREATE FUNCTION foreign_table_reserves_open_requests()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR default 'reserves_open_requests';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub '
    ' FOREIGN KEY (reserve_pub)'
    ' REFERENCES reserves (reserve_pub) ON DELETE CASCADE'
  );
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves_open_requests'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_open_requests'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserves_open_requests'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
