--
-- This file is part of TALER
-- Copyright (C) 2014--2022 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_refresh_commitments(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refresh_commitments';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(melt_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',rc BYTEA PRIMARY KEY CHECK (LENGTH(rc)=64)'
      ',old_coin_pub BYTEA NOT NULL'
      ',old_coin_sig BYTEA NOT NULL CHECK(LENGTH(old_coin_sig)=64)'
      ',amount_with_fee_val INT8 NOT NULL'
      ',amount_with_fee_frac INT4 NOT NULL'
      ',noreveal_index INT4 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (rc)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Commitments made when melting coins and the gamma value chosen by the exchange.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'The gamma value chosen by the exchange in the cut-and-choose protocol'
    ,'noreveal_index'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Commitment made by the client, hash over the various client inputs in the cut-and-choose protocol'
    ,'rc'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Coin being melted in the refresh process.'
    ,'old_coin_pub'
    ,table_name
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_refresh_commitments(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refresh_commitments';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);

  -- Note: index spans partitions, may need to be materialized.
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_old_coin_pub_index '
    'ON ' || table_name || ' '
    '(old_coin_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_melt_serial_id_key'
    ' UNIQUE (melt_serial_id)'
  );
END
$$;


CREATE FUNCTION foreign_table_refresh_commitments()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'refresh_commitments';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (old_coin_pub) '
    ' REFERENCES known_coins (coin_pub) ON DELETE CASCADE'
  );
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('refresh_commitments'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('refresh_commitments'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('refresh_commitments'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
