--
-- This file is part of TALER
-- Copyright (C) 2014--2022 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_purse_requests(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_requests';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(purse_requests_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
      ',merge_pub BYTEA NOT NULL CHECK (LENGTH(merge_pub)=32)'
      ',purse_creation INT8 NOT NULL'
      ',purse_expiration INT8 NOT NULL'
      ',h_contract_terms BYTEA NOT NULL CHECK (LENGTH(h_contract_terms)=64)'
      ',age_limit INT4 NOT NULL'
      ',flags INT4 NOT NULL'
      ',in_reserve_quota BOOLEAN NOT NULL DEFAULT(FALSE)'
      ',amount_with_fee_val INT8 NOT NULL'
      ',amount_with_fee_frac INT4 NOT NULL'
      ',purse_fee_val INT8 NOT NULL'
      ',purse_fee_frac INT4 NOT NULL'
      ',balance_val INT8 NOT NULL DEFAULT (0)'
      ',balance_frac INT4 NOT NULL DEFAULT (0)'
      ',purse_sig BYTEA NOT NULL CHECK(LENGTH(purse_sig)=64)'
      ',PRIMARY KEY (purse_pub)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Requests establishing purses, associating them with a contract but without a target reserve'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Local time when the purse was created. Determines applicable purse fees.'
    ,'purse_creation'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'When the purse is set to expire'
    ,'purse_expiration'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Hash of the contract the parties are to agree to'
    ,'h_contract_terms'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'see the enum TALER_WalletAccountMergeFlags'
    ,'flags'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'set to TRUE if this purse currently counts against the number of free purses in the respective reserve'
    ,'in_reserve_quota'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total amount expected to be in the purse'
    ,'amount_with_fee_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Purse fee the client agreed to pay from the reserve (accepted by the exchange at the time the purse was created). Zero if in_reserve_quota is TRUE.'
    ,'purse_fee_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'Total amount actually in the purse (updated)'
    ,'balance_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature of the purse affirming the purse parameters, of type TALER_SIGNATURE_PURSE_REQUEST'
    ,'purse_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;

CREATE FUNCTION constrain_table_purse_requests(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'purse_requests';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);

  -- FIXME: change to materialized index by merge_pub!
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_merge_pub '
    'ON ' || table_name || ' '
    '(merge_pub);'
  );
  -- FIXME: drop index on master (crosses partitions)?
  -- Or use materialized index? (needed?)
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_purse_expiration '
    'ON ' || table_name || ' '
    '(purse_expiration);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_purse_requests_serial_id_key'
    ' UNIQUE (purse_requests_serial_id) '
  );
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('purse_requests'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('purse_requests'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
