--
-- This file is part of TALER
-- Copyright (C) 2014--2022 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--


CREATE FUNCTION create_table_history_requests(
  IN shard_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'history_requests';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(history_request_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL CHECK (LENGTH(reserve_pub)=32)'
      ',request_timestamp INT8 NOT NULL'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',history_fee_val INT8 NOT NULL'
      ',history_fee_frac INT4 NOT NULL'
      ',PRIMARY KEY (reserve_pub,request_timestamp)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,shard_suffix
  );
  PERFORM comment_partitioned_table(
     'Paid history requests issued by a client against a reserve'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'When was the history request made'
    ,'request_timestamp'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature approving payment for the history request'
    ,'reserve_sig'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'History fee approved by the signature'
    ,'history_fee_val'
    ,table_name
    ,shard_suffix
  );
END $$;


CREATE FUNCTION constrain_table_history_requests(
  IN partition_suffix VARCHAR
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  partition_name VARCHAR;
BEGIN
  partition_name = concat_ws('_', 'history_requests', partition_suffix);

  EXECUTE FORMAT (
    'ALTER TABLE ' || partition_name ||
    ' ADD CONSTRAINT ' || partition_name || '_serial_id'
    ' UNIQUE (history_request_serial_id)'
  );
END
$$;


CREATE FUNCTION foreign_table_history_requests()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'history_requests';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub) '
    ' REFERENCES reserves(reserve_pub) ON DELETE CASCADE'
  );
END $$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('history_requests'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('history_requests'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('history_requests'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
