--
-- This file is part of TALER
-- Copyright (C) 2014--2022 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_close_requests(
  IN partition_suffix VARCHAR DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'close_requests';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(close_request_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL CHECK (LENGTH(reserve_pub)=32)'
      ',close_timestamp INT8 NOT NULL'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',close_val INT8 NOT NULL'
      ',close_frac INT4 NOT NULL'
      ',close_fee_val INT8 NOT NULL'
      ',close_fee_frac INT4 NOT NULL'
      ',payto_uri VARCHAR NOT NULL'
      ',done BOOL NOT NULL DEFAULT(FALSE)'
      ',PRIMARY KEY (reserve_pub,close_timestamp)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Explicit requests by a reserve owner to close a reserve immediately'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'When the request was created by the client'
    ,'close_timestamp'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature affirming that the reserve is to be closed'
    ,'reserve_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Balance of the reserve at the time of closing, to be wired to the associated bank account (minus the closing fee)'
    ,'close_val'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the credited bank account. Optional.'
    ,'payto_uri'
    ,table_name
    ,partition_suffix
  );
END $$;


CREATE FUNCTION constrain_table_close_requests(
  IN partition_suffix VARCHAR
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'close_requests';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_close_request_uuid_index '
    'ON ' || table_name || ' '
    '(close_request_serial_id);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_close_request_done_index '
    'ON ' || table_name || ' '
    '(done);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_close_request_uuid_pkey'
    ' UNIQUE (close_request_serial_id)'
  );
END
$$;


CREATE FUNCTION foreign_table_close_requests()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name VARCHAR DEFAULT 'close_requests';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub) '
    ' REFERENCES reserves(reserve_pub) ON DELETE CASCADE'
  );
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('close_requests'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('close_requests'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('close_requests'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
