/*
  This file is part of TALER
  Copyright (C) 2014-2022 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more details.

  You should have received a copy of the GNU Affero General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file taler-exchange-httpd_reserves_attest.c
 * @brief Handle /reserves/$RESERVE_PUB/attest requests
 * @author Florian Dold
 * @author Benedikt Mueller
 * @author Christian Grothoff
 */
#include "platform.h"
#include <gnunet/gnunet_util_lib.h>
#include <jansson.h>
#include "taler_dbevents.h"
#include "taler_kyclogic_lib.h"
#include "taler_json_lib.h"
#include "taler_mhd_lib.h"
#include "taler-exchange-httpd_keys.h"
#include "taler-exchange-httpd_reserves_attest.h"
#include "taler-exchange-httpd_responses.h"


/**
 * How far do we allow a client's time to be off when
 * checking the request timestamp?
 */
#define TIMESTAMP_TOLERANCE \
  GNUNET_TIME_relative_multiply (GNUNET_TIME_UNIT_MINUTES, 15)


/**
 * Closure for #reserve_attest_transaction.
 */
struct ReserveAttestContext
{
  /**
   * Public key of the reserve the inquiry is about.
   */
  struct TALER_ReservePublicKeyP reserve_pub;

  /**
   * Hash of the payto URI of this reserve.
   */
  struct TALER_PaytoHashP h_payto;

  /**
   * Timestamp of the request.
   */
  struct GNUNET_TIME_Timestamp timestamp;

  /**
   * Expiration time for the attestation.
   */
  struct GNUNET_TIME_Timestamp etime;

  /**
   * List of requested details.
   */
  json_t *details;

  /**
   * Client signature approving the request.
   */
  struct TALER_ReserveSignatureP reserve_sig;

  /**
   * Attributes we are affirming.
   */
  json_t *json_attest;

  /**
   * Error code encountered in interaction with KYC provider.
   */
  enum TALER_ErrorCode ec;

  /**
   * Set to true if we did not find the reserve.
   */
  bool not_found;

};


/**
 * Send reserve attest to client.
 *
 * @param connection connection to the client
 * @param rhc reserve attest to return
 * @return MHD result code
 */
static MHD_RESULT
reply_reserve_attest_success (struct MHD_Connection *connection,
                              const struct ReserveAttestContext *rhc)
{
  struct TALER_ExchangeSignatureP exchange_sig;
  struct TALER_ExchangePublicKeyP exchange_pub;
  enum TALER_ErrorCode ec;
  struct GNUNET_TIME_Timestamp now;

  if (NULL == rhc->json_attest)
  {
    GNUNET_break (0);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_INTERNAL_SERVER_ERROR,
                                       TALER_EC_GENERIC_JSON_ALLOCATION_FAILURE,
                                       NULL);
  }
  now = GNUNET_TIME_timestamp_get ();
  ec = TALER_exchange_online_reserve_attest_details_sign (
    &TEH_keys_exchange_sign_,
    now,
    rhc->etime,
    &rhc->reserve_pub,
    rhc->json_attest,
    &exchange_pub,
    &exchange_sig);
  if (TALER_EC_NONE != ec)
  {
    GNUNET_break (0);
    return TALER_MHD_reply_with_ec (connection,
                                    ec,
                                    NULL);
  }
  return TALER_MHD_REPLY_JSON_PACK (
    connection,
    MHD_HTTP_OK,
    GNUNET_JSON_pack_data_auto ("exchange_sig",
                                &exchange_sig),
    GNUNET_JSON_pack_data_auto ("exchange_pub",
                                &exchange_pub),
    GNUNET_JSON_pack_array_steal ("attest",
                                  rhc->json_attest));
}


/**
 * Function called with information about all applicable
 * legitimization processes for the given user.  Finds the
 * available attributes and merges them into our result
 * set based on the details requested by the client.
 *
 * @param cls our `struct ReserveAttestContext *`
 * @param provider_section KYC provider configuration section
 * @param provider_user_id UID at a provider (can be NULL)
 * @param legi_id legitimization process ID (can be NULL)
 */
static void
kyc_process_cb (void *cls,
                const char *provider_section,
                const char *provider_user_id,
                const char *legi_id)
{
  struct ReserveAttestContext *rsc = cls;
  struct GNUNET_TIME_Timestamp etime;
  json_t *attrs;
  bool match = false;

  rsc->ec = TALER_KYCLOGIC_user_to_attributes (provider_section,
                                               provider_user_id,
                                               legi_id,
                                               &etime,
                                               &attrs);
  if (TALER_EC_NONE != rsc->ec)
    return;
  if (GNUNET_TIME_absolute_is_past (etime.abs_time))
  {
    json_decref (attrs);
    return;
  }
  {
    json_t *val;
    const char *name;

    json_object_foreach (attrs, name, val)
    {
      bool requested = false;
      size_t idx;
      json_t *str;

      if (NULL != json_object_get (rsc->json_attest,
                                   name))
        continue; /* duplicate */
      json_array_foreach (rsc->details, idx, str)
      {
        if (0 == strcmp (json_string_value (str),
                         name))
        {
          requested = true;
          break;
        }
      }
      if (! requested)
        continue;
      match = true;
      GNUNET_assert (0 ==
                     json_object_set (rsc->json_attest, /* NOT set_new! */
                                      name,
                                      val));
    }
  }
  json_decref (attrs);
  if (! match)
    return;
  rsc->etime = GNUNET_TIME_timestamp_min (etime,
                                          rsc->etime);
}


/**
 * Function implementing /reserves/$RID/attest transaction.  Given the public
 * key of a reserve, return the associated transaction attest.  Runs the
 * transaction logic; IF it returns a non-error code, the transaction logic
 * MUST NOT queue a MHD response.  IF it returns an hard error, the
 * transaction logic MUST queue a MHD response and set @a mhd_ret.  IF it
 * returns the soft error code, the function MAY be called again to retry and
 * MUST not queue a MHD response.
 *
 * @param cls a `struct ReserveAttestContext *`
 * @param connection MHD request which triggered the transaction
 * @param[out] mhd_ret set to MHD response status for @a connection,
 *             if transaction failed (!); unused
 * @return transaction status
 */
static enum GNUNET_DB_QueryStatus
reserve_attest_transaction (void *cls,
                            struct MHD_Connection *connection,
                            MHD_RESULT *mhd_ret)
{
  struct ReserveAttestContext *rsc = cls;
  enum GNUNET_DB_QueryStatus qs;

  rsc->json_attest = json_array ();
  GNUNET_assert (NULL != rsc->json_attest);
  qs = TEH_plugin->iterate_kyc_reference (TEH_plugin->cls,
                                          &rsc->h_payto,
                                          &kyc_process_cb,
                                          rsc);
  switch (qs)
  {
  case GNUNET_DB_STATUS_HARD_ERROR:
    GNUNET_break (0);
    *mhd_ret
      = TALER_MHD_reply_with_error (connection,
                                    MHD_HTTP_INTERNAL_SERVER_ERROR,
                                    TALER_EC_GENERIC_DB_FETCH_FAILED,
                                    "iterate_kyc_reference");
    return qs;
  case GNUNET_DB_STATUS_SOFT_ERROR:
    GNUNET_break (0);
    return qs;
  case GNUNET_DB_STATUS_SUCCESS_NO_RESULTS:
    rsc->not_found = true;
    return qs;
  case GNUNET_DB_STATUS_SUCCESS_ONE_RESULT:
    rsc->not_found = false;
    break;
  }
  return qs;
}


MHD_RESULT
TEH_handler_reserves_attest (struct TEH_RequestContext *rc,
                             const json_t *root,
                             const char *const args[1])
{
  struct ReserveAttestContext rsc = {
    .etime = GNUNET_TIME_UNIT_FOREVER_TS
  };
  MHD_RESULT mhd_ret;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_timestamp ("request_timestamp",
                                &rsc.timestamp),
    GNUNET_JSON_spec_json ("details",
                           &rsc.details),
    GNUNET_JSON_spec_fixed_auto ("reserve_sig",
                                 &rsc.reserve_sig),
    GNUNET_JSON_spec_end ()
  };
  struct GNUNET_TIME_Timestamp now;

  if (GNUNET_OK !=
      GNUNET_STRINGS_string_to_data (args[0],
                                     strlen (args[0]),
                                     &rsc.reserve_pub,
                                     sizeof (rsc.reserve_pub)))
  {
    GNUNET_break_op (0);
    return TALER_MHD_reply_with_error (rc->connection,
                                       MHD_HTTP_BAD_REQUEST,
                                       TALER_EC_GENERIC_RESERVE_PUB_MALFORMED,
                                       args[0]);
  }
  {
    enum GNUNET_GenericReturnValue res;

    res = TALER_MHD_parse_json_data (rc->connection,
                                     root,
                                     spec);
    if (GNUNET_SYSERR == res)
    {
      GNUNET_break (0);
      return MHD_NO; /* hard failure */
    }
    if (GNUNET_NO == res)
    {
      GNUNET_break_op (0);
      return MHD_YES; /* failure */
    }
  }
  now = GNUNET_TIME_timestamp_get ();
  if (! GNUNET_TIME_absolute_approx_eq (now.abs_time,
                                        rsc.timestamp.abs_time,
                                        TIMESTAMP_TOLERANCE))
  {
    GNUNET_break_op (0);
    return TALER_MHD_reply_with_error (rc->connection,
                                       MHD_HTTP_BAD_REQUEST,
                                       TALER_EC_EXCHANGE_GENERIC_CLOCK_SKEW,
                                       NULL);
  }

  if (GNUNET_OK !=
      TALER_wallet_reserve_attest_request_verify (rsc.timestamp,
                                                  rsc.details,
                                                  &rsc.reserve_pub,
                                                  &rsc.reserve_sig))
  {
    GNUNET_break_op (0);
    return TALER_MHD_reply_with_error (rc->connection,
                                       MHD_HTTP_FORBIDDEN,
                                       TALER_EC_EXCHANGE_RESERVES_ATTEST_BAD_SIGNATURE,
                                       NULL);
  }

  {
    char *payto_uri;

    payto_uri = TALER_reserve_make_payto (TEH_base_url,
                                          &rsc.reserve_pub);
    TALER_payto_hash (payto_uri,
                      &rsc.h_payto);
    GNUNET_free (payto_uri);
  }

  if (GNUNET_OK !=
      TEH_DB_run_transaction (rc->connection,
                              "post reserve attest",
                              TEH_MT_REQUEST_OTHER,
                              &mhd_ret,
                              &reserve_attest_transaction,
                              &rsc))
  {
    return mhd_ret;
  }
  if (rsc.not_found)
  {
    json_decref (rsc.json_attest);
    return TALER_MHD_reply_with_error (rc->connection,
                                       MHD_HTTP_NOT_FOUND,
                                       TALER_EC_EXCHANGE_GENERIC_RESERVE_UNKNOWN,
                                       args[0]);
  }
  if (TALER_EC_NONE != rsc.ec)
  {
    json_decref (rsc.json_attest);
    return TALER_MHD_reply_with_ec (rc->connection,
                                    rsc.ec,
                                    NULL);
  }
  mhd_ret = reply_reserve_attest_success (rc->connection,
                                          &rsc);
  return mhd_ret;
}


/* end of taler-exchange-httpd_reserves_attest.c */
