/*
     This file is part of GNU Taler
     Copyright (C) 2012-2020 Taler Systems SA

     GNU Taler is free software: you can redistribute it and/or modify it
     under the terms of the GNU Lesser General Public License as published
     by the Free Software Foundation, either version 3 of the License,
     or (at your option) any later version.

     GNU Taler is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     Lesser General Public License for more details.

     You should have received a copy of the GNU Lesser General Public License
     along with this program.  If not, see <http://www.gnu.org/licenses/>.

     SPDX-License-Identifier: LGPL3.0-or-later

     Note: the LGPL does not apply to all components of GNU Taler,
     but it does apply to this file.
 */
#include "taler_error_codes.h"
#include <stddef.h>


/**
 * A pair containing an error code and its hint.
 */
struct ErrorCodeAndHint
{
  /**
   * The error code.
   */
  enum TALER_ErrorCode ec;

  /**
   * The hint.
   */
  const char *hint;
};


/**
 * The list of all error codes with their hints.
 */
static const struct ErrorCodeAndHint code_hint_pairs[] = {

  {
    .ec = TALER_EC_NONE,
    .hint = "Special code to indicate no error (or no \"code\" present)."
  },
  {
    .ec = TALER_EC_INVALID,
    .hint = "Special code to indicate that a non-integer error code was returned in the JSON response."
  },
  {
    .ec = TALER_EC_INVALID_RESPONSE,
    .hint = "The response we got from the server was not even in JSON format."
  },
  {
    .ec = TALER_EC_NOT_IMPLEMENTED,
    .hint = "Generic implementation error: this function was not yet implemented."
  },
  {
    .ec = TALER_EC_EXCHANGE_BAD_CONFIGURATION,
    .hint = "Exchange is badly configured and thus cannot operate."
  },
  {
    .ec = TALER_EC_INTERNAL_INVARIANT_FAILURE,
    .hint = "Internal assertion error."
  },
  {
    .ec = TALER_EC_TIMEOUT,
    .hint = "Operation timed out."
  },
  {
    .ec = TALER_EC_JSON_ALLOCATION_FAILURE,
    .hint = "Exchange failed to allocate memory for building JSON reply."
  },
  {
    .ec = TALER_EC_METHOD_INVALID,
    .hint = "HTTP method invalid for this URL."
  },
  {
    .ec = TALER_EC_OPERATION_INVALID,
    .hint = "Operation specified invalid for this endpoint."
  },
  {
    .ec = TALER_EC_ENDPOINT_UNKNOWN,
    .hint = "There is no endpoint defined for the URL provided by the client."
  },
  {
    .ec = TALER_EC_URI_TOO_LONG,
    .hint = "The URI is longer than the longest URI the HTTP server is willing to parse."
  },
  {
    .ec = TALER_EC_WRONG_NUMBER_OF_SEGMENTS,
    .hint = "The number of segments included in the URI does not match the number of segments expected by the endpoint."
  },
  {
    .ec = TALER_EC_HOLE_IN_WIRE_FEE_STRUCTURE,
    .hint = "The start and end-times in the wire fee structure leave a hole. This is not allowed. Generated as an error on the client-side."
  },
  {
    .ec = TALER_EC_VERSION_MALFORMED,
    .hint = "The version string given does not follow the expected CURRENT:REVISION:AGE Format.  Generated as an error on the client side."
  },
  {
    .ec = TALER_EC_CLIENT_INTERNAL_FAILURE,
    .hint = "The client-side experienced an internal failure. Generated as an error on the client side."
  },
  {
    .ec = TALER_EC_UPLOAD_EXCEEDS_LIMIT,
    .hint = "The body is too large to be permissible for the endpoint."
  },
  {
    .ec = TALER_EC_PAYTO_MALFORMED,
    .hint = "The payto:// URI we got is malformed."
  },
  {
    .ec = TALER_EC_OPERATION_UNKNOWN,
    .hint = "Operation specified unknown for this endpoint."
  },
  {
    .ec = TALER_EC_ALLOCATION_FAILURE,
    .hint = "Exchange failed to allocate memory."
  },
  {
    .ec = TALER_EC_DB_SETUP_FAILED,
    .hint = "The exchange failed to even just initialize its connection to the database."
  },
  {
    .ec = TALER_EC_DB_START_FAILED,
    .hint = "The exchange encountered an error event to just start the database transaction."
  },
  {
    .ec = TALER_EC_DB_COMMIT_FAILED_HARD,
    .hint = "The exchange encountered an error event to commit the database transaction (hard, unrecoverable error)."
  },
  {
    .ec = TALER_EC_DB_COMMIT_FAILED_ON_RETRY,
    .hint = "The exchange encountered an error event to commit the database transaction, even after repeatedly retrying it there was always a conflicting transaction. (This indicates a repeated serialization error; should only happen if some client maliciously tries to create conflicting concurrent transactions.)"
  },
  {
    .ec = TALER_EC_PARSER_OUT_OF_MEMORY,
    .hint = "The exchange had insufficient memory to parse the request."
  },
  {
    .ec = TALER_EC_JSON_INVALID,
    .hint = "The JSON in the client's request to the exchange was malformed. (Generic parse error)."
  },
  {
    .ec = TALER_EC_JSON_INVALID_WITH_DETAILS,
    .hint = "The JSON in the client's request to the exchange was malformed. Details about the location of the parse error are provided."
  },
  {
    .ec = TALER_EC_PARAMETER_MISSING,
    .hint = "A required parameter in the request to the exchange was missing."
  },
  {
    .ec = TALER_EC_PARAMETER_MALFORMED,
    .hint = "A parameter in the request to the exchange was malformed."
  },
  {
    .ec = TALER_EC_COIN_HISTORY_DB_ERROR_INSUFFICIENT_FUNDS,
    .hint = "The exchange failed to obtain the transaction history of the given coin from the database while generating an insufficient funds errors. This can happen during /deposit or /recoup requests."
  },
  {
    .ec = TALER_EC_INTERNAL_LOGIC_ERROR,
    .hint = "Internal logic error.  Some server-side function failed that really should not."
  },
  {
    .ec = TALER_EC_PAYTO_WRONG_METHOD,
    .hint = "The method specified in a payto:// URI is not one we expected."
  },
  {
    .ec = TALER_EC_COIN_CONFLICTING_DENOMINATION_KEY,
    .hint = "The same coin was already used with a different denomination previously."
  },
  {
    .ec = TALER_EC_DB_COIN_HISTORY_STORE_ERROR,
    .hint = "We failed to update the database of known coins."
  },
  {
    .ec = TALER_EC_COINS_INVALID_COIN_PUB,
    .hint = "The public key of given to a /coins/ handler was malformed."
  },
  {
    .ec = TALER_EC_RESERVES_INVALID_RESERVE_PUB,
    .hint = "The reserve key of given to a /reserves/ handler was malformed."
  },
  {
    .ec = TALER_EC_TRANSFERS_INVALID_WTID,
    .hint = "The public key of given to a /transfers/ handler was malformed."
  },
  {
    .ec = TALER_EC_DEPOSITS_INVALID_H_WIRE,
    .hint = "The wire hash of given to a /deposits/ handler was malformed."
  },
  {
    .ec = TALER_EC_DEPOSITS_INVALID_MERCHANT_PUB,
    .hint = "The merchant key of given to a /deposits/ handler was malformed."
  },
  {
    .ec = TALER_EC_DEPOSITS_INVALID_H_CONTRACT_TERMS,
    .hint = "The hash of the contract terms given to a /deposits/ handler was malformed."
  },
  {
    .ec = TALER_EC_DEPOSITS_INVALID_COIN_PUB,
    .hint = "The coin public key of given to a /deposits/ handler was malformed."
  },
  {
    .ec = TALER_EC_DEPOSITS_INVALID_BODY_BY_EXCHANGE,
    .hint = "The body returned by the exchange for a /deposits/ request was malformed. Error created client-side."
  },
  {
    .ec = TALER_EC_DEPOSITS_INVALID_SIGNATURE_BY_EXCHANGE,
    .hint = "The signature returned by the exchange in a /deposits/ request was malformed. Error created client-side."
  },
  {
    .ec = TALER_EC_WITHDRAW_INSUFFICIENT_FUNDS,
    .hint = "The given reserve does not have sufficient funds to admit the requested withdraw operation at this time.  The response includes the current \"balance\" of the reserve as well as the transaction \"history\" that lead to this balance."
  },
  {
    .ec = TALER_EC_WITHDRAW_RESERVE_UNKNOWN,
    .hint = "The exchange has no information about the \"reserve_pub\" that was given."
  },
  {
    .ec = TALER_EC_WITHDRAW_AMOUNT_FEE_OVERFLOW,
    .hint = "The amount to withdraw together with the fee exceeds the numeric range for Taler amounts.  This is not a client failure, as the coin value and fees come from the exchange's configuration."
  },
  {
    .ec = TALER_EC_AMOUNT_DEPOSITS_OVERFLOW,
    .hint = "All of the deposited amounts into this reserve total up to a value that is too big for the numeric range for Taler amounts. This is not a client failure, as the transaction history comes from the exchange's configuration."
  },
  {
    .ec = TALER_EC_WITHDRAW_HISTORIC_DENOMINATION_KEY_NOT_FOUND,
    .hint = "For one of the historic withdrawals from this reserve, the exchange could not find the denomination key. This is not a client failure, as the transaction history comes from the exchange's configuration."
  },
  {
    .ec = TALER_EC_WITHDRAW_AMOUNT_WITHDRAWALS_OVERFLOW,
    .hint = "All of the withdrawals from reserve total up to a value that is too big for the numeric range for Taler amounts. This is not a client failure, as the transaction history comes from the exchange's configuration."
  },
  {
    .ec = TALER_EC_WITHDRAW_RESERVE_WITHOUT_WIRE_TRANSFER,
    .hint = "The exchange somehow knows about this reserve, but there seem to have been no wire transfers made.  This is not a client failure, as this is a database consistency issue of the exchange."
  },
  {
    .ec = TALER_EC_WITHDRAW_SIGNATURE_FAILED,
    .hint = "The exchange failed to create the signature using the denomination key."
  },
  {
    .ec = TALER_EC_WITHDRAW_DB_STORE_ERROR,
    .hint = "The exchange failed to store the withdraw operation in its database."
  },
  {
    .ec = TALER_EC_WITHDRAW_DB_FETCH_ERROR,
    .hint = "The exchange failed to check against historic withdraw data from database (as part of ensuring the idempotency of the operation)."
  },
  {
    .ec = TALER_EC_WITHDRAW_DENOMINATION_KEY_NOT_FOUND,
    .hint = "The exchange is not aware of the denomination key the wallet requested for the withdrawal."
  },
  {
    .ec = TALER_EC_WITHDRAW_RESERVE_SIGNATURE_INVALID,
    .hint = "The signature of the reserve is not valid."
  },
  {
    .ec = TALER_EC_WITHDRAW_HISTORY_DB_ERROR_INSUFFICIENT_FUNDS,
    .hint = "When computing the reserve history, we ended up with a negative overall balance, which should be impossible."
  },
  {
    .ec = TALER_EC_WITHDRAW_RESERVE_HISTORY_IMPOSSIBLE,
    .hint = "When computing the reserve history, we ended up with a negative overall balance, which should be impossible."
  },
  {
    .ec = TALER_EC_WITHDRAW_VALIDITY_IN_FUTURE,
    .hint = "Validity period of the coin to be withdrawn is in the future."
  },
  {
    .ec = TALER_EC_WITHDRAW_VALIDITY_IN_PAST,
    .hint = "Withdraw period of the coin to be withdrawn is in the past."
  },
  {
    .ec = TALER_EC_DENOMINATION_KEY_LOST,
    .hint = "Withdraw period of the coin to be withdrawn is in the past."
  },
  {
    .ec = TALER_EC_WITHDRAW_RESERVE_BALANCE_CORRUPT,
    .hint = "The exchange's database entry with the reserve balance summary is inconsistent with its own history of the reserve."
  },
  {
    .ec = TALER_EC_WITHDRAW_REPLY_MALFORMED,
    .hint = "The exchange responded with a reply that did not satsify the protocol. This error is not used in the protocol but created client-side."
  },
  {
    .ec = TALER_EC_WITHDRAW_UNBLIND_FAILURE,
    .hint = "The client failed to unblind the blind signature. This error is not used in the protocol but created client-side."
  },
  {
    .ec = TALER_EC_RESERVE_STATUS_DB_ERROR,
    .hint = "The exchange failed to obtain the transaction history of the given reserve from the database."
  },
  {
    .ec = TALER_EC_RESERVE_STATUS_UNKNOWN,
    .hint = "The reserve status was requested using a unknown key, to be returned with 404 Not Found."
  },
  {
    .ec = TALER_EC_RESERVE_STATUS_REPLY_MALFORMED,
    .hint = "The exchange responded with a reply that did not satsify the protocol. This error is not used in the protocol but created client-side."
  },
  {
    .ec = TALER_EC_DEPOSIT_INSUFFICIENT_FUNDS,
    .hint = "The respective coin did not have sufficient residual value for the /deposit operation (i.e. due to double spending). The \"history\" in the response provides the transaction history of the coin proving this fact."
  },
  {
    .ec = TALER_EC_DEPOSIT_HISTORY_DB_ERROR,
    .hint = "The exchange failed to obtain the transaction history of the given coin from the database (this does not happen merely because the coin is seen by the exchange for the first time)."
  },
  {
    .ec = TALER_EC_DEPOSIT_STORE_DB_ERROR,
    .hint = "The exchange failed to store the /depost information in the database."
  },
  {
    .ec = TALER_EC_DEPOSIT_DB_DENOMINATION_KEY_UNKNOWN,
    .hint = "The exchange database is unaware of the denomination key that signed the coin (however, the exchange process is; this is not supposed to happen; it can happen if someone decides to purge the DB behind the back of the exchange process).  Hence the deposit is being refused."
  },
  {
    .ec = TALER_EC_DEPOSIT_DENOMINATION_KEY_UNKNOWN,
    .hint = "The exchange was trying to lookup the denomination key for the purpose of a DEPOSIT operation. However, the denomination key is unavailable for that purpose. This can be because it is entirely unknown to the exchange or not in the validity period for the deposit operation. Hence the deposit is being refused."
  },
  {
    .ec = TALER_EC_DEPOSIT_COIN_SIGNATURE_INVALID,
    .hint = "The signature made by the coin over the deposit permission is not valid."
  },
  {
    .ec = TALER_EC_DEPOSIT_DENOMINATION_SIGNATURE_INVALID,
    .hint = "The signature of the denomination key over the coin is not valid."
  },
  {
    .ec = TALER_EC_DEPOSIT_NEGATIVE_VALUE_AFTER_FEE,
    .hint = "The stated value of the coin after the deposit fee is subtracted would be negative."
  },
  {
    .ec = TALER_EC_DEPOSIT_REFUND_DEADLINE_AFTER_WIRE_DEADLINE,
    .hint = "The stated refund deadline is after the wire deadline."
  },
  {
    .ec = TALER_EC_DEPOSIT_INVALID_WIRE_FORMAT_TYPE,
    .hint = "The exchange does not recognize the validity of or support the given wire format type."
  },
  {
    .ec = TALER_EC_DEPOSIT_INVALID_WIRE_FORMAT_JSON,
    .hint = "The exchange failed to canonicalize and hash the given wire format. For example, the merchant failed to provide the \"salt\" or a valid payto:// URI in the wire details.  Note that while the exchange will do some basic sanity checking on the wire details, it cannot warrant that the banking system will ultimately be able to route to the specified address, even if this check passed."
  },
  {
    .ec = TALER_EC_DEPOSIT_INVALID_WIRE_FORMAT_CONTRACT_HASH_CONFLICT,
    .hint = "The hash of the given wire address does not match the wire hash specified in the proposal data."
  },
  {
    .ec = TALER_EC_DEPOSIT_INVALID_WIRE_FORMAT_ACCOUNT_NUMBER,
    .hint = "The exchange detected that the given account number is invalid for the selected wire format type."
  },
  {
    .ec = TALER_EC_DEPOSIT_INVALID_TIMESTAMP,
    .hint = "Timestamp included in deposit permission is intolerably far off with respect to the clock of the exchange."
  },
  {
    .ec = TALER_EC_DEPOSIT_DENOMINATION_VALIDITY_IN_FUTURE,
    .hint = "Validity period of the denomination key is in the future."
  },
  {
    .ec = TALER_EC_DEPOSIT_DENOMINATION_EXPIRED,
    .hint = "Denomination key of the coin is past the deposit deadline."
  },
  {
    .ec = TALER_EC_DEPOSIT_INVALID_SIGNATURE_BY_EXCHANGE,
    .hint = "The signature provided by the exchange is not valid. Error created client-side."
  },
  {
    .ec = TALER_EC_DEPOSIT_CURRENCY_MISMATCH,
    .hint = "The currency specified for the deposit is different from the currency of the coin."
  },
  {
    .ec = TALER_EC_MELT_INSUFFICIENT_FUNDS,
    .hint = "The respective coin did not have sufficient residual value for the /refresh/melt operation.  The \"history\" in this response provdes the \"residual_value\" of the coin, which may be less than its \"original_value\"."
  },
  {
    .ec = TALER_EC_MELT_DENOMINATION_KEY_NOT_FOUND,
    .hint = "The respective coin did not have sufficient residual value for the /refresh/melt operation.  The \"history\" in this response provdes the \"residual_value\" of the coin, which may be less than its \"original_value\"."
  },
  {
    .ec = TALER_EC_MELT_COIN_HISTORY_COMPUTATION_FAILED,
    .hint = "The exchange had an internal error reconstructing the transaction history of the coin that was being melted."
  },
  {
    .ec = TALER_EC_MELT_DB_FETCH_ERROR,
    .hint = "The exchange failed to check against historic melt data from database (as part of ensuring the idempotency of the operation)."
  },
  {
    .ec = TALER_EC_MELT_DB_STORE_SESSION_ERROR,
    .hint = "The exchange failed to store session data in the database."
  },
  {
    .ec = TALER_EC_MELT_FEES_EXCEED_CONTRIBUTION,
    .hint = "The exchange encountered melt fees exceeding the melted coin's contribution."
  },
  {
    .ec = TALER_EC_MELT_DENOMINATION_SIGNATURE_INVALID,
    .hint = "The denomination key signature on the melted coin is invalid."
  },
  {
    .ec = TALER_EC_MELT_COIN_SIGNATURE_INVALID,
    .hint = "The signature made with the coin to be melted is invalid."
  },
  {
    .ec = TALER_EC_MELT_HISTORY_DB_ERROR_INSUFFICIENT_FUNDS,
    .hint = "The exchange failed to obtain the transaction history of the given coin from the database while generating an insufficient funds errors."
  },
  {
    .ec = TALER_EC_MELT_COIN_EXPIRED_NO_ZOMBIE,
    .hint = "The denomination of the given coin has past its expiration date and it is also not a valid zombie (that is, was not refreshed with the fresh coin being subjected to recoup)."
  },
  {
    .ec = TALER_EC_MELT_INVALID_SIGNATURE_BY_EXCHANGE,
    .hint = "The signature returned by the exchange in a melt request was malformed. Error created client-side."
  },
  {
    .ec = TALER_EC_MELT_CURRENCY_MISMATCH,
    .hint = "The currency specified for the melt amount is different from the currency of the coin."
  },
  {
    .ec = TALER_EC_REFRESH_RECOUP_DENOMINATION_KEY_NOT_FOUND,
    .hint = "The exchange is unaware of the denomination key that was used to sign the melted zombie coin."
  },
  {
    .ec = TALER_EC_REFRESH_RECOUP_DENOMINATION_VALIDITY_IN_FUTURE,
    .hint = "Validity period of the denomination key is in the future."
  },
  {
    .ec = TALER_EC_REFRESH_RECOUP_DENOMINATION_EXPIRED,
    .hint = "Denomination key of the coin is past the deposit deadline."
  },
  {
    .ec = TALER_EC_REFRESH_ZOMBIE_DENOMINATION_EXPIRED,
    .hint = "Denomination key of the coin is past the deposit deadline."
  },
  {
    .ec = TALER_EC_REVEAL_COMMITMENT_VIOLATION,
    .hint = "The provided transfer keys do not match up with the original commitment.  Information about the original commitment is included in the response."
  },
  {
    .ec = TALER_EC_REVEAL_SIGNING_ERROR,
    .hint = "Failed to produce the blinded signatures over the coins to be returned."
  },
  {
    .ec = TALER_EC_REVEAL_SESSION_UNKNOWN,
    .hint = "The exchange is unaware of the refresh session specified in the request."
  },
  {
    .ec = TALER_EC_REVEAL_DB_FETCH_SESSION_ERROR,
    .hint = "The exchange failed to retrieve valid session data from the database."
  },
  {
    .ec = TALER_EC_REVEAL_DB_FETCH_REVEAL_ERROR,
    .hint = "The exchange failed to retrieve previously revealed data from the database."
  },
  {
    .ec = TALER_EC_REVEAL_DB_COMMIT_ERROR,
    .hint = "The exchange failed to retrieve commitment data from the database."
  },
  {
    .ec = TALER_EC_REVEAL_CNC_TRANSFER_ARRAY_SIZE_INVALID,
    .hint = "The size of the cut-and-choose dimension of the private transfer keys request does not match #TALER_CNC_KAPPA - 1."
  },
  {
    .ec = TALER_EC_REVEAL_NEW_DENOMS_ARRAY_SIZE_EXCESSIVE,
    .hint = "The number of coins to be created in refresh exceeds the limits of the exchange. private transfer keys request does not match #TALER_CNC_KAPPA - 1."
  },
  {
    .ec = TALER_EC_REVEAL_NEW_DENOMS_ARRAY_SIZE_MISMATCH,
    .hint = "The number of envelopes given does not match the number of denomination keys given."
  },
  {
    .ec = TALER_EC_REVEAL_COST_CALCULATION_OVERFLOW,
    .hint = "The exchange encountered a numeric overflow totaling up the cost for the refresh operation."
  },
  {
    .ec = TALER_EC_REVEAL_AMOUNT_INSUFFICIENT,
    .hint = "The exchange's cost calculation shows that the melt amount is below the costs of the transaction."
  },
  {
    .ec = TALER_EC_REVEAL_FRESH_DENOMINATION_KEY_NOT_FOUND,
    .hint = "The exchange is unaware of the denomination key that was requested for one of the fresh coins."
  },
  {
    .ec = TALER_EC_REVEAL_LINK_SIGNATURE_INVALID,
    .hint = "The signature made with the coin over the link data is invalid."
  },
  {
    .ec = TALER_EC_REVEAL_KEYS_MISSING,
    .hint = "The exchange failed to generate the signature as it could not find the signing key for the denomination."
  },
  {
    .ec = TALER_EC_REVEAL_INVALID_RCH,
    .hint = "The refresh session hash given to a /refreshes/ handler was malformed."
  },
  {
    .ec = TALER_EC_REVEAL_REPLY_MALFORMED,
    .hint = "The exchange responded with a reply that did not satsify the protocol. This error is not used in the protocol but created client-side."
  },
  {
    .ec = TALER_EC_LINK_COIN_UNKNOWN,
    .hint = "The coin specified in the link request is unknown to the exchange."
  },
  {
    .ec = TALER_EC_LINK_REPLY_MALFORMED,
    .hint = "The exchange responded with a reply that did not satsify the protocol. This error is not used in the protocol but created client-side."
  },
  {
    .ec = TALER_EC_REFUND_COIN_NOT_FOUND,
    .hint = "The exchange knows literally nothing about the coin we were asked to refund. But without a transaction history, we cannot issue a refund. This is kind-of OK, the owner should just refresh it directly without executing the refund."
  },
  {
    .ec = TALER_EC_REFUND_CONFLICT_DEPOSIT_INSUFFICIENT,
    .hint = "We could not process the refund request as the coin's transaction history does not permit the requested refund because then refunds would exceed the deposit amount.  The \"history\" in the response proves this."
  },
  {
    .ec = TALER_EC_REFUND_INCONSITENT_AMOUNT,
    .hint = "We could not process the refund request as the same refund transaction ID was already used with a different amount. Retrying with a different refund transaction ID may work. The \"history\" in the response proves this by providing the conflicting entry."
  },
  {
    .ec = TALER_EC_REFUND_DEPOSIT_NOT_FOUND,
    .hint = "The exchange knows about the coin we were asked to refund, but not about the specific /deposit operation.  Hence, we cannot issue a refund (as we do not know if this merchant public key is authorized to do a refund)."
  },
  {
    .ec = TALER_EC_REFUND_CURRENCY_MISMATCH,
    .hint = "The currency specified for the refund is different from the currency of the coin."
  },
  {
    .ec = TALER_EC_REFUND_DB_INCONSISTENT,
    .hint = "When we tried to check if we already paid out the coin, the exchange's database suddenly disagreed with data it previously provided (internal inconsistency)."
  },
  {
    .ec = TALER_EC_REFUND_MERCHANT_ALREADY_PAID,
    .hint = "The exchange can no longer refund the customer/coin as the money was already transferred (paid out) to the merchant. (It should be past the refund deadline.)"
  },
  {
    .ec = TALER_EC_REFUND_INSUFFICIENT_FUNDS,
    .hint = "The amount the exchange was asked to refund exceeds (with fees) the total amount of the deposit (including fees)."
  },
  {
    .ec = TALER_EC_REFUND_DENOMINATION_KEY_NOT_FOUND,
    .hint = "The exchange failed to recover information about the denomination key of the refunded coin (even though it recognizes the key).  Hence it could not check the fee structure."
  },
  {
    .ec = TALER_EC_REFUND_FEE_TOO_LOW,
    .hint = "The refund fee specified for the request is lower than the refund fee charged by the exchange for the given denomination key of the refunded coin."
  },
  {
    .ec = TALER_EC_REFUND_STORE_DB_ERROR,
    .hint = "The exchange failed to store the refund information to its database."
  },
  {
    .ec = TALER_EC_REFUND_FEE_CURRENCY_MISMATCH,
    .hint = "The refund fee is specified in a different currency than the refund amount."
  },
  {
    .ec = TALER_EC_REFUND_FEE_ABOVE_AMOUNT,
    .hint = "The refunded amount is smaller than the refund fee, which would result in a negative refund."
  },
  {
    .ec = TALER_EC_REFUND_MERCHANT_SIGNATURE_INVALID,
    .hint = "The signature of the merchant is invalid."
  },
  {
    .ec = TALER_EC_REFUND_MERCHANT_SIGNING_FAILED,
    .hint = "Merchant backend failed to create the refund confirmation signature."
  },
  {
    .ec = TALER_EC_REFUND_INVALID_SIGNATURE_BY_EXCHANGE,
    .hint = "The signature returned by the exchange in a refund request was malformed. Error created client-side."
  },
  {
    .ec = TALER_EC_REFUND_INVALID_FAILURE_PROOF_BY_EXCHANGE,
    .hint = "The failure proof returned by the exchange is incorrect. Error code generated client-side."
  },
  {
    .ec = TALER_EC_REFUND_DATABASE_LOOKUP_ERROR,
    .hint = "The exchange failed to lookup information for the refund from its database."
  },
  {
    .ec = TALER_EC_ADMIN_ADD_INCOMING_WIREFORMAT_UNSUPPORTED,
    .hint = "The wire format specified in the \"sender_account_details\" is not understood or not supported by this exchange."
  },
  {
    .ec = TALER_EC_ADMIN_ADD_INCOMING_CURRENCY_UNSUPPORTED,
    .hint = "The currency specified in the \"amount\" parameter is not supported by this exhange."
  },
  {
    .ec = TALER_EC_ADMIN_ADD_INCOMING_DB_STORE,
    .hint = "The exchange failed to store information about the incoming transfer in its database."
  },
  {
    .ec = TALER_EC_TRANSFERS_GET_DB_FETCH_FAILED,
    .hint = "The exchange encountered an error (that is not about not finding the wire transfer) trying to lookup a wire transfer identifier in the database."
  },
  {
    .ec = TALER_EC_TRANSFERS_GET_DB_INCONSISTENT,
    .hint = "The exchange found internally inconsistent data when resolving a wire transfer identifier in the database."
  },
  {
    .ec = TALER_EC_TRANSFERS_GET_WTID_NOT_FOUND,
    .hint = "The exchange did not find information about the specified wire transfer identifier in the database."
  },
  {
    .ec = TALER_EC_TRANSFERS_GET_WIRE_FEE_NOT_FOUND,
    .hint = "The exchange did not find information about the wire transfer fees it charged."
  },
  {
    .ec = TALER_EC_TRANSFERS_GET_WIRE_FEE_INCONSISTENT,
    .hint = "The exchange found a wire fee that was above the total transfer value (and thus could not have been charged)."
  },
  {
    .ec = TALER_EC_TRANSFERS_GET_REPLY_MALFORMED,
    .hint = "The exchange responded with a reply that did not satsify the protocol. This error is not used in the protocol but created client-side."
  },
  {
    .ec = TALER_EC_DEPOSITS_GET_DB_FEE_INCONSISTENT,
    .hint = "The exchange found internally inconsistent fee data when resolving a transaction in the database."
  },
  {
    .ec = TALER_EC_DEPOSITS_GET_DB_FETCH_FAILED,
    .hint = "The exchange encountered an error (that is not about not finding the transaction) trying to lookup a transaction in the database."
  },
  {
    .ec = TALER_EC_DEPOSITS_GET_NOT_FOUND,
    .hint = "The exchange did not find information about the specified transaction in the database."
  },
  {
    .ec = TALER_EC_DEPOSITS_GET_WTID_RESOLUTION_ERROR,
    .hint = "The exchange failed to identify the wire transfer of the transaction (or information about the plan that it was supposed to still happen in the future)."
  },
  {
    .ec = TALER_EC_DEPOSITS_GET_MERCHANT_SIGNATURE_INVALID,
    .hint = "The signature of the merchant is invalid."
  },
  {
    .ec = TALER_EC_RECOUP_DENOMINATION_KEY_UNKNOWN,
    .hint = "The given denomination key is not in the \"recoup\" set of the exchange right now."
  },
  {
    .ec = TALER_EC_RECOUP_SIGNATURE_INVALID,
    .hint = "The given coin signature is invalid for the request."
  },
  {
    .ec = TALER_EC_RECOUP_DENOMINATION_SIGNATURE_INVALID,
    .hint = "The signature of the denomination key over the coin is not valid."
  },
  {
    .ec = TALER_EC_RECOUP_DB_FETCH_FAILED,
    .hint = "The exchange failed to access its own database about reserves."
  },
  {
    .ec = TALER_EC_RECOUP_WITHDRAW_NOT_FOUND,
    .hint = "The exchange could not find the corresponding withdraw operation. The request is denied."
  },
  {
    .ec = TALER_EC_RECOUP_HISTORY_DB_ERROR,
    .hint = "The exchange obtained an internally inconsistent transaction history for the given coin."
  },
  {
    .ec = TALER_EC_RECOUP_DB_PUT_FAILED,
    .hint = "The exchange failed to store information about the recoup to be performed in the database."
  },
  {
    .ec = TALER_EC_RECOUP_COIN_BALANCE_ZERO,
    .hint = "The coin's remaining balance is zero.  The request is denied."
  },
  {
    .ec = TALER_EC_RECOUP_BLINDING_FAILED,
    .hint = "The exchange failed to reproduce the coin's blinding."
  },
  {
    .ec = TALER_EC_RECOUP_COIN_BALANCE_NEGATIVE,
    .hint = "The coin's remaining balance is zero.  The request is denied."
  },
  {
    .ec = TALER_EC_RECOUP_DENOMINATION_VALIDITY_IN_FUTURE,
    .hint = "Validity period of the denomination key is in the future."
  },
  {
    .ec = TALER_EC_RECOUP_REPLY_MALFORMED,
    .hint = "The exchange responded with a reply that did not satsify the protocol. This error is not used in the protocol but created client-side."
  },
  {
    .ec = TALER_EC_KEYS_HAVE_NOT_NUMERIC,
    .hint = "The \"have\" parameter was not a natural number."
  },
  {
    .ec = TALER_EC_KEYS_MISSING,
    .hint = "We currently cannot find any keys."
  },
  {
    .ec = TALER_EC_KEYS_TIMETRAVEL_FORBIDDEN,
    .hint = "This exchange does not allow clients to request /keys for times other than the current (exchange) time."
  },
  {
    .ec = TALER_EC_KEYS_INVALID,
    .hint = "The keys response was malformed. This error is generated client-side."
  },
  {
    .ec = TALER_EC_INSTANCE_UNKNOWN,
    .hint = "The backend could not find the merchant instance specified in the request."
  },
  {
    .ec = TALER_EC_PROPOSAL_INSTANCE_CONFIGURATION_LACKS_WIRE,
    .hint = "The backend lacks a wire transfer method configuration option for the given instance."
  },
  {
    .ec = TALER_EC_MERCHANT_FAILED_TO_LOAD_TEMPLATE,
    .hint = "The backend could not locate a required template to generate an HTML reply."
  },
  {
    .ec = TALER_EC_MERCHANT_FAILED_TO_EXPAND_TEMPLATE,
    .hint = "The backend could not expand the template to generate an HTML reply."
  },
  {
    .ec = TALER_EC_PAY_MERCHANT_INVALID_RESPONSE,
    .hint = "The merchant failed to provide a meaningful response to a /pay request.  This error is created client-side."
  },
  {
    .ec = TALER_EC_PAY_INSUFFICIENT_FUNDS,
    .hint = "The exchange responded saying that funds were insufficient (for example, due to double-spending)."
  },
  {
    .ec = TALER_EC_PAY_DB_STORE_PAY_ERROR,
    .hint = "The merchant failed to commit the exchanges' response to a /deposit request to its database."
  },
  {
    .ec = TALER_EC_PAY_EXCHANGE_REJECTED,
    .hint = "The specified exchange is not supported/trusted by this merchant."
  },
  {
    .ec = TALER_EC_PAY_DENOMINATION_KEY_NOT_FOUND,
    .hint = "The denomination key used for payment is not listed among the denomination keys of the exchange."
  },
  {
    .ec = TALER_EC_PAY_DENOMINATION_KEY_AUDITOR_FAILURE,
    .hint = "The denomination key used for payment is not audited by an auditor approved by the merchant."
  },
  {
    .ec = TALER_EC_PAY_AMOUNT_OVERFLOW,
    .hint = "There was an integer overflow totaling up the amounts or deposit fees in the payment."
  },
  {
    .ec = TALER_EC_PAY_FEES_EXCEED_PAYMENT,
    .hint = "The deposit fees exceed the total value of the payment."
  },
  {
    .ec = TALER_EC_PAY_PAYMENT_INSUFFICIENT_DUE_TO_FEES,
    .hint = "After considering deposit and wire fees, the payment is insufficient to satisfy the required amount for the contract.  The client should revisit the logic used to calculate fees it must cover."
  },
  {
    .ec = TALER_EC_PAY_PAYMENT_INSUFFICIENT,
    .hint = "Even if we do not consider deposit and wire fees, the payment is insufficient to satisfy the required amount for the contract."
  },
  {
    .ec = TALER_EC_PAY_COIN_SIGNATURE_INVALID,
    .hint = "The signature over the contract of one of the coins was invalid."
  },
  {
    .ec = TALER_EC_PAY_EXCHANGE_TIMEOUT,
    .hint = "We failed to contact the exchange for the /pay request."
  },
  {
    .ec = TALER_EC_PAY_EXCHANGE_LOOKUP_FAILED,
    .hint = "When we tried to find information about the exchange to issue the deposit, we failed.  This usually only happens if the merchant backend is somehow unable to get its own HTTP client logic to work."
  },
  {
    .ec = TALER_EC_PAY_REFUND_DEADLINE_PAST_WIRE_TRANSFER_DEADLINE,
    .hint = "The refund deadline in the contract is after the transfer deadline."
  },
  {
    .ec = TALER_EC_PAY_COINS_ARRAY_EMPTY,
    .hint = "The request fails to provide coins for the payment."
  },
  {
    .ec = TALER_EC_PAY_DB_FETCH_PAY_ERROR,
    .hint = "The merchant failed to fetch the contract terms from the merchant's database."
  },
  {
    .ec = TALER_EC_PAY_DB_FETCH_TRANSACTION_ERROR,
    .hint = "The merchant failed to fetch the merchant's previous state with respect to transactions from its database."
  },
  {
    .ec = TALER_EC_PAY_DB_STORE_TRANSACTION_ERROR,
    .hint = "The merchant failed to store the merchant's state with respect to the transaction in its database."
  },
  {
    .ec = TALER_EC_PAY_EXCHANGE_KEYS_FAILURE,
    .hint = "The exchange failed to provide a valid response to the merchant's /keys request."
  },
  {
    .ec = TALER_EC_PAY_OFFER_EXPIRED,
    .hint = "The payment is too late, the offer has expired."
  },
  {
    .ec = TALER_EC_PAY_MERCHANT_FIELD_MISSING,
    .hint = "The \"merchant\" field is missing in the proposal data."
  },
  {
    .ec = TALER_EC_PAY_FAILED_COMPUTE_PROPOSAL_HASH,
    .hint = "Failed computing a hash code (likely server out-of-memory)."
  },
  {
    .ec = TALER_EC_PAY_WIRE_HASH_UNKNOWN,
    .hint = "Failed to locate merchant's account information matching the wire hash given in the proposal."
  },
  {
    .ec = TALER_EC_PAY_WIRE_FEE_CURRENCY_MISMATCH,
    .hint = "We got different currencies for the wire fee and the maximum wire fee."
  },
  {
    .ec = TALER_EC_PAY_EXCHANGE_REPLY_MALFORMED,
    .hint = "The exchange had a failure when trying to process the request, returning a malformed response."
  },
  {
    .ec = TALER_EC_PAY_WRONG_INSTANCE,
    .hint = "A unknown merchant public key was included in the payment.  That happens typically when the wallet sends the payment to the wrong merchant instance."
  },
  {
    .ec = TALER_EC_PAY_EXCHANGE_HAS_NO_KEYS,
    .hint = "The exchange failed to give us a response when we asked for /keys."
  },
  {
    .ec = TALER_EC_PAY_DENOMINATION_DEPOSIT_EXPIRED,
    .hint = "The deposit time for the denomination has expired."
  },
  {
    .ec = TALER_EC_PAY_PROPOSAL_NOT_FOUND,
    .hint = "The proposal is not known to the backend."
  },
  {
    .ec = TALER_EC_PAY_EXCHANGE_WIRE_FEE_ADDITION_FAILED,
    .hint = "The exchange of the deposited coin charges a wire fee that could not be added to the total (total amount too high)."
  },
  {
    .ec = TALER_EC_PAY_REFUNDED,
    .hint = "The contract was not fully paid because of refunds. Note that clients MAY treat this as paid if, for example, contracts must be executed despite of refunds."
  },
  {
    .ec = TALER_EC_PAY_REFUNDS_EXCEED_PAYMENTS,
    .hint = "According to our database, we have refunded more than we were paid (which should not be possible)."
  },
  {
    .ec = TALER_EC_PAY_ABORT_REFUND_REFUSED_PAYMENT_COMPLETE,
    .hint = "Legacy stuff. Remove me with protocol v1."
  },
  {
    .ec = TALER_EC_PAY_EXCHANGE_FAILED,
    .hint = "The payment failed at the exchange."
  },
  {
    .ec = TALER_EC_PAID_DB_ERROR,
    .hint = "The merchant backend couldn't verify the order payment because of a database failure."
  },
  {
    .ec = TALER_EC_PAID_ORDER_UNKNOWN,
    .hint = "The order is not known."
  },
  {
    .ec = TALER_EC_PAID_CONTRACT_HASH_MISMATCH,
    .hint = "The contract hash does not match the given order ID."
  },
  {
    .ec = TALER_EC_PAID_COIN_SIGNATURE_INVALID,
    .hint = "The signature of the merchant is not valid for the given contract hash."
  },
  {
    .ec = TALER_EC_ABORT_EXCHANGE_KEYS_FAILURE,
    .hint = "The merchant failed to contact the exchange."
  },
  {
    .ec = TALER_EC_ABORT_EXCHANGE_REFUND_FAILED,
    .hint = "The merchant failed to send the exchange the refund request."
  },
  {
    .ec = TALER_EC_ABORT_EXCHANGE_LOOKUP_FAILED,
    .hint = "The merchant failed to find the exchange to process the lookup."
  },
  {
    .ec = TALER_EC_ABORT_DB_STORE_ABORT_ERROR,
    .hint = "The merchant failed to store the abort request in its database."
  },
  {
    .ec = TALER_EC_ABORT_DB_STORE_TRANSACTION_ERROR,
    .hint = "The merchant failed to repeatedly serialize the transaction."
  },
  {
    .ec = TALER_EC_ABORT_DB_FETCH_TRANSACTION_ERROR,
    .hint = "The merchant failed in the lookup part of the transaction."
  },
  {
    .ec = TALER_EC_ABORT_CONTRACT_NOT_FOUND,
    .hint = "The merchant could not find the contract."
  },
  {
    .ec = TALER_EC_ABORT_REFUND_REFUSED_PAYMENT_COMPLETE,
    .hint = "The payment was already completed and thus cannot be aborted anymore."
  },
  {
    .ec = TALER_EC_ABORT_CONTRACT_HASH_MISSMATCH,
    .hint = "The hash provided by the wallet does not match the order."
  },
  {
    .ec = TALER_EC_ABORT_COINS_ARRAY_EMPTY,
    .hint = "The array of coins cannot be empty."
  },
  {
    .ec = TALER_EC_ABORT_EXCHANGE_TIMEOUT,
    .hint = "The merchant experienced a timeout processing the request."
  },
  {
    .ec = TALER_EC_FORGET_ORDER_NOT_FOUND,
    .hint = "The merchant could not find the order."
  },
  {
    .ec = TALER_EC_FORGET_PATH_SYNTAX_INCORRECT,
    .hint = "One of the paths to forget is malformed."
  },
  {
    .ec = TALER_EC_FORGET_PATH_NOT_FORGETTABLE,
    .hint = "One of the paths to forget was not marked as forgettable."
  },
  {
    .ec = TALER_EC_FORGET_ORDER_DB_START_ERROR,
    .hint = "The merchant backend cannot forget part of an order because it failed to start the database transaction."
  },
  {
    .ec = TALER_EC_FORGET_ORDER_DB_COMMIT_ERROR,
    .hint = "The merchant backend cannot forget part of an order because it failed to commit the database transaction."
  },
  {
    .ec = TALER_EC_HISTORY_TIMESTAMP_OVERFLOW,
    .hint = "Integer overflow with specified timestamp argument detected."
  },
  {
    .ec = TALER_EC_HISTORY_DB_FETCH_ERROR,
    .hint = "Failed to retrieve history from merchant database."
  },
  {
    .ec = TALER_EC_POLL_PAYMENT_CONTRACT_NOT_FOUND,
    .hint = "The backend could not find the contract specified in the request."
  },
  {
    .ec = TALER_EC_POLL_PAYMENT_REPLY_MALFORMED,
    .hint = "The response provided by the merchant backend was malformed. This error is created client-side."
  },
  {
    .ec = TALER_EC_TRACK_TRANSACTION_EXCHANGE_TIMEOUT,
    .hint = "We failed to contact the exchange for the /track/transaction request."
  },
  {
    .ec = TALER_EC_TRACK_TRANSACTION_EXCHANGE_KEYS_FAILURE,
    .hint = "We failed to get a valid /keys response from the exchange for the /track/transaction request."
  },
  {
    .ec = TALER_EC_TRACK_TRANSACTION_TRANSACTION_UNKNOWN,
    .hint = "The backend could not find the transaction specified in the request."
  },
  {
    .ec = TALER_EC_TRACK_TRANSACTION_DB_FETCH_TRANSACTION_ERROR,
    .hint = "The backend had a database access error trying to retrieve transaction data from its database."
  },
  {
    .ec = TALER_EC_TRACK_TRANSACTION_DB_FETCH_PAYMENT_ERROR,
    .hint = "The backend had a database access error trying to retrieve payment data from its database."
  },
  {
    .ec = TALER_EC_TRACK_TRANSACTION_DB_NO_DEPOSITS_ERROR,
    .hint = "The backend found no applicable deposits in the database. This is odd, as we know about the transaction, but not about deposits we made for the transaction."
  },
  {
    .ec = TALER_EC_TRACK_TRANSACTION_COIN_TRACE_ERROR,
    .hint = "We failed to obtain a wire transfer identifier for one of the coins in the transaction."
  },
  {
    .ec = TALER_EC_TRACK_TRANSACTION_WIRE_TRANSFER_TRACE_ERROR,
    .hint = "We failed to obtain the full wire transfer identifier for the transfer one of the coins was aggregated into."
  },
  {
    .ec = TALER_EC_TRACK_TRANSACTION_CONFLICTING_REPORTS,
    .hint = "We got conflicting reports from the exhange with respect to which transfers are included in which aggregate."
  },
  {
    .ec = TALER_EC_GET_TRANSFERS_DB_FETCH_ERROR,
    .hint = "We did failed to retrieve information from our database."
  },
  {
    .ec = TALER_EC_POST_TRANSFERS_EXCHANGE_TIMEOUT,
    .hint = "We failed to contact the exchange for the /track/transfer request."
  },
  {
    .ec = TALER_EC_POST_TRANSFERS_EXCHANGE_KEYS_FAILURE,
    .hint = "We failed to obtain an acceptable /keys response from the exchange for the /track/transfer request."
  },
  {
    .ec = TALER_EC_POST_TRANSFERS_DB_STORE_COIN_ERROR,
    .hint = "We failed to persist coin wire transfer information in our merchant database."
  },
  {
    .ec = TALER_EC_POST_TRANSFERS_REQUEST_ERROR,
    .hint = "We internally failed to execute the /track/transfer request."
  },
  {
    .ec = TALER_EC_POST_TRANSFERS_DB_STORE_TRANSFER_ERROR,
    .hint = "We failed to persist wire transfer information in our merchant database."
  },
  {
    .ec = TALER_EC_POST_TRANSFERS_EXCHANGE_ERROR,
    .hint = "The exchange returned an error from /track/transfer."
  },
  {
    .ec = TALER_EC_POST_TRANSFERS_DB_FETCH_DEPOSIT_ERROR,
    .hint = "We failed to fetch deposit information from our merchant database."
  },
  {
    .ec = TALER_EC_POST_TRANSFERS_DB_INTERNAL_LOGIC_ERROR,
    .hint = "We encountered an internal logic error."
  },
  {
    .ec = TALER_EC_POST_TRANSFERS_CONFLICTING_REPORTS,
    .hint = "The exchange gave conflicting information about a coin which has been wire transferred."
  },
  {
    .ec = TALER_EC_POST_TRANSFERS_JSON_RESPONSE_ERROR,
    .hint = "The merchant backend had problems in creating the JSON response."
  },
  {
    .ec = TALER_EC_POST_TRANSFERS_JSON_BAD_WIRE_FEE,
    .hint = "The exchange charged a different wire fee than what it originally advertised, and it is higher."
  },
  {
    .ec = TALER_EC_POST_TRANSFERS_ACCOUNT_NOT_FOUND,
    .hint = "We did not find the account that the transfer was made to."
  },
  {
    .ec = TALER_EC_POST_TRANSFERS_DB_STORE_ERROR,
    .hint = "We did failed to store information in our database."
  },
  {
    .ec = TALER_EC_POST_TRANSFERS_DB_LOOKUP_ERROR,
    .hint = "We did failed to retrieve information from our database."
  },
  {
    .ec = TALER_EC_POST_INSTANCES_BAD_CURRENCY,
    .hint = "The merchant backend cannot create an instance with the given default max deposit fee or default max wire fee because the fee currencies are incompatible with the merchant's currency in the config."
  },
  {
    .ec = TALER_EC_POST_INSTANCES_ALREADY_EXISTS,
    .hint = "The merchant backend cannot create an instance under the given identifier as one already exists. Use PATCH to modify the existing entry."
  },
  {
    .ec = TALER_EC_POST_INSTANCES_BAD_PAYTO_URIS,
    .hint = "The merchant backend cannot create an instance because the specified bank accounts are somehow invalid."
  },
  {
    .ec = TALER_EC_POST_INSTANCES_DB_START_ERROR,
    .hint = "The merchant backend cannot create an instance because it failed to start the database transaction."
  },
  {
    .ec = TALER_EC_POST_INSTANCES_DB_COMMIT_ERROR,
    .hint = "The merchant backend cannot create an instance because it failed to commit the database transaction."
  },
  {
    .ec = TALER_EC_DELETE_INSTANCES_ID_DB_HARD_FAILURE,
    .hint = "The merchant backend cannot delete an instance because it failed to commit the database transaction."
  },
  {
    .ec = TALER_EC_DELETE_INSTANCES_ID_NO_SUCH_INSTANCE,
    .hint = "The merchant backend cannot delete the data because it already does not exist."
  },
  {
    .ec = TALER_EC_PATCH_INSTANCES_BAD_PAYTO_URIS,
    .hint = "The merchant backend cannot update an instance because the specified bank accounts are somehow invalid."
  },
  {
    .ec = TALER_EC_PATCH_INSTANCES_DB_START_ERROR,
    .hint = "The merchant backend cannot patch an instance because it failed to start the database transaction."
  },
  {
    .ec = TALER_EC_PATCH_INSTANCES_DB_COMMIT_ERROR,
    .hint = "The merchant backend cannot patch an instance because it failed to commit the database transaction."
  },
  {
    .ec = TALER_EC_MAP_IN_UNMATCHED_HASH,
    .hint = "The hash provided in the request of /map/in does not match the contract sent alongside in the same request."
  },
  {
    .ec = TALER_EC_PROPOSAL_STORE_DB_ERROR,
    .hint = "The backend encountered an error while trying to store the h_contract_terms into the database."
  },
  {
    .ec = TALER_EC_PROPOSAL_LOOKUP_DB_ERROR,
    .hint = "The backend encountered an error while trying to retrieve the proposal data from database.  Likely to be an internal error."
  },
  {
    .ec = TALER_EC_PROPOSAL_LOOKUP_NOT_FOUND,
    .hint = "The proposal being looked up is not found on this merchant."
  },
  {
    .ec = TALER_EC_PROPOSAL_NO_LOCALTIME,
    .hint = "The proposal had no timestamp and the backend failed to obtain the local time. Likely to be an internal error."
  },
  {
    .ec = TALER_EC_PROPOSAL_ORDER_PARSE_ERROR,
    .hint = "The order provided to the backend could not be parsed, some required fields were missing or ill-formed."
  },
  {
    .ec = TALER_EC_PROPOSAL_STORE_DB_ERROR_HARD,
    .hint = "The backend encountered an error while trying to find the existing proposal in the database."
  },
  {
    .ec = TALER_EC_PROPOSAL_STORE_DB_ERROR_SOFT,
    .hint = "The backend encountered an error while trying to find the existing proposal in the database."
  },
  {
    .ec = TALER_EC_PROPOSAL_STORE_DB_ERROR_ALREADY_EXISTS,
    .hint = "The backend encountered an error: the proposal already exists."
  },
  {
    .ec = TALER_EC_PROPOSAL_ORDER_BAD_CURRENCY,
    .hint = "The order provided to the backend uses an amount in a currency that does not match the backend's configuration."
  },
  {
    .ec = TALER_EC_PROPOSAL_REPLY_MALFORMED,
    .hint = "The response provided by the merchant backend was malformed. This error is created client-side."
  },
  {
    .ec = TALER_EC_ORDERS_DELETE_NO_SUCH_ORDER,
    .hint = "The order provided to the backend could not be deleted, it is not known."
  },
  {
    .ec = TALER_EC_ORDERS_DELETE_AWAITING_PAYMENT,
    .hint = "The order provided to the backend could not be deleted, our offer is still valid and awaiting payment."
  },
  {
    .ec = TALER_EC_ORDERS_DELETE_DB_HARD_FAILURE,
    .hint = "The order provided to the backend could not be deleted, due to a database error."
  },
  {
    .ec = TALER_EC_ORDERS_LOOKUP_PRODUCT_DB_HARD_FAILURE,
    .hint = "The order provided to the backend could not be completed, due to a database error trying to fetch product inventory data."
  },
  {
    .ec = TALER_EC_ORDERS_LOOKUP_PRODUCT_DB_SOFT_FAILURE,
    .hint = "The order provided to the backend could not be completed, due to a database serialization error (which should be impossible) trying to fetch product inventory data."
  },
  {
    .ec = TALER_EC_ORDERS_LOOKUP_PRODUCT_NOT_FOUND,
    .hint = "The order provided to the backend could not be completed, because a product to be completed via inventory data is not actually in our inventory."
  },
  {
    .ec = TALER_EC_ORDERS_GET_DB_LOOKUP_ERROR,
    .hint = "We could not obtain a list of all orders because of a database failure."
  },
  {
    .ec = TALER_EC_ORDERS_CLAIM_HARD_DB_ERROR,
    .hint = "We could not claim the order because of a database failure."
  },
  {
    .ec = TALER_EC_ORDERS_CLAIM_SOFT_DB_ERROR,
    .hint = "We could not claim the order because of a database serialization failure."
  },
  {
    .ec = TALER_EC_ORDERS_CLAIM_NOT_FOUND,
    .hint = "We could not claim the order because the backend is unaware of it."
  },
  {
    .ec = TALER_EC_ORDERS_ALREADY_CLAIMED,
    .hint = "We could not claim the order because someone else claimed it first."
  },
  {
    .ec = TALER_EC_GET_PRODUCTS_NOT_FOUND,
    .hint = "The merchant backend couldn't find a product with the specified id."
  },
  {
    .ec = TALER_EC_GET_PRODUCTS_DB_LOOKUP_ERROR,
    .hint = "The merchant backend failed to lookup the products."
  },
  {
    .ec = TALER_EC_PRODUCTS_POST_DB_START_ERROR,
    .hint = "The merchant backend failed to start the transaction."
  },
  {
    .ec = TALER_EC_PRODUCTS_POST_CONFLICT_PRODUCT_EXISTS,
    .hint = "The product ID exists."
  },
  {
    .ec = TALER_EC_PRODUCTS_POST_DB_COMMIT_SOFT_ERROR,
    .hint = "The merchant backend failed to serialize the transaction."
  },
  {
    .ec = TALER_EC_PRODUCTS_POST_DB_COMMIT_HARD_ERROR,
    .hint = "The merchant backend failed to commit the transaction."
  },
  {
    .ec = TALER_EC_PRODUCTS_PATCH_DB_COMMIT_HARD_ERROR,
    .hint = "The merchant backend failed to commit the transaction."
  },
  {
    .ec = TALER_EC_PRODUCTS_PATCH_UNKNOWN_PRODUCT,
    .hint = "The merchant backend did not find the product to be updated."
  },
  {
    .ec = TALER_EC_PRODUCTS_PATCH_TOTAL_LOST_REDUCED,
    .hint = "The update would have reduced the total amount of product lost, which is not allowed."
  },
  {
    .ec = TALER_EC_PRODUCTS_PATCH_TOTAL_LOST_EXCEEDS_STOCKS,
    .hint = "The update would have mean that more stocks were lost than what remains from total inventory after sales, which is not allowed."
  },
  {
    .ec = TALER_EC_PRODUCTS_PATCH_TOTAL_STOCKED_REDUCED,
    .hint = "The update would have reduced the total amount of product in stock, which is not allowed."
  },
  {
    .ec = TALER_EC_PRODUCTS_LOCK_INSUFFICIENT_STOCKS,
    .hint = "The lock request is for more products than we have left (unlocked) in stock."
  },
  {
    .ec = TALER_EC_PRODUCTS_LOCK_UNKNOWN_PRODUCT,
    .hint = "The lock request is for an unknown product."
  },
  {
    .ec = TALER_EC_PRODUCTS_DELETE_DB_HARD_FAILURE,
    .hint = "The deletion request resulted in a hard database error."
  },
  {
    .ec = TALER_EC_PRODUCTS_DELETE_NO_SUCH_PRODUCT,
    .hint = "The deletion request was for a product unknown to the backend."
  },
  {
    .ec = TALER_EC_PRODUCTS_DELETE_CONFLICTING_LOCK,
    .hint = "The deletion request is for a product that is locked."
  },
  {
    .ec = TALER_EC_REFUND_LOOKUP_INVALID_RESPONSE,
    .hint = "The merchant returned a malformed response. Error created client-side."
  },
  {
    .ec = TALER_EC_REFUND_ORDER_ID_UNKNOWN,
    .hint = "The frontend gave an unknown order id to issue the refund to."
  },
  {
    .ec = TALER_EC_REFUND_INCONSISTENT_AMOUNT,
    .hint = "The amount to be refunded is inconsistent: either is lower than the previous amount being awarded, or it is too big to be paid back. In this second case, the fault stays on the business dept. side."
  },
  {
    .ec = TALER_EC_REFUND_LOOKUP_DB_ERROR,
    .hint = "The backend encountered an error while trying to retrieve the payment data from database.  Likely to be an internal error."
  },
  {
    .ec = TALER_EC_REFUND_MERCHANT_DB_COMMIT_ERROR,
    .hint = "The backend encountered an error while trying to retrieve the payment data from database.  Likely to be an internal error."
  },
  {
    .ec = TALER_EC_PAY_DB_STORE_PAYMENTS_ERROR,
    .hint = "Payments are stored in a single db transaction; this error indicates that one db operation within that transaction failed.  This might involve storing of coins or other related db operations, like starting/committing the db transaction or marking a contract as paid."
  },
  {
    .ec = TALER_EC_PAY_REFUND_SIGNATURE_FAILED,
    .hint = "The backend failed to sign the refund request."
  },
  {
    .ec = TALER_EC_REFUND_LOOKUP_NO_REFUND,
    .hint = "The merchant backend is not available of any applicable refund(s) for this order."
  },
  {
    .ec = TALER_EC_REFUND_ORDER_ID_UNPAID,
    .hint = "The frontend gave an unpaid order id to issue the refund to."
  },
  {
    .ec = TALER_EC_RESERVES_POST_UNSUPPORTED_WIRE_METHOD,
    .hint = "The requested wire method is not supported by the exchange."
  },
  {
    .ec = TALER_EC_RESERVES_POST_DB_COMMIT_HARD_ERROR,
    .hint = "The backend failed to commit the result to the database."
  },
  {
    .ec = TALER_EC_GET_RESERVES_DB_LOOKUP_ERROR,
    .hint = "The backend failed to fetch the requested information from the database."
  },
  {
    .ec = TALER_EC_TIP_AUTHORIZE_INSTANCE_DOES_NOT_TIP,
    .hint = "The backend knows the instance that was supposed to support the tip, but it was not configured for tipping (i.e. has no exchange associated with it).  Likely to be a configuration error."
  },
  {
    .ec = TALER_EC_TIP_AUTHORIZE_RESERVE_EXPIRED,
    .hint = "The reserve that was used to fund the tips has expired."
  },
  {
    .ec = TALER_EC_TIP_AUTHORIZE_RESERVE_UNKNOWN,
    .hint = "The reserve that was used to fund the tips was not found in the DB."
  },
  {
    .ec = TALER_EC_TIP_AUTHORIZE_INSUFFICIENT_FUNDS,
    .hint = "The backend knows the instance that was supposed to support the tip, and it was configured for tipping. However, the funds remaining are insufficient to cover the tip, and the merchant should top up the reserve."
  },
  {
    .ec = TALER_EC_TIP_AUTHORIZE_DB_HARD_ERROR,
    .hint = "The backend had trouble accessing the database to persist information about the tip authorization. Returned with an HTTP status code of internal error."
  },
  {
    .ec = TALER_EC_TIP_AUTHORIZE_DB_SOFT_ERROR,
    .hint = "The backend had trouble accessing the database to persist information about the tip authorization. The problem might be fixable by repeating the transaction."
  },
  {
    .ec = TALER_EC_TIP_QUERY_RESERVE_STATUS_FAILED_EXCHANGE_DOWN,
    .hint = "The backend failed to obtain a reserve status from the exchange."
  },
  {
    .ec = TALER_EC_TIP_QUERY_RESERVE_HISTORY_FAILED_EMPTY,
    .hint = "The backend got an empty (!) reserve history from the exchange."
  },
  {
    .ec = TALER_EC_TIP_QUERY_RESERVE_HISTORY_INVALID_NO_DEPOSIT,
    .hint = "The backend got an invalid reserve history (fails to start with a deposit) from the exchange."
  },
  {
    .ec = TALER_EC_TIP_QUERY_RESERVE_UNKNOWN_TO_EXCHANGE,
    .hint = "The backend got an 404 response from the exchange when it inquired about the reserve history."
  },
  {
    .ec = TALER_EC_TIP_QUERY_RESERVE_CURRENCY_MISMATCH,
    .hint = "The backend got a reserve with a currency that does not match the backend's currency."
  },
  {
    .ec = TALER_EC_TIP_QUERY_RESERVE_HISTORY_ARITHMETIC_ISSUE_DEPOSIT,
    .hint = "The backend got a reserve history with amounts it cannot process (addition failure in deposits)."
  },
  {
    .ec = TALER_EC_TIP_QUERY_RESERVE_HISTORY_ARITHMETIC_ISSUE_WITHDRAW,
    .hint = "The backend got a reserve history with amounts it cannot process (addition failure in withdraw amounts)."
  },
  {
    .ec = TALER_EC_TIP_QUERY_RESERVE_HISTORY_ARITHMETIC_ISSUE_CLOSED,
    .hint = "The backend got a reserve history with amounts it cannot process (addition failure in closing amounts)."
  },
  {
    .ec = TALER_EC_TIP_QUERY_RESERVE_HISTORY_ARITHMETIC_ISSUE_INCONSISTENT,
    .hint = "The backend got a reserve history with inconsistent amounts."
  },
  {
    .ec = TALER_EC_TIP_QUERY_DB_ERROR,
    .hint = "The backend encountered a database error querying tipping reserves."
  },
  {
    .ec = TALER_EC_TIP_QUERY_RESERVE_HISTORY_FAILED,
    .hint = "The backend got an unexpected resever history reply from the exchange."
  },
  {
    .ec = TALER_EC_TIP_QUERY_RESERVE_HISTORY_ARITHMETIC_ISSUE_RECOUP,
    .hint = "The backend got a reserve history with amounts it cannot process (addition failure in withdraw amounts)."
  },
  {
    .ec = TALER_EC_TIP_QUERY_INSTANCE_DOES_NOT_TIP,
    .hint = "The backend knows the instance that was supposed to support the tip, but it was not configured for tipping (i.e. has no exchange associated with it).  Likely to be a configuration error."
  },
  {
    .ec = TALER_EC_TIP_QUERY_TIP_ID_UNKNOWN,
    .hint = "The tip id is unknown.  This could happen if the tip id is wrong or the tip authorization expired."
  },
  {
    .ec = TALER_EC_RESERVES_DELETE_DB_HARD_FAILURE,
    .hint = "The reserve could not be deleted due to a database failure."
  },
  {
    .ec = TALER_EC_RESERVES_DELETE_NO_SUCH_RESERVE,
    .hint = "The reserve could not be deleted because it is unknown."
  },
  {
    .ec = TALER_EC_TIP_LOOKUP_RESERVE_DB_FAILURE,
    .hint = "The backend got an unexpected error trying to lookup reserve details from the backend."
  },
  {
    .ec = TALER_EC_TIP_AUTHORIZE_DB_SERIALIZATION_FAILURE,
    .hint = "The backend repeatedly failed to serialize the transaction to authorize the tip."
  },
  {
    .ec = TALER_EC_TIP_AUTHORIZE_DB_START_FAILURE,
    .hint = "The backend failed to start the transaction to authorize the tip."
  },
  {
    .ec = TALER_EC_TIP_AUTHORIZE_DB_LOOKUP_RESERVE_FAILURE,
    .hint = "The backend failed looking up the reserve needed to authorize the tip."
  },
  {
    .ec = TALER_EC_TIP_AUTHORIZE_DB_RESERVE_NOT_FOUND,
    .hint = "The backend failed to find a reserve needed to authorize the tip."
  },
  {
    .ec = TALER_EC_TIP_AUTHORIZE_DB_RESERVE_INVARIANT_FAILURE,
    .hint = "The backend encountered an internal invariant violation."
  },
  {
    .ec = TALER_EC_TIP_AUTHORIZE_DB_RESERVE_EXPIRED,
    .hint = "The selected exchange expired."
  },
  {
    .ec = TALER_EC_TIP_AUTHORIZE_DB_UPDATE_RESERVE_FAILURE,
    .hint = "The backend failed updating the reserve needed to authorize the tip."
  },
  {
    .ec = TALER_EC_TIP_ENABLE_DB_TRANSACTION_ERROR,
    .hint = "The backend had trouble accessing the database to persist information about enabling tips. Returned with an HTTP status code of internal error."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_TIP_ID_UNKNOWN,
    .hint = "The tip ID is unknown.  This could happen if the tip has expired."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_NO_FUNDS,
    .hint = "The amount requested exceeds the remaining tipping balance for this tip ID. Returned with an HTTP status code of \"Conflict\" (as it conflicts with a previous pickup operation)."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_DB_ERROR_SOFT,
    .hint = "We encountered a DB error, repeating the request may work."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_DB_ERROR_HARD,
    .hint = "We encountered a DB error, repeating the request will not help. This is an internal server error."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_AMOUNT_CHANGED,
    .hint = "The same pickup ID was already used for picking up a different amount. This points to a very strange internal error as the pickup ID is derived from the denomination key which is tied to a particular amount. Hence this should also be an internal server error."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_EXCHANGE_DOWN,
    .hint = "We failed to contact the exchange to obtain the denomination keys."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_EXCHANGE_LACKED_KEYS,
    .hint = "We contacted the exchange to obtain any denomination keys, but got no valid keys."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_EXCHANGE_LACKED_KEY,
    .hint = "We contacted the exchange to obtain at least one of the denomination keys specified in the request. Returned with a response code \"not found\" (404)."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_EXCHANGE_AMOUNT_OVERFLOW,
    .hint = "We encountered an arithmetic issue totaling up the amount to withdraw."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_EXCHANGE_TOO_MANY_PLANCHETS,
    .hint = "The number of planchets specified exceeded the limit."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_WITHDRAW_FAILED,
    .hint = "The merchant failed to initialize the withdraw operation."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_WITHDRAW_FAILED_AT_EXCHANGE,
    .hint = "The merchant failed to initialize the withdraw operation."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_UNBLIND_FAILURE,
    .hint = "The client failed to unblind the signature returned by the merchant. Generated client-side."
  },
  {
    .ec = TALER_EC_GET_TIPS_DB_LOOKUP_ERROR,
    .hint = "Merchant failed to access its database to lookup the tip."
  },
  {
    .ec = TALER_EC_GET_TIPS_ID_UNKNOWN,
    .hint = "Merchant failed find the tip in its database."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_CONTACT_EXCHANGE_ERROR,
    .hint = "The merchant failed to contact the exchange."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_EXCHANGE_KEYS_ERROR,
    .hint = "The merchant failed to obtain keys from the exchange."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_DB_STORE_HARD_ERROR,
    .hint = "The merchant failed to store data in its own database."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_EXCHANGE_TIMEOUT,
    .hint = "The merchant failed to get a timely response from the exchange."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_EXCHANGE_ERROR,
    .hint = "The exchange returned a failure code for the withdraw operation."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_SUMMATION_FAILED,
    .hint = "The merchant failed to add up the amounts to compute the pick up value."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_HAS_EXPIRED,
    .hint = "The tip expired."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_AMOUNT_EXCEEDS_TIP_REMAINING,
    .hint = "The requested withdraw amount exceeds the amount remaining to be picked up."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_DB_STORE_SOFT_ERROR,
    .hint = "The merchant failed to store data in its own database."
  },
  {
    .ec = TALER_EC_TIP_PICKUP_DENOMINATION_UNKNOWN,
    .hint = "The merchant did not find the specified denomination key in the exchange's key set."
  },
  {
    .ec = TALER_EC_GET_ORDERS_DB_LOOKUP_ERROR,
    .hint = "We failed to fetch contract terms from our merchant database."
  },
  {
    .ec = TALER_EC_GET_ORDERS_ID_UNKNOWN,
    .hint = "We failed to find the contract terms from our merchant database."
  },
  {
    .ec = TALER_EC_GET_ORDERS_EXCHANGE_TIMEOUT,
    .hint = "The merchant had a timeout contacting the exchange, thus not providing wire details in the response."
  },
  {
    .ec = TALER_EC_GET_ORDERS_EXCHANGE_TRACKING_FAILURE,
    .hint = "The exchange failed to provide a valid answer to the tracking request, thus those details are not in the response."
  },
  {
    .ec = TALER_EC_GET_ORDERS_DB_STORE_TRACKING_FAILURE,
    .hint = "The merchant backend failed to persist tracking details in its database, thus those details are not in the response."
  },
  {
    .ec = TALER_EC_GET_ORDERS_AMOUNT_ARITHMETIC_FAILURE,
    .hint = "The merchant backend encountered a failure in computing the deposit total."
  },
  {
    .ec = TALER_EC_GET_ORDERS_EXCHANGE_LOOKUP_FAILURE,
    .hint = "The merchant backend failed trying to contact the exchange for tracking details, thus those details are not in the response."
  },
  {
    .ec = TALER_EC_GET_ORDERS_EXCHANGE_REQUEST_FAILURE,
    .hint = "The merchant backend failed to construct the request for tracking to the exchange, thus tracking details are not in the response."
  },
  {
    .ec = TALER_EC_GET_ORDERS_DB_FETCH_CONTRACT_TERMS_ERROR,
    .hint = "The merchant backend had a database failure trying to find information about the contract of the order."
  },
  {
    .ec = TALER_EC_GET_ORDERS_ORDER_NOT_FOUND,
    .hint = "The merchant backend could not find an order with the given identifier."
  },
  {
    .ec = TALER_EC_GET_ORDERS_FAILED_COMPUTE_PROPOSAL_HASH,
    .hint = "The merchant backend could not compute the hash of the proposal."
  },
  {
    .ec = TALER_EC_GET_ORDERS_DB_FETCH_PAYMENT_STATUS,
    .hint = "The merchant backend could not fetch the payment status from its database."
  },
  {
    .ec = TALER_EC_GET_ORDERS_DB_FETCH_TRANSACTION_ERROR,
    .hint = "The merchant backend had an error looking up information in its database."
  },
  {
    .ec = TALER_EC_GET_ORDERS_CONTRACT_CONTENT_INVALID,
    .hint = "The contract obtained from the merchant backend was malformed."
  },
  {
    .ec = TALER_EC_CHECK_PAYMENT_DB_FETCH_CONTRACT_TERMS_ERROR,
    .hint = "We failed to contract terms from our merchant database."
  },
  {
    .ec = TALER_EC_CHECK_PAYMENT_DB_FETCH_ORDER_ERROR,
    .hint = "We failed to contract terms from our merchant database."
  },
  {
    .ec = TALER_EC_CHECK_PAYMENT_ORDER_ID_UNKNOWN,
    .hint = "The order id we're checking is unknown, likely the frontend did not create the order first."
  },
  {
    .ec = TALER_EC_CHECK_PAYMENT_FAILED_COMPUTE_PROPOSAL_HASH,
    .hint = "Failed computing a hash code (likely server out-of-memory)."
  },
  {
    .ec = TALER_EC_CHECK_PAYMENT_SESSION_SIGNATURE_INVALID,
    .hint = "Signature \"session_sig\" failed to verify."
  },
  {
    .ec = TALER_EC_GET_ORDER_WRONG_CONTRACT,
    .hint = "The order we found does not match the provided contract hash."
  },
  {
    .ec = TALER_EC_CHECK_PAYMENT_RESPONSE_MALFORMED,
    .hint = "The response we received from the merchant is malformed. This error is generated client-side."
  },
  {
    .ec = TALER_EC_GET_ORDERS_EXCHANGE_LOOKUP_START_FAILURE,
    .hint = "The merchant backend failed trying to contact the exchange for tracking details, thus those details are not in the response."
  },
  {
    .ec = TALER_EC_MERCHANT_ORDER_GET_REPLY_MALFORMED,
    .hint = "The response we received from the merchant is malformed. This error is generated client-side."
  },
  {
    .ec = TALER_EC_MERCHANT_GET_ORDER_INVALID_TOKEN,
    .hint = "The token used to authenticate the client is invalid for this order."
  },
  {
    .ec = TALER_EC_MERCHANT_PRIVATE_GET_ORDERS_STATUS_DB_LOOKUP_ERROR,
    .hint = "The merchant backup failed to lookup the order status in the database."
  },
  {
    .ec = TALER_EC_MERCHANT_PRIVATE_GET_ORDERS_CONTRACT_DB_LOOKUP_ERROR,
    .hint = "The merchant backup failed to lookup the contract terms in the database."
  },
  {
    .ec = TALER_EC_MERCHANT_PRIVATE_GET_ORDERS_PARSE_CONTRACT_ERROR,
    .hint = "The merchant backup failed to parse the order contract terms."
  },
  {
    .ec = TALER_EC_MERCHANT_PRIVATE_GET_ORDERS_REFUND_DB_LOOKUP_ERROR,
    .hint = "The merchant backup failed to lookup the refunds in the database."
  },
  {
    .ec = TALER_EC_MERCHANT_PRIVATE_GET_ORDERS_BY_FILTER_DB_LOOKUP_ERROR,
    .hint = "The merchant backup failed to lookup filtered orders in the database."
  },
  {
    .ec = TALER_EC_DEPOSIT_CONFIRMATION_SIGNATURE_INVALID,
    .hint = "The signature from the exchange on the deposit confirmation is invalid.  Returned with a \"400 Bad Request\" status code."
  },
  {
    .ec = TALER_EC_DEPOSIT_CONFIRMATION_STORE_DB_ERROR,
    .hint = "The auditor had trouble storing the deposit confirmation in its database."
  },
  {
    .ec = TALER_EC_LIST_EXCHANGES_DB_ERROR,
    .hint = "The auditor had trouble retrieving the exchange list from its database."
  },
  {
    .ec = TALER_EC_AUDITOR_EXCHANGE_STORE_DB_ERROR,
    .hint = "The auditor had trouble storing an exchange in its database."
  },
  {
    .ec = TALER_EC_AUDITOR_EXCHANGES_REPLY_MALFORMED,
    .hint = "The auditor (!) responded with a reply that did not satsify the protocol. This error is not used in the protocol but created client- side."
  },
  {
    .ec = TALER_EC_TEST_ECDH_ERROR,
    .hint = "The exchange failed to compute ECDH."
  },
  {
    .ec = TALER_EC_TEST_EDDSA_INVALID,
    .hint = "The EdDSA test signature is invalid."
  },
  {
    .ec = TALER_EC_TEST_EDDSA_ERROR,
    .hint = "The exchange failed to compute the EdDSA test signature."
  },
  {
    .ec = TALER_EC_TEST_RSA_GEN_ERROR,
    .hint = "The exchange failed to generate an RSA key."
  },
  {
    .ec = TALER_EC_TEST_RSA_PUB_ERROR,
    .hint = "The exchange failed to compute the public RSA key."
  },
  {
    .ec = TALER_EC_TEST_RSA_SIGN_ERROR,
    .hint = "The exchange failed to compute the RSA signature."
  },
  {
    .ec = TALER_EC_SERVER_JSON_INVALID,
    .hint = "The JSON in the server's response was malformed.  This response is provided with HTTP status code of 0."
  },
  {
    .ec = TALER_EC_SERVER_SIGNATURE_INVALID,
    .hint = "A signature in the server's response was malformed.  This response is provided with HTTP status code of 0."
  },
  {
    .ec = TALER_EC_BANK_SAME_ACCOUNT,
    .hint = "Wire transfer attempted with credit and debit party being the same bank account."
  },
  {
    .ec = TALER_EC_BANK_UNALLOWED_DEBIT,
    .hint = "Wire transfer impossible, due to financial limitation of the party that attempted the payment."
  },
  {
    .ec = TALER_EC_BANK_CURRENCY_MISMATCH,
    .hint = "Arithmetic operation between two amounts of different currency was attempted."
  },
  {
    .ec = TALER_EC_BANK_PARAMETER_MISSING_OR_INVALID,
    .hint = "At least one GET parameter was either missing or invalid for the requested operation."
  },
  {
    .ec = TALER_EC_BANK_JSON_INVALID,
    .hint = "JSON body sent was invalid for the requested operation."
  },
  {
    .ec = TALER_EC_BANK_NEGATIVE_NUMBER_AMOUNT,
    .hint = "Negative number was used (as value and/or fraction) to initiate a Amount object."
  },
  {
    .ec = TALER_EC_BANK_NUMBER_TOO_BIG,
    .hint = "A number too big was used (as value and/or fraction) to initiate a amount object."
  },
  {
    .ec = TALER_EC_BANK_LOGIN_FAILED,
    .hint = "Could not login for the requested operation."
  },
  {
    .ec = TALER_EC_BANK_UNKNOWN_ACCOUNT,
    .hint = "The bank account referenced in the requested operation was not found. Returned along \"400 Not found\"."
  },
  {
    .ec = TALER_EC_BANK_TRANSACTION_NOT_FOUND,
    .hint = "The transaction referenced in the requested operation (typically a reject operation), was not found."
  },
  {
    .ec = TALER_EC_BANK_BAD_FORMAT_AMOUNT,
    .hint = "Bank received a malformed amount string."
  },
  {
    .ec = TALER_EC_BANK_REJECT_NO_RIGHTS,
    .hint = "The client does not own the account credited by the transaction which is to be rejected, so it has no rights do reject it.  To be returned along HTTP 403 Forbidden."
  },
  {
    .ec = TALER_EC_BANK_UNMANAGED_EXCEPTION,
    .hint = "This error code is returned when no known exception types captured the exception, and comes along with a 500 Internal Server Error."
  },
  {
    .ec = TALER_EC_BANK_SOFT_EXCEPTION,
    .hint = "This error code is used for all those exceptions that do not really need a specific error code to return to the client, but need to signal the middleware that the bank is not responding with 500 Internal Server Error.  Used for example when a client is trying to register with a unavailable username."
  },
  {
    .ec = TALER_EC_BANK_TRANSFER_REQUEST_UID_REUSED,
    .hint = "The request UID for a request to transfer funds has already been used, but with different details for the transfer."
  },
  {
    .ec = TALER_EC_BANK_WITHDRAWAL_OPERATION_RESERVE_SELECTION_CONFLICT,
    .hint = "The withdrawal operation already has a reserve selected.  The current request conflicts with the existing selection."
  },
  {
    .ec = TALER_EC_SYNC_DB_HARD_FETCH_ERROR,
    .hint = "The sync service failed to access its database."
  },
  {
    .ec = TALER_EC_SYNC_BACKUP_UNKNOWN,
    .hint = "The sync service failed find the record in its database."
  },
  {
    .ec = TALER_EC_SYNC_ACCOUNT_UNKNOWN,
    .hint = "The sync service failed find the account in its database."
  },
  {
    .ec = TALER_EC_SYNC_BAD_IF_NONE_MATCH,
    .hint = "The SHA-512 hash provided in the If-None-Match header is malformed."
  },
  {
    .ec = TALER_EC_SYNC_BAD_IF_MATCH,
    .hint = "The SHA-512 hash provided in the If-Match header is malformed or missing."
  },
  {
    .ec = TALER_EC_SYNC_BAD_SYNC_SIGNATURE,
    .hint = "The signature provided in the \"Sync-Signature\" header is malformed or missing."
  },
  {
    .ec = TALER_EC_SYNC_INVALID_SIGNATURE,
    .hint = "The signature provided in the \"Sync-Signature\" header does not match the account, old or new Etags."
  },
  {
    .ec = TALER_EC_SYNC_MALFORMED_CONTENT_LENGTH,
    .hint = "The \"Content-length\" field for the upload is either not a number, or too big."
  },
  {
    .ec = TALER_EC_SYNC_EXCESSIVE_CONTENT_LENGTH,
    .hint = "The \"Content-length\" field for the upload is too big based on the server's terms of service."
  },
  {
    .ec = TALER_EC_SYNC_OUT_OF_MEMORY_ON_CONTENT_LENGTH,
    .hint = "The server is out of memory to handle the upload. Trying again later may succeed."
  },
  {
    .ec = TALER_EC_SYNC_INVALID_UPLOAD,
    .hint = "The uploaded data does not match the Etag."
  },
  {
    .ec = TALER_EC_SYNC_DATABASE_FETCH_ERROR,
    .hint = "We failed to check for existing upload data in the database."
  },
  {
    .ec = TALER_EC_SYNC_SHUTDOWN,
    .hint = "HTTP server was being shutdown while this operation was pending."
  },
  {
    .ec = TALER_EC_SYNC_PAYMENT_TIMEOUT,
    .hint = "HTTP server experienced a timeout while awaiting promised payment."
  },
  {
    .ec = TALER_EC_SYNC_PAYMENT_CREATE_DB_ERROR,
    .hint = "Sync could not store order data in its own database."
  },
  {
    .ec = TALER_EC_SYNC_PAYMENT_CONFIRM_DB_ERROR,
    .hint = "Sync could not store payment confirmation in its own database."
  },
  {
    .ec = TALER_EC_SYNC_PAYMENT_CHECK_ORDER_DB_ERROR,
    .hint = "Sync could not fetch information about possible existing orders from its own database."
  },
  {
    .ec = TALER_EC_SYNC_PAYMENT_CREATE_BACKEND_ERROR,
    .hint = "Sync could not setup the payment request with its own backend."
  },
  {
    .ec = TALER_EC_SYNC_PREVIOUS_BACKUP_UNKNOWN,
    .hint = "The sync service failed find the backup to be updated in its database."
  },
  {
    .ec = TALER_EC_SYNC_DB_SOFT_FETCH_ERROR,
    .hint = "The sync service had a serialization failure when accessing its database."
  },
  {
    .ec = TALER_EC_SYNC_DB_INCONSISTENT_FETCH_ERROR,
    .hint = "The sync service first found information, and then later not. This could happen if a backup was garbage collected just when it was being accessed. Trying again may give a different answer."
  },
  {
    .ec = TALER_EC_SYNC_MISSING_CONTENT_LENGTH,
    .hint = "The \"Content-length\" field for the upload is missing."
  },
  {
    .ec = TALER_EC_WALLET_EXCHANGE_PROTOCOL_VERSION_INCOMPATIBLE,
    .hint = "The wallet does not implement a version of the exchange protocol that is compatible with the protocol version of the exchange."
  },
  {
    .ec = TALER_EC_WALLET_UNEXPECTED_EXCEPTION,
    .hint = "The wallet encountered an unexpected exception.  This is likely a bug in the wallet implementation."
  },
  {
    .ec = TALER_EC_WALLET_RECEIVED_MALFORMED_RESPONSE,
    .hint = "The wallet received a response from a server, but the response can't be parsed."
  },
  {
    .ec = TALER_EC_WALLET_NETWORK_ERROR,
    .hint = "The wallet tried to make a network request, but it received no response."
  },
  {
    .ec = TALER_EC_WALLET_HTTP_REQUEST_THROTTLED,
    .hint = "The wallet tried to make a network request, but it was throttled."
  },
  {
    .ec = TALER_EC_WALLET_UNEXPECTED_REQUEST_ERROR,
    .hint = "The wallet made a request to a service, but received an error response it does not know how to handle."
  },
  {
    .ec = TALER_EC_WALLET_EXCHANGE_DENOMINATIONS_INSUFFICIENT,
    .hint = "The denominations offered by the exchange are insufficient.  Likely the exchange is badly configured or not maintained."
  },
  {
    .ec = TALER_EC_WALLET_CORE_API_OPERATION_UNKNOWN,
    .hint = "The wallet does not support the operation requested by a client."
  },
  {
    .ec = TALER_EC_WALLET_INVALID_TALER_PAY_URI,
    .hint = "The given taler://pay URI is invalid."
  },
  {
    .ec = TALER_EC_WALLET_EXCHANGE_COIN_SIGNATURE_INVALID,
    .hint = "The signature on a coin by the exchange's denomination key is invalid after unblinding it."
  },
  {
    .ec = TALER_EC_WALLET_WITHDRAW_RESERVE_UNKNOWN_AT_EXCHANGE,
    .hint = "The exchange does not know about the reserve (yet), and thus withdrawal can't progress."
  },
  {
    .ec = TALER_EC_WALLET_CORE_NOT_AVAILABLE,
    .hint = "The wallet core service is not available."
  },
  {
    .ec = TALER_EC_WALLET_WITHDRAWAL_OPERATION_ABORTED_BY_BANK,
    .hint = "The bank has aborted a withdrawal operation, and thus a withdrawal can't complete."
  },
  {
    .ec = TALER_EC_WALLET_HTTP_REQUEST_TIMEOUT,
    .hint = "An HTTP request made by the wallet timed out."
  },
  {
    .ec = TALER_EC_WALLET_ORDER_ALREADY_CLAIMED,
    .hint = "The order has already been claimed by another wallet."
  },
  {
    .ec = TALER_EC_WALLET_WITHDRAWAL_GROUP_INCOMPLETE,
    .hint = "A group of withdrawal operations (typically for the same reserve at the same exchange) has errors and will be tried again later."
  },
  {
    .ec = TALER_EC_WALLET_TIPPING_COIN_SIGNATURE_INVALID,
    .hint = "The signature on a coin by the exchange's denomination key (obtained through the merchant via tipping) is invalid after unblinding it."
  },
  {
    .ec = TALER_EC_WALLET_BANK_INTEGRATION_PROTOCOL_VERSION_INCOMPATIBLE,
    .hint = "The wallet does not implement a version of the bank integration API that is compatible with the version offered by the bank."
  },
  {
    .ec = TALER_EC_END,
    .hint = "End of error code range."
  },

};


/**
 * The length of @e code_hint_pairs.
 */
static const unsigned int code_hint_pairs_length = 467;


/**
 * Returns a hint for a given error code.
 *
 * @param ec the error code.
 * @return the hint if it could be found, otherwise "<no hint found>".
 */
const char *
TALER_ErrorCode_get_hint (enum TALER_ErrorCode ec)
{
  unsigned int lower = 0;
  unsigned int upper = code_hint_pairs_length - 1;
  unsigned int mid = upper / 2;
  while (lower <= upper)
  {
    mid = (upper + lower) / 2;
    if (code_hint_pairs[mid].ec < ec)
    {
      lower = mid + 1;
    }
    else if (code_hint_pairs[mid].ec > ec)
    {
      upper = mid - 1;
    }
    else
    {
      return code_hint_pairs[mid].hint;
    }
  }
  return "<no hint found>";
}
