/*
  This file is part of TALER
  Copyright (C) 2018 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 3, or
  (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/

/**
 * @file bank-lib/testing_api_helpers.c
 * @brief convenience functions for bank-lib tests.
 * @author Marcello Stanisci
 */

#include "platform.h"
#include <gnunet/gnunet_util_lib.h>
#include "taler_testing_bank_lib.h"
#include "taler_fakebank_lib.h"


#define BANK_FAIL() \
  do {GNUNET_break (0); return NULL; } while (0)


/**
 * Keep each bank account credentials at index:
 * bank account number - 1
 */
struct TALER_BANK_AuthenticationData AUTHS[] = {

  /* Bank credentials */
  {.method = TALER_BANK_AUTH_BASIC,
   .details.basic.username = TALER_TESTING_BANK_USERNAME,
   .details.basic.password = TALER_TESTING_BANK_PASSWORD},

  /* Exchange credentials */
  {.method = TALER_BANK_AUTH_BASIC,
   .details.basic.username = TALER_TESTING_EXCHANGE_USERNAME,
   .details.basic.password = TALER_TESTING_EXCHANGE_PASSWORD },

  /* User credentials */
  {.method = TALER_BANK_AUTH_BASIC,
   .details.basic.username = TALER_TESTING_USER_USERNAME,
   .details.basic.password = TALER_TESTING_USER_PASSWORD }
};


/**
 * Runs the Fakebank by guessing / extracting the portnumber
 * from the base URL.
 *
 * @param bank_url bank's base URL.
 * @return the fakebank process handle, or NULL if any
 *         error occurs.
 */
struct TALER_FAKEBANK_Handle *
TALER_TESTING_run_fakebank (const char *bank_url)
{
  const char *port;
  long pnum;
  struct TALER_FAKEBANK_Handle *fakebankd;

  port = strrchr (bank_url,
                  (unsigned char) ':');
  if (NULL == port)
    pnum = 80;
  else
    pnum = strtol (port + 1, NULL, 10);
  GNUNET_log (GNUNET_ERROR_TYPE_INFO,
              "Starting Fakebank on port %u (%s)\n",
              (unsigned int) pnum,
              bank_url);
  fakebankd = TALER_FAKEBANK_start ((uint16_t) pnum);
  if (NULL == fakebankd)
  {
    GNUNET_break (0);
    return NULL;
  }
  return fakebankd;
}


/**
 * Look for substring in a programs' name.
 *
 * @param prog program's name to look into
 * @param marker chunk to find in @a prog
 */
int
TALER_TESTING_has_in_name (const char *prog_name,
                           const char *marker)
{
  size_t name_pos;
  size_t pos;

  if (! prog_name || ! marker)
    return GNUNET_NO;

  pos = 0;
  name_pos = 0;
  while (prog_name[pos])
  {
    if ('/' == prog_name[pos])
      name_pos = pos + 1;
    pos++;
  }
  if (name_pos == pos)
    return GNUNET_YES;
  return strstr (prog_name + name_pos, marker) != NULL;
}


/**
 * Start the (Python) bank process.  Assume the port
 * is available and the database is clean.  Use the "prepare
 * bank" function to do such tasks.
 *
 * @param config_filename configuration filename.
 * @param bank_url base URL of the bank, used by `wget' to check
 *        that the bank was started right.
 *
 * @return the process, or NULL if the process could not
 *         be started.
 */
struct GNUNET_OS_Process *
TALER_TESTING_run_bank (const char *config_filename,
                        const char *bank_url)
{
  struct GNUNET_OS_Process *bank_proc;
  unsigned int iter;
  char *wget_cmd;
  char *database;
  char *serve_cfg;
  char *serve_arg;
  struct GNUNET_CONFIGURATION_Handle *cfg;

  cfg = GNUNET_CONFIGURATION_create ();
  if (GNUNET_OK !=
      GNUNET_CONFIGURATION_load (cfg,
                                 config_filename))
  {
    GNUNET_break (0);
    GNUNET_CONFIGURATION_destroy (cfg);
    exit (77);
  }

  if (GNUNET_OK !=
      GNUNET_CONFIGURATION_get_value_string (cfg,
                                             "bank",
                                             "database",
                                             &database))
  {
    GNUNET_log_config_missing (GNUNET_ERROR_TYPE_WARNING,
                               "bank",
                               "database");
    GNUNET_break (0);
    GNUNET_CONFIGURATION_destroy (cfg);
    exit (77);
  }

  if (GNUNET_OK !=
      GNUNET_CONFIGURATION_get_value_string (cfg,
                                             "bank",
                                             "serve",
                                             &serve_cfg))
  {
    GNUNET_log_config_missing (GNUNET_ERROR_TYPE_WARNING,
                               "bank",
                               "serve");
    GNUNET_break (0);
    GNUNET_CONFIGURATION_destroy (cfg);
    GNUNET_free (database);
    exit (77);
  }
  GNUNET_CONFIGURATION_destroy (cfg);

  serve_arg = "serve-http";
  if (0 != strcmp ("http", serve_cfg))
    serve_arg = "serve-uwsgi";
  GNUNET_free (serve_cfg);
  bank_proc = GNUNET_OS_start_process
                (GNUNET_NO,
                GNUNET_OS_INHERIT_STD_ALL,
                NULL, NULL, NULL,
                "taler-bank-manage-testing",
                "taler-bank-manage-testing",
                config_filename,
                database,
                serve_arg, NULL);
  GNUNET_free (database);
  if (NULL == bank_proc)
  {
    BANK_FAIL ();
  }

  GNUNET_asprintf (&wget_cmd,
                   "wget -q -t 1 -T 1 %s -o /dev/null -O /dev/null",
                   bank_url);

  /* give child time to start and bind against the socket */
  fprintf (stderr,
           "Waiting for `taler-bank-manage' to be ready");
  iter = 0;
  do
  {
    if (10 == iter)
    {
      fprintf (
        stderr,
        "Failed to launch `taler-bank-manage' (or `wget')\n");
      GNUNET_OS_process_kill (bank_proc,
                              SIGTERM);
      GNUNET_OS_process_wait (bank_proc);
      GNUNET_OS_process_destroy (bank_proc);
      GNUNET_free (wget_cmd);
      BANK_FAIL ();
    }
    fprintf (stderr, ".");
    sleep (1);
    iter++;
  }
  while (0 != system (wget_cmd));
  GNUNET_free (wget_cmd);
  fprintf (stderr, "\n");

  return bank_proc;

}


/**
 * Prepare the bank execution.  Check if the port is available
 * and reset database.
 *
 * @param config_filename configuration file name.
 *
 * @return the base url, or NULL upon errors.  Must be freed
 *         by the caller.
 */
char *
TALER_TESTING_prepare_bank (const char *config_filename)
{
  struct GNUNET_CONFIGURATION_Handle *cfg;
  unsigned long long port;
  struct GNUNET_OS_Process *dbreset_proc;
  enum GNUNET_OS_ProcessStatusType type;
  unsigned long code;
  char *base_url;
  char *database;

  cfg = GNUNET_CONFIGURATION_create ();

  if (GNUNET_OK !=
      GNUNET_CONFIGURATION_load (cfg, config_filename))
  {
    GNUNET_CONFIGURATION_destroy (cfg);
    BANK_FAIL ();
  }
  if (GNUNET_OK !=
      GNUNET_CONFIGURATION_get_value_string (cfg,
                                             "bank",
                                             "DATABASE",
                                             &database))
  {
    GNUNET_log_config_missing (GNUNET_ERROR_TYPE_ERROR,
                               "bank",
                               "DATABASE");
    GNUNET_CONFIGURATION_destroy (cfg);
    BANK_FAIL ();
  }

  if (GNUNET_OK !=
      GNUNET_CONFIGURATION_get_value_number (cfg,
                                             "bank",
                                             "HTTP_PORT",
                                             &port))
  {
    GNUNET_log_config_missing (GNUNET_ERROR_TYPE_ERROR,
                               "bank",
                               "HTTP_PORT");
    GNUNET_CONFIGURATION_destroy (cfg);
    GNUNET_free (database);
    BANK_FAIL ();
  }
  GNUNET_CONFIGURATION_destroy (cfg);

  if (GNUNET_OK != GNUNET_NETWORK_test_port_free
        (IPPROTO_TCP, (uint16_t) port))
  {
    fprintf (stderr,
             "Required port %llu not available, skipping.\n",
             port);
    BANK_FAIL ();
  }

  /* DB preparation */
  if (NULL ==
      (dbreset_proc = GNUNET_OS_start_process (
         GNUNET_NO,
         GNUNET_OS_INHERIT_STD_ALL,
         NULL, NULL, NULL,
         "taler-bank-manage",
         "taler-bank-manage",
         "-c", "bank.conf",
         "--with-db", database,
         "django",
         "flush",
         "--no-input", NULL)))
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Failed to flush the bank db.\n");
    GNUNET_free (database);
    BANK_FAIL ();
  }
  GNUNET_free (database);

  if (GNUNET_SYSERR ==
      GNUNET_OS_process_wait_status (dbreset_proc,
                                     &type,
                                     &code))
  {
    GNUNET_OS_process_destroy (dbreset_proc);
    BANK_FAIL ();
  }
  if ( (type == GNUNET_OS_PROCESS_EXITED) &&
       (0 != code) )
  {
    fprintf (stderr,
             "Failed to setup database\n");
    BANK_FAIL ();
  }
  if ( (type != GNUNET_OS_PROCESS_EXITED) ||
       (0 != code) )
  {
    fprintf (stderr,
             "Unexpected error running"
             " `taler-bank-manage django flush..'!\n");
    BANK_FAIL ();
  }

  GNUNET_OS_process_destroy (dbreset_proc);

  GNUNET_asprintf (&base_url,
                   "http://localhost:%llu/",
                   port);
  return base_url;
}


/* end of testing_api_helpers.c */
