BEGIN;
SELECT _v.register_patch('exchange-0002', NULL, NULL);
SET search_path TO exchange;
CREATE DOMAIN gnunet_hashcode
  AS BYTEA
  CHECK(LENGTH(VALUE) = 32);
CREATE TYPE taler_amount
  AS
  (val INT8
  ,frac INT4
  );
COMMENT ON TYPE taler_amount
  IS 'Stores an amount, fraction is in units of 1/100000000 of the base value';
CREATE TYPE exchange_do_array_reserve_insert_return_type
  AS
  (transaction_duplicate BOOLEAN
  ,ruuid INT8
  );
COMMENT ON TYPE exchange_do_array_reserve_insert_return_type
  IS 'Return type for exchange_do_array_reserves_insert() stored procedure';
CREATE TYPE exchange_do_select_deposits_missing_wire_return_type
  AS
  (
    batch_deposit_serial_id INT8,
    total_amount taler_amount,
    wire_target_h_payto BYTEA,
    deadline INT8
  );
COMMENT ON TYPE exchange_do_select_deposits_missing_wire_return_type
  IS 'Return type for exchange_do_select_deposits_missing_wire';
CREATE TABLE denominations
  (denominations_serial BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,denom_pub_hash BYTEA PRIMARY KEY CHECK (LENGTH(denom_pub_hash)=64)
  ,denom_type INT4 NOT NULL DEFAULT (1) 
  ,age_mask INT4 NOT NULL DEFAULT (0)
  ,denom_pub BYTEA NOT NULL
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  ,valid_from INT8 NOT NULL
  ,expire_withdraw INT8 NOT NULL
  ,expire_deposit INT8 NOT NULL
  ,expire_legal INT8 NOT NULL
  ,coin taler_amount NOT NULL
  ,fee_withdraw taler_amount NOT NULL
  ,fee_deposit taler_amount NOT NULL
  ,fee_refresh taler_amount NOT NULL
  ,fee_refund taler_amount NOT NULL
  );
COMMENT ON TABLE denominations
  IS 'Main denominations table. All the valid denominations the exchange knows about.';
COMMENT ON COLUMN denominations.denom_type
  IS 'determines cipher type for blind signatures used with this denomination; 0 is for RSA';
COMMENT ON COLUMN denominations.age_mask
  IS 'bitmask with the age restrictions that are being used for this denomination; 0 if denomination does not support the use of age restrictions';
COMMENT ON COLUMN denominations.denominations_serial
  IS 'needed for exchange-auditor replication logic';
CREATE INDEX denominations_by_expire_legal_index
  ON denominations
  (expire_legal);
CREATE TABLE IF NOT EXISTS denomination_revocations
  (denom_revocations_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,denominations_serial INT8 PRIMARY KEY REFERENCES denominations (denominations_serial) ON DELETE CASCADE
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  );
COMMENT ON TABLE denomination_revocations
  IS 'remembering which denomination keys have been revoked';
CREATE FUNCTION create_table_wire_targets(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(wire_target_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',wire_target_h_payto BYTEA PRIMARY KEY CHECK (LENGTH(wire_target_h_payto)=32)'
      ',payto_uri TEXT NOT NULL'
    ') %s ;'
    ,'wire_targets'
    ,'PARTITION BY HASH (wire_target_h_payto)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'All senders and recipients of money via the exchange'
    ,'wire_targets'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Full payto URI. Can identify a regular bank account, or also be a URI identifying a reserve-account (for P2P payments)'
    ,'payto_uri'
    ,'wire_targets'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Unsalted hash of (full) payto_uri'
    ,'wire_target_h_payto'
    ,'wire_targets'
    ,partition_suffix
  );
END $$;
CREATE FUNCTION constrain_table_wire_targets(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wire_targets';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wire_target_serial_id_key'
    ' UNIQUE (wire_target_serial_id)'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('wire_targets'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('wire_targets'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE TABLE kyc_alerts
  (h_payto BYTEA PRIMARY KEY CHECK (LENGTH(h_payto)=32)
  ,trigger_type INT4 NOT NULL
  ,UNIQUE(trigger_type,h_payto)
  );
COMMENT ON TABLE kyc_alerts
  IS 'alerts about completed KYC events reliably notifying other components (even if they are not running)';
COMMENT ON COLUMN kyc_alerts.h_payto
  IS 'hash of the normalized payto://-URI for which the KYC status changed';
COMMENT ON COLUMN kyc_alerts.trigger_type
  IS 'identifies the receiver of the alert, as the same h_payto may require multiple components to be notified';
CREATE TABLE wire_fee
  (wire_fee_serial BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,wire_method TEXT NOT NULL
  ,start_date INT8 NOT NULL
  ,end_date INT8 NOT NULL
  ,wire_fee taler_amount NOT NULL
  ,closing_fee taler_amount NOT NULL
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  ,PRIMARY KEY (wire_method, start_date)
  );
COMMENT ON TABLE wire_fee
  IS 'list of the wire fees of this exchange, by date';
COMMENT ON COLUMN wire_fee.wire_fee_serial
  IS 'needed for exchange-auditor replication logic';
CREATE INDEX wire_fee_by_end_date_index
  ON wire_fee
  (end_date);
CREATE TABLE global_fee
  (global_fee_serial BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,start_date INT8 NOT NULL
  ,end_date INT8 NOT NULL
  ,history_fee taler_amount NOT NULL
  ,account_fee taler_amount NOT NULL
  ,purse_fee taler_amount NOT NULL
  ,purse_timeout INT8 NOT NULL
  ,history_expiration INT8 NOT NULL
  ,purse_account_limit INT4 NOT NULL
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  ,PRIMARY KEY (start_date)
  );
COMMENT ON TABLE global_fee
  IS 'list of the global fees of this exchange, by date';
COMMENT ON COLUMN global_fee.global_fee_serial
  IS 'needed for exchange-auditor replication logic';
CREATE INDEX global_fee_by_end_date_index
  ON global_fee
  (end_date);
CREATE TABLE wire_accounts
  (payto_uri TEXT PRIMARY KEY
  ,master_sig BYTEA CHECK (LENGTH(master_sig)=64)
  ,is_active BOOLEAN NOT NULL
  ,last_change INT8 NOT NULL
  ,conversion_url TEXT DEFAULT (NULL)
  ,debit_restrictions TEXT DEFAULT (NULL)
  ,credit_restrictions TEXT DEFAULT (NULL)
  );
COMMENT ON TABLE wire_accounts
  IS 'Table with current and historic bank accounts of the exchange. Entries never expire as we need to remember the last_change column indefinitely.';
COMMENT ON COLUMN wire_accounts.payto_uri
  IS 'payto URI (RFC 8905) with the bank account of the exchange.';
COMMENT ON COLUMN wire_accounts.master_sig
  IS 'Signature of purpose TALER_SIGNATURE_MASTER_WIRE_DETAILS';
COMMENT ON COLUMN wire_accounts.is_active
  IS 'true if we are currently supporting the use of this account.';
COMMENT ON COLUMN wire_accounts.last_change
  IS 'Latest time when active status changed. Used to detect replays of old messages.';
COMMENT ON COLUMN wire_accounts.conversion_url
  IS 'URL of a currency conversion service if conversion is needed when this account is used; NULL if there is no conversion.';
COMMENT ON COLUMN wire_accounts.debit_restrictions
  IS 'JSON array describing restrictions imposed when debiting this account. Empty for no restrictions, NULL if account was migrated from previous database revision or account is disabled.';
COMMENT ON COLUMN wire_accounts.credit_restrictions
  IS 'JSON array describing restrictions imposed when crediting this account. Empty for no restrictions, NULL if account was migrated from previous database revision or account is disabled.';
CREATE TABLE auditors
  (auditor_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,auditor_pub BYTEA PRIMARY KEY CHECK (LENGTH(auditor_pub)=32)
  ,auditor_name TEXT NOT NULL
  ,auditor_url TEXT NOT NULL
  ,is_active BOOLEAN NOT NULL
  ,last_change INT8 NOT NULL
  );
COMMENT ON TABLE auditors
  IS 'Table with auditors the exchange uses or has used in the past. Entries never expire as we need to remember the last_change column indefinitely.';
COMMENT ON COLUMN auditors.auditor_pub
  IS 'Public key of the auditor.';
COMMENT ON COLUMN auditors.auditor_url
  IS 'The base URL of the auditor.';
COMMENT ON COLUMN auditors.is_active
  IS 'true if we are currently supporting the use of this auditor.';
COMMENT ON COLUMN auditors.last_change
  IS 'Latest time when active status changed. Used to detect replays of old messages.';
CREATE TABLE auditor_denom_sigs
  (auditor_denom_serial BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,auditor_uuid INT8 NOT NULL REFERENCES auditors (auditor_uuid) ON DELETE CASCADE
  ,denominations_serial INT8 NOT NULL REFERENCES denominations (denominations_serial) ON DELETE CASCADE
  ,auditor_sig BYTEA CHECK (LENGTH(auditor_sig)=64)
  ,PRIMARY KEY (denominations_serial, auditor_uuid)
  );
COMMENT ON TABLE auditor_denom_sigs
  IS 'Table with auditor signatures on exchange denomination keys.';
COMMENT ON COLUMN auditor_denom_sigs.auditor_uuid
  IS 'Identifies the auditor.';
COMMENT ON COLUMN auditor_denom_sigs.denominations_serial
  IS 'Denomination the signature is for.';
COMMENT ON COLUMN auditor_denom_sigs.auditor_sig
  IS 'Signature of the auditor, of purpose TALER_SIGNATURE_AUDITOR_EXCHANGE_KEYS.';
CREATE TABLE exchange_sign_keys
  (esk_serial BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,exchange_pub BYTEA PRIMARY KEY CHECK (LENGTH(exchange_pub)=32)
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  ,valid_from INT8 NOT NULL
  ,expire_sign INT8 NOT NULL
  ,expire_legal INT8 NOT NULL
  );
COMMENT ON TABLE exchange_sign_keys
  IS 'Table with master public key signatures on exchange online signing keys.';
COMMENT ON COLUMN exchange_sign_keys.exchange_pub
  IS 'Public online signing key of the exchange.';
COMMENT ON COLUMN exchange_sign_keys.master_sig
  IS 'Signature affirming the validity of the signing key of purpose TALER_SIGNATURE_MASTER_SIGNING_KEY_VALIDITY.';
COMMENT ON COLUMN exchange_sign_keys.valid_from
  IS 'Time when this online signing key will first be used to sign messages.';
COMMENT ON COLUMN exchange_sign_keys.expire_sign
  IS 'Time when this online signing key will no longer be used to sign.';
COMMENT ON COLUMN exchange_sign_keys.expire_legal
  IS 'Time when this online signing key legally expires.';
CREATE TABLE signkey_revocations
  (signkey_revocations_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,esk_serial INT8 PRIMARY KEY REFERENCES exchange_sign_keys (esk_serial) ON DELETE CASCADE
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  );
COMMENT ON TABLE signkey_revocations
  IS 'Table storing which online signing keys have been revoked';
CREATE TABLE extensions
  (extension_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,name TEXT NOT NULL UNIQUE
  ,manifest BYTEA
  );
COMMENT ON TABLE extensions
  IS 'Configurations of the activated extensions';
COMMENT ON COLUMN extensions.name
  IS 'Name of the extension';
COMMENT ON COLUMN extensions.manifest
  IS 'Manifest of the extension as JSON-blob, maybe NULL.  It contains common meta-information and extension-specific configuration.';
CREATE FUNCTION create_table_policy_fulfillments(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'policy_fulfillments';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(h_fulfillment_proof gnunet_hashcode PRIMARY KEY'
      ',fulfillment_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',fulfillment_timestamp INT8 NOT NULL'
      ',fulfillment_proof TEXT'
      ',policy_hash_codes gnunet_hashcode[] NOT NULL'
      ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (h_fulfillment_proof)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
    'Proofs of fulfillment of policies that were set in deposits'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'Timestamp of the arrival of a proof of fulfillment'
    ,'fulfillment_timestamp'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'JSON object with a proof of the fulfillment of a policy. Supported details depend on the policy extensions supported by the exchange.'
    ,'fulfillment_proof'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'Hash of the fulfillment_proof'
    ,'h_fulfillment_proof'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'Array of the policy_hash_code''s of all policy_details that are fulfilled by this proof'
    ,'policy_hash_codes'
    ,table_name
    ,partition_suffix
  );
END
$$;
COMMENT ON FUNCTION create_table_policy_fulfillments
  IS 'Creates the policy_fulfillments table';
CREATE FUNCTION constrain_table_policy_fulfillments(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  partition_name TEXT;
BEGIN
  partition_name = concat_ws('_', 'policy_fulfillments', partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || partition_name ||
    ' ADD CONSTRAINT ' || partition_name || '_serial_id '
    ' UNIQUE (h_fulfillment_proof, fulfillment_id)'
  );
END
$$;
INSERT INTO exchange_tables
  (name
  ,version
  ,action
  ,partitioned
  ,by_range)
VALUES
  ('policy_fulfillments', 'exchange-0002', 'create', TRUE ,FALSE),
  ('policy_fulfillments', 'exchange-0002', 'constrain', TRUE ,FALSE);
CREATE FUNCTION create_table_policy_details(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'policy_details';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(policy_details_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',policy_hash_code gnunet_hashcode NOT NULL'
      ',policy_json TEXT NOT NULL'
      ',deadline INT8 NOT NULL'
      ',commitment taler_amount NOT NULL'
      ',accumulated_total taler_amount NOT NULL'
      ',fee taler_amount NOT NULL'
      ',transferable taler_amount NOT NULL'
      ',fulfillment_state SMALLINT NOT NULL CHECK(fulfillment_state between 0 and 5)'
      ',h_fulfillment_proof gnunet_hashcode'
    ') %s;'
    ,table_name
    ,'PARTITION BY HASH (h_fulfillment_proof)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
    'Policies that were provided with deposits via policy extensions.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'ID (GNUNET_HashCode) that identifies a policy.  Will be calculated by the policy extension based on the content'
    ,'policy_hash_code'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'JSON object with options set that the exchange needs to consider when executing a deposit. Supported details depend on the policy extensions supported by the exchange.'
    ,'policy_json'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'Deadline until the policy must be marked as fulfilled (maybe "forever")'
    ,'deadline'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'The amount that this policy commits to.  Invariant: commitment >= fee'
    ,'commitment'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'The sum of all contributions of all deposit that reference this policy.  Invariant: The fulfilment_state must be Insufficient as long as accumulated_total < commitment'
    ,'accumulated_total'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'The fee for this policy, due when the policy is fulfilled or timed out'
    ,'fee'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'The amount that on fulfillment or timeout will be transferred to the payto-URI''s of the corresponding deposit''s.  The policy fees must have been already deducted from it.  Invariant: fee+transferable <= accumulated_total.  The remaining amount (accumulated_total - fee - transferable) can be refreshed by the owner of the coins when the state is Timeout or Success.'
    ,'transferable'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'State of the fulfillment:
       - 0 (Failure)
       - 1 (Insufficient)
       - 2 (Ready)
       - 4 (Success)
       - 5 (Timeout)'
    ,'fulfillment_state'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'Reference to the proof of the fulfillment of this policy, if it exists.  Invariant: If not NULL, this entry''s .hash_code MUST be part of the corresponding policy_fulfillments.policy_hash_codes array.'
    ,'h_fulfillment_proof'
    ,table_name
    ,partition_suffix
  );
END
$$;
COMMENT ON FUNCTION create_table_policy_details
  IS 'Creates the policy_details table';
CREATE FUNCTION constrain_table_policy_details(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  partition_name TEXT;
BEGIN
  partition_name = concat_ws('_', 'policy_details', partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || partition_name ||
    ' ADD CONSTRAINT ' || partition_name || '_unique_serial_id '
    ' UNIQUE (policy_details_serial_id)'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || partition_name ||
    ' ADD CONSTRAINT ' || partition_name || '_unique_hash_fulfillment_proof '
    ' UNIQUE (policy_hash_code, h_fulfillment_proof)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || partition_name || '_policy_hash_code'
    ' ON ' || partition_name ||
    ' (policy_hash_code);'
  );
END
$$;
CREATE OR REPLACE FUNCTION foreign_table_policy_details()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'policy_details';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_policy_fulfillments'
    ' FOREIGN KEY (h_fulfillment_proof) '
    ' REFERENCES policy_fulfillments (h_fulfillment_proof) ON DELETE RESTRICT'
  );
END
$$;
INSERT INTO exchange_tables
  (name
  ,version
  ,action
  ,partitioned
  ,by_range)
VALUES
  ('policy_details', 'exchange-0002', 'create', TRUE ,FALSE),
  ('policy_details', 'exchange-0002', 'constrain', TRUE ,FALSE),
  ('policy_details', 'exchange-0002', 'foreign', TRUE ,FALSE);
CREATE TABLE profit_drains
  (profit_drain_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,wtid BYTEA PRIMARY KEY CHECK (LENGTH(wtid)=32)
  ,account_section TEXT NOT NULL
  ,payto_uri TEXT NOT NULL
  ,trigger_date INT8 NOT NULL
  ,amount taler_amount NOT NULL
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  ,executed BOOLEAN NOT NULL DEFAULT FALSE
  );
COMMENT ON TABLE profit_drains
  IS 'transactions to be performed to move profits from the escrow account of the exchange to a regular account';
COMMENT ON COLUMN profit_drains.wtid
  IS 'randomly chosen nonce, unique to prevent double-submission';
COMMENT ON COLUMN profit_drains.account_section
  IS 'specifies the configuration section in the taler-exchange-drain configuration with the wire account to drain';
COMMENT ON COLUMN profit_drains.payto_uri
  IS 'specifies the account to be credited';
COMMENT ON COLUMN profit_drains.trigger_date
  IS 'set by taler-exchange-offline at the time of making the signature; not necessarily the exact date of execution of the wire transfer, just for orientation';
COMMENT ON COLUMN profit_drains.amount
  IS 'amount to be transferred';
COMMENT ON COLUMN profit_drains.master_sig
  IS 'EdDSA signature of type TALER_SIGNATURE_MASTER_DRAIN_PROFIT';
COMMENT ON COLUMN profit_drains.executed
  IS 'set to TRUE by taler-exchange-drain on execution of the transaction, not replicated to auditor';
CREATE FUNCTION create_table_legitimization_processes(
  IN shard_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(legitimization_process_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_payto BYTEA NOT NULL CHECK (LENGTH(h_payto)=32)'
      ',start_time INT8 NOT NULL'
      ',expiration_time INT8 NOT NULL DEFAULT (0)'
      ',provider_section TEXT NOT NULL'
      ',provider_user_id TEXT DEFAULT NULL'
      ',provider_legitimization_id TEXT DEFAULT NULL'
      ',redirect_url TEXT DEFAULT NULL'
      ',finished BOOLEAN DEFAULT (FALSE)'
    ') %s ;'
    ,'legitimization_processes'
    ,'PARTITION BY HASH (h_payto)'
    ,shard_suffix
  );
  PERFORM comment_partitioned_table(
    'List of legitimization processes (ongoing and completed) by account and provider'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'unique ID for this legitimization process at the exchange'
    ,'legitimization_process_serial_id'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'hash over the normalized payto URI; foreign key linking the entry to the wire_targets table, NOT a primary key (multiple legitimizations are possible per wire target)'
    ,'h_payto'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'time when the KYC check was initiated, useful for garbage collection (absolute time, not rounded)'
    ,'start_time'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'URL where the user should go to begin the KYC process'
    ,'redirect_url'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'in the future if the respective KYC check was passed successfully; an absolute time (not rounded)'
    ,'expiration_time'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Configuration file section with details about this provider'
    ,'provider_section'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifier for the user at the provider that was used for the legitimization. NULL if provider is unaware.'
    ,'provider_user_id'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifier for the specific legitimization process at the provider. NULL if legitimization was not started.'
    ,'provider_legitimization_id'
    ,'legitimization_processes'
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Set to TRUE when the specific legitimization process is finished.'
    ,'finished'
    ,'legitimization_processes'
    ,shard_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_legitimization_processes(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  partition_name TEXT;
BEGIN
  partition_name = concat_ws('_', 'legitimization_processes', partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || partition_name
    || ' '
      'ADD CONSTRAINT ' || partition_name || '_serial_key '
        'UNIQUE (legitimization_process_serial_id)');
  EXECUTE FORMAT (
    'CREATE INDEX IF NOT EXISTS ' || partition_name || '_by_provider_and_legi_index '
        'ON '|| partition_name || ' '
        '(provider_section,provider_legitimization_id)'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || partition_name || '_by_provider_and_legi_index '
    'IS ' || quote_literal('used (rarely) in kyc_provider_account_lookup') || ';'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('legitimization_processes'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('legitimization_processes'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_legitimization_requirements(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(legitimization_requirement_serial_id INT8 GENERATED BY DEFAULT AS IDENTITY'
      ',h_payto BYTEA NOT NULL CHECK (LENGTH(h_payto)=32)'
      ',reserve_pub BYTEA'
      ',required_checks TEXT NOT NULL'
      ',UNIQUE (h_payto, required_checks)'
    ') %s ;'
    ,'legitimization_requirements'
    ,'PARTITION BY HASH (h_payto)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'List of required legitimizations by account'
    ,'legitimization_requirements'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'unique ID for this legitimization requirement at the exchange'
    ,'legitimization_requirement_serial_id'
    ,'legitimization_requirements'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'foreign key linking the entry to the wire_targets table, NOT a primary key (multiple legitimizations are possible per wire target)'
    ,'h_payto'
    ,'legitimization_requirements'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'if h_payto refers to a reserve, this is its public key, NULL otherwise.  It allows to lookup the corresponding reserve when the KYC process is done.'
    ,'reserve_pub'
    ,'legitimization_requirements'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'space-separated list of required checks'
    ,'required_checks'
    ,'legitimization_requirements'
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_legitimization_requirements(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  partition_name TEXT;
BEGIN
  partition_name = concat_ws('_', 'legitimization_requirements', partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || partition_name || ' '
    'ADD CONSTRAINT ' || partition_name || '_serial_id_key '
    'UNIQUE (legitimization_requirement_serial_id)');
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('legitimization_requirements'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('legitimization_requirements'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_reserves(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'reserves';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(reserve_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA PRIMARY KEY CHECK(LENGTH(reserve_pub)=32)'
      ',current_balance taler_amount NOT NULL DEFAULT (0, 0)'
      ',purses_active INT8 NOT NULL DEFAULT(0)'
      ',purses_allowed INT8 NOT NULL DEFAULT(0)'
      ',birthday INT4 NOT NULL DEFAULT(0)'
      ',expiration_date INT8 NOT NULL'
      ',gc_date INT8 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Summarizes the balance of a reserve. Updated when new funds are added or withdrawn.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'EdDSA public key of the reserve. Knowledge of the private key implies ownership over the balance.'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Current balance remaining with the reserve.'
    ,'current_balance'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Number of purses that were created by this reserve that are not expired and not fully paid.'
    ,'purses_active'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Number of purses that this reserve is allowed to have active at most.'
    ,'purses_allowed'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Used to trigger closing of reserves that have not been drained after some time'
    ,'expiration_date'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Used to forget all information about a reserve during garbage collection'
    ,'gc_date'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Birthday of the user in days after 1970, or 0 if user is an adult and is not subject to age restrictions'
    ,'birthday'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_reserves(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'reserves';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_unique_uuid'
    ' UNIQUE (reserve_uuid)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_expiration_index '
    'ON ' || table_name || ' '
    '(expiration_date'
    ',current_balance'
    ');'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_expiration_index '
    'IS ' || quote_literal('used in get_expired_reserves') || ';'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve_uuid_index '
    'ON ' || table_name || ' '
    '(reserve_uuid);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_gc_date_index '
    'ON ' || table_name || ' '
    '(gc_date);'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_gc_date_index '
    'IS ' || quote_literal('for reserve garbage collection') || ';'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_reserve_history (
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'reserve_history';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(reserve_history_serial_id INT8 GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL CHECK (LENGTH(reserve_pub)=32)'
      ',table_name TEXT NOT NULL'
      ',serial_id INT8 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
    'Links to tables with entries that affected the transaction history of a reserve.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'For which reserve is this a history entry'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'In which table is the history entry'
    ,'table_name'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Which is the generated serial ID of the entry in the table'
    ,'serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Monotonic counter, used to generate Etags for caching'
    ,'reserve_history_serial_id'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_reserve_history(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'reserve_history';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_reserve_history_serial_id_pkey'
    ' PRIMARY KEY (reserve_history_serial_id) '
    ',ADD CONSTRAINT ' || table_name || '_reserve_entry_key'
    ' UNIQUE (reserve_pub, table_name, serial_id)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_reserve_by_time'
    ' ON ' || table_name || ' '
    '(reserve_pub'
    ',reserve_history_serial_id DESC'
    ');'
  );
END
$$;
CREATE FUNCTION foreign_table_reserve_history()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'reserve_history';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub) '
    ' REFERENCES reserves (reserve_pub) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserve_history'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserve_history'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserve_history'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE)
    ;
CREATE FUNCTION create_table_reserves_in(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_in';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(reserve_in_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA PRIMARY KEY'
      ',wire_reference INT8 NOT NULL'
      ',credit taler_amount NOT NULL'
      ',wire_source_h_payto BYTEA CHECK (LENGTH(wire_source_h_payto)=32)'
      ',exchange_account_section TEXT NOT NULL'
      ',execution_date INT8 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'list of transfers of funds into the reserves, one per incoming wire transfer'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the debited bank account and KYC status'
    ,'wire_source_h_payto'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the reserve. Private key signifies ownership of the remaining balance.'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Amount that was transferred into the reserve'
    ,'credit'
    ,table_name
    ,partition_suffix
  );
END $$;
CREATE FUNCTION constrain_table_reserves_in(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_in';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_reserve_in_serial_id_key'
    ' UNIQUE (reserve_in_serial_id)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve_in_serial_id_index '
    'ON ' || table_name || ' '
    '(reserve_in_serial_id);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_exch_accnt_reserve_in_serial_id_idx '
    'ON ' || table_name || ' '
    '(exchange_account_section'
    ',reserve_in_serial_id ASC'
    ');'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_exch_accnt_reserve_in_serial_id_idx '
    'IS ' || quote_literal ('for pg_select_reserves_in_above_serial_id_by_account') || ';'
  );
END
$$;
CREATE FUNCTION foreign_table_reserves_in()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'reserves_in';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub) '
    ' REFERENCES reserves(reserve_pub) ON DELETE CASCADE'
  );
END $$;
CREATE OR REPLACE FUNCTION reserves_in_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO reserve_history
    (reserve_pub
    ,table_name
    ,serial_id)
  VALUES
    (NEW.reserve_pub
    ,'reserves_in'
    ,NEW.reserve_in_serial_id);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION reserves_in_insert_trigger()
  IS 'Automatically generate reserve history entry.';
CREATE FUNCTION master_table_reserves_in()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER reserves_in_on_insert
    AFTER INSERT
     ON reserves_in
     FOR EACH ROW EXECUTE FUNCTION reserves_in_insert_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves_in'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_in'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserves_in'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('reserves_in'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_reserves_close(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_close';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(close_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL'
      ',execution_date INT8 NOT NULL'
      ',wtid BYTEA NOT NULL CHECK (LENGTH(wtid)=32)'
      ',wire_target_h_payto BYTEA CHECK (LENGTH(wire_target_h_payto)=32)'
      ',amount taler_amount NOT NULL'
      ',closing_fee taler_amount NOT NULL'
      ',close_request_row INT8 NOT NULL DEFAULT(0)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'wire transfers executed by the reserve to close reserves'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the credited bank account (and KYC status). Note that closing does not depend on KYC.'
    ,'wire_target_h_payto'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_reserves_close(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_close';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_close_uuid_pkey'
    ' PRIMARY KEY (close_uuid)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve_pub_index '
    'ON ' || table_name || ' (reserve_pub);'
  );
END $$;
CREATE FUNCTION foreign_table_reserves_close()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_close';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub)'
    ' REFERENCES reserves (reserve_pub) ON DELETE CASCADE'
  );
END $$;
CREATE OR REPLACE FUNCTION reserves_close_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO reserve_history
    (reserve_pub
    ,table_name
    ,serial_id)
  VALUES
    (NEW.reserve_pub
    ,'reserves_close'
    ,NEW.close_uuid);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION reserves_close_insert_trigger()
  IS 'Automatically generate reserve history entry.';
CREATE FUNCTION master_table_reserves_close()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER reserves_close_on_insert
    AFTER INSERT
     ON reserves_close
     FOR EACH ROW EXECUTE FUNCTION reserves_close_insert_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves_close'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_close'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserves_close'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('reserves_close'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_close_requests(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'close_requests';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(close_request_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL CHECK (LENGTH(reserve_pub)=32)'
      ',close_timestamp INT8 NOT NULL'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',close taler_amount NOT NULL'
      ',close_fee taler_amount NOT NULL'
      ',payto_uri TEXT NOT NULL'
      ',done BOOL NOT NULL DEFAULT(FALSE)'
      ',PRIMARY KEY (reserve_pub,close_timestamp)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Explicit requests by a reserve owner to close a reserve immediately'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'When the request was created by the client'
    ,'close_timestamp'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature affirming that the reserve is to be closed'
    ,'reserve_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Balance of the reserve at the time of closing, to be wired to the associated bank account (minus the closing fee)'
    ,'close'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the credited bank account. Optional.'
    ,'payto_uri'
    ,table_name
    ,partition_suffix
  );
END $$;
CREATE FUNCTION constrain_table_close_requests(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'close_requests';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_close_request_uuid_index '
    'ON ' || table_name || ' '
    '(close_request_serial_id);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_close_request_done_index '
    'ON ' || table_name || ' '
    '(done);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_close_request_uuid_pkey'
    ' UNIQUE (close_request_serial_id)'
  );
END
$$;
CREATE FUNCTION foreign_table_close_requests()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'close_requests';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub) '
    ' REFERENCES reserves(reserve_pub) ON DELETE CASCADE'
  );
END
$$;
CREATE OR REPLACE FUNCTION close_requests_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO reserve_history
    (reserve_pub
    ,table_name
    ,serial_id)
  VALUES
    (NEW.reserve_pub
    ,'close_requests'
    ,NEW.close_request_serial_id);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION close_requests_insert_trigger()
  IS 'Automatically generate reserve history entry.';
CREATE FUNCTION master_table_close_requests()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER close_requests_on_insert
    AFTER INSERT
     ON close_requests
     FOR EACH ROW EXECUTE FUNCTION close_requests_insert_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('close_requests'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('close_requests'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('close_requests'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('close_requests'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_reserves_open_deposits(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_open_deposits';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(reserve_open_deposit_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',reserve_pub BYTEA NOT NULL CHECK (LENGTH(reserve_pub)=32)'
      ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)'
      ',coin_sig BYTEA NOT NULL CHECK (LENGTH(coin_sig)=64)'
      ',contribution taler_amount NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'coin contributions paying for a reserve to remain open'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the specific reserve being paid for (possibly together with reserve_sig).'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_reserves_open_deposits(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_open_deposits';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name || ' '
      'ADD CONSTRAINT ' || table_name || '_coin_unique '
        'PRIMARY KEY (coin_pub,coin_sig)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_uuid '
    'ON ' || table_name || ' '
    '(reserve_open_deposit_uuid);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve '
    'ON ' || table_name || ' '
    '(reserve_pub);'
  );
END
$$;
CREATE OR REPLACE FUNCTION reserves_open_deposits_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO exchange.coin_history
    (coin_pub
    ,table_name
    ,serial_id)
 VALUES
     (NEW.coin_pub
    ,'reserves_open_deposits'
    ,NEW.reserve_open_deposit_uuid);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION reserves_open_deposits_insert_trigger()
  IS 'Automatically generate coin history entry.';
CREATE FUNCTION master_table_reserves_open_deposits()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER reserves_open_deposits_on_insert
    AFTER INSERT
     ON reserves_open_deposits
     FOR EACH ROW EXECUTE FUNCTION reserves_open_deposits_insert_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves_open_deposits'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_open_deposits'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserves_open_deposits'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_reserves_open_requests(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_open_requests';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(open_request_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL'
      ',request_timestamp INT8 NOT NULL'
      ',expiration_date INT8 NOT NULL'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',reserve_payment taler_amount NOT NULL'
      ',requested_purse_limit INT4 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table (
     'requests to keep a reserve open'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column (
     'Fee to pay for the request from the reserve balance itself.'
    ,'reserve_payment'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_reserves_open_requests(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_open_requests';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_by_uuid'
    ' PRIMARY KEY (open_request_uuid)'
    ',ADD CONSTRAINT ' || table_name || '_by_time'
    ' UNIQUE (reserve_pub,request_timestamp)'
  );
END
$$;
CREATE FUNCTION foreign_table_reserves_open_requests()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_open_requests';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub '
    ' FOREIGN KEY (reserve_pub)'
    ' REFERENCES reserves (reserve_pub) ON DELETE CASCADE'
  );
END
$$;
CREATE OR REPLACE FUNCTION reserves_open_requests_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO reserve_history
    (reserve_pub
    ,table_name
    ,serial_id)
  VALUES
    (NEW.reserve_pub
    ,'reserves_open_requests'
    ,NEW.open_request_uuid);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION reserves_open_requests_insert_trigger()
  IS 'Automatically generate reserve history entry.';
CREATE FUNCTION master_table_reserves_open_requests()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER reserves_open_requests_on_insert
    AFTER INSERT
     ON reserves_open_requests
     FOR EACH ROW EXECUTE FUNCTION reserves_open_requests_insert_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves_open_requests'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_open_requests'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserves_open_requests'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('reserves_open_requests'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_reserves_out(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_out';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(reserve_out_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_blind_ev BYTEA CHECK (LENGTH(h_blind_ev)=64) UNIQUE'
      ',denominations_serial INT8 NOT NULL'
      ',denom_sig BYTEA NOT NULL'
      ',reserve_uuid INT8 NOT NULL'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',execution_date INT8 NOT NULL'
      ',amount_with_fee taler_amount NOT NULL'
    ') %s ;'
    ,'reserves_out'
    ,'PARTITION BY HASH (h_blind_ev)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table (
     'Withdraw operations performed on reserves.'
    ,'reserves_out'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column (
     'Hash of the blinded coin, used as primary key here so that broken clients that use a non-random coin or blinding factor fail to withdraw (otherwise they would fail on deposit when the coin is not unique there).'
    ,'h_blind_ev'
    ,'reserves_out'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column (
     'We do not CASCADE ON DELETE for the foreign constrain here, as we may keep the denomination data alive'
    ,'denominations_serial'
    ,'reserves_out'
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_reserves_out(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_out';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_reserve_out_serial_id_key'
    ' UNIQUE (reserve_out_serial_id)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve_uuid_and_execution_date_index '
    'ON ' || table_name || ' '
    '(reserve_uuid, execution_date);'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_reserve_uuid_and_execution_date_index '
    'IS ' || quote_literal('for do_gc, do_recoup_by_reserve, select_kyc_relevant_withdraw_events and a few others') || ';'
  );
END
$$;
CREATE FUNCTION foreign_table_reserves_out()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_out';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_denom'
    ' FOREIGN KEY (denominations_serial)'
    ' REFERENCES denominations (denominations_serial)'
    ',ADD CONSTRAINT ' || table_name || '_foreign_reserve '
    ' FOREIGN KEY (reserve_uuid)'
    ' REFERENCES reserves (reserve_uuid) ON DELETE CASCADE'
  );
END
$$;
CREATE FUNCTION reserves_out_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO reserve_history
    (reserve_pub
    ,table_name
    ,serial_id)
  SELECT
     res.reserve_pub
    ,'reserves_out'
    ,NEW.reserve_out_serial_id
  FROM
    reserves res
  WHERE res.reserve_uuid = NEW.reserve_uuid;
  RETURN NEW;
END $$;
COMMENT ON FUNCTION reserves_out_insert_trigger()
  IS 'Replicate reserve_out inserts into reserve_history table.';
CREATE FUNCTION master_table_reserves_out()
RETURNS void
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER reserves_out_on_insert
  AFTER INSERT
   ON reserves_out
   FOR EACH ROW EXECUTE FUNCTION reserves_out_insert_trigger();
END $$;
COMMENT ON FUNCTION master_table_reserves_out()
  IS 'Setup triggers to replicate reserve_out into reserve_history.';
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves_out'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_out'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserves_out'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('reserves_out'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_known_coins(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'known_coins';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(known_coin_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',denominations_serial INT8 NOT NULL'
      ',coin_pub BYTEA NOT NULL PRIMARY KEY CHECK (LENGTH(coin_pub)=32)'
      ',age_commitment_hash BYTEA CHECK (LENGTH(age_commitment_hash)=32)'
      ',denom_sig BYTEA NOT NULL'
      ',remaining taler_amount NOT NULL DEFAULT(0,0)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'information about coins and their signatures, so we do not have to store the signatures more than once if a coin is involved in multiple operations'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Denomination of the coin, determines the value of the original coin and applicable fees for coin-specific operations.'
    ,'denominations_serial'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'EdDSA public key of the coin'
    ,'coin_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Value of the coin that remains to be spent'
    ,'remaining'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Optional hash of the age commitment for age restrictions as per DD 24 (active if denom_type has the respective bit set)'
    ,'age_commitment_hash'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'This is the signature of the exchange that affirms that the coin is a valid coin. The specific signature type depends on denom_type of the denomination.'
    ,'denom_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_known_coins(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'known_coins';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_known_coin_id_key'
    ' UNIQUE (known_coin_id)'
  );
END
$$;
CREATE FUNCTION foreign_table_known_coins()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'known_coins';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_denominations'
    ' FOREIGN KEY (denominations_serial) '
    ' REFERENCES denominations (denominations_serial) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('known_coins'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('known_coins'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('known_coins'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_coin_history (
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'coin_history';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(coin_history_serial_id INT8 GENERATED BY DEFAULT AS IDENTITY'
      ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)'
      ',table_name TEXT NOT NULL'
      ',serial_id INT8 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
    'Links to tables with entries that affected the transaction history of a coin.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'For which coin is this a history entry'
    ,'coin_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'In which table is the history entry'
    ,'table_name'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Which is the generated serial ID of the entry in the table'
    ,'serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Monotonic counter, used to generate Etags for caching'
    ,'coin_history_serial_id'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_coin_history(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'coin_history';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_coin_history_serial_id_pkey'
    ' PRIMARY KEY (coin_history_serial_id) '
    ',ADD CONSTRAINT ' || table_name || '_coin_entry_key'
    ' UNIQUE (coin_pub, table_name, serial_id)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_coin_by_time'
    ' ON ' || table_name || ' '
    '(coin_pub'
    ',coin_history_serial_id DESC'
    ');'
  );
END
$$;
CREATE FUNCTION foreign_table_coin_history()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'coin_history';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (coin_pub) '
    ' REFERENCES known_coins (coin_pub) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('coin_history'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('coin_history'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('coin_history'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE)
    ;
CREATE FUNCTION create_table_refresh_commitments(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_commitments';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(melt_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',rc BYTEA PRIMARY KEY CHECK (LENGTH(rc)=64)'
      ',old_coin_pub BYTEA NOT NULL'
      ',old_coin_sig BYTEA NOT NULL CHECK(LENGTH(old_coin_sig)=64)'
      ',amount_with_fee taler_amount NOT NULL'
      ',noreveal_index INT4 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (rc)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Commitments made when melting coins and the gamma value chosen by the exchange.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'The gamma value chosen by the exchange in the cut-and-choose protocol'
    ,'noreveal_index'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Commitment made by the client, hash over the various client inputs in the cut-and-choose protocol'
    ,'rc'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Coin being melted in the refresh process.'
    ,'old_coin_pub'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_refresh_commitments(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_commitments';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_old_coin_pub_index '
    'ON ' || table_name || ' '
    '(old_coin_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_melt_serial_id_key'
    ' UNIQUE (melt_serial_id)'
  );
END
$$;
CREATE FUNCTION foreign_table_refresh_commitments()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_commitments';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (old_coin_pub) '
    ' REFERENCES known_coins (coin_pub) ON DELETE CASCADE'
  );
END
$$;
CREATE OR REPLACE FUNCTION refresh_commitments_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO exchange.coin_history
    (coin_pub
    ,table_name
    ,serial_id)
 VALUES
     (NEW.old_coin_pub
    ,'refresh_commitments'
    ,NEW.melt_serial_id);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION refresh_commitments_insert_trigger()
  IS 'Automatically generate coin history entry.';
CREATE FUNCTION master_table_refresh_commitments()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER refresh_commitments_on_insert
    AFTER INSERT
     ON refresh_commitments
     FOR EACH ROW EXECUTE FUNCTION refresh_commitments_insert_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('refresh_commitments'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('refresh_commitments'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('refresh_commitments'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
   ('refresh_commitments'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_refresh_revealed_coins(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_revealed_coins';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(rrc_serial BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',melt_serial_id INT8 NOT NULL'
      ',freshcoin_index INT4 NOT NULL'
      ',link_sig BYTEA NOT NULL CHECK(LENGTH(link_sig)=64)'
      ',denominations_serial INT8 NOT NULL'
      ',coin_ev BYTEA NOT NULL'
      ',h_coin_ev BYTEA NOT NULL CHECK(LENGTH(h_coin_ev)=64)'
      ',ev_sig BYTEA NOT NULL'
      ',ewv BYTEA NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (melt_serial_id)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Revelations about the new coins that are to be created during a melting session.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'needed for exchange-auditor replication logic'
    ,'rrc_serial'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the refresh commitment (rc) of the melt operation.'
    ,'melt_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'index of the fresh coin being created (one melt operation may result in multiple fresh coins)'
    ,'freshcoin_index'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature of type WALLET_COIN_LINK, proves exchange did not tamper with the link data'
    ,'link_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'envelope of the new coin to be signed'
    ,'coin_ev'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'exchange contributed values in the creation of the fresh coin (see /csr)'
    ,'ewv'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash of the envelope of the new coin to be signed (for lookups)'
    ,'h_coin_ev'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'exchange signature over the envelope'
    ,'ev_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_refresh_revealed_coins(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_revealed_coins';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_coins_by_melt_serial_id_index '
    'ON ' || table_name || ' '
    '(melt_serial_id);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_rrc_serial_key'
    ' UNIQUE (rrc_serial) '
    ',ADD CONSTRAINT ' || table_name || '_coin_ev_key'
    ' UNIQUE (coin_ev) '
    ',ADD CONSTRAINT ' || table_name || '_h_coin_ev_key'
    ' UNIQUE (h_coin_ev) '
    ',ADD PRIMARY KEY (melt_serial_id, freshcoin_index)'
  );
END
$$;
CREATE FUNCTION foreign_table_refresh_revealed_coins()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_revealed_coins';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_melt'
    ' FOREIGN KEY (melt_serial_id)'
    ' REFERENCES refresh_commitments (melt_serial_id) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_denom'
    ' FOREIGN KEY (denominations_serial)'
    ' REFERENCES denominations (denominations_serial) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('refresh_revealed_coins'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('refresh_revealed_coins'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('refresh_revealed_coins'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_refresh_transfer_keys(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_transfer_keys';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(rtc_serial BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',melt_serial_id INT8 PRIMARY KEY'
      ',transfer_pub BYTEA NOT NULL CHECK(LENGTH(transfer_pub)=32)'
      ',transfer_privs BYTEA NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (melt_serial_id)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Transfer keys of a refresh operation (the data revealed to the exchange).'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'needed for exchange-auditor replication logic'
    ,'rtc_serial'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the refresh commitment (rc) of the operation.'
    ,'melt_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'transfer public key for the gamma index'
    ,'transfer_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'array of TALER_CNC_KAPPA-1 transfer private keys that have been revealed, with the gamma entry being skipped'
    ,'transfer_privs'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_refresh_transfer_keys(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_transfer_keys';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_rtc_serial_key'
    ' UNIQUE (rtc_serial)'
  );
END
$$;
CREATE FUNCTION foreign_table_refresh_transfer_keys()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_transfer_keys';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || 'foreign_melt_serial_id'
    ' FOREIGN KEY (melt_serial_id)'
    ' REFERENCES refresh_commitments (melt_serial_id) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('refresh_transfer_keys'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('refresh_transfer_keys'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('refresh_transfer_keys'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_batch_deposits(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'batch_deposits';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(batch_deposit_serial_id INT8 GENERATED BY DEFAULT AS IDENTITY'
      ',shard INT8 NOT NULL'
      ',merchant_pub BYTEA NOT NULL CHECK (LENGTH(merchant_pub)=32)'
      ',wallet_timestamp INT8 NOT NULL'
      ',exchange_timestamp INT8 NOT NULL'
      ',refund_deadline INT8 NOT NULL'
      ',wire_deadline INT8 NOT NULL'
      ',h_contract_terms BYTEA NOT NULL CHECK (LENGTH(h_contract_terms)=64)'
      ',wallet_data_hash BYTEA CHECK (LENGTH(wallet_data_hash)=64) DEFAULT NULL'
      ',wire_salt BYTEA NOT NULL CHECK (LENGTH(wire_salt)=16)'
      ',wire_target_h_payto BYTEA CHECK (LENGTH(wire_target_h_payto)=32)'
      ',policy_details_serial_id INT8'
      ',policy_blocked BOOLEAN NOT NULL DEFAULT FALSE'
      ',done BOOLEAN NOT NULL DEFAULT FALSE'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (shard)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
    'Information about the contracts for which we have received (batch) deposits.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Used for load sharding in the materialized indices. Should be set based on merchant_pub. 64-bit value because we need an *unsigned* 32-bit value.'
    ,'shard'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Unsalted hash of the target bank account; also used to lookup the KYC status'
    ,'wire_target_h_payto'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash over data provided by the wallet upon payment to select a more specific contract'
    ,'wallet_data_hash'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Salt used when hashing the payto://-URI to get the h_wire that was used by the coin deposit signatures; not used to calculate wire_target_h_payto (as that one is unsalted)'
    ,'wire_salt'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Set to TRUE once we have included this (batch) deposit (and all associated coins) in some aggregate wire transfer to the merchant'
    ,'done'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'True if the aggregation of the (batch) deposit is currently blocked by some policy extension mechanism. Used to filter out deposits that must not be processed by the canonical deposit logic.'
    ,'policy_blocked'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'References policy extensions table, NULL if extensions are not used'
    ,'policy_details_serial_id'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_batch_deposits(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'batch_deposits';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_batch_deposit_serial_id_pkey'
    ' PRIMARY KEY (batch_deposit_serial_id) '
    ',ADD CONSTRAINT ' || table_name || '_merchant_pub_h_contract_terms'
    ' UNIQUE (shard, merchant_pub, h_contract_terms)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_ready '
    'ON ' || table_name || ' '
    '(shard ASC'
    ',wire_deadline ASC'
    ') WHERE NOT (done OR policy_blocked);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_for_matching '
    'ON ' || table_name || ' '
    '(shard ASC'
    ',refund_deadline ASC'
    ',wire_target_h_payto'
    ') WHERE NOT (done OR policy_blocked);'
  );
END
$$;
CREATE OR REPLACE FUNCTION foreign_table_batch_deposits()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'batch_deposits';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_policy_details'
    ' FOREIGN KEY (policy_details_serial_id) '
    ' REFERENCES policy_details (policy_details_serial_id) ON DELETE RESTRICT'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('batch_deposits'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('batch_deposits'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('batch_deposits'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE)
    ;
CREATE FUNCTION create_table_coin_deposits(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'coin_deposits';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(coin_deposit_serial_id INT8 GENERATED BY DEFAULT AS IDENTITY'
      ',batch_deposit_serial_id INT8 NOT NULL'
      ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)'
      ',coin_sig BYTEA NOT NULL CHECK (LENGTH(coin_sig)=64)'
      ',amount_with_fee taler_amount NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
    'Coins which have been deposited with the respective per-coin signatures.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Link to information about the batch deposit this coin was used for'
    ,'batch_deposit_serial_id'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_coin_deposits(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'coin_deposits';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_coin_deposit_serial_id_pkey'
    ' PRIMARY KEY (coin_deposit_serial_id) '
    ',ADD CONSTRAINT ' || table_name || '_unique_coin_sig'
    ' UNIQUE (coin_pub, coin_sig)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_batch '
    'ON ' || table_name || ' '
    '(batch_deposit_serial_id);'
  );
END
$$;
CREATE FUNCTION foreign_table_coin_deposits()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'coin_deposits';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (coin_pub) '
    ' REFERENCES known_coins (coin_pub) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_batch_deposits_id'
    ' FOREIGN KEY (batch_deposit_serial_id) '
    ' REFERENCES batch_deposits (batch_deposit_serial_id) ON DELETE CASCADE'
  );
END
$$;
CREATE OR REPLACE FUNCTION coin_deposits_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO exchange.coin_history
    (coin_pub
    ,table_name
    ,serial_id)
 VALUES
     (NEW.coin_pub
    ,'coin_deposits'
    ,NEW.coin_deposit_serial_id);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION coin_deposits_insert_trigger()
  IS 'Automatically generate coin history entry.';
CREATE FUNCTION master_table_coin_deposits()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER coin_deposits_on_insert
    AFTER INSERT
     ON coin_deposits
     FOR EACH ROW EXECUTE FUNCTION coin_deposits_insert_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('coin_deposits'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('coin_deposits'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('coin_deposits'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('coin_deposits'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE)
    ;
CREATE FUNCTION create_table_refunds(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refunds';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(refund_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)'
      ',batch_deposit_serial_id INT8 NOT NULL'
      ',merchant_sig BYTEA NOT NULL CHECK(LENGTH(merchant_sig)=64)'
      ',rtransaction_id INT8 NOT NULL'
      ',amount_with_fee taler_amount NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Data on coins that were refunded. Technically, refunds always apply against specific deposit operations involving a coin. The combination of coin_pub, merchant_pub, h_contract_terms and rtransaction_id MUST be unique, and we usually select by coin_pub so that one goes first.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies ONLY the merchant_pub, h_contract_terms and coin_pub. Multiple deposits may match a refund, this only identifies one of them.'
    ,'batch_deposit_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'used by the merchant to make refunds unique in case the same coin for the same deposit gets a subsequent (higher) refund'
    ,'rtransaction_id'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_refunds (
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refunds';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_coin_pub_index '
    'ON ' || table_name || ' '
    '(coin_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_refund_serial_id_key'
    ' UNIQUE (refund_serial_id) '
    ',ADD PRIMARY KEY (batch_deposit_serial_id, rtransaction_id) '
  );
END
$$;
CREATE FUNCTION foreign_table_refunds ()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refunds';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (coin_pub) '
    ' REFERENCES known_coins (coin_pub) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_deposit'
    ' FOREIGN KEY (batch_deposit_serial_id) '
    ' REFERENCES batch_deposits (batch_deposit_serial_id) ON DELETE CASCADE'
  );
END
$$;
CREATE OR REPLACE FUNCTION refunds_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO exchange.coin_history
    (coin_pub
    ,table_name
    ,serial_id)
 VALUES
     (NEW.coin_pub
    ,'refunds'
    ,NEW.refund_serial_id);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION refunds_insert_trigger()
  IS 'Automatically generate coin history entry.';
CREATE FUNCTION master_table_refunds()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER refunds_on_insert
    AFTER INSERT
     ON refunds
     FOR EACH ROW EXECUTE FUNCTION refunds_insert_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('refunds'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('refunds'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('refunds'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('refunds'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_wire_out(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wire_out';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I'
      '(wireout_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',execution_date INT8 NOT NULL'
      ',wtid_raw BYTEA UNIQUE NOT NULL CHECK (LENGTH(wtid_raw)=32)'
      ',wire_target_h_payto BYTEA CHECK (LENGTH(wire_target_h_payto)=32)'
      ',exchange_account_section TEXT NOT NULL'
      ',amount taler_amount NOT NULL'
      ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (wtid_raw)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'wire transfers the exchange has executed'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'identifies the configuration section with the debit account of this payment'
    ,'exchange_account_section'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the credited bank account and KYC status'
    ,'wire_target_h_payto'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_wire_out(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wire_out';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_wire_target_h_payto_index '
    'ON ' || table_name || ' '
    '(wire_target_h_payto);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wireout_uuid_pkey'
    ' PRIMARY KEY (wireout_uuid)'
  );
END
$$;
CREATE FUNCTION wire_out_delete_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  DELETE FROM exchange.aggregation_tracking
   WHERE wtid_raw = OLD.wtid_raw;
  RETURN OLD;
END $$;
COMMENT ON FUNCTION wire_out_delete_trigger()
  IS 'Replicate reserve_out deletions into aggregation_tracking. This replaces an earlier use of an ON DELETE CASCADE that required a DEFERRABLE constraint and conflicted with nice partitioning.';
CREATE FUNCTION master_table_wire_out()
RETURNS void
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER wire_out_on_delete
    AFTER DELETE
      ON wire_out
     FOR EACH ROW EXECUTE FUNCTION wire_out_delete_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('wire_out'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('wire_out'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('wire_out'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_aggregation_transient(
  IN shard_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'aggregation_transient';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
    '(amount taler_amount NOT NULL'
    ',wire_target_h_payto BYTEA CHECK (LENGTH(wire_target_h_payto)=32)'
    ',merchant_pub BYTEA CHECK (LENGTH(merchant_pub)=32)'
    ',exchange_account_section TEXT NOT NULL'
    ',legitimization_requirement_serial_id INT8 NOT NULL DEFAULT(0)'
    ',wtid_raw BYTEA NOT NULL CHECK (LENGTH(wtid_raw)=32)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (wire_target_h_payto)'
    ,shard_suffix
  );
  PERFORM comment_partitioned_table(
    'aggregations currently happening (lacking wire_out, usually because the amount is too low); this table is not replicated'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
       'Sum of all of the aggregated deposits (without deposit fees)'
      ,'amount'
      ,table_name
      ,shard_suffix
  );
  PERFORM comment_partitioned_column(
       'identifier of the wire transfer'
      ,'wtid_raw'
      ,table_name
      ,shard_suffix
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('aggregation_transient'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_aggregation_tracking(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'aggregation_tracking';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(aggregation_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',batch_deposit_serial_id INT8 PRIMARY KEY'
      ',wtid_raw BYTEA NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (batch_deposit_serial_id)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'mapping from wire transfer identifiers (WTID) to deposits (and back)'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'identifier of the wire transfer'
    ,'wtid_raw'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_aggregation_tracking(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'aggregation_tracking';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_wtid_raw_index '
    'ON ' || table_name || ' '
    '(wtid_raw);'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_wtid_raw_index '
    'IS ' || quote_literal('for lookup_transactions') || ';'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_aggregation_serial_id_key'
    ' UNIQUE (aggregation_serial_id) '
  );
END
$$;
CREATE FUNCTION foreign_table_aggregation_tracking()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'aggregation_tracking';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_deposit'
    ' FOREIGN KEY (batch_deposit_serial_id)'
    ' REFERENCES batch_deposits (batch_deposit_serial_id)'
    ' ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('aggregation_tracking'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('aggregation_tracking'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('aggregation_tracking'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_recoup(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'recoup';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(recoup_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)'
      ',coin_sig BYTEA NOT NULL CHECK(LENGTH(coin_sig)=64)'
      ',coin_blind BYTEA NOT NULL CHECK(LENGTH(coin_blind)=32)'
      ',amount taler_amount NOT NULL'
      ',recoup_timestamp INT8 NOT NULL'
      ',reserve_out_serial_id INT8 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub);'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Information about recoups that were executed between a coin and a reserve. In this type of recoup, the amount is credited back to the reserve from which the coin originated.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Coin that is being debited in the recoup. Do not CASCADE ON DROP on the coin_pub, as we may keep the coin alive!'
    ,'coin_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the h_blind_ev of the recouped coin and provides the link to the credited reserve.'
    ,'reserve_out_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature by the coin affirming the recoup, of type TALER_SIGNATURE_WALLET_COIN_RECOUP'
    ,'coin_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Denomination blinding key used when creating the blinded coin from the planchet. Secret revealed during the recoup to provide the linkage between the coin and the withdraw operation.'
    ,'coin_blind'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_recoup(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'recoup';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_coin_pub_index '
    'ON ' || table_name || ' '
    '(coin_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_recoup_uuid_key'
    ' UNIQUE (recoup_uuid) '
  );
END
$$;
CREATE FUNCTION foreign_table_recoup()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'recoup';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserves_out'
    ' FOREIGN KEY (reserve_out_serial_id) '
    ' REFERENCES reserves_out (reserve_out_serial_id) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (coin_pub) '
    ' REFERENCES known_coins (coin_pub)'
  );
END
$$;
CREATE FUNCTION create_table_recoup_by_reserve(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'recoup_by_reserve';
BEGIN
  PERFORM create_partitioned_table(
  'CREATE TABLE %I'
    '(reserve_out_serial_id INT8 NOT NULL' 
    ',coin_pub BYTEA CHECK (LENGTH(coin_pub)=32)' 
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_out_serial_id)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Information in this table is strictly redundant with that of recoup, but saved by a different primary key for fast lookups by reserve_out_serial_id.'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_recoup_by_reserve(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'recoup_by_reserve';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_main_index '
    'ON ' || table_name || ' '
    '(reserve_out_serial_id);'
  );
END
$$;
CREATE FUNCTION recoup_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO recoup_by_reserve
    (reserve_out_serial_id
    ,coin_pub)
  VALUES
    (NEW.reserve_out_serial_id
    ,NEW.coin_pub);
  INSERT INTO coin_history
    (coin_pub
    ,table_name
    ,serial_id)
  VALUES
    (NEW.coin_pub
    ,'recoup'
    ,NEW.recoup_uuid);
  INSERT INTO reserve_history
    (reserve_pub
    ,table_name
    ,serial_id)
  SELECT
     res.reserve_pub
    ,'recoup'
    ,NEW.recoup_uuid
  FROM reserves_out rout
  JOIN reserves res
    USING (reserve_uuid)
    WHERE rout.reserve_out_serial_id = NEW.reserve_out_serial_id;
  RETURN NEW;
END $$;
COMMENT ON FUNCTION recoup_insert_trigger()
  IS 'Replicates recoup inserts into recoup_by_reserve table and updates the coin_history table.';
CREATE FUNCTION recoup_delete_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  DELETE FROM recoup_by_reserve
   WHERE reserve_out_serial_id = OLD.reserve_out_serial_id
     AND coin_pub = OLD.coin_pub;
  RETURN OLD;
END $$;
COMMENT ON FUNCTION recoup_delete_trigger()
  IS 'Replicate recoup deletions into recoup_by_reserve table.';
CREATE FUNCTION master_table_recoup()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER recoup_on_insert
    AFTER INSERT
     ON recoup
     FOR EACH ROW EXECUTE FUNCTION recoup_insert_trigger();
  CREATE TRIGGER recoup_on_delete
    AFTER DELETE
      ON recoup
     FOR EACH ROW EXECUTE FUNCTION recoup_delete_trigger();
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('recoup'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('recoup'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('recoup'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('recoup_by_reserve'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('recoup_by_reserve'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('recoup'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_recoup_refresh(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'recoup_refresh';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
    '(recoup_refresh_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
    ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)'
    ',known_coin_id BIGINT NOT NULL'
    ',coin_sig BYTEA NOT NULL CHECK(LENGTH(coin_sig)=64)'
    ',coin_blind BYTEA NOT NULL CHECK(LENGTH(coin_blind)=32)'
    ',amount taler_amount NOT NULL'
    ',recoup_timestamp INT8 NOT NULL'
    ',rrc_serial INT8 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Table of coins that originated from a refresh operation and that were recouped. Links the (fresh) coin to the melted operation (and thus the old coin). A recoup on a refreshed coin credits the old coin and debits the fresh coin.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Refreshed coin of a revoked denomination where the residual value is credited to the old coin. Do not CASCADE ON DROP on the coin_pub, as we may keep the coin alive!'
    ,'coin_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Used for garbage collection (in the absence of foreign constraints, in the future)'
    ,'known_coin_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Link to the refresh operation. Also identifies the h_blind_ev of the recouped coin (as h_coin_ev).'
    ,'rrc_serial'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Denomination blinding key used when creating the blinded coin from the planchet. Secret revealed during the recoup to provide the linkage between the coin and the refresh operation.'
    ,'coin_blind'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_recoup_refresh(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'recoup_refresh';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_rrc_serial_index'
    ' ON ' || table_name || ' '
    '(rrc_serial);'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_rrc_serial_index '
    'IS ' || quote_literal('used in exchange_do_melt for zombie coins (rare)') || ';'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_coin_pub_index'
    ' ON ' || table_name || ' '
    '(coin_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_recoup_refresh_uuid_key'
    ' UNIQUE (recoup_refresh_uuid) '
  );
END
$$;
CREATE FUNCTION foreign_table_recoup_refresh()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'recoup_refresh';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (coin_pub) '
    ' REFERENCES known_coins (coin_pub)'
    ',ADD CONSTRAINT ' || table_name || '_foreign_known_coin_id'
    ' FOREIGN KEY (known_coin_id) '
    ' REFERENCES known_coins (known_coin_id) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_rrc_serial'
    ' FOREIGN KEY (rrc_serial) '
    ' REFERENCES refresh_revealed_coins (rrc_serial) ON DELETE CASCADE'
  );
END
$$;
CREATE OR REPLACE FUNCTION recoup_refresh_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO exchange.coin_history
    (coin_pub
    ,table_name
    ,serial_id)
  VALUES
     (NEW.coin_pub
    ,'recoup_refresh::NEW'
    ,NEW.recoup_refresh_uuid);
  INSERT INTO exchange.coin_history
    (coin_pub
    ,table_name
    ,serial_id)
  SELECT
     melt.old_coin_pub
    ,'recoup_refresh::OLD'
    ,NEW.recoup_refresh_uuid
    FROM refresh_revealed_coins rrc
    JOIN refresh_commitments melt
      USING (melt_serial_id)
    WHERE rrc.rrc_serial = NEW.rrc_serial;
  RETURN NEW;
END $$;
COMMENT ON FUNCTION coin_deposits_insert_trigger()
  IS 'Automatically generate coin history entry.';
CREATE FUNCTION master_table_recoup_refresh()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER recoup_refresh_on_insert
    AFTER INSERT
     ON recoup_refresh
     FOR EACH ROW EXECUTE FUNCTION recoup_refresh_insert_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('recoup_refresh'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('recoup_refresh'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('recoup_refresh'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('recoup_refresh'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_prewire(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'prewire';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(prewire_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY'
      ',wire_method TEXT NOT NULL'
      ',finished BOOLEAN NOT NULL DEFAULT FALSE'
      ',failed BOOLEAN NOT NULL DEFAULT FALSE'
      ',buf BYTEA NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (prewire_uuid)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'pre-commit data for wire transfers we are about to execute'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'set to TRUE if the bank responded with a non-transient failure to our transfer request'
    ,'failed'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'set to TRUE once bank confirmed receiving the wire transfer request'
    ,'finished'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'serialized data to send to the bank to execute the wire transfer'
    ,'buf'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_prewire(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'prewire';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_finished_index '
    'ON ' || table_name || ' '
    '(finished)'
    ' WHERE finished;'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_finished_index '
    'IS ' || quote_literal('for do_gc') || ';'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_failed_finished_index '
    'ON ' || table_name || ' '
    '(prewire_uuid)'
    ' WHERE finished=FALSE'
    '   AND failed=FALSE;'
  );
  EXECUTE FORMAT (
    'COMMENT ON INDEX ' || table_name || '_by_failed_finished_index '
    'IS ' || quote_literal('for wire_prepare_data_get') || ';'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('prewire'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('prewire'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_cs_nonce_locks(
  partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(cs_nonce_lock_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',nonce BYTEA PRIMARY KEY CHECK (LENGTH(nonce)=32)'
      ',op_hash BYTEA NOT NULL CHECK (LENGTH(op_hash)=64)'
      ',max_denomination_serial INT8 NOT NULL'
    ') %s ;'
    ,'cs_nonce_locks'
    ,'PARTITION BY HASH (nonce)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'ensures a Clause Schnorr client nonce is locked for use with an operation identified by a hash'
    ,'cs_nonce_locks'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'actual nonce submitted by the client'
    ,'nonce'
    ,'cs_nonce_locks'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash (RC for refresh, blind coin hash for withdraw) the nonce may be used with'
    ,'op_hash'
    ,'cs_nonce_locks'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Maximum number of a CS denomination serial the nonce could be used with, for GC'
    ,'max_denomination_serial'
    ,'cs_nonce_locks'
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_cs_nonce_locks(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'cs_nonce_locks';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_cs_nonce_lock_serial_id_key'
    ' UNIQUE (cs_nonce_lock_serial_id)'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('cs_nonce_locks'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('cs_nonce_locks'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_purse_requests(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_requests';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(purse_requests_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
      ',merge_pub BYTEA NOT NULL CHECK (LENGTH(merge_pub)=32)'
      ',purse_creation INT8 NOT NULL'
      ',purse_expiration INT8 NOT NULL'
      ',h_contract_terms BYTEA NOT NULL CHECK (LENGTH(h_contract_terms)=64)'
      ',age_limit INT4 NOT NULL'
      ',flags INT4 NOT NULL'
      ',in_reserve_quota BOOLEAN NOT NULL DEFAULT(FALSE)'
      ',was_decided BOOLEAN NOT NULL DEFAULT(FALSE)'
      ',amount_with_fee taler_amount NOT NULL'
      ',purse_fee taler_amount NOT NULL'
      ',balance taler_amount NOT NULL DEFAULT (0,0)'
      ',purse_sig BYTEA NOT NULL CHECK(LENGTH(purse_sig)=64)'
      ',PRIMARY KEY (purse_pub)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Requests establishing purses, associating them with a contract but without a target reserve'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Local time when the purse was created. Determines applicable purse fees.'
    ,'purse_creation'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'When the purse is set to expire'
    ,'purse_expiration'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Hash of the contract the parties are to agree to'
    ,'h_contract_terms'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'see the enum TALER_WalletAccountMergeFlags'
    ,'flags'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'set to TRUE if this purse currently counts against the number of free purses in the respective reserve'
    ,'in_reserve_quota'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total amount expected to be in the purse'
    ,'amount_with_fee'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Purse fee the client agreed to pay from the reserve (accepted by the exchange at the time the purse was created). Zero if in_reserve_quota is TRUE.'
    ,'purse_fee'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'Total amount actually in the purse (updated)'
    ,'balance'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature of the purse affirming the purse parameters, of type TALER_SIGNATURE_PURSE_REQUEST'
    ,'purse_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_purse_requests(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_requests';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_merge_pub '
    'ON ' || table_name || ' '
    '(merge_pub);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_purse_expiration '
    'ON ' || table_name || ' '
    '(purse_expiration) ' ||
    'WHERE NOT was_decided;'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_purse_requests_serial_id_key'
    ' UNIQUE (purse_requests_serial_id) '
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('purse_requests'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('purse_requests'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_purse_merges(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_merges';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
    '(purse_merge_request_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
    ',partner_serial_id INT8'
    ',reserve_pub BYTEA NOT NULL CHECK(length(reserve_pub)=32)'
    ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
    ',merge_sig BYTEA NOT NULL CHECK (LENGTH(merge_sig)=64)'
    ',merge_timestamp INT8 NOT NULL'
    ',PRIMARY KEY (purse_pub)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Merge requests where a purse-owner requested merging the purse into the account'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'identifies the partner exchange, NULL in case the target reserve lives at this exchange'
    ,'partner_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the target reserve'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'signature by the purse private key affirming the merge, of type TALER_SIGNATURE_WALLET_PURSE_MERGE'
    ,'merge_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'when was the merge message signed'
    ,'merge_timestamp'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_purse_merges(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_merges';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_purse_merge_request_serial_id_key'
    ' UNIQUE (purse_merge_request_serial_id) '
  );
END
$$;
CREATE FUNCTION foreign_table_purse_merges()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_merges';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_partner_serial_id'
    ' FOREIGN KEY (partner_serial_id) '
    ' REFERENCES partners(partner_serial_id) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_purse_pub'
    ' FOREIGN KEY (purse_pub) '
    ' REFERENCES purse_requests (purse_pub) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('purse_merges'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('purse_merges'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('purse_merges'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_account_merges(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'account_merges';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I '
      '(account_merge_request_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL CHECK (LENGTH(reserve_pub)=32)'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
      ',wallet_h_payto BYTEA NOT NULL CHECK (LENGTH(wallet_h_payto)=32)'
      ',PRIMARY KEY (purse_pub)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Merge requests where a purse- and account-owner requested merging the purse into the account'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the target reserve'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash over the normalized (!) payto:// URI that identifies the receiving wallet'
    ,'wallet_h_payto'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'signature by the reserve private key affirming the merge, of type TALER_SIGNATURE_WALLET_ACCOUNT_MERGE'
    ,'reserve_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_account_merges(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'account_merges';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve_pub '
    'ON ' || table_name || ' '
    '(reserve_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_account_merge_request_serial_id_key'
    ' UNIQUE (account_merge_request_serial_id) '
  );
END
$$;
CREATE FUNCTION foreign_table_account_merges()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'account_merges';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub) '
    ' REFERENCES reserves (reserve_pub) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_purse_pub'
    ' FOREIGN KEY (purse_pub) '
    ' REFERENCES purse_requests (purse_pub)'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('account_merges'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('account_merges'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('account_merges'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_purse_decision(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_decision';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(purse_decision_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
      ',action_timestamp INT8 NOT NULL'
      ',refunded BOOL NOT NULL'
      ',PRIMARY KEY (purse_pub)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Purses that were decided upon (refund or merge)'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_purse_decision(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_decision';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_purse_action_serial_id_key'
    ' UNIQUE (purse_decision_serial_id) '
  );
END
$$;
CREATE OR REPLACE FUNCTION purse_decision_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  UPDATE purse_requests
     SET was_decided=TRUE
   WHERE purse_pub=NEW.purse_pub;
  IF NEW.refunded
  THEN
    INSERT INTO coin_history
      (coin_pub
      ,table_name
      ,serial_id)
    SELECT
      pd.coin_pub
     ,'purse_decision'
     ,NEW.purse_decision_serial_id
    FROM purse_deposits pd
    WHERE purse_pub = NEW.purse_pub;
  ELSE
    INSERT INTO reserve_history
      (reserve_pub
      ,table_name
      ,serial_id)
    SELECT
      reserve_pub
     ,'purse_decision'
     ,NEW.purse_decision_serial_id
    FROM purse_merges
    WHERE purse_pub=NEW.purse_pub;
  END IF;
  RETURN NEW;
END $$;
COMMENT ON FUNCTION purse_decision_insert_trigger()
  IS 'Automatically generate coin history entry and update decision status for the purse.';
CREATE FUNCTION master_table_purse_decision()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER purse_decision_on_insert
    AFTER INSERT
     ON purse_decision
     FOR EACH ROW EXECUTE FUNCTION purse_decision_insert_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('purse_decision'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('purse_decision'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('purse_decision'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_contracts(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'contracts';
BEGIN
  PERFORM create_partitioned_table(
     'CREATE TABLE %I '
     '(contract_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
     ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
     ',pub_ckey BYTEA NOT NULL CHECK (LENGTH(pub_ckey)=32)'
     ',contract_sig BYTEA NOT NULL CHECK (LENGTH(contract_sig)=64)'
     ',e_contract BYTEA NOT NULL'
     ',purse_expiration INT8 NOT NULL'
     ',PRIMARY KEY (purse_pub)'
     ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'encrypted contracts associated with purses'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the purse that the contract is associated with'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'signature over the encrypted contract by the purse contract key'
    ,'contract_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public ECDH key used to encrypt the contract, to be used with the purse private key for decryption'
    ,'pub_ckey'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'AES-GCM encrypted contract terms (contains gzip compressed JSON after decryption)'
    ,'e_contract'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_contracts(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'contracts';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_contract_serial_id_key'
    ' UNIQUE (contract_serial_id) '
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('contracts'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('contracts'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_history_requests(
  IN shard_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'history_requests';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(history_request_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_pub BYTEA NOT NULL CHECK (LENGTH(reserve_pub)=32)'
      ',request_timestamp INT8 NOT NULL'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',history_fee taler_amount NOT NULL'
      ',PRIMARY KEY (reserve_pub,request_timestamp)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,shard_suffix
  );
  PERFORM comment_partitioned_table(
     'Paid history requests issued by a client against a reserve'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'When was the history request made'
    ,'request_timestamp'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature approving payment for the history request'
    ,'reserve_sig'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'History fee approved by the signature'
    ,'history_fee'
    ,table_name
    ,shard_suffix
  );
END $$;
CREATE FUNCTION constrain_table_history_requests(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  partition_name TEXT;
BEGIN
  partition_name = concat_ws('_', 'history_requests', partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || partition_name ||
    ' ADD CONSTRAINT ' || partition_name || '_serial_id'
    ' UNIQUE (history_request_serial_id)'
  );
END
$$;
CREATE FUNCTION foreign_table_history_requests()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'history_requests';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub) '
    ' REFERENCES reserves(reserve_pub) ON DELETE CASCADE'
  );
END $$;
CREATE OR REPLACE FUNCTION history_requests_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO reserve_history
    (reserve_pub
    ,table_name
    ,serial_id)
  VALUES
    (NEW.reserve_pub
    ,'history_requests'
    ,NEW.history_request_serial_id);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION history_requests_insert_trigger()
  IS 'Automatically generate reserve history entry.';
CREATE FUNCTION master_table_history_requests()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER history_requests_on_insert
    AFTER INSERT
     ON history_requests
     FOR EACH ROW EXECUTE FUNCTION history_requests_insert_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('history_requests'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('history_requests'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('history_requests'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('history_requests'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_purse_deposits(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_deposits';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(purse_deposit_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',partner_serial_id INT8'
      ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
      ',coin_pub BYTEA NOT NULL'
      ',amount_with_fee taler_amount NOT NULL'
      ',coin_sig BYTEA NOT NULL CHECK(LENGTH(coin_sig)=64)'
      ',PRIMARY KEY (purse_pub,coin_pub)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Requests depositing coins into a purse'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'identifies the partner exchange, NULL in case the target purse lives at this exchange'
    ,'partner_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the coin being deposited'
    ,'coin_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total amount being deposited'
    ,'amount_with_fee'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature of the coin affirming the deposit into the purse, of type TALER_SIGNATURE_PURSE_DEPOSIT'
    ,'coin_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_purse_deposits(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_deposits';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_purse_deposit_serial_id_key'
    ' UNIQUE (purse_deposit_serial_id) '
  );
END
$$;
CREATE FUNCTION foreign_table_purse_deposits()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_deposits';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_partner'
    ' FOREIGN KEY (partner_serial_id) '
    ' REFERENCES partners(partner_serial_id) ON DELETE CASCADE'
    ',ADD CONSTRAINT ' || table_name || '_foreign_coin_pub'
    ' FOREIGN KEY (coin_pub) '
    ' REFERENCES known_coins (coin_pub) ON DELETE CASCADE'
  );
END
$$;
CREATE OR REPLACE FUNCTION purse_deposits_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO exchange.coin_history
    (coin_pub
    ,table_name
    ,serial_id)
 VALUES
    (NEW.coin_pub
    ,'purse_deposits'
    ,NEW.purse_deposit_serial_id);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION purse_deposits_insert_trigger()
  IS 'Automatically generate coin history entry.';
CREATE FUNCTION master_table_purse_deposits()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER purse_deposits_on_insert
    AFTER INSERT
     ON purse_deposits
     FOR EACH ROW EXECUTE FUNCTION purse_deposits_insert_trigger();
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('purse_deposits'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('purse_deposits'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('purse_deposits'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE),
    ('purse_deposits'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_wads_in(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wads_in';
BEGIN
  PERFORM create_partitioned_table(
     'CREATE TABLE %I '
     '(wad_in_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
     ',wad_id BYTEA PRIMARY KEY CHECK (LENGTH(wad_id)=24)'
     ',origin_exchange_url TEXT NOT NULL'
     ',amount taler_amount NOT NULL'
     ',arrival_time INT8 NOT NULL'
     ',UNIQUE (wad_id, origin_exchange_url)'
     ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (wad_id)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Incoming exchange-to-exchange wad wire transfers'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Unique identifier of the wad, part of the wire transfer subject'
    ,'wad_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Base URL of the originating URL, also part of the wire transfer subject'
    ,'origin_exchange_url'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Actual amount that was received by our exchange'
    ,'amount'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Time when the wad was received'
    ,'arrival_time'
    ,table_name
    ,partition_suffix
  );
END $$;
CREATE FUNCTION constrain_table_wads_in(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wads_in';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wad_in_serial_id_key'
    ' UNIQUE (wad_in_serial_id) '
    ',ADD CONSTRAINT ' || table_name || '_wad_is_origin_exchange_url_key'
    ' UNIQUE (wad_id, origin_exchange_url) '
  );
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('wads_in'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('wads_in'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_wad_in_entries(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wad_in_entries';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(wad_in_entry_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',wad_in_serial_id INT8'
      ',reserve_pub BYTEA NOT NULL CHECK(LENGTH(reserve_pub)=32)'
      ',purse_pub BYTEA PRIMARY KEY CHECK(LENGTH(purse_pub)=32)'
      ',h_contract BYTEA NOT NULL CHECK(LENGTH(h_contract)=64)'
      ',purse_expiration INT8 NOT NULL'
      ',merge_timestamp INT8 NOT NULL'
      ',amount_with_fee taler_amount NOT NULL'
      ',wad_fee taler_amount NOT NULL'
      ',deposit_fees taler_amount NOT NULL'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',purse_sig BYTEA NOT NULL CHECK (LENGTH(purse_sig)=64)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'list of purses aggregated in a wad according to the sending exchange'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'wad for which the given purse was included in the aggregation'
    ,'wad_in_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'target account of the purse (must be at the local exchange)'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public key of the purse that was merged'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'hash of the contract terms of the purse'
    ,'h_contract'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Time when the purse was set to expire'
    ,'purse_expiration'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Time when the merge was approved'
    ,'merge_timestamp'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total amount in the purse'
    ,'amount_with_fee'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total wad fees paid by the purse'
    ,'wad_fee'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total deposit fees paid when depositing coins into the purse'
    ,'deposit_fees'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature by the receiving reserve, of purpose TALER_SIGNATURE_ACCOUNT_MERGE'
    ,'reserve_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature by the purse of purpose TALER_SIGNATURE_PURSE_MERGE'
    ,'purse_sig'
    ,table_name
    ,partition_suffix
  );
END $$;
CREATE FUNCTION constrain_table_wad_in_entries(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wad_in_entries';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wad_in_entry_serial_id_key'
    ' UNIQUE (wad_in_entry_serial_id) '
  );
END $$;
CREATE FUNCTION foreign_table_wad_in_entries()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wad_in_entries';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_wad_in'
    ' FOREIGN KEY(wad_in_serial_id)'
    ' REFERENCES wads_in (wad_in_serial_id) ON DELETE CASCADE'
  );
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('wad_in_entries'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('wad_in_entries'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('wad_in_entries'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_wads_out(
  IN shard_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wads_out';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I '
      '(wad_out_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',wad_id BYTEA PRIMARY KEY CHECK (LENGTH(wad_id)=24)'
      ',partner_serial_id INT8 NOT NULL'
      ',amount taler_amount NOT NULL'
      ',execution_time INT8 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (wad_id)'
    ,shard_suffix
  );
  PERFORM comment_partitioned_table(
     'Wire transfers made to another exchange to transfer purse funds'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Unique identifier of the wad, part of the wire transfer subject'
    ,'wad_id'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'target exchange of the wad'
    ,'partner_serial_id'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Amount that was wired'
    ,'amount'
    ,table_name
    ,shard_suffix
  );
  PERFORM comment_partitioned_column(
     'Time when the wire transfer was scheduled'
    ,'execution_time'
    ,table_name
    ,shard_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_wads_out(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wads_out';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wad_out_serial_id_key'
    ' UNIQUE (wad_out_serial_id) '
  );
END
$$;
CREATE FUNCTION foreign_table_wads_out()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wads_out';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_partner'
    ' FOREIGN KEY(partner_serial_id)'
    ' REFERENCES partners(partner_serial_id) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('wads_out'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('wads_out'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('wads_out'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_wad_out_entries(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wad_out_entries';
BEGIN
  PERFORM create_partitioned_table(
     'CREATE TABLE %I '
     '(wad_out_entry_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
     ',wad_out_serial_id INT8'
     ',reserve_pub BYTEA NOT NULL CHECK(LENGTH(reserve_pub)=32)'
     ',purse_pub BYTEA PRIMARY KEY CHECK(LENGTH(purse_pub)=32)'
     ',h_contract BYTEA NOT NULL CHECK(LENGTH(h_contract)=64)'
     ',purse_expiration INT8 NOT NULL'
     ',merge_timestamp INT8 NOT NULL'
     ',amount_with_fee taler_amount NOT NULL'
     ',wad_fee taler_amount NOT NULL'
     ',deposit_fees taler_amount NOT NULL'
     ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
     ',purse_sig BYTEA NOT NULL CHECK (LENGTH(purse_sig)=64)'
     ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
    'Purses combined into a wad'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Wad the purse was part of'
    ,'wad_out_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Target reserve for the purse'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Hash of the contract associated with the purse'
    ,'h_contract'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Time when the purse expires'
    ,'purse_expiration'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Time when the merge was approved'
    ,'merge_timestamp'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total amount in the purse'
    ,'amount_with_fee'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
    'Wad fee charged to the purse'
    ,'wad_fee'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Total deposit fees charged to the purse'
    ,'deposit_fees'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature by the receiving reserve, of purpose TALER_SIGNATURE_ACCOUNT_MERGE'
    ,'reserve_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature by the purse of purpose TALER_SIGNATURE_PURSE_MERGE'
    ,'purse_sig'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_wad_out_entries(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wad_out_entries';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_wad_out_entry_serial_id_key'
    ' UNIQUE (wad_out_entry_serial_id) '
  );
END
$$;
CREATE FUNCTION foreign_table_wad_out_entries()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'wad_out_entries';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_wad_out'
    ' FOREIGN KEY(wad_out_serial_id)'
    ' REFERENCES wads_out (wad_out_serial_id) ON DELETE CASCADE'
  );
END
$$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('wad_out_entries'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('wad_out_entries'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('wad_out_entries'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE TABLE work_shards
  (shard_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,last_attempt INT8 NOT NULL
  ,start_row INT8 NOT NULL
  ,end_row INT8 NOT NULL
  ,completed BOOLEAN NOT NULL DEFAULT FALSE
  ,job_name TEXT NOT NULL
  ,PRIMARY KEY (job_name, start_row)
  );
COMMENT ON TABLE work_shards
  IS 'coordinates work between multiple processes working on the same job';
COMMENT ON COLUMN work_shards.shard_serial_id
  IS 'unique serial number identifying the shard';
COMMENT ON COLUMN work_shards.last_attempt
  IS 'last time a worker attempted to work on the shard';
COMMENT ON COLUMN work_shards.completed
  IS 'set to TRUE once the shard is finished by a worker';
COMMENT ON COLUMN work_shards.start_row
  IS 'row at which the shard scope starts, inclusive';
COMMENT ON COLUMN work_shards.end_row
  IS 'row at which the shard scope ends, exclusive';
COMMENT ON COLUMN work_shards.job_name
  IS 'unique name of the job the workers on this shard are performing';
CREATE INDEX work_shards_by_job_name_completed_last_attempt_index
  ON work_shards
  (job_name
  ,completed
  ,last_attempt ASC
  );
CREATE INDEX work_shards_by_end_row_index
  ON work_shards
  (end_row DESC);
CREATE INDEX work_shards_by_rows
  ON work_shards
  (job_name
  ,start_row
  ,end_row);
CREATE UNLOGGED TABLE revolving_work_shards
  (shard_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,last_attempt INT8 NOT NULL
  ,start_row INT4 NOT NULL
  ,end_row INT4 NOT NULL
  ,active BOOLEAN NOT NULL DEFAULT FALSE
  ,job_name TEXT NOT NULL
  ,PRIMARY KEY (job_name, start_row)
  );
COMMENT ON TABLE revolving_work_shards
  IS 'coordinates work between multiple processes working on the same job with partitions that need to be repeatedly processed; unlogged because on system crashes the locks represented by this table will have to be cleared anyway, typically using "taler-exchange-dbinit -s"';
COMMENT ON COLUMN revolving_work_shards.shard_serial_id
  IS 'unique serial number identifying the shard';
COMMENT ON COLUMN revolving_work_shards.last_attempt
  IS 'last time a worker attempted to work on the shard';
COMMENT ON COLUMN revolving_work_shards.active
  IS 'set to TRUE when a worker is active on the shard';
COMMENT ON COLUMN revolving_work_shards.start_row
  IS 'row at which the shard scope starts, inclusive';
COMMENT ON COLUMN revolving_work_shards.end_row
  IS 'row at which the shard scope ends, exclusive';
COMMENT ON COLUMN revolving_work_shards.job_name
  IS 'unique name of the job the workers on this shard are performing';
CREATE INDEX revolving_work_shards_by_job_name_active_last_attempt_index
  ON revolving_work_shards
  (job_name
  ,active
  ,last_attempt
  );
CREATE TABLE partners
  (partner_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,partner_master_pub BYTEA NOT NULL CHECK(LENGTH(partner_master_pub)=32)
  ,start_date INT8 NOT NULL
  ,end_date INT8 NOT NULL
  ,next_wad INT8 NOT NULL DEFAULT (0)
  ,wad_frequency INT8 NOT NULL
  ,wad_fee taler_amount NOT NULL
  ,master_sig BYTEA NOT NULL CHECK (LENGTH(master_sig)=64)
  ,partner_base_url TEXT NOT NULL
  ,PRIMARY KEY (partner_master_pub, start_date)
  );
COMMENT ON TABLE partners
  IS 'exchanges we do wad transfers to';
COMMENT ON COLUMN partners.partner_master_pub
  IS 'offline master public key of the partner';
COMMENT ON COLUMN partners.start_date
  IS 'starting date of the partnership';
COMMENT ON COLUMN partners.end_date
  IS 'end date of the partnership';
COMMENT ON COLUMN partners.next_wad
  IS 'at what time should we do the next wad transfer to this partner (frequently updated); set to forever after the end_date';
COMMENT ON COLUMN partners.wad_frequency
  IS 'how often do we promise to do wad transfers';
COMMENT ON COLUMN partners.wad_fee
  IS 'how high is the fee for a wallet to be added to a wad to this partner';
COMMENT ON COLUMN partners.partner_base_url
  IS 'base URL of the REST API for this partner';
COMMENT ON COLUMN partners.master_sig
  IS 'signature of our master public key affirming the partnership, of purpose TALER_SIGNATURE_MASTER_PARTNER_DETAILS';
CREATE INDEX IF NOT EXISTS partner_by_wad_time
  ON partners (next_wad ASC);
CREATE TABLE partner_accounts
  (payto_uri TEXT PRIMARY KEY
  ,partner_serial_id INT8 REFERENCES partners(partner_serial_id) ON DELETE CASCADE
  ,partner_master_sig BYTEA CHECK (LENGTH(partner_master_sig)=64)
  ,last_seen INT8 NOT NULL
  );
CREATE INDEX IF NOT EXISTS partner_accounts_index_by_partner_and_time
  ON partner_accounts (partner_serial_id,last_seen);
COMMENT ON TABLE partner_accounts
  IS 'Table with bank accounts of the partner exchange. Entries never expire as we need to remember the signature for the auditor.';
COMMENT ON COLUMN partner_accounts.payto_uri
  IS 'payto URI (RFC 8905) with the bank account of the partner exchange.';
COMMENT ON COLUMN partner_accounts.partner_master_sig
  IS 'Signature of purpose TALER_SIGNATURE_MASTER_WIRE_DETAILS by the partner master public key';
COMMENT ON COLUMN partner_accounts.last_seen
  IS 'Last time we saw this account as being active at the partner exchange. Used to select the most recent entry, and to detect when we should check again.';
CREATE OR REPLACE FUNCTION create_table_purse_actions(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_actions';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I'
      '(purse_pub BYTEA NOT NULL PRIMARY KEY CHECK(LENGTH(purse_pub)=32)'
      ',action_date INT8 NOT NULL'
      ',partner_serial_id INT8'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'purses awaiting some action by the router'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'public (contract) key of the purse'
    ,'purse_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'when is the purse ready for action'
    ,'action_date'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'wad target of an outgoing wire transfer, 0 for local, NULL if the purse is unmerged and thus the target is still unknown'
    ,'partner_serial_id'
    ,table_name
    ,partition_suffix
  );
END $$;
CREATE OR REPLACE FUNCTION purse_requests_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO
    purse_actions
    (purse_pub
    ,action_date)
  VALUES
    (NEW.purse_pub
    ,NEW.purse_expiration);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION purse_requests_insert_trigger()
  IS 'When a purse is created, insert it into the purse_action table to take action when the purse expires.';
CREATE OR REPLACE FUNCTION master_table_purse_actions()
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_actions';
BEGIN
  CREATE INDEX IF NOT EXISTS purse_action_by_target
    ON purse_actions
    (partner_serial_id,action_date);
  CREATE TRIGGER purse_requests_on_insert
    AFTER INSERT
    ON purse_requests
    FOR EACH ROW EXECUTE FUNCTION purse_requests_insert_trigger();
  COMMENT ON TRIGGER purse_requests_on_insert
          ON purse_requests
    IS 'Here we install an entry for the purse expiration.';
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('purse_actions'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('purse_actions'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE OR REPLACE FUNCTION create_table_purse_deletion(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_deletion';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I'
      '(purse_deletion_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',purse_sig BYTEA CHECK (LENGTH(purse_sig)=64)'
      ',purse_pub BYTEA NOT NULL CHECK (LENGTH(purse_pub)=32)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (purse_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'signatures affirming explicit purse deletions'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'signature of type WALLET_PURSE_DELETE'
    ,'purse_sig'
    ,table_name
    ,partition_suffix
  );
END $$;
COMMENT ON FUNCTION create_table_purse_deletion
  IS 'Creates the purse_deletion table';
CREATE OR REPLACE FUNCTION constrain_table_purse_deletion(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_deletion';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
      ' ADD CONSTRAINT ' || table_name || '_delete_serial_key '
        'UNIQUE (purse_deletion_serial_id)'
  );
END $$;
CREATE OR REPLACE FUNCTION master_table_purse_requests_was_deleted (
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'purse_requests';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE exchange.' || table_name ||
    ' ADD COLUMN'
    ' was_deleted BOOLEAN NOT NULL DEFAULT(FALSE)'
  );
  COMMENT ON COLUMN purse_requests.was_deleted
    IS 'TRUE if the purse was explicitly deleted (purse must have an entry in the purse_deletion table)';
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('purse_deletion'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('purse_deletion'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('purse_requests_was_deleted'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
CREATE OR REPLACE FUNCTION create_table_kyc_attributes(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'kyc_attributes';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I'
      '(kyc_attributes_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_payto BYTEA PRIMARY KEY CHECK (LENGTH(h_payto)=32)'
      ',kyc_prox BYTEA NOT NULL CHECK (LENGTH(kyc_prox)=32)'
      ',provider TEXT NOT NULL'
      ',satisfied_checks TEXT[] NOT NULL'
      ',collection_time INT8 NOT NULL'
      ',expiration_time INT8 NOT NULL'
      ',encrypted_attributes BYTEA NOT NULL'
      ',legitimization_serial INT8 NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (h_payto)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'KYC data about particular payment addresses'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash of payto://-URI the attributes are about'
    ,'h_payto'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'short hash of normalized full name and birthdate; used to efficiently find likely duplicate users'
    ,'kyc_prox'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'time when the attributes were collected by the provider'
    ,'collection_time'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'time when the attributes should no longer be considered validated'
    ,'expiration_time'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'configuration section name of the provider that affirmed the attributes'
    ,'provider'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     '(encrypted) JSON object (as string) with the attributes'
    ,'encrypted_attributes'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Reference the legitimization process for which these attributes are gathered for.'
    ,'legitimization_serial'
    ,table_name
    ,partition_suffix
  );
END $$;
COMMENT ON FUNCTION create_table_kyc_attributes
  IS 'Creates the kyc_attributes table';
CREATE OR REPLACE FUNCTION constrain_table_kyc_attributes(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'kyc_attributes';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
      ' ADD CONSTRAINT ' || table_name || '_serial_key '
        'UNIQUE (kyc_attributes_serial_id)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_similarity_index '
    'ON ' || table_name || ' '
    '(kyc_prox);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_expiration_time '
    'ON ' || table_name || ' '
    '(expiration_time ASC);'
  );
END $$;
CREATE OR REPLACE FUNCTION foreign_table_kyc_attributes()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'kyc_attributes';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_legitimization_processes'
    ' FOREIGN KEY (legitimization_serial) '
    ' REFERENCES legitimization_processes (legitimization_process_serial_id)' 
  );
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('kyc_attributes'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('kyc_attributes'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('kyc_attributes'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
CREATE OR REPLACE FUNCTION create_table_aml_status(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'aml_status';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I'
      '(aml_status_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_payto BYTEA PRIMARY KEY CHECK (LENGTH(h_payto)=32)'
      ',threshold taler_amount NOT NULL DEFAULT(0,0)'
      ',status INT4 NOT NULL DEFAULT(0)'
      ',kyc_requirement INT8 NOT NULL DEFAULT(0)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (h_payto)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'AML status for a particular payment destination'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash of the payto://-URI this AML status is about'
    ,'h_payto'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'monthly inbound transaction limit below which we are OK (if status is 1)'
    ,'threshold'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     '0 for all OK, 1 for AML decision required, 2 for account is frozen (prevents further transactions)'
    ,'status'
    ,table_name
    ,partition_suffix
  );
END $$;
COMMENT ON FUNCTION create_table_aml_status
  IS 'Creates the aml_status table';
CREATE OR REPLACE FUNCTION constrain_table_aml_status(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'aml_status';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
      ' ADD CONSTRAINT ' || table_name || '_serial_key '
        'UNIQUE (aml_status_serial_id)'
  );
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('aml_status'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('aml_status'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE TABLE aml_staff
  (aml_staff_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY UNIQUE
  ,decider_pub BYTEA PRIMARY KEY CHECK (LENGTH(decider_pub)=32)
  ,master_sig BYTEA CHECK (LENGTH(master_sig)=64)
  ,decider_name TEXT NOT NULL
  ,is_active BOOLEAN NOT NULL
  ,read_only BOOLEAN NOT NULL
  ,last_change INT8 NOT NULL
  );
COMMENT ON TABLE aml_staff
  IS 'Table with AML staff members the exchange uses or has used in the past. Entries never expire as we need to remember the last_change column indefinitely.';
COMMENT ON COLUMN aml_staff.decider_pub
  IS 'Public key of the AML staff member.';
COMMENT ON COLUMN aml_staff.master_sig
  IS 'The master public key signature on the AML staff member status, of type TALER_SIGNATURE_MASTER_AML_KEY.';
COMMENT ON COLUMN aml_staff.decider_name
  IS 'Name of the staff member.';
COMMENT ON COLUMN aml_staff.is_active
  IS 'true if we are currently supporting the use of this AML staff member.';
COMMENT ON COLUMN aml_staff.is_active
  IS 'true if the member has read-only access.';
COMMENT ON COLUMN aml_staff.last_change
  IS 'Latest time when active status changed. Used to detect replays of old messages.';
CREATE OR REPLACE FUNCTION create_table_aml_history(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'aml_history';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE IF NOT EXISTS %I'
      '(aml_history_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_payto BYTEA CHECK (LENGTH(h_payto)=32)'
      ',new_threshold taler_amount NOT NULL DEFAULT(0,0)'
      ',new_status INT4 NOT NULL DEFAULT(0)'
      ',decision_time INT8 NOT NULL DEFAULT(0)'
      ',justification TEXT NOT NULL'
      ',kyc_requirements TEXT'
      ',kyc_req_row INT8 NOT NULL DEFAULT(0)'
      ',decider_pub BYTEA CHECK (LENGTH(decider_pub)=32)'
      ',decider_sig BYTEA CHECK (LENGTH(decider_sig)=64)'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (h_payto)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'AML decision history for a particular payment destination'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash of the payto://-URI this AML history is about'
    ,'h_payto'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'new monthly inbound transaction limit below which we are OK'
    ,'new_threshold'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     '0 for all OK, 1 for AML decision required, 2 for account is frozen (prevents further transactions)'
    ,'new_status'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'when was the status changed'
    ,'decision_time'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'human-readable justification for the status change'
    ,'justification'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Public key of the staff member who made the AML decision'
    ,'decider_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Additional KYC requirements imposed by the AML staff member. Serialized JSON array of strings.'
    ,'kyc_requirements'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Row in the KYC table for this KYC requirement, 0 for none.'
    ,'kyc_req_row'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature key of the staff member affirming the AML decision; of type AML_DECISION'
    ,'decider_sig'
    ,table_name
    ,partition_suffix
  );
END $$;
COMMENT ON FUNCTION create_table_aml_history
  IS 'Creates the aml_history table';
CREATE OR REPLACE FUNCTION constrain_table_aml_history(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'aml_history';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
      ' ADD CONSTRAINT ' || table_name || '_serial_key '
        'UNIQUE (aml_history_serial_id)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_main_index '
    'ON ' || table_name || ' '
    '(h_payto, decision_time DESC);'
  );
END $$;
INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('aml_history'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('aml_history'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE);
CREATE FUNCTION create_table_age_withdraw(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'age_withdraw';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(age_withdraw_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_commitment BYTEA NOT NULL CONSTRAINT h_commitment_length CHECK(LENGTH(h_commitment)=64)'
      ',max_age SMALLINT NOT NULL CONSTRAINT max_age_positive CHECK(max_age>=0)'
      ',amount_with_fee taler_amount NOT NULL'
      ',reserve_pub BYTEA NOT NULL CONSTRAINT reserve_pub_length CHECK(LENGTH(reserve_pub)=32)'
      ',reserve_sig BYTEA NOT NULL CONSTRAINT reserve_sig_length CHECK(LENGTH(reserve_sig)=64)'
      ',noreveal_index SMALLINT NOT NULL CONSTRAINT noreveal_index_positive CHECK(noreveal_index>=0)'
      ',h_blind_evs BYTEA[] NOT NULL CONSTRAINT h_blind_evs_length CHECK(cardinality(h_blind_evs)=cardinality(denom_serials))'
      ',denom_serials INT8[] NOT NULL CONSTRAINT denom_serials_array_length CHECK(cardinality(denom_serials)=cardinality(denom_sigs))'
      ',denom_sigs BYTEA[] NOT NULL CONSTRAINT denom_sigs_array_length CHECK(cardinality(denom_sigs)=cardinality(denom_serials))'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (reserve_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Commitments made when withdrawing coins with age restriction and the gamma value chosen by the exchange. '
     'It also contains the blindly signed coins, their signatures and denominations.'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'The gamma value chosen by the exchange in the cut-and-choose protocol'
    ,'noreveal_index'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'The maximum age (in years) that the client commits to with this request'
    ,'max_age'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Commitment made by the client, hash over the various client inputs in the cut-and-choose protocol'
    ,'h_commitment'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Reference to the public key of the reserve from which the coins are going to be withdrawn'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Signature of the reserve''s private key over the age-withdraw request'
    ,'reserve_sig'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Array of references to the denominations'
    ,'denom_serials'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Array of the blinded envelopes of the chosen fresh coins, with value as given by the denomination in the corresponding slot in denom_serials'
    ,'h_blind_evs'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Array of signatures over each blinded envelope'
    ,'denom_sigs'
    ,table_name
    ,partition_suffix
  );
END
$$;
CREATE FUNCTION constrain_table_age_withdraw(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'age_withdraw';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD PRIMARY KEY (h_commitment);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_h_commitment_reserve_pub_key'
    ' UNIQUE (h_commitment, reserve_pub);'
  );
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_age_withdraw_id_key'
    ' UNIQUE (age_withdraw_id);'
  );
END
$$;
CREATE FUNCTION foreign_table_age_withdraw()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'age_withdraw';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_reserve_pub'
    ' FOREIGN KEY (reserve_pub)'
    ' REFERENCES reserves(reserve_pub) ON DELETE CASCADE;'
  );
END
$$;
INSERT INTO exchange_tables
  (name
  ,version
  ,action
  ,partitioned
  ,by_range)
VALUES
  ('age_withdraw', 'exchange-0002', 'create', TRUE ,FALSE),
  ('age_withdraw', 'exchange-0002', 'constrain',TRUE ,FALSE),
  ('age_withdraw', 'exchange-0002', 'foreign', TRUE ,FALSE);
COMMIT;
