BEGIN;
SET search_path TO exchange;
CREATE OR REPLACE FUNCTION amount_normalize(
    IN amount taler_amount
  ,OUT normalized taler_amount
)
LANGUAGE plpgsql
AS $$
BEGIN
  normalized.val = amount.val + amount.frac / 100000000;
  normalized.frac = amount.frac % 100000000;
END $$;
COMMENT ON FUNCTION amount_normalize
  IS 'Returns the normalized amount by adding to the .val the value of (.frac / 100000000) and removing the modulus 100000000 from .frac.';
CREATE OR REPLACE FUNCTION amount_add(
   IN a taler_amount
  ,IN b taler_amount
  ,OUT sum taler_amount
)
LANGUAGE plpgsql
AS $$
BEGIN
  sum = (a.val + b.val, a.frac + b.frac);
  CALL amount_normalize(sum ,sum);
  IF (sum.val > (1<<52))
  THEN
    RAISE EXCEPTION 'addition overflow';
  END IF;
END $$;
COMMENT ON FUNCTION amount_add
  IS 'Returns the normalized sum of two amounts. It raises an exception when the resulting .val is larger than 2^52';
CREATE OR REPLACE FUNCTION amount_left_minus_right(
  IN l taler_amount
 ,IN r taler_amount
 ,OUT diff taler_amount
 ,OUT ok BOOLEAN
)
LANGUAGE plpgsql
AS $$
BEGIN
IF (l.val > r.val)
THEN
  ok = TRUE;
  IF (l.frac >= r.frac)
  THEN
    diff.val = l.val - r.val;
    diff.frac = l.frac - r.frac;
  ELSE
    diff.val = l.val - r.val - 1;
    diff.frac = l.frac + 100000000 - r.frac;
  END IF;
ELSE
  IF (l.val = r.val) AND (l.frac >= r.frac)
  THEN
    diff.val = 0;
    diff.frac = l.frac - r.frac;
    ok = TRUE;
  ELSE
    diff = (-1, -1);
    ok = FALSE;
  END IF;
END IF;
RETURN;
END $$;
COMMENT ON FUNCTION amount_left_minus_right
  IS 'Subtracts the right amount from the left and returns the difference and TRUE, if the left amount is larger than the right, or an invalid amount and FALSE otherwise.';
CREATE OR REPLACE FUNCTION exchange_do_batch_withdraw(
  IN amount taler_amount,
  IN rpub BYTEA,
  IN now INT8,
  IN min_reserve_gc INT8,
  IN do_age_check BOOLEAN,
  OUT reserve_found BOOLEAN,
  OUT balance_ok BOOLEAN,
  OUT reserve_balance taler_amount,
  OUT age_ok BOOLEAN,
  OUT allowed_maximum_age INT2, 
  OUT ruuid INT8)
LANGUAGE plpgsql
AS $$
DECLARE
  reserve RECORD;
  balance taler_amount;
  not_before date;
BEGIN
SELECT current_balance
      ,reserve_uuid
      ,birthday
      ,gc_date
  INTO reserve
  FROM exchange.reserves
 WHERE reserves.reserve_pub=rpub;
IF NOT FOUND
THEN
  reserve_found=FALSE;
  balance_ok=FALSE;
  reserve_balance.frac = 0;
  reserve_balance.val = 0;
  age_ok=FALSE;
  allowed_maximum_age=0;
  ruuid=2;
  RETURN;
END IF;
reserve_found=TRUE;
reserve_balance = reserve.current_balance;
ruuid = reserve.reserve_uuid;
IF ((NOT do_age_check) OR (reserve.birthday = 0))
THEN
  age_ok = TRUE;
  allowed_maximum_age = -1;
ELSE
  not_before=date '1970-01-01' + reserve.birthday;
  allowed_maximum_age = extract(year from age(current_date, not_before));
  balance_ok=FALSE;
  age_ok = FALSE;
  RETURN;
END IF;
IF (reserve_balance.val > amount.val)
THEN
  IF (reserve_balance.frac >= amount.frac)
  THEN
    balance.val=reserve_balance.val - amount.val;
    balance.frac=reserve_balance.frac - amount.frac;
  ELSE
    balance.val=reserve_balance.val - amount.val - 1;
    balance.frac=reserve_balance.frac + 100000000 - amount.frac;
  END IF;
ELSE
  IF (reserve_balance.val = amount.val) AND (reserve_balance.frac >= amount.frac)
  THEN
    balance.val=0;
    balance.frac=reserve_balance.frac - amount.frac;
  ELSE
    balance_ok=FALSE;
    RETURN;
  END IF;
END IF;
min_reserve_gc=GREATEST(min_reserve_gc,reserve.gc_date);
UPDATE reserves SET
  gc_date=min_reserve_gc
 ,current_balance=balance
WHERE
  reserves.reserve_pub=rpub;
balance_ok=TRUE;
END $$;
COMMENT ON FUNCTION exchange_do_batch_withdraw(taler_amount, BYTEA, INT8, INT8, BOOLEAN)
  IS 'Checks whether the reserve has sufficient balance for a withdraw operation (or the request is repeated and was previously approved) and that age requirements are formally met. If so updates the database with the result. Excludes storing the planchets.';
CREATE OR REPLACE FUNCTION exchange_do_batch_withdraw_insert(
  IN cs_nonce BYTEA,
  IN amount taler_amount,
  IN h_denom_pub BYTEA, 
  IN ruuid INT8,
  IN reserve_sig BYTEA,
  IN h_coin_envelope BYTEA,
  IN denom_sig BYTEA,
  IN now INT8,
  OUT out_denom_unknown BOOLEAN,
  OUT out_nonce_reuse BOOLEAN,
  OUT out_conflict BOOLEAN)
LANGUAGE plpgsql
AS $$
DECLARE
  denom_serial INT8;
BEGIN
out_denom_unknown=TRUE;
out_conflict=TRUE;
out_nonce_reuse=TRUE;
SELECT denominations_serial
  INTO denom_serial
  FROM exchange.denominations
 WHERE denom_pub_hash=h_denom_pub;
IF NOT FOUND
THEN
  out_denom_unknown=TRUE;
  ASSERT false, 'denomination unknown';
  RETURN;
END IF;
out_denom_unknown=FALSE;
INSERT INTO exchange.reserves_out
  (h_blind_ev
  ,denominations_serial
  ,denom_sig
  ,reserve_uuid
  ,reserve_sig
  ,execution_date
  ,amount_with_fee)
VALUES
  (h_coin_envelope
  ,denom_serial
  ,denom_sig
  ,ruuid
  ,reserve_sig
  ,now
  ,amount)
ON CONFLICT DO NOTHING;
IF NOT FOUND
THEN
  out_conflict=TRUE;
  RETURN;
END IF;
out_conflict=FALSE;
out_nonce_reuse=FALSE;
IF NOT NULL cs_nonce
THEN
  INSERT INTO exchange.cs_nonce_locks
    (nonce
    ,max_denomination_serial
    ,op_hash)
  VALUES
    (cs_nonce
    ,denom_serial
    ,h_coin_envelope)
  ON CONFLICT DO NOTHING;
  IF NOT FOUND
  THEN
    SELECT 1
      FROM exchange.cs_nonce_locks
     WHERE nonce=cs_nonce
       AND op_hash=h_coin_envelope;
    IF NOT FOUND
    THEN
      out_nonce_reuse=TRUE;
      ASSERT false, 'nonce reuse attempted by client';
      RETURN;
    END IF;
  END IF;
END IF;
END $$;
COMMENT ON FUNCTION exchange_do_batch_withdraw_insert(BYTEA, taler_amount, BYTEA, INT8, BYTEA, BYTEA, BYTEA, INT8)
  IS 'Stores information about a planchet for a batch withdraw operation. Checks if the planchet already exists, and in that case indicates a conflict';
CREATE OR REPLACE FUNCTION exchange_do_age_withdraw(
  IN amount_with_fee taler_amount,
  IN rpub BYTEA,
  IN rsig BYTEA,
  IN now INT8,
  IN min_reserve_gc INT8,
  IN h_commitment BYTEA,
  IN maximum_age_committed INT2, 
  IN noreveal_index INT2,
  IN blinded_evs BYTEA[],
  IN denom_serials INT8[],
  IN denom_sigs BYTEA[],
  OUT reserve_found BOOLEAN,
  OUT balance_ok BOOLEAN,
  OUT reserve_balance taler_amount,
  OUT age_ok BOOLEAN,
  OUT required_age INT2, 
  OUT reserve_birthday INT4,
  OUT conflict BOOLEAN)
LANGUAGE plpgsql
AS $$
DECLARE
  reserve RECORD;
  difference RECORD;
  balance taler_amount;
  not_before date;
  earliest_date date;
BEGIN
SELECT current_balance
      ,birthday
      ,gc_date
  INTO reserve
  FROM exchange.reserves
 WHERE reserves.reserve_pub=rpub;
IF NOT FOUND
THEN
  reserve_found=FALSE;
  age_ok = FALSE;
  required_age=-1;
  conflict=FALSE;
  reserve_balance.val = 0;
  reserve_balance.frac = 0;
  balance_ok=FALSE;
  RETURN;
END IF;
reserve_found = TRUE;
conflict=FALSE; 
reserve_balance = reserve.current_balance;
reserve_birthday = reserve.birthday;
IF (reserve.birthday <> 0)
THEN
  not_before=date '1970-01-01' + reserve.birthday;
  earliest_date = current_date - make_interval(maximum_age_committed);
  IF (earliest_date < not_before)
  THEN
    required_age = extract(year from age(current_date, not_before));
    age_ok = FALSE;
    balance_ok=TRUE; 
    RETURN;
  END IF;
END IF;
age_ok = TRUE;
required_age=0;
SELECT *
INTO difference
FROM amount_left_minus_right(reserve_balance
                            ,amount_with_fee);
balance_ok = difference.ok;
IF NOT balance_ok
THEN
  RETURN;
END IF;
balance = difference.diff;
min_reserve_gc=GREATEST(min_reserve_gc,reserve.gc_date);
UPDATE reserves SET
  gc_date=min_reserve_gc
 ,current_balance=balance
WHERE
  reserves.reserve_pub=rpub;
INSERT INTO exchange.age_withdraw
  (h_commitment
  ,max_age
  ,amount_with_fee
  ,reserve_pub
  ,reserve_sig
  ,noreveal_index
  ,denom_serials
  ,h_blind_evs
  ,denom_sigs)
VALUES
  (h_commitment
  ,maximum_age_committed
  ,amount_with_fee
  ,rpub
  ,rsig
  ,noreveal_index
  ,denom_serials
  ,blinded_evs
  ,denom_sigs)
ON CONFLICT DO NOTHING;
IF NOT FOUND
THEN
  conflict=TRUE;
  RETURN;
ELSE
  conflict=FALSE;
END IF;
END $$;
COMMENT ON FUNCTION exchange_do_age_withdraw(taler_amount, BYTEA, BYTEA, INT8, INT8, BYTEA, INT2, INT2, BYTEA[], INT8[], BYTEA[])
  IS 'Checks whether the reserve has sufficient balance for an age-withdraw operation (or the request is repeated and was previously approved) and that age requirements are met. If so updates the database with the result. Includes storing the blinded planchets and denomination signatures, or signaling conflict';
CREATE OR REPLACE FUNCTION exchange_do_deposit(
  IN in_shard INT8,
  IN in_merchant_pub BYTEA,
  IN in_wallet_timestamp INT8,
  IN in_exchange_timestamp INT8,
  IN in_refund_deadline INT8,
  IN in_wire_deadline INT8,
  IN in_h_contract_terms BYTEA,
  IN in_wallet_data_hash BYTEA, 
  IN in_wire_salt BYTEA,
  IN in_wire_target_h_payto BYTEA,
  IN in_policy_details_serial_id INT8, 
  IN in_policy_blocked BOOLEAN,
  IN in_receiver_wire_account TEXT,
  IN ina_coin_pub BYTEA[],
  IN ina_coin_sig BYTEA[],
  IN ina_amount_with_fee taler_amount[],
  OUT out_exchange_timestamp INT8,
  OUT out_insufficient_balance_coin_index INT4, 
  OUT out_conflict BOOL
 )
LANGUAGE plpgsql
AS $$
DECLARE
  wtsi INT8; 
  bdsi INT8; 
  i INT4;
  ini_amount_with_fee taler_amount;
  ini_coin_pub BYTEA;
  ini_coin_sig BYTEA;
BEGIN
INSERT INTO wire_targets
    (wire_target_h_payto
    ,payto_uri)
  VALUES
    (in_wire_target_h_payto
    ,in_receiver_wire_account)
  ON CONFLICT DO NOTHING 
  RETURNING
    wire_target_serial_id
  INTO
    wtsi;
IF NOT FOUND
THEN
  SELECT
    wire_target_serial_id
  INTO
    wtsi
  FROM wire_targets
  WHERE
    wire_target_h_payto=in_wire_target_h_payto;
END IF;
INSERT INTO batch_deposits
  (shard
  ,merchant_pub
  ,wallet_timestamp
  ,exchange_timestamp
  ,refund_deadline
  ,wire_deadline
  ,h_contract_terms
  ,wallet_data_hash
  ,wire_salt
  ,wire_target_h_payto
  ,policy_details_serial_id
  ,policy_blocked
  )
  VALUES
  (in_shard
  ,in_merchant_pub
  ,in_wallet_timestamp
  ,in_exchange_timestamp
  ,in_refund_deadline
  ,in_wire_deadline
  ,in_h_contract_terms
  ,in_wallet_data_hash
  ,in_wire_salt
  ,in_wire_target_h_payto
  ,in_policy_details_serial_id
  ,in_policy_blocked)
  ON CONFLICT DO NOTHING 
  RETURNING
    batch_deposit_serial_id
  INTO
    bdsi;
IF NOT FOUND
THEN
  SELECT
      exchange_timestamp
     ,batch_deposit_serial_id
   INTO
      out_exchange_timestamp
     ,bdsi
   FROM batch_deposits
   WHERE shard=in_shard
     AND merchant_pub=in_merchant_pub
     AND h_contract_terms=in_h_contract_terms
     AND wire_target_h_payto=in_wire_target_h_payto
     AND ( (wallet_data_hash=in_wallet_data_hash) OR
           (wallet_data_hash IS NULL AND in_wallet_data_hash IS NULL) )
     AND wire_salt=in_wire_salt
     AND wallet_timestamp=in_wallet_timestamp
     AND refund_deadline=in_refund_deadline
     AND wire_deadline=in_wire_deadline
     AND ( (policy_details_serial_id=in_policy_details_serial_id) OR
           (policy_details_serial_id IS NULL AND in_policy_details_serial_id IS NULL) );
  IF NOT FOUND
  THEN
    out_conflict=TRUE;
    RETURN;
  END IF;
END IF;
out_conflict=FALSE;
FOR i IN 1..array_length(ina_coin_pub,1)
LOOP
  ini_coin_pub = ina_coin_pub[i];
  ini_coin_sig = ina_coin_sig[i];
  ini_amount_with_fee = ina_amount_with_fee[i];
  INSERT INTO coin_deposits
    (batch_deposit_serial_id
    ,coin_pub
    ,coin_sig
    ,amount_with_fee
    )
    VALUES
    (bdsi
    ,ini_coin_pub
    ,ini_coin_sig
    ,ini_amount_with_fee
    )
    ON CONFLICT DO NOTHING;
  IF FOUND
  THEN
    UPDATE known_coins kc
      SET
        remaining.frac=(kc.remaining).frac-ini_amount_with_fee.frac
          + CASE
              WHEN (kc.remaining).frac < ini_amount_with_fee.frac
              THEN 100000000
              ELSE 0
            END,
        remaining.val=(kc.remaining).val-ini_amount_with_fee.val
          - CASE
              WHEN (kc.remaining).frac < ini_amount_with_fee.frac
              THEN 1
              ELSE 0
            END
      WHERE coin_pub=ini_coin_pub
        AND ( ((kc.remaining).val > ini_amount_with_fee.val) OR
              ( ((kc.remaining).frac >= ini_amount_with_fee.frac) AND
                ((kc.remaining).val >= ini_amount_with_fee.val) ) );
    IF NOT FOUND
    THEN
      out_insufficient_balance_coin_index=i-1;
      RETURN;
    END IF;
  END IF;
END LOOP; 
END $$;
CREATE OR REPLACE FUNCTION exchange_do_melt(
  IN in_cs_rms BYTEA,
  IN in_amount_with_fee taler_amount,
  IN in_rc BYTEA,
  IN in_old_coin_pub BYTEA,
  IN in_old_coin_sig BYTEA,
  IN in_known_coin_id INT8, 
  IN in_noreveal_index INT4,
  IN in_zombie_required BOOLEAN,
  OUT out_balance_ok BOOLEAN,
  OUT out_zombie_bad BOOLEAN,
  OUT out_noreveal_index INT4)
LANGUAGE plpgsql
AS $$
DECLARE
  denom_max INT8;
BEGIN
INSERT INTO exchange.refresh_commitments
  (rc
  ,old_coin_pub
  ,old_coin_sig
  ,amount_with_fee
  ,noreveal_index
  )
  VALUES
  (in_rc
  ,in_old_coin_pub
  ,in_old_coin_sig
  ,in_amount_with_fee
  ,in_noreveal_index)
  ON CONFLICT DO NOTHING;
IF NOT FOUND
THEN
  out_noreveal_index=-1;
  SELECT
     noreveal_index
    INTO
     out_noreveal_index
    FROM exchange.refresh_commitments
   WHERE rc=in_rc;
  out_balance_ok=FOUND;
  out_zombie_bad=FALSE; 
  RETURN;
END IF;
IF in_zombie_required
THEN
  PERFORM
    FROM recoup_refresh
   WHERE rrc_serial IN
    (SELECT rrc_serial
       FROM refresh_revealed_coins
      WHERE melt_serial_id IN
      (SELECT melt_serial_id
         FROM refresh_commitments
        WHERE old_coin_pub=in_old_coin_pub));
  IF NOT FOUND
  THEN
    out_zombie_bad=TRUE;
    out_balance_ok=FALSE;
    RETURN;
  END IF;
END IF;
out_zombie_bad=FALSE; 
UPDATE known_coins kc
  SET
    remaining.frac=(kc.remaining).frac-in_amount_with_fee.frac
       + CASE
         WHEN (kc.remaining).frac < in_amount_with_fee.frac
         THEN 100000000
         ELSE 0
         END,
    remaining.val=(kc.remaining).val-in_amount_with_fee.val
       - CASE
         WHEN (kc.remaining).frac < in_amount_with_fee.frac
         THEN 1
         ELSE 0
         END
  WHERE coin_pub=in_old_coin_pub
    AND ( ((kc.remaining).val > in_amount_with_fee.val) OR
          ( ((kc.remaining).frac >= in_amount_with_fee.frac) AND
            ((kc.remaining).val >= in_amount_with_fee.val) ) );
IF NOT FOUND
THEN
  out_noreveal_index=-1;
  out_balance_ok=FALSE;
  RETURN;
END IF;
IF NOT NULL in_cs_rms
THEN
  SELECT
      denominations_serial
    INTO
      denom_max
    FROM exchange.denominations
      ORDER BY denominations_serial DESC
      LIMIT 1;
  INSERT INTO exchange.cs_nonce_locks
    (nonce
    ,max_denomination_serial
    ,op_hash)
  VALUES
    (cs_rms
    ,denom_serial
    ,in_rc)
  ON CONFLICT DO NOTHING;
  IF NOT FOUND
  THEN
    SELECT 1
      FROM exchange.cs_nonce_locks
     WHERE nonce=cs_rms
       AND op_hash=in_rc;
    IF NOT FOUND
    THEN
       out_balance_ok=FALSE;
       out_zombie_bad=FALSE;
       out_noreveal_index=42; 
       ASSERT false, 'nonce reuse attempted by client';
    END IF;
  END IF;
END IF;
out_balance_ok=TRUE;
out_noreveal_index=in_noreveal_index;
END $$;
CREATE OR REPLACE FUNCTION exchange_do_select_deposits_missing_wire(
  IN in_min_serial_id INT8)
RETURNS SETOF exchange_do_select_deposits_missing_wire_return_type
LANGUAGE plpgsql
AS $$
DECLARE
  missing CURSOR
  FOR
  SELECT
    batch_deposit_serial_id
   ,wire_target_h_payto
   ,wire_deadline
    FROM batch_deposits
    WHERE batch_deposit_serial_id > in_min_serial_id
    ORDER BY batch_deposit_serial_id ASC;
DECLARE
  my_total_val INT8; 
DECLARE
  my_total_frac INT8; 
DECLARE
  my_total taler_amount; 
DECLARE
  my_batch_record RECORD;
DECLARE
  i RECORD;
BEGIN
OPEN missing;
LOOP
  FETCH NEXT FROM missing INTO i;
  EXIT WHEN NOT FOUND;
  SELECT
    SUM((cdep.amount_with_fee).val) AS total_val
   ,SUM((cdep.amount_with_fee).frac::INT8) AS total_frac
    INTO
      my_batch_record
    FROM coin_deposits cdep
    WHERE cdep.batch_deposit_serial_id = i.batch_deposit_serial_id;
  my_total_val=my_batch_record.total_val;
  my_total_frac=my_batch_record.total_frac;
  my_total.val = my_total_val + my_total_frac / 100000000;
  my_total.frac = my_total_frac % 100000000;
  RETURN NEXT (
       i.batch_deposit_serial_id
      ,my_total
      ,i.wire_target_h_payto
      ,i.wire_deadline);
END LOOP;
CLOSE missing;
RETURN;
END $$;
CREATE OR REPLACE FUNCTION exchange_do_select_justification_missing_wire(
  IN in_wire_target_h_payto BYTEA,
  IN in_current_time INT8,
  OUT out_payto_uri TEXT, 
  OUT out_kyc_pending TEXT, 
  OUT out_aml_status INT4, 
  OUT out_aml_limit taler_amount) 
LANGUAGE plpgsql
AS $$
DECLARE
  my_required_checks TEXT[];
DECLARE
  my_aml_data RECORD;
DECLARE
  satisfied CURSOR FOR
  SELECT satisfied_checks
    FROM kyc_attributes
   WHERE h_payto=in_wire_target_h_payto
     AND expiration_time < in_current_time;
DECLARE
  i RECORD;
BEGIN
  out_payto_uri = NULL;
  SELECT payto_uri
    INTO out_payto_uri
    FROM wire_targets
   WHERE wire_target_h_payto=my_wire_target_h_payto;
  my_required_checks = NULL;
  SELECT string_to_array (required_checks, ' ')
    INTO my_required_checks
    FROM legitimization_requirements
    WHERE h_payto=my_wire_target_h_payto;
  SELECT
      new_threshold
     ,kyc_requirements
     ,new_status
    INTO
      my_aml_data
     FROM aml_history
    WHERE h_payto=in_wire_target_h_payto
    ORDER BY aml_history_serial_id 
      DESC LIMIT 1;
  IF FOUND
  THEN
    out_aml_limit=my_aml_data.new_threshold;
    out_aml_status=my_aml_data.kyc_status;
    my_required_checks
       = array_cat (my_required_checks,
                    my_aml_data.kyc_requirements);
  ELSE
    out_aml_limit=NULL;
    out_aml_status=0; 
  END IF;
  OPEN satisfied;
  LOOP
    FETCH NEXT FROM satisfied INTO i;
    EXIT WHEN NOT FOUND;
    FOR i in 1..array_length(i.satisfied_checks)
    LOOP
      my_required_checks
        = array_remove (my_required_checks,
                        i.satisfied_checks[i]);
    END LOOP;
  END LOOP;
  IF ( (my_required_checks IS NOT NULL) AND
       (0 < array_length(my_satisfied_checks)) )
  THEN
    out_kyc_pending
      = array_to_string (my_required_checks, ' ');
  END IF;
  RETURN;
END $$;
CREATE OR REPLACE FUNCTION exchange_do_refund(
  IN in_amount_with_fee taler_amount,
  IN in_amount taler_amount,
  IN in_deposit_fee taler_amount,
  IN in_h_contract_terms BYTEA,
  IN in_rtransaction_id INT8,
  IN in_deposit_shard INT8,
  IN in_known_coin_id INT8,
  IN in_coin_pub BYTEA,
  IN in_merchant_pub BYTEA,
  IN in_merchant_sig BYTEA,
  OUT out_not_found BOOLEAN,
  OUT out_refund_ok BOOLEAN,
  OUT out_gone BOOLEAN,
  OUT out_conflict BOOLEAN)
LANGUAGE plpgsql
AS $$
DECLARE
  bdsi INT8; 
DECLARE
  tmp_val INT8; 
DECLARE
  tmp_frac INT8; 
DECLARE
  tmp taler_amount; 
DECLARE
  deposit taler_amount; 
BEGIN
SELECT
   bdep.batch_deposit_serial_id
  ,(cdep.amount_with_fee).val
  ,(cdep.amount_with_fee).frac
  ,bdep.done
 INTO
   bdsi
  ,deposit.val
  ,deposit.frac
  ,out_gone
 FROM batch_deposits bdep
 JOIN coin_deposits cdep
   USING (batch_deposit_serial_id)
 WHERE cdep.coin_pub=in_coin_pub
  AND shard=in_deposit_shard
  AND merchant_pub=in_merchant_pub
  AND h_contract_terms=in_h_contract_terms;
IF NOT FOUND
THEN
  out_refund_ok=FALSE;
  out_conflict=FALSE;
  out_not_found=TRUE;
  out_gone=FALSE;
  RETURN;
END IF;
INSERT INTO refunds
  (batch_deposit_serial_id
  ,coin_pub
  ,merchant_sig
  ,rtransaction_id
  ,amount_with_fee
  )
  VALUES
  (bdsi
  ,in_coin_pub
  ,in_merchant_sig
  ,in_rtransaction_id
  ,in_amount_with_fee
  )
  ON CONFLICT DO NOTHING;
IF NOT FOUND
THEN
   PERFORM
   FROM exchange.refunds
   WHERE coin_pub=in_coin_pub
     AND batch_deposit_serial_id=bdsi
     AND rtransaction_id=in_rtransaction_id
     AND amount_with_fee=in_amount_with_fee;
  IF NOT FOUND
  THEN
    out_refund_ok=FALSE;
    out_conflict=TRUE;
    out_not_found=FALSE;
    RETURN;
  END IF;
  out_refund_ok=TRUE;
  out_conflict=FALSE;
  out_not_found=FALSE;
  out_gone=FALSE;
  RETURN;
END IF;
IF out_gone
THEN
  out_refund_ok=FALSE;
  out_conflict=FALSE;
  out_not_found=FALSE;
  RETURN;
END IF;
SELECT
   SUM((refs.amount_with_fee).val) 
  ,SUM(CAST((refs.amount_with_fee).frac AS INT8)) 
  INTO
   tmp_val
  ,tmp_frac
  FROM refunds refs
  WHERE coin_pub=in_coin_pub
    AND batch_deposit_serial_id=bdsi;
IF tmp_val IS NULL
THEN
  RAISE NOTICE 'failed to sum up existing refunds';
  out_refund_ok=FALSE;
  out_conflict=FALSE;
  out_not_found=FALSE;
  RETURN;
END IF;
tmp.val = tmp_val + tmp_frac / 100000000;
tmp.frac = tmp_frac % 100000000;
IF (tmp.val < deposit.val)
THEN
  out_refund_ok=TRUE;
ELSE
  IF (tmp.val = deposit.val) AND (tmp.frac <= deposit.frac)
  THEN
    out_refund_ok=TRUE;
  ELSE
    out_refund_ok=FALSE;
  END IF;
END IF;
IF (tmp.val = deposit.val) AND (tmp.frac = deposit.frac)
THEN
  in_amount.frac = in_amount.frac + in_deposit_fee.frac;
  in_amount.val = in_amount.val + in_deposit_fee.val;
  in_amount.val = in_amount.val + in_amount.frac / 100000000;
  in_amount.frac = in_amount.frac % 100000000;
END IF;
UPDATE known_coins kc
  SET
    remaining.frac=(kc.remaining).frac+in_amount.frac
       - CASE
         WHEN (kc.remaining).frac+in_amount.frac >= 100000000
         THEN 100000000
         ELSE 0
         END,
    remaining.val=(kc.remaining).val+in_amount.val
       + CASE
         WHEN (kc.remaining).frac+in_amount.frac >= 100000000
         THEN 1
         ELSE 0
         END
  WHERE coin_pub=in_coin_pub;
out_conflict=FALSE;
out_not_found=FALSE;
END $$;
COMMENT ON FUNCTION exchange_do_refund(taler_amount, taler_amount, taler_amount, BYTEA, INT8, INT8, INT8, BYTEA, BYTEA, BYTEA)
  IS 'Executes a refund operation, checking that the corresponding deposit was sufficient to cover the refunded amount';
CREATE OR REPLACE FUNCTION exchange_do_recoup_to_reserve(
  IN in_reserve_pub BYTEA,
  IN in_reserve_out_serial_id INT8,
  IN in_coin_blind BYTEA,
  IN in_coin_pub BYTEA,
  IN in_known_coin_id INT8,
  IN in_coin_sig BYTEA,
  IN in_reserve_gc INT8,
  IN in_reserve_expiration INT8,
  IN in_recoup_timestamp INT8,
  OUT out_recoup_ok BOOLEAN,
  OUT out_internal_failure BOOLEAN,
  OUT out_recoup_timestamp INT8)
LANGUAGE plpgsql
AS $$
DECLARE
  tmp taler_amount; 
  balance taler_amount; 
  new_balance taler_amount; 
  reserve RECORD;
  rval RECORD;
BEGIN
out_internal_failure=FALSE;
SELECT
   remaining
 INTO
   rval
FROM exchange.known_coins
  WHERE coin_pub=in_coin_pub;
IF NOT FOUND
THEN
  out_internal_failure=TRUE;
  out_recoup_ok=FALSE;
  RETURN;
END IF;
tmp := rval.remaining;
IF tmp.val + tmp.frac = 0
THEN
  SELECT
    recoup_timestamp
  INTO
    out_recoup_timestamp
    FROM exchange.recoup
    WHERE coin_pub=in_coin_pub;
  out_recoup_ok=FOUND;
  RETURN;
END IF;
UPDATE known_coins
  SET
     remaining.val = 0
    ,remaining.frac = 0
  WHERE coin_pub=in_coin_pub;
SELECT current_balance
  INTO reserve
  FROM reserves
 WHERE reserve_pub=in_reserve_pub;
balance = reserve.current_balance;
new_balance.frac=balance.frac+tmp.frac
   - CASE
     WHEN balance.frac+tmp.frac >= 100000000
     THEN 100000000
     ELSE 0
     END;
new_balance.val=balance.val+tmp.val
   + CASE
     WHEN balance.frac+tmp.frac >= 100000000
     THEN 1
     ELSE 0
     END;
UPDATE reserves
  SET
    current_balance = new_balance,
    gc_date=GREATEST(gc_date, in_reserve_gc),
    expiration_date=GREATEST(expiration_date, in_reserve_expiration)
  WHERE reserve_pub=in_reserve_pub;
IF NOT FOUND
THEN
  RAISE NOTICE 'failed to increase reserve balance from recoup';
  out_recoup_ok=TRUE;
  out_internal_failure=TRUE;
  RETURN;
END IF;
INSERT INTO exchange.recoup
  (coin_pub
  ,coin_sig
  ,coin_blind
  ,amount
  ,recoup_timestamp
  ,reserve_out_serial_id
  )
VALUES
  (in_coin_pub
  ,in_coin_sig
  ,in_coin_blind
  ,tmp
  ,in_recoup_timestamp
  ,in_reserve_out_serial_id);
out_recoup_ok=TRUE;
out_recoup_timestamp=in_recoup_timestamp;
END $$;
CREATE OR REPLACE FUNCTION exchange_do_recoup_to_coin(
  IN in_old_coin_pub BYTEA,
  IN in_rrc_serial INT8,
  IN in_coin_blind BYTEA,
  IN in_coin_pub BYTEA,
  IN in_known_coin_id INT8,
  IN in_coin_sig BYTEA,
  IN in_recoup_timestamp INT8,
  OUT out_recoup_ok BOOLEAN,
  OUT out_internal_failure BOOLEAN,
  OUT out_recoup_timestamp INT8)
LANGUAGE plpgsql
AS $$
DECLARE
  rval RECORD;
DECLARE
  tmp taler_amount; 
BEGIN
out_internal_failure=FALSE;
SELECT
   remaining
 INTO
   rval
FROM exchange.known_coins
  WHERE coin_pub=in_coin_pub;
IF NOT FOUND
THEN
  out_internal_failure=TRUE;
  out_recoup_ok=FALSE;
  RETURN;
END IF;
tmp := rval.remaining;
IF tmp.val + tmp.frac = 0
THEN
  SELECT
      recoup_timestamp
    INTO
      out_recoup_timestamp
    FROM recoup_refresh
    WHERE coin_pub=in_coin_pub;
  out_recoup_ok=FOUND;
  RETURN;
END IF;
UPDATE known_coins
  SET
     remaining.val = 0
    ,remaining.frac = 0
  WHERE coin_pub=in_coin_pub;
UPDATE known_coins kc
  SET
    remaining.frac=(kc.remaining).frac+tmp.frac
       - CASE
         WHEN (kc.remaining).frac+tmp.frac >= 100000000
         THEN 100000000
         ELSE 0
         END,
    remaining.val=(kc.remaining).val+tmp.val
       + CASE
         WHEN (kc.remaining).frac+tmp.frac >= 100000000
         THEN 1
         ELSE 0
         END
  WHERE coin_pub=in_old_coin_pub;
IF NOT FOUND
THEN
  RAISE NOTICE 'failed to increase old coin balance from recoup';
  out_recoup_ok=TRUE;
  out_internal_failure=TRUE;
  RETURN;
END IF;
INSERT INTO recoup_refresh
  (coin_pub
  ,known_coin_id
  ,coin_sig
  ,coin_blind
  ,amount
  ,recoup_timestamp
  ,rrc_serial
  )
VALUES
  (in_coin_pub
  ,in_known_coin_id
  ,in_coin_sig
  ,in_coin_blind
  ,tmp
  ,in_recoup_timestamp
  ,in_rrc_serial);
out_recoup_ok=TRUE;
out_recoup_timestamp=in_recoup_timestamp;
END $$;
CREATE OR REPLACE PROCEDURE exchange_do_gc(
  IN in_ancient_date INT8,
  IN in_now INT8)
LANGUAGE plpgsql
AS $$
DECLARE
  reserve_uuid_min INT8; 
  melt_min INT8; 
  coin_min INT8; 
  batch_deposit_min INT8; 
  reserve_out_min INT8; 
  denom_min INT8; 
BEGIN
DELETE FROM prewire
  WHERE finished=TRUE;
DELETE FROM wire_fee
  WHERE end_date < in_ancient_date;
DELETE FROM reserves
  WHERE gc_date < in_now
    AND current_balance = (0, 0);
SELECT
     reserve_out_serial_id
  INTO
     reserve_out_min
  FROM reserves_out
  ORDER BY reserve_out_serial_id ASC
  LIMIT 1;
DELETE FROM recoup
  WHERE reserve_out_serial_id < reserve_out_min;
SELECT
     reserve_uuid
  INTO
     reserve_uuid_min
  FROM reserves
  ORDER BY reserve_uuid ASC
  LIMIT 1;
DELETE FROM reserves_out
  WHERE reserve_uuid < reserve_uuid_min;
DELETE FROM denominations
  WHERE expire_legal < in_now
    AND denominations_serial NOT IN
      (SELECT DISTINCT denominations_serial
         FROM reserves_out)
    AND denominations_serial NOT IN
      (SELECT DISTINCT denominations_serial
         FROM known_coins
        WHERE coin_pub IN
          (SELECT DISTINCT coin_pub
             FROM recoup))
    AND denominations_serial NOT IN
      (SELECT DISTINCT denominations_serial
         FROM known_coins
        WHERE coin_pub IN
          (SELECT DISTINCT coin_pub
             FROM recoup_refresh));
SELECT
     melt_serial_id
  INTO
     melt_min
  FROM refresh_commitments
  ORDER BY melt_serial_id ASC
  LIMIT 1;
DELETE FROM refresh_revealed_coins
  WHERE melt_serial_id < melt_min;
DELETE FROM refresh_transfer_keys
  WHERE melt_serial_id < melt_min;
SELECT
     known_coin_id
  INTO
     coin_min
  FROM known_coins
  ORDER BY known_coin_id ASC
  LIMIT 1;
DELETE FROM recoup_refresh
  WHERE known_coin_id < coin_min;
DELETE FROM batch_deposits
  WHERE wire_deadline < in_ancient_date;
SELECT
     batch_deposit_serial_id
  INTO
     batch_deposit_min
  FROM coin_deposits
  ORDER BY batch_deposit_serial_id ASC
  LIMIT 1;
DELETE FROM refunds
  WHERE batch_deposit_serial_id < batch_deposit_min;
DELETE FROM aggregation_tracking
  WHERE batch_deposit_serial_id < batch_deposit_min;
DELETE FROM coin_deposits
  WHERE batch_deposit_serial_id < batch_deposit_min;
SELECT
     denominations_serial
  INTO
     denom_min
  FROM denominations
  ORDER BY denominations_serial ASC
  LIMIT 1;
DELETE FROM cs_nonce_locks
  WHERE max_denomination_serial <= denom_min;
END $$;
CREATE OR REPLACE FUNCTION exchange_do_purse_delete(
  IN in_purse_pub BYTEA,
  IN in_purse_sig BYTEA,
  IN in_now INT8,
  OUT out_decided BOOLEAN,
  OUT out_found BOOLEAN)
LANGUAGE plpgsql
AS $$
DECLARE
  my_deposit record;
DECLARE
  my_in_reserve_quota BOOLEAN;
BEGIN
PERFORM refunded FROM purse_decision
  WHERE purse_pub=in_purse_pub;
IF FOUND
THEN
  out_found=TRUE;
  out_decided=TRUE;
  RETURN;
END IF;
out_decided=FALSE;
SELECT in_reserve_quota
  INTO my_in_reserve_quota
  FROM exchange.purse_requests
 WHERE purse_pub=in_purse_pub;
out_found=FOUND;
IF NOT FOUND
THEN
  RETURN;
END IF;
INSERT INTO exchange.purse_deletion
  (purse_pub
  ,purse_sig)
VALUES
  (in_purse_pub
  ,in_purse_sig)
ON CONFLICT DO NOTHING;
IF NOT FOUND
THEN
  RETURN;
END IF;
DELETE FROM contracts
  WHERE purse_pub=in_purse_pub;
INSERT INTO purse_decision
  (purse_pub
  ,action_timestamp
  ,refunded)
VALUES
  (in_purse_pub
  ,in_now
  ,TRUE);
IF (my_in_reserve_quota)
THEN
  UPDATE reserves
    SET purses_active=purses_active-1
  WHERE reserve_pub IN
    (SELECT reserve_pub
       FROM exchange.purse_merges
      WHERE purse_pub=in_purse_pub
     LIMIT 1);
END IF;
FOR my_deposit IN
  SELECT coin_pub
        ,amount_with_fee
    FROM exchange.purse_deposits
  WHERE purse_pub = in_purse_pub
LOOP
  UPDATE known_coins kc SET
    remaining.frac=(kc.remaining).frac+(my_deposit.amount_with_fee).frac
     - CASE
       WHEN (kc.remaining).frac+(my_deposit.amount_with_fee).frac >= 100000000
       THEN 100000000
       ELSE 0
       END,
    remaining.val=(kc.remaining).val+(my_deposit.amount_with_fee).val
     + CASE
       WHEN (kc.remaining).frac+(my_deposit.amount_with_fee).frac >= 100000000
       THEN 1
       ELSE 0
       END
    WHERE coin_pub = my_deposit.coin_pub;
END LOOP;
END $$;
COMMENT ON FUNCTION exchange_do_purse_delete(BYTEA,BYTEA,INT8)
  IS 'Delete a previously undecided purse and refund the coins (if any).';
CREATE OR REPLACE FUNCTION exchange_do_purse_deposit(
  IN in_partner_id INT8,
  IN in_purse_pub BYTEA,
  IN in_amount_with_fee taler_amount,
  IN in_coin_pub BYTEA,
  IN in_coin_sig BYTEA,
  IN in_amount_without_fee taler_amount,
  IN in_reserve_expiration INT8,
  IN in_now INT8,
  OUT out_balance_ok BOOLEAN,
  OUT out_late BOOLEAN,
  OUT out_conflict BOOLEAN)
LANGUAGE plpgsql
AS $$
DECLARE
  was_merged BOOLEAN;
DECLARE
  psi INT8; 
DECLARE
  my_amount taler_amount; 
DECLARE
  was_paid BOOLEAN;
DECLARE
  my_in_reserve_quota BOOLEAN;
DECLARE
  my_reserve_pub BYTEA;
DECLARE
  rval RECORD;
BEGIN
INSERT INTO purse_deposits
  (partner_serial_id
  ,purse_pub
  ,coin_pub
  ,amount_with_fee
  ,coin_sig)
  VALUES
  (in_partner_id
  ,in_purse_pub
  ,in_coin_pub
  ,in_amount_with_fee
  ,in_coin_sig)
  ON CONFLICT DO NOTHING;
IF NOT FOUND
THEN
  PERFORM
  FROM purse_deposits
  WHERE purse_pub = in_purse_pub
    AND coin_pub = in_coin_pub
    AND coin_sig = in_coin_sig;
  IF NOT FOUND
  THEN
    out_balance_ok=FALSE;
    out_late=FALSE;
    out_conflict=TRUE;
    RETURN;
  ELSE
    out_late=FALSE;
    out_balance_ok=TRUE;
    out_conflict=FALSE;
    RETURN;
  END IF;
END IF;
PERFORM
  FROM exchange.purse_deletion
  WHERE purse_pub = in_purse_pub;
IF FOUND
THEN
  out_late=TRUE;
  out_balance_ok=FALSE;
  out_conflict=FALSE;
  RETURN;
END IF;
UPDATE known_coins kc
  SET
    remaining.frac=(kc.remaining).frac-in_amount_with_fee.frac
       + CASE
         WHEN (kc.remaining).frac < in_amount_with_fee.frac
         THEN 100000000
         ELSE 0
         END,
    remaining.val=(kc.remaining).val-in_amount_with_fee.val
       - CASE
         WHEN (kc.remaining).frac < in_amount_with_fee.frac
         THEN 1
         ELSE 0
         END
  WHERE coin_pub=in_coin_pub
    AND ( ((kc.remaining).val > in_amount_with_fee.val) OR
          ( ((kc.remaining).frac >= in_amount_with_fee.frac) AND
            ((kc.remaining).val >= in_amount_with_fee.val) ) );
IF NOT FOUND
THEN
  out_balance_ok=FALSE;
  out_late=FALSE;
  out_conflict=FALSE;
  RETURN;
END IF;
UPDATE purse_requests pr
  SET
    balance.frac=(pr.balance).frac+in_amount_without_fee.frac
       - CASE
         WHEN (pr.balance).frac+in_amount_without_fee.frac >= 100000000
         THEN 100000000
         ELSE 0
         END,
    balance.val=(pr.balance).val+in_amount_without_fee.val
       + CASE
         WHEN (pr.balance).frac+in_amount_without_fee.frac >= 100000000
         THEN 1
         ELSE 0
         END
  WHERE purse_pub=in_purse_pub;
out_conflict=FALSE;
out_balance_ok=TRUE;
SELECT COALESCE(partner_serial_id,0)
      ,reserve_pub
  INTO psi
      ,my_reserve_pub
  FROM purse_merges
 WHERE purse_pub=in_purse_pub;
IF NOT FOUND
THEN
  out_late=FALSE;
  RETURN;
END IF;
SELECT
    amount_with_fee
   ,in_reserve_quota
  INTO
    rval
  FROM exchange.purse_requests preq
  WHERE (purse_pub=in_purse_pub)
    AND ( ( ( ((preq.amount_with_fee).val <= (preq.balance).val)
          AND ((preq.amount_with_fee).frac <= (preq.balance).frac) )
         OR ((preq.amount_with_fee).val < (preq.balance).val) ) );
IF NOT FOUND
THEN
  out_late=FALSE;
  RETURN;
END IF;
my_amount := rval.amount_with_fee;
my_in_reserve_quota := rval.in_reserve_quota;
INSERT INTO purse_decision
  (purse_pub
  ,action_timestamp
  ,refunded)
VALUES
  (in_purse_pub
  ,in_now
  ,FALSE)
ON CONFLICT DO NOTHING;
IF NOT FOUND
THEN
  out_late=TRUE;
  RETURN;
END IF;
out_late=FALSE;
IF (my_in_reserve_quota)
THEN
  UPDATE reserves
    SET purses_active=purses_active-1
  WHERE reserve_pub IN
    (SELECT reserve_pub
       FROM purse_merges
      WHERE purse_pub=my_purse_pub
     LIMIT 1);
END IF;
IF (0 != psi)
THEN
  UPDATE purse_actions
     SET action_date=0 
        ,partner_serial_id=psi
   WHERE purse_pub=in_purse_pub;
ELSE
  INSERT INTO reserves
    (reserve_pub
    ,current_balance
    ,expiration_date
    ,gc_date)
  VALUES
    (my_reserve_pub
    ,my_amount
    ,in_reserve_expiration
    ,in_reserve_expiration)
  ON CONFLICT DO NOTHING;
  IF NOT FOUND
  THEN
    UPDATE reserves
      SET
       current_balance.frac=(current_balance).frac+my_amount.frac
        - CASE
          WHEN (current_balance).frac + my_amount.frac >= 100000000
            THEN 100000000
          ELSE 0
          END
      ,current_balance.val=(current_balance).val+my_amount.val
        + CASE
          WHEN (current_balance).frac + my_amount.frac >= 100000000
            THEN 1
          ELSE 0
          END
      ,expiration_date=GREATEST(expiration_date,in_reserve_expiration)
      ,gc_date=GREATEST(gc_date,in_reserve_expiration)
      WHERE reserve_pub=my_reserve_pub;
  END IF;
END IF;
END $$;
CREATE OR REPLACE FUNCTION exchange_do_purse_merge(
  IN in_purse_pub BYTEA,
  IN in_merge_sig BYTEA,
  IN in_merge_timestamp INT8,
  IN in_reserve_sig BYTEA,
  IN in_partner_url TEXT,
  IN in_reserve_pub BYTEA,
  IN in_wallet_h_payto BYTEA,
  IN in_expiration_date INT8,
  OUT out_no_partner BOOLEAN,
  OUT out_no_balance BOOLEAN,
  OUT out_conflict BOOLEAN)
LANGUAGE plpgsql
AS $$
DECLARE
  my_amount taler_amount;
DECLARE
  my_purse_fee taler_amount;
DECLARE
  my_partner_serial_id INT8;
DECLARE
  my_in_reserve_quota BOOLEAN;
DECLARE
  rval RECORD;
DECLARE
  reserve RECORD;
DECLARE
  balance taler_amount;
BEGIN
INSERT INTO reserves
  (reserve_pub
  ,expiration_date
  ,gc_date)
  VALUES
  (in_reserve_pub
  ,in_expiration_date
  ,in_expiration_date)
  ON CONFLICT DO NOTHING;
IF in_partner_url IS NULL
THEN
  my_partner_serial_id=NULL;
ELSE
  SELECT
    partner_serial_id
  INTO
    my_partner_serial_id
  FROM exchange.partners
  WHERE partner_base_url=in_partner_url
    AND start_date <= in_merge_timestamp
    AND end_date > in_merge_timestamp;
  IF NOT FOUND
  THEN
    out_no_partner=TRUE;
    out_conflict=FALSE;
    RETURN;
  END IF;
END IF;
out_no_partner=FALSE;
SELECT amount_with_fee
      ,purse_fee
      ,in_reserve_quota
  INTO rval
  FROM purse_requests pr
  WHERE purse_pub=in_purse_pub
    AND (pr.balance).val >= (pr.amount_with_fee).val
    AND ( (pr.balance).frac >= (pr.amount_with_fee).frac OR
          (pr.balance).val > (pr.amount_with_fee).val );
IF NOT FOUND
THEN
  out_no_balance=TRUE;
  out_conflict=FALSE;
  RETURN;
END IF;
my_amount := rval.amount_with_fee;
my_purse_fee := rval.purse_fee;
my_in_reserve_quota := rval.in_reserve_quota;
out_no_balance=FALSE;
INSERT INTO purse_merges
    (partner_serial_id
    ,reserve_pub
    ,purse_pub
    ,merge_sig
    ,merge_timestamp)
  VALUES
    (my_partner_serial_id
    ,in_reserve_pub
    ,in_purse_pub
    ,in_merge_sig
    ,in_merge_timestamp)
  ON CONFLICT DO NOTHING;
IF NOT FOUND
THEN
  PERFORM
  FROM purse_merges
  WHERE purse_pub=in_purse_pub
     AND merge_sig=in_merge_sig;
  IF NOT FOUND
  THEN
     out_conflict=TRUE;
     RETURN;
  END IF;
  out_conflict=FALSE;
  RETURN;
END IF;
INSERT INTO purse_decision
  (purse_pub
  ,action_timestamp
  ,refunded)
VALUES
  (in_purse_pub
  ,in_merge_timestamp
  ,FALSE)
ON CONFLICT DO NOTHING;
IF NOT FOUND
THEN
  out_conflict=TRUE;
  RETURN;
END IF;
out_conflict=FALSE;
IF (my_in_reserve_quota)
THEN
  UPDATE reserves
    SET purses_active=purses_active-1
  WHERE reserve_pub IN
    (SELECT reserve_pub
       FROM purse_merges
      WHERE purse_pub=my_purse_pub
     LIMIT 1);
END IF;
INSERT INTO account_merges
  (reserve_pub
  ,reserve_sig
  ,purse_pub
  ,wallet_h_payto)
  VALUES
  (in_reserve_pub
  ,in_reserve_sig
  ,in_purse_pub
  ,in_wallet_h_payto);
IF (0 != my_partner_serial_id)
THEN
  UPDATE purse_actions
     SET action_date=0 
        ,partner_serial_id=my_partner_serial_id
   WHERE purse_pub=in_purse_pub;
ELSE
  my_amount.val = my_amount.val + my_purse_fee.val;
  my_amount.frac = my_amount.frac + my_purse_fee.frac;
  my_amount.val = my_amount.val + my_amount.frac / 100000000;
  my_amount.frac = my_amount.frac % 100000000;
  SELECT *
   INTO reserve
   FROM exchange.reserves
  WHERE reserve_pub=in_reserve_pub;
  balance = reserve.current_balance;
  balance.frac=balance.frac+my_amount.frac
     - CASE
       WHEN balance.frac + my_amount.frac >= 100000000
       THEN 100000000
       ELSE 0
       END;
  balance.val=balance.val+my_amount.val
     + CASE
       WHEN balance.frac + my_amount.frac >= 100000000
       THEN 1
       ELSE 0
       END;
  UPDATE exchange.reserves
  SET current_balance=balance
  WHERE reserve_pub=in_reserve_pub;
END IF;
RETURN;
END $$;
COMMENT ON FUNCTION exchange_do_purse_merge(BYTEA, BYTEA, INT8, BYTEA, TEXT, BYTEA, BYTEA, INT8)
  IS 'Checks that the partner exists, the purse has not been merged with a different reserve and that the purse is full. If so, persists the merge data and either merges the purse with the reserve or marks it as ready for the taler-exchange-router. Caller MUST abort the transaction on failures so as to not persist data by accident.';
CREATE OR REPLACE FUNCTION exchange_do_reserve_purse(
  IN in_purse_pub BYTEA,
  IN in_merge_sig BYTEA,
  IN in_merge_timestamp INT8,
  IN in_reserve_expiration INT8,
  IN in_reserve_gc INT8,
  IN in_reserve_sig BYTEA,
  IN in_reserve_quota BOOLEAN,
  IN in_purse_fee taler_amount,
  IN in_reserve_pub BYTEA,
  IN in_wallet_h_payto BYTEA,
  OUT out_no_funds BOOLEAN,
  OUT out_no_reserve BOOLEAN,
  OUT out_conflict BOOLEAN)
LANGUAGE plpgsql
AS $$
BEGIN
INSERT INTO purse_merges
    (partner_serial_id
    ,reserve_pub
    ,purse_pub
    ,merge_sig
    ,merge_timestamp)
  VALUES
    (NULL
    ,in_reserve_pub
    ,in_purse_pub
    ,in_merge_sig
    ,in_merge_timestamp)
  ON CONFLICT DO NOTHING;
IF NOT FOUND
THEN
  PERFORM
  FROM purse_merges
  WHERE purse_pub=in_purse_pub
     AND merge_sig=in_merge_sig;
  IF NOT FOUND
  THEN
     out_conflict=TRUE;
     out_no_reserve=FALSE;
     out_no_funds=FALSE;
     RETURN;
  END IF;
  out_conflict=FALSE;
  out_no_funds=FALSE;
  out_no_reserve=FALSE;
  RETURN;
END IF;
out_conflict=FALSE;
PERFORM
  FROM exchange.reserves
 WHERE reserve_pub=in_reserve_pub;
out_no_reserve = NOT FOUND;
IF (in_reserve_quota)
THEN
  IF (out_no_reserve)
  THEN
    out_no_funds=TRUE;
    RETURN;
  END IF;
  UPDATE exchange.reserves
     SET purses_active=purses_active+1
   WHERE reserve_pub=in_reserve_pub
     AND purses_active < purses_allowed;
  IF NOT FOUND
  THEN
    out_no_funds=TRUE;
    RETURN;
  END IF;
ELSE
  IF (out_no_reserve)
  THEN
    IF ( (0 != in_purse_fee.val) OR
         (0 != in_purse_fee.frac) )
    THEN
      out_no_funds=TRUE;
      RETURN;
    END IF;
    INSERT INTO exchange.reserves
      (reserve_pub
      ,expiration_date
      ,gc_date)
    VALUES
      (in_reserve_pub
      ,in_reserve_expiration
      ,in_reserve_gc);
  ELSE
    UPDATE exchange.reserves
      SET
        current_balance.frac=(current_balance).frac-in_purse_fee.frac
         + CASE
         WHEN (current_balance).frac < in_purse_fee.frac
         THEN 100000000
         ELSE 0
         END,
       current_balance.val=(current_balance).val-in_purse_fee.val
         - CASE
         WHEN (current_balance).frac < in_purse_fee.frac
         THEN 1
         ELSE 0
         END
      WHERE reserve_pub=in_reserve_pub
        AND ( ((current_balance).val > in_purse_fee.val) OR
              ( ((current_balance).frac >= in_purse_fee.frac) AND
                ((current_balance).val >= in_purse_fee.val) ) );
    IF NOT FOUND
    THEN
      out_no_funds=TRUE;
      RETURN;
    END IF;
  END IF;
END IF;
out_no_funds=FALSE;
INSERT INTO account_merges
  (reserve_pub
  ,reserve_sig
  ,purse_pub
  ,wallet_h_payto)
  VALUES
  (in_reserve_pub
  ,in_reserve_sig
  ,in_purse_pub
  ,in_wallet_h_payto);
END $$;
COMMENT ON FUNCTION exchange_do_reserve_purse(BYTEA, BYTEA, INT8, INT8, INT8, BYTEA, BOOLEAN, taler_amount, BYTEA, BYTEA)
  IS 'Create a purse for a reserve.';
CREATE OR REPLACE FUNCTION exchange_do_expire_purse(
  IN in_start_time INT8,
  IN in_end_time INT8,
  IN in_now INT8,
  OUT out_found BOOLEAN)
LANGUAGE plpgsql
AS $$
DECLARE
  my_purse_pub BYTEA;
DECLARE
  my_deposit record;
DECLARE
  my_in_reserve_quota BOOLEAN;
BEGIN
SELECT purse_pub
      ,in_reserve_quota
  INTO my_purse_pub
      ,my_in_reserve_quota
  FROM purse_requests
 WHERE (purse_expiration >= in_start_time) AND
       (purse_expiration < in_end_time) AND
       NOT was_decided
  ORDER BY purse_expiration ASC
 LIMIT 1;
out_found = FOUND;
IF NOT FOUND
THEN
  RETURN;
END IF;
INSERT INTO purse_decision
  (purse_pub
  ,action_timestamp
  ,refunded)
VALUES
  (my_purse_pub
  ,in_now
  ,TRUE);
NOTIFY X8DJSPNYJMNZDAP7GN6YQ4EZVSQXMF3HRP4VAR347WP9SZYP1C200;
IF (my_in_reserve_quota)
THEN
  UPDATE reserves
    SET purses_active=purses_active-1
  WHERE reserve_pub IN
    (SELECT reserve_pub
       FROM exchange.purse_merges
      WHERE purse_pub=my_purse_pub
     LIMIT 1);
END IF;
FOR my_deposit IN
  SELECT coin_pub
        ,amount_with_fee
    FROM purse_deposits
  WHERE purse_pub = my_purse_pub
LOOP
  UPDATE known_coins kc SET
    remaining.frac=(kc.remaining).frac+(my_deposit.amount_with_fee).frac
     - CASE
       WHEN (kc.remaining).frac+(my_deposit.amount_with_fee).frac >= 100000000
       THEN 100000000
       ELSE 0
       END,
    remaining.val=(kc.remaining).val+(my_deposit.amount_with_fee).val
     + CASE
       WHEN (kc.remaining).frac+(my_deposit.amount_with_fee).frac >= 100000000
       THEN 1
       ELSE 0
       END
    WHERE coin_pub = my_deposit.coin_pub;
  END LOOP;
END $$;
COMMENT ON FUNCTION exchange_do_expire_purse(INT8,INT8,INT8)
  IS 'Finds an expired purse in the given time range and refunds the coins (if any).';
CREATE OR REPLACE FUNCTION exchange_do_reserve_open_deposit(
  IN in_coin_pub BYTEA,
  IN in_known_coin_id INT8,
  IN in_coin_sig BYTEA,
  IN in_reserve_sig BYTEA,
  IN in_reserve_pub BYTEA,
  IN in_coin_total taler_amount,
  OUT out_insufficient_funds BOOLEAN)
LANGUAGE plpgsql
AS $$
BEGIN
INSERT INTO exchange.reserves_open_deposits
  (reserve_sig
  ,reserve_pub
  ,coin_pub
  ,coin_sig
  ,contribution
  )
  VALUES
  (in_reserve_sig
  ,in_reserve_pub
  ,in_coin_pub
  ,in_coin_sig
  ,in_coin_total
  )
  ON CONFLICT DO NOTHING;
IF NOT FOUND
THEN
  out_insufficient_funds=FALSE;
  RETURN;
END IF;
UPDATE exchange.known_coins kc
  SET
    remaining.frac=(kc.remaining).frac-in_coin_total.frac
       + CASE
         WHEN (kc.remaining).frac < in_coin_total.frac
         THEN 100000000
         ELSE 0
         END,
    remaining.val=(kc.remaining).val-in_coin_total.val
       - CASE
         WHEN (kc.remaining).frac < in_coin_total.frac
         THEN 1
         ELSE 0
         END
  WHERE coin_pub=in_coin_pub
    AND ( ((kc.remaining).val > in_coin_total.val) OR
          ( ((kc.remaining).frac >= in_coin_total.frac) AND
            ((kc.remaining).val >= in_coin_total.val) ) );
IF NOT FOUND
THEN
  out_insufficient_funds=TRUE;
  RETURN;
END IF;
out_insufficient_funds=FALSE;
END $$;
CREATE OR REPLACE FUNCTION exchange_do_reserve_open(
  IN in_reserve_pub BYTEA,
  IN in_total_paid taler_amount,
  IN in_reserve_payment taler_amount,
  IN in_min_purse_limit INT4,
  IN in_default_purse_limit INT4,
  IN in_reserve_sig BYTEA,
  IN in_desired_expiration INT8,
  IN in_reserve_gc_delay INT8,
  IN in_now INT8,
  IN in_open_fee taler_amount,
  OUT out_open_cost taler_amount,
  OUT out_final_expiration INT8,
  OUT out_no_reserve BOOLEAN,
  OUT out_no_funds BOOLEAN,
  OUT out_reserve_balance taler_amount)
LANGUAGE plpgsql
AS $$
DECLARE
  my_balance taler_amount;
  my_cost taler_amount;
  my_cost_tmp INT8;
  my_years_tmp INT4;
  my_years INT4;
  my_needs_update BOOL;
  my_expiration_date INT8;
  reserve RECORD;
BEGIN
SELECT current_balance
      ,expiration_date
      ,purses_allowed
  INTO reserve
  FROM reserves
 WHERE reserve_pub=in_reserve_pub;
IF NOT FOUND
THEN
  RAISE NOTICE 'reserve not found';
  out_no_reserve = TRUE;
  out_no_funds = TRUE;
  out_reserve_balance.val = 0;
  out_reserve_balance.frac = 0;
  out_open_cost.val = 0;
  out_open_cost.frac = 0;
  out_final_expiration = 0;
  RETURN;
END IF;
out_no_reserve = FALSE;
out_reserve_balance = reserve.current_balance;
IF (reserve.expiration_date < in_now)
THEN
  my_expiration_date = in_now;
ELSE
  my_expiration_date = reserve.expiration_date;
END IF;
my_cost.val = 0;
my_cost.frac = 0;
my_needs_update = FALSE;
my_years = 0;
IF (my_expiration_date < in_desired_expiration)
THEN
  my_years = (31535999999999 + in_desired_expiration - my_expiration_date) / 31536000000000;
  reserve.purses_allowed = in_default_purse_limit;
  my_expiration_date = my_expiration_date + 31536000000000 * my_years;
END IF;
IF (reserve.purses_allowed < in_min_purse_limit)
THEN
  my_years = (31535999999999 + in_desired_expiration - in_now) / 31536000000000;
  my_expiration_date = in_now + 31536000000000 * my_years;
  my_years_tmp = (in_min_purse_limit + in_default_purse_limit - reserve.purses_allowed - 1) / in_default_purse_limit;
  my_years = my_years + my_years_tmp;
  reserve.purses_allowed = reserve.purses_allowed + (in_default_purse_limit * my_years_tmp);
END IF;
IF (my_years > 0)
THEN
  my_cost.val = my_years * in_open_fee.val;
  my_cost_tmp = my_years * in_open_fee.frac / 100000000;
  IF (CAST (my_cost.val + my_cost_tmp AS INT8) < my_cost.val)
  THEN
    out_open_cost.val=9223372036854775807;
    out_open_cost.frac=2147483647;
    out_final_expiration=my_expiration_date;
    out_no_funds=FALSE;
    RAISE NOTICE 'arithmetic issue computing amount';
  RETURN;
  END IF;
  my_cost.val = CAST (my_cost.val + my_cost_tmp AS INT8);
  my_cost.frac = my_years * in_open_fee.frac % 100000000;
  my_needs_update = TRUE;
END IF;
IF NOT my_needs_update
THEN
  out_final_expiration = reserve.expiration_date;
  out_open_cost.val = 0;
  out_open_cost.frac = 0;
  out_no_funds=FALSE;
  RAISE NOTICE 'no change required';
  RETURN;
END IF;
IF ( (in_total_paid.val < my_cost.val) OR
     ( (in_total_paid.val = my_cost.val) AND
       (in_total_paid.frac < my_cost.frac) ) )
THEN
  out_open_cost.val = my_cost.val;
  out_open_cost.frac = my_cost.frac;
  out_no_funds=FALSE;
  IF (reserve.expiration_date >= in_desired_expiration)
  THEN
    RAISE NOTICE 'forcing low expiration time';
    out_final_expiration = 0;
  ELSE
    out_final_expiration = reserve.expiration_date;
  END IF;
  RAISE NOTICE 'amount paid too low';
  RETURN;
END IF;
IF (out_reserve_balance.val > in_reserve_payment.val)
THEN
  IF (out_reserve_balance.frac >= in_reserve_payment.frac)
  THEN
    my_balance.val=out_reserve_balance.val - in_reserve_payment.val;
    my_balance.frac=out_reserve_balance.frac - in_reserve_payment.frac;
  ELSE
    my_balance.val=out_reserve_balance.val - in_reserve_payment.val - 1;
    my_balance.frac=out_reserve_balance.frac + 100000000 - in_reserve_payment.frac;
  END IF;
ELSE
  IF (out_reserve_balance.val = in_reserve_payment.val) AND (out_reserve_balance.frac >= in_reserve_payment.frac)
  THEN
    my_balance.val=0;
    my_balance.frac=out_reserve_balance.frac - in_reserve_payment.frac;
  ELSE
    out_final_expiration = reserve.expiration_date;
    out_open_cost.val = my_cost.val;
    out_open_cost.frac = my_cost.frac;
    out_no_funds=TRUE;
    RAISE NOTICE 'reserve balance too low';
  RETURN;
  END IF;
END IF;
UPDATE reserves SET
  current_balance=my_balance
 ,gc_date=reserve.expiration_date + in_reserve_gc_delay
 ,expiration_date=my_expiration_date
 ,purses_allowed=reserve.purses_allowed
WHERE
 reserve_pub=in_reserve_pub;
out_final_expiration=my_expiration_date;
out_open_cost = my_cost;
out_no_funds=FALSE;
RETURN;
END $$;
CREATE OR REPLACE FUNCTION exchange_do_insert_or_update_policy_details(
  IN in_policy_hash_code BYTEA,
  IN in_policy_json TEXT,
  IN in_deadline INT8,
  IN in_commitment taler_amount,
  IN in_accumulated_total taler_amount,
  IN in_fee taler_amount,
  IN in_transferable taler_amount,
  IN in_fulfillment_state SMALLINT,
  OUT out_policy_details_serial_id INT8,
  OUT out_accumulated_total taler_amount,
  OUT out_fulfillment_state SMALLINT)
LANGUAGE plpgsql
AS $$
DECLARE
    cur_commitment taler_amount;
DECLARE
    cur_accumulated_total taler_amount;
DECLARE
    rval RECORD;
BEGIN
       INSERT INTO policy_details
               (policy_hash_code,
                policy_json,
                deadline,
                commitment,
                accumulated_total,
                fee,
                transferable,
                fulfillment_state)
       VALUES (in_policy_hash_code,
                in_policy_json,
                in_deadline,
                in_commitment,
                in_accumulated_total,
                in_fee,
                in_transferable,
                in_fulfillment_state)
       ON CONFLICT (policy_hash_code) DO NOTHING
       RETURNING policy_details_serial_id INTO out_policy_details_serial_id;
       IF FOUND THEN
               out_accumulated_total = in_accumulated_total;
               out_fulfillment_state = in_fulfillment_state;
               RETURN;
       END IF;
       SELECT policy_details_serial_id
         ,commitment
         ,accumulated_total
       INTO rval
       FROM policy_details
       WHERE policy_hash_code = in_policy_hash_code;
       out_policy_details_serial_id := rval.policy_details_serial_id;
       cur_commitment := rval.commitment;
       cur_accumulated_total := rval.accumulated_total;
       out_accumulated_total.val = cur_accumulated_total.val + in_accumulated_total.val;
       out_accumulated_total.frac = cur_accumulated_total.frac + in_accumulated_total.frac;
       out_accumulated_total.val = out_accumulated_total.val + out_accumulated_total.frac / 100000000;
       out_accumulated_total.frac = out_accumulated_total.frac % 100000000;
       IF (out_accumulated_total.val > (1 << 52))
       THEN
               RAISE EXCEPTION 'accumulation overflow';
       END IF;
       IF (out_fullfillment_state = 2) 
       THEN
               IF (out_accumulated_total.val >= cur_commitment.val OR
                       (out_accumulated_total.val = cur_commitment.val AND
                               out_accumulated_total.frac >= cur_commitment.frac))
               THEN
                       out_fulfillment_state = 3; 
               END IF;
       END IF;
       UPDATE exchange.policy_details
       SET
               accumulated = out_accumulated_total,
               fulfillment_state = out_fulfillment_state
       WHERE
               policy_details_serial_id = out_policy_details_serial_id;
END $$;
CREATE OR REPLACE FUNCTION exchange_do_insert_aml_decision(
  IN in_h_payto BYTEA,
  IN in_new_threshold taler_amount,
  IN in_new_status INT4,
  IN in_decision_time INT8,
  IN in_justification TEXT,
  IN in_decider_pub BYTEA,
  IN in_decider_sig BYTEA,
  IN in_notify_s TEXT,
  IN in_kyc_requirements TEXT,
  IN in_requirement_row INT8,
  OUT out_invalid_officer BOOLEAN,
  OUT out_last_date INT8)
LANGUAGE plpgsql
AS $$
BEGIN
PERFORM
  FROM aml_staff
  WHERE decider_pub=in_decider_pub
    AND is_active
    AND NOT read_only;
IF NOT FOUND
THEN
  out_invalid_officer=TRUE;
  out_last_date=0;
  RETURN;
END IF;
out_invalid_officer=FALSE;
SELECT decision_time
  INTO out_last_date
  FROM aml_history
  WHERE h_payto=in_h_payto
  ORDER BY decision_time DESC;
IF FOUND
THEN
  IF out_last_date >= in_decision_time
  THEN
    RETURN;
  END IF;
  UPDATE aml_status
    SET threshold=in_new_threshold
       ,status=in_new_status
       ,kyc_requirement=in_requirement_row
   WHERE h_payto=in_h_payto;
  ASSERT FOUND, 'cannot have AML decision history but no AML status';
ELSE
  out_last_date = 0;
  INSERT INTO aml_status
    (h_payto
    ,threshold
    ,status
    ,kyc_requirement)
    VALUES
    (in_h_payto
    ,in_new_threshold
    ,in_new_status
    ,in_requirement_row)
    ON CONFLICT (h_payto) DO
    UPDATE SET
       threshold=in_new_threshold
      ,status=in_new_status;
END IF;
INSERT INTO aml_history
  (h_payto
  ,new_threshold
  ,new_status
  ,decision_time
  ,justification
  ,kyc_requirements
  ,kyc_req_row
  ,decider_pub
  ,decider_sig
  ) VALUES
  (in_h_payto
  ,in_new_threshold
  ,in_new_status
  ,in_decision_time
  ,in_justification
  ,in_kyc_requirements
  ,in_requirement_row
  ,in_decider_pub
  ,in_decider_sig);
IF 0 = in_new_status
THEN
  INSERT INTO kyc_alerts
    (h_payto
    ,trigger_type)
    VALUES
    (in_h_payto,1);
   EXECUTE FORMAT (
     'NOTIFY %s'
    ,in_notify_s);
END IF;
END $$;
COMMENT ON FUNCTION exchange_do_insert_aml_decision(BYTEA, taler_amount, INT4, INT8, TEXT, BYTEA, BYTEA, TEXT, TEXT, INT8)
  IS 'Checks whether the AML officer is eligible to make AML decisions and if so inserts the decision into the table';
CREATE OR REPLACE FUNCTION exchange_do_insert_aml_officer(
  IN in_decider_pub BYTEA,
  IN in_master_sig BYTEA,
  IN in_decider_name TEXT,
  IN in_is_active BOOLEAN,
  IN in_read_only BOOLEAN,
  IN in_last_change INT8,
  OUT out_last_change INT8)
LANGUAGE plpgsql
AS $$
BEGIN
INSERT INTO exchange.aml_staff
  (decider_pub
  ,master_sig
  ,decider_name
  ,is_active
  ,read_only
  ,last_change
  ) VALUES
  (in_decider_pub
  ,in_master_sig
  ,in_decider_name
  ,in_is_active
  ,in_read_only
  ,in_last_change)
 ON CONFLICT DO NOTHING;
IF FOUND
THEN
  out_last_change=0;
  RETURN;
END IF;
SELECT last_change
  INTO out_last_change
  FROM exchange.aml_staff
  WHERE decider_pub=in_decider_pub;
ASSERT FOUND, 'cannot have INSERT conflict but no AML staff record';
IF out_last_change >= in_last_change
THEN
 RETURN;
END IF;
UPDATE exchange.aml_staff
  SET master_sig=in_master_sig
     ,decider_name=in_decider_name
     ,is_active=in_is_active
     ,read_only=in_read_only
     ,last_change=in_last_change
  WHERE decider_pub=in_decider_pub;
END $$;
COMMENT ON FUNCTION exchange_do_insert_aml_officer(BYTEA, BYTEA, TEXT, BOOL, BOOL, INT8)
  IS 'Inserts or updates AML staff record, making sure the update is more recent than the previous change';
CREATE OR REPLACE FUNCTION exchange_do_insert_kyc_attributes(
  IN in_process_row INT8,
  IN in_h_payto BYTEA,
  IN in_kyc_prox BYTEA,
  IN in_provider_section TEXT,
  IN in_satisfied_checks TEXT[],
  IN in_birthday INT4,
  IN in_provider_account_id TEXT,
  IN in_provider_legitimization_id TEXT,
  IN in_collection_time_ts INT8,
  IN in_expiration_time INT8,
  IN in_expiration_time_ts INT8,
  IN in_enc_attributes BYTEA,
  IN in_require_aml BOOLEAN,
  IN in_kyc_completed_notify_s TEXT,
  OUT out_ok BOOLEAN)
LANGUAGE plpgsql
AS $$
DECLARE
   orig_reserve_pub BYTEA;
   orig_reserve_found BOOLEAN;
BEGIN
INSERT INTO exchange.kyc_attributes
  (h_payto
  ,kyc_prox
  ,provider
  ,satisfied_checks
  ,collection_time
  ,expiration_time
  ,encrypted_attributes
  ,legitimization_serial
 ) VALUES
  (in_h_payto
  ,in_kyc_prox
  ,in_provider_section
  ,in_satisfied_checks
  ,in_collection_time_ts
  ,in_expiration_time_ts
  ,in_enc_attributes
  ,in_process_row);
UPDATE legitimization_processes
  SET provider_user_id=in_provider_account_id
     ,provider_legitimization_id=in_provider_legitimization_id
     ,expiration_time=GREATEST(expiration_time,in_expiration_time)
     ,finished=TRUE
 WHERE h_payto=in_h_payto
   AND legitimization_process_serial_id=in_process_row
   AND provider_section=in_provider_section;
out_ok = FOUND;
SELECT reserve_pub
  INTO orig_reserve_pub
  FROM exchange.legitimization_requirements
 WHERE h_payto=in_h_payto
   AND NOT reserve_pub IS NULL;
orig_reserve_found = FOUND;
IF orig_reserve_found
THEN
  UPDATE exchange.reserves
     SET birthday=in_birthday
   WHERE reserve_pub=orig_reserve_pub;
END IF;
IF in_require_aml
THEN
  INSERT INTO exchange.aml_status
    (h_payto
    ,status)
   VALUES
    (in_h_payto
    ,1)
  ON CONFLICT (h_payto) DO
    UPDATE SET status=EXCLUDED.status | 1;
END IF;
EXECUTE FORMAT (
 'NOTIFY %s'
 ,in_kyc_completed_notify_s);
INSERT INTO kyc_alerts
 (h_payto
 ,trigger_type)
 VALUES
 (in_h_payto,1);
END $$;
COMMENT ON FUNCTION exchange_do_insert_kyc_attributes(INT8, BYTEA, BYTEA, TEXT, TEXT[], INT4, TEXT, TEXT, INT8, INT8, INT8, BYTEA, BOOL, TEXT)
  IS 'Inserts new KYC attributes and updates the status of the legitimization process and the AML status for the account';
CREATE OR REPLACE FUNCTION exchange_do_array_reserves_insert(
  IN in_gc_date INT8,
  IN in_reserve_expiration INT8,
  IN ina_reserve_pub BYTEA[],
  IN ina_wire_ref INT8[],
  IN ina_credit taler_amount[],
  IN ina_exchange_account_name TEXT[],
  IN ina_execution_date INT8[],
  IN ina_wire_source_h_payto BYTEA[],
  IN ina_payto_uri TEXT[],
  IN ina_notify TEXT[])
RETURNS SETOF exchange_do_array_reserve_insert_return_type
LANGUAGE plpgsql
AS $$
DECLARE
  conflict BOOL;
  dup BOOL;
  uuid INT8;
  i INT4;
  ini_reserve_pub BYTEA;
  ini_wire_ref INT8;
  ini_credit taler_amount;
  ini_exchange_account_name TEXT;
  ini_execution_date INT8;
  ini_wire_source_h_payto BYTEA;
  ini_payto_uri TEXT;
  ini_notify TEXT;
BEGIN
  FOR i IN 1..array_length(ina_reserve_pub,1)
  LOOP
    ini_reserve_pub = ina_reserve_pub[i];
    ini_wire_ref = ina_wire_ref[i];
    ini_credit = ina_credit[i];
    ini_exchange_account_name = ina_exchange_account_name[i];
    ini_execution_date = ina_execution_date[i];
    ini_wire_source_h_payto = ina_wire_source_h_payto[i];
    ini_payto_uri = ina_payto_uri[i];
    ini_notify = ina_notify[i];
    INSERT INTO wire_targets
      (wire_target_h_payto
      ,payto_uri
      ) VALUES (
        ini_wire_source_h_payto
       ,ini_payto_uri
      )
    ON CONFLICT DO NOTHING;
    INSERT INTO reserves
      (reserve_pub
      ,current_balance
      ,expiration_date
      ,gc_date
    ) VALUES (
      ini_reserve_pub
     ,ini_credit
     ,in_reserve_expiration
     ,in_gc_date
    )
    ON CONFLICT DO NOTHING
    RETURNING reserve_uuid
      INTO uuid;
    conflict = NOT FOUND;
    INSERT INTO reserves_in
      (reserve_pub
      ,wire_reference
      ,credit
      ,exchange_account_section
      ,wire_source_h_payto
      ,execution_date
    ) VALUES (
      ini_reserve_pub
     ,ini_wire_ref
     ,ini_credit
     ,ini_exchange_account_name
     ,ini_wire_source_h_payto
     ,ini_execution_date
    )
    ON CONFLICT DO NOTHING;
    IF NOT FOUND
    THEN
      IF conflict
      THEN
        dup = TRUE;
      else
        dup = FALSE;
      END IF;
    ELSE
      IF NOT conflict
      THEN
        EXECUTE FORMAT (
          'NOTIFY %s'
          ,ini_notify);
      END IF;
      dup = FALSE;
    END IF;
    RETURN NEXT (dup,uuid);
  END LOOP;
  RETURN;
END $$;
CREATE OR REPLACE FUNCTION exchange_do_batch_reserves_update(
  IN in_reserve_pub BYTEA,
  IN in_expiration_date INT8,
  IN in_wire_ref INT8,
  IN in_credit taler_amount,
  IN in_exchange_account_name TEXT,
  IN in_wire_source_h_payto BYTEA,
  IN in_notify text,
  OUT out_duplicate BOOLEAN)
LANGUAGE plpgsql
AS $$
BEGIN
  INSERT INTO reserves_in
    (reserve_pub
    ,wire_reference
    ,credit
    ,exchange_account_section
    ,wire_source_h_payto
    ,execution_date)
    VALUES
    (in_reserve_pub
    ,in_wire_ref
    ,in_credit
    ,in_exchange_account_name
    ,in_wire_source_h_payto
    ,in_expiration_date)
    ON CONFLICT DO NOTHING;
  IF FOUND
  THEN
    out_duplicate = FALSE;
    UPDATE reserves rs
      SET
         current_balance.frac = (rs.current_balance).frac+in_credit.frac
           - CASE
             WHEN (rs.current_balance).frac + in_credit.frac >= 100000000
               THEN 100000000
             ELSE 1
             END
        ,current_balance.val = (rs.current_balance).val+in_credit.val
           + CASE
             WHEN (rs.current_balance).frac + in_credit.frac >= 100000000
               THEN 1
             ELSE 0
             END
             ,expiration_date=GREATEST(expiration_date,in_expiration_date)
             ,gc_date=GREATEST(gc_date,in_expiration_date)
            WHERE reserve_pub=in_reserve_pub;
    EXECUTE FORMAT (
      'NOTIFY %s'
      ,in_notify);
  ELSE
    out_duplicate = TRUE;
  END IF;
  RETURN;
END $$;
CREATE OR REPLACE FUNCTION exchange_do_get_link_data(
  IN in_coin_pub BYTEA
)
RETURNS SETOF record
LANGUAGE plpgsql
AS $$
DECLARE
  curs CURSOR
  FOR
  SELECT
   melt_serial_id
  FROM refresh_commitments
  WHERE old_coin_pub=in_coin_pub;
DECLARE
  i RECORD;
BEGIN
OPEN curs;
LOOP
    FETCH NEXT FROM curs INTO i;
    EXIT WHEN NOT FOUND;
    RETURN QUERY
      SELECT
       tp.transfer_pub
      ,denoms.denom_pub
      ,rrc.ev_sig
      ,rrc.ewv
      ,rrc.link_sig
      ,rrc.freshcoin_index
      ,rrc.coin_ev
      FROM refresh_revealed_coins rrc
       JOIN refresh_transfer_keys tp
         ON (tp.melt_serial_id=rrc.melt_serial_id)
       JOIN denominations denoms
         ON (rrc.denominations_serial=denoms.denominations_serial)
       WHERE rrc.melt_serial_id =i.melt_serial_id
       ORDER BY tp.transfer_pub,
       rrc.freshcoin_index ASC
       ;
END LOOP;
CLOSE curs;
END $$;
CREATE OR REPLACE FUNCTION exchange_do_batch4_known_coin(
  IN in_coin_pub1 BYTEA,
  IN in_denom_pub_hash1 BYTEA,
  IN in_h_age_commitment1 BYTEA,
  IN in_denom_sig1 BYTEA,
  IN in_coin_pub2 BYTEA,
  IN in_denom_pub_hash2 BYTEA,
  IN in_h_age_commitment2 BYTEA,
  IN in_denom_sig2 BYTEA,
  IN in_coin_pub3 BYTEA,
  IN in_denom_pub_hash3 BYTEA,
  IN in_h_age_commitment3 BYTEA,
  IN in_denom_sig3 BYTEA,
  IN in_coin_pub4 BYTEA,
  IN in_denom_pub_hash4 BYTEA,
  IN in_h_age_commitment4 BYTEA,
  IN in_denom_sig4 BYTEA,
  OUT existed1 BOOLEAN,
  OUT existed2 BOOLEAN,
  OUT existed3 BOOLEAN,
  OUT existed4 BOOLEAN,
  OUT known_coin_id1 INT8,
  OUT known_coin_id2 INT8,
  OUT known_coin_id3 INT8,
  OUT known_coin_id4 INT8,
  OUT denom_pub_hash1 BYTEA,
  OUT denom_pub_hash2 BYTEA,
  OUT denom_pub_hash3 BYTEA,
  OUT denom_pub_hash4 BYTEA,
  OUT age_commitment_hash1 BYTEA,
  OUT age_commitment_hash2 BYTEA,
  OUT age_commitment_hash3 BYTEA,
  OUT age_commitment_hash4 BYTEA)
LANGUAGE plpgsql
AS $$
BEGIN
WITH dd AS (
SELECT
  denominations_serial,
  coin
  FROM denominations
    WHERE denom_pub_hash
    IN
     (in_denom_pub_hash1,
      in_denom_pub_hash2,
      in_denom_pub_hash3,
      in_denom_pub_hash4)
     ),
     input_rows AS (
     VALUES
      (in_coin_pub1,
      in_denom_pub_hash1,
      in_h_age_commitment1,
      in_denom_sig1),
      (in_coin_pub2,
      in_denom_pub_hash2,
      in_h_age_commitment2,
      in_denom_sig2),
      (in_coin_pub3,
      in_denom_pub_hash3,
      in_h_age_commitment3,
      in_denom_sig3),
      (in_coin_pub4,
      in_denom_pub_hash4,
      in_h_age_commitment4,
      in_denom_sig4)
      ),
      ins AS (
      INSERT INTO known_coins (
      coin_pub,
      denominations_serial,
      age_commitment_hash,
      denom_sig,
      remaining
      )
      SELECT
        ir.coin_pub,
        dd.denominations_serial,
        ir.age_commitment_hash,
        ir.denom_sig,
        dd.coin
        FROM input_rows ir
        JOIN dd
          ON dd.denom_pub_hash = ir.denom_pub_hash
          ON CONFLICT DO NOTHING
          RETURNING known_coin_id
      ),
       exists AS (
         SELECT
         CASE
           WHEN
             ins.known_coin_id IS NOT NULL
             THEN
               FALSE
             ELSE
               TRUE
         END AS existed,
         ins.known_coin_id,
         dd.denom_pub_hash,
         kc.age_commitment_hash
         FROM input_rows ir
         LEFT JOIN ins
           ON ins.coin_pub = ir.coin_pub
         LEFT JOIN known_coins kc
           ON kc.coin_pub = ir.coin_pub
         LEFT JOIN dd
           ON dd.denom_pub_hash = ir.denom_pub_hash
         )
SELECT
 exists.existed AS existed1,
 exists.known_coin_id AS known_coin_id1,
 exists.denom_pub_hash AS denom_pub_hash1,
 exists.age_commitment_hash AS age_commitment_hash1,
 (
   SELECT exists.existed
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash2
 ) AS existed2,
 (
   SELECT exists.known_coin_id
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash2
 ) AS known_coin_id2,
 (
   SELECT exists.denom_pub_hash
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash2
 ) AS denom_pub_hash2,
 (
   SELECT exists.age_commitment_hash
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash2
 )AS age_commitment_hash2,
 (
   SELECT exists.existed
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash3
 ) AS existed3,
 (
   SELECT exists.known_coin_id
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash3
 ) AS known_coin_id3,
 (
   SELECT exists.denom_pub_hash
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash3
 ) AS denom_pub_hash3,
 (
   SELECT exists.age_commitment_hash
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash3
 )AS age_commitment_hash3,
 (
   SELECT exists.existed
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash4
 ) AS existed4,
 (
   SELECT exists.known_coin_id
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash4
 ) AS known_coin_id4,
 (
   SELECT exists.denom_pub_hash
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash4
 ) AS denom_pub_hash4,
 (
   SELECT exists.age_commitment_hash
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash4
 )AS age_commitment_hash4
FROM exists;
RETURN;
END $$;
CREATE OR REPLACE FUNCTION exchange_do_batch2_known_coin(
  IN in_coin_pub1 BYTEA,
  IN in_denom_pub_hash1 BYTEA,
  IN in_h_age_commitment1 BYTEA,
  IN in_denom_sig1 BYTEA,
  IN in_coin_pub2 BYTEA,
  IN in_denom_pub_hash2 BYTEA,
  IN in_h_age_commitment2 BYTEA,
  IN in_denom_sig2 BYTEA,
  OUT existed1 BOOLEAN,
  OUT existed2 BOOLEAN,
  OUT known_coin_id1 INT8,
  OUT known_coin_id2 INT8,
  OUT denom_pub_hash1 BYTEA,
  OUT denom_pub_hash2 BYTEA,
  OUT age_commitment_hash1 BYTEA,
  OUT age_commitment_hash2 BYTEA)
LANGUAGE plpgsql
AS $$
BEGIN
WITH dd AS (
SELECT
  denominations_serial,
  coin
  FROM denominations
    WHERE denom_pub_hash
    IN
     (in_denom_pub_hash1,
      in_denom_pub_hash2)
     ),
     input_rows AS (
     VALUES
      (in_coin_pub1,
      in_denom_pub_hash1,
      in_h_age_commitment1,
      in_denom_sig1),
      (in_coin_pub2,
      in_denom_pub_hash2,
      in_h_age_commitment2,
      in_denom_sig2)
      ),
      ins AS (
      INSERT INTO known_coins (
      coin_pub,
      denominations_serial,
      age_commitment_hash,
      denom_sig,
      remaining
      )
      SELECT
        ir.coin_pub,
        dd.denominations_serial,
        ir.age_commitment_hash,
        ir.denom_sig,
        dd.coin
        FROM input_rows ir
        JOIN dd
          ON dd.denom_pub_hash = ir.denom_pub_hash
          ON CONFLICT DO NOTHING
          RETURNING known_coin_id
      ),
       exists AS (
       SELECT
        CASE
          WHEN ins.known_coin_id IS NOT NULL
          THEN
            FALSE
          ELSE
            TRUE
        END AS existed,
        ins.known_coin_id,
        dd.denom_pub_hash,
        kc.age_commitment_hash
        FROM input_rows ir
        LEFT JOIN ins
          ON ins.coin_pub = ir.coin_pub
        LEFT JOIN known_coins kc
          ON kc.coin_pub = ir.coin_pub
        LEFT JOIN dd
          ON dd.denom_pub_hash = ir.denom_pub_hash
     )
SELECT
 exists.existed AS existed1,
 exists.known_coin_id AS known_coin_id1,
 exists.denom_pub_hash AS denom_pub_hash1,
 exists.age_commitment_hash AS age_commitment_hash1,
 (
   SELECT exists.existed
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash2
 ) AS existed2,
 (
   SELECT exists.known_coin_id
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash2
 ) AS known_coin_id2,
 (
   SELECT exists.denom_pub_hash
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash2
 ) AS denom_pub_hash2,
 (
   SELECT exists.age_commitment_hash
   FROM exists
   WHERE exists.denom_pub_hash = in_denom_pub_hash2
 )AS age_commitment_hash2
FROM exists;
RETURN;
END $$;
CREATE OR REPLACE FUNCTION exchange_do_batch1_known_coin(
  IN in_coin_pub1 BYTEA,
  IN in_denom_pub_hash1 BYTEA,
  IN in_h_age_commitment1 BYTEA,
  IN in_denom_sig1 BYTEA,
  OUT existed1 BOOLEAN,
  OUT known_coin_id1 INT8,
  OUT denom_pub_hash1 BYTEA,
  OUT age_commitment_hash1 BYTEA)
LANGUAGE plpgsql
AS $$
BEGIN
WITH dd AS (
SELECT
  denominations_serial,
  coin
  FROM denominations
    WHERE denom_pub_hash
    IN
     (in_denom_pub_hash1,
      in_denom_pub_hash2)
     ),
     input_rows AS (
     VALUES
      (in_coin_pub1,
      in_denom_pub_hash1,
      in_h_age_commitment1,
      in_denom_sig1)
      ),
      ins AS (
      INSERT INTO known_coins (
      coin_pub,
      denominations_serial,
      age_commitment_hash,
      denom_sig,
      remaining
      )
      SELECT
        ir.coin_pub,
        dd.denominations_serial,
        ir.age_commitment_hash,
        ir.denom_sig,
        dd.coin
        FROM input_rows ir
        JOIN dd
          ON dd.denom_pub_hash = ir.denom_pub_hash
          ON CONFLICT DO NOTHING
          RETURNING known_coin_id
      ),
       exists AS (
       SELECT
        CASE
          WHEN ins.known_coin_id IS NOT NULL
          THEN
            FALSE
          ELSE
            TRUE
        END AS existed,
        ins.known_coin_id,
        dd.denom_pub_hash,
        kc.age_commitment_hash
        FROM input_rows ir
        LEFT JOIN ins
          ON ins.coin_pub = ir.coin_pub
        LEFT JOIN known_coins kc
          ON kc.coin_pub = ir.coin_pub
        LEFT JOIN dd
          ON dd.denom_pub_hash = ir.denom_pub_hash
       )
SELECT
 exists.existed AS existed1,
 exists.known_coin_id AS known_coin_id1,
 exists.denom_pub_hash AS denom_pub_hash1,
 exists.age_commitment_hash AS age_commitment_hash1
FROM exists;
RETURN;
END $$;
BEGIN;
CREATE OR REPLACE FUNCTION auditor_wake_coins_helper_trigger()
    RETURNS trigger
    LANGUAGE plpgsql
AS $$
BEGIN
    NOTIFY X5V5R0DDFMXS0R3W058R3W4RPDVMK35YZCS0S5VZS583J0NR0PE2G;
RETURN NEW;
END $$;
COMMENT ON FUNCTION auditor_wake_coins_helper_trigger()
    IS 'Call auditor_call_db_notify on new entry';
CREATE OR REPLACE FUNCTION auditor_wake_purses_helper_trigger()
    RETURNS trigger
    LANGUAGE plpgsql
AS $$
BEGIN
    NOTIFY X908G8PNPMJYA59YGGTJND1TKTBFNG8C7TREHG3X5SJ9EQAJY4Z00;
RETURN NEW;
END $$;
COMMENT ON FUNCTION auditor_wake_purses_helper_trigger()
    IS 'Call auditor_call_db_notify on new entry';
CREATE OR REPLACE FUNCTION auditor_wake_deposits_helper_trigger()
    RETURNS trigger
    LANGUAGE plpgsql
AS $$
BEGIN
    NOTIFY XZD0FASMJD3XCY3Z0CGXNJQ8CMWSCW80JN6796098N71CXPH70TQ0;
RETURN NEW;
END $$;
COMMENT ON FUNCTION auditor_wake_deposits_helper_trigger()
    IS 'Call auditor_call_db_notify on new entry';
CREATE OR REPLACE FUNCTION auditor_wake_reserves_helper_trigger()
    RETURNS trigger
    LANGUAGE plpgsql
AS $$
BEGIN
    NOTIFY XMF69RJQB7EN06KGSQ02VFD3723CE86VXA5GRE8H7XNNS6BDYF0G0;
RETURN NEW;
END $$;
COMMENT ON FUNCTION auditor_wake_reserves_helper_trigger()
    IS 'Call auditor_call_db_notify on new entry';
CREATE OR REPLACE FUNCTION auditor_wake_wire_helper_trigger()
    RETURNS trigger
    LANGUAGE plpgsql
AS $$
BEGIN
    NOTIFY X1RYYSTS139MBHVEXJ6CZZTY76MAMEEF87SRRWC8WM00HCCW6D12G;
RETURN NEW;
END $$;
COMMENT ON FUNCTION auditor_wake_wire_helper_trigger()
    IS 'Call auditor_call_db_notify on new entry';
CREATE OR REPLACE FUNCTION auditor_wake_aggregation_helper_trigger()
    RETURNS trigger
    LANGUAGE plpgsql
AS $$
BEGIN
    NOTIFY XWRPZ889FPA6TMGJ15JVTCMKEFVJEWCEKF1TEZHTDQHBYSV49M31G;
RETURN NEW;
END $$;
COMMENT ON FUNCTION auditor_wake_aggregation_helper_trigger()
    IS 'Call auditor_call_db_notify on new entry';
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation0
    AFTER INSERT ON exchange.batch_deposits
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation1
    AFTER INSERT ON exchange.partners
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation2
    AFTER INSERT ON exchange.wire_targets
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation3
    AFTER INSERT ON exchange.reserves_open_deposits
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation4
    AFTER INSERT ON exchange.aggregation_tracking
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation5
    AFTER INSERT ON exchange.purse_requests
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation6
    AFTER INSERT ON exchange.refresh_revealed_coins
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation7
    AFTER INSERT ON exchange.reserves
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation8
    AFTER INSERT ON exchange.purse_deposits
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation9
    AFTER INSERT ON exchange.reserves_out
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation10
    AFTER INSERT ON exchange.recoup
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation11
    AFTER INSERT ON exchange.coin_history
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation12
    AFTER INSERT ON exchange.coin_deposits
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation13
    AFTER INSERT ON exchange.wire_out
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation14
    AFTER INSERT ON exchange.refunds
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation15
    AFTER INSERT ON exchange.refresh_commitments
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation16
    AFTER INSERT ON exchange.purse_decision
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation17
    AFTER INSERT ON exchange.known_coins
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_aggregation18
    AFTER INSERT ON exchange.recoup_refresh
EXECUTE FUNCTION auditor_wake_aggregation_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins0
    AFTER INSERT ON exchange.purse_merges
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins1
    AFTER INSERT ON exchange.batch_deposits
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins2
    AFTER INSERT ON exchange.partners
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins3
    AFTER INSERT ON exchange.denomination_revocations
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins4
    AFTER INSERT ON exchange.wire_targets
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins5
    AFTER INSERT ON exchange.auditors
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins6
    AFTER INSERT ON exchange.purse_requests
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins7
    AFTER INSERT ON exchange.refresh_revealed_coins
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins8
    AFTER INSERT ON exchange.reserves
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins9
    AFTER INSERT ON exchange.purse_deposits
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins10
    AFTER INSERT ON exchange.reserves_out
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins11
    AFTER INSERT ON exchange.recoup
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins12
    AFTER INSERT ON exchange.auditor_denom_sigs
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins13
    AFTER INSERT ON exchange.coin_deposits
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins14
    AFTER INSERT ON exchange.refunds
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins15
    AFTER INSERT ON exchange.refresh_commitments
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins16
    AFTER INSERT ON exchange.purse_decision
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins17
    AFTER INSERT ON exchange.known_coins
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_coins18
    AFTER INSERT ON exchange.recoup_refresh
EXECUTE FUNCTION auditor_wake_coins_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_purses0
    AFTER INSERT ON exchange.purse_merges
EXECUTE FUNCTION auditor_wake_purses_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_purses1
    AFTER INSERT ON exchange.account_merges
EXECUTE FUNCTION auditor_wake_purses_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_purses2
    AFTER INSERT ON exchange.purse_deposits
EXECUTE FUNCTION auditor_wake_purses_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_purses3
    AFTER INSERT ON exchange.global_fee
EXECUTE FUNCTION auditor_wake_purses_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_purses4
    AFTER INSERT ON exchange.purse_requests
EXECUTE FUNCTION auditor_wake_purses_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_purses5
    AFTER INSERT ON exchange.partners
EXECUTE FUNCTION auditor_wake_purses_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_purses6
    AFTER INSERT ON exchange.purse_decision
EXECUTE FUNCTION auditor_wake_purses_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_purses7
    AFTER INSERT ON exchange.known_coins
EXECUTE FUNCTION auditor_wake_purses_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_deposits0
    AFTER INSERT ON exchange.wire_targets
EXECUTE FUNCTION auditor_wake_deposits_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_deposits1
    AFTER INSERT ON exchange.batch_deposits
EXECUTE FUNCTION auditor_wake_deposits_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_deposits2
    AFTER INSERT ON exchange.known_coins
EXECUTE FUNCTION auditor_wake_deposits_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_deposits3
    AFTER INSERT ON exchange.coin_deposits
EXECUTE FUNCTION auditor_wake_deposits_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_reserves0
    AFTER INSERT ON exchange.wire_fee
EXECUTE FUNCTION auditor_wake_reserves_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_reserves1
    AFTER INSERT ON exchange.reserves
EXECUTE FUNCTION auditor_wake_reserves_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_reserves2
    AFTER INSERT ON exchange.reserves_close
EXECUTE FUNCTION auditor_wake_reserves_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_reserves3
    AFTER INSERT ON exchange.purse_merges
EXECUTE FUNCTION auditor_wake_reserves_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_reserves4
    AFTER INSERT ON exchange.wire_targets
EXECUTE FUNCTION auditor_wake_reserves_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_reserves5
    AFTER INSERT ON exchange.reserves_out
EXECUTE FUNCTION auditor_wake_reserves_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_reserves6
    AFTER INSERT ON exchange.recoup
EXECUTE FUNCTION auditor_wake_reserves_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_reserves7
    AFTER INSERT ON exchange.purse_requests
EXECUTE FUNCTION auditor_wake_reserves_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_reserves8
    AFTER INSERT ON exchange.reserves_open_requests
EXECUTE FUNCTION auditor_wake_reserves_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_reserves9
    AFTER INSERT ON exchange.denomination_revocations
EXECUTE FUNCTION auditor_wake_reserves_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_reserves10
    AFTER INSERT ON exchange.purse_decision
EXECUTE FUNCTION auditor_wake_reserves_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_reserves11
    AFTER INSERT ON exchange.known_coins
EXECUTE FUNCTION auditor_wake_reserves_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_reserves12
    AFTER INSERT ON exchange.reserves_in
EXECUTE FUNCTION auditor_wake_reserves_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_wire0
    AFTER INSERT ON exchange.reserves
EXECUTE FUNCTION auditor_wake_wire_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_wire1
    AFTER INSERT ON exchange.wire_targets
EXECUTE FUNCTION auditor_wake_wire_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_wire2
    AFTER INSERT ON exchange.aggregation_tracking
EXECUTE FUNCTION auditor_wake_wire_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_wire3
    AFTER INSERT ON exchange.wire_out
EXECUTE FUNCTION auditor_wake_wire_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_wire4
    AFTER INSERT ON exchange.reserves_close
EXECUTE FUNCTION auditor_wake_wire_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_wire5
    AFTER INSERT ON exchange.profit_drains
EXECUTE FUNCTION auditor_wake_wire_helper_trigger();
CREATE OR REPLACE TRIGGER auditor_exchange_notify_helper_wire6
    AFTER INSERT ON exchange.reserves_in
EXECUTE FUNCTION auditor_wake_wire_helper_trigger();
COMMIT;
COMMIT;
