#  This file is part of TALER
#  (C) 2014, 2015, 2016 INRIA
#
#  TALER is free software; you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation; either version 3, or
# (at your option) any later version.
#
#  TALER is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public
# License along with TALER; see the file COPYING.  If not, see
# <http://www.gnu.org/licenses/>
#
#  @author Marcello Stanisci


"""
definitions of JSON schemas for validating data
"""

import json
from validictory import validate
from validictory.validator import \
    (RequiredFieldValidationError,
     FieldValidationError)

from django.conf import settings

class UnknownCurrencyException(ValueError):
    def __init__(self, hint, http_status_code):
        self.hint = hint
        self.http_status_code = http_status_code
        super().__init__()

class URLParameterMissing(ValueError):
    def __init__(self, param, http_status_code):
        self.hint = "URL parameter '%s' is missing" % param
        self.http_status_code = http_status_code
        super().__init__()

class URLParameterMalformed(ValueError):
    def __init__(self, param, http_status_code):
        self.hint = "URL parameter '%s' is malformed" % param
        self.http_status_code = http_status_code
        super().__init__()

class JSONFieldException(ValueError):
    def __init__(self, hint, http_status_code):
        self.hint = hint
        self.http_status_code = http_status_code
        super().__init__()

'''
AMOUNT_SCHEMA = {
    "type": "object",
    "properties": {
        "value": {"type": "integer"},
        "fraction": {"type": "integer"},
        "currency": {"type": "string",
                     "pattern": "^"+settings.TALER_CURRENCY+"$"}
    }
}'''

AMOUNT_SCHEMA = {
    "type": "string",
    "pattern": "^"+settings.TALER_CURRENCY+":([0-9]+)\.?([0-9]+)?$"}

WITHDRAW_SESSION_SCHEMA = {
    "type": "object",
    "properties": {
        "amount": {"type": AMOUNT_SCHEMA},
        "reserve_pub": {"type": "string"},
        "exchange_account_number": {"type": "integer"},
        "sender_wiredetails": {
            "type": "object",
            "properties": {
                "type": {"type": "string"},
                "bank_url": {"type": "string"},
                "account_number": {"type": "integer"}
            }
        }
    }
}

WIREDETAILS_SCHEMA = {
    "type": "object",
    "properties": {
        "test": {
            "type": "object",
            "properties": {
                "type": {"type": "string"},
                "account_number": {"type": "integer"},
                "bank_url": {"type": "string"},
                "name": {"type": "string", "required": False},
            }
        }
    }
}

AUTH_SCHEMA = {
    "type": "object",
    "properties": {
        "type": {"type": "string",
                 "pattern": "^basic$"},
        "data": {"type": "object", "required": False}
    }
}

REJECT_REQUEST_SCHEMA = {
    "type": "object",
    "properties": {
        "auth": AUTH_SCHEMA,
        "row_id": {"type": "integer"},
        "account_number": {"type": "integer"}
    }
}

HISTORY_REQUEST_SCHEMA = {
    "type": "object",
    "properties": {
        "auth": {"type": "string", "pattern": "^basic$"},
        "cancelled": {"type": "string",
                      "pattern": "^(omit|show)$",
                      "required": False},
        "delta": {"type": "string",
                  "pattern": r"^([\+-])?([0-9])+$"},
        "start": {"type": "string",
                  "pattern": "^([0-9]+)$",
                  "required": False},
        "direction": {"type": "string",
                      "pattern": r"^(debit|credit|both|cancel\+|cancel-)$"},
        "account_number": {"type": "string",
                           "pattern": "^([0-9]+)$",
                           "required": False}
    }
}

INCOMING_REQUEST_SCHEMA = {
    "type": "object",
    "properties": {
        "amount": {"type": AMOUNT_SCHEMA},
        "subject": {"type": "string"},
        "credit_account": {"type": "integer"},
        "auth": AUTH_SCHEMA
    }
}

PIN_TAN_ARGS = {
    "type": "object",
    "properties": {
        "amount_value": {"format": "str_to_int"},
        "amount_fraction": {"format": "str_to_int"},
        "amount_currency": {"type": "string"},
        "exchange": {"type": "string"},
        "reserve_pub": {"type": "string"},
        "exchange_wire_details": {"format": "wiredetails_string"}
    }
}

def validate_pintan_types(validator, fieldname, value, format_option):
    del validator # pacify PEP checkers
    try:
        if format_option == "str_to_int":
            int(value)
        if format_option == "wiredetails_string":
            data = json.loads(value)
            validate_wiredetails(data)
    except Exception:
        raise FieldValidationError(
            "Malformed '%s'" % fieldname, fieldname, value)

def validate_pin_tan(data):
    format_dict = {
        "str_to_int": validate_pintan_types,
        "wiredetails_string": validate_pintan_types}
    validate(data, PIN_TAN_ARGS, format_validators=format_dict)

def validate_reject(data):
    validate(data, REJECT_REQUEST_SCHEMA)

def validate_history(data):
    validate(data, HISTORY_REQUEST_SCHEMA)

def validate_wiredetails(wiredetails):
    validate(wiredetails, WIREDETAILS_SCHEMA)

def validate_add_incoming(data):
    validate(data, INCOMING_REQUEST_SCHEMA)

def check_withdraw_session(data):
    validate(data, WITHDRAW_SESSION_SCHEMA)

def validate_data(request, data):
    switch = {
        "/reject": validate_reject,
        "/history": validate_history,
        "/admin/add/incoming": validate_add_incoming,
        "/pin/verify": check_withdraw_session,
        "/pin/question": validate_pin_tan
    }
    try:
        switch.get(request.path_info)(data)
    except  RequiredFieldValidationError as exc:
        if request.method == "GET":
            raise URLParameterMissing(exc.fieldname, 400)
        raise JSONFieldException(
            "Field '%s' is missing" % exc.fieldname, 400)
    except FieldValidationError as exc:
        if exc.fieldname == "currency":
            raise UnknownCurrencyException("Unknown currency", 406)
        if request.method == "GET":
            raise URLParameterMalformed(exc.fieldname, 400)
        raise JSONFieldException(
            "Malformed '%s' field" % exc.fieldname, 400)
