#  This file is part of TALER
#  (C) 2014, 2015, 2016 INRIA
#
#  TALER is free software; you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation; either version 3, or
# (at your option) any later version. TALER is distributed in the
# hope that it will be useful, but WITHOUT ANY WARRANTY; without
# even the implied warranty of MERCHANTABILITY or FITNESS FOR A
# PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License
# along with TALER; see the file COPYING.  If not, see
# <http://www.gnu.org/licenses/>
#
#  @author Marcello Stanisci
#  @author Florian Dold

from __future__ import unicode_literals
from typing import Any, Tuple
from django.contrib.auth.models import User
from django.db import models
from django.conf import settings
from django.core.exceptions import \
    ValidationError, \
    ObjectDoesNotExist
from .amount import Amount, BadFormatAmount

class AmountField(models.Field):

    description = 'Amount object in Taler style'

    def deconstruct(self) -> Tuple[str, str, list, dict]:
        name, path, args, kwargs = super(
            AmountField, self).deconstruct()
        return name, path, args, kwargs

    def db_type(self, connection: Any) -> str:
        return "varchar"

    # Pass stringified object to db connector
    def get_prep_value(self, value: Amount) -> str:
        if not value:
            return "%s:0.0" % settings.TALER_CURRENCY
        return value.stringify(settings.TALER_DIGITS)

    @staticmethod
    def from_db_value(value: str, *args) -> Amount:
        del args # pacify PEP checkers
        if value is None:
            return Amount.parse(settings.TALER_CURRENCY)
        return Amount.parse(value)

    def to_python(self, value: Any) -> Amount:
        if isinstance(value, Amount):
            return value
        try:
            if value is None:
                return Amount.parse(settings.TALER_CURRENCY)
            return Amount.parse(value)
        except BadFormatAmount:
            raise ValidationError(
                "Invalid input for an amount string: %s" % value)

def get_zero_amount() -> Amount:
    return Amount(settings.TALER_CURRENCY)

class BankAccountDoesNotExist(ObjectDoesNotExist):
    hint = "Specified bank account does not exist"
    http_status_code = 404

class BankTransactionDoesNotExist(ObjectDoesNotExist):
    hint = "Specified bank transaction does not exist"
    http_status_code = 404

class BankAccount(models.Model):
    is_public = models.BooleanField(default=False)
    debit = models.BooleanField(default=False)
    account_no = models.AutoField(primary_key=True)
    user = models.OneToOneField(User, on_delete=models.CASCADE)
    amount = AmountField(default=get_zero_amount)
    DoesNotExist = BankAccountDoesNotExist

class BankTransaction(models.Model):
    amount = AmountField(default=False)
    debit_account = models.ForeignKey(
        BankAccount,
        on_delete=models.CASCADE,
        db_index=True,
        related_name="debit_account")
    credit_account = models.ForeignKey(
        BankAccount,
        on_delete=models.CASCADE,
        db_index=True,
        related_name="credit_account")
    subject = models.CharField(
        default="(no subject given)", max_length=200)
    date = models.DateTimeField(
        auto_now=True, db_index=True)
    cancelled = models.BooleanField(default=False)
