import logging
from django.http import JsonResponse
from django.shortcuts import redirect
from .models import BankAccount, BankTransaction
from .views import \
    (DebitLimitException, SameAccountException,
     LoginFailed, RejectNoRightsException)
from .schemas import \
    (URLParameterMissing, URLParameterMalformed,
     JSONFieldException, UnknownCurrencyException)
from .amount import CurrencyMismatch, BadFormatAmount

LOGGER = logging.getLogger()

class ExceptionMiddleware:

    def __init__(self, get_response):
        self.get_response = get_response

        self.excs = {
            BankAccount.DoesNotExist: 0,
            BankTransaction.DoesNotExist: 1,
            SameAccountException: 2,
            DebitLimitException: 3,
            URLParameterMissing: 8,
            URLParameterMalformed: 9,
            JSONFieldException: 6,
            CurrencyMismatch: 11,
            BadFormatAmount: 11,
            LoginFailed: 12,
            RejectNoRightsException: 13,
            UnknownCurrencyException: 14}

        self.apis = {
            "/reject": 5300,
            "/history": 5200,
            "/admin/add/incoming": 5100}

        self.render = {
            "/profile": "profile",
            "/register": "index",
            "/public-accounts": "index",
            "/pin/verify": "profile"}


    def __call__(self, request):
        return self.get_response(request)

    def process_exception(self, request, exception):
        if not self.excs.get(exception.__class__):
            return None
        taler_ec = self.excs.get(exception.__class__)
        # The way error codes compose matches definitions found
        # at [1].
        taler_ec += self.apis.get(request.path, 1000)
        render_to = self.render.get(request.path)
        if not render_to:
            return JsonResponse({"ec": taler_ec,
                                 "error": exception.hint},
                                status=exception.http_status_code)
        request.session["profile_hint"] = \
            True, False, exception.hint
        return redirect(render_to)

# [1] https://git.taler.net/exchange.git/tree/src/include/taler_error_codes.h#n1502
