#  This file is part of TALER
#  (C) 2017 INRIA
#
#  TALER is free software; you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free Software
#  Foundation; either version 3, or (at your option) any later version.
#
#  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along with
#  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
#
#  @author Florian Dold

import os
from urllib.parse import urlparse
from django.urls import reverse, get_script_prefix
from django.conf import settings
from jinja2 import Environment


def is_absolute(urloc):
    return bool(urlparse(urloc).netloc)


def join_urlparts(*parts):
    ret = ""
    part = 0
    while part < len(parts):
        buf = parts[part]
        part += 1
        if ret.endswith("/"):
            buf = buf.lstrip("/")
        elif ret and not buf.startswith("/"):
            buf = "/" + buf
        ret += buf
    return ret


def static(urloc):
    if is_absolute(urloc):
        return urloc
    return join_urlparts(get_script_prefix(), settings.STATIC_URL, urloc)


def settings_value(name):
    return getattr(settings, name, "")


def url(url_name, **kwargs):
    # strangely, Django's 'reverse' function
    # takes a named parameter 'kwargs' instead
    # of real kwargs.
    return reverse(url_name, kwargs=kwargs)


def env_get(name, default=None):
    return os.environ.get(name, default)


def environment(**options):
    env = Environment(**options)
    env.globals.update({
        'static': static,
        'url': url,
        'settings_value': settings_value,
        'env': env_get,
    })
    return env
