"""
Django settings for talerbank.

For more information on this file, see
https://docs.djangoproject.com/en/1.9/topics/settings/

For the full list of settings and their values, see
https://docs.djangoproject.com/en/1.9/ref/settings/
"""

import os
import logging
import base64
from .talerconfig import TalerConfig, ConfigurationError
import sys
import urllib.parse
import re

logger = logging.getLogger(__name__)

logger.info("DJANGO_SETTINGS_MODULE: %s" % os.environ.get("DJANGO_SETTINGS_MODULE"))

tc = TalerConfig.from_file(os.environ.get("TALER_CONFIG_FILE"))

# Build paths inside the project like this: os.path.join(BASE_DIR, ...)
BASE_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))

# Quick-start development settings - unsuitable for production
# See https://docs.djangoproject.com/en/1.9/howto/deployment/checklist/


SECRET_KEY = os.environ.get("TALER_BANK_SECRET_KEY", None)

if not SECRET_KEY:
    logging.info("secret key not configured in TALER_BANK_SECRET_KEY env variable, generating random secret")
    SECRET_KEY = base64.b64encode(os.urandom(32)).decode('utf-8')

# SECURITY WARNING: don't run with debug turned on in production!
DEBUG = True

ALLOWED_HOSTS = ["*"]

LOGIN_URL = "login"

LOGIN_REDIRECT_URL = "index"


# Application definition

INSTALLED_APPS = [
    'django.contrib.admin',
    'django.contrib.auth',
    'django.contrib.contenttypes',
    'django.contrib.sessions',
    'django.contrib.messages',
    'django.contrib.staticfiles',
    'talerbank.app'
]

MIDDLEWARE_CLASSES = [
    'django.middleware.security.SecurityMiddleware',
    'django.contrib.sessions.middleware.SessionMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.csrf.CsrfViewMiddleware',
    'django.contrib.auth.middleware.AuthenticationMiddleware',
    'django.contrib.auth.middleware.SessionAuthenticationMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'django.middleware.clickjacking.XFrameOptionsMiddleware',
]

TEMPLATES = [
    {
        'BACKEND': 'django.template.backends.jinja2.Jinja2',
        'DIRS': [os.path.join(BASE_DIR, "talerbank/app/static/web-common/"),
                 os.path.join(BASE_DIR, "talerbank/app/templates")],
        'OPTIONS': {
            'environment': 'talerbank.jinja2.environment',
            },
    },
]

# Disable those, since they don't work with
# jinja2 anyways.
TEMPLATE_CONTEXT_PROCESSORS = []

WSGI_APPLICATION = 'talerbank.wsgi.application'


# Database
# https://docs.djangoproject.com/en/1.9/ref/settings/#databases

DATABASES = {}

dbname = tc.value_string("bank", "database", required=True)
# db given in cli argument takes precedence over config
dbname = os.environ.get("TALER_BANK_ALTDB", dbname)

if not dbname:
    raise Exception("DB not specified (neither in config or as cli argument)")

logger.info("dbname: %s" % dbname)

check_dbstring_format = re.search("[a-z]+:///[a-z]+", dbname)
if not check_dbstring_format:
    logger.error("Bad db string given, respect the format 'dbtype:///dbname'")
    sys.exit(2)

dbconfig = {}
db_url = urllib.parse.urlparse(dbname)


if ((db_url.scheme not in ("postgres")) or ("" == db_url.scheme)):
    logger.error("DB '%s' is not supported" % db_url.scheme)
    sys.exit(1)
if db_url.scheme == "postgres":
    dbconfig["ENGINE"] = 'django.db.backends.postgresql_psycopg2'
    dbconfig["NAME"] = db_url.path.lstrip("/")

if not db_url.netloc:
    p = urllib.parse.parse_qs(db_url.query)
    if ("host" not in p) or len(p["host"]) == 0:
        host = None
    else:
        host = p["host"][0]
else:
    host = db_url.netloc

if host:
    dbconfig["HOST"] = host

DATABASES["default"] = dbconfig

# Password validation
# https://docs.djangoproject.com/en/1.9/ref/settings/#auth-password-validators

AUTH_PASSWORD_VALIDATORS = [
    {
        'NAME': 'django.contrib.auth.password_validation.UserAttributeSimilarityValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.MinimumLengthValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.CommonPasswordValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.NumericPasswordValidator',
    },
]


# Internationalization
# https://docs.djangoproject.com/en/1.9/topics/i18n/

LANGUAGE_CODE = 'en-us'

TIME_ZONE = 'UTC'

USE_I18N = True

USE_L10N = True

USE_TZ = True


# Static files (CSS, JavaScript, Images)
# https://docs.djangoproject.com/en/1.9/howto/static-files/

STATIC_URL = '/static/'

STATICFILES_DIRS = [
    os.path.join(BASE_DIR, "talerbank/app/static"),
    os.path.join(BASE_DIR, "talerbank/app/static/web-common"),
]

# Currently we don't use "collectstatic", so this value isn't used.
# Instead, we serve static files directly from the installed python package
# via the "django.contrib.staticfiles" app.
# We must set it to something valid though, # or django will give us warnings.
STATIC_ROOT = '/tmp/talerbankstatic/'

ROOT_URLCONF = "talerbank.app.urls"

try:
    TALER_CURRENCY = tc.value_string("taler", "currency", required=True)
except ConfigurationError as e:
    logger.error(e)
    sys.exit(3)

TALER_MAX_DEBT = tc.value_string("bank", "MAX_DEBT", default="%s:50" % TALER_CURRENCY)
TALER_MAX_DEBT_BANK = tc.value_string("bank", "MAX_DEBT_BANK", default="%s:0" % TALER_CURRENCY)

TALER_DIGITS = 2
TALER_PREDEFINED_ACCOUNTS = ['Tor', 'GNUnet', 'Taler', 'FSF', 'Tutorial']
TALER_EXPECTS_DONATIONS = ['Tor', 'GNUnet', 'Taler', 'FSF']
TALER_SUGGESTED_EXCHANGE = tc.value_string("bank", "suggested_exchange")
