#  This file is part of TALER
#  (C) 2014, 2015, 2016 INRIA
#
#  TALER is free software; you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free Software
#  Foundation; either version 3, or (at your option) any later version.
#
#  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along with
#  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
#
#  @author Marcello Stanisci

from django.test import TestCase, Client
from django.core.urlresolvers import reverse
from django.conf import settings
from django.contrib.auth.models import User
from .models import BankAccount, BankTransaction
from . import urls
from . import amounts
from .views import wire_transfer
import json

import logging

logger = logging.getLogger(__name__)

def clearDb():
    User.objects.all().delete()
    BankAccount.objects.all().delete()
    BankTransaction.objects.all().delete()


class RegisterTestCase(TestCase):
    """User registration"""

    def setUp(self):
        bank = User.objects.create_user(username='Bank')
        ba = BankAccount(user=bank, currency=settings.TALER_CURRENCY)
        ba.account_no = 1
        ba.save() 

    def tearDown(self):
        clearDb()

    def test_register(self):
        c = Client()
        response = c.post(reverse("register", urlconf=urls),
                          {"username": "test_register",
                           "password": "test_register"},
                           follow=True)
        self.assertIn(("/profile", 302), response.redirect_chain)
        # this assertion tests "/profile""s view
        self.assertEqual(200, response.status_code)


class RegisterWrongCurrencyTestCase(TestCase):
    """User registration"""

    # Activating this user with a faulty currency.
    def setUp(self):
        bank = User.objects.create_user(username='Bank')
        ba = BankAccount(user=bank, currency="XYZ")
        ba.account_no = 1
        ba.save() 

    def tearDown(self):
        clearDb()

    def test_register(self):
        c = Client()
        response = c.post(reverse("register", urlconf=urls),
                          {"username": "test_register",
                           "password": "test_register"},
                           follow=True)
        # A currency mismatch is expected when the 100 KUDOS will be
        # attempted to be given to the new user.
        # This scenario expects a 500 Internal server error response to
        # be returned.
        self.assertEqual(500, response.status_code)

class LoginTestCase(TestCase):
    """User login"""

    def setUp(self):
        user = User.objects.create_user(username="test_user",
                                        password="test_password")
        user_account = BankAccount(user=user,
                                   currency=settings.TALER_CURRENCY)
        user_account.save()

    def tearDown(self):
        clearDb()
    
    def test_login(self):
        c = Client()
        response = c.post(reverse("login", urlconf=urls),
                          {"username": "test_user",
                           "password": "test_password"},
                           follow=True)
        self.assertIn(("/profile", 302), response.redirect_chain)

        # Send wrong password
        response = c.post(reverse("login", urlconf=urls),
                          {"username": "test_user",
                           "password": "test_passwordoo"},
                           follow=True)
        # The current logic -- django's default -- returns 200 OK
        # even when the login didn't succeed.
        # So the test here ensures that the bank just doesn't crash.
        self.assertEqual(200, response.status_code)

class AmountTestCase(TestCase):
    
    def test_cmp(self):
        a1 = dict(value=1, fraction=0, currency="X")
        _a1 = dict(value=1, fraction=0, currency="X")
        a2 = dict(value=2, fraction=0, currency="X")
        self.assertEqual(-1, amounts.amount_cmp(a1, a2))
        self.assertEqual(1, amounts.amount_cmp(a2, a1))
        self.assertEqual(0, amounts.amount_cmp(a1, _a1))

    # Trying to compare amount of different currencies
    def test_cmp_diff_curr(self):
        a1 = dict(value=1, fraction=0, currency="X")
        a2 = dict(value=2, fraction=0, currency="Y")
        try:
            amounts.amount_cmp(a1, a2)
        except amounts.CurrencyMismatchException:
            self.assertTrue(True)
            return
        # Should never get here
        self.assertTrue(False)


class AddIncomingTestCase(TestCase):
    """Test money transfer's API"""

    def setUp(self):
        bank = User.objects.create_user(username="bank_user",
                                        password="bank_password")
        bank_account = BankAccount(user=bank,
                                   currency=settings.TALER_CURRENCY)
        user = User.objects.create_user(username="user_user",
                                        password="user_password")
        user_account = BankAccount(user=user,
                                   currency=settings.TALER_CURRENCY)
        bank_account.save()
        user_account.save()

    def tearDown(self):
        clearDb()

    def test_add_incoming(self):
        c = Client()
        data = '{"auth": {"type": "basic"}, \
                 "credit_account": 1, \
                 "wtid": "TESTWTID", \
                 "exchange_url": "https://exchange.test", \
                 "amount": \
                   {"value": 1, \
                    "fraction": 0, \
                    "currency": "%s"}}' \
               % settings.TALER_CURRENCY
        response = c.post(reverse("add-incoming", urlconf=urls),
                          data=data,
                          content_type="application/json",
                          follow=True, **{"HTTP_X_TALER_BANK_USERNAME": "user_user", "HTTP_X_TALER_BANK_PASSWORD": "user_password"})
        self.assertEqual(200, response.status_code)
        data = '{"auth": {"type": "basic"}, \
                 "credit_account": 1, \
                 "wtid": "TESTWTID", \
                 "exchange_url": "https://exchange.test", \
                 "amount": \
                   {"value": 1, \
                    "fraction": 0, \
                    "currency": "%s"}}' \
               % "WRONGCURRENCY"
        response = c.post(reverse("add-incoming", urlconf=urls),
                          data=data,
                          content_type="application/json",
                          follow=True, **{"HTTP_X_TALER_BANK_USERNAME": "user_user", "HTTP_X_TALER_BANK_PASSWORD": "user_password"})
        self.assertEqual(406, response.status_code)

        # Try to go debit
        data = '{"auth": {"type": "basic"}, \
                 "credit_account": 1, \
                 "wtid": "TESTWTID", \
                 "exchange_url": "https://exchange.test", \
                 "amount": \
                   {"value": 50, \
                    "fraction": 1, \
                    "currency": "%s"}}' \
               % settings.TALER_CURRENCY
        response = c.post(reverse("add-incoming", urlconf=urls),
                          data=data,
                          content_type="application/json",
                          follow=True, **{"HTTP_X_TALER_BANK_USERNAME": "user_user", "HTTP_X_TALER_BANK_PASSWORD": "user_password"})
        self.assertEqual(403, response.status_code)


class HistoryTestCase(TestCase):

    def setUp(self):
        user = User.objects.create_user(username='User', password="Password")
        ub = BankAccount(user=user, currency=settings.TALER_CURRENCY)
        ub.account_no = 1
        ub.balance_obj = dict(value=100, fraction=0, currency=settings.TALER_CURRENCY)
        ub.save() 
        user_passive = User.objects.create_user(username='UserP', password="PasswordP")
        ub_p = BankAccount(user=user_passive, currency=settings.TALER_CURRENCY)
        ub_p.account_no = 2
        ub_p.save() 
        wire_transfer(dict(value=1, fraction=0, currency=settings.TALER_CURRENCY), ub, ub_p, subject="a")
        wire_transfer(dict(value=1, fraction=0, currency=settings.TALER_CURRENCY), ub, ub_p, subject="b")
        wire_transfer(dict(value=1, fraction=0, currency=settings.TALER_CURRENCY), ub, ub_p, subject="c")
        wire_transfer(dict(value=1, fraction=0, currency=settings.TALER_CURRENCY), ub, ub_p, subject="d")
        wire_transfer(dict(value=1, fraction=0, currency=settings.TALER_CURRENCY), ub, ub_p, subject="e")
        wire_transfer(dict(value=1, fraction=0, currency=settings.TALER_CURRENCY), ub, ub_p, subject="f")
        wire_transfer(dict(value=1, fraction=0, currency=settings.TALER_CURRENCY), ub, ub_p, subject="g")
        wire_transfer(dict(value=1, fraction=0, currency=settings.TALER_CURRENCY), ub, ub_p, subject="h")

    def tearDown(self):
        clearDb()

    def test_history(self):
        c = Client()

        response = c.get(reverse("history", urlconf=urls), {"auth": "basic", "delta": "+4"},
                         **{"HTTP_X_TALER_BANK_USERNAME": "User", "HTTP_X_TALER_BANK_PASSWORD": "Password"})
        self.assertEqual(200, response.status_code)

        # Get a delta=+1 record in the middle of the list
        response = c.get(reverse("history", urlconf=urls), {"auth": "basic", "delta": "+1", "start": "5"},
                         **{"HTTP_X_TALER_BANK_USERNAME": "User", "HTTP_X_TALER_BANK_PASSWORD": "Password"})
        data = response.content.decode("utf-8")
        data = json.loads(data)
        self.assertEqual(data["data"][0]["row_id"], 6)
        # Get latest record
        response = c.get(reverse("history", urlconf=urls), {"auth": "basic", "delta": "-1"},
                         **{"HTTP_X_TALER_BANK_USERNAME": "User", "HTTP_X_TALER_BANK_PASSWORD": "Password"})
        data = response.content.decode("utf-8")
        data = json.loads(data)
        self.assertEqual(data["data"][0]["wt_subject"], "h")
        # Get non-existent record: the latest plus one in the future: transaction "h" takes row_id 11
        response = c.get(reverse("history", urlconf=urls), {"auth": "basic", "delta": "1", "start": "11"},
                         **{"HTTP_X_TALER_BANK_USERNAME": "User", "HTTP_X_TALER_BANK_PASSWORD": "Password"})
        response_txt = response.content.decode("utf-8")
        self.assertEqual(204, response.status_code)
        # Get credit records
        response = c.get(reverse("history", urlconf=urls), {"auth": "basic", "delta": "+1", "direction": "credit"},
                         **{"HTTP_X_TALER_BANK_USERNAME": "User", "HTTP_X_TALER_BANK_PASSWORD": "Password"})
        self.assertEqual(204, response.status_code)
        # Get debit records
        response = c.get(reverse("history", urlconf=urls), {"auth": "basic", "delta": "+1", "direction": "debit"},
                         **{"HTTP_X_TALER_BANK_USERNAME": "User", "HTTP_X_TALER_BANK_PASSWORD": "Password"})
        self.assertNotEqual(204, response.status_code)
        # Query about non-owned account
        response = c.get(reverse("history", urlconf=urls), {"auth": "basic", "delta": "+1", "account_number": 2},
                         **{"HTTP_X_TALER_BANK_USERNAME": "User", "HTTP_X_TALER_BANK_PASSWORD": "Password"})
        self.assertEqual(403, response.status_code)
        # Query about non-existent account
        response = c.get(reverse("history", urlconf=urls), {"auth": "basic", "delta": "-1", "account_number": 9},
                         **{"HTTP_X_TALER_BANK_USERNAME": "User", "HTTP_X_TALER_BANK_PASSWORD": "Password"})
        self.assertEqual(404, response.status_code)


# This tests whether a bank account goes debit and then goes >=0  again
class DebitTestCase(TestCase):

    def setUp(self):
        u = User.objects.create_user(username='U')
        u0 = User.objects.create_user(username='U0')
        ua = BankAccount(user=u, currency=settings.TALER_CURRENCY)
        u0a = BankAccount(user=u0, currency=settings.TALER_CURRENCY)

        ua.save()
        u0a.save()

    def test_green(self):
        u = User.objects.get(username='U')
        ub = BankAccount.objects.get(user=u)
        self.assertEqual(False, ub.debit)

    def test_red(self):
        u = User.objects.get(username='U')
        u0 = User.objects.get(username='U0')

        ub = BankAccount.objects.get(user=u)
        ub0 = BankAccount.objects.get(user=u0)

        wire_transfer(dict(value=10, fraction=0, currency=settings.TALER_CURRENCY),
                      ub0,
                      ub,
                      "Go green")
        tmp = amounts.get_zero()
        tmp["value"] = 10

        self.assertEqual(0, amounts.amount_cmp(ub.balance_obj, tmp))
        self.assertEqual(False, ub.debit)
        self.assertEqual(True, ub0.debit)

        wire_transfer(dict(value=11, fraction=0, currency=settings.TALER_CURRENCY),
                      ub,
                      ub0,
                      "Go red")

        self.assertEqual(True, ub.debit)
        self.assertEqual(False, ub0.debit)

        tmp["value"] = 1

        self.assertEqual(0, amounts.amount_cmp(ub0.balance_obj, tmp))

class ParseAmountTestCase(TestCase):
     def test_parse_amount(self):
         ret = amounts.parse_amount("KUDOS:4")
         self.assertJSONEqual('{"value": 4, "fraction": 0, "currency": "KUDOS"}', json.dumps(ret))
         ret = amounts.parse_amount("KUDOS:4.00")
         self.assertJSONEqual('{"value": 4, "fraction": 0, "currency": "KUDOS"}', json.dumps(ret))
         ret = amounts.parse_amount("KUDOS:4.3")
         self.assertJSONEqual('{"value": 4, "fraction": 30000000, "currency": "KUDOS"}', json.dumps(ret))
         try:
             amounts.parse_amount("Buggy")
         except amounts.BadFormatAmount:
             return
         # make sure the control doesn't get here
         self.assertEqual(True, False)
