#  This file is part of TALER
#  (C) 2014, 2015, 2016 INRIA
#
#  TALER is free software; you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free Software
#  Foundation; either version 3, or (at your option) any later version.
#
#  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along with
#  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
#
#  @author Marcello Stanisci
#  @author Florian Dold

from __future__ import unicode_literals
from django.contrib.auth.models import User
from django.db import models


class BankAccount(models.Model):
    is_public = models.BooleanField(default=False)
    # Handier than keeping the amount signed, for two reasons:
    # (1) checking if someone is in debt is less verbose: with signed
    # amounts we have to check if the amount is less than zero; this
    # way we only check if a boolean is true. (2) The bank logic is
    # ready to welcome a data type for amounts which doesn't have any
    # sign notion, like Taler amounts do.
    debit = models.BooleanField(default=False)
    balance_value = models.IntegerField(default=0)
    balance_fraction = models.IntegerField(default=0)
    # From today's (16/10/2017) Mumble talk, it emerged that bank shouldn't
    # store amounts as floats, but: normal banks should not care about
    # Taler when representing values around in their databases..
    balance = models.FloatField(default=0)
    currency = models.CharField(max_length=12, default="")
    account_no = models.AutoField(primary_key=True)
    user = models.OneToOneField(User, on_delete=models.CASCADE)
    def _get_balance(self):
        return dict(value=self.balance_value,
                    fraction=self.balance_fraction,
                    currency=self.currency)
    def _set_balance(self, amount):
        self.balance_value = amount["value"]
        self.balance_fraction = amount["fraction"]
        self.currency = amount["currency"]
    balance_obj = property(_get_balance, _set_balance)

class BankTransaction(models.Model):
    amount_value = models.IntegerField(default=0)
    amount_fraction = models.IntegerField(default=0)
    currency = models.CharField(max_length=12)
    debit_account = models.ForeignKey(BankAccount,
                                      on_delete=models.CASCADE,
                                      related_name="debit_account")
    credit_account = models.ForeignKey(BankAccount,
                                       on_delete=models.CASCADE,
                                       related_name="credit_account")
    subject = models.CharField(default="(no subject given)", max_length=200)
    date = models.DateTimeField(auto_now=True)
    def _get_amount(self):
        return dict(value=self.amount_value,
                    fraction=self.amount_fraction,
                    currency=self.currency)
    def _set_amount(self, amount):
        self.amount_value = amount["value"]
        self.amount_fraction = amount["fraction"]
        self.currency = amount["currency"]
    amount_obj = property(_get_amount, _set_amount)
