#  This file is part of TALER
#  (C) 2016 INRIA
#
#  TALER is free software; you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free Software
#  Foundation; either version 3, or (at your option) any later version.
#
#  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along with
#  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
#
#  @author Marcello Stanisci
#  @author Florian Dold


import re
import math
import logging
from django.conf import settings

logger = logging.getLogger(__name__)

FRACTION = 100000000

class CurrencyMismatchException(Exception):
    def __init__(self, msg=None, status_code=0):
        self.msg = msg
        # HTTP status code to be returned as response for
        # this exception
        self.status_code = status_code

class BadFormatAmount(Exception):
    def __init__(self, msg=None, status_code=0):
        self.msg = msg
        # HTTP status code to be returned as response for
        # this exception
        self.status_code = status_code


def check_currency(a1, a2):
    if a1["currency"] != a2["currency"]:
        logger.error("Different currencies given: %s vs %s" % (a1["currency"], a2["currency"]))
        raise CurrencyMismatchException

def get_zero():
    return dict(value=0, fraction=0, currency=settings.TALER_CURRENCY)

def amount_add(a1, a2):
    check_currency(a1, a2)
    a1_float = floatify(a1)
    a2_float = floatify(a2)
    return parse_amount("%s:%s" % (a2["currency"], str(a1_float + a2_float)))

def amount_sub(a1, a2):
    check_currency(a1, a2)
    a1_float = floatify(a1)
    a2_float = floatify(a2)
    sub = a1_float - a2_float
    fmt = "%s:%s" % (a2["currency"], str(sub))
    return parse_amount(fmt)

# Return -1 if a1 < a2, 0 if a1 == a2, 1 if a1 > a2
def amount_cmp(a1, a2):
    check_currency(a1, a2)
    a1_float = floatify(a1)
    a2_float = floatify(a2)

    if a1_float < a2_float:
        return -1
    elif a1_float == a2_float:
        return 0

    return 1


def floatify(amount_dict):
    return amount_dict['value'] + (float(amount_dict['fraction']) / float(FRACTION))

def stringify(amount_float, digits=2):
    o = "".join(["%.", "%sf" % digits])
    return o % amount_float

def parse_amount(amount_str):
    """
    Parse amount of return None if not a
    valid amount string
    """
    parsed = re.search("^\s*([-_*A-Za-z0-9]+):([0-9]+)(\.[0-9]+)?\s*$", amount_str)
    if not parsed:
        raise BadFormatAmount
    value = int(parsed.group(2))
    fraction = 0
    if parsed.group(3) is not None:
        for i, digit in enumerate(parsed.group(3)[1:]):
            fraction += int(int(digit) * (FRACTION / 10 ** (i+1)))
    return {'value': value,
            'fraction': fraction,
            'currency': parsed.group(1)}
