#  This file is part of TALER
#  (C) 2014, 2015, 2016 INRIA
#
#  TALER is free software; you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free Software
#  Foundation; either version 3, or (at your option) any later version.
#
#  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along with
#  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
#
#  @author Marcello Stanisci
#  @author Florian Dold

import re
import django.contrib.auth
import django.contrib.auth.views
import django.contrib.auth.forms
from django.db import transaction
from django import forms
from django.conf import settings
from django.contrib.auth.decorators import login_required
from django.http import JsonResponse, HttpResponse, HttpResponseBadRequest, HttpResponseServerError
from django.shortcuts import render, redirect
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_POST, require_GET
from simplemathcaptcha.fields import MathCaptchaField, MathCaptchaWidget
from django.core.urlresolvers import reverse
from django.contrib.auth.models import User
from django.db.models import Q
import json
import logging
import time
import hashlib
import requests
from urllib.parse import urljoin
from . import amounts
from . import schemas
from .models import BankAccount, BankTransaction

logger = logging.getLogger(__name__)

class DebtLimitExceededException(Exception):
    pass
class SameAccountException(Exception):
    pass

class MyAuthenticationForm(django.contrib.auth.forms.AuthenticationForm):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.fields["username"].widget.attrs["autofocus"] = True
        self.fields["username"].widget.attrs["placeholder"] = "Username"
        self.fields["password"].widget.attrs["placeholder"] = "Password"


def ignore(request):
    return HttpResponse()

def javascript_licensing(request):
    return render(request, "javascript.html")

def login_view(request):
    just_logged_out = get_session_flag(request, "just_logged_out")
    response = django.contrib.auth.views.login(
            request,
            authentication_form=MyAuthenticationForm,
            template_name="login.html",
            extra_context={"user": request.user})
    # sometimes the response is a redirect and not a template response
    if hasattr(response, "context_data"):
        response.context_data["just_logged_out"] = just_logged_out
    return response

def get_session_flag(request, name):
    """
    Get a flag from the session and clear it.
    """
    if name in request.session:
        del request.session[name]
        return True
    return False


# Check if user's logged in.  Check if he/she has withdrawn or
# registered; render profile page.

@login_required
def profile_page(request):
    just_withdrawn = get_session_flag(request, "just_withdrawn")
    just_registered = get_session_flag(request, "just_registered")
    no_initial_bonus = get_session_flag(request, "no_initial_bonus")
    user_account = BankAccount.objects.get(user=request.user)
    history = extract_history(user_account)
    reserve_pub = request.session.get("reserve_pub")

    context = dict(
        name=user_account.user.username,
        balance=amounts.stringify(amounts.floatify(user_account.balance_obj)),
        currency=user_account.currency,
        precision=settings.TALER_DIGITS,
        account_no=user_account.account_no,
        history=history,
        just_withdrawn=just_withdrawn,
        just_registered=just_registered,
        no_initial_bonus=no_initial_bonus,
    )
    if settings.TALER_SUGGESTED_EXCHANGE:
        context["suggested_exchange"] = settings.TALER_SUGGESTED_EXCHANGE

    response = render(request, "profile_page.html", context)
    if just_withdrawn:
       response["X-Taler-Operation"] = "confirm-reserve"
       response["X-Taler-Reserve-Pub"] = reserve_pub
       response.status_code = 202
    return response


class Pin(forms.Form):
    pin = MathCaptchaField(
        widget=MathCaptchaWidget(
            attrs=dict(autocomplete="off", autofocus=True),
            question_tmpl="<div lang=\"en\">What is %(num1)i %(operator)s %(num2)i ?</div>"))


@require_GET
@login_required
def pin_tan_question(request):
    for param in ("amount_value",
                  "amount_fraction",
                  "amount_currency",
                  "exchange",
                  "reserve_pub",
                  "wire_details"):
        if param not in request.GET:
            return HttpResponseBadRequest("parameter {} missing".format(param))
    try:
        value = int(request.GET.get("amount_value", None))
    except ValueError:
        return HttpResponseBadRequest("invalid parameters: \"amount_value\" not given or NaN")
    try:
        fraction = int(request.GET.get("amount_fraction", None))
    except ValueError:
        return HttpResponseBadRequest("invalid parameters: \"amount_fraction\" not given or NaN")
    try:
        currency = request.GET.get("amount_currency", None)
    except ValueError:
        return HttpResponseBadRequest("invalid parameters: \"amount_currency\" not given")
    amount = {"value": value,
              "fraction": fraction,
              "currency": currency}
    user_account = BankAccount.objects.get(user=request.user)
    wiredetails = json.loads(request.GET["wire_details"])
    if not isinstance(wiredetails, dict) or "test" not in wiredetails:
        return HttpResponseBadRequest(
                "This bank only supports the test wire transfer method. "
                "The exchange does not seem to support it.")
    try:
        schemas.validate_wiredetails(wiredetails)
        schemas.validate_amount(amount)
    except ValueError as error:
        return HttpResponseBadRequest("invalid parameters (%s)" % error)
    # parameters we store in the session are (more or less) validated
    request.session["exchange_account_number"] = wiredetails["test"]["account_number"]
    request.session["amount"] = amount
    request.session["exchange_url"] = request.GET["exchange"]
    request.session["reserve_pub"] = request.GET["reserve_pub"]
    request.session["sender_wiredetails"] = dict(
        type="TEST",
        bank_uri=request.build_absolute_uri(reverse("index")),
        account_number=user_account.account_no
    )
    previous_failed = get_session_flag(request, "captcha_failed")
    context = dict(
        form=Pin(auto_id=False),
        amount=amounts.floatify(amount),
        previous_failed=previous_failed,
        exchange=request.GET["exchange"],
    )
    return render(request, "pin_tan.html", context)


@require_POST
@login_required
def pin_tan_verify(request):
    try:
        given = request.POST["pin_0"]
        hashed_result = request.POST["pin_1"]
        question_url = request.POST["question_url"]
    except Exception:  # FIXME narrow the Exception type
        return redirect("profile")
    hasher = hashlib.new("sha1")
    hasher.update(settings.SECRET_KEY.encode("utf-8"))
    hasher.update(given.encode("utf-8"))
    hashed_attempt = hasher.hexdigest()
    if hashed_attempt != hashed_result:
        request.session["captcha_failed"] = True
        return redirect(question_url)
    # We recover the info about reserve creation from the session (and
    # not from POST parameters), since we don't what the user to
    # change it after we've verified it.
    try:
        amount = request.session["amount"]
        exchange_url = request.session["exchange_url"]
        reserve_pub = request.session["reserve_pub"]
        exchange_account_number = request.session["exchange_account_number"]
        sender_wiredetails = request.session["sender_wiredetails"]
    except KeyError:
        # This is not a withdraw session, we redirect the user to the
        # profile page.
        return redirect("profile")
    try:
        BankAccount.objects.get(account_no=exchange_account_number)
    except BankAccount.DoesNotExist:
        raise HttpResponseBadRequest("The bank account #{} of exchange {} does not exist".format(exchange_account_no, exchange_url))
    logging.info("asking exchange {} to create reserve {}".format(exchange_url, reserve_pub))
    json_body = dict(
            reserve_pub=reserve_pub,
            execution_date="/Date(" + str(int(time.time())) + ")/",
            sender_account_details=sender_wiredetails,
             # just something unique
            transfer_details=dict(timestamp=int(time.time() * 1000)),
            amount=amount,
    )
    user_account = BankAccount.objects.get(user=request.user)
    exchange_account = BankAccount.objects.get(account_no=exchange_account_number)
    try:
        wire_transfer(amount, user_account, exchange_account, reserve_pub)
    except DebtLimitExceededException:
        logger.warning("Withdrawal impossible due to debt limit exceeded")
        request.session["debt_limit"] = True
        return redirect("profile")
    except amounts.BadFormatAmount as e:
        return HttpResponse(e.msg, status=e.status_code) 
    except amounts.CurrencyMismatchException as e:
        return HttpResponse(e.msg, status=e.status_code) 
    except SameAccountException:
        logger.error("Odd situation: SameAccountException should NOT occur in this function")
        return HttpResponse("internal server error", status=500)

    request_url = urljoin(exchange_url, "admin/add/incoming")
    res = requests.post(request_url, json=json_body)
    if res.status_code != 200:
        return render(request, "error_exchange.html", dict(
            message="Could not transfer funds to the exchange.  The exchange ({}) gave a bad response.".format(exchange_url),
            response_text=res.text,
            response_status=res.status_code,
        ))
    request.session["just_withdrawn"] = True
    return redirect("profile")


class UserReg(forms.Form):
    username = forms.CharField()
    password = forms.CharField(widget=forms.PasswordInput())


def register(request):
    """
    register a new user giving 100 KUDOS bonus
    """
    if request.method != "POST":
        return render(request, "register.html")
    form = UserReg(request.POST)
    if not form.is_valid():
        return render(request, "register.html", dict(wrong_field=True))
    username = form.cleaned_data["username"]
    password = form.cleaned_data["password"]
    if User.objects.filter(username=username).exists():
        return render(request, "register.html", dict(not_available=True))
    with transaction.atomic():
        user = User.objects.create_user(username=username, password=password)
        user_account = BankAccount(user=user, currency=settings.TALER_CURRENCY)
        user_account.save()
    bank_internal_account = BankAccount.objects.get(account_no=1)
    amount = dict(value=100, fraction=0, currency=settings.TALER_CURRENCY)
    try:
        wire_transfer(amount, bank_internal_account, user_account, "Joining bonus")
    except DebtLimitExceededException:
        logger.info("Debt situation encountered")
        request.session["no_initial_bonus"] = True
    except amounts.CurrencyMismatchException as e:
        return HttpResponse(e.msg, status=e.status_code)
    except amounts.BadFormatAmount as e:
        return HttpResponse(e.msg, status=e.status_code)
    except SameAccountException:
        logger.error("Odd situation: SameAccountException should NOT occur in this function")
        return HttpResponse("internal server error", status=500)
        
    request.session["just_registered"] = True
    user = django.contrib.auth.authenticate(username=username, password=password)
    django.contrib.auth.login(request, user)
    return redirect("profile")


def logout_view(request):
    """
    Log out the user and redirect to index page.
    """
    django.contrib.auth.logout(request)
    request.session["just_logged_out"] = True
    return redirect("index")


def extract_history(account):
    history = []
    related_transactions = BankTransaction.objects.filter(
            Q(debit_account=account) | Q(credit_account=account))
    for item in related_transactions:
        if item.credit_account == account:
            counterpart = item.debit_account
            sign = 1
        else:
            counterpart = item.credit_account
            sign = -1
        entry = dict(
            float_amount=amounts.stringify(amounts.floatify(item.amount_obj) * sign),
            float_currency=item.currency,
            counterpart=counterpart.account_no,
            counterpart_username=counterpart.user.username,
            subject=item.subject,
            date=item.date.strftime("%d/%m/%y %H:%M"),
        )
        history.append(entry)
    return history


def public_accounts(request, name=None):
    if not name:
        name = settings.TALER_PREDEFINED_ACCOUNTS[0]
    try:
        user = User.objects.get(username=name)
        account = BankAccount.objects.get(user=user, is_public=True)
    except User.DoesNotExist:
        return HttpResponse("account '{}' not found".format(name), status=404)
    except BankAccount.DoesNotExist:
        return HttpResponse("account '{}' not found".format(name), status=404)
    public_accounts = BankAccount.objects.filter(is_public=True)
    history = extract_history(account)
    context = dict(
        public_accounts=public_accounts,
        selected_account=dict(
            name=name,
            number=account.account_no,
            history=history,
        )
    )
    return render(request, "public_accounts.html", context)

@require_GET
def history(request):
    """
    This API is used to get a list of transactions related to one user.
    """
    # login caller
    user_account = auth_and_login(request)
    if not user_account:
        return JsonResponse(dict(error="authentication failed: bad credentials OR auth method"),
                            status=401)
    # delta
    delta = request.GET.get("delta")
    if not delta:
        return HttpResponseBadRequest()
    #FIXME: make the '+' sign optional
    parsed_delta = re.search("([\+-])?([0-9]+)", delta)
    try:
        parsed_delta.group(0)
    except AttributeError:
        return JsonResponse(dict(error="Bad 'delta' parameter"), status=400)
    delta = int(parsed_delta.group(2))
    # start
    start = request.GET.get("start")
    if start:
        start = int(start)

    sign = parsed_delta.group(1)

    if ("+" == sign) or (not sign):
        sign = ""
    # Assuming Q() means 'true'
    sign_filter = Q()
    if "-" == sign and start:
        sign_filter = Q(id__lt=start)
    elif "" == sign and start:
        sign_filter = Q(id__gt=start)
    # direction (debit/credit)
    direction = request.GET.get("direction")

    # target account
    target_account = request.GET.get("account_number")
    if not target_account:
        target_account = user_account.bankaccount
    else:
        try:
            target_account = BankAccount.objects.get(account_no=target_account)
        except BankAccount.DoesNotExist:
            logger.error("Attempted /history about non existent account")
            return JsonResponse(dict(error="Queried account does not exist"), status=404)

    if target_account != user_account.bankaccount:
        return JsonResponse(dict(error="Querying unowned accounts not allowed"), status=403)

    query_string = Q(debit_account=target_account) | Q(credit_account=target_account)
    history = []

    if "credit" == direction:
        query_string = Q(credit_account=target_account)
    if "debit" == direction:
        query_string = Q(debit_account=target_account)

    qs = BankTransaction.objects.filter(query_string, sign_filter).order_by("%sid" % sign)[:delta]
    if 0 == qs.count():
        return HttpResponse(status=204)
    for entry in qs:
        counterpart = entry.credit_account.account_no
        sign_ = "-"
        if entry.credit_account.account_no == user_account.bankaccount.account_no:
            counterpart = entry.debit_account.account_no
            sign_ = "+"
        history.append(dict(counterpart=counterpart,
                            amount=entry.amount_obj,
                            sign=sign_,
                            wt_subject=entry.subject,
                            row_id=entry.id,
                            date="/Date(" + str(int(entry.date.timestamp())) + ")/"))
    return JsonResponse(dict(data=history), status=200)


def auth_and_login(request):
    """Return user instance after checking authentication
       credentials, False if errors occur"""

    auth_type = None
    if "POST" == request.method:
        data = json.loads(request.body.decode("utf-8"))
        auth_type = data["auth"]["type"]
    if "GET" == request.method:
        auth_type = request.GET.get("auth")

    if "basic" != auth_type:
        logger.error("auth method not supported")
        return False

    username = request.META.get("HTTP_X_TALER_BANK_USERNAME")
    password = request.META.get("HTTP_X_TALER_BANK_PASSWORD")
    # logger.info("Trying to log '%s/%s' in" % (username, password))
    if not username or not password:
        return False
    return django.contrib.auth.authenticate(username=username,
                                            password=password)


@csrf_exempt
@require_POST
def add_incoming(request):
    """
    Internal API used by exchanges to notify the bank
    of incoming payments.

    This view is CSRF exempt, since it is not used from
    within the browser, and only over the private admin interface.
    """
    data = json.loads(request.body.decode("utf-8"))
    subject = "%s %s" % (data["wtid"], data["exchange_url"])
    try:
        schemas.validate_incoming_request(data)
    except ValueError as error:
        logger.error("Bad data POSTed: %s" % error)
        return JsonResponse(dict(error="invalid data POSTed: %s" % error), status=400)

    user_account = auth_and_login(request)

    if not user_account:
        return JsonResponse(dict(error="authentication failed"),
                            status=401)

    try:
        credit_account = BankAccount.objects.get(user=data["credit_account"])
    except BankAccount.DoesNotExist:
        return HttpResponse(status=404)
    try:
        transaction = wire_transfer(data["amount"],
                                    user_account.bankaccount,
                                    credit_account,
                                    subject)
        return JsonResponse(dict(serial_id=transaction.id, timestamp="/Date(%s)/" % int(transaction.date.timestamp())))
    except amounts.BadFormatAmount as e:
        return JsonResponse(dict(error=e.msg), status=e.status_code)
    except SameAccountException:
        return JsonResponse(dict(error="debit and credit account are the same"), status=422)
    except DebtLimitExceededException:
        return JsonResponse(dict(error="debit count has reached its debt limit", status=403 ),
                             status=403)
    except amounts.CurrencyMismatchException as e:
        return JsonResponse(dict(error=e.msg), status=e.status_code)

@login_required
@require_POST
def withdraw_nojs(request):

    try:
        amount = amounts.parse_amount(request.POST.get("kudos_amount", ""))
    except amounts.BadFormatAmount:
        logger.error("Amount did not pass parsing")
        return HttpResponseBadRequest()

    response = HttpResponse(status=202)
    response["X-Taler-Operation"] = "create-reserve"
    response["X-Taler-Callback-Url"] = reverse("pin-question")
    response["X-Taler-Wt-Types"] = '["TEST"]'
    response["X-Taler-Amount"] = json.dumps(amount)
    if settings.TALER_SUGGESTED_EXCHANGE:
        response["X-Taler-Suggested-Exchange"] = settings.TALER_SUGGESTED_EXCHANGE
    return response


def wire_transfer(amount,
                  debit_account,
                  credit_account,
                  subject):
    if debit_account.pk == credit_account.pk:
        logger.error("Debit and credit account are the same!")
        raise SameAccountException()

    transaction_item = BankTransaction(amount_value=amount["value"],
                                       amount_fraction=amount["fraction"],
                                       currency=amount["currency"],
                                       credit_account=credit_account,
                                       debit_account=debit_account,
                                       subject=subject)

    try:
        if debit_account.debit:
            debit_account.balance_obj = amounts.amount_add(debit_account.balance_obj,
                                                           amount)
    
        elif -1 == amounts.amount_cmp(debit_account.balance_obj, amount):
            debit_account.debit = True
            debit_account.balance_obj = amounts.amount_sub(amount,
                                                           debit_account.balance_obj)
        else:
            debit_account.balance_obj = amounts.amount_sub(debit_account.balance_obj,
                                                           amount)

        if False == credit_account.debit:
            credit_account.balance_obj = amounts.amount_add(credit_account.balance_obj,
                                                            amount)
    
        elif 1 == amounts.amount_cmp(amount, credit_account.balance_obj):
            credit_account.debit = False
            credit_account.balance_obj = amounts.amount_sub(amount,
                                                            credit_account.balance_obj)
        else:
            credit_account.balance_obj = amounts.amount_sub(credit_account.balance_obj,
                                                            amount)
    except amounts.CurrencyMismatchException:
        msg = "The amount to be transferred (%s) doesn't match the bank's currency (%s)" % (amount["currency"], settings.TALER_CURRENCY)
        status_code = 406
        if settings.TALER_CURRENCY != credit_account.balance_obj["currency"]:
            logger.error("Internal inconsistency: credit account's currency (%s) differs from bank's (%s)" % (credit_account.balance_obj["currency"], settings.TALER_CURRENCY))
            msg = "Internal server error"
            status_code = 500
        elif settings.TALER_CURRENCY != debit_account.balance_obj["currency"]:
            logger.error("Internal inconsistency: debit account's currency (%s) differs from bank's (%s)" % (debit_account.balance_obj["currency"], settings.TALER_CURRENCY))
            msg = "Internal server error"
            status_code = 500
        logger.error(msg)
        raise amounts.CurrencyMismatchException(msg=msg, status_code=status_code)

    # Check here if any account went beyond the allowed
    # debit threshold.

    try:
        threshold = amounts.parse_amount(settings.TALER_MAX_DEBT)

        if debit_account.user.username == "Bank":
            threshold = amounts.parse_amount(settings.TALER_MAX_DEBT_BANK)
    except amounts.BadFormatAmount:
        logger.error("MAX_DEBT|MAX_DEBT_BANK had the wrong format")
        raise amounts.BadFormatAmount(msg="internal server error", status_code=500)

    try:
        if 1 == amounts.amount_cmp(debit_account.balance_obj, threshold) \
           and 0 != amounts.amount_cmp(amounts.get_zero(), threshold) \
           and debit_account.debit:
            logger.error("Negative balance '%s' not allowed." % json.dumps(debit_account.balance_obj))
            logger.info("%s's threshold is: '%s'." % (debit_account.user.username, json.dumps(threshold)))
            raise DebtLimitExceededException()
    except amounts.CurrencyMismatchException:
        logger.error("(Internal) currency mismatch between debt threshold and debit account")
        raise amounts.CurrencyMismatchException(msg="internal server error", status_code=500)

    with transaction.atomic():
        debit_account.save()
        credit_account.save()
        transaction_item.save()

    return transaction_item
