#  This file is part of TALER
#  (C) 2014, 2015, 2106 INRIA
#
#  TALER is free software; you can redistribute it and/or modify it under the
#  terms of the GNU General Public License as published by the Free Software
#  Foundation; either version 3, or (at your option) any later version.
#
#  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along with
#  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
#
#  @author Marcello Stanisci
#  @author Florian Dold

import sys
import logging
from django.contrib.auth.models import User
from django.db.utils import ProgrammingError, DataError, OperationalError
from django.core.management.base import BaseCommand
from ...models import BankAccount
from django.conf import settings

logger = logging.getLogger(__name__)


def demo_accounts():
    for name in settings.TALER_PREDEFINED_ACCOUNTS:
        try:
            User.objects.get(username=name)
        except User.DoesNotExist:
            u = User.objects.create_user(username=name, password='x')
            b = BankAccount(user=u,
                            currency=settings.TALER_CURRENCY,
                            is_public=True)
            b.save()
            logger.info("Creating account '%s' with number %s", name, b.account_no)


def ensure_account(name):
    logger.info("ensuring account '{}'".format(name))
    user = None
    try:
        user = User.objects.get(username=name)
    except (OperationalError, ProgrammingError):
        logger.error("likely causes: non existent DB or unmigrated project\n"
                     "(try 'taler-bank-manage django migrate' in the latter case)",
                     stack_info=False,
                     exc_info=True)
        sys.exit(1)
    except User.DoesNotExist:
        logger.info("Creating *user* account '{}'".format(name))
        user = User.objects.create_user(username=name, password='x')

    try:
        BankAccount.objects.get(user=user)

    except BankAccount.DoesNotExist:
        acc = BankAccount(user=user,
                          currency=settings.TALER_CURRENCY,
                          is_public=True)
        acc.save()
        logger.info("Creating *bank* account number '{}' for user '{}'".format(acc.account_no, name))


def basic_accounts():
    ensure_account("Bank")
    ensure_account("Exchange")


class Command(BaseCommand):
    help = "Provide initial user accounts"

    def handle(self, *args, **options):
        basic_accounts()
        demo_accounts()
