"""
Django settings for talerbank.

For more information on this file, see
https://docs.djangoproject.com/en/1.9/topics/settings/

For the full list of settings and their values, see
https://docs.djangoproject.com/en/1.9/ref/settings/
"""

import os
import logging
import base64
from .talerconfig import TalerConfig
import sys
import urllib.parse

logger = logging.getLogger(__name__)

tc = TalerConfig.from_file(os.environ.get("TALER_CONFIG_FILE"))

# Build paths inside the project like this: os.path.join(BASE_DIR, ...)
BASE_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))

# Quick-start development settings - unsuitable for production
# See https://docs.djangoproject.com/en/1.9/howto/deployment/checklist/


SECRET_KEY = os.environ.get("TALER_BANK_SECRET_KEY", None)

if not SECRET_KEY:
    logging.info("secret key not configured in TALER_BANK_SECRET_KEY env variable, generating random secret")
    SECRET_KEY = base64.b64encode(os.urandom(32)).decode('utf-8')

# SECURITY WARNING: don't run with debug turned on in production!
DEBUG = True

ALLOWED_HOSTS = ["*"]


# Application definition

INSTALLED_APPS = [
    'django.contrib.admin',
    'django.contrib.auth',
    'django.contrib.contenttypes',
    'django.contrib.sessions',
    'django.contrib.messages',
    'django.contrib.staticfiles',
    'talerbank.app'
]

MIDDLEWARE_CLASSES = [
    'django.middleware.security.SecurityMiddleware',
    'django.contrib.sessions.middleware.SessionMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.csrf.CsrfViewMiddleware',
    'django.contrib.auth.middleware.AuthenticationMiddleware',
    'django.contrib.auth.middleware.SessionAuthenticationMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'django.middleware.clickjacking.XFrameOptionsMiddleware',
    'talerbank.app.middleware.ExpectedExceptionsMiddleware',
]

# To be dynamically set at launch time (by *.wsgi scripts)
ROOT_URLCONF = ''

TEMPLATES = [
    {
        'BACKEND': 'django.template.backends.django.DjangoTemplates',
        'DIRS': [],
        'APP_DIRS': True,
        'OPTIONS': {
            'context_processors': [
                'django.template.context_processors.debug',
                'django.template.context_processors.request',
                'django.contrib.auth.context_processors.auth',
                'django.contrib.messages.context_processors.messages',
            ],
        },
    },
]

WSGI_APPLICATION = 'talerbank.wsgi.application'


# Database
# https://docs.djangoproject.com/en/1.9/ref/settings/#databases

# parse a database URL, django can't natively do this!
dbname = tc.value_string("bank", "database", required=True)
dbconfig = {}
db_url = urllib.parse.urlparse(dbname)
if db_url.scheme != "postgres":
    raise Exception("only postgres db is supported ('{}' not understood)".format(dbname))
dbconfig['ENGINE'] = 'django.db.backends.postgresql_psycopg2'
dbconfig['NAME'] = db_url.path.lstrip("/")

if not db_url.netloc:
    p = urllib.parse.parse_qs(db_url.query)
    if ("host" not in p) or len(p["host"]) == 0:
        host = None
    else:
        host = p["host"][0]
else:
    host = db_url.netloc

if host:
    dbconfig["HOST"] = host

logger.info("db string '%s'", dbname)
logger.info("db info '%s'", dbconfig)

DATABASES = {
    'default': dbconfig
}


# Password validation
# https://docs.djangoproject.com/en/1.9/ref/settings/#auth-password-validators

AUTH_PASSWORD_VALIDATORS = [
    {
        'NAME': 'django.contrib.auth.password_validation.UserAttributeSimilarityValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.MinimumLengthValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.CommonPasswordValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.NumericPasswordValidator',
    },
]


# Internationalization
# https://docs.djangoproject.com/en/1.9/topics/i18n/

LANGUAGE_CODE = 'en-us'

TIME_ZONE = 'UTC'

USE_I18N = True

USE_L10N = True

USE_TZ = True


# Static files (CSS, JavaScript, Images)
# https://docs.djangoproject.com/en/1.9/howto/static-files/

STATIC_URL = '/static/'

STATICFILES_DIRS = [
    os.path.join(BASE_DIR, "talerbank/app/static"),
]

# Currently we don't use "collectstatic", so this value isn't used.
# Instead, we serve static files directly from the installed python package
# via the "django.contrib.staticfiles" app.
# We must set it to something valid though, # or django will give us warnings.
STATIC_ROOT = '/tmp/talerbankstatic/'



TALER_WIREDETAILS_COUNTER = 0
TALER_CURRENCY = tc.value_string("taler", "currency", required=True)
# How many digits we want to be shown for amounts fractional part
NDIGITS = tc.value_int("bank", "ndigits")
if NDIGITS is None:
    NDIGITS = 2
# Taler-compliant fractional part in amount objects, currently 1e8
FRACTION = tc.value_int("bank", "fraction")
if FRACTION is None:
    FRACTION = 100000000
TALER_PREDEFINED_ACCOUNTS = ['Tor', 'GNUnet', 'Taler', 'FSF', 'Tutorial']
TALER_EXPECTS_DONATIONS = ['Tor', 'GNUnet', 'Taler', 'FSF']
