#  This file is part of TALER
#  (C) 2014, 2015, 2016 INRIA
#
#  TALER is free software; you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free Software
#  Foundation; either version 3, or (at your option) any later version.
#
#  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along with
#  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
#
#  @author Marcello Stanisci
#  @author Florian Dold

from django.contrib.auth.decorators import login_required
from django.http import HttpResponse
from django.conf import settings
from django.shortcuts import render, redirect
from django.contrib.auth import authenticate, login, logout
from .models import BankAccount
from .history import extract_history
from .amounts import stringify_amount
from . import errors
import logging

logger = logging.getLogger(__name__)

def ignore(request):
    return HttpResponse()

def javascript_licensing(request):
    return render(request, 'javascript.html')

def home_page(request, js=None):

    logger.info("js: %s" % js)

    if js: 
        request.session['js'] = js
    else:
        js = request.session.get('js', 'no_js')

    if request.method == 'POST':
        username = request.POST['username']
        password = request.POST['password']
        user = authenticate(username=username, password=password)
        if user is None:
            request.session['wrong_login'] = True
            return redirect("/")
        if not user.is_active:
            return render(request, 'account_disabled.html', {'name': user.username,
                                                             'currency': settings.TALER_CURRENCY})
        login(request, user)
        request.session["account_no"] = user.bankaccount.account_no
        logger.info("Redirecting to /profile, js: %s" % js)
        return redirect("/profile")
    wrong = False
    if "wrong_login" in request.session:
        wrong = request.session['wrong_login']
        del request.session['wrong_login']
    if "logged_out" in request.session:
        del request.session['logged_out']
        return render(request,
                      'home_page.html',
                      {'currency': settings.TALER_CURRENCY,
                       'logged_out': True,
                       'js': js})
    if request.user.is_authenticated():
        return redirect("profile")
    return render(request,
                  'home_page.html',
                  {'currency': settings.TALER_CURRENCY,
                   'wrong': wrong,
                   'js': js})


@login_required
def profile_page(request, withdraw=False, registration=False):
    if 'registration_successful' in request.session:
        del request.session['registration_successful']
        registration = "success"
    if 'withdrawal_successful' in request.session:
        del request.session['withdrawal_successful']
        withdraw = "success"
    user_account = BankAccount.objects.get(account_no=request.session['account_no'])
    history = extract_history(user_account)
    logger.info(str(history))
    reserve_pub = request.session.get('reserve_pub')

    # Should never hit an empty session['js']
    js = request.session.get('js', 'no_js')
    logger.info("js: %s" % js)
    response = render(request,
                      'profile_page.html',
                      {'name': user_account.user.username,
                       'balance': stringify_amount(user_account.balance),
                       'currency': user_account.currency,
                       'precision': settings.NDIGITS,
                       'account_no': user_account.account_no,
                       'history': history,
                       'withdraw': withdraw,
                       'registration': registration,
                       'js': js})
    if js and withdraw == "success":
       response['X-Taler-Reserve-Pub'] = reserve_pub
       response.status_code = 202

    return response    
