#  This file is part of TALER
#  (C) 2014, 2015, 2016 INRIA
#
#  TALER is free software; you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free Software
#  Foundation; either version 3, or (at your option) any later version.
#
#  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along with
#  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
#
#  @author Marcello Stanisci

from .models import BankAccount
from .errors import internal_error_handler
from django.contrib.auth.decorators import login_required
from django.shortcuts import render, redirect
from django.conf import settings
from .user import get_bank_account_from_username
from .amounts import stringify_amount
import logging

logger = logging.getLogger(__name__)

def get_public_accounts():
    try:
        return BankAccount.objects.filter(is_public=True)
    except BankAccount.DoesNotExist:
        return []



def extract_history(bank_account):
    ret = []
    for item in bank_account.history_set.all():
        entry = {'float_amount': stringify_amount(item.amount),
                 'float_currency': item.currency,
                 'direction': 'FROM' if item.direction == 'IN' else 'TO',
                 'counterpart': item.counterpart.account_no,
                 'subject': item.subject,
                 'date': item.date.strftime("%d/%m/%y %H:%M")}  # Yes, ugly.
        logger.info(item.counterpart.user.username)
        # we don't make visible regular users' usernames
        if item.counterpart.user.username in settings.TALER_PREDEFINED_ACCOUNTS + ["Bank", "Exchange"]:
            entry['counterpart_username'] = item.counterpart.user.username
        ret.append(entry)
    return ret


def public_accounts_process(request):
    accounts = []
    for item in get_public_accounts():
        accounts.append({'account_name': item.user.username})
    sel_account_name = request.GET.get('account')
    if not sel_account_name:
        return redirect("/public-accounts?account=Tor")
    sel_account = get_bank_account_from_username(sel_account_name)
    if sel_account is False:
        return internal_error_handler(request, "User '%s' does not exist" % (sel_account_name,))
    history = extract_history(sel_account)
    return render(request,
                  'public_histories_reloaded.html',
                  {'public_accounts': accounts,
                   'currency': settings.TALER_CURRENCY,
                   'selected_account':
                       {'account_name': sel_account_name,
                        'account_no': sel_account.account_no,
                        'history': history}
                   })
