#  This file is part of TALER
#  (C) 2014, 2015, 2016 INRIA
#
#  TALER is free software; you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free Software
#  Foundation; either version 3, or (at your option) any later version.
#
#  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along with
#  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
#
#  @author Marcello Stanisci

from django.shortcuts import render
from django.http import JsonResponse
import logging

logger = logging.getLogger(__name__)

class RenderedException(Exception):
    """An exception that is rendered by us"""
    def render(self, request):
        raise Exception("RenderedException must override render()")


class BadGetParameter(RenderedException):
    def __init__(self, name=None):
        self.name = name
    def render(self, request):
        # the bank has no REST service available on GET,
        # so no JSON returned in this case
        return bad_get_parameter_handler(request, name)

# Raised either at withdrawal time (prettier rendering needed (?)),
# or when the exchange does deposits (JSON object returned is fine)
class BadAmount(RenderedException):
    def __init__(self, name=None):
        self.name = name
    def render(self, request):
        data = {"error": "bad amount"}
        if self.name:
            data["parameter"] = self.name
        return JsonResponse(data, status=400)


class BadWireDetails(RenderedException):
    def __init__(self, name=None, hint=None):
        self.name = name
        self.hint = hint
    def render(self, request):
        data = {"error": "bad wire details"}
        if self.name:
            data["parameter"] = self.name
        if self.hint:
            data["hint"] = self.hint
        return JsonResponse(data, status=400)


class BadIncomingRequest(RenderedException):
    """Thrown for an /admin/add/incoming call"""
    def __init__(self, name=None, hint=None):
        self.name = name
        self.hint = hint
    def render(self, request):
        data = {"error": "malformed JSON"}
        if self.hint:
            data["hint"] = self.hint
        return JsonResponse(data, status=400)


class NonExistentAccount(RenderedException):
    """Thrown whenever a non existent account is referenced"""
    def __init__(self, name=None, hint=None):
        self.name = name
        self.hint = hint
    def render(self, request):
        data = {"error": "non existent account"}
        if self.hint:
            logger.error(self.hint)
            data["hint"] = self.hint
        return JsonResponse(data, status=400)

class NoWireMethodMatch(RenderedException):
    def __init__(self):
        pass
    def render(self, request):
      return non_suppoerted_wire_method(request)


class BadPostValue(RenderedException):
    def __init__(self):
        pass
    def render(self, request):
      return bad_post_value_handler(request)


class WrongMethod(RenderedException):
    def __init__(self, allowed_method=None):
        self.allowed_method = allowed_method
    def render(self, request):
        data = {"error": "wrong method"}
        if self.allowed_method:
            data["parameter"] = self.allowed_method
        return JsonResponse(data)


class ExchangeUnknown(RenderedException):
    def __init__(self):
        pass
    def render(self, request):
      return exchange_unknown_handler(request)


class CurrencyMismatch(RenderedException):
    def __init__(self, allowed_method=None):
        pass
    def render(self, request):
        data = {"error": "currency mismatch"}
        return JsonResponse(data)


def no_bank_account_handler(request):
    return internal_error_handler(
        request,
        "(The bank itself has no account,"
        " please run 'taler-bank-manage --preaccounts')")


def non_supported_wire_method(request):
    return render(request, 'error.html', {'type': 'non_supported_method'}, status=400)


def non_existent_db_handler(request):
    return internal_error_handler(request, "(db does not exist)")


def internal_error_handler(request, hint=False):
    return render(request, 'error.html', {'type': 'internal_error', 'hint': hint}, status=500)


def user_not_logged_handler(request):
    return render(request, 'error.html', {'type': 'not_logged'}, status=401)


def exchange_unknown_handler(request):
    return render(request, 'error.html', {'type': 'exchange_unknown'}, status=401)


def wrong_method_handler(request, err):
    return render(request,
                  'error.html',
                  {'type': 'custom',
                   'custom_title': "Wrong method",
                   'custom_message': "Only " + err.allowed_method + " allowed"},
                  status=405)


def bad_get_parameter_handler(request, name):
    msg = "Invalid parameter {!s}in GET request"
    if name:
        msg = msg.format(name + " ")
    else:
        msg = msg.format("")

    return render(request,
                  'error.html',
                  {'type': 'custom',
                   'custom_title': "Bad request",
                   'custom_message': msg},
                  status=405)


def bad_post_value_handler(request):
    return render(request,
                  'error.html',
                  {'type': 'custom',
                   'custom_title': "Bad request",
                   'custom_message': "Bad value in POSTed data"},
                  status=400)
