#  This file is part of TALER
#  (C) 2014, 2015, 2016 INRIA
#
#  TALER is free software; you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free Software
#  Foundation; either version 3, or (at your option) any later version.
#
#  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along with
#  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
#
#  @author Marcello Stanisci

import logging
import django.db
from django.contrib.auth import authenticate, login, logout
from django.contrib.auth.models import User
from django.shortcuts import render, redirect
from django import forms
from django.conf import settings
from .models import BankAccount
from .errors import bad_get_parameter_handler
from .funds import wire_transfer_in_out

logger = logging.getLogger(__name__)

class UserReg(forms.Form):
    username = forms.CharField()
    password = forms.CharField(widget=forms.PasswordInput())


def register_process(request):
    """
    register a new user giving 100 KUDOS bonus
    """
    wrong_field = False
    not_available = False
    if request.method == 'POST':
        form = UserReg(request.POST)
        if form.is_valid():
            try:
                username = form.cleaned_data['username']
                password = form.cleaned_data['password']
                user = User.objects.create_user(username=username,
                                                password=password)
                account = BankAccount(user=user,
                                      currency=settings.TALER_CURRENCY)
                account.save()
                wire_transfer_in_out({'value': 100,
                                      'fraction': 0,
                                      'currency': settings.TALER_CURRENCY},
                                     1,
                                     account.account_no,
                                     "Joining bonus")
                request.session['account_no'] = account.account_no
                request.session['registration_successful'] = True
                user = authenticate(username=username, password=password)
                login(request, user)
                return redirect("/profile")
            except django.db.IntegrityError:
                not_available = True
        else:
            wrong_field = True
    return render(request,
                  'register.html',
                  {'wrong': wrong_field,
                   'currency': settings.TALER_CURRENCY,
                   'not_available': not_available})


def get_bank_account_from_username(username):
    try:
        user_account = User.objects.get(username=username)
        return user_account.bankaccount
    except User.DoesNotExist:
        logger.warn("user '%s' does not exist", username)
        return None


def logout_process(request):
    del request.session["account_no"]
    logout(request)
    request.session['logged_out'] = True
    return redirect("/")


def view_public_accno_process(request):
    """
    Return a HTML page showing which account number has the
    pubblic account given in parameter 'account'.
    """
    account_name = request.GET.get('account')
    if not account_name:
        return bad_get_parameter_handler(request)
    # FIXME the following function must be exception-driven
    bank_account = get_bank_account_from_username(account_name)
    if not bank_account or not bank_account.is_public:
        return render(request,
                      'error.html',
                      {'type': 'custom',
                       'custom_title': "Unknown public account",
                       'custom_message': "Wrong account name given"},
                      status=405)
    return render(request,
                  'public_account_details.html',
                  {'account_no': bank_account.account_no,
                   'account_name': account_name})
