#  This file is part of TALER
#  (C) 2014, 2015, 2016 INRIA
#
#  TALER is free software; you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free Software
#  Foundation; either version 3, or (at your option) any later version.
#
#  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along with
#  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
#
#  @author Marcello Stanisci

from django.test import TestCase, TransactionTestCase, Client
from django.core.urlresolvers import reverse
from django.contrib.auth.models import User
from .user import register_process
from .models import BankAccount
from django.conf import settings
from .funds import wire_transfer_in_out
from .errors import BadWireDetails
from django.db import connection
from .management.commands import provide_accounts
from . import history
from .amounts import floatify
import logging
import json

logger = logging.getLogger(__name__)


def create_bank_and_user():
    bank = User.objects.create_user(username='Bank')
    ba = BankAccount(user=bank, currency=settings.TALER_CURRENCY)
    ba.save()    
    user = User.objects.create_user(username='test_login', password='test_login')
    ua = BankAccount(user=user, currency=settings.TALER_CURRENCY)
    ua.save()


def reset_db():
    users = User.objects.all()
    for u in users:
        u.delete()
    cursor = connection.cursor()
    # Zero the 'account_no' autoincrement counter, because
    # tables' deletion occurring after each test* function
    # does NOT reset that counter, and this causes problem when
    # creating 'Bank' user, since it gets account_no != 1, and
    # this breaks wiretransfers involving the bank
    cursor.execute("ALTER SEQUENCE app_bankaccount_account_no_seq RESTART")

class UserTestCase(TestCase):
    """Test user registration/login/logout"""

    def setUp(self):
        create_bank_and_user()

    def tearDown(self):
        reset_db()

    def test_register(self):
        c = Client()
        response = c.post(reverse('register'),
                          {'username': 'test_register',
                           'password': 'test_register'},
                           follow=True)
        self.assertIn(("/profile", 302), response.redirect_chain)
        # this assertion tests "/profile"'s view
        self.assertEqual(200, response.status_code)
    

    def test_login(self):
        c = Client()
        response = c.post(reverse('login'),
                          {'username': 'test_login',
                           'password': 'test_login'},
                          follow=True)
        self.assertEqual(200, response.status_code)
        response = c.get(reverse('logout'))
        self.assertEqual(302, response.status_code)


class FundsTestCase(TestCase):
    def setUp(self):
        logging.disable(logging.CRITICAL)
        create_bank_and_user()

    def tearDown(self):
        logging.disable(logging.NOTSET)
        reset_db()

    def test_wiretransfer(self):
        wire_transfer_in_out({'value': 100,
                              'fraction': 0,
                              'currency': settings.TALER_CURRENCY},
                             1,
                             2,
                             "Joining bonus")
    def test_addincoming(self):
        c = Client()
        data = {'amount': {'value': 1, 'fraction': 0, 'currency': settings.TALER_CURRENCY},
                'debit_account': 1,
                'credit_account': 2,
                'wtid': 'TEST'}
        response = c.post(reverse('add-incoming'), json.dumps(data), content_type="application/json")
        self.assertEqual(200, response.status_code)

class HistoryTestCase(TestCase):
    def setUp(self):
        logging.disable(logging.CRITICAL)
        provide_accounts.basic_accounts()
        provide_accounts.demo_accounts()

    def tearDown(self):
        logging.disable(logging.NOTSET)

    # Extract history from public accounts
    def test_public_history(self):
        c = Client() 
        response = c.get(reverse('public-history'), follow=True)
        self.assertEqual(200, response.status_code)
        response = c.get(reverse('public-history'), {'account': 'GNUnet'})
        self.assertEqual(200, response.status_code)


class CaptchaTestCase(TestCase):
    def test_pin_tan_question(self): 
        c = Client()
        User.objects.create_user(username='test_pintan', password='test_pintan')
        c.login(username='test_pintan', password='test_pintan')
        wiredetails = {'test':
                          {'type': 'test',
                           'account_number': 0,
                           'bank_uri': 'http://test',
                           'name': 'test',
                           'salt': 'test',
                           'sig': 'test',
                           'uri': 'test'}}   
        data = {'amount_value': 1,
                'amount_fraction': 0,
                'amount_currency': settings.TALER_CURRENCY,
                'exchange': 'http://test',
                'reserve_pub': 'TEST',
                'wire_details': json.dumps(wiredetails)}
        response = c.get(reverse('pin_tan'), data)
        self.assertEqual(200, response.status_code)


class FloatifyTestCase(TestCase):
    def test_floatify(self):
        data = {'value': 3, 'fraction': 700, 'currency': settings.TALER_CURRENCY}
        result = 3 + (float(700) / float(1000000))
        ret = floatify(data)
        self.assertEqual(result, ret)
