#  This file is part of TALER
#  (C) 2014, 2015, 2016 INRIA
#
#  TALER is free software; you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free Software
#  Foundation; either version 3, or (at your option) any later version.
#
#  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along with
#  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
#
#  @author Marcello Stanisci

from . import schemas
from .errors import WrongMethod, BadWireDetails, CurrencyMismatch, NonExistentAccount
from . import amounts
from .models import BankAccount, History
from django.views.decorators.csrf import csrf_exempt
from django.http import JsonResponse
from django.shortcuts import render, redirect
from django.http import HttpResponseServerError
from django.contrib.auth.decorators import login_required
from urllib.parse import urlparse, urljoin
from django.conf import settings
import requests
import time
import json
import logging

logger = logging.getLogger(__name__)



def check_exchange_account_no(account_no):
    try:
        BankAccount.objects.get(account_no=account_no)
    except BankAccount.DoesNotExist:
        raise errors.ExchangeUnknown()


class Reserve:
    def __init__(self, amount, exchange, exchange_account, reserve_pub, sender_account_details, wiredetails_counter=0):
        schemas.validate_amount(amount)
        self.amount = amount
        self.exchange = exchange
        self.exchange_account = exchange_account
        self.reserve_pub = reserve_pub
        self.sender_account_details = sender_account_details
        self.transfer_details = wiredetails_counter

# The CSRF exempt is due to the fact Django looks for an anti-CSRF token
# In any POST it gets. Since the following function is meant to serve mints,
# And mints don't send any such token, it is necessary to disable it.
# Those tokens are normally hidden fields in Django-generated HTML forms.


@csrf_exempt
def add_incoming(request):
    logger.info("handling /admin/add/incoming")
    if request.method != 'POST':
        raise WrongMethod('GET')
    data = json.loads(request.body.decode('utf-8'))
    logger.info("valid uploaded data")
    schemas.validate_incoming_request(data)
    logger.info("add_incoming for debit account %s and credit account %s, WTID %s",
                data['debit_account'],
		data['credit_account'],
		data['wtid'])

    wire_transfer_in_out(data['amount'],
                         data['debit_account'],
                         data['credit_account'],
                         data['wtid'])
    return JsonResponse({'outcome': 'ok'}, status=200)


@login_required
def withdraw_process(request):
    return render(request, 'withdraw.html', {'account_no': request.session["account_no"]})


def create_reserve_at_exchange(request, success_url, reserve_set):
    if not isinstance(reserve_set, Reserve):
        return HttpResponseServerError
    o = urlparse(reserve_set.exchange)
    if o.scheme == '' or o.netloc == '' or o.path != '/':
        return JsonResponse({'error': 'bad exchange base url',
                             'given_url': reserve_set.mint}, status=400)
    amount = {'value': reserve_set.amount['value'],
              'fraction': reserve_set.amount['fraction'],
              'currency': reserve_set.amount['currency']}
    check_exchange_account_no(reserve_set.exchange_account)
    json_body = {'reserve_pub': reserve_set.reserve_pub,
                 'execution_date': "/Date(" + str(int(time.time())) + ")/",
                 'sender_account_details': reserve_set.sender_account_details,
                 'transfer_details': {'uuid': reserve_set.transfer_details},
                 'amount': amount}
    res = requests.post(urljoin(reserve_set.exchange, '/admin/add/incoming'), json=json_body)
    logger.info("making request to /admin/add/incoming with %s", json_body)
    if res.status_code != 200:
        return JsonResponse({'error': "internal error",
                             'hint': "wire transfer failed (bad response)",
                             'status': res.status_code,
                             'body': res.text}, status=500)
    wire_transfer_in_out(amount,
                         request.session['account_no'],
                         reserve_set.exchange_account,
                         reserve_set.reserve_pub)
    request.session['withdrawal_successful'] = True
    return redirect('/profile')



def wire_transfer_in_out(amount,
                         debit,
                         credit,
                         wtid):
    if debit == credit:
        raise BadWireDetails(hint="debit and credit accounts are the same")
    try:
        debit_account = BankAccount.objects.get(account_no=debit)
    except BankAccount.DoesNotExist: 
        raise NonExistentAccount(hint="account " + str(debit) + " unknown")
    try:
        credit_account = BankAccount.objects.get(account_no=credit)
    except BankAccount.DoesNotExist:
        raise NonExistentAccount(hint="account " + str(credit) + " unknown")
    wire_transfer(amount,
                  debit_account,
                  "OUT",
                  credit_account,
                  wtid)
    wire_transfer(amount,
                  credit_account,
                  "IN",
                  debit_account,
                  wtid)


# transfer funds from/to 'account' (used as a subroutine)
def wire_transfer(amount,
                  account,
                  direction,
                  counterpart,
                  wtid="not given"):
    if account.currency != amount['currency']:
        logger.error("currency %s and currency %s mismatch", account.currency, amount['currency'])
        raise CurrencyMismatch()
    float_amount = amounts.floatify(amount)
    if "IN" == direction:
        account.balance += float_amount
    else:
        account.balance -= float_amount
    account.save()
    history_item = History(amount=float_amount,
                           currency=amount['currency'],
                           direction=direction,
                           counterpart=counterpart,
                           subject=wtid,
                           account=account)
    history_item.save()
