#  This file is part of TALER
#  (C) 2014, 2015, 2016 INRIA
#
#  TALER is free software; you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free Software
#  Foundation; either version 3, or (at your option) any later version.
#
#  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along with
#  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
#
#  @author Marcello Stanisci

import hashlib
import json
import logging
from urllib.parse import urlunparse, unquote, urlparse, ParseResult
from simplemathcaptcha.fields import MathCaptchaField, MathCaptchaWidget
from django.http import HttpResponse
from django.shortcuts import render
from django.conf import settings
from django import forms
from django.contrib.auth.decorators import login_required
from django.views.decorators.http import require_POST, require_GET
from .funds import Reserve, create_reserve_at_exchange
from . import schemas
from . import amounts
from . import errors
from .user import get_bank_account_from_username

logger = logging.getLogger(__name__)


class Pin(forms.Form):
    pin = MathCaptchaField(
        widget=MathCaptchaWidget(
            question_tmpl="<div lang=\"en\">What is %(num1)i %(operator)s %(num2)i ?</div>"))


@require_GET
@login_required
def pin_tan_question(request):
    for param in ["amount_value",
                  "amount_fraction",
                  "amount_currency",
                  "exchange",
                  "reserve_pub",
                  "wire_details"]:
        if param not in request.GET:
            raise errors.BadGetParameter(param)
    try:
        amount = {'value': int(request.GET['amount_value']),
                  'fraction': int(request.GET['amount_fraction']),
                  'currency': request.GET['amount_currency']}
    except ValueError:
        raise errors.BadGetParameter()
    wiredetails = json.loads(unquote(request.GET['wire_details']))
    schemas.validate_wiredetails(wiredetails)
    request.session['account_number'] = wiredetails['test']['account_number']
    request.session['wire_details'] = wiredetails['test']
    schemas.validate_amount(amount)
    request.session['amount'] = amount
    request.session['exchange'] = request.GET['exchange']
    request.session['reserve_pub'] = request.GET['reserve_pub']
    return render(request, 'pin_tan.html', {'form': Pin(auto_id=False),
                                            'amount': amounts.floatify(amount),
                                            'currency': settings.TALER_CURRENCY,
                                            'exchange': request.GET['exchange']})


@require_POST
@login_required
def pin_tan_verify(request):
    try:
        given = request.POST['pin_0']
        hashed_result = request.POST['pin_1']
    except Exception:  # FIXME narrow the Exception type
        raise errors.BadPostValue()
    hasher = hashlib.new("sha1")
    hasher.update(settings.SECRET_KEY.encode('utf-8'))
    hasher.update(given.encode('utf-8'))
    hashed_attempt = hasher.hexdigest()
    if hashed_attempt == hashed_result:
        for param in ["amount", "exchange", "reserve_pub"]:
            if param not in request.session:
                return HttpResponse("Not a withdraw session", status=400)
        settings.TALER_WIREDETAILS_COUNTER += 1
        sender_wiredetails = {'type': 'TEST',
                              'bank_uri': urlunparse(ParseResult(scheme=request.scheme, netloc=request.META['HTTP_HOST'],
                                                     path='/', params='', query='', fragment='')),
                              'account_number': request.session['account_no']}
        reserve = Reserve(request.session['amount'],
                          request.session['exchange'],
                          request.session['account_number'],
                          request.session['reserve_pub'],
                          sender_wiredetails,
                          wiredetails_counter=settings.TALER_WIREDETAILS_COUNTER)
        success_url = urlunparse([request.scheme,
                                 request.META['HTTP_HOST'],
                                 "/success.html", '', '', ''])
        return create_reserve_at_exchange(request, success_url, reserve)

    else:
        return render(request, 'error.html', {'type': "wrong_pin"}, status=400)
