/*
 * This file is part of LibEuFin.
 * Copyright (C) 2019 Stanisci and Dold.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

package tech.libeufin.sandbox

import com.fasterxml.jackson.annotation.JsonProperty
import tech.libeufin.util.PaymentInfo

data class WithdrawalRequest(
    /**
     * Note: the currency is redundant, because at each point during
     * the execution the Demobank should have a handle of the currency.
     */
    val amount: String // $CURRENCY:X.Y
)

data class Demobank(
    val currency: String,
    val name: String,
    val userDebtLimit: Int,
    val bankDebtLimit: Int,
    val allowRegistrations: Boolean
)
/**
 * Used to show the list of Ebics hosts that exist
 * in the system.
 */
data class EbicsHostsResponse(
    val ebicsHosts: List<String>
)

data class EbicsHostCreateRequest(
    val hostID: String,
    val ebicsVersion: String
)

/**
 * List type that show all the payments existing in the system.
 */
data class AccountTransactions(
    val payments: MutableList<PaymentInfo> = mutableListOf()
)

/**
 * Used to create AND show one Ebics subscriber.
 */
data class EbicsSubscriberInfo(
    val hostID: String,
    val partnerID: String,
    val userID: String,
    val systemID: String? = null,
    val demobankAccountLabel: String
)

data class AdminGetSubscribers(
    var subscribers: MutableList<EbicsSubscriberInfo> = mutableListOf()
)

/**
 * The following definition is obsolete because it
 * doesn't allow to specify a demobank that will host
 * the Ebics subscriber.  */
data class EbicsSubscriberObsoleteApi(
    val hostID: String,
    val partnerID: String,
    val userID: String,
    val systemID: String? = null
)

/**
 * Allows the admin to associate a new bank account
 * to a EBICS subscriber.
 */
data class EbicsBankAccountRequest(
    val subscriber: EbicsSubscriberObsoleteApi,
    val iban: String,
    val bic: String,
    val name: String,
    /**
     * This value labels the bank account to be created
     * AND its owner.  The 'owner' is a bank's customer
     * whose username equals this label AND has the rights
     * over such bank accounts.
     */
    val label: String
)

data class CustomerRegistration(
    val username: String,
    val password: String,
    val isPublic: Boolean = false,
    // When missing, it's autogenerated.
    val iban: String?,
    // When missing, stays null in the DB.
    val name: String?
)

// Could be used as a general bank account info container.
data class PublicAccountInfo(
    val balance: String,
    val iban: String,
    // Name / Label of the bank account _and_ of the
    // Sandbox username that owns it.
    val accountLabel: String
    // more ..?
)

data class CamtParams(
    // name/label of the bank account to query.
    val bankaccount: String,
    val type: Int,
    // need range parameter
)

data class TalerWithdrawalStatus(
    val selection_done: Boolean,
    val transfer_done: Boolean,
    val amount: String,
    val wire_types: List<String> = listOf("iban"),
    val suggested_exchange: String? = null,
    val sender_wire: String? = null,
    val aborted: Boolean,
    // Not needed with CLI wallets.
    val confirm_transfer_url: String?
)

data class TalerWithdrawalSelection(
    val reserve_pub: String,
    val selected_exchange: String?
)

data class NewTransactionReq(
    /**
     * This Payto address must contain the wire transfer
     * subject among its query parameters -- 'message' parameter.
     */
    val paytoUri: String,
    // $currency:X.Y format
    val amount: String?
)

data class SandboxConfig(
    val currency: String,
    val version: String,
    val name: String
)