/*
   This file is part of Challenger
   Copyright (C) 2023 Taler Systems SA

   Challenger is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   Challenger is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   Challenger; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file challengerdb/pg_validate_solve_pin.c
 * @brief Implementation of the validate_solve_pin function for Postgres
 * @author Christian Grothoff
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_validate_solve_pin.h"
#include "pg_helper.h"


enum GNUNET_DB_QueryStatus
CH_PG_validate_solve_pin (void *cls,
                          const struct CHALLENGER_ValidationNonceP *nonce,
                          uint32_t new_pin,
                          bool *solved)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_auto_from_type (nonce),
    GNUNET_PQ_query_param_uint32 (&new_pin),
    GNUNET_PQ_query_param_end
  };
  struct GNUNET_PQ_ResultSpec rs[] = {
    GNUNET_PQ_result_spec_bool ("solved",
                                solved),
    GNUNET_PQ_result_spec_end
  };

  /* We set all remaining attempts to zero to prevent
     user from changing the address after already having
     succeeded with the process. */
  PREPARE (pg,
           "validate_solve_pin",
           "UPDATE validations SET"
           "  auth_attempts_left=CASE"
           "    WHEN last_pin = $2"
           "    THEN 0"
           "    ELSE auth_attempts_left - 1"
           "  END"
           " ,address_attempts_left=CASE"
           "    WHEN last_pin = $2"
           "    THEN 0"
           "    ELSE address_attempts_left"
           "  END"
           " ,pin_transmissions_left=CASE"
           "    WHEN last_pin = $2"
           "    THEN 0"
           "    ELSE pin_transmissions_left"
           "  END"
           " WHERE nonce=$1"
           "   AND auth_attempts_left > 0"
           " RETURNING"
           "   (last_pin = $2) AS solved;");
  return GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                   "validate_solve_pin",
                                                   params,
                                                   rs);
}
