/*
   This file is part of Challenger
   Copyright (C) 2023 Taler Systems SA

   Challenger is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   Challenger is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   Challenger; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file challengerdb/pg_challenge_set_address_and_pin.c
 * @brief Implementation of the challenge_set_address_and_pin function for Postgres
 * @author Christian Grothoff
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_challenge_set_address_and_pin.h"
#include "pg_helper.h"


enum GNUNET_DB_QueryStatus
CH_PG_challenge_set_address_and_pin (
  void *cls,
  const struct CHALLENGER_ValidationNonceP *nonce,
  const json_t *address,
  struct GNUNET_TIME_Relative validation_duration,
  uint32_t *tan,
  struct GNUNET_TIME_Absolute *last_tx_time,
  uint32_t *auth_attempts_left,
  bool *pin_transmit)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_TIME_Absolute now
    = GNUNET_TIME_absolute_get ();
  struct GNUNET_TIME_Absolute next_tx_time
    = GNUNET_TIME_absolute_subtract (now,
                                     validation_duration);
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_auto_from_type (nonce),
    TALER_PQ_query_param_json (address),
    GNUNET_PQ_query_param_absolute_time (&next_tx_time),
    GNUNET_PQ_query_param_absolute_time (&now),
    GNUNET_PQ_query_param_uint32 (tan),
    GNUNET_PQ_query_param_end
  };
  struct GNUNET_PQ_ResultSpec rs[] = {
    GNUNET_PQ_result_spec_absolute_time ("last_tx_time",
                                         last_tx_time),
    GNUNET_PQ_result_spec_uint32 ("last_pin",
                                  tan),
    GNUNET_PQ_result_spec_bool ("pin_transmit",
                                pin_transmit),
    GNUNET_PQ_result_spec_uint32 ("auth_attempts_left",
                                  auth_attempts_left),
    GNUNET_PQ_result_spec_end
  };

  PREPARE (pg,
           "challenge_set_address_and_pin",
           "WITH decisions AS ("
           "  SELECT "
           "    ( (address IS NULL) OR"
           "      (address != $2) ) AND"
           "    (address_attempts_left > 0)"
           "      AS addr_changed"
           "   ,( (pin_transmissions_left > 0) OR"
           "      (address_attempts_left > 0) ) AND"
           "    ( (address IS NULL) OR"
           "      (address != $2) OR"
           "      (last_tx_time < $3) ) AS send_pin"
           "    FROM validations"
           "    WHERE nonce=$1"
           "),"
           "result AS ("
           "UPDATE validations SET"
           "  address_attempts_left=CASE"
           "    WHEN (SELECT addr_changed FROM decisions)"
           "    THEN address_attempts_left - 1 "
           "    ELSE address_attempts_left "
           "  END"
           " ,last_pin = CASE "
           "    WHEN (SELECT addr_changed FROM decisions)"
           "    THEN $5"
           "    ELSE last_pin"
           "  END"
           " ,pin_transmissions_left=CASE"
           "    WHEN (SELECT addr_changed FROM decisions)"
           "    THEN 3 "
           "    ELSE CASE"
           "      WHEN (SELECT send_pin FROM decisions)"
           "      THEN pin_transmissions_left - 1"
           "      ELSE pin_transmissions_left"
           "    END"
           "  END"
           " ,auth_attempts_left=CASE"
           "    WHEN (SELECT addr_changed FROM decisions)"
           "    THEN 3 "
           "    ELSE auth_attempts_left"
           "  END"
           " ,last_tx_time=CASE"
           "    WHEN (SELECT send_pin FROM decisions)"
           "    THEN $4"
           "    ELSE last_tx_time"
           "  END"
           " ,address=CASE"
           "    WHEN (SELECT addr_changed FROM decisions)"
           "    THEN $2"
           "    ELSE address"
           "  END"
           " WHERE nonce=$1"
           " RETURNING"
           "   last_tx_time"
           "  ,last_pin"
           "  ,auth_attempts_left"
           ")"
           " SELECT"
           "  last_tx_time"
           " ,decisions.send_pin AS pin_transmit"
           " ,last_pin"
           " ,auth_attempts_left"
           " FROM result"
           " FULL OUTER JOIN decisions ON (TRUE);");
  return GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                   "challenge_set_address_and_pin",
                                                   params,
                                                   rs);
}
