/*
  This file is part of Challenger
  Copyright (C) 2023 Taler Systems SA

  Challenger is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  Challenger is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more details.

  You should have received a copy of the GNU Affero General Public License along with
  Challenger; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file challenger-httpd_common.c
 * @brief common helper functions
 * @author Christian Grothoff
 */
#include "platform.h"
#include "challenger-httpd_common.h"


#define RFC_8959_PREFIX "secret-token:"

const char *
CH_get_client_secret (struct MHD_Connection *connection)
{
  const char *bearer = "Bearer ";
  const char *auth;
  const char *tok;

  auth = MHD_lookup_connection_value (connection,
                                      MHD_HEADER_KIND,
                                      MHD_HTTP_HEADER_AUTHORIZATION);
  if (NULL == auth)
    return NULL;
  if (0 != strncmp (auth,
                    bearer,
                    strlen (bearer)))
  {
    return NULL;
  }
  tok = auth + strlen (bearer);
  while (' ' == *tok)
    tok++;
  if (0 != strncasecmp (tok,
                        RFC_8959_PREFIX,
                        strlen (RFC_8959_PREFIX)))
  {
    return NULL;
  }
  return tok;
}


char *
CH_compute_code (const struct CHALLENGER_ValidationNonceP *nonce,
                 const char *client_secret,
                 const char *client_scope,
                 const json_t *address,
                 const char *client_redirect_uri)
{
  char *code;
  char *ns;
  char *hs;
  struct GNUNET_ShortHashCode h;
  char *astr;

  astr = json_dumps (address,
                     JSON_COMPACT);
  GNUNET_assert (GNUNET_YES ==
                 GNUNET_CRYPTO_kdf (&h,
                                    sizeof (h),
                                    nonce,
                                    sizeof (nonce),
                                    client_secret,
                                    strlen (client_secret),
                                    astr,
                                    strlen (astr),
                                    client_redirect_uri,
                                    strlen (client_redirect_uri),
                                    client_scope,
                                    NULL != client_scope
                                    ? strlen (client_scope)
                                    : 0,
                                    NULL,
                                    0));
  free (astr);
  ns = GNUNET_STRINGS_data_to_string_alloc (nonce,
                                            sizeof (*nonce));
  hs = GNUNET_STRINGS_data_to_string_alloc (&h,
                                            sizeof (h));
  GNUNET_asprintf (&code,
                   "%s-%s",
                   ns,
                   hs);
  GNUNET_free (ns);
  GNUNET_free (hs);
  return code;
}


enum GNUNET_GenericReturnValue
CH_code_to_nonce (const char *code,
                  struct CHALLENGER_ValidationNonceP *nonce)
{
  const char *dash = strchr (code, '-');

  if (NULL == dash)
  {
    GNUNET_break_op (0);
    return GNUNET_SYSERR;
  }
  if (GNUNET_OK !=
      GNUNET_STRINGS_string_to_data (code,
                                     (size_t) (dash - code),
                                     nonce,
                                     sizeof (*nonce)))
  {
    GNUNET_break_op (0);
    return GNUNET_SYSERR;
  }
  return GNUNET_OK;
}


MHD_RESULT
TALER_MHD_reply_with_oauth_error (
  struct MHD_Connection *connection,
  unsigned int http_status,
  const char *oauth_error,
  enum TALER_ErrorCode ec,
  const char *detail)
{
  struct MHD_Response *resp;
  MHD_RESULT mret;

  resp = TALER_MHD_make_json (
    GNUNET_JSON_PACK (
      TALER_MHD_PACK_EC (ec),
      GNUNET_JSON_pack_string ("error",
                               oauth_error),
      GNUNET_JSON_pack_allow_null (
        GNUNET_JSON_pack_string ("detail",
                                 detail))));
  if (MHD_HTTP_UNAUTHORIZED == http_status)
    GNUNET_break (MHD_YES ==
                  MHD_add_response_header (
                    resp,
                    "WWW-Authenticate",
                    "Bearer, error=\"invalid_token\""));
  mret = MHD_queue_response (connection,
                             http_status,
                             resp);
  MHD_destroy_response (resp);
  return mret;
}
