/*
  This file is part of Challenger
  Copyright (C) 2023, 2024 Taler Systems SA

  Challenger is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  Challenger is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more details.

  You should have received a copy of the GNU Affero General Public License along with
  Challenger; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file challenger-httpd_authorize.c
 * @brief functions to handle incoming requests for authorizations
 * @author Christian Grothoff
 */
#include "platform.h"
#include "challenger-httpd.h"
#include <gnunet/gnunet_util_lib.h>
#include <taler/taler_templating_lib.h>
#include "challenger-httpd_authorize.h"
#include "challenger-httpd_common.h"


/**
 * Generate error reply in the format requested by
 * the client.
 *
 * @param hc our context
 * @param template error template to use
 * @param http_status HTTP status to return
 * @param ec error code to return
 * @param hint human-readable hint to give
 */
static MHD_RESULT
reply_error (struct CH_HandlerContext *hc,
             const char *template,
             unsigned int http_status,
             enum TALER_ErrorCode ec,
             const char *hint)
{
  if (0 == CH_get_output_type (hc->connection))
    return TALER_TEMPLATING_reply_error (
      hc->connection,
      template,
      http_status,
      ec,
      hint);
  return TALER_MHD_reply_with_error (
    hc->connection,
    http_status,
    ec,
    hint);
}


MHD_RESULT
CH_handler_authorize (struct CH_HandlerContext *hc,
                      const char *upload_data,
                      size_t *upload_data_size)
{
  const char *response_type;
  unsigned long long client_id;
  const char *redirect_uri;
  const char *state;
  const char *scope;
  struct CHALLENGER_ValidationNonceP nonce;

  (void) upload_data;
  (void) upload_data_size;
  if (GNUNET_OK !=
      GNUNET_STRINGS_string_to_data (hc->path,
                                     strlen (hc->path),
                                     &nonce,
                                     sizeof (nonce)))
  {
    GNUNET_break_op (0);
    return reply_error (
      hc,
      "invalid-request",
      MHD_HTTP_NOT_FOUND,
      TALER_EC_GENERIC_PARAMETER_MISSING,
      hc->path);
  }
  response_type
    = MHD_lookup_connection_value (hc->connection,
                                   MHD_GET_ARGUMENT_KIND,
                                   "response_type");
  if (NULL == response_type)
  {
    GNUNET_break_op (0);
    return reply_error (
      hc,
      "invalid-request",
      MHD_HTTP_BAD_REQUEST,
      TALER_EC_GENERIC_PARAMETER_MISSING,
      "response_type");
  }
  if (0 != strcmp (response_type,
                   "code"))
  {
    GNUNET_break_op (0);
    return reply_error (
      hc,
      "invalid-request",
      MHD_HTTP_BAD_REQUEST,
      TALER_EC_GENERIC_PARAMETER_MALFORMED,
      "response_type (must be 'code')");
  }

  {
    const char *client_id_str;
    char dummy;

    client_id_str
      = MHD_lookup_connection_value (hc->connection,
                                     MHD_GET_ARGUMENT_KIND,
                                     "client_id");
    if (NULL == client_id_str)
    {
      GNUNET_break_op (0);
      return reply_error (
        hc,
        "invalid_request",
        MHD_HTTP_BAD_REQUEST,
        TALER_EC_GENERIC_PARAMETER_MISSING,
        "client_id");
    }
    if (1 != sscanf (client_id_str,
                     "%llu%c",
                     &client_id,
                     &dummy))
    {
      GNUNET_break_op (0);
      return reply_error (
        hc,
        "invalid-request",
        MHD_HTTP_BAD_REQUEST,
        TALER_EC_GENERIC_PARAMETER_MALFORMED,
        "client_id");
    }
  }
  redirect_uri
    = MHD_lookup_connection_value (hc->connection,
                                   MHD_GET_ARGUMENT_KIND,
                                   "redirect_uri");
  /* Note: this is a somewhat arbitrary restriction, as the rest of
     this code would support other schemas just fine.  However, #7838
     (RFC 7636) should be implemented before lifting this restriction,
     as otherwise the service might be accidentally used with public
     clients which would then be insecure. */
  if ( (NULL != redirect_uri) &&
       (0 != strncmp (redirect_uri,
                      "http://",
                      strlen ("http://"))) &&
       (0 != strncmp (redirect_uri,
                      "https://",
                      strlen ("https://"))) )
  {
    GNUNET_break_op (0);
    return reply_error (
      hc,
      "invalid-request",
      MHD_HTTP_BAD_REQUEST,
      TALER_EC_GENERIC_PARAMETER_MALFORMED,
      "redirect_uri (has to start with 'http://' or 'https://')");
  }
  state
    = MHD_lookup_connection_value (hc->connection,
                                   MHD_GET_ARGUMENT_KIND,
                                   "state");
  if (NULL == state)
    state = "";

  scope
    = MHD_lookup_connection_value (hc->connection,
                                   MHD_GET_ARGUMENT_KIND,
                                   "scope");
  {
    json_t *last_address;
    uint32_t address_attempts_left;
    uint32_t pin_transmissions_left;
    uint32_t auth_attempts_left;
    enum GNUNET_DB_QueryStatus qs;

    /* authorize_start will return 0 if a 'redirect_uri' was
       configured for the client and this one differs. */
    qs = CH_db->authorize_start (CH_db->cls,
                                 &nonce,
                                 client_id,
                                 scope,
                                 state,
                                 redirect_uri,
                                 &last_address,
                                 &address_attempts_left,
                                 &pin_transmissions_left,
                                 &auth_attempts_left);
    switch (qs)
    {
    case GNUNET_DB_STATUS_HARD_ERROR:
      GNUNET_break (0);
      return reply_error (
        hc,
        "internal-error",
        MHD_HTTP_INTERNAL_SERVER_ERROR,
        TALER_EC_GENERIC_DB_STORE_FAILED,
        "authorize_start");
    case GNUNET_DB_STATUS_SOFT_ERROR:
      GNUNET_break (0);
      return MHD_NO;
    case GNUNET_DB_STATUS_SUCCESS_NO_RESULTS:
      GNUNET_break_op (0);
      GNUNET_log (GNUNET_ERROR_TYPE_INFO,
                  "Failed to find authorization process of client %llu for nonce `%s'\n",
                  client_id,
                  hc->path);
      return reply_error (
        hc,
        "validation-unknown",
        MHD_HTTP_NOT_FOUND,
        TALER_EC_CHALLENGER_GENERIC_VALIDATION_UNKNOWN,
        NULL);
    case GNUNET_DB_STATUS_SUCCESS_ONE_RESULT:
      break;
    }
    if ( (0 == address_attempts_left) &&
         (0 == pin_transmissions_left) &&
         (0 == auth_attempts_left) )
    {
      GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                  "Refusing authorization: zero attempts left\n");
      json_decref (last_address);
      return TALER_MHD_redirect_with_oauth_status (
        hc->connection,
        redirect_uri,
        state,
        "unauthorized_client",
        "client exceeded authorization attempts limit (too many addresses)",
        NULL);
    }
    {
      enum GNUNET_GenericReturnValue ret;
      json_t *args;
      char *form;
      struct MHD_Response *resp;
      unsigned int http_status;
      MHD_RESULT res;

      GNUNET_asprintf (&form,
                       "enter-%s-form",
                       CH_address_type);
      args = GNUNET_JSON_PACK (
        GNUNET_JSON_pack_allow_null (
          GNUNET_JSON_pack_object_incref ("restrictions",
                                          CH_restrictions)),
        GNUNET_JSON_pack_bool ("fix_address",
                               0 == address_attempts_left),
        GNUNET_JSON_pack_string ("nonce",
                                 hc->path),
        GNUNET_JSON_pack_allow_null (
          GNUNET_JSON_pack_object_steal ("last_address",
                                         last_address)),
        GNUNET_JSON_pack_uint64 ("changes_left",
                                 address_attempts_left)
        );
      http_status = MHD_HTTP_OK;
      if (0 == CH_get_output_type (hc->connection))
      {
        ret = TALER_TEMPLATING_build (
          hc->connection,
          &http_status,
          form,
          NULL,
          NULL,
          args,
          &resp);
        GNUNET_free (form);
        json_decref (args);
        if (GNUNET_SYSERR == ret)
        {
          GNUNET_break (0);
          return MHD_NO;
        }
        GNUNET_break (GNUNET_OK == ret);
      }
      else
      {
        GNUNET_break (0 ==
                      json_object_del (args,
                                       "nonce"));
        resp = TALER_MHD_make_json_steal (args);
      }
      GNUNET_break (MHD_YES ==
                    MHD_add_response_header (resp,
                                             MHD_HTTP_HEADER_CACHE_CONTROL,
                                             "no-store,no-cache"));
      res = MHD_queue_response (hc->connection,
                                http_status,
                                resp);
      MHD_destroy_response (resp);
      return res;
    }
  }
}
