;;; latin1-disp.el --- display tables for other ISO 8859 on Latin-1 terminals -*-coding: iso-2022-7bit;-*-

;; Copyright (C) 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011
;;   Free Software Foundation, Inc.

;; Author: Dave Love <fx@gnu.org>
;; Keywords: i18n

;; This file is part of GNU Emacs.

;; GNU Emacs is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs.  If not, see <http://www.gnu.org/licenses/>.

;;; Commentary:

;; This package sets up display of ISO 8859-n for n>1 by substituting
;; Latin-1 characters and sequences of them for characters which can't
;; be displayed, either because we're on a tty or because we don't
;; have the relevant window system fonts available.  For instance,
;; Latin-9 is very similar to Latin-1, so we can display most Latin-9
;; characters using the Latin-1 characters at the same code point and
;; fall back on more-or-less mnemonic ASCII sequences for the rest.

;; For the Latin charsets the ASCII sequences are mostly consistent
;; with the Quail prefix input sequences.  Latin-4 uses the Quail
;; postfix sequences since a prefix method isn't defined for Latin-4.

;; [A different approach is taken in the DOS display tables in
;; term/internal.el, and the relevant ASCII sequences from there are
;; available as an alternative; see `latin1-display-mnemonic'.  Only
;; these sequences are used for Arabic, Cyrillic, Greek and Hebrew.]

;; If you don't even have Latin-1, see iso-ascii.el and use the
;; complete tables from internal.el.  The ASCII sequences used here
;; are mostly in the same style as iso-ascii.

;;; Code:

;; Ensure `standard-display-table' is set up:
(require 'disp-table)

(defconst latin1-display-sets '(latin-2 latin-3 latin-4 latin-5 latin-8
		                latin-9 arabic cyrillic greek hebrew)
  "The ISO8859 character sets with defined Latin-1 display sequences.
These are the nicknames for the sets and correspond to Emacs language
environments.")

(defgroup latin1-display ()
  "Set up display tables for ISO8859 characters using Latin-1."
  :version "21.1"
  :link '(emacs-commentary-link "latin1-disp")
  :group 'i18n)

(defcustom latin1-display-format "{%s}"
  "A format string used to display the ASCII sequences.
The default encloses the sequence in braces, but you could just use
\"%s\" to avoid the braces, maybe with a non-default value of
`latin1-display-face'."
  :group 'latin1-display
  :type 'string)

;;;###autoload
(defcustom latin1-display nil
  "Set up Latin-1/ASCII display for ISO8859 character sets.
This is done for each character set in the list `latin1-display-sets',
if no font is available to display it.  Characters are displayed using
the corresponding Latin-1 characters where they match.  Otherwise
ASCII sequences are used, mostly following the Latin prefix input
methods.  Some different ASCII sequences are used if
`latin1-display-mnemonic' is non-nil.

This option also treats some characters in the `mule-unicode-...'
charsets if you don't have a Unicode font with which to display them.

Setting this variable directly does not take effect;
use either \\[customize] or the function `latin1-display'."
  :group 'latin1-display
  :type 'boolean
  :require 'latin1-disp
  :initialize 'custom-initialize-default
  :set (lambda (symbol value)
	 (if value
	     (apply #'latin1-display latin1-display-sets)
	   (latin1-display))))

;;;###autoload
(defun latin1-display (&rest sets)
  "Set up Latin-1/ASCII display for the arguments character SETS.
See option `latin1-display' for the method.  The members of the list
must be in `latin1-display-sets'.  With no arguments, reset the
display for all of `latin1-display-sets'. See also
`latin1-display-setup'."
  (if sets
      (progn
	(mapc #'latin1-display-setup sets)
	(unless (char-displayable-p #x101) ; a with macron
	  ;; Extra stuff for windows-1252, in particular.
	  (mapc
	   (lambda (l)
	     (apply 'latin1-display-char l))
	   '((?\$,1rz(B ",") ;; SINGLE LOW-9 QUOTATION MARK
	     (?\$,1r~(B ",,")	;; DOUBLE LOW-9 QUOTATION MARK
	     (?\$,1s&(B "...") ;; HORIZONTAL ELLIPSIS
	     (?\$,1s0(B "o/oo") ;; PER MILLE SIGN
	     (?\$,1s9(B "<") ;; SINGLE LEFT-POINTING ANGLE QUOTATION MARK
	     (?\$,1r|(B "``")	;; LEFT DOUBLE QUOTATION MARK
	     (?\$,1r}(B "''")	;; RIGHT DOUBLE QUOTATION MARK
	     (?\$,1rs(B "-") ;; EN DASH
	     (?\$,1rt(B "--")	;; EM DASH
	     (?\$,1ub(B "TM")	;; TRADE MARK SIGN
	     (?\$,1s:(B ">") ;; SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
	     (?$,1s"(B  ",A7(B")
	     )))
	  (setq latin1-display t))
    (mapc #'latin1-display-reset latin1-display-sets)
    (set-char-table-range standard-display-table '(#x0100 . #x33FF) nil)
    (set-char-table-range standard-display-table '(#xE000 . #xFFFF) nil)
    (setq latin1-display nil)
    (redraw-display)))

(defcustom latin1-display-mnemonic nil
  "Non-nil means to display potentially more mnemonic sequences.
These are taken from the tables in `internal.el' rather than the Quail
input sequences."
  :type 'boolean
  :group 'latin1-display)

(defcustom latin1-display-face 'default
  "Face to use for displaying substituted ASCII sequences."
  :type 'face
  :version "22.1"
  :group 'latin1-display)

(defun latin1-display-char (char display &optional alt-display)
  "Make an entry in `standard-display-table' for CHAR using string DISPLAY.
If ALT-DISPLAY is provided, use that instead if
`latin1-display-mnemonic' is non-nil.  The actual string displayed is
formatted using `latin1-display-format'.

DISPLAY and ALT-DISPLAY are either strings or vectors.  String values
are formatted using `latin1-display-format' and passed to
`standard-display-ascii'; vectors are put into `standard-display-table'
asis."
  (if (and (stringp alt-display)
	   latin1-display-mnemonic)
      (setq display alt-display))
  (if (stringp display)
      (if (eq 'default latin1-display-face)
	  (standard-display-ascii char (format latin1-display-format display))
	(aset standard-display-table char
	      (vconcat (mapcar (lambda (c) (make-glyph-code c latin1-display-face))
			       display))))
    (aset standard-display-table char
	  (make-glyph-code display latin1-display-face))))

(defun latin1-display-identities (charset)
  "Display each character in CHARSET as the corresponding Latin-1 character.
CHARSET is a symbol which is the nickname of a language environment
using an ISO8859 character set."
  (if (eq charset 'cyrillic)
      (setq charset 'cyrillic-iso))
  (let ((i 128)
	(set (car (remq 'ascii (get-language-info charset 'charset)))))
    (while (<= i 255)
      (let ((ch (decode-char set i)))
	(if ch
	    (aset standard-display-table ch (vector i))))
      (setq i (1+ i)))))

(defun latin1-display-reset (language)
  "Set up the default display for each character of LANGUAGE's charset.
LANGUAGE is a symbol naming a language environment using an ISO8859
character set."
  (if (eq language 'cyrillic)
      (setq language 'cyrillic-iso))
  (let ((charset (if (eq language 'arabic)
		     'arabic-iso8859-6
		   (car (remq 'ascii (get-language-info language
							'charset))))))
    (map-charset-chars #'(lambda (range arg)
			   (standard-display-default (car range) (cdr range)))
		       charset))
  (sit-for 0))

(defun latin1-display-check-font (language)
  "Return non-nil if we have a font with an encoding for LANGUAGE.
LANGUAGE is a symbol naming a language environment using an ISO8859
character set: `latin-2', `hebrew' etc."
  (if (eq language 'cyrillic)
      (setq language 'cyrillic-iso))
  (let* ((info (get-language-info language 'charset))
	 (char (and info (decode-char (car (remq 'ascii info)) ?\ ))))
    (and char (char-displayable-p char))))

;; Backwards compatibility.
(defalias 'latin1-char-displayable-p 'char-displayable-p)
(make-obsolete 'latin1-char-displayable-p 'char-displayable-p "22.1")

(defun latin1-display-setup (set &optional force)
  "Set up Latin-1 display for characters in the given SET.
SET must be a member of `latin1-display-sets'.  Normally, check
whether a font for SET is available and don't set the display if it
is.  If FORCE is non-nil, set up the display regardless."
  (cond
   ((eq set 'latin-2)
    (latin1-display-identities set)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (apply 'latin1-display-char l)))
       '((?,BF(B "'C" "C'")
	 (?,BP(B "'D" "/D")
	 (?,B&(B "'S" "S'")
	 (?,Bf(B "'c" "c'")
	 (?,Bp(B "'d" "/d")
	 (?,BE(B "'L" "L'")
	 (?,Bq(B "'n" "n'")
	 (?,BQ(B "'N" "N'")
	 (?,B`(B "'r" "r'")
	 (?,B@(B "'R" "R'")
	 (?,B6(B "'s" "s'")
	 (?,B<(B "'z" "z'")
	 (?,B,(B "'Z" "Z'")
	 (?,B!(B "`A" "A;")
	 (?,BJ(B "`E" "E;")
	 (?,B#(B "`L" "/L")
	 (?,B*(B "`S" ",S")
	 (?,B^(B "`T" ",T")
	 (?,B/(B "`Z" "Z^.")
	 (?,B1(B "`a" "a;")
	 (?,B3(B "`l" "/l")
	 (?,Bj(B "`e" "e;")
	 (?,B:(B "`s" ",s")
	 (?,B~(B "`t" ",t")
	 (?,B?(B "`z" "z^.")
	 (?,B(B "`." "'.")
	 (?,BC(B "~A" "A(")
	 (?,BH(B "~C" "C<")
	 (?,BO(B "~D" "D<")
	 (?,BL(B "~E" "E<")
	 (?,Bl(B "~e" "e<")
	 (?,B%(B "~L" "L<")
	 (?,BR(B "~N" "N<")
	 (?,BU(B "~O" "O''")
	 (?,BX(B "~R" "R<")
	 (?,B)(B "~S" "S<")
	 (?,B+(B "~T" "T<")
	 (?,B[(B "~U" "U''")
	 (?,B.(B "~Z" "Z<")
	 (?,Bc(B "~a" "a(}")
	 (?,Bh(B "~c" "c<")
	 (?,Bo(B "~d" "d<")
	 (?,B5(B "~l" "l<")
	 (?,Br(B "~n" "n<")
	 (?,Bu(B "~o" "o''")
	 (?,Bx(B "~r" "r<")
	 (?,B9(B "~s" "s<")
	 (?,B;(B "~t" "t<")
	 (?,B{(B "~u" "u''")
	 (?,B>(B "~z" "z<")
	 (?,B7(B "~v" "'<")			; ?,B"(B in latin-pre
	 (?,B"(B "~~" "'(")
	 (?,By(B "uu" "u^0")
	 (?,BY(B "UU" "U^0")
	 (?,BD(B "\"A")
	 (?,Bd(B "\"a")
	 (?,BK(B "\"E" "E:")
	 (?,Bk(B "\"e")
	 (?,B=(B "''" "'")
	 (?,B7(B "'<")			; Lynx's rendering of caron
	 )))

   ((eq set 'latin-3)
    (latin1-display-identities set)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (apply 'latin1-display-char l)))
       '((?,C!(B "/H")
	 (?,C"(B "~`" "'(")
	 (?,C&(B "^H" "H^")
	 (?,C6(B "^h" "h^")
	 (?,C)(B ".I" "I^.")
	 (?,C*(B ",S")
	 (?,C+(B "~G" "G(")
	 (?,C,(B "^J" "J^")
	 (?,C/(B ".Z" "Z^.")
	 (?,C1(B "/h")
	 (?,C9(B ".i" "i^.")
	 (?,C:(B ",s")
	 (?,C;(B "~g" "g(")
	 (?,C<(B "^j" "j^")
	 (?,C?(B ".Z" "z^.")
	 (?,CE(B ".c" "C^.")
	 (?,CF(B "^C" "C^")
	 (?,CU(B ".G" "G^.")
	 (?,CX(B "^G" "G^")
	 (?,C](B "~U" "U(")
	 (?,C^(B "^S" "S^")
	 (?,Ce(B ".C" "c^.")
	 (?,Cf(B "^c" "c^")
	 (?,Cu(B ".g" "g^.")
	 (?,Cx(B "^g" "g^")
	 (?,C}(B "~u" "u(")
	 (?,C~(B "^s" "s^")
	 (?,C(B "/." "^."))))

   ((eq set 'latin-4)
    (latin1-display-identities set)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (apply 'latin1-display-char l)))
       '((?,D!(B "A," "A;")
	 (?,D"(B "k/" "kk")
	 (?,D#(B "R," ",R")
	 (?,D%(B "I~" "?I")
	 (?,D&(B "L," ",L")
	 (?,D)(B "S~" "S<")
	 (?,D*(B "E-")
	 (?,D+(B "G," ",G")
	 (?,D,(B "T/" "/T")
	 (?,D.(B "Z~" "Z<")
	 (?,D1(B "a," "a;")
	 (?,D2(B "';")
	 (?,D3(B "r," ",r")
	 (?,D5(B "i~" "~i")
	 (?,D6(B "l," ",l")
	 (?,D7(B "'<")
	 (?,D9(B "s~" "s<")
	 (?,D:(B "e-")
	 (?,D;(B "g," ",g")
	 (?,D<(B "t/" "/t")
	 (?,D=(B "N/" "NG")
	 (?,D>(B "z~" "z<")
	 (?,D?(B "n/" "ng")
	 (?,D@(B "A-")
	 (?,DG(B "I," "I;")
	 (?,DH(B "C~" "C<")
	 (?,DJ(B "E," "E;")
	 (?,DL(B "E." "E^.")
	 (?,DO(B "I-")
	 (?,DQ(B "N," ",N")
	 (?,DR(B "O-")
	 (?,DS(B "K," ",K")
	 (?,DY(B "U," "U;")
	 (?,D](B "U~" "~U")
	 (?,D^(B "U-")
	 (?,D`(B "a-")
	 (?,Dg(B "i," "i;")
	 (?,Dh(B "c~" "c<")
	 (?,Dj(B "e," "e;")
	 (?,Dl(B "e." "e^.")
	 (?,Do(B "i-")
	 (?,Dp(B "d/" "/d")
	 (?,Dq(B "n," ",n")
	 (?,Dr(B "o-")
	 (?,Ds(B "k," ",k")
	 (?,Dy(B "u," "u;")
	 (?,D}(B "u~" "~u")
	 (?,D~(B "u-")
	 (?,D(B "^."))))

   ((eq set 'latin-5)
    (latin1-display-identities set)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	 (apply 'latin1-display-char l)))
       '((?,Mp(B "~g" "g(")
	 (?,MP(B "~G" "G(")
	 (?,M](B ".I" "I^.")
	 (?,M~(B ",s")
	 (?,M^(B ",S")
	 (?,Mj(B "^e" "e<")			; from latin-post
	 (?,Ml(B ".e" "e^.")
	 (?,Mo(B "\"i" "i-")		; from latin-post
	 (?,M}(B ".i" "i."))))

   ((eq set 'latin-8)
    (latin1-display-identities set)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (apply 'latin1-display-char l)))
       '((?,_!(B ".B" "B`")
	 (?,_"(B ".b" "b`")
	 (?,_%(B ".c" "c`")
	 (?,_$(B ".C" "C`")
	 (?,_&(B ".D" "D`")
	 (?,_+(B ".d" "d`")
	 (?,_8(B "`w")
	 (?,_((B "`W")
	 (?,_:(B "'w" "w'")
	 (?,_*(B "'W" "W'")
	 (?,_<(B "`y")
	 (?,_,(B "`Y")
	 (?,_1(B ".f" "f`")
	 (?,_0(B ".F" "F`")
	 (?,_3(B ".g" "g`")
	 (?,_2(B ".G" "G`")
	 (?,_5(B ".m" "m`")
	 (?,_4(B ".M" "M`")
	 (?,_9(B ".p" "p`")
	 (?,_7(B ".P" "P`")
	 (?,_?(B ".s" "s`")
	 (?,_;(B ".S" "S`")
	 (?,_>(B "\"w")
	 (?,_=(B "\"W")
	 (?,_p(B "^w" "w^")
	 (?,_P(B "^W" "W^")
	 (?,_w(B ".t" "t`")
	 (?,_W(B ".T" "T`")
	 (?,_~(B "^y" "y^")
	 (?,_^(B "^Y" "Y^")
	 (?,_/(B "\"Y"))))

   ((eq set 'latin-9)
    (latin1-display-identities set)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (apply 'latin1-display-char l)))
       '((?,b((B "~s" "s<")
	 (?,b&(B "~S" "S<")
	 (?,b$(B "Euro" "E=")
	 (?,b8(B "~z" "z<")
	 (?,b4(B "~Z" "Z<")
	 (?,b>(B "\"Y")
	 (?,b=(B "oe")
	 (?,b<(B "OE"))))

   ((eq set 'greek)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (apply 'latin1-display-char l)))
       '((?,F!(B "9'")
	 (?,F"(B "'9")
	 (?,F/(B "-M")
	 (?,F5(B "'%")
	 (?,F6(B "'A")
	 (?,F8(B "'E")
	 (?,F9(B "'H")
	 (?,F:(B "'I")
	 (?,F<(B "'O")
	 (?,F>(B "'Y")
	 (?,F?(B "W%")
	 (?,F@(B "i3")
	 (?,FC(B "G*")
	 (?,FD(B "D*")
	 (?,FH(B "TH")
	 (?,FK(B "L*")
	 (?,FN(B "C*")
	 (?,FP(B "P*")
	 (?,FS(B "S*")
	 (?,FV(B "F*")
	 (?,FX(B "Q*")
	 (?,FY(B "W*")
	 (?,FZ(B "\"I")
	 (?,F[(B "\"Y")
	 (?,F\(B "a%")
	 (?,F](B "e%")
	 (?,F^(B "y%")
	 (?,F_(B "i%")
	 (?,F`(B "u3")
	 (?,Fa(B "a*")
	 (?,Fb(B "b*")
	 (?,Fc(B "g*")
	 (?,Fd(B "d*")
	 (?,Fe(B "e*")
	 (?,Ff(B "z*")
	 (?,Fg(B "y*")
	 (?,Fh(B "h*")
	 (?,Fi(B "i*")
	 (?,Fj(B "k")
	 (?,Fk(B "l*")
	 (?,Fl(B "m*")
	 (?,Fm(B "n*")
	 (?,Fn(B "c*")
	 (?,Fp(B "p*")
	 (?,Fq(B "r*")
	 (?,Fr(B "*s")
	 (?,Fs(B "s*")
	 (?,Ft(B "t*")
	 (?,Fu(B "u")
	 (?,Fv(B "f*")
	 (?,Fw(B "x*")
	 (?,Fx(B "q*")
	 (?,Fy(B "w*")
	 (?,Fz(B "\"i")
	 (?,F{(B "\"u")
	 (?,F|(B "'o")
	 (?,F}(B "'u")
	 (?,F~(B "'w")))
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (aset standard-display-table (car l) (string-to-vector (cadr l)))))
       '((?,FA(B "A")
	 (?,FB(B "B")
	 (?,FE(B "E")
	 (?,FF(B "Z")
	 (?,FG(B "H")
	 (?,FI(B "I")
	 (?,FJ(B "J")
	 (?,FL(B "M")
	 (?,FM(B "N")
	 (?,FO(B "O")
	 (?,FQ(B "P")
	 (?,FT(B "T")
	 (?,FU(B "Y")
	 (?,FW(B "X")
	 (?,Fo(B "o"))))

   ((eq set 'hebrew)
    ;; Don't start with identities, since we don't have definitions
    ;; for a lot of Hebrew in internal.el.  (Intlfonts is also
    ;; missing some glyphs.)
    (let ((i 34))
      (while (<= i 62)
	(let ((ch (decode-char 'hebrew-iso8859-8 i)))
	  (if ch
	      (aset standard-display-table ch
		    (vector (decode-char 'latin-iso8859-1 i)))))
	(setq i (1+ i))))
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (aset standard-display-table (car l) (string-to-vector (cadr l)))))
       '((?,H_(B "=2")
	 (?,H`(B "A+")
	 (?,Ha(B "B+")
	 (?,Hb(B "G+")
	 (?,Hc(B "D+")
	 (?,Hd(B "H+")
	 (?,He(B "W+")
	 (?,Hf(B "Z+")
	 (?,Hg(B "X+")
	 (?,Hh(B "Tj")
	 (?,Hi(B "J+")
	 (?,Hj(B "K%")
	 (?,Hk(B "K+")
	 (?,Hl(B "L+")
	 (?,Hm(B "M%")
	 (?,Hn(B "M+")
	 (?,Ho(B "N%")
	 (?,Hp(B "N+")
	 (?,Hq(B "S+")
	 (?,Hr(B "E+")
	 (?,Hs(B "P%")
	 (?,Ht(B "P+")
	 (?,Hu(B "Zj")
	 (?,Hv(B "ZJ")
	 (?,Hw(B "Q+")
	 (?,Hx(B "R+")
	 (?,Hy(B "Sh")
	 (?,Hz(B "T+"))))

   ;; Arabic probably isn't so useful in the absence of Arabic
   ;; language support...
   ((eq set 'arabic)
    (setq set 'arabic)
    (or (char-displayable-p ?,G (B)
	(aset standard-display-table ?,G (B ",A (B"))
    (or (char-displayable-p ?,G$(B)
	(aset standard-display-table ?,G$(B ",A$(B"))
    (or (char-displayable-p ?,G-(B)
	(aset standard-display-table ?,G-(B ",A-(B"))
    (mapc (lambda (l)
	    (or (char-displayable-p (car l))
		(apply  'latin1-display-char l)))
	    '((?,G,(B ",+")
	      (?,G;(B ";+")
	      (?,G?(B "?+")
	      (?,GA(B "H'")
	      (?,GB(B "aM")
	      (?,GC(B "aH")
	      (?,GD(B "wH")
	      (?,GE(B "ah")
	      (?,GF(B "yH")
	      (?,GG(B "a+")
	      (?,GH(B "b+")
	      (?,GI(B "tm")
	      (?,GJ(B "t+")
	      (?,GK(B "tk")
	      (?,GL(B "g+")
	      (?,GM(B "hk")
	      (?,GN(B "x+")
	      (?,GO(B "d+")
	      (?,GP(B "dk")
	      (?,GQ(B "r+")
	      (?,GR(B "z+")
	      (?,GS(B "s+")
	      (?,GT(B "sn")
	      (?,GU(B "c+")
	      (?,GV(B "dd")
	      (?,GW(B "tj")
	      (?,GX(B "zH")
	      (?,GY(B "e+")
	      (?,GZ(B "i+")
	      (?,G`(B "++")
	      (?,Ga(B "f+")
	      (?,Gb(B "q+")
	      (?,Gc(B "k+")
	      (?,Gd(B "l+")
	      (?,Ge(B "m+")
	      (?,Gf(B "n+")
	      (?,Gg(B "h+")
	      (?,Gh(B "w+")
	      (?,Gi(B "j+")
	      (?,Gj(B "y+")
	      (?,Gk(B ":+")
	      (?,Gl(B "\"+")
	      (?,Gm(B "=+")
	      (?,Gn(B "/+")
	      (?,Go(B "'+")
	      (?,Gp(B "1+")
	      (?,Gq(B "3+")
	      (?,Gr(B "0+"))))

   ((eq set 'cyrillic)
    (setq set 'cyrillic-iso)
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (apply 'latin1-display-char l)))
       '((?,L"(B "Dj")
	 (?,L#(B "Gj")
	 (?,L$(B "IE")
	 (?,L)(B "Lj")
	 (?,L*(B "Nj")
	 (?,L+(B "Ts")
	 (?,L,(B "Kj")
	 (?,L.(B "V%")
	 (?,L/(B "Dzh")
	 (?,L1(B "B=")
	 (?,L3(B ",Ab(B")
	 (?,L4(B "D")
	 (?,L6(B "Z%")
	 (?,L7(B "3")
	 (?,L8(B "U")
	 (?,L9(B "J=")
	 (?,L;(B "L=")
	 (?,L?(B "P=")
	 (?,LC(B "Y")
	 (?,LD(B ",Ah(B")
	 (?,LF(B "C=")
	 (?,LG(B "C%")
	 (?,LH(B "S%")
	 (?,LI(B "Sc")
	 (?,LJ(B "=\"")
	 (?,LK(B "Y=")
	 (?,LL(B "%\"")
	 (?,LM(B "Ee")
	 (?,LN(B "Yu")
	 (?,LO(B "Ya")
	 (?,LQ(B "b")
	 (?,LR(B "v=")
	 (?,LS(B "g=")
	 (?,LT(B "g")
	 (?,LV(B "z%")
	 (?,LW(B "z=")
	 (?,LX(B "u")
	 (?,LY(B "j=")
	 (?,LZ(B "k")
	 (?,L[(B "l=")
	 (?,L\(B "m=")
	 (?,L](B "n=")
	 (?,L_(B "n")
	 (?,L`(B "p")
	 (?,Lb(B "t=")
	 (?,Ld(B "f=")
	 (?,Lf(B "c=")
	 (?,Lg(B "c%")
	 (?,Lh(B "s%")
	 (?,Li(B "sc")
	 (?,Lj(B "='")
	 (?,Lk(B "y=")
	 (?,Ll(B "%'")
	 (?,Lm(B "ee")
	 (?,Ln(B "yu")
	 (?,Lo(B "ya")
	 (?,Lp(B "N0")
	 (?,Lr(B "dj")
	 (?,Ls(B "gj")
	 (?,Lt(B "ie")
	 (?,Ly(B "lj")
	 (?,Lz(B "nj")
	 (?,L{(B "ts")
	 (?,L|(B "kj")
	 (?,L~(B "v%")
	 (?,L(B "dzh")))
    (mapc
     (lambda (l)
       (or (char-displayable-p (car l))
	   (aset standard-display-table (car l) (string-to-vector (cadr l)))))
       '((?,L!(B ",AK(B")
	 (?,L%(B "S")
	 (?,L&(B "I")
	 (?,L'(B ",AO(B")
	 (?,L((B "J")
	 (?,Lq(B ",Ak(B")
	 (?,L}(B ",A'(B")
	 (?,L-(B "-")
	 (?,L0(B "A")
	 (?,L2(B "B")
	 (?,L5(B "E")
	 (?,L:(B "K")
	 (?,L<(B "M")
	 (?,L=(B "H")
	 (?,L>(B "O")
	 (?,L@(B "P")
	 (?,LA(B "C")
	 (?,LB(B "T")
	 (?,LE(B "X")
	 (?,LP(B "a")
	 (?,LU(B "e")
	 (?,L^(B "o")
	 (?,La(B "c")
	 (?,Lc(B "y")
	 (?,Le(B "x")
	 (?,Lu(B "s")
	 (?,Lv(B "i")
	 (?,Lw(B ",Ao(B")
	 (?,Lx(B "j"))))

   (t (error "Unsupported character set: %S" set)))

  (sit-for 0))

;;;###autoload
(defcustom latin1-display-ucs-per-lynx nil
  "Set up Latin-1/ASCII display for Unicode characters.
This uses the transliterations of the Lynx browser.  The display isn't
changed if the display can render Unicode characters.

Setting this variable directly does not take effect;
use either \\[customize] or the function `latin1-display'."
  :group 'latin1-display
  :type 'boolean
  :require 'latin1-disp
  :initialize 'custom-initialize-default
  :set (lambda (symbol value)
	 (if value
	     (latin1-display-ucs-per-lynx 1)
	   (latin1-display-ucs-per-lynx -1))))

(defun latin1-display-ucs-per-lynx (arg)
  "Set up Latin-1/ASCII display for Unicode characters.
This uses the transliterations of the Lynx browser.

With argument ARG, turn such display on if ARG is positive, otherwise
turn it off and display Unicode characters literally.  The display
isn't changed if the display can render Unicode characters."
  (interactive "p")
  (if (> arg 0)
      (unless (char-displayable-p #x101) ; a with macron
	;; It doesn't look as though we have a Unicode font.
	(let ((latin1-display-format "%s"))
	  (mapc
	   (lambda (l)
	     (apply 'latin1-display-char l))
	   ;; Table derived by running Lynx on a suitable list of
	   ;; characters in a utf-8 file, except for some added by
	   ;; hand at the end.
	   '((?\$,1  (B "A")
	     (?\$,1 !(B "a")
	     (?\$,1 "(B "A")
	     (?\$,1 #(B "a")
	     (?\$,1 $(B "A")
	     (?\$,1 %(B "a")
	     (?\$,1 &(B "C")
	     (?\$,1 '(B "c")
	     (?\$,1 ((B "C")
	     (?\$,1 )(B "c")
	     (?\$,1 *(B "C")
	     (?\$,1 +(B "c")
	     (?\$,1 ,(B "C")
	     (?\$,1 -(B "c")
	     (?\$,1 .(B "D")
	     (?\$,1 /(B "d")
	     (?\$,1 0(B ",AP(B")
	     (?\$,1 1(B "d/")
	     (?\$,1 2(B "E")
	     (?\$,1 3(B "e")
	     (?\$,1 4(B "E")
	     (?\$,1 5(B "e")
	     (?\$,1 6(B "E")
	     (?\$,1 7(B "e")
	     (?\$,1 8(B "E")
	     (?\$,1 9(B "e")
	     (?\$,1 :(B "E")
	     (?\$,1 ;(B "e")
	     (?\$,1 <(B "G")
	     (?\$,1 =(B "g")
	     (?\$,1 >(B "G")
	     (?\$,1 ?(B "g")
	     (?\$,1 @(B "G")
	     (?\$,1 A(B "g")
	     (?\$,1 B(B "G")
	     (?\$,1 C(B "g")
	     (?\$,1 D(B "H")
	     (?\$,1 E(B "h")
	     (?\$,1 F(B "H/")
	     (?\$,1 G(B "H")
	     (?\$,1 H(B "I")
	     (?\$,1 I(B "i")
	     (?\$,1 J(B "I")
	     (?\$,1 K(B "i")
	     (?\$,1 L(B "I")
	     (?\$,1 M(B "i")
	     (?\$,1 N(B "I")
	     (?\$,1 O(B "i")
	     (?\$,1 P(B "I")
	     (?\$,1 Q(B "i")
	     (?\$,1 R(B "IJ")
	     (?\$,1 S(B "ij")
	     (?\$,1 T(B "J")
	     (?\$,1 U(B "j")
	     (?\$,1 V(B "K")
	     (?\$,1 W(B "k")
	     (?\$,1 X(B "kk")
	     (?\$,1 Y(B "L")
	     (?\$,1 Z(B "l")
	     (?\$,1 [(B "L")
	     (?\$,1 \(B "l")
	     (?\$,1 ](B "L")
	     (?\$,1 ^(B "l")
	     (?\$,1 _(B "L.")
	     (?\$,1 `(B "l.")
	     (?\$,1 a(B "L/")
	     (?\$,1 b(B "l/")
	     (?\$,1 c(B "N")
	     (?\$,1 d(B "n")
	     (?\$,1 e(B "N")
	     (?\$,1 f(B "n")
	     (?\$,1 g(B "N")
	     (?\$,1 h(B "n")
	     (?\$,1 i(B "'n")
	     (?\$,1 j(B "NG")
	     (?\$,1 k(B "N")
	     (?\$,1 l(B "O")
	     (?\$,1 m(B "o")
	     (?\$,1 n(B "O")
	     (?\$,1 o(B "o")
	     (?\$,1 p(B "O\"")
	     (?\$,1 q(B "o\"")
	     (?\$,1 r(B "OE")
	     (?\$,1 s(B "oe")
	     (?\$,1 t(B "R")
	     (?\$,1 u(B "r")
	     (?\$,1 v(B "R")
	     (?\$,1 w(B "r")
	     (?\$,1 x(B "R")
	     (?\$,1 y(B "r")
	     (?\$,1 z(B "S")
	     (?\$,1 {(B "s")
	     (?\$,1 |(B "S")
	     (?\$,1 }(B "s")
	     (?\$,1 ~(B "S")
	     (?\$,1 (B "s")
	     (?\$,1! (B "S")
	     (?\$,1!!(B "s")
	     (?\$,1!"(B "T")
	     (?\$,1!#(B "t")
	     (?\$,1!$(B "T")
	     (?\$,1!%(B "t")
	     (?\$,1!&(B "T/")
	     (?\$,1!'(B "t/")
	     (?\$,1!((B "U")
	     (?\$,1!)(B "u")
	     (?\$,1!*(B "U")
	     (?\$,1!+(B "u")
	     (?\$,1!,(B "U")
	     (?\$,1!-(B "u")
	     (?\$,1!.(B "U")
	     (?\$,1!/(B "u")
	     (?\$,1!0(B "U\"")
	     (?\$,1!1(B "u\"")
	     (?\$,1!2(B "U")
	     (?\$,1!3(B "u")
	     (?\$,1!4(B "W")
	     (?\$,1!5(B "w")
	     (?\$,1!6(B "Y")
	     (?\$,1!7(B "y")
	     (?\$,1!8(B "Y")
	     (?\$,1!9(B "Z")
	     (?\$,1!:(B "z")
	     (?\$,1!;(B "Z")
	     (?\$,1!<(B "z")
	     (?\$,1!=(B "Z")
	     (?\$,1!>(B "z")
	     (?\$,1!?(B "s1")
	     (?\$,1!G(B "C2")
	     (?\$,1!H(B "c2")
	     (?\$,1!Q(B "F2")
	     (?\$,1!R(B " f")
	     (?\$,1!X(B "K2")
	     (?\$,1!Y(B "k2")
	     (?\$,1!`(B "O9")
	     (?\$,1!a(B "o9")
	     (?\$,1!b(B "OI")
	     (?\$,1!c(B "oi")
	     (?\$,1!f(B "yr")
	     (?\$,1!o(B "U9")
	     (?\$,1!p(B "u9")
	     (?\$,1!u(B "Z/")
	     (?\$,1!v(B "z/")
	     (?\$,1!w(B "ED")
	     (?\$,1"-(B "A")
	     (?\$,1".(B "a")
	     (?\$,1"/(B "I")
	     (?\$,1"0(B "i")
	     (?\$,1"1(B "O")
	     (?\$,1"2(B "o")
	     (?\$,1"3(B "U")
	     (?\$,1"4(B "u")
	     (?\$,1"5(B "U:-")
	     (?\$,1"6(B "u:-")
	     (?\$,1"7(B "U:'")
	     (?\$,1"8(B "u:'")
	     (?\$,1"9(B "U:<")
	     (?\$,1":(B "u:<")
	     (?\$,1";(B "U:!")
	     (?\$,1"<(B "u:!")
	     (?\$,1">(B "A1")
	     (?\$,1"?(B "a1")
	     (?\$,1"@(B "A7")
	     (?\$,1"A(B "a7")
	     (?\$,1"B(B "A3")
	     (?\$,1"C(B "a3")
	     (?\$,1"D(B "G/")
	     (?\$,1"E(B "g/")
	     (?\$,1"F(B "G")
	     (?\$,1"G(B "g")
	     (?\$,1"H(B "K")
	     (?\$,1"I(B "k")
	     (?\$,1"J(B "O")
	     (?\$,1"K(B "o")
	     (?\$,1"L(B "O1")
	     (?\$,1"M(B "o1")
	     (?\$,1"N(B "EZ")
	     (?\$,1"O(B "ez")
	     (?\$,1"P(B "j")
	     (?\$,1"T(B "G")
	     (?\$,1"U(B "g")
	     (?\$,1"Z(B "AA'")
	     (?\$,1"[(B "aa'")
	     (?\$,1"\(B "AE'")
	     (?\$,1"](B "ae'")
	     (?\$,1"^(B "O/'")
	     (?\$,1"_(B "o/'")
	     (?\$,1"`(B "A!!")
	     (?\$,1"a(B "a!!")
	     (?\$,1"b(B "A)")
	     (?\$,1"c(B "a)")
	     (?\$,1"d(B "E!!")
	     (?\$,1"e(B "e!!")
	     (?\$,1"f(B "E)")
	     (?\$,1"g(B "e)")
	     (?\$,1"h(B "I!!")
	     (?\$,1"i(B "i!!")
	     (?\$,1"j(B "I)")
	     (?\$,1"k(B "i)")
	     (?\$,1"l(B "O!!")
	     (?\$,1"m(B "o!!")
	     (?\$,1"n(B "O)")
	     (?\$,1"o(B "o)")
	     (?\$,1"p(B "R!!")
	     (?\$,1"q(B "r!!")
	     (?\$,1"r(B "R)")
	     (?\$,1"s(B "r)")
	     (?\$,1"t(B "U!!")
	     (?\$,1"u(B "u!!")
	     (?\$,1"v(B "U)")
	     (?\$,1"w(B "u)")
	     (?\$,1"}(B "Z")
	     (?\$,1#Q(B "A")
	     (?\$,1#R(B "A.")
	     (?\$,1#S(B "b`")
	     (?\$,1#T(B "O")
	     (?\$,1#V(B "d.")
	     (?\$,1#W(B "d`")
	     (?\$,1#X(B "@<umd>")
	     (?\$,1#Y(B "@")
	     (?\$,1#Z(B "R")
	     (?\$,1#[(B "E")
	     (?\$,1#\(B "V\"")
	     (?\$,1#](B "R<umd>")
	     (?\$,1#^(B "O\"")
	     (?\$,1#_(B "J")
	     (?\$,1#`(B "g`")
	     (?\$,1#a(B "g")
	     (?\$,1#b(B "G")
	     (?\$,1#c(B "Q")
	     (?\$,1#d(B "o-")
	     (?\$,1#e(B "j<rnd>")
	     (?\$,1#f(B "h<?>")
	     (?\$,1#h(B "i\"")
	     (?\$,1#i(B "I")
	     (?\$,1#j(B "I")
	     (?\$,1#k(B "L")
	     (?\$,1#l(B "L")
	     (?\$,1#m(B "l.")
	     (?\$,1#n(B "z<lat>")
	     (?\$,1#o(B "u-")
	     (?\$,1#p(B "j<vel>")
	     (?\$,1#q(B "M")
	     (?\$,1#s(B "n.")
	     (?\$,1#t(B "n\"")
	     (?\$,1#u(B "@.")
	     (?\$,1#v(B "&.")
	     (?\$,1#w(B "U")
	     (?\$,1#y(B "r")
	     (?\$,1#z(B "*<lat>")
	     (?\$,1#{(B "r.")
	     (?\$,1#}(B "*.")
	     (?\$,1#~(B "*")
	     (?\$,1$ (B "R")
	     (?\$,1$!(B "g\"")
	     (?\$,1$"(B "s.")
	     (?\$,1$#(B "S")
	     (?\$,1$$(B "J`")
	     (?\$,1$'(B "t!")
	     (?\$,1$((B "t.")
	     (?\$,1$)(B "u\"")
	     (?\$,1$*(B "U")
	     (?\$,1$+(B "r<lbd>")
	     (?\$,1$,(B "V")
	     (?\$,1$-(B "w<vls>")
	     (?\$,1$.(B "l^")
	     (?\$,1$/(B "I.")
	     (?\$,1$0(B "z.")
	     (?\$,1$2(B "Z")
	     (?\$,1$4(B "?")
	     (?\$,1$5(B "H<vcd>")
	     (?\$,1$6(B "l!")
	     (?\$,1$7(B "c!")
	     (?\$,1$8(B "p!")
	     (?\$,1$9(B "b<trl>")
	     (?\$,1$;(B "G`")
	     (?\$,1$=(B "j")
	     (?\$,1$>(B "k!")
	     (?\$,1$?(B "L")
	     (?\$,1$@(B "q`")
	     (?\$,1$D(B "d3")
	     (?\$,1$F(B "ts")
	     (?\$,1$G(B "tS")
	     (?\$,1$P(B "<h>")
	     (?\$,1$Q(B "<?>")
	     (?\$,1$R(B ";")
	     (?\$,1$S(B "<r>")
	     (?\$,1$W(B "<w>")
	     (?\$,1$[(B ";S")
	     (?\$,1$\(B "`")
	     (?\$,1$f(B "^")
	     (?\$,1$g(B "'<")
	     (?\$,1$h(B "|")
	     (?\$,1$i(B "1-")
	     (?\$,1$k(B "1!")
	     (?\$,1$p(B ":")
	     (?\$,1$q(B ":\\")
	     (?\$,1$v(B "+")
	     (?\$,1$w(B "-")
	     (?\$,1$x(B "'(")
	     (?\$,1$y(B "'.")
	     (?\$,1$z(B "'0")
	     (?\$,1${(B "';")
	     (?\$,1$|(B "~")
	     (?\$,1$}(B "'\"")
	     (?\$,1%%(B "_T")
	     (?\$,1%&(B "_H")
	     (?\$,1%'(B "_M")
	     (?\$,1%((B "_L")
	     (?\$,1%)(B "_B")
	     (?\$,1%,(B "_v")
	     (?\$,1%.(B "''")
	     (?\$,1%@(B "`")
	     (?\$,1%A(B "'")
	     (?\$,1%B(B "^")
	     (?\$,1%C(B "~")
	     (?\$,1%D(B ",A/(B")
	     (?\$,1%G(B ",A7(B")
	     (?\$,1%H(B ",A((B")
	     (?\$,1%J(B ",A0(B")
	     (?\$,1%K(B "''")
	     (?\$,1%M(B "|")
	     (?\$,1%N(B "||")
	     (?\$,1%O(B "``")
	     (?\$,1%a(B ";")
	     (?\$,1%b(B ".")
	     (?\$,1%c(B ".")
	     (?\$,1%d(B "<?>")
	     (?\$,1%e(B "<o>")
	     (?\$,1%f(B ",")
	     (?\$,1%g(B ",A8(B")
	     (?\$,1%i(B "-")
	     (?\$,1%j(B "[")
	     (?\$,1%k(B "<w>")
	     (?\$,1%t(B "~")
	     (?\$,1%w(B "/")
	     (?\$,1%x(B "/")
	     (?\$,1& (B "`")
	     (?\$,1&!(B "'")
	     (?\$,1&"(B "~")
	     (?\$,1&$(B "'%")
	     (?\$,1&%(B "j3")
	     (?\$,1&'(B "=")
	     (?\$,1&@(B "~~")
	     (?\$,1&T(B "'")
	     (?\$,1&U(B ",")
	     (?\$,1&Z(B "j3")
	     (?\$,1&^(B "?%")
	     (?\$,1&d(B "'*")
	     (?\$,1&e(B "'%")
	     (?\$,1&f(B "A'")
	     (?\$,1&g(B ",A7(B")
	     (?\$,1&h(B "E'")
	     (?\$,1&i(B "Y%")
	     (?\$,1&j(B "I'")
	     (?\$,1&l(B "O'")
	     (?\$,1&n(B "U%")
	     (?\$,1&o(B "W%")
	     (?\$,1&p(B "i3")
	     (?\$,1&q(B "A")
	     (?\$,1&r(B "B")
	     (?\$,1&s(B "G")
	     (?\$,1&t(B "D")
	     (?\$,1&u(B "E")
	     (?\$,1&v(B "Z")
	     (?\$,1&w(B "Y")
	     (?\$,1&x(B "TH")
	     (?\$,1&y(B "I")
	     (?\$,1&z(B "K")
	     (?\$,1&{(B "L")
	     (?\$,1&|(B "M")
	     (?\$,1&}(B "N")
	     (?\$,1&~(B "C")
	     (?\$,1&(B "O")
	     (?\$,1' (B "P")
	     (?\$,1'!(B "R")
	     (?\$,1'#(B "S")
	     (?\$,1'$(B "T")
	     (?\$,1'%(B "U")
	     (?\$,1'&(B "F")
	     (?\$,1''(B "X")
	     (?\$,1'((B "Q")
	     (?\$,1')(B "W*")
	     (?\$,1'*(B "J")
	     (?\$,1'+(B "V*")
	     (?\$,1',(B "a'")
	     (?\$,1'-(B "e'")
	     (?\$,1'.(B "y%")
	     (?\$,1'/(B "i'")
	     (?\$,1'0(B "u3")
	     (?\$,1'1(B "a")
	     (?\$,1'2(B "b")
	     (?\$,1'3(B "g")
	     (?\$,1'4(B "d")
	     (?\$,1'5(B "e")
	     (?\$,1'6(B "z")
	     (?\$,1'7(B "y")
	     (?\$,1'8(B "th")
	     (?\$,1'9(B "i")
	     (?\$,1':(B "k")
	     (?\$,1';(B "l")
	     (?\$,1'<(B ",A5(B")
	     (?\$,1'=(B "n")
	     (?\$,1'>(B "c")
	     (?\$,1'?(B "o")
	     (?\$,1'@(B "p")
	     (?\$,1'A(B "r")
	     (?\$,1'B(B "*s")
	     (?\$,1'C(B "s")
	     (?\$,1'D(B "t")
	     (?\$,1'E(B "u")
	     (?\$,1'F(B "f")
	     (?\$,1'G(B "x")
	     (?\$,1'H(B "q")
	     (?\$,1'I(B "w")
	     (?\$,1'J(B "j")
	     (?\$,1'K(B "v*")
	     (?\$,1'L(B "o'")
	     (?\$,1'M(B "u%")
	     (?\$,1'N(B "w%")
	     (?\$,1'P(B "beta ")
	     (?\$,1'Q(B "theta ")
	     (?\$,1'R(B "upsi ")
	     (?\$,1'U(B "phi ")
	     (?\$,1'V(B "pi ")
	     (?\$,1'W(B "k.")
	     (?\$,1'Z(B "T3")
	     (?\$,1'[(B "t3")
	     (?\$,1'\(B "M3")
	     (?\$,1'](B "m3")
	     (?\$,1'^(B "K3")
	     (?\$,1'_(B "k3")
	     (?\$,1'`(B "P3")
	     (?\$,1'a(B "p3")
	     (?\$,1'p(B "kappa ")
	     (?\$,1'q(B "rho ")
	     (?\$,1's(B "J")
	     (?\$,1't(B "'%")
	     (?\$,1'u(B "j3")
	     (?\$,1(!(B "IO")
	     (?\$,1("(B "D%")
	     (?\$,1(#(B "G%")
	     (?\$,1($(B "IE")
	     (?\$,1(%(B "DS")
	     (?\$,1(&(B "II")
	     (?\$,1('(B "YI")
	     (?\$,1(((B "J%")
	     (?\$,1()(B "LJ")
	     (?\$,1(*(B "NJ")
	     (?\$,1(+(B "Ts")
	     (?\$,1(,(B "KJ")
	     (?\$,1(.(B "V%")
	     (?\$,1(/(B "DZ")
	     (?\$,1(0(B "A")
	     (?\$,1(1(B "B")
	     (?\$,1(2(B "V")
	     (?\$,1(3(B "G")
	     (?\$,1(4(B "D")
	     (?\$,1(5(B "E")
	     (?\$,1(6(B "ZH")
	     (?\$,1(7(B "Z")
	     (?\$,1(8(B "I")
	     (?\$,1(9(B "J")
	     (?\$,1(:(B "K")
	     (?\$,1(;(B "L")
	     (?\$,1(<(B "M")
	     (?\$,1(=(B "N")
	     (?\$,1(>(B "O")
	     (?\$,1(?(B "P")
	     (?\$,1(@(B "R")
	     (?\$,1(A(B "S")
	     (?\$,1(B(B "T")
	     (?\$,1(C(B "U")
	     (?\$,1(D(B "F")
	     (?\$,1(E(B "H")
	     (?\$,1(F(B "C")
	     (?\$,1(G(B "CH")
	     (?\$,1(H(B "SH")
	     (?\$,1(I(B "SCH")
	     (?\$,1(J(B "\"")
	     (?\$,1(K(B "Y")
	     (?\$,1(L(B "'")
	     (?\$,1(M(B "`E")
	     (?\$,1(N(B "YU")
	     (?\$,1(O(B "YA")
	     (?\$,1(P(B "a")
	     (?\$,1(Q(B "b")
	     (?\$,1(R(B "v")
	     (?\$,1(S(B "g")
	     (?\$,1(T(B "d")
	     (?\$,1(U(B "e")
	     (?\$,1(V(B "zh")
	     (?\$,1(W(B "z")
	     (?\$,1(X(B "i")
	     (?\$,1(Y(B "j")
	     (?\$,1(Z(B "k")
	     (?\$,1([(B "l")
	     (?\$,1(\(B "m")
	     (?\$,1(](B "n")
	     (?\$,1(^(B "o")
	     (?\$,1(_(B "p")
	     (?\$,1(`(B "r")
	     (?\$,1(a(B "s")
	     (?\$,1(b(B "t")
	     (?\$,1(c(B "u")
	     (?\$,1(d(B "f")
	     (?\$,1(e(B "h")
	     (?\$,1(f(B "c")
	     (?\$,1(g(B "ch")
	     (?\$,1(h(B "sh")
	     (?\$,1(i(B "sch")
	     (?\$,1(j(B "\"")
	     (?\$,1(k(B "y")
	     (?\$,1(l(B "'")
	     (?\$,1(m(B "`e")
	     (?\$,1(n(B "yu")
	     (?\$,1(o(B "ya")
	     (?\$,1(q(B "io")
	     (?\$,1(r(B "d%")
	     (?\$,1(s(B "g%")
	     (?\$,1(t(B "ie")
	     (?\$,1(u(B "ds")
	     (?\$,1(v(B "ii")
	     (?\$,1(w(B "yi")
	     (?\$,1(x(B "j%")
	     (?\$,1(y(B "lj")
	     (?\$,1(z(B "nj")
	     (?\$,1({(B "ts")
	     (?\$,1(|(B "kj")
	     (?\$,1(~(B "v%")
	     (?\$,1((B "dz")
	     (?\$,1)"(B "Y3")
	     (?\$,1)#(B "y3")
	     (?\$,1)*(B "O3")
	     (?\$,1)+(B "o3")
	     (?\$,1)2(B "F3")
	     (?\$,1)3(B "f3")
	     (?\$,1)4(B "V3")
	     (?\$,1)5(B "v3")
	     (?\$,1)@(B "C3")
	     (?\$,1)A(B "c3")
	     (?\$,1)P(B "G3")
	     (?\$,1)Q(B "g3")
	     (?\$,1*4(B "AE")
	     (?\$,1*5(B "ae")
	     (?\$,1,T(B "i")
	     (?\$,1,W(B "a")
	     (?\$,1,X(B "o")
	     (?\$,1,\(B "u")
	     (?\$,1,_(B "h")
	     (?\$,1,b(B ":")
	     (?\$,1,p(B "#")
	     (?\$,1,q(B "B+")
	     (?\$,1,r(B "G+")
	     (?\$,1,s(B "D+")
	     (?\$,1,t(B "H+")
	     (?\$,1,u(B "W+")
	     (?\$,1,v(B "Z+")
	     (?\$,1,w(B "X+")
	     (?\$,1,x(B "Tj")
	     (?\$,1,y(B "J+")
	     (?\$,1,z(B "K%")
	     (?\$,1,{(B "K+")
	     (?\$,1,|(B "L+")
	     (?\$,1,}(B "M%")
	     (?\$,1,~(B "M+")
	     (?\$,1,(B "N%")
	     (?\$,1- (B "N+")
	     (?\$,1-!(B "S+")
	     (?\$,1-"(B "E+")
	     (?\$,1-#(B "P%")
	     (?\$,1-$(B "P+")
	     (?\$,1-%(B "Zj")
	     (?\$,1-&(B "ZJ")
	     (?\$,1-'(B "Q+")
	     (?\$,1-((B "R+")
	     (?\$,1-)(B "Sh")
	     (?\$,1-*(B "T+")
	     (?\$,1-0(B "v")
	     (?\$,1-1(B "oy")
	     (?\$,1-2(B "ey")
	     (?\$,1-L(B ",+")
	     (?\$,1-[(B ";+")
	     (?\$,1-_(B "?+")
	     (?\$,1-a(B "H'")
	     (?\$,1-b(B "aM")
	     (?\$,1-c(B "aH")
	     (?\$,1-d(B "wH")
	     (?\$,1-e(B "ah")
	     (?\$,1-f(B "yH")
	     (?\$,1-g(B "a+")
	     (?\$,1-h(B "b+")
	     (?\$,1-i(B "tm")
	     (?\$,1-j(B "t+")
	     (?\$,1-k(B "tk")
	     (?\$,1-l(B "g+")
	     (?\$,1-m(B "hk")
	     (?\$,1-n(B "x+")
	     (?\$,1-o(B "d+")
	     (?\$,1-p(B "dk")
	     (?\$,1-q(B "r+")
	     (?\$,1-r(B "z+")
	     (?\$,1-s(B "s+")
	     (?\$,1-t(B "sn")
	     (?\$,1-u(B "c+")
	     (?\$,1-v(B "dd")
	     (?\$,1-w(B "tj")
	     (?\$,1-x(B "zH")
	     (?\$,1-y(B "e+")
	     (?\$,1-z(B "i+")
	     (?\$,1. (B "++")
	     (?\$,1.!(B "f+")
	     (?\$,1."(B "q+")
	     (?\$,1.#(B "k+")
	     (?\$,1.$(B "l+")
	     (?\$,1.%(B "m+")
	     (?\$,1.&(B "n+")
	     (?\$,1.'(B "h+")
	     (?\$,1.((B "w+")
	     (?\$,1.)(B "j+")
	     (?\$,1.*(B "y+")
	     (?\$,1.+(B ":+")
	     (?\$,1.,(B "\"+")
	     (?\$,1.-(B "=+")
	     (?\$,1..(B "/+")
	     (?\$,1./(B "'+")
	     (?\$,1.0(B "1+")
	     (?\$,1.1(B "3+")
	     (?\$,1.2(B "0+")
	     (?\$,1.@(B "0a")
	     (?\$,1.A(B "1a")
	     (?\$,1.B(B "2a")
	     (?\$,1.C(B "3a")
	     (?\$,1.D(B "4a")
	     (?\$,1.E(B "5a")
	     (?\$,1.F(B "6a")
	     (?\$,1.G(B "7a")
	     (?\$,1.H(B "8a")
	     (?\$,1.I(B "9a")
	     (?\$,1.P(B "aS")
	     (?\$,1.^(B "p+")
	     (?\$,1.a(B "hH")
	     (?\$,1.f(B "tc")
	     (?\$,1.x(B "zj")
	     (?\$,1/$(B "v+")
	     (?\$,1//(B "gf")
	     (?\$,1/p(B "0a")
	     (?\$,1/q(B "1a")
	     (?\$,1/r(B "2a")
	     (?\$,1/s(B "3a")
	     (?\$,1/t(B "4a")
	     (?\$,1/u(B "5a")
	     (?\$,1/v(B "6a")
	     (?\$,1/w(B "7a")
	     (?\$,1/x(B "8a")
	     (?\$,1/y(B "9a")
	     (?\$,1M@(B "he")
	     (?\$,1MA(B "hu")
	     (?\$,1MB(B "hi")
	     (?\$,1MC(B "ha")
	     (?\$,1MD(B "hE")
	     (?\$,1ME(B "h")
	     (?\$,1MF(B "ho")
	     (?\$,1MH(B "le")
	     (?\$,1MI(B "lu")
	     (?\$,1MJ(B "li")
	     (?\$,1MK(B "la")
	     (?\$,1ML(B "lE")
	     (?\$,1MM(B "l")
	     (?\$,1MN(B "lo")
	     (?\$,1MO(B "lWa")
	     (?\$,1MP(B "He")
	     (?\$,1MQ(B "Hu")
	     (?\$,1MR(B "Hi")
	     (?\$,1MS(B "Ha")
	     (?\$,1MT(B "HE")
	     (?\$,1MU(B "H")
	     (?\$,1MV(B "Ho")
	     (?\$,1MW(B "HWa")
	     (?\$,1MX(B "me")
	     (?\$,1MY(B "mu")
	     (?\$,1MZ(B "mi")
	     (?\$,1M[(B "ma")
	     (?\$,1M\(B "mE")
	     (?\$,1M](B "m")
	     (?\$,1M^(B "mo")
	     (?\$,1M_(B "mWa")
	     (?\$,1M`(B "`se")
	     (?\$,1Ma(B "`su")
	     (?\$,1Mb(B "`si")
	     (?\$,1Mc(B "`sa")
	     (?\$,1Md(B "`sE")
	     (?\$,1Me(B "`s")
	     (?\$,1Mf(B "`so")
	     (?\$,1Mg(B "`sWa")
	     (?\$,1Mh(B "re")
	     (?\$,1Mi(B "ru")
	     (?\$,1Mj(B "ri")
	     (?\$,1Mk(B "ra")
	     (?\$,1Ml(B "rE")
	     (?\$,1Mm(B "r")
	     (?\$,1Mn(B "ro")
	     (?\$,1Mo(B "rWa")
	     (?\$,1Mp(B "se")
	     (?\$,1Mq(B "su")
	     (?\$,1Mr(B "si")
	     (?\$,1Ms(B "sa")
	     (?\$,1Mt(B "sE")
	     (?\$,1Mu(B "s")
	     (?\$,1Mv(B "so")
	     (?\$,1Mw(B "sWa")
	     (?\$,1Mx(B "xe")
	     (?\$,1My(B "xu")
	     (?\$,1Mz(B "xi")
	     (?\$,1M{(B "xa")
	     (?\$,1M|(B "xE")
	     (?\$,1M}(B "xa")
	     (?\$,1M~(B "xo")
	     (?\$,1M(B "xWa")
	     (?\$,1N (B "qe")
	     (?\$,1N!(B "qu")
	     (?\$,1N"(B "qi")
	     (?\$,1N#(B "qa")
	     (?\$,1N$(B "qE")
	     (?\$,1N%(B "q")
	     (?\$,1N&(B "qo")
	     (?\$,1N((B "qWe")
	     (?\$,1N*(B "qWi")
	     (?\$,1N+(B "qWa")
	     (?\$,1N,(B "qWE")
	     (?\$,1N-(B "qW")
	     (?\$,1N0(B "Qe")
	     (?\$,1N1(B "Qu")
	     (?\$,1N2(B "Qi")
	     (?\$,1N3(B "Qa")
	     (?\$,1N4(B "QE")
	     (?\$,1N5(B "Q")
	     (?\$,1N6(B "Qo")
	     (?\$,1N8(B "QWe")
	     (?\$,1N:(B "QWi")
	     (?\$,1N;(B "QWa")
	     (?\$,1N<(B "QWE")
	     (?\$,1N=(B "QW")
	     (?\$,1N@(B "be")
	     (?\$,1NA(B "bu")
	     (?\$,1NB(B "bi")
	     (?\$,1NC(B "ba")
	     (?\$,1ND(B "bE")
	     (?\$,1NE(B "b")
	     (?\$,1NF(B "bo")
	     (?\$,1NG(B "bWa")
	     (?\$,1NH(B "ve")
	     (?\$,1NI(B "vu")
	     (?\$,1NJ(B "vi")
	     (?\$,1NK(B "va")
	     (?\$,1NL(B "vE")
	     (?\$,1NM(B "v")
	     (?\$,1NN(B "vo")
	     (?\$,1NO(B "vWa")
	     (?\$,1NP(B "te")
	     (?\$,1NQ(B "tu")
	     (?\$,1NR(B "ti")
	     (?\$,1NS(B "ta")
	     (?\$,1NT(B "tE")
	     (?\$,1NU(B "t")
	     (?\$,1NV(B "to")
	     (?\$,1NW(B "tWa")
	     (?\$,1NX(B "ce")
	     (?\$,1NY(B "cu")
	     (?\$,1NZ(B "ci")
	     (?\$,1N[(B "ca")
	     (?\$,1N\(B "cE")
	     (?\$,1N](B "c")
	     (?\$,1N^(B "co")
	     (?\$,1N_(B "cWa")
	     (?\$,1N`(B "`he")
	     (?\$,1Na(B "`hu")
	     (?\$,1Nb(B "`hi")
	     (?\$,1Nc(B "`ha")
	     (?\$,1Nd(B "`hE")
	     (?\$,1Ne(B "`h")
	     (?\$,1Nf(B "`ho")
	     (?\$,1Nh(B "hWe")
	     (?\$,1Nj(B "hWi")
	     (?\$,1Nk(B "hWa")
	     (?\$,1Nl(B "hWE")
	     (?\$,1Nm(B "hW")
	     (?\$,1Np(B "na")
	     (?\$,1Nq(B "nu")
	     (?\$,1Nr(B "ni")
	     (?\$,1Ns(B "na")
	     (?\$,1Nt(B "nE")
	     (?\$,1Nu(B "n")
	     (?\$,1Nv(B "no")
	     (?\$,1Nw(B "nWa")
	     (?\$,1Nx(B "Ne")
	     (?\$,1Ny(B "Nu")
	     (?\$,1Nz(B "Ni")
	     (?\$,1N{(B "Na")
	     (?\$,1N|(B "NE")
	     (?\$,1N}(B "N")
	     (?\$,1N~(B "No")
	     (?\$,1N(B "NWa")
	     (?\$,1O (B "e")
	     (?\$,1O!(B "u")
	     (?\$,1O"(B "i")
	     (?\$,1O#(B "a")
	     (?\$,1O$(B "E")
	     (?\$,1O%(B "I")
	     (?\$,1O&(B "o")
	     (?\$,1O'(B "e3")
	     (?\$,1O((B "ke")
	     (?\$,1O)(B "ku")
	     (?\$,1O*(B "ki")
	     (?\$,1O+(B "ka")
	     (?\$,1O,(B "kE")
	     (?\$,1O-(B "k")
	     (?\$,1O.(B "ko")
	     (?\$,1O0(B "kWe")
	     (?\$,1O2(B "kWi")
	     (?\$,1O3(B "kWa")
	     (?\$,1O4(B "kWE")
	     (?\$,1O5(B "kW")
	     (?\$,1O8(B "Ke")
	     (?\$,1O9(B "Ku")
	     (?\$,1O:(B "Ki")
	     (?\$,1O;(B "Ka")
	     (?\$,1O<(B "KE")
	     (?\$,1O=(B "K")
	     (?\$,1O>(B "Ko")
	     (?\$,1O@(B "KWe")
	     (?\$,1OB(B "KWi")
	     (?\$,1OC(B "KWa")
	     (?\$,1OD(B "KWE")
	     (?\$,1OE(B "KW")
	     (?\$,1OH(B "we")
	     (?\$,1OI(B "wu")
	     (?\$,1OJ(B "wi")
	     (?\$,1OK(B "wa")
	     (?\$,1OL(B "wE")
	     (?\$,1OM(B "w")
	     (?\$,1ON(B "wo")
	     (?\$,1OP(B "`e")
	     (?\$,1OQ(B "`u")
	     (?\$,1OR(B "`i")
	     (?\$,1OS(B "`a")
	     (?\$,1OT(B "`E")
	     (?\$,1OU(B "`I")
	     (?\$,1OV(B "`o")
	     (?\$,1OX(B "ze")
	     (?\$,1OY(B "zu")
	     (?\$,1OZ(B "zi")
	     (?\$,1O[(B "za")
	     (?\$,1O\(B "zE")
	     (?\$,1O](B "z")
	     (?\$,1O^(B "zo")
	     (?\$,1O_(B "zWa")
	     (?\$,1O`(B "Ze")
	     (?\$,1Oa(B "Zu")
	     (?\$,1Ob(B "Zi")
	     (?\$,1Oc(B "Za")
	     (?\$,1Od(B "ZE")
	     (?\$,1Oe(B "Z")
	     (?\$,1Of(B "Zo")
	     (?\$,1Og(B "ZWa")
	     (?\$,1Oh(B "ye")
	     (?\$,1Oi(B "yu")
	     (?\$,1Oj(B "yi")
	     (?\$,1Ok(B "ya")
	     (?\$,1Ol(B "yE")
	     (?\$,1Om(B "y")
	     (?\$,1On(B "yo")
	     (?\$,1Oo(B "yWa")
	     (?\$,1Op(B "de")
	     (?\$,1Oq(B "du")
	     (?\$,1Or(B "di")
	     (?\$,1Os(B "da")
	     (?\$,1Ot(B "dE")
	     (?\$,1Ou(B "d")
	     (?\$,1Ov(B "do")
	     (?\$,1Ow(B "dWa")
	     (?\$,1Ox(B "De")
	     (?\$,1Oy(B "Du")
	     (?\$,1Oz(B "Di")
	     (?\$,1O{(B "Da")
	     (?\$,1O|(B "DE")
	     (?\$,1O}(B "D")
	     (?\$,1O~(B "Do")
	     (?\$,1O(B "DWa")
	     (?\$,1P (B "je")
	     (?\$,1P!(B "ju")
	     (?\$,1P"(B "ji")
	     (?\$,1P#(B "ja")
	     (?\$,1P$(B "jE")
	     (?\$,1P%(B "j")
	     (?\$,1P&(B "jo")
	     (?\$,1P'(B "jWa")
	     (?\$,1P((B "ga")
	     (?\$,1P)(B "gu")
	     (?\$,1P*(B "gi")
	     (?\$,1P+(B "ga")
	     (?\$,1P,(B "gE")
	     (?\$,1P-(B "g")
	     (?\$,1P.(B "go")
	     (?\$,1P0(B "gWu")
	     (?\$,1P2(B "gWi")
	     (?\$,1P3(B "gWa")
	     (?\$,1P4(B "gWE")
	     (?\$,1P5(B "gW")
	     (?\$,1P8(B "Ge")
	     (?\$,1P9(B "Gu")
	     (?\$,1P:(B "Gi")
	     (?\$,1P;(B "Ga")
	     (?\$,1P<(B "GE")
	     (?\$,1P=(B "G")
	     (?\$,1P>(B "Go")
	     (?\$,1P?(B "GWa")
	     (?\$,1P@(B "Te")
	     (?\$,1PA(B "Tu")
	     (?\$,1PB(B "Ti")
	     (?\$,1PC(B "Ta")
	     (?\$,1PD(B "TE")
	     (?\$,1PE(B "T")
	     (?\$,1PF(B "To")
	     (?\$,1PG(B "TWa")
	     (?\$,1PH(B "Ce")
	     (?\$,1PI(B "Ca")
	     (?\$,1PJ(B "Cu")
	     (?\$,1PK(B "Ca")
	     (?\$,1PL(B "CE")
	     (?\$,1PM(B "C")
	     (?\$,1PN(B "Co")
	     (?\$,1PO(B "CWa")
	     (?\$,1PP(B "Pe")
	     (?\$,1PQ(B "Pu")
	     (?\$,1PR(B "Pi")
	     (?\$,1PS(B "Pa")
	     (?\$,1PT(B "PE")
	     (?\$,1PU(B "P")
	     (?\$,1PV(B "Po")
	     (?\$,1PW(B "PWa")
	     (?\$,1PX(B "SWe")
	     (?\$,1PY(B "SWu")
	     (?\$,1PZ(B "SWi")
	     (?\$,1P[(B "SWa")
	     (?\$,1P\(B "SWE")
	     (?\$,1P](B "SW")
	     (?\$,1P^(B "SWo")
	     (?\$,1P_(B "SWa")
	     (?\$,1P`(B "`Sa")
	     (?\$,1Pa(B "`Su")
	     (?\$,1Pb(B "`Si")
	     (?\$,1Pc(B "`Sa")
	     (?\$,1Pd(B "`SE")
	     (?\$,1Pe(B "`S")
	     (?\$,1Pf(B "`So")
	     (?\$,1Ph(B "fa")
	     (?\$,1Pi(B "fu")
	     (?\$,1Pj(B "fi")
	     (?\$,1Pk(B "fa")
	     (?\$,1Pl(B "fE")
	     (?\$,1Pm(B "o")
	     (?\$,1Pn(B "fo")
	     (?\$,1Po(B "fWa")
	     (?\$,1Pp(B "pe")
	     (?\$,1Pq(B "pu")
	     (?\$,1Pr(B "pi")
	     (?\$,1Ps(B "pa")
	     (?\$,1Pt(B "pE")
	     (?\$,1Pu(B "p")
	     (?\$,1Pv(B "po")
	     (?\$,1Pw(B "pWa")
	     (?\$,1Px(B "mYa")
	     (?\$,1Py(B "rYa")
	     (?\$,1Pz(B "fYa")
	     (?\$,1Q (B " ")
	     (?\$,1Q!(B ":")
	     (?\$,1Q"(B "::")
	     (?\$,1Q#(B ",")
	     (?\$,1Q$(B ";")
	     (?\$,1Q%(B "-:")
	     (?\$,1Q&(B ":-")
	     (?\$,1Q'(B "`?")
	     (?\$,1Q((B ":|:")
	     (?\$,1Q)(B "`1")
	     (?\$,1Q*(B "`2")
	     (?\$,1Q+(B "`3")
	     (?\$,1Q,(B "`4")
	     (?\$,1Q-(B "`5")
	     (?\$,1Q.(B "`6")
	     (?\$,1Q/(B "`7")
	     (?\$,1Q0(B "`8")
	     (?\$,1Q1(B "`9")
	     (?\$,1Q2(B "`10")
	     (?\$,1Q3(B "`20")
	     (?\$,1Q4(B "`30")
	     (?\$,1Q5(B "`40")
	     (?\$,1Q6(B "`50")
	     (?\$,1Q7(B "`60")
	     (?\$,1Q8(B "`70")
	     (?\$,1Q9(B "`80")
	     (?\$,1Q:(B "`90")
	     (?\$,1Q;(B "`100")
	     (?\$,1Q<(B "`10000")
	     (?\$,1m@(B "A-0")
	     (?\$,1mA(B "a-0")
	     (?\$,1mB(B "B.")
	     (?\$,1mC(B "b.")
	     (?\$,1mD(B "B-.")
	     (?\$,1mE(B "b-.")
	     (?\$,1mF(B "B_")
	     (?\$,1mG(B "b_")
	     (?\$,1mH(B "C,'")
	     (?\$,1mI(B "c,'")
	     (?\$,1mJ(B "D.")
	     (?\$,1mK(B "d.")
	     (?\$,1mL(B "D-.")
	     (?\$,1mM(B "d-.")
	     (?\$,1mN(B "D_")
	     (?\$,1mO(B "d_")
	     (?\$,1mP(B "D,")
	     (?\$,1mQ(B "d,")
	     (?\$,1mR(B "D->")
	     (?\$,1mS(B "d->")
	     (?\$,1mT(B "E-!")
	     (?\$,1mU(B "e-!")
	     (?\$,1mV(B "E-'")
	     (?\$,1mW(B "e-'")
	     (?\$,1mX(B "E->")
	     (?\$,1mY(B "e->")
	     (?\$,1mZ(B "E-?")
	     (?\$,1m[(B "e-?")
	     (?\$,1m\(B "E,(")
	     (?\$,1m](B "e,(")
	     (?\$,1m^(B "F.")
	     (?\$,1m_(B "f.")
	     (?\$,1m`(B "G-")
	     (?\$,1ma(B "g-")
	     (?\$,1mb(B "H.")
	     (?\$,1mc(B "h.")
	     (?\$,1md(B "H-.")
	     (?\$,1me(B "h-.")
	     (?\$,1mf(B "H:")
	     (?\$,1mg(B "h:")
	     (?\$,1mh(B "H,")
	     (?\$,1mi(B "h,")
	     (?\$,1mj(B "H-(")
	     (?\$,1mk(B "h-(")
	     (?\$,1ml(B "I-?")
	     (?\$,1mm(B "i-?")
	     (?\$,1mn(B "I:'")
	     (?\$,1mo(B "i:'")
	     (?\$,1mp(B "K'")
	     (?\$,1mq(B "k'")
	     (?\$,1mr(B "K-.")
	     (?\$,1ms(B "k-.")
	     (?\$,1mt(B "K_")
	     (?\$,1mu(B "k_")
	     (?\$,1mv(B "L-.")
	     (?\$,1mw(B "l-.")
	     (?\$,1mx(B "L--.")
	     (?\$,1my(B "l--.")
	     (?\$,1mz(B "L_")
	     (?\$,1m{(B "l_")
	     (?\$,1m|(B "L->")
	     (?\$,1m}(B "l->")
	     (?\$,1m~(B "M'")
	     (?\$,1m(B "m'")
	     (?\$,1n (B "M.")
	     (?\$,1n!(B "m.")
	     (?\$,1n"(B "M-.")
	     (?\$,1n#(B "m-.")
	     (?\$,1n$(B "N.")
	     (?\$,1n%(B "n.")
	     (?\$,1n&(B "N-.")
	     (?\$,1n'(B "n-.")
	     (?\$,1n((B "N_")
	     (?\$,1n)(B "n_")
	     (?\$,1n*(B "N->")
	     (?\$,1n+(B "n->")
	     (?\$,1n,(B "O?'")
	     (?\$,1n-(B "o?'")
	     (?\$,1n.(B "O?:")
	     (?\$,1n/(B "o?:")
	     (?\$,1n0(B "O-!")
	     (?\$,1n1(B "o-!")
	     (?\$,1n2(B "O-'")
	     (?\$,1n3(B "o-'")
	     (?\$,1n4(B "P'")
	     (?\$,1n5(B "p'")
	     (?\$,1n6(B "P.")
	     (?\$,1n7(B "p.")
	     (?\$,1n8(B "R.")
	     (?\$,1n9(B "r.")
	     (?\$,1n:(B "R-.")
	     (?\$,1n;(B "r-.")
	     (?\$,1n<(B "R--.")
	     (?\$,1n=(B "r--.")
	     (?\$,1n>(B "R_")
	     (?\$,1n?(B "r_")
	     (?\$,1n@(B "S.")
	     (?\$,1nA(B "s.")
	     (?\$,1nB(B "S-.")
	     (?\$,1nC(B "s-.")
	     (?\$,1nD(B "S'.")
	     (?\$,1nE(B "s'.")
	     (?\$,1nF(B "S<.")
	     (?\$,1nG(B "s<.")
	     (?\$,1nH(B "S.-.")
	     (?\$,1nI(B "s.-.")
	     (?\$,1nJ(B "T.")
	     (?\$,1nK(B "t.")
	     (?\$,1nL(B "T-.")
	     (?\$,1nM(B "t-.")
	     (?\$,1nN(B "T_")
	     (?\$,1nO(B "t_")
	     (?\$,1nP(B "T->")
	     (?\$,1nQ(B "t->")
	     (?\$,1nR(B "U--:")
	     (?\$,1nS(B "u--:")
	     (?\$,1nT(B "U-?")
	     (?\$,1nU(B "u-?")
	     (?\$,1nV(B "U->")
	     (?\$,1nW(B "u->")
	     (?\$,1nX(B "U?'")
	     (?\$,1nY(B "u?'")
	     (?\$,1nZ(B "U-:")
	     (?\$,1n[(B "u-:")
	     (?\$,1n\(B "V?")
	     (?\$,1n](B "v?")
	     (?\$,1n^(B "V-.")
	     (?\$,1n_(B "v-.")
	     (?\$,1n`(B "W!")
	     (?\$,1na(B "w!")
	     (?\$,1nb(B "W'")
	     (?\$,1nc(B "w'")
	     (?\$,1nd(B "W:")
	     (?\$,1ne(B "w:")
	     (?\$,1nf(B "W.")
	     (?\$,1ng(B "w.")
	     (?\$,1nh(B "W-.")
	     (?\$,1ni(B "w-.")
	     (?\$,1nj(B "X.")
	     (?\$,1nk(B "x.")
	     (?\$,1nl(B "X:")
	     (?\$,1nm(B "x:")
	     (?\$,1nn(B "Y.")
	     (?\$,1no(B "y.")
	     (?\$,1np(B "Z>")
	     (?\$,1nq(B "z>")
	     (?\$,1nr(B "Z-.")
	     (?\$,1ns(B "z-.")
	     (?\$,1nt(B "Z_")
	     (?\$,1nu(B "z_")
	     (?\$,1nv(B "h_")
	     (?\$,1nw(B "t:")
	     (?\$,1nx(B "w0")
	     (?\$,1ny(B "y0")
	     (?\$,1o (B "A-.")
	     (?\$,1o!(B "a-.")
	     (?\$,1o"(B "A2")
	     (?\$,1o#(B "a2")
	     (?\$,1o$(B "A>'")
	     (?\$,1o%(B "a>'")
	     (?\$,1o&(B "A>!")
	     (?\$,1o'(B "a>!")
	     (?\$,1o((B "A>2")
	     (?\$,1o)(B "a>2")
	     (?\$,1o*(B "A>?")
	     (?\$,1o+(B "a>?")
	     (?\$,1o,(B "A>-.")
	     (?\$,1o-(B "a>-.")
	     (?\$,1o.(B "A('")
	     (?\$,1o/(B "a('")
	     (?\$,1o0(B "A(!")
	     (?\$,1o1(B "a(!")
	     (?\$,1o2(B "A(2")
	     (?\$,1o3(B "a(2")
	     (?\$,1o4(B "A(?")
	     (?\$,1o5(B "a(?")
	     (?\$,1o6(B "A(-.")
	     (?\$,1o7(B "a(-.")
	     (?\$,1o8(B "E-.")
	     (?\$,1o9(B "e-.")
	     (?\$,1o:(B "E2")
	     (?\$,1o;(B "e2")
	     (?\$,1o<(B "E?")
	     (?\$,1o=(B "e?")
	     (?\$,1o>(B "E>'")
	     (?\$,1o?(B "e>'")
	     (?\$,1o@(B "E>!")
	     (?\$,1oA(B "e>!")
	     (?\$,1oB(B "E>2")
	     (?\$,1oC(B "e>2")
	     (?\$,1oD(B "E>?")
	     (?\$,1oE(B "e>?")
	     (?\$,1oF(B "E>-.")
	     (?\$,1oG(B "e>-.")
	     (?\$,1oH(B "I2")
	     (?\$,1oI(B "i2")
	     (?\$,1oJ(B "I-.")
	     (?\$,1oK(B "i-.")
	     (?\$,1oL(B "O-.")
	     (?\$,1oM(B "o-.")
	     (?\$,1oN(B "O2")
	     (?\$,1oO(B "o2")
	     (?\$,1oP(B "O>'")
	     (?\$,1oQ(B "o>'")
	     (?\$,1oR(B "O>!")
	     (?\$,1oS(B "o>!")
	     (?\$,1oT(B "O>2")
	     (?\$,1oU(B "o>2")
	     (?\$,1oV(B "O>?")
	     (?\$,1oW(B "o>?")
	     (?\$,1oX(B "O>-.")
	     (?\$,1oY(B "o>-.")
	     (?\$,1oZ(B "O9'")
	     (?\$,1o[(B "o9'")
	     (?\$,1o\(B "O9!")
	     (?\$,1o](B "o9!")
	     (?\$,1o^(B "O92")
	     (?\$,1o_(B "o92")
	     (?\$,1o`(B "O9?")
	     (?\$,1oa(B "o9?")
	     (?\$,1ob(B "O9-.")
	     (?\$,1oc(B "o9-.")
	     (?\$,1od(B "U-.")
	     (?\$,1oe(B "u-.")
	     (?\$,1of(B "U2")
	     (?\$,1og(B "u2")
	     (?\$,1oh(B "U9'")
	     (?\$,1oi(B "u9'")
	     (?\$,1oj(B "U9!")
	     (?\$,1ok(B "u9!")
	     (?\$,1ol(B "U92")
	     (?\$,1om(B "u92")
	     (?\$,1on(B "U9?")
	     (?\$,1oo(B "u9?")
	     (?\$,1op(B "U9-.")
	     (?\$,1oq(B "u9-.")
	     (?\$,1or(B "Y!")
	     (?\$,1os(B "y!")
	     (?\$,1ot(B "Y-.")
	     (?\$,1ou(B "y-.")
	     (?\$,1ov(B "Y2")
	     (?\$,1ow(B "y2")
	     (?\$,1ox(B "Y?")
	     (?\$,1oy(B "y?")
	     (?\$,1p (B "a")
	     (?\$,1p!(B "ha")
	     (?\$,1p"(B "`a")
	     (?\$,1p#(B "h`a")
	     (?\$,1p$(B "a'")
	     (?\$,1p%(B "ha'")
	     (?\$,1p&(B "a~")
	     (?\$,1p'(B "ha~")
	     (?\$,1p((B "A")
	     (?\$,1p)(B "hA")
	     (?\$,1p*(B "`A")
	     (?\$,1p+(B "h`A")
	     (?\$,1p,(B "A'")
	     (?\$,1p-(B "hA'")
	     (?\$,1p.(B "A~")
	     (?\$,1p/(B "hA~")
	     (?\$,1p1(B "he")
	     (?\$,1p9(B "hE")
	     (?\$,1pQ(B "hi")
	     (?\$,1pY(B "hI")
	     (?\$,1pa(B "ho")
	     (?\$,1pi(B "hO")
	     (?\$,1pq(B "hu")
	     (?\$,1py(B "hU")
	     (?\$,1q(B ",,")
	     (?\$,1r (B "?*")
	     (?\$,1r!(B "?:")
	     (?\$,1r-(B ",!")
	     (?\$,1r.(B ",'")
	     (?\$,1r/(B "?,")
	     (?\$,1r=(B ";!")
	     (?\$,1r>(B ";'")
	     (?\$,1r?(B "?;")
	     (?\$,1rE(B "rh")
	     (?\$,1rL(B "Rh")
	     (?\$,1rM(B "!:")
	     (?\$,1rO(B "!*")
	     (?\$,1r^(B ";;")
	     (?\$,1r`(B " ")
	     (?\$,1ra(B "  ")
	     (?\$,1rb(B " ")
	     (?\$,1rc(B "  ")
	     (?\$,1rd(B " ")
	     (?\$,1re(B " ")
	     (?\$,1rf(B " ")
	     (?\$,1rg(B ",A (B")
	     (?\$,1rh(B " ")
	     (?\$,1ri(B " ")
	     (?\$,1rp(B "-")
	     (?\$,1rq(B "-")
	     (?\$,1rs(B "-")
	     (?\$,1rt(B "--")
	     (?\$,1ru(B "-")
	     (?\$,1rv(B "||")
	     (?\$,1rw(B "=2")
	     (?\$,1rx(B "`")
	     (?\$,1ry(B "'")
	     (?\$,1rz(B "'")
	     (?\$,1r{(B "'")
	     (?\$,1r|(B "\"")
	     (?\$,1r}(B "\"")
	     (?\$,1r~(B "\"")
	     (?\$,1r(B "\"")
	     (?\$,1s (B "/-")
	     (?\$,1s!(B "/=")
	     (?\$,1s"(B " o ")
	     (?\$,1s$(B ".")
	     (?\$,1s%(B "..")
	     (?\$,1s&(B "...")
	     (?\$,1s'(B ",A7(B")
	     (?\$,1s0(B " 0/00")
	     (?\$,1s2(B "'")
	     (?\$,1s3(B "''")
	     (?\$,1s4(B "'''")
	     (?\$,1s5(B "`")
	     (?\$,1s6(B "``")
	     (?\$,1s7(B "```")
	     (?\$,1s8(B "Ca")
	     (?\$,1s9(B "<")
	     (?\$,1s:(B ">")
	     (?\$,1s;(B ":X")
	     (?\$,1s<(B "!!")
	     (?\$,1s>(B "'-")
	     (?\$,1sC(B "-")
	     (?\$,1sD(B "/")
	     (?\$,1sH(B "?!")
	     (?\$,1sI(B "!?")
	     (?\$,1sp(B "^0")
	     (?\$,1st(B "^4")
	     (?\$,1su(B "^5")
	     (?\$,1sv(B "^6")
	     (?\$,1sw(B "^7")
	     (?\$,1sx(B "^8")
	     (?\$,1sy(B "^9")
	     (?\$,1sz(B "^+")
	     (?\$,1s{(B "^-")
	     (?\$,1s|(B "^=")
	     (?\$,1s}(B "^(")
	     (?\$,1s~(B "^)")
	     (?\$,1s(B "^n")
	     (?\$,1t (B "_0")
	     (?\$,1t!(B "_1")
	     (?\$,1t"(B "_2")
	     (?\$,1t#(B "_3")
	     (?\$,1t$(B "_4")
	     (?\$,1t%(B "_5")
	     (?\$,1t&(B "_6")
	     (?\$,1t'(B "_7")
	     (?\$,1t((B "_8")
	     (?\$,1t)(B "_9")
	     (?\$,1t*(B "_+")
	     (?\$,1t+(B "_-")
	     (?\$,1t,(B "_=")
	     (?\$,1t-(B "(")
	     (?\$,1t.(B ")")
	     (?\$,1tC(B "Ff")
	     (?\$,1tD(B "Li")
	     (?\$,1tG(B "Pt")
	     (?\$,1tI(B "W=")
	     (?\$,1tL(B "EUR")
	     (?\$,1u@(B "a/c")
	     (?\$,1uA(B "a/s")
	     (?\$,1uC(B "oC")
	     (?\$,1uE(B "c/o")
	     (?\$,1uF(B "c/u")
	     (?\$,1uI(B "oF")
	     (?\$,1uJ(B "g")
	     (?\$,1uN(B "h")
	     (?\$,1uO(B "\\hbar")
	     (?\$,1uQ(B "Im")
	     (?\$,1uS(B "l")
	     (?\$,1uV(B "No.")
	     (?\$,1uW(B "PO")
	     (?\$,1uX(B "P")
	     (?\$,1u\(B "Re")
	     (?\$,1u^(B "Rx")
	     (?\$,1u`(B "(SM)")
	     (?\$,1ua(B "TEL")
	     (?\$,1ub(B "(TM)")
	     (?\$,1uf(B "Ohm")
	     (?\$,1uj(B "K")
	     (?\$,1uk(B "Ang.")
	     (?\$,1un(B "est.")
	     (?\$,1ut(B "o")
	     (?\$,1uu(B "Aleph ")
	     (?\$,1uv(B "Bet ")
	     (?\$,1uw(B "Gimel ")
	     (?\$,1ux(B "Dalet ")
	     (?\$,1v3(B " 1/3")
	     (?\$,1v4(B " 2/3")
	     (?\$,1v5(B " 1/5")
	     (?\$,1v6(B " 2/5")
	     (?\$,1v7(B " 3/5")
	     (?\$,1v8(B " 4/5")
	     (?\$,1v9(B " 1/6")
	     (?\$,1v:(B " 5/6")
	     (?\$,1v;(B " 1/8")
	     (?\$,1v<(B " 3/8")
	     (?\$,1v=(B " 5/8")
	     (?\$,1v>(B " 7/8")
	     (?\$,1v?(B " 1/")
	     (?\$,1v@(B "I")
	     (?\$,1vA(B "II")
	     (?\$,1vB(B "III")
	     (?\$,1vC(B "IV")
	     (?\$,1vD(B "V")
	     (?\$,1vE(B "VI")
	     (?\$,1vF(B "VII")
	     (?\$,1vG(B "VIII")
	     (?\$,1vH(B "IX")
	     (?\$,1vI(B "X")
	     (?\$,1vJ(B "XI")
	     (?\$,1vK(B "XII")
	     (?\$,1vL(B "L")
	     (?\$,1vM(B "C")
	     (?\$,1vN(B "D")
	     (?\$,1vO(B "M")
	     (?\$,1vP(B "i")
	     (?\$,1vQ(B "ii")
	     (?\$,1vR(B "iii")
	     (?\$,1vS(B "iv")
	     (?\$,1vT(B "v")
	     (?\$,1vU(B "vi")
	     (?\$,1vV(B "vii")
	     (?\$,1vW(B "viii")
	     (?\$,1vX(B "ix")
	     (?\$,1vY(B "x")
	     (?\$,1vZ(B "xi")
	     (?\$,1v[(B "xii")
	     (?\$,1v\(B "l")
	     (?\$,1v](B "c")
	     (?\$,1v^(B "d")
	     (?\$,1v_(B "m")
	     (?\$,1v`(B "1000RCD")
	     (?\$,1va(B "5000R")
	     (?\$,1vb(B "10000R")
	     (?\$,1vp(B "<-")
	     (?\$,1vq(B "-^")
	     (?\$,1vr(B "->")
	     (?\$,1vs(B "-v")
	     (?\$,1vt(B "<->")
	     (?\$,1vu(B "UD")
	     (?\$,1vv(B "<!!")
	     (?\$,1vw(B "//>")
	     (?\$,1vx(B "!!>")
	     (?\$,1vy(B "<//")
	     (?\$,1w((B "UD-")
	     (?\$,1w5(B "RET")
	     (?\$,1w@(B ">V")
	     (?\$,1wP(B "<=")
	     (?\$,1wQ(B "^^")
	     (?\$,1wR(B "=>")
	     (?\$,1wS(B "vv")
	     (?\$,1wT(B "<=>")
	     (?\$,1x (B "FA")
	     (?\$,1x"(B "\\partial")
	     (?\$,1x#(B "TE")
	     (?\$,1x%(B "{}")
	     (?\$,1x&(B "Delta")
	     (?\$,1x'(B "Nabla")
	     (?\$,1x((B "(-")
	     (?\$,1x)(B "!(-")
	     (?\$,1x*(B "(-")
	     (?\$,1x+(B "-)")
	     (?\$,1x,(B "!-)")
	     (?\$,1x-(B "-)")
	     (?\$,1x.(B " qed")
	     (?\$,1x/(B "\\prod")
	     (?\$,1x1(B "\\sum")
	     (?\$,1x2(B " -")
	     (?\$,1x3(B "-/+")
	     (?\$,1x4(B ".+")
	     (?\$,1x5(B "/")
	     (?\$,1x6(B " - ")
	     (?\$,1x7(B "*")
	     (?\$,1x8(B " ,A0(B ")
	     (?\$,1x9(B "sb")
	     (?\$,1x:(B " SQRT ")
	     (?\$,1x;(B " ROOT,A3(B ")
	     (?\$,1x<(B " ROOT4 ")
	     (?\$,1x=(B "0(")
	     (?\$,1x>(B "infty")
	     (?\$,1x?(B "-L")
	     (?\$,1x@(B "-V")
	     (?\$,1xE(B "PP")
	     (?\$,1xF(B " !PP ")
	     (?\$,1xG(B "AND")
	     (?\$,1xH(B "OR")
	     (?\$,1xI(B "(U")
	     (?\$,1xJ(B ")U")
	     (?\$,1xK(B "\int ")
	     (?\$,1xL(B "DI")
	     (?\$,1xN(B "Io")
	     (?\$,1xT(B ".:")
	     (?\$,1xU(B ":.")
	     (?\$,1xV(B ":R")
	     (?\$,1xW(B "::")
	     (?\$,1x\(B "?1")
	     (?\$,1x^(B "CG")
	     (?\$,1xc(B "?-")
	     (?\$,1xe(B "?=")
	     (?\$,1xh(B "~=")
	     (?\$,1xi(B " !~= ")
	     (?\$,1xl(B "=?")
	     (?\$,1xs(B "HI")
	     (?\$,1xt(B ":=")
	     (?\$,1xu(B "=:")
	     (?\$,1y (B "!=")
	     (?\$,1y!(B "=3")
	     (?\$,1y"(B " !=3 ")
	     (?\$,1y$(B "=<")
	     (?\$,1y%(B ">=")
	     (?\$,1y&(B ".LE.")
	     (?\$,1y'(B ".GE.")
	     (?\$,1y((B ".LT.NOT.EQ.")
	     (?\$,1y)(B ".GT.NOT.EQ.")
	     (?\$,1y*(B "<<")
	     (?\$,1y+(B ">>")
	     (?\$,1y.(B "!<")
	     (?\$,1y/(B "!>")
	     (?\$,1y6(B " <> ")
	     (?\$,1y7(B " >< ")
	     (?\$,1yB(B "(C")
	     (?\$,1yC(B ")C")
	     (?\$,1yD(B " !(C ")
	     (?\$,1yE(B " !)C ")
	     (?\$,1yF(B "(_")
	     (?\$,1yG(B ")_")
	     (?\$,1yU(B "(+)")
	     (?\$,1yV(B "(-)")
	     (?\$,1yW(B "(,AW(B)")
	     (?\$,1yX(B "(/)")
	     (?\$,1yY(B "(,A7(B)")
	     (?\$,1yZ(B "(,A0(B)")
	     (?\$,1y[(B "(*)")
	     (?\$,1y\(B "(=)")
	     (?\$,1y](B "(-)")
	     (?\$,1y^(B "[+]")
	     (?\$,1y_(B "[-]")
	     (?\$,1y`(B "[,AW(B]")
	     (?\$,1ya(B "[,A7(B]")
	     (?\$,1ye(B "-T")
	     (?\$,1yg(B " MODELS ")
	     (?\$,1yh(B " TRUE ")
	     (?\$,1yi(B " FORCES ")
	     (?\$,1yl(B " !PROVES ")
	     (?\$,1ym(B " NOT TRUE ")
	     (?\$,1yn(B " !FORCES ")
	     (?\$,1yr(B " NORMAL SUBGROUP OF ")
	     (?\$,1ys(B " CONTAINS AS NORMAL SUBGROUP ")
	     (?\$,1yt(B " NORMAL SUBGROUP OF OR EQUAL TO ")
	     (?\$,1yu(B " CONTAINS AS NORMAL SUBGROUP OR EQUAL TO ")
	     (?\$,1yx(B " MULTIMAP ")
	     (?\$,1yz(B " INTERCALATE ")
	     (?\$,1y{(B " XOR ")
	     (?\$,1y|(B " NAND ")
	     (?\$,1z%(B " ,A7(B ")
	     (?\$,1z6(B "<.")
	     (?\$,1z7(B ">.")
	     (?\$,1z8(B "<<<")
	     (?\$,1z9(B ">>>")
	     (?\$,1zN(B ":3")
	     (?\$,1zO(B ".3")
	     (?\$,1zb(B "Eh")
	     (?\$,1zg(B "~~")
	     (?\$,1zh(B "<7")
	     (?\$,1zi(B ">7")
	     (?\$,1zj(B "7<")
	     (?\$,1zk(B "7>")
	     (?\$,1zp(B "NI")
	     (?\$,1zr(B "(A")
	     (?\$,1zu(B "TR")
	     (?\$,1zx(B "88")
	     (?\$,1{ (B "Iu")
	     (?\$,1{!(B "Il")
	     (?\$,1{"(B ":(")
	     (?\$,1{#(B ":)")
	     (?\$,1{$(B "|^|")
	     (?\$,1{'(B "[X]")
	     (?\$,1{)(B "</")
	     (?\$,1{*(B "/>")
	     (?\$,1}c(B "Vs")
	     (?\$,1~ (B "1h")
	     (?\$,1~!(B "3h")
	     (?\$,1~"(B "2h")
	     (?\$,1~#(B "4h")
	     (?\$,1~&(B "1j")
	     (?\$,1~'(B "2j")
	     (?\$,1~((B "3j")
	     (?\$,1~)(B "4j")
	     (?\$,1~@(B "1-o")
	     (?\$,1~A(B "2-o")
	     (?\$,1~B(B "3-o")
	     (?\$,1~C(B "4-o")
	     (?\$,1~D(B "5-o")
	     (?\$,1~E(B "6-o")
	     (?\$,1~F(B "7-o")
	     (?\$,1~G(B "8-o")
	     (?\$,1~H(B "9-o")
	     (?\$,1~I(B "10-o")
	     (?\$,1~J(B "11-o")
	     (?\$,1~K(B "12-o")
	     (?\$,1~L(B "13-o")
	     (?\$,1~M(B "14-o")
	     (?\$,1~N(B "15-o")
	     (?\$,1~O(B "16-o")
	     (?\$,1~P(B "17-o")
	     (?\$,1~Q(B "18-o")
	     (?\$,1~R(B "19-o")
	     (?\$,1~S(B "20-o")
	     (?\$,1~T(B "(1)")
	     (?\$,1~U(B "(2)")
	     (?\$,1~V(B "(3)")
	     (?\$,1~W(B "(4)")
	     (?\$,1~X(B "(5)")
	     (?\$,1~Y(B "(6)")
	     (?\$,1~Z(B "(7)")
	     (?\$,1~[(B "(8)")
	     (?\$,1~\(B "(9)")
	     (?\$,1~](B "(10)")
	     (?\$,1~^(B "(11)")
	     (?\$,1~_(B "(12)")
	     (?\$,1~`(B "(13)")
	     (?\$,1~a(B "(14)")
	     (?\$,1~b(B "(15)")
	     (?\$,1~c(B "(16)")
	     (?\$,1~d(B "(17)")
	     (?\$,1~e(B "(18)")
	     (?\$,1~f(B "(19)")
	     (?\$,1~g(B "(20)")
	     (?\$,1~h(B "1.")
	     (?\$,1~i(B "2.")
	     (?\$,1~j(B "3.")
	     (?\$,1~k(B "4.")
	     (?\$,1~l(B "5.")
	     (?\$,1~m(B "6.")
	     (?\$,1~n(B "7.")
	     (?\$,1~o(B "8.")
	     (?\$,1~p(B "9.")
	     (?\$,1~q(B "10.")
	     (?\$,1~r(B "11.")
	     (?\$,1~s(B "12.")
	     (?\$,1~t(B "13.")
	     (?\$,1~u(B "14.")
	     (?\$,1~v(B "15.")
	     (?\$,1~w(B "16.")
	     (?\$,1~x(B "17.")
	     (?\$,1~y(B "18.")
	     (?\$,1~z(B "19.")
	     (?\$,1~{(B "20.")
	     (?\$,1~|(B "(a)")
	     (?\$,1~}(B "(b)")
	     (?\$,1~~(B "(c)")
	     (?\$,1~(B "(d)")
	     (?\$,1 (B "(e)")
	     (?\$,1!(B "(f)")
	     (?\$,1"(B "(g)")
	     (?\$,1#(B "(h)")
	     (?\$,1$(B "(i)")
	     (?\$,1%(B "(j)")
	     (?\$,1&(B "(k)")
	     (?\$,1'(B "(l)")
	     (?\$,1((B "(m)")
	     (?\$,1)(B "(n)")
	     (?\$,1*(B "(o)")
	     (?\$,1+(B "(p)")
	     (?\$,1,(B "(q)")
	     (?\$,1-(B "(r)")
	     (?\$,1.(B "(s)")
	     (?\$,1/(B "(t)")
	     (?\$,10(B "(u)")
	     (?\$,11(B "(v)")
	     (?\$,12(B "(w)")
	     (?\$,13(B "(x)")
	     (?\$,14(B "(y)")
	     (?\$,15(B "(z)")
	     (?\$,16(B "A-o")
	     (?\$,17(B "B-o")
	     (?\$,18(B "C-o")
	     (?\$,19(B "D-o")
	     (?\$,1:(B "E-o")
	     (?\$,1;(B "F-o")
	     (?\$,1<(B "G-o")
	     (?\$,1=(B "H-o")
	     (?\$,1>(B "I-o")
	     (?\$,1?(B "J-o")
	     (?\$,1@(B "K-o")
	     (?\$,1A(B "L-o")
	     (?\$,1B(B "M-o")
	     (?\$,1C(B "N-o")
	     (?\$,1D(B "O-o")
	     (?\$,1E(B "P-o")
	     (?\$,1F(B "Q-o")
	     (?\$,1G(B "R-o")
	     (?\$,1H(B "S-o")
	     (?\$,1I(B "T-o")
	     (?\$,1J(B "U-o")
	     (?\$,1K(B "V-o")
	     (?\$,1L(B "W-o")
	     (?\$,1M(B "X-o")
	     (?\$,1N(B "Y-o")
	     (?\$,1O(B "Z-o")
	     (?\$,1P(B "a-o")
	     (?\$,1Q(B "b-o")
	     (?\$,1R(B "c-o")
	     (?\$,1S(B "d-o")
	     (?\$,1T(B "e-o")
	     (?\$,1U(B "f-o")
	     (?\$,1V(B "g-o")
	     (?\$,1W(B "h-o")
	     (?\$,1X(B "i-o")
	     (?\$,1Y(B "j-o")
	     (?\$,1Z(B "k-o")
	     (?\$,1[(B "l-o")
	     (?\$,1\(B "m-o")
	     (?\$,1](B "n-o")
	     (?\$,1^(B "o-o")
	     (?\$,1_(B "p-o")
	     (?\$,1`(B "q-o")
	     (?\$,1a(B "r-o")
	     (?\$,1b(B "s-o")
	     (?\$,1c(B "t-o")
	     (?\$,1d(B "u-o")
	     (?\$,1e(B "v-o")
	     (?\$,1f(B "w-o")
	     (?\$,1g(B "x-o")
	     (?\$,1h(B "y-o")
	     (?\$,1i(B "z-o")
	     (?\$,1j(B "0-o")
	     (?\$,2  (B "-")
	     (?\$,2 !(B "=")
	     (?\$,2 "(B "|")
	     (?\$,2 #(B "|")
	     (?\$,2 $(B "-")
	     (?\$,2 %(B "=")
	     (?\$,2 &(B "|")
	     (?\$,2 '(B "|")
	     (?\$,2 ((B "-")
	     (?\$,2 )(B "=")
	     (?\$,2 *(B "|")
	     (?\$,2 +(B "|")
	     (?\$,2 ,(B "+")
	     (?\$,2 -(B "+")
	     (?\$,2 .(B "+")
	     (?\$,2 /(B "+")
	     (?\$,2 0(B "+")
	     (?\$,2 1(B "+")
	     (?\$,2 2(B "+")
	     (?\$,2 3(B "+")
	     (?\$,2 4(B "+")
	     (?\$,2 5(B "+")
	     (?\$,2 6(B "+")
	     (?\$,2 7(B "+")
	     (?\$,2 8(B "+")
	     (?\$,2 9(B "+")
	     (?\$,2 :(B "+")
	     (?\$,2 ;(B "+")
	     (?\$,2 <(B "+")
	     (?\$,2 =(B "+")
	     (?\$,2 >(B "+")
	     (?\$,2 ?(B "+")
	     (?\$,2 @(B "+")
	     (?\$,2 A(B "+")
	     (?\$,2 B(B "+")
	     (?\$,2 C(B "+")
	     (?\$,2 D(B "+")
	     (?\$,2 E(B "+")
	     (?\$,2 F(B "+")
	     (?\$,2 G(B "+")
	     (?\$,2 H(B "+")
	     (?\$,2 I(B "+")
	     (?\$,2 J(B "+")
	     (?\$,2 K(B "+")
	     (?\$,2 L(B "+")
	     (?\$,2 M(B "+")
	     (?\$,2 N(B "+")
	     (?\$,2 O(B "+")
	     (?\$,2 P(B "+")
	     (?\$,2 Q(B "+")
	     (?\$,2 R(B "+")
	     (?\$,2 S(B "+")
	     (?\$,2 T(B "+")
	     (?\$,2 U(B "+")
	     (?\$,2 V(B "+")
	     (?\$,2 W(B "+")
	     (?\$,2 X(B "+")
	     (?\$,2 Y(B "+")
	     (?\$,2 Z(B "+")
	     (?\$,2 [(B "+")
	     (?\$,2 \(B "+")
	     (?\$,2 ](B "+")
	     (?\$,2 ^(B "+")
	     (?\$,2 _(B "+")
	     (?\$,2 `(B "+")
	     (?\$,2 a(B "+")
	     (?\$,2 b(B "+")
	     (?\$,2 c(B "+")
	     (?\$,2 d(B "+")
	     (?\$,2 e(B "+")
	     (?\$,2 f(B "+")
	     (?\$,2 g(B "+")
	     (?\$,2 h(B "+")
	     (?\$,2 i(B "+")
	     (?\$,2 j(B "+")
	     (?\$,2 k(B "+")
	     (?\$,2 l(B "+")
	     (?\$,2 m(B "+")
	     (?\$,2 n(B "+")
	     (?\$,2 o(B "+")
	     (?\$,2 p(B "+")
	     (?\$,2 q(B "+")
	     (?\$,2 r(B "+")
	     (?\$,2 s(B "+")
	     (?\$,2 t(B "+")
	     (?\$,2 u(B "+")
	     (?\$,2 v(B "+")
	     (?\$,2 w(B "+")
	     (?\$,2 x(B "+")
	     (?\$,2 y(B "+")
	     (?\$,2 z(B "+")
	     (?\$,2 {(B "+")
	     (?\$,2 |(B "+")
	     (?\$,2 }(B "+")
	     (?\$,2 ~(B "+")
	     (?\$,2 (B "+")
	     (?\$,2! (B "+")
	     (?\$,2!!(B "+")
	     (?\$,2!"(B "+")
	     (?\$,2!#(B "+")
	     (?\$,2!$(B "+")
	     (?\$,2!%(B "+")
	     (?\$,2!&(B "+")
	     (?\$,2!'(B "+")
	     (?\$,2!((B "+")
	     (?\$,2!)(B "+")
	     (?\$,2!*(B "+")
	     (?\$,2!+(B "+")
	     (?\$,2!,(B "+")
	     (?\$,2!1(B "/")
	     (?\$,2!2(B "\\")
	     (?\$,2!@(B "TB")
	     (?\$,2!D(B "LB")
	     (?\$,2!H(B "FB")
	     (?\$,2!L(B "lB")
	     (?\$,2!P(B "RB")
	     (?\$,2!Q(B ".S")
	     (?\$,2!R(B ":S")
	     (?\$,2!S(B "?S")
	     (?\$,2!`(B "fS")
	     (?\$,2!a(B "OS")
	     (?\$,2!b(B "RO")
	     (?\$,2!c(B "Rr")
	     (?\$,2!d(B "RF")
	     (?\$,2!e(B "RY")
	     (?\$,2!f(B "RH")
	     (?\$,2!g(B "RZ")
	     (?\$,2!h(B "RK")
	     (?\$,2!i(B "RX")
	     (?\$,2!j(B "sB")
	     (?\$,2!l(B "SR")
	     (?\$,2!m(B "Or")
	     (?\$,2!r(B "^")
	     (?\$,2!s(B "uT")
	     (?\$,2!v(B "|>")
	     (?\$,2!w(B "Tr")
	     (?\$,2!z(B "|>")
	     (?\$,2!|(B "v")
	     (?\$,2!}(B "dT")
	     (?\$,2" (B "<|")
	     (?\$,2"!(B "Tl")
	     (?\$,2"$(B "<|")
	     (?\$,2"&(B "Db")
	     (?\$,2"'(B "Dw")
	     (?\$,2"*(B "LZ")
	     (?\$,2"+(B "0m")
	     (?\$,2".(B "0o")
	     (?\$,2"/(B "0M")
	     (?\$,2"0(B "0L")
	     (?\$,2"1(B "0R")
	     (?\$,2"8(B "Sn")
	     (?\$,2"9(B "Ic")
	     (?\$,2"B(B "Fd")
	     (?\$,2"C(B "Bd")
	     (?\$,2"O(B "Ci")
	     (?\$,2"e(B "*2")
	     (?\$,2"f(B "*1")
	     (?\$,2"n(B "TEL")
	     (?\$,2"o(B "tel")
	     (?\$,2"|(B "<--")
	     (?\$,2"~(B "-->")
	     (?\$,2#!(B "CAUTION ")
	     (?\$,2#'(B "XP")
	     (?\$,2#9(B ":-(")
	     (?\$,2#:(B ":-)")
	     (?\$,2#;(B "(-:")
	     (?\$,2#<(B "SU")
	     (?\$,2#@(B "f.")
	     (?\$,2#B(B "m.")
	     (?\$,2#`(B "cS")
	     (?\$,2#a(B "cH")
	     (?\$,2#b(B "cD")
	     (?\$,2#c(B "cC")
	     (?\$,2#d(B "cS-")
	     (?\$,2#e(B "cH-")
	     (?\$,2#f(B "cD-")
	     (?\$,2#g(B "cC-")
	     (?\$,2#i(B "Md")
	     (?\$,2#j(B "M8")
	     (?\$,2#k(B "M2")
	     (?\$,2#l(B "M16")
	     (?\$,2#m(B "b")
	     (?\$,2#n(B "Mx")
	     (?\$,2#o(B "#")
	     (?\$,2%S(B "X")
	     (?\$,2%W(B "X")
	     (?\$,2%`(B "-X")
	     (?\$,2=@(B " ")
	     (?\$,2=A(B ",_")
	     (?\$,2=B(B "._")
	     (?\$,2=C(B "+\"")
	     (?\$,2=D(B "JIS")
	     (?\$,2=E(B "*_")
	     (?\$,2=F(B ";_")
	     (?\$,2=G(B "0_")
	     (?\$,2=J(B "<+")
	     (?\$,2=K(B ">+")
	     (?\$,2=L(B "<'")
	     (?\$,2=M(B ">'")
	     (?\$,2=N(B "<\"")
	     (?\$,2=O(B ">\"")
	     (?\$,2=P(B "(\"")
	     (?\$,2=Q(B ")\"")
	     (?\$,2=R(B "=T")
	     (?\$,2=S(B "=_")
	     (?\$,2=T(B "('")
	     (?\$,2=U(B ")'")
	     (?\$,2=V(B "(I")
	     (?\$,2=W(B ")I")
	     (?\$,2=Z(B "[[")
	     (?\$,2=[(B "]]")
	     (?\$,2=\(B "-?")
	     (?\$,2=`(B "=T:)")
	     (?\$,2=(B " ")
	     (?\$,2>!(B "A5")
	     (?\$,2>"(B "a5")
	     (?\$,2>#(B "I5")
	     (?\$,2>$(B "i5")
	     (?\$,2>%(B "U5")
	     (?\$,2>&(B "u5")
	     (?\$,2>'(B "E5")
	     (?\$,2>((B "e5")
	     (?\$,2>)(B "O5")
	     (?\$,2>*(B "o5")
	     (?\$,2>+(B "ka")
	     (?\$,2>,(B "ga")
	     (?\$,2>-(B "ki")
	     (?\$,2>.(B "gi")
	     (?\$,2>/(B "ku")
	     (?\$,2>0(B "gu")
	     (?\$,2>1(B "ke")
	     (?\$,2>2(B "ge")
	     (?\$,2>3(B "ko")
	     (?\$,2>4(B "go")
	     (?\$,2>5(B "sa")
	     (?\$,2>6(B "za")
	     (?\$,2>7(B "si")
	     (?\$,2>8(B "zi")
	     (?\$,2>9(B "su")
	     (?\$,2>:(B "zu")
	     (?\$,2>;(B "se")
	     (?\$,2><(B "ze")
	     (?\$,2>=(B "so")
	     (?\$,2>>(B "zo")
	     (?\$,2>?(B "ta")
	     (?\$,2>@(B "da")
	     (?\$,2>A(B "ti")
	     (?\$,2>B(B "di")
	     (?\$,2>C(B "tU")
	     (?\$,2>D(B "tu")
	     (?\$,2>E(B "du")
	     (?\$,2>F(B "te")
	     (?\$,2>G(B "de")
	     (?\$,2>H(B "to")
	     (?\$,2>I(B "do")
	     (?\$,2>J(B "na")
	     (?\$,2>K(B "ni")
	     (?\$,2>L(B "nu")
	     (?\$,2>M(B "ne")
	     (?\$,2>N(B "no")
	     (?\$,2>O(B "ha")
	     (?\$,2>P(B "ba")
	     (?\$,2>Q(B "pa")
	     (?\$,2>R(B "hi")
	     (?\$,2>S(B "bi")
	     (?\$,2>T(B "pi")
	     (?\$,2>U(B "hu")
	     (?\$,2>V(B "bu")
	     (?\$,2>W(B "pu")
	     (?\$,2>X(B "he")
	     (?\$,2>Y(B "be")
	     (?\$,2>Z(B "pe")
	     (?\$,2>[(B "ho")
	     (?\$,2>\(B "bo")
	     (?\$,2>](B "po")
	     (?\$,2>^(B "ma")
	     (?\$,2>_(B "mi")
	     (?\$,2>`(B "mu")
	     (?\$,2>a(B "me")
	     (?\$,2>b(B "mo")
	     (?\$,2>c(B "yA")
	     (?\$,2>d(B "ya")
	     (?\$,2>e(B "yU")
	     (?\$,2>f(B "yu")
	     (?\$,2>g(B "yO")
	     (?\$,2>h(B "yo")
	     (?\$,2>i(B "ra")
	     (?\$,2>j(B "ri")
	     (?\$,2>k(B "ru")
	     (?\$,2>l(B "re")
	     (?\$,2>m(B "ro")
	     (?\$,2>n(B "wA")
	     (?\$,2>o(B "wa")
	     (?\$,2>p(B "wi")
	     (?\$,2>q(B "we")
	     (?\$,2>r(B "wo")
	     (?\$,2>s(B "n5")
	     (?\$,2>t(B "vu")
	     (?\$,2>{(B "\"5")
	     (?\$,2>|(B "05")
	     (?\$,2>}(B "*5")
	     (?\$,2>~(B "+5")
	     (?\$,2?!(B "a6")
	     (?\$,2?"(B "A6")
	     (?\$,2?#(B "i6")
	     (?\$,2?$(B "I6")
	     (?\$,2?%(B "u6")
	     (?\$,2?&(B "U6")
	     (?\$,2?'(B "e6")
	     (?\$,2?((B "E6")
	     (?\$,2?)(B "o6")
	     (?\$,2?*(B "O6")
	     (?\$,2?+(B "Ka")
	     (?\$,2?,(B "Ga")
	     (?\$,2?-(B "Ki")
	     (?\$,2?.(B "Gi")
	     (?\$,2?/(B "Ku")
	     (?\$,2?0(B "Gu")
	     (?\$,2?1(B "Ke")
	     (?\$,2?2(B "Ge")
	     (?\$,2?3(B "Ko")
	     (?\$,2?4(B "Go")
	     (?\$,2?5(B "Sa")
	     (?\$,2?6(B "Za")
	     (?\$,2?7(B "Si")
	     (?\$,2?8(B "Zi")
	     (?\$,2?9(B "Su")
	     (?\$,2?:(B "Zu")
	     (?\$,2?;(B "Se")
	     (?\$,2?<(B "Ze")
	     (?\$,2?=(B "So")
	     (?\$,2?>(B "Zo")
	     (?\$,2??(B "Ta")
	     (?\$,2?@(B "Da")
	     (?\$,2?A(B "Ti")
	     (?\$,2?B(B "Di")
	     (?\$,2?C(B "TU")
	     (?\$,2?D(B "Tu")
	     (?\$,2?E(B "Du")
	     (?\$,2?F(B "Te")
	     (?\$,2?G(B "De")
	     (?\$,2?H(B "To")
	     (?\$,2?I(B "Do")
	     (?\$,2?J(B "Na")
	     (?\$,2?K(B "Ni")
	     (?\$,2?L(B "Nu")
	     (?\$,2?M(B "Ne")
	     (?\$,2?N(B "No")
	     (?\$,2?O(B "Ha")
	     (?\$,2?P(B "Ba")
	     (?\$,2?Q(B "Pa")
	     (?\$,2?R(B "Hi")
	     (?\$,2?S(B "Bi")
	     (?\$,2?T(B "Pi")
	     (?\$,2?U(B "Hu")
	     (?\$,2?V(B "Bu")
	     (?\$,2?W(B "Pu")
	     (?\$,2?X(B "He")
	     (?\$,2?Y(B "Be")
	     (?\$,2?Z(B "Pe")
	     (?\$,2?[(B "Ho")
	     (?\$,2?\(B "Bo")
	     (?\$,2?](B "Po")
	     (?\$,2?^(B "Ma")
	     (?\$,2?_(B "Mi")
	     (?\$,2?`(B "Mu")
	     (?\$,2?a(B "Me")
	     (?\$,2?b(B "Mo")
	     (?\$,2?c(B "YA")
	     (?\$,2?d(B "Ya")
	     (?\$,2?e(B "YU")
	     (?\$,2?f(B "Yu")
	     (?\$,2?g(B "YO")
	     (?\$,2?h(B "Yo")
	     (?\$,2?i(B "Ra")
	     (?\$,2?j(B "Ri")
	     (?\$,2?k(B "Ru")
	     (?\$,2?l(B "Re")
	     (?\$,2?m(B "Ro")
	     (?\$,2?n(B "WA")
	     (?\$,2?o(B "Wa")
	     (?\$,2?p(B "Wi")
	     (?\$,2?q(B "We")
	     (?\$,2?r(B "Wo")
	     (?\$,2?s(B "N6")
	     (?\$,2?t(B "Vu")
	     (?\$,2?u(B "KA")
	     (?\$,2?v(B "KE")
	     (?\$,2?w(B "Va")
	     (?\$,2?x(B "Vi")
	     (?\$,2?y(B "Ve")
	     (?\$,2?z(B "Vo")
	     (?\$,2?{(B ".6")
	     (?\$,2?|(B "-6")
	     (?\$,2?}(B "*6")
	     (?\$,2?~(B "+6")
	     (?\$,2@%(B "b4")
	     (?\$,2@&(B "p4")
	     (?\$,2@'(B "m4")
	     (?\$,2@((B "f4")
	     (?\$,2@)(B "d4")
	     (?\$,2@*(B "t4")
	     (?\$,2@+(B "n4")
	     (?\$,2@,(B "l4")
	     (?\$,2@-(B "g4")
	     (?\$,2@.(B "k4")
	     (?\$,2@/(B "h4")
	     (?\$,2@0(B "j4")
	     (?\$,2@1(B "q4")
	     (?\$,2@2(B "x4")
	     (?\$,2@3(B "zh")
	     (?\$,2@4(B "ch")
	     (?\$,2@5(B "sh")
	     (?\$,2@6(B "r4")
	     (?\$,2@7(B "z4")
	     (?\$,2@8(B "c4")
	     (?\$,2@9(B "s4")
	     (?\$,2@:(B "a4")
	     (?\$,2@;(B "o4")
	     (?\$,2@<(B "e4")
	     (?\$,2@=(B "eh4")
	     (?\$,2@>(B "ai")
	     (?\$,2@?(B "ei")
	     (?\$,2@@(B "au")
	     (?\$,2@A(B "ou")
	     (?\$,2@B(B "an")
	     (?\$,2@C(B "en")
	     (?\$,2@D(B "aN")
	     (?\$,2@E(B "eN")
	     (?\$,2@F(B "er")
	     (?\$,2@G(B "i4")
	     (?\$,2@H(B "u4")
	     (?\$,2@I(B "iu")
	     (?\$,2@J(B "v4")
	     (?\$,2@K(B "nG")
	     (?\$,2@L(B "gn")
	     (?\$,2B|(B "(JU)")
	     (?\$,2C (B "1c")
	     (?\$,2C!(B "2c")
	     (?\$,2C"(B "3c")
	     (?\$,2C#(B "4c")
	     (?\$,2C$(B "5c")
	     (?\$,2C%(B "6c")
	     (?\$,2C&(B "7c")
	     (?\$,2C'(B "8c")
	     (?\$,2C((B "9c")
	     (?\$,2C)(B "10c")
	     (?\$,2C(B "KSC")
	     (?\$,2GB(B "am")
	     (?\$,2GX(B "pm")
	     (?\$,3h (B "ff")
	     (?\$,3h!(B "fi")
	     (?\$,3h"(B "fl")
	     (?\$,3h#(B "ffi")
	     (?\$,3h$(B "ffl")
	     (?\$,3h%(B "St")
	     (?\$,3h&(B "st")
	     (?\$,3q=(B "3+;")
	     (?\$,3qB(B "aM.")
	     (?\$,3qD(B "aH.")
	     (?\$,3qH(B "ah.")
	     (?\$,3qM(B "a+-")
	     (?\$,3qN(B "a+.")
	     (?\$,3qO(B "b+-")
	     (?\$,3qP(B "b+.")
	     (?\$,3qQ(B "b+,")
	     (?\$,3qR(B "b+;")
	     (?\$,3qS(B "tm-")
	     (?\$,3qT(B "tm.")
	     (?\$,3qU(B "t+-")
	     (?\$,3qV(B "t+.")
	     (?\$,3qW(B "t+,")
	     (?\$,3qX(B "t+;")
	     (?\$,3qY(B "tk-")
	     (?\$,3qZ(B "tk.")
	     (?\$,3q[(B "tk,")
	     (?\$,3q\(B "tk;")
	     (?\$,3q](B "g+-")
	     (?\$,3q^(B "g+.")
	     (?\$,3q_(B "g+,")
	     (?\$,3q`(B "g+;")
	     (?\$,3qa(B "hk-")
	     (?\$,3qb(B "hk.")
	     (?\$,3qc(B "hk,")
	     (?\$,3qd(B "hk;")
	     (?\$,3qe(B "x+-")
	     (?\$,3qf(B "x+.")
	     (?\$,3qg(B "x+,")
	     (?\$,3qh(B "x+;")
	     (?\$,3qi(B "d+-")
	     (?\$,3qj(B "d+.")
	     (?\$,3qk(B "dk-")
	     (?\$,3ql(B "dk.")
	     (?\$,3qm(B "r+-")
	     (?\$,3qn(B "r+.")
	     (?\$,3qo(B "z+-")
	     (?\$,3qp(B "z+.")
	     (?\$,3qq(B "s+-")
	     (?\$,3qr(B "s+.")
	     (?\$,3qs(B "s+,")
	     (?\$,3qt(B "s+;")
	     (?\$,3qu(B "sn-")
	     (?\$,3qv(B "sn.")
	     (?\$,3qw(B "sn,")
	     (?\$,3qx(B "sn;")
	     (?\$,3qy(B "c+-")
	     (?\$,3qz(B "c+.")
	     (?\$,3q{(B "c+,")
	     (?\$,3q|(B "c+;")
	     (?\$,3q}(B "dd-")
	     (?\$,3q~(B "dd.")
	     (?\$,3q(B "dd,")
	     (?\$,3r (B "dd;")
	     (?\$,3r!(B "tj-")
	     (?\$,3r"(B "tj.")
	     (?\$,3r#(B "tj,")
	     (?\$,3r$(B "tj;")
	     (?\$,3r%(B "zH-")
	     (?\$,3r&(B "zH.")
	     (?\$,3r'(B "zH,")
	     (?\$,3r((B "zH;")
	     (?\$,3r)(B "e+-")
	     (?\$,3r*(B "e+.")
	     (?\$,3r+(B "e+,")
	     (?\$,3r,(B "e+;")
	     (?\$,3r-(B "i+-")
	     (?\$,3r.(B "i+.")
	     (?\$,3r/(B "i+,")
	     (?\$,3r0(B "i+;")
	     (?\$,3r1(B "f+-")
	     (?\$,3r2(B "f+.")
	     (?\$,3r3(B "f+,")
	     (?\$,3r4(B "f+;")
	     (?\$,3r5(B "q+-")
	     (?\$,3r6(B "q+.")
	     (?\$,3r7(B "q+,")
	     (?\$,3r8(B "q+;")
	     (?\$,3r9(B "k+-")
	     (?\$,3r:(B "k+.")
	     (?\$,3r;(B "k+,")
	     (?\$,3r<(B "k+;")
	     (?\$,3r=(B "l+-")
	     (?\$,3r>(B "l+.")
	     (?\$,3r?(B "l+,")
	     (?\$,3r@(B "l+;")
	     (?\$,3rA(B "m+-")
	     (?\$,3rB(B "m+.")
	     (?\$,3rC(B "m+,")
	     (?\$,3rD(B "m+;")
	     (?\$,3rE(B "n+-")
	     (?\$,3rF(B "n+.")
	     (?\$,3rG(B "n+,")
	     (?\$,3rH(B "n+;")
	     (?\$,3rI(B "h+-")
	     (?\$,3rJ(B "h+.")
	     (?\$,3rK(B "h+,")
	     (?\$,3rL(B "h+;")
	     (?\$,3rM(B "w+-")
	     (?\$,3rN(B "w+.")
	     (?\$,3rO(B "j+-")
	     (?\$,3rP(B "j+.")
	     (?\$,3rQ(B "y+-")
	     (?\$,3rR(B "y+.")
	     (?\$,3rS(B "y+,")
	     (?\$,3rT(B "y+;")
	     (?\$,3rU(B "lM-")
	     (?\$,3rV(B "lM.")
	     (?\$,3rW(B "lH-")
	     (?\$,3rX(B "lH.")
	     (?\$,3rY(B "lh-")
	     (?\$,3rZ(B "lh.")
	     (?\$,3r[(B "la-")
	     (?\$,3r\(B "la.")
	     (?\$,3ra(B "!")
	     (?\$,3rb(B "\"")
	     (?\$,3rc(B "#")
	     (?\$,3rd(B "$")
	     (?\$,3re(B "%")
	     (?\$,3rf(B "&")
	     (?\$,3rg(B "'")
	     (?\$,3rh(B "(")
	     (?\$,3ri(B ")")
	     (?\$,3rj(B "*")
	     (?\$,3rk(B "+")
	     (?\$,3rl(B ",")
	     (?\$,3rm(B "-")
	     (?\$,3rn(B ".")
	     (?\$,3ro(B "/")
	     (?\$,3rp(B "0")
	     (?\$,3rq(B "1")
	     (?\$,3rr(B "2")
	     (?\$,3rs(B "3")
	     (?\$,3rt(B "4")
	     (?\$,3ru(B "5")
	     (?\$,3rv(B "6")
	     (?\$,3rw(B "7")
	     (?\$,3rx(B "8")
	     (?\$,3ry(B "9")
	     (?\$,3rz(B ":")
	     (?\$,3r{(B ";")
	     (?\$,3r|(B "<")
	     (?\$,3r}(B "=")
	     (?\$,3r~(B ">")
	     (?\$,3r(B "?")
	     (?\$,3s (B "@")
	     (?\$,3s!(B "A")
	     (?\$,3s"(B "B")
	     (?\$,3s#(B "C")
	     (?\$,3s$(B "D")
	     (?\$,3s%(B "E")
	     (?\$,3s&(B "F")
	     (?\$,3s'(B "G")
	     (?\$,3s((B "H")
	     (?\$,3s)(B "I")
	     (?\$,3s*(B "J")
	     (?\$,3s+(B "K")
	     (?\$,3s,(B "L")
	     (?\$,3s-(B "M")
	     (?\$,3s.(B "N")
	     (?\$,3s/(B "O")
	     (?\$,3s0(B "P")
	     (?\$,3s1(B "Q")
	     (?\$,3s2(B "R")
	     (?\$,3s3(B "S")
	     (?\$,3s4(B "T")
	     (?\$,3s5(B "U")
	     (?\$,3s6(B "V")
	     (?\$,3s7(B "W")
	     (?\$,3s8(B "X")
	     (?\$,3s9(B "Y")
	     (?\$,3s:(B "Z")
	     (?\$,3s;(B "[")
	     (?\$,3s<(B "\\")
	     (?\$,3s=(B "]")
	     (?\$,3s>(B "^")
	     (?\$,3s?(B "_")
	     (?\$,3s@(B "`")
	     (?\$,3sA(B "a")
	     (?\$,3sB(B "b")
	     (?\$,3sC(B "c")
	     (?\$,3sD(B "d")
	     (?\$,3sE(B "e")
	     (?\$,3sF(B "f")
	     (?\$,3sG(B "g")
	     (?\$,3sH(B "h")
	     (?\$,3sI(B "i")
	     (?\$,3sJ(B "j")
	     (?\$,3sK(B "k")
	     (?\$,3sL(B "l")
	     (?\$,3sM(B "m")
	     (?\$,3sN(B "n")
	     (?\$,3sO(B "o")
	     (?\$,3sP(B "p")
	     (?\$,3sQ(B "q")
	     (?\$,3sR(B "r")
	     (?\$,3sS(B "s")
	     (?\$,3sT(B "t")
	     (?\$,3sU(B "u")
	     (?\$,3sV(B "v")
	     (?\$,3sW(B "w")
	     (?\$,3sX(B "x")
	     (?\$,3sY(B "y")
	     (?\$,3sZ(B "z")
	     (?\$,3s[(B "{")
	     (?\$,3s\(B "|")
	     (?\$,3s](B "}")
	     (?\$,3s^(B "~")
	     (?\$,3sa(B ".")
	     (?\$,3sb(B "\"")
	     (?\$,3sc(B "\"")
	     (?\$,3sd(B ",")
	     ;; Not from Lynx
	     (?$,3r_(B "")
	     (?$,3u=(B "?")))))
    (aset standard-display-table
	  (make-char 'mule-unicode-0100-24ff) nil)
    (aset standard-display-table
	  (make-char 'mule-unicode-2500-33ff) nil)
    (aset standard-display-table
	  (make-char 'mule-unicode-e000-ffff) nil))
  (setq latin1-display-ucs-per-lynx (not latin1-display-ucs-per-lynx))
  (redraw-display))

(provide 'latin1-disp)

;; arch-tag: 68b2872e-d667-4f48-8e2f-ec2ba2d29406
;;; latin1-disp.el ends here
