/*
 * Electric(tm) VLSI Design System
 *
 * File: tecmocmossub.c
 * MOSIS CMOS Submicron technology description
 * Written by: Steven M. Rubin, Static Free Software
 * The MOSIS 6-metal, 2-poly submicron rules,
 * interpreted by Richard "Bic" Schediwy
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

/*
 * Assumptions made from the MOSIS submicron design rules
 * (MOSIS rule is in square brackets []):
 *
 * Arc widths (minimum sizes):
 *   metal1: 3 [7.1]
 *   metal2: 3 [9.1]
 *   metal3: 5 (if 3-metal process) [15.1]
 *   metal3: 3 (if 4-metal process) [15.1]
 *   metal4: 6 [22.1]
 *   poly1:  2 [3.1]
 *   poly2:  3 [11.1]
 *   p/n active (active&select&well):
 *     active: 3 [2.1], select extends by 2 [4.2], well extends by 6 [2.3]
 *   active: 3 [2.1]
 *
 * Pin/Node sizes (minimum sizes):
 *   metal1: 3 [7.1]
 *   metal2: 3 [9.1]
 *   metal3: 5 (if 3-metal process) [15.1]
 *   metal3: 3 (if 4-metal process) [15.1]
 *   metal4: 6 [22.1]
 *   poly1:  2 [3.1]
 *   poly2:  3 [11.1]
 *   active: 3 [2.1]
 *   select: 2 [4.4]
 *   well:  12 [1.1]
 *
 * Special nodes:
 *   p/n active-to-metal1 contact:
 *     cuts 2x2 [6.1], separated 3 [6.3]
 *     metal1 extends around cut by 1 (4x4) [7.3]
 *     active extends around cut by 1.5 (5x5) [6.2]
 *     select extends around active by 1 (7x7) [4.3]  {CORRECTION: by 2 (8x8) [4.2]}
 *     well extends around active by 6 (17x17) [2.3]
 *   poly1-to-metal1 contact:
 *     cuts 2x2 [5.1], separated 3 [5.3]
 *     metal1 extends around cut by 1 (4x4) [7.3]
 *     poly1 extends around cut by 1.5 (5x5) [5.2]
 *   poly2-to-metal1 contact:
 *     cuts 2x2 [5.1], separated 3 [5.3]
 *     metal1 extends around cut by 1 (4x4) [7.3]
 *     poly2 size: 3 (3x3) [11.1]
 *   poly1-to-poly2 (capacitor) contact:
 *     cuts 2x2 [5.1], separated 3 [5.3]
 *     poly2 size: 3 (3x3) [11.1]
 *     poly1 extends around poly2 by 2 (7x7) [11.3]
 *   Transistors:
 *     active is 3 wide [2.1] and sticks out by 3 (3x8) [3.4]
 *     poly1 is 2 wide [3.1] and sticks out by 2 (7x2) [3.3]
 *     transistor area is 3x2
 *     select surrounds active by 2 (7x12) [4.2]
 *     well surrounds active by 6 (15x20) [2.3]
 *   Via1:
 *     cuts 2x2 [8.1], separated 3 [8.2]
 *     metal1 extends around cut by 1 (4x4) [8.3]
 *     metal2 extends around cut by 1 (4x4) [9.3]
 *   Via2:
 *     cuts 2x2 [14.1], separated 3 [14.2]
 *     metal2 extends around cut by 1 (4x4) [14.3]
 *     metal3 extends around cut by: 2 (6x6) (if 3-metal process) [15.3]
 *     metal3 extends around cut by: 1 (4x4) (if 4-metal process) [15.3]
 *   Via3:
 *     cuts 2x2 [21.1], separated 4 [21.2]
 *     metal3 extends around cut by: 1 (4x4) [21.3]
 *     metal4 extends around cut by: 2 (6x6) [22.3]
 *   Substrate/well contact:
 *     select extends around active by 2 [4.2]
 *     well extends around active by 6 [2.3]
 *
 * DRC:
 *   metal1-to-metal1: 3 [7.2]
 *   metal2-to-metal2: 4 [9.2]
 *   metal3-to-metal3: 3 [15.2]
 *   metal4-to-metal4: 6 [22.2]
 *   poly1-to-poly1: 3 [3.2]
 *   poly1-to-active: 1 [3.5]
 *   poly2-to-poly2: 3 [11.2]
 *   poly2-to-active: 1 [3.5]
 *   poly2-to-polyCut: 3 [11.5]
 *   active-to-active: 3 [2.2]
 *   select-to-trans: 3 [4.1]
 *   polyCut/actCut-to-polyCut/actCut: 3 [5.3]
 *   polyCut/actCut-to-via1: 2 [8.4]
 *   polyCut-to-active: 2 [5.4]
 *   actCut-to-poly: 2 [6.4]
 *   via1-to-via1: 3 [8.2]
 *   via1-to-via2: 2 [14.4]
 *   via2-to-via2: 2 [14.2]
 *   via3-to-via3: 4 [21.2]
 */

/*
 * #metals:     Metal-1     Metal-2     Metal-3     Metal-4     Metal-5     Metal-6
 *
 *  2-metals:   3 wide      3 wide
 *              3 apart     4 apart
 *              1 over via  1 over via
 *
 *  3-metals:   3 wide      3 wide      5 wide
 *              3 apart     3 apart     3 apart
 *              1 over via  1 over via  2 over via
 *
 *  4-metals:   3 wide      3 wide      3 wide      3 wide
 *              3 apart     3 apart     3 apart     3 apart
 *              1 over via  1 over via  1 over via  1 over via
 *
 *  5-metals:   3 wide      3 wide      3 wide      3 wide      4 wide
 *              3 apart     3 apart     3 apart     3 apart     4 apart
 *              1 over via  1 over via  1 over via  1 over via  1 over via
 *
 *  6-metals:   3 wide      3 wide      3 wide      3 wide      3 wide      4 wide
 *              3 apart     3 apart     3 apart     3 apart     3 apart     4 apart
 *              1 over via  1 over via  1 over via  1 over via  1 over via  1 over via
 */

#include "config.h"
#if TECMOCMOSSUB

#include "global.h"
#include "egraphics.h"
#include "tech.h"
#include "tecmocmossub.h"
#include "efunction.h"

/*
 * Can switch from 4-metal rules (the default) to 2/3/5/6-metal rules
 * Can switch from full-geometry (the default) to stick-figures
 */

/* the options table */
static KEYWORD mocmossubopt[] =
{
	{"2-metal-rules",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"3-metal-rules",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"4-metal-rules",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"5-metal-rules",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"6-metal-rules",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"full-graphics",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	{"stick-display",              0,{NOKEY,NOKEY,NOKEY,NOKEY,NOKEY}},
	TERMKEY
};
COMCOMP mocmossub_parse = {mocmossubopt, NOTOPLIST, NONEXTLIST, NOPARAMS,
	NOBACKUP, 0, " \t", "MOSIS CMOS Submicron option", "show current options"};

typedef struct
{
	NODEPROTO *np[2];
	PORTPROTO *pp[2][4];
	INTBIG     portcount;
} PRIMSWAP;

PRIMSWAP mocmossub_primswap[7];

static INTBIG      mocmossub_state;			/* settings */
       TECHNOLOGY *mocmossub_tech;
static INTSML      mocmossub_arrowbox;

void mocmossub_setstate(INTBIG newstate);
void mocmossub_switchnp(void);
INTSML mocmossub_nodepolys(NODEINST *ni);
void mocmossub_shapenodepoly(NODEINST *ni, INTSML box, POLYGON *poly);
void mocmossub_nodesizeoffset(NODEINST *ni, INTBIG *lx, INTBIG *ly, INTBIG *hx, INTBIG *hy);
INTSML mocmossub_arcpolys(ARCINST *ai);
void mocmossub_shapearcpoly(ARCINST *ai, INTSML box, POLYGON *poly);
INTBIG mocmossub_arcwidthoffset(ARCINST *ai);
void mocmossub_setupprimswap(INTBIG index1, INTBIG index2, PRIMSWAP *swap);
void mocmossub_setlayerrules(char *layername, INTBIG spacing, INTBIG minwidth);
void mocmossub_setarcwidth(char *arcname, INTBIG width);
void mocmossub_setnodesize(char *nodename, INTBIG size, INTSML portoffset);
void mocmossub_setmetalonvia(char *nodename, INTBIG layer, INTBIG *boxdesc, INTBIG nodeoffset);

/******************** LAYERS ********************/

#define	MAXLAYERS   40		/* total layers below         */
#define	LMETAL1      0		/* metal layer 1              */
#define	LMETAL2      1		/* metal layer 2              */
#define	LMETAL3      2		/* metal layer 3              */
#define	LMETAL4      3		/* metal layer 4              */
#define	LMETAL5      4		/* metal layer 5              */
#define	LMETAL6      5		/* metal layer 6              */
#define	LPOLY1       6		/* polysilicon                */
#define	LPOLY2       7		/* polysilicon 2 (electrode)  */
#define	LSACT        8		/* P (or N) active            */
#define	LDACT        9		/* N (or P) active            */
#define	LSELECTP    10		/* P-type select              */
#define	LSELECTN    11		/* N-type select              */
#define	LWELLP      12		/* P-type well                */
#define	LWELLN      13		/* N-type well                */
#define	LPOLYCUT    14		/* poly contact cut           */
#define	LACTCUT     15		/* active contact cut         */
#define	LVIA1       16		/* metal1-to-metal2 via       */
#define	LVIA2       17		/* metal2-to-metal3 via       */
#define	LVIA3       18		/* metal3-to-metal4 via       */
#define	LVIA4       19		/* metal4-to-metal5 via       */
#define	LVIA5       20		/* metal5-to-metal6 via       */
#define	LPASS       21		/* passivation (overglass)    */
#define	LTRANS      22		/* transistor                 */
#define	LPOLYCAP    23		/* polysilicon capacitor      */
#define	LSACTWELL   24		/* P active in well           */
#define	LMET1P      25		/* pseudo metal 1             */
#define	LMET2P      26		/* pseudo metal 2             */
#define	LMET3P      27		/* pseudo metal 3             */
#define	LMET4P      28		/* pseudo metal 4             */
#define	LMET5P      29		/* pseudo metal 5             */
#define	LMET6P      30		/* pseudo metal 6             */
#define	LPOLY1P     31		/* pseudo polysilicon 1       */
#define	LPOLY2P     32		/* pseudo polysilicon 2       */
#define	LSACTP      33		/* pseudo P (or N) active     */
#define	LDACTP      34		/* pseudo N (or P) active     */
#define	LSELECTPP   35		/* pseudo P-type select       */
#define	LSELECTNP   36		/* pseudo N-type select       */
#define	LWELLPP     37		/* pseudo P-type well         */
#define	LWELLNP     38		/* pseudo N-type well         */
#define	LFRAME      39		/* pad frame boundary         */

static GRAPHICS mocmossub_m1_lay = {LAYERT1,COLORT1, SOLIDC, PATTERNED,
/* metal-1 layer */		{0x2222, /*   X   X   X   X  */
						0x0000,  /*                  */
						0x8888,  /* X   X   X   X    */
						0x0000,  /*                  */
						0x2222,  /*   X   X   X   X  */
						0x0000,  /*                  */
						0x8888,  /* X   X   X   X    */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_m2_lay = {LAYERT4,COLORT4, SOLIDC, PATTERNED,
/* metal-2 layer */		{0x1010, /*    X       X     */
						0x2020,  /*   X       X      */
						0x4040,  /*  X       X       */
						0x8080,  /* X       X        */
						0x0101,  /*        X       X */
						0x0202,  /*       X       X  */
						0x0404,  /*      X       X   */
						0x0808}, /*     X       X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_m3_lay = {LAYERT5,COLORT5, SOLIDC, PATTERNED,
/* metal-3 layer */		{0x2222, /*   X   X   X   X  */
						0x0000,  /*                  */
						0x8888,  /* X   X   X   X    */
						0x0000,  /*                  */
						0x2222,  /*   X   X   X   X  */
						0x0000,  /*                  */
						0x8888,  /* X   X   X   X    */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_m4_lay = {LAYERO,LBLUE, PATTERNED, PATTERNED,
/* metal-4 layer */		{0x0808, /*     X       X    */
						0x1818,  /*    XX      XX    */
						0x2828,  /*   X X     X X    */
						0x4848,  /*  X  X    X  X    */
						0xFCFC,  /* XXXXXX  XXXXXXX  */
						0x0808,  /*     X       X    */
						0x0808,  /*     X       X    */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_m5_lay = {LAYERO,LRED, PATTERNED, PATTERNED,
/* metal-5 layer */		{0xFCFC, /* XXXXXX  XXXXXX   */
						0x8080,  /* X       X        */
						0x8080,  /* X       X        */
						0xF8F8,  /* XXXXX   XXXXX    */
						0x0404,  /*      X       X   */
						0x0404,  /*      X       X   */
						0xF8F8,  /* XXXXX   XXXXX    */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_m6_lay = {LAYERO,CYAN, PATTERNED, PATTERNED,
/* metal-6 layer */		{0x1818, /*    XX      XX    */
						0x6060,  /*  XX      XX      */
						0x8080,  /* X       X        */
						0xF8F8,  /* XXXXX   XXXXX    */
						0x8484,  /* X    X  X    X   */
						0x8484,  /* X    X  X    X   */
						0x7878,  /*  XXXX    XXXX    */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_p1_lay = {LAYERT2,COLORT2, SOLIDC, PATTERNED,
/* poly layer */		{0x1111, /*    X   X   X   X */
						0xFFFF,  /* XXXXXXXXXXXXXXXX */
						0x1111,  /*    X   X   X   X */
						0x5555,  /*  X X X X X X X X */
						0x1111,  /*    X   X   X   X */
						0xFFFF,  /* XXXXXXXXXXXXXXXX */
						0x1111,  /*    X   X   X   X */
						0x5555}, /*  X X X X X X X X */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_p2_lay = {LAYERO,ORANGE, PATTERNED, PATTERNED,
/* poly2 layer */		{0xAFAF, /* X X XXXXX X XXXX */
						0x8888,  /* X   X   X   X    */
						0xFAFA,  /* XXXXX X XXXXX X  */
						0x8888,  /* X   X   X   X    */
						0xAFAF,  /* X X XXXXX X XXXX */
						0x8888,  /* X   X   X   X    */
						0xFAFA,  /* XXXXX X XXXXX X  */
						0x8888}, /* X   X   X   X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_sa_lay = {LAYERT3,COLORT3, SOLIDC, PATTERNED,
/* P active layer */	{0x0000, /*                  */
						0x0303,  /*       XX      XX */
						0x4848,  /*  X  X    X  X    */
						0x0303,  /*       XX      XX */
						0x0000,  /*                  */
						0x3030,  /*   XX      XX     */
						0x8484,  /* X    X  X    X   */
						0x3030}, /*   XX      XX     */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_da_lay = {LAYERT3,COLORT3, SOLIDC, PATTERNED,
/* N active layer */	{0x0000, /*                  */
						0x0303,  /*       XX      XX */
						0x4848,  /*  X  X    X  X    */
						0x0303,  /*       XX      XX */
						0x0000,  /*                  */
						0x3030,  /*   XX      XX     */
						0x8484,  /* X    X  X    X   */
						0x3030}, /*   XX      XX     */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_ssp_lay = {LAYERO,YELLOW, PATTERNED, PATTERNED,
/* P Select layer */	{0x1010, /*    X       X     */
						0x2020,  /*   X       X      */
						0x4040,  /*  X       X       */
						0x8080,  /* X       X        */
						0x0101,  /*        X       X */
						0x0202,  /*       X       X  */
						0x0404,  /*      X       X   */
						0x0808}, /*     X       X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_ssn_lay = {LAYERO,YELLOW, PATTERNED, PATTERNED,
/* N Select layer */	{0x0100, /*        X         */
						0x0000,  /*                  */
						0x0000,  /*                  */
						0x0000,  /*                  */
						0x0001,  /*                X */
						0x0000,  /*                  */
						0x0000,  /*                  */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_wp_lay = {LAYERO,BROWN, PATTERNED, PATTERNED,
/* P Well layer */		{0x0202, /*       X       X  */
						0x0101,  /*        X       X */
						0x8080,  /* X       X        */
						0x4040,  /*  X       X       */
						0x2020,  /*   X       X      */
						0x1010,  /*    X       X     */
						0x0808,  /*     X       X    */
						0x0404}, /*      X       X   */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_wn_lay = {LAYERO,BROWN, PATTERNED, PATTERNED,
/* N Well implant */	{0x0002, /*               X  */
						0x0000,  /*                  */
						0x0000,  /*                  */
						0x0000,  /*                  */
						0x0200,  /*       X          */
						0x0000,  /*                  */
						0x0000,  /*                  */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pc_lay = {LAYERO,BLACK, SOLIDC, SOLIDC,
/* poly cut layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_ac_lay = {LAYERO,BLACK, SOLIDC, SOLIDC,
/* active cut layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_v1_lay = {LAYERO,LGRAY, SOLIDC, SOLIDC,
/* via1 layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_v2_lay = {LAYERO,LGRAY, SOLIDC, SOLIDC,
/* via2 layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_v3_lay = {LAYERO,LGRAY, SOLIDC, SOLIDC,
/* via3 layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_v4_lay = {LAYERO,LGRAY, SOLIDC, SOLIDC,
/* via4 layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_v5_lay = {LAYERO,LGRAY, SOLIDC, SOLIDC,
/* via5 layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_ovs_lay = {LAYERO,DGRAY, PATTERNED, PATTERNED,
/* passivation layer */	{0x1C1C, /*    XXX     XXX   */
						0x3E3E,  /*   XXXXX   XXXXX  */
						0x3636,  /*   XX XX   XX XX  */
						0x3E3E,  /*   XXXXX   XXXXX  */
						0x1C1C,  /*    XXX     XXX   */
						0x0000,  /*                  */
						0x0000,  /*                  */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_tr_lay = {LAYERN,ALLOFF, SOLIDC, SOLIDC,
/* transistor layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_cp_lay = {LAYERO,BLACK, SOLIDC, SOLIDC,
/* poly cap layer */	{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};
static GRAPHICS mocmossub_saw_lay = {LAYERT3,COLORT3, SOLIDC, PATTERNED,
/* P act well layer */	{0x0000, /*                  */
						0x0303,  /*       XX      XX */
						0x4848,  /*  X  X    X  X    */
						0x0303,  /*       XX      XX */
						0x0000,  /*                  */
						0x3030,  /*   XX      XX     */
						0x8484,  /* X    X  X    X   */
						0x3030}, /*   XX      XX     */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pm1_lay ={LAYERT1,COLORT1, SOLIDC, PATTERNED,
/* pseudo metal 1 */	{0x2222, /*   X   X   X   X  */
						0x0000,  /*                  */
						0x8888,  /* X   X   X   X    */
						0x0000,  /*                  */
						0x2222,  /*   X   X   X   X  */
						0x0000,  /*                  */
						0x8888,  /* X   X   X   X    */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pm2_lay = {LAYERT4,COLORT4, SOLIDC, PATTERNED,
/* pseudo metal-2 */	{0x1010, /*    X       X     */
						0x2020,  /*   X       X      */
						0x4040,  /*  X       X       */
						0x8080,  /* X       X        */
						0x0101,  /*        X       X */
						0x0202,  /*       X       X  */
						0x0404,  /*      X       X   */
						0x0808}, /*     X       X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pm3_lay = {LAYERT5,COLORT5, SOLIDC, PATTERNED,
/* pseudo metal-3 */	{0x1010, /*    X       X     */
						0x2020,  /*   X       X      */
						0x4040,  /*  X       X       */
						0x8080,  /* X       X        */
						0x0101,  /*        X       X */
						0x0202,  /*       X       X  */
						0x0404,  /*      X       X   */
						0x0808}, /*     X       X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pm4_lay = {LAYERO,LBLUE, PATTERNED, PATTERNED,
/* pseudo metal-4 */	{0x0808, /*     X       X    */
						0x1818,  /*    XX      XX    */
						0x2828,  /*   X X     X X    */
						0x4848,  /*  X  X    X  X    */
						0xFCFC,  /* XXXXXX  XXXXXXX  */
						0x0808,  /*     X       X    */
						0x0808,  /*     X       X    */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pm5_lay = {LAYERO,LRED, PATTERNED, PATTERNED,
/* pseudo metal-5 */	{0xFCFC, /* XXXXXX  XXXXXX   */
						0x8080,  /* X       X        */
						0x8080,  /* X       X        */
						0xF8F8,  /* XXXXX   XXXXX    */
						0x0404,  /*      X       X   */
						0x0404,  /*      X       X   */
						0xF8F8,  /* XXXXX   XXXXX    */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pm6_lay = {LAYERO,CYAN, PATTERNED, PATTERNED,
/* pseudo metal-6 */	{0x1818, /*    XX      XX    */
						0x6060,  /*  XX      XX      */
						0x8080,  /* X       X        */
						0xF8F8,  /* XXXXX   XXXXX    */
						0x8484,  /* X    X  X    X   */
						0x8484,  /* X    X  X    X   */
						0x7878,  /*  XXXX    XXXX    */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pp1_lay = {LAYERT2,COLORT2, SOLIDC, PATTERNED,
/* pseudo poly layer */	{0x1111, /*    X   X   X   X */
						0xFFFF,  /* XXXXXXXXXXXXXXXX */
						0x1111,  /*    X   X   X   X */
						0x5555,  /*  X X X X X X X X */
						0x1111,  /*    X   X   X   X */
						0xFFFF,  /* XXXXXXXXXXXXXXXX */
						0x1111,  /*    X   X   X   X */
						0x5555}, /*  X X X X X X X X */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pp2_lay = {LAYERO,ORANGE, PATTERNED, PATTERNED,
/* pseudo poly2 layer */{0xAFAF, /* X X XXXXX X XXXX */
						0x8888,  /* X   X   X   X    */
						0xFAFA,  /* XXXXX X XXXXX X  */
						0x8888,  /* X   X   X   X    */
						0xAFAF,  /* X X XXXXX X XXXX */
						0x8888,  /* X   X   X   X    */
						0xFAFA,  /* XXXXX X XXXXX X  */
						0x8888}, /* X   X   X   X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_psa_lay = {LAYERT3,COLORT3, SOLIDC, PATTERNED,
/* pseudo P active */	{0x0000, /*                  */
						0x0303,  /*       XX      XX */
						0x4848,  /*  X  X    X  X    */
						0x0303,  /*       XX      XX */
						0x0000,  /*                  */
						0x3030,  /*   XX      XX     */
						0x8484,  /* X    X  X    X   */
						0x3030}, /*   XX      XX     */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pda_lay = {LAYERT3,COLORT3, SOLIDC, PATTERNED,
/* pseudo N active */	{0x0000, /*                  */
						0x0303,  /*       XX      XX */
						0x4848,  /*  X  X    X  X    */
						0x0303,  /*       XX      XX */
						0x0000,  /*                  */
						0x3030,  /*   XX      XX     */
						0x8484,  /* X    X  X    X   */
						0x3030}, /*   XX      XX     */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pssp_lay = {LAYERO,YELLOW,PATTERNED, PATTERNED,
/* pseudo P Select */	{0x1010, /*    X       X     */
						0x2020,  /*   X       X      */
						0x4040,  /*  X       X       */
						0x8080,  /* X       X        */
						0x0101,  /*        X       X */
						0x0202,  /*       X       X  */
						0x0404,  /*      X       X   */
						0x0808}, /*     X       X    */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pssn_lay = {LAYERO,YELLOW,PATTERNED, PATTERNED,
/* pseudo N Select */	{0x0100, /*        X         */
						0x0000,  /*                  */
						0x0000,  /*                  */
						0x0000,  /*                  */
						0x0001,  /*                X */
						0x0000,  /*                  */
						0x0000,  /*                  */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pwp_lay = {LAYERO,BROWN, PATTERNED, PATTERNED,
/* pseudo P Well */		{0x0202, /*       X       X  */
						0x0101,  /*        X       X */
						0x8080,  /* X       X        */
						0x4040,  /*  X       X       */
						0x2020,  /*   X       X      */
						0x1010,  /*    X       X     */
						0x0808,  /*     X       X    */
						0x0404}, /*      X       X   */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pwn_lay = {LAYERO,BROWN, PATTERNED, PATTERNED,
/* pseudo N Well */		{0x0002, /*               X  */
						0x0000,  /*                  */
						0x0000,  /*                  */
						0x0000,  /*                  */
						0x0200,  /*       X          */
						0x0000,  /*                  */
						0x0000,  /*                  */
						0x0000}, /*                  */
						NOVARIABLE, 0};
static GRAPHICS mocmossub_pf_lay = {LAYERO, RED, SOLIDC, PATTERNED,
/* pad frame */		{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};

/* these tables must be updated together */
GRAPHICS *mocmossub_layers[MAXLAYERS+1] = {
	&mocmossub_m1_lay, &mocmossub_m2_lay, &mocmossub_m3_lay,	/* metal 1/2/3 */
	&mocmossub_m4_lay, &mocmossub_m5_lay, &mocmossub_m6_lay,	/* metal 4/5/6 */
	&mocmossub_p1_lay, &mocmossub_p2_lay,						/* poly 1/2 */
	&mocmossub_sa_lay, &mocmossub_da_lay,						/* P/N active */
	&mocmossub_ssp_lay, &mocmossub_ssn_lay,						/* P/N select */
	&mocmossub_wp_lay, &mocmossub_wn_lay,						/* P/N well */
	&mocmossub_pc_lay, &mocmossub_ac_lay,						/* poly/act cut */
	&mocmossub_v1_lay, &mocmossub_v2_lay, &mocmossub_v3_lay,	/* via 1/2/3 */
	&mocmossub_v4_lay, &mocmossub_v5_lay,						/* via 4/5 */
	&mocmossub_ovs_lay,											/* overglass */
	&mocmossub_tr_lay,											/* transistor */
	&mocmossub_cp_lay,											/* poly cap */
	&mocmossub_saw_lay,											/* P active well */
	&mocmossub_pm1_lay, &mocmossub_pm2_lay,						/* pseudo metal 1/2 */
	&mocmossub_pm3_lay, &mocmossub_pm4_lay,						/* pseudo metal 3/4 */
	&mocmossub_pm5_lay, &mocmossub_pm6_lay,						/* pseudo metal 5/6 */
	&mocmossub_pp1_lay, &mocmossub_pp2_lay,						/* pseudo poly 1/2 */
	&mocmossub_psa_lay, &mocmossub_pda_lay,						/* pseudo P/N active */
	&mocmossub_pssp_lay, &mocmossub_pssn_lay,					/* pseudo P/N select */
	&mocmossub_pwp_lay, &mocmossub_pwn_lay,						/* pseudo P/N well */
	&mocmossub_pf_lay, NOGRAPHICS};								/* pad frame */
static char *mocmossub_layer_names[MAXLAYERS] = {
	"Metal-1", "Metal-2", "Metal-3",							/* metal 1/2/3 */
	"Metal-4", "Metal-5", "Metal-6",							/* metal 4/5/6 */
	"Polysilicon-1", "Polysilicon-2",							/* poly 1/2 */
	"P-Active", "N-Active",										/* P/N active */
	"P-Select", "N-Select",										/* P/N select */
	"P-Well", "N-Well",											/* P/N well */
	"Poly-Cut", "Active-Cut",									/* poly/act cut */
	"Via1", "Via2", "Via3",										/* via 1/2/3 */
	"Via4", "Via5",												/* via 4/5 */
	"Passivation",												/* overglass */
	"Transistor",												/* transistor */
	"Poly-Cap",													/* poly cap */
	"P-Active-Well",											/* P active well */
	"Pseudo-Metal-1", "Pseudo-Metal-2",							/* pseudo metal 1/2 */
	"Pseudo-Metal-3", "Pseudo-Metal-4",							/* pseudo metal 3/4 */
	"Pseudo-Metal-5", "Pseudo-Metal-6",							/* pseudo metal 5/6 */
	"Pseudo-Polysilicon", "Pseudo-Electrode",					/* pseudo poly 1/2 */
	"Pseudo-P-Active", "Pseudo-N-Active",						/* pseudo P/N active */
	"Pseudo-P-Select", "Pseudo-N-Select",						/* pseudo P/N select */
	"Pseudo-P-Well", "Pseudo-N-Well",							/* pseudo P/N well */
	"Pad-Frame"};												/* pad frame */
static INTBIG mocmossub_layer_function[MAXLAYERS] = {
	LFMETAL1|LFTRANS1, LFMETAL2|LFTRANS4, LFMETAL3|LFTRANS5,	/* metal 1/2/3 */
	LFMETAL4, LFMETAL5, LFMETAL6,								/* metal 4/5/6 */
	LFPOLY1|LFTRANS2, LFPOLY2,									/* poly 1/2 */
	LFDIFF|LFPTYPE|LFTRANS3, LFDIFF|LFNTYPE|LFTRANS3,			/* P/N active */
	LFIMPLANT|LFPTYPE, LFIMPLANT|LFNTYPE,						/* P/N select */
	LFWELL|LFPTYPE, LFWELL|LFNTYPE,								/* P/N well */
	LFCONTACT1|LFCONPOLY, LFCONTACT1|LFCONDIFF,					/* poly/act cut */
	LFCONTACT2|LFCONMETAL, LFCONTACT3|LFCONMETAL, LFCONTACT4|LFCONMETAL,	/* via 1/2/3 */
	LFCONTACT5|LFCONMETAL, LFCONTACT6|LFCONMETAL,				/* via 4/5 */
	LFOVERGLASS,												/* overglass */
	LFTRANSISTOR|LFPSEUDO,										/* transistor */
	LFCAP,														/* poly cap */
	LFDIFF|LFPTYPE|LFTRANS3,									/* P active well */
	LFMETAL1|LFPSEUDO|LFTRANS1,LFMETAL2|LFPSEUDO|LFTRANS4,		/* pseudo metal 1/2 */
	LFMETAL3|LFPSEUDO|LFTRANS5, LFMETAL4|LFPSEUDO,				/* pseudo metal 3/4 */
	LFMETAL5|LFPSEUDO, LFMETAL6|LFPSEUDO,						/* pseudo metal 5/6 */
	LFPOLY1|LFPSEUDO|LFTRANS2, LFPOLY2|LFPSEUDO,				/* pseudo poly 1/2 */
	LFDIFF|LFPTYPE|LFPSEUDO|LFTRANS3,							/* pseudo P/N active */
		LFDIFF|LFNTYPE|LFPSEUDO|LFTRANS3,
	LFIMPLANT|LFPTYPE|LFPSEUDO, LFIMPLANT|LFNTYPE|LFPSEUDO,		/* pseudo P/N select */
	LFWELL|LFPTYPE|LFPSEUDO, LFWELL|LFNTYPE|LFPSEUDO,			/* pseudo P/N well */
	LFART};														/* pad frame */
static char *mocmossub_cif_layers[MAXLAYERS] = {
	"CMF", "CMS", "CMT", "CMQ", "CMP", "CM6",					/* metal 1/2/3/4/5/6 */
	"CPG", "CEL",												/* poly 1/2 */
	"CAA", "CAA",												/* P/N active */
	"CSP", "CSN",												/* P/N select */
	"CWP", "CWN",												/* P/N well */
	"CCG", "CCG",												/* poly/act cut */
	"CVA", "CVS", "CVT", "CVQ", "CV5",							/* via 1/2/3/4/5 */
	"COG",														/* overglass */
	"",															/* transistor */
	"CPC",														/* poly cap */
	"CAA",														/* P active well */
	"", "", "", "", "", "",										/* pseudo metal 1/2/3/4/5/6 */
	"", "",														/* pseudo poly 1/2 */
	"", "",														/* pseudo P/N active */
	"CSP", "CSN",												/* pseudo P/N select */
	"CWP", "CWN",												/* pseudo P/N well */
	"CX"};														/* pad frame */
static INTBIG mocmossub_gds_layers[MAXLAYERS] = {
	49, 51, 62, 31, 33, 38,										/* metal 1/2/3/4/5/6 */
	46, 56,														/* poly 1/2 */
	43, 43,														/* P/N active */
	44, 45,														/* P/N select */
	41, 42,														/* P/N well */
	25, 25,														/* poly/act cut */
	50, 61, 30, 32, 39,											/* via 1/2/3/4/5 */
	52,															/* overglass */
	-1,															/* transistor */
	28,															/* poly cap */
	43,															/* P active well */
	-1, -1, -1, -1, -1, -1,										/* pseudo metal 1/2/3/4/5/6 */
	-1, -1,														/* pseudo poly 1/2 */
	-1, -1,														/* pseudo P/N active */
	-1, -1,														/* pseudo P/N select */
	-1, -1,														/* pseudo P/N well */
	19};														/* pad frame */
static INTBIG mocmossub_minimum_width[MAXLAYERS] = {
	K3, K3, K3, K3, K3, K3,										/* metal 1/2/3/4/5/6 */
	K2, XX,														/* poly 1/2 */
	K3, K3,														/* P/N active */
	K2, K2,														/* P/N select */
	K12, K12,													/* P/N well */
	XX, XX,														/* poly/act cut */
	XX, XX, XX, XX, XX,											/* via 1/2/3/4/5 */
	XX,															/* overglass */
	XX,															/* transistor */
	XX,															/* poly cap */
	XX,															/* P active well */
	XX, XX, XX, XX, XX, XX,										/* pseudo metal 1/2/3/4/5/6 */
	XX, XX,														/* pseudo poly 1/2 */
	XX, XX,														/* pseudo P/N active */
	K2, K2,														/* pseudo P/N select */
	K12, K12,													/* pseudo P/N well */
	XX};														/* pad frame */
static char *mocmossub_skill_layers[MAXLAYERS] = {
	"metal1", "metal2", "metal3",								/* metal 1/2/3 */
	"metal4", "metal5", "metal6",								/* metal 4/5/6 */
	"poly", "",													/* poly 1/2 */
	"aa", "aa",													/* P/N active */
	"pplus", "nplus",											/* P/N select */
	"pwell", "nwell",											/* P/N well */
	"pcont", "acont",											/* poly/act cut */
	"via", "via2", "via3", "via4", "via5",						/* via 1/2/3/4/5 */
	"glasscut",													/* overglass */
	"",															/* transistor */
	"",															/* poly cap */
	"aa",														/* P active well */
	"", "", "", "", "", "",										/* pseudo metal 1/2/3/4/5/6 */
	"", "",														/* pseudo poly 1/2 */
	"", "",														/* pseudo P/N active */
	"pplus", "nplus",											/* pseudo P/N select */
	"pwell", "nwell",											/* pseudo P/N well */
	""};														/* pad frame */
static INTBIG mocmossub_3dthick_layers[MAXLAYERS] = {
	0, 0, 0, 0, 0, 0,											/* metal 1/2/3/4/5/6 */
	0, 0,														/* poly 1/2 */
	0, 0,														/* P/N active */
	0, 0,														/* P/N select */
	0, 0,														/* P/N well */
	2, 4,														/* poly/act cut */
	2, 2, 2, 2, 2,												/* via 1/2/3/4/5 */
	0,															/* overglass */
	0,															/* transistor */
	0,															/* poly cap */
	0,															/* P active well */
	0, 0, 0, 0, 0, 0,											/* pseudo metal 1/2/3/4/5/6 */
	0, 0,														/* pseudo poly 1/2 */
	0, 0,														/* pseudo P/N active */
	0, 0,														/* pseudo P/N select */
	0, 0,														/* pseudo P/N well */
	0};															/* pad frame */
static INTBIG mocmossub_3dheight_layers[MAXLAYERS] = {
	17, 19, 21, 23, 25, 27,										/* metal 1/2/3/4/5/6 */
	15, 16,														/* poly 1/2 */
	13, 13,														/* P/N active */
	12, 12,														/* P/N select */
	11, 11,														/* P/N well */
	16, 15,														/* poly/act cut */
	18, 20, 22, 24, 26,											/* via 1/2/3/4/5 */
	30,															/* overglass */
	31,															/* transistor */
	28,															/* poly cap */
	29,															/* P active well */
	17, 19, 21, 23, 25, 27,										/* pseudo metal 1/2/3/4/5/6 */
	12, 13,														/* pseudo poly 1/2 */
	11, 11,														/* pseudo P/N active */
	2, 2,														/* pseudo P/N select */
	0, 0,														/* pseudo P/N well */
	33};														/* pad frame */
/* there are no available letters */
static char *mocmossub_layer_letters[MAXLAYERS] = {
	"m", "h", "r", "q", "a", "c",								/* metal 1/2/3/4/5/6 */
	"p", "l",													/* poly 1/2 */
	"s", "d",													/* P/N active */
	"e", "f",													/* P/N select */
	"w", "n",													/* P/N well */
	"j", "k",													/* poly/act cut */
	"v", "u", "z", "i", "y",									/* via 1/2/3/4/5 */
	"o",														/* overglass */
	"t",														/* transistor */
	"g",														/* poly cap */
	"x",														/* P active well */
	"M", "H", "R", "Q", "A", "C",								/* pseudo metal 1/2/3/4/5/6 */
	"P", "L",													/* pseudo poly 1/2 */
	"S", "D",													/* pseudo P/N active */
	"E", "F",													/* pseudo P/N select */
	"W", "N",													/* pseudo P/N well */
	"b"};														/* pad frame */
/* The low 5 bits map Metal-1, Poly-1, Active, Metal-2, and Metal-3 */
static TECH_COLORMAP mocmossub_colmap[32] =
{                  /*     Metal-3 Metal-2 Active Polysilicon-1 Metal-1 */
	{200,200,200}, /*  0:                                              */
	{ 96,209,255}, /*  1:                                      Metal-1 */
	{255,155,192}, /*  2:                        Polysilicon-1         */
	{111,144,177}, /*  3:                        Polysilicon-1 Metal-1 */
	{107,226, 96}, /*  4:                 Active                       */
	{ 83,179,160}, /*  5:                 Active               Metal-1 */
	{161,151,126}, /*  6:                 Active Polysilicon-1         */
	{110,171,152}, /*  7:                 Active Polysilicon-1 Metal-1 */
	{224, 95,255}, /*  8:         Metal-2                              */
	{135,100,191}, /*  9:         Metal-2                      Metal-1 */
	{170, 83,170}, /* 10:         Metal-2        Polysilicon-1         */
	{152,104,175}, /* 11:         Metal-2        Polysilicon-1 Metal-1 */
	{150,124,163}, /* 12:         Metal-2 Active                       */
	{129,144,165}, /* 13:         Metal-2 Active               Metal-1 */
	{155,133,151}, /* 14:         Metal-2 Active Polysilicon-1         */
	{141,146,153}, /* 15:         Metal-2 Active Polysilicon-1 Metal-1 */
	{247,251, 20}, /* 16: Metal-3                                      */
	{154,186, 78}, /* 17: Metal-3                              Metal-1 */
	{186,163, 57}, /* 18: Metal-3                Polysilicon-1         */
	{167,164, 99}, /* 19: Metal-3                Polysilicon-1 Metal-1 */
	{156,197, 41}, /* 20: Metal-3         Active                       */
	{138,197, 83}, /* 21: Metal-3         Active               Metal-1 */
	{161,184, 69}, /* 22: Metal-3         Active Polysilicon-1         */
	{147,183, 97}, /* 23: Metal-3         Active Polysilicon-1 Metal-1 */
	{186,155, 76}, /* 24: Metal-3 Metal-2                              */
	{155,163,119}, /* 25: Metal-3 Metal-2                      Metal-1 */
	{187,142, 97}, /* 26: Metal-3 Metal-2        Polysilicon-1         */
	{165,146,126}, /* 27: Metal-3 Metal-2        Polysilicon-1 Metal-1 */
	{161,178, 82}, /* 28: Metal-3 Metal-2 Active                       */
	{139,182,111}, /* 29: Metal-3 Metal-2 Active               Metal-1 */
	{162,170, 97}, /* 30: Metal-3 Metal-2 Active Polysilicon-1         */
	{147,172,116}  /* 31: Metal-3 Metal-2 Active Polysilicon-1 Metal-1 */
};

/******************** DESIGN RULES ********************/

#define	X	XX
#define	A	K1
#define	B	K2
#define	C	K3
#define	D	K4
#define	E	K5
#define	F	K6
#define	R	K18

/* layers that can connect to other layers when electrically disconnected */
static INTBIG mocmossub_unconnectedtable[] = {
/*          M M M M M M P P P N S S W W P A V V V V V P T P P M M M M M M P P P N S S W W P */
/*          e e e e e e o o A A e e e e o c i i i i i a r C a e e e e e e o o A A e e e e a */
/*          t t t t t t l l c c l l l l l t a a a a a s a a c t t t t t t l l c c l l l l d */
/*          1 2 3 4 5 6 y y t t P N l l y C 1 2 3 4 5 s n p t 1 2 3 4 5 6 1 2 t t P N P N F */
/*                      1 2         P N C               s   W P P P P P P P P P P P P P P r */
/* Met1  */ C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met2  */   C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met3  */     C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met4  */       C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met5  */         C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met6  */           C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Poly1 */             C,X,A,A,X,X,X,X,X,B,X,X,X,X,X,X,X,X,A,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Poly2 */               C,A,A,X,X,X,X,C,X,X,X,X,X,X,X,X,X,A,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PAct  */                 C,C,X,X,X,X,B,X,X,X,X,X,X,X,X,X,C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* NAct  */                   C,X,X,X,X,B,X,X,X,X,X,X,X,X,X,C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* SelP  */                     B,0,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* SelN  */                       B,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* WellP */                         R,0,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* WellN */                           R,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PolyC */                             C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* ActC  */                               C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via1  */                                 C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via2  */                                   C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via3  */                                     D,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via4  */                                       C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via5  */                                         C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Pass  */                                           X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Trans */                                             X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PCap  */                                               X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PactW */                                                 C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met1P */                                                   X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met2P */                                                     X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met3P */                                                       X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met4P */                                                         X,X,X,X,X,X,X,X,X,X,X,X,
/* Met5P */                                                           X,X,X,X,X,X,X,X,X,X,X,
/* Met6P */                                                             X,X,X,X,X,X,X,X,X,X,
/* Poly1P */                                                              X,X,X,X,X,X,X,X,X,
/* Poly2P */                                                                X,X,X,X,X,X,X,X,
/* PActP */                                                                   X,X,X,X,X,X,X,
/* NActP */                                                                     X,X,X,X,X,X,
/* SelPP */                                                                       X,X,X,X,X,
/* SelNP */                                                                         X,X,X,X,
/* WelPP */                                                                           X,X,X,
/* WelNP */                                                                             X,X,
/* PadFr */                                                                               X,
};

/* layers that can connect to other layers when electrically connected */
static INTBIG mocmossub_connectedtable[] = {
/*          M M M M M M P P P N S S W W P A V V V V V P T P P M M M M M M P P P N S S W W P */
/*          e e e e e e o o A A e e e e o c i i i i i a r C a e e e e e e o o A A e e e e a */
/*          t t t t t t l l c c l l l l l t a a a a a s a a c t t t t t t l l c c l l l l d */
/*          1 2 3 4 5 6 y y t t P N l l y C 1 2 3 4 5 s n p t 1 2 3 4 5 6 1 2 t t P N P N F */
/*                      1 2         P N C               s   W P P P P P P P P P P P P P P r */
/* Met1  */ C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met2  */   C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met3  */     C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met4  */       C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met5  */         C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met6  */           C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Poly1 */             C,X,A,A,X,X,X,X,X,X,X,X,X,X,X,X,X,X,A,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Poly2 */               C,A,A,X,X,X,X,X,X,X,X,X,X,X,X,X,X,A,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PAct  */                 C,C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* NAct  */                   C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* SelP  */                     X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* SelN  */                       X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* WellP */                         F,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* WellN */                           F,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PolyC */                             C,C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* ActC  */                               C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via1  */                                 C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via2  */                                   C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via3  */                                     D,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via4  */                                       C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Via5  */                                         C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Pass  */                                           X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Trans */                                             X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PCap  */                                               X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* PactW */                                                 C,X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met1P */                                                   X,X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met2P */                                                     X,X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met3P */                                                       X,X,X,X,X,X,X,X,X,X,X,X,X,
/* Met4P */                                                         X,X,X,X,X,X,X,X,X,X,X,X,
/* Met5P */                                                           X,X,X,X,X,X,X,X,X,X,X,
/* Met6P */                                                             X,X,X,X,X,X,X,X,X,X,
/* Poly1P */                                                              X,X,X,X,X,X,X,X,X,
/* Poly2P */                                                                X,X,X,X,X,X,X,X,
/* PActP */                                                                   X,X,X,X,X,X,X,
/* NActP */                                                                     X,X,X,X,X,X,
/* SelPP */                                                                       X,X,X,X,X,
/* SelNP */                                                                         X,X,X,X,
/* WelPP */                                                                           X,X,X,
/* WelNP */                                                                             X,X,
/* PadFr */                                                                               X,
};

/******************** ARCS ********************/

#define	ARCPROTOCOUNT 11
#define	AMETAL1        0	/* metal-1                   */
#define	AMETAL2        1	/* metal-2                   */
#define	AMETAL3        2	/* metal-3                   */
#define	AMETAL4        3	/* metal-4                   */
#define	AMETAL5        4	/* metal-5                   */
#define	AMETAL6        5	/* metal-6                   */
#define	APOLY1         6	/* polysilicon-1             */
#define	APOLY2         7	/* polysilicon-2             */
#define	ASACT          8	/* P-active (or N)           */
#define	ADACT          9	/* N-active (or P)           */
#define	AACT          10	/* General active            */

/* metal 1 arc */
static TECH_ARCLAY mocmossub_al_m1[] = {{LMETAL1,0,FILLED }};
static TECH_ARCS mocmossub_a_m1 = {
	"Metal-1",K3,AMETAL1,											/* name */
	1,mocmossub_al_m1,												/* layers */
	(APMETAL1<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* metal 2 arc */
static TECH_ARCLAY mocmossub_al_m2[] = {{LMETAL2,0,FILLED }};
static TECH_ARCS mocmossub_a_m2 = {
	"Metal-2",K3,AMETAL2,											/* name */
	1,mocmossub_al_m2,												/* layers */
	(APMETAL2<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* metal 3 arc */
static TECH_ARCLAY mocmossub_al_m3[] = {{LMETAL3,0,FILLED }};
static TECH_ARCS mocmossub_a_m3 = {
	"Metal-3",K3,AMETAL3,											/* name */
	1,mocmossub_al_m3,												/* layers */
	(APMETAL3<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* metal 4 arc */
static TECH_ARCLAY mocmossub_al_m4[] = {{LMETAL4,0,FILLED }};
static TECH_ARCS mocmossub_a_m4 = {
	"Metal-4",K3,AMETAL4,											/* name */
	1,mocmossub_al_m4,												/* layers */
	(APMETAL4<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* metal 5 arc */
static TECH_ARCLAY mocmossub_al_m5[] = {{LMETAL5,0,FILLED }};
static TECH_ARCS mocmossub_a_m5 = {
	"Metal-5",K3,AMETAL5,											/* name */
	1,mocmossub_al_m5,												/* layers */
	(APMETAL5<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* metal 6 arc */
static TECH_ARCLAY mocmossub_al_m6[] = {{LMETAL6,0,FILLED }};
static TECH_ARCS mocmossub_a_m6 = {
	"Metal-6",K3,AMETAL6,											/* name */
	1,mocmossub_al_m6,												/* layers */
	(APMETAL6<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* polysilicon 1 arc */
static TECH_ARCLAY mocmossub_al_p1[] = {{LPOLY1,0,FILLED }};
static TECH_ARCS mocmossub_a_po1 = {
	"Polysilicon-1",K2,APOLY1,										/* name */
	1,mocmossub_al_p1,												/* layers */
	(APPOLY1<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* polysilicon 2 arc */
static TECH_ARCLAY mocmossub_al_p2[] = {{LPOLY2,0,FILLED }};
static TECH_ARCS mocmossub_a_po2 = {
	"Polysilicon-2",K3,APOLY2,										/* name */
	1,mocmossub_al_p2,												/* layers */
	(APPOLY2<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* P-active arc */
static TECH_ARCLAY mocmossub_al_pa[] = {{LSACT,K12,FILLED}, {LWELLN,0,FILLED},
	{LSELECTP,K8,FILLED}};
static TECH_ARCS mocmossub_a_pa = {
	"P-Active",K15,ASACT,											/* name */
	3,mocmossub_al_pa,												/* layers */
	(APDIFFP<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* N-active arc */
static TECH_ARCLAY mocmossub_al_na[] = {{LDACT,K12,FILLED}, {LWELLP,0,FILLED},
	{LSELECTN,K8,FILLED}};
static TECH_ARCS mocmossub_a_na = {
	"N-Active",K15,ADACT,											/* name */
	3,mocmossub_al_na,												/* layers */
	(APDIFFN<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

/* General active arc */
static TECH_ARCLAY mocmossub_al_a[] = {{LDACT,0,FILLED}, {LSACT,0,FILLED}};
static TECH_ARCS mocmossub_a_a = {
	"Active",K3,AACT,												/* name */
	2,mocmossub_al_a,												/* layers */
	(APDIFF<<AFUNCTIONSH)|WANTFIXANG|CANWIPE|(90<<AANGLEINCSH)};	/* userbits */

TECH_ARCS *mocmossub_arcprotos[ARCPROTOCOUNT+1] = {
	&mocmossub_a_m1, &mocmossub_a_m2, &mocmossub_a_m3,
	&mocmossub_a_m4, &mocmossub_a_m5, &mocmossub_a_m6,
	&mocmossub_a_po1, &mocmossub_a_po2,
	&mocmossub_a_pa, &mocmossub_a_na,
	&mocmossub_a_a, ((TECH_ARCS *)-1)};

static INTBIG mocmossub_arc_widoff[ARCPROTOCOUNT] = {0,0,0,0,0,0, 0,0, K12,K12, 0};

/******************** PORTINST CONNECTIONS ********************/

static INTBIG mocmossub_pc_m1[]   = {-1, AMETAL1, ALLGEN, -1};
static INTBIG mocmossub_pc_m1a[]  = {-1, AMETAL1, AACT, ALLGEN, -1};
static INTBIG mocmossub_pc_m2[]   = {-1, AMETAL2, ALLGEN, -1};
static INTBIG mocmossub_pc_m3[]   = {-1, AMETAL3, ALLGEN, -1};
static INTBIG mocmossub_pc_m4[]   = {-1, AMETAL4, ALLGEN, -1};
static INTBIG mocmossub_pc_m5[]   = {-1, AMETAL5, ALLGEN, -1};
static INTBIG mocmossub_pc_m6[]   = {-1, AMETAL6, ALLGEN, -1};
static INTBIG mocmossub_pc_p1[]   = {-1, APOLY1, ALLGEN, -1};
static INTBIG mocmossub_pc_p2[]   = {-1, APOLY2, ALLGEN, -1};
static INTBIG mocmossub_pc_pa[]   = {-1, ASACT, ALLGEN, -1};
static INTBIG mocmossub_pc_a[]    = {-1, AACT, ASACT, ADACT, ALLGEN,-1};
static INTBIG mocmossub_pc_na[]   = {-1, ADACT, ALLGEN, -1};
static INTBIG mocmossub_pc_pam1[] = {-1, ASACT, AMETAL1, ALLGEN, -1};
static INTBIG mocmossub_pc_nam1[] = {-1, ADACT, AMETAL1, ALLGEN, -1};
static INTBIG mocmossub_pc_pm1[]  = {-1, APOLY1, AMETAL1, ALLGEN, -1};
static INTBIG mocmossub_pc_pm2[]  = {-1, APOLY2, AMETAL1, ALLGEN, -1};
static INTBIG mocmossub_pc_pm12[] = {-1, APOLY1, APOLY2, AMETAL1, ALLGEN, -1};
static INTBIG mocmossub_pc_m1m2[] = {-1, AMETAL1, AMETAL2, ALLGEN, -1};
static INTBIG mocmossub_pc_m2m3[] = {-1, AMETAL2, AMETAL3, ALLGEN, -1};
static INTBIG mocmossub_pc_m3m4[] = {-1, AMETAL3, AMETAL4, ALLGEN, -1};
static INTBIG mocmossub_pc_m4m5[] = {-1, AMETAL4, AMETAL5, ALLGEN, -1};
static INTBIG mocmossub_pc_m5m6[] = {-1, AMETAL5, AMETAL6, ALLGEN, -1};
static INTBIG mocmossub_pc_null[] = {-1, ALLGEN, -1};

/******************** NODES ********************/

#define	NODEPROTOCOUNT 50
#define	NMETAL1P        1	/* metal-1 pin */
#define	NMETAL2P        2	/* metal-2 pin */
#define	NMETAL3P        3	/* metal-3 pin */
#define	NMETAL4P        4	/* metal-4 pin */
#define	NMETAL5P        5	/* metal-5 pin */
#define	NMETAL6P        6	/* metal-6 pin */
#define	NPOLY1P         7	/* polysilicon-1 pin */
#define	NPOLY2P         8	/* polysilicon-2 pin */
#define	NSACTP          9	/* P-active (or N) pin */
#define	NDACTP         10	/* N-active (or P) pin */
#define	NACTP          11	/* General active pin */
#define	NMETSACTC      12	/* metal-1-P-active (or N) contact */
#define	NMETDACTC      13	/* metal-1-N-active (or P) contact */
#define	NMETPOLY1C     14	/* metal-1-polysilicon-1 contact */
#define	NMETPOLY2C     15	/* metal-1-polysilicon-2 contact */
#define	NMETPOLY12C    16	/* metal-1-polysilicon-1-2 capacitor/contact */
#define	NSTRANS        17	/* P-transistor (or N) */
#define	NDTRANS        18	/* N-transistor (or P) */
#define	NVIA1          19	/* metal-1-metal-2 contact */
#define	NVIA2          20	/* metal-2-metal-3 contact */
#define	NVIA3          21	/* metal-3-metal-4 contact */
#define	NVIA4          22	/* metal-4-metal-5 contact */
#define	NVIA5          23	/* metal-5-metal-6 contact */
#define	NWBUT          24	/* metal-1-Well contact */
#define	NSBUT          25	/* metal-1-Substrate contact */
#define	NMETAL1N       26	/* metal-1 node */
#define	NMETAL2N       27	/* metal-2 node */
#define	NMETAL3N       28	/* metal-3 node */
#define	NMETAL4N       29	/* metal-4 node */
#define	NMETAL5N       30	/* metal-5 node */
#define	NMETAL6N       31	/* metal-6 node */
#define	NPOLY1N        32	/* polysilicon-1 node */
#define	NPOLY2N        33	/* polysilicon-2 node */
#define	NACTIVEN       34	/* active(P) node */
#define	NDACTIVEN      35	/* N-active node */
#define	NSELECTPN      36	/* P-select node */
#define	NSELECTNN      37	/* N-select node */
#define	NPCUTN         38	/* poly cut node */
#define	NACUTN         39	/* active cut node */
#define	NVIA1N         40	/* via-1 node */
#define	NVIA2N         41	/* via-2 node */
#define	NVIA3N         42	/* via-3 node */
#define	NVIA4N         43	/* via-4 node */
#define	NVIA5N         44	/* via-5 node */
#define	NWELLPN        45	/* P-well node */
#define	NWELLNN        46	/* N-well node */
#define	NPASSN         47	/* passivation node */
#define	NPADFRN        48	/* pad frame node */
#define	NPOLYCAPN      49	/* polysilicon-capacitor node */
#define	NPACTWELLN     50	/* P-active-Well node */

/* for geometry */
static INTBIG mocmossub_cutbox[8]  = {LEFTIN1,  BOTIN1,   LEFTIN3,   BOTIN3};/*adjust*/
static INTBIG mocmossub_fullbox[8] = {LEFTEDGE, BOTEDGE,  RIGHTEDGE, TOPEDGE};
static INTBIG mocmossub_in0hbox[8] = {LEFTIN0H, BOTIN0H,  RIGHTIN0H, TOPIN0H};
static INTBIG mocmossub_in1box[8]  = {LEFTIN1,  BOTIN1,   RIGHTIN1,  TOPIN1};
static INTBIG mocmossub_in2box[8]  = {LEFTIN2,  BOTIN2,   RIGHTIN2,  TOPIN2};
static INTBIG mocmossub_in4box[8]  = {LEFTIN4,  BOTIN4,   RIGHTIN4,  TOPIN4};
static INTBIG mocmossub_in6box[8]  = {LEFTIN6,  BOTIN6,   RIGHTIN6,  TOPIN6};
static INTBIG mocmossub_in6hbox[8] = {LEFTIN6H, BOTIN6H,  RIGHTIN6H, TOPIN6H};
static INTBIG mocmossub_min5box[16]= {LEFTIN5,  BOTIN5,   RIGHTIN5,  TOPIN5,
									 CENTERL2, CENTERD2, CENTERR2,  CENTERU2};
#if 0    /* new MOSIS CMOS Deep rules require gate to overlap by 2.5, not 2.0 */
static INTBIG mocmossub_trnpbox[8] = {LEFTIN4,  BOTIN9,   RIGHTIN4,  TOPIN9};
#else
static INTBIG mocmossub_trnpbox[8] = {LEFTIN3H, BOTIN9,   RIGHTIN3H, TOPIN9};
#endif
static INTBIG mocmossub_trntbox[8] = {LEFTIN6,  BOTIN9,   RIGHTIN6,  TOPIN9};
static INTBIG mocmossub_trna1box[8]= {LEFTIN6,  BOTIN9,   RIGHTIN6,  TOPIN9};
static INTBIG mocmossub_trna2box[8]= {LEFTIN6,  TOPIN9,   RIGHTIN6,  TOPIN6};
static INTBIG mocmossub_trna3box[8]= {LEFTIN6,  BOTIN6,   RIGHTIN6,  BOTIN9};

/* metal-1-pin */
static TECH_PORTS mocmossub_pm1_p[] = {				/* ports */
	{mocmossub_pc_m1, "metal-1", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_pm1_l[] = {			/* layers */
	{LMET1P, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pm1 = {
	"Metal-1-Pin",NMETAL1P,NONODEPROTO,				/* name */
	K3,K3,											/* size */
	1,mocmossub_pm1_p,								/* ports */
	1,mocmossub_pm1_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0,0,0};								/* characteristics */

/* metal-2-pin */
static TECH_PORTS mocmossub_pm2_p[] = {				/* ports */
	{mocmossub_pc_m2, "metal-2", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_pm2_l[] = {			/* layers */
	{LMET2P, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pm2 = {
	"Metal-2-Pin",NMETAL2P,NONODEPROTO,				/* name */
	K3,K3,											/* size */
	1,mocmossub_pm2_p,								/* ports */
	1,mocmossub_pm2_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0,0,0};								/* characteristics */

/* metal-3-pin */
static TECH_PORTS mocmossub_pm3_p[] = {				/* ports */
	{mocmossub_pc_m3, "metal-3", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_pm3_l[] = {			/* layers */
	{LMET3P, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pm3 = {
	"Metal-3-Pin",NMETAL3P,NONODEPROTO,				/* name */
	K3,K3,											/* size */
	1,mocmossub_pm3_p,								/* ports */
	1,mocmossub_pm3_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0,0,0};								/* characteristics */

/* metal-4-pin */
static TECH_PORTS mocmossub_pm4_p[] = {				/* ports */
	{mocmossub_pc_m4, "metal-4", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_pm4_l[] = {			/* layers */
	{LMET4P, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pm4 = {
	"Metal-4-Pin",NMETAL4P,NONODEPROTO,				/* name */
	K3,K3,											/* size */
	1,mocmossub_pm4_p,								/* ports */
	1,mocmossub_pm4_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0,0,0};								/* characteristics */

/* metal-5-pin */
static TECH_PORTS mocmossub_pm5_p[] = {				/* ports */
	{mocmossub_pc_m5, "metal-5", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_pm5_l[] = {			/* layers */
	{LMET5P, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pm5 = {
	"Metal-5-Pin",NMETAL5P,NONODEPROTO,				/* name */
	K3,K3,											/* size */
	1,mocmossub_pm5_p,								/* ports */
	1,mocmossub_pm5_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK|NNOTUSED,	/* userbits */
	0,0,0,0,0,0,0,0,0};								/* characteristics */

/* metal-6-pin */
static TECH_PORTS mocmossub_pm6_p[] = {				/* ports */
	{mocmossub_pc_m6, "metal-6", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_pm6_l[] = {			/* layers */
	{LMET6P, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pm6 = {
	"Metal-6-Pin",NMETAL6P,NONODEPROTO,				/* name */
	K3,K3,											/* size */
	1,mocmossub_pm6_p,								/* ports */
	1,mocmossub_pm6_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK|NNOTUSED,	/* userbits */
	0,0,0,0,0,0,0,0,0};								/* characteristics */

/* polysilicon-1-pin */
static TECH_PORTS mocmossub_pp1_p[] = {				/* ports */
	{mocmossub_pc_p1, "polysilicon-1", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1, BOTIN1, RIGHTIN1, TOPIN1}};
static TECH_POLYGON mocmossub_pp1_l[] = {			/* layers */
	{LPOLY1P, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pp1 = {
	"Polysilicon-1-Pin",NPOLY1P,NONODEPROTO,		/* name */
	K2,K2,											/* size */
	1,mocmossub_pp1_p,								/* ports */
	1,mocmossub_pp1_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0,0,0};								/* characteristics */

/* polysilicon-2-pin */
static TECH_PORTS mocmossub_pp2_p[] = {				/* ports */
	{mocmossub_pc_p2, "polysilicon-2", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_pp2_l[] = {			/* layers */
	{LPOLY2P, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pp2 = {
	"Polysilicon-2-Pin",NPOLY2P,NONODEPROTO,		/* name */
	K3,K3,											/* size */
	1,mocmossub_pp2_p,								/* ports */
	1,mocmossub_pp2_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0,0,0};								/* characteristics */

/* P-active-pin (or N) */
static TECH_PORTS mocmossub_psa_p[] = {				/* ports */
	{mocmossub_pc_pa, "p-active", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN7H, BOTIN7H, RIGHTIN7H, TOPIN7H}};
static TECH_POLYGON mocmossub_psa_l[] = {			/* layers */
	{LSACTP,    0, 4, CROSSED, BOX, mocmossub_in6box},
	{LWELLNP,  -1, 4, CROSSED, BOX, mocmossub_fullbox},
	{LSELECTPP,-1, 4, CROSSED, BOX, mocmossub_in4box}};
static TECH_NODES mocmossub_psa = {
	"P-Active-Pin",NSACTP,NONODEPROTO,				/* name */
	K15,K15,										/* size */
	1,mocmossub_psa_p,								/* ports */
	3,mocmossub_psa_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0,0,0};								/* characteristics */

/* N-active-pin (or P) */
static TECH_PORTS mocmossub_pda_p[] = {				/* ports */
	{mocmossub_pc_na, "n-active", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN7H, BOTIN7H, RIGHTIN7H, TOPIN7H}};
static TECH_POLYGON mocmossub_pda_l[] = {			/* layers */
	{LDACTP,    0, 4, CROSSED, BOX, mocmossub_in6box},
	{LWELLPP,  -1, 4, CROSSED, BOX, mocmossub_fullbox},
	{LSELECTNP,-1, 4, CROSSED, BOX, mocmossub_in4box}};
static TECH_NODES mocmossub_pda = {
	"N-Active-Pin",NDACTP,NONODEPROTO,				/* name */
	K15,K15,										/* size */
	1,mocmossub_pda_p,								/* ports */
	3,mocmossub_pda_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0,0,0};								/* characteristics */

/* General active-pin */
static TECH_PORTS mocmossub_pa_p[] = {				/* ports */
	{mocmossub_pc_a, "active", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}}	;
static TECH_POLYGON mocmossub_pa_l[] = {			/* layers */
	{LDACTP, 0, 4, CROSSED, BOX, mocmossub_fullbox},
	{LSACTP, 0, 4, CROSSED, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pa = {
	"Active-Pin",NACTP,NONODEPROTO,					/* name */
	K3,K3,											/* size */
	1,mocmossub_pa_p,								/* ports */
	2,mocmossub_pa_l,								/* layers */
	(NPPIN<<NFUNCTIONSH)|ARCSWIPE|ARCSHRINK,		/* userbits */
	0,0,0,0,0,0,0,0,0};								/* characteristics */

/* metal-1-P-active-contact (or N) */
static TECH_PORTS mocmossub_mpa_p[] = {				/* ports */
	{mocmossub_pc_pam1, "metal-1-p-act", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN8, BOTIN8, RIGHTIN8, TOPIN8}};
static TECH_POLYGON mocmossub_mpa_l[] = {			/* layers */
	{LMETAL1,  0, 4, FILLEDRECT, BOX, mocmossub_in6hbox},
	{LSACT,    0, 4, FILLEDRECT, BOX, mocmossub_in6box},
	{LWELLN,  -1, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LSELECTP,-1, 4, FILLEDRECT, BOX, mocmossub_in4box},
	{LACTCUT,  0, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_mpa = {
	"Metal-1-P-Active-Con",NMETSACTC,NONODEPROTO,	/* name */
	K17,K17,										/* size */
	1,mocmossub_mpa_p,								/* ports */
	5,mocmossub_mpa_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,H1,K4,0,0,0,0};					/* characteristics */

/* metal-1-N-active-contact (or P) */
static TECH_PORTS mocmossub_mna_p[] = {				/* ports */
	{mocmossub_pc_nam1, "metal-1-n-act", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN8, BOTIN8, RIGHTIN8, TOPIN8}};
static TECH_POLYGON mocmossub_mna_l[] = {			/* layers */
	{LMETAL1,   0, 4, FILLEDRECT, BOX, mocmossub_in6hbox},
	{LDACT,     0, 4, FILLEDRECT, BOX, mocmossub_in6box},
	{LWELLP,   -1, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LSELECTN, -1, 4, FILLEDRECT, BOX, mocmossub_in4box},
	{LACTCUT,   0, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_mna = {
	"Metal-1-N-Active-Con",NMETDACTC,NONODEPROTO,	/* name */
	K17,K17,										/* size */
	1,mocmossub_mna_p,								/* ports */
	5,mocmossub_mna_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,H1,K4,0,0,0,0};					/* characteristics */

/* metal-1-polysilicon-1-contact */
static TECH_PORTS mocmossub_mp1_p[] = {				/* ports */
	{mocmossub_pc_pm1, "metal-1-polysilicon-1", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN2, BOTIN2, RIGHTIN2, TOPIN2}};
static TECH_POLYGON mocmossub_mp1_l[] = {			/* layers */
	{LMETAL1,  0, 4, FILLEDRECT, BOX, mocmossub_in0hbox},
	{LPOLY1,   0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LPOLYCUT, 0, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_mp1 = {
	"Metal-1-Polysilicon-1-Con",NMETPOLY1C,NONODEPROTO,/* name */
	K5,K5,											/* size */
	1,mocmossub_mp1_p,								/* ports */
	3,mocmossub_mp1_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,H1,K4,0,0,0,0};					/* characteristics */

/* metal-1-polysilicon-2-contact */
static TECH_PORTS mocmossub_mp2_p[] = {				/* ports */
	{mocmossub_pc_pm2, "metal-1-polysilicon-2", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_mp2_l[] = {			/* layers */
	{LMETAL1,  0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LPOLY2,   0, 4, FILLEDRECT, BOX, mocmossub_in0hbox},
	{LPOLYCUT, 0, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_mp2 = {
	"Metal-1-Polysilicon-2-Con",NMETPOLY2C,NONODEPROTO,/* name */
	K4,K4,											/* size */
	1,mocmossub_mp2_p,								/* ports */
	3,mocmossub_mp2_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,K1,K4,0,0,0,0};					/* characteristics */

/* metal-1-polysilicon-1-2-contact */
static TECH_PORTS mocmossub_mp12_p[] = {				/* ports */
	{mocmossub_pc_pm12, "metal-1-polysilicon-1-2", NOPORTPROTO, (180<<PORTARANGESH),
		CENTER, CENTER, CENTER, CENTER}};
static TECH_POLYGON mocmossub_mp12_l[] = {			/* layers */
	{LPOLY1,   0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LPOLY2,   0, 4, FILLEDRECT, BOX, mocmossub_in2box},
	{LPOLYCUT, 0, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_mp12 = {
	"Metal-1-Polysilicon-1-2-Con",NMETPOLY12C,NONODEPROTO,/* name */
	K7,K7,											/* size */
	1,mocmossub_mp12_p,								/* ports */
	3,mocmossub_mp12_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,H2,K4,0,0,0,0};					/* characteristics */

/* P-transistor (or N) */
static TECH_PORTS mocmossub_tpa_p[] = {				/* ports */
	{mocmossub_pc_p1, "p-trans-poly-left",  NOPORTPROTO, (180<<PORTANGLESH)|
		(90<<PORTARANGESH),                LEFTIN4,  BOTIN10, LEFTIN4,   TOPIN10},
	{mocmossub_pc_pa, "p-trans-diff-top",   NOPORTPROTO, (90<<PORTANGLESH)|
		(90<<PORTARANGESH)|(1<<PORTNETSH), LEFTIN7H,  TOPIN6H, RIGHTIN7H,  TOPIN6},
	{mocmossub_pc_p1, "p-trans-poly-right", NOPORTPROTO, (0<<PORTANGLESH)|
		(90<<PORTARANGESH),                RIGHTIN4, BOTIN10, RIGHTIN4,  TOPIN10},
	{mocmossub_pc_pa, "p-trans-diff-bottom",NOPORTPROTO, (270<<PORTANGLESH)|
		(90<<PORTARANGESH)|(2<<PORTNETSH), LEFTIN7H,  BOTIN6, RIGHTIN7H,  BOTIN6H}};
static TECH_SERPENT mocmossub_tpa_l[] = {			/* graphical layers */
	{{LSACT,    1, 4, FILLEDRECT, BOX,    mocmossub_in6box},  K4, K4,  0},
	{{LPOLY1,   0, 4, FILLEDRECT, BOX,    mocmossub_trnpbox}, K1, K1, H2},
	{{LWELLN,  -1, 4, FILLEDRECT, BOX,    mocmossub_fullbox}, K10,K10,K6},
	{{LSELECTP,-1, 4, FILLEDRECT, BOX,    mocmossub_in4box},  K6, K6, K2}};
static TECH_SERPENT mocmossub_tpaE_l[] = {			/* electric layers */
	{{LSACT,    1, 4, FILLEDRECT, BOX,    mocmossub_trna2box}, K4,-K1,  0},
	{{LSACT,    3, 4, FILLEDRECT, BOX,    mocmossub_trna3box},-K1, K4,  0},
	{{LPOLY1,   0, 4, FILLEDRECT, BOX,    mocmossub_trnpbox},  K1, K1, H2},
	{{LWELLN,  -1, 4, FILLEDRECT, BOX,    mocmossub_fullbox},  K10,K10,K6},
	{{LSELECTP,-1, 4, FILLEDRECT, BOX,    mocmossub_in4box},   K6, K6, K2}};
static TECH_NODES mocmossub_tpa = {
	"P-Transistor",NSTRANS,NONODEPROTO,				/* name */
	K15,K20,										/* size */
	4,mocmossub_tpa_p,								/* ports */
	4,(TECH_POLYGON *)0,							/* layers */
	NODESHRINK|(NPTRAPMOS<<NFUNCTIONSH)|HOLDSTRACE,	/* userbits */
	SERPTRANS,5,H1,H2,K2,K1,K2,mocmossub_tpa_l,mocmossub_tpaE_l};/* characteristics */

/* N-transistor (or P) */
static TECH_PORTS mocmossub_tna_p[] = {				/* ports */
	{mocmossub_pc_p1, "n-trans-poly-left", NOPORTPROTO, (180<<PORTANGLESH)|
		(90<<PORTARANGESH),                LEFTIN4,  BOTIN10, LEFTIN4,   TOPIN10},
	{mocmossub_pc_na, "n-trans-diff-top",  NOPORTPROTO, (90<<PORTANGLESH)|
		(90<<PORTARANGESH)|(1<<PORTNETSH), LEFTIN7H,  TOPIN6H, RIGHTIN7H,  TOPIN6},
	{mocmossub_pc_p1, "n-trans-poly-right",NOPORTPROTO, (0<<PORTANGLESH)|
		(90<<PORTARANGESH),                RIGHTIN4, BOTIN10, RIGHTIN4,  TOPIN10},
	{mocmossub_pc_na, "n-trans-diff-bot",  NOPORTPROTO, (270<<PORTANGLESH)|
		(90<<PORTARANGESH)|(2<<PORTNETSH), LEFTIN7H,  BOTIN6, RIGHTIN7H,  BOTIN6H}};
static TECH_SERPENT mocmossub_tna_l[] = {			/* graphical layers */
	{{LDACT,     1, 4, FILLEDRECT, BOX,    mocmossub_in6box},  K4, K4,  0},
	{{LPOLY1,    0, 4, FILLEDRECT, BOX,    mocmossub_trnpbox}, K1, K1, H2},
	{{LWELLP,   -1, 4, FILLEDRECT, BOX,    mocmossub_fullbox}, K10,K10,K6},
	{{LSELECTN, -1, 4, FILLEDRECT, BOX,    mocmossub_in4box},  K6, K6, K2}};
static TECH_SERPENT mocmossub_tnaE_l[] = {			/* electric layers */
	{{LDACT,     1, 4, FILLEDRECT, BOX,    mocmossub_trna2box}, K4,-K1,  0},
	{{LDACT,     3, 4, FILLEDRECT, BOX,    mocmossub_trna3box},-K1, K4,  0},
	{{LPOLY1,    0, 4, FILLEDRECT, BOX,    mocmossub_trnpbox},  K1, K1, H2},
	{{LWELLP,   -1, 4, FILLEDRECT, BOX,    mocmossub_fullbox},  K10,K10,K6},
	{{LSELECTN, -1, 4, FILLEDRECT, BOX,    mocmossub_in4box},   K6, K6, K2}};
static TECH_NODES mocmossub_tna = {
	"N-Transistor",NDTRANS,NONODEPROTO,				/* name */
	K15,K20,										/* size */
	4,mocmossub_tna_p,								/* ports */
	4,(TECH_POLYGON *)0,							/* layers */
	NODESHRINK|(NPTRANMOS<<NFUNCTIONSH)|HOLDSTRACE,	/* userbits */
	SERPTRANS,5,H1,H2,K2,K1,K2,mocmossub_tna_l,mocmossub_tnaE_l};/* characteristics */

/* metal-1-metal-2-contact */
static TECH_PORTS mocmossub_m1m2_p[] = {				/* ports */
	{mocmossub_pc_m1m2, "metal-1-metal-2", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_m1m2_l[] = {			/* layers */
	{LMETAL1, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LMETAL2, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LVIA1,   0, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_m1m2 = {
	"Metal-1-Metal-2-Con",NVIA1,NONODEPROTO,		/* name */
	K4,K4,											/* size */
	1,mocmossub_m1m2_p,								/* ports */
	3,mocmossub_m1m2_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,K1,K4,0,0,0,0};					/* characteristics */

/* metal-2-metal-3-contact */
static TECH_PORTS mocmossub_m2m3_p[] = {			/* ports */
	{mocmossub_pc_m2m3, "metal-2-metal-3", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN2H, BOTIN2H, RIGHTIN2H, TOPIN2H}};
static TECH_POLYGON mocmossub_m2m3_l[] = {			/* layers */
	{LMETAL2, 0, 4, FILLEDRECT, BOX, mocmossub_in1box},
	{LMETAL3, 0, 4, FILLEDRECT, BOX, mocmossub_in1box},
	{LVIA2,   0, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_m2m3 = {
	"Metal-2-Metal-3-Con",NVIA2,NONODEPROTO,		/* name */
	K6,K6,											/* size */
	1,mocmossub_m2m3_p,								/* ports */
	3,mocmossub_m2m3_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,K1,K4,0,0,0,0};					/* characteristics */

/* metal-3-metal-4-contact */
static TECH_PORTS mocmossub_m3m4_p[] = {			/* ports */
	{mocmossub_pc_m3m4, "metal-3-metal-4", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_m3m4_l[] = {			/* layers */
	{LMETAL3, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LMETAL4, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LVIA3,   0, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_m3m4 = {
	"Metal-3-Metal-4-Con",NVIA3,NONODEPROTO,		/* name */
	K4,K4,											/* size */
	1,mocmossub_m3m4_p,								/* ports */
	3,mocmossub_m3m4_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,K1,K4,0,0,0,0};					/* characteristics */

/* metal-4-metal-5-contact */
static TECH_PORTS mocmossub_m4m5_p[] = {			/* ports */
	{mocmossub_pc_m4m5, "metal-4-metal-5", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_m4m5_l[] = {			/* layers */
	{LMETAL4, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LMETAL5, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LVIA4,   0, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_m4m5 = {
	"Metal-4-Metal-5-Con",NVIA4,NONODEPROTO,		/* name */
	K4,K4,											/* size */
	1,mocmossub_m4m5_p,								/* ports */
	3,mocmossub_m4m5_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH)|NNOTUSED,				/* userbits */
	MULTICUT,K2,K2,K1,K4,0,0,0,0};					/* characteristics */

/* metal-5-metal-6-contact */
static TECH_PORTS mocmossub_m5m6_p[] = {			/* ports */
	{mocmossub_pc_m5m6, "metal-5-metal-6", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_m5m6_l[] = {			/* layers */
	{LMETAL5, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LMETAL6, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox},
	{LVIA5,   0, 4, FILLEDRECT, BOX, mocmossub_cutbox}};
static TECH_NODES mocmossub_m5m6 = {
	"Metal-5-Metal-6-Con",NVIA5,NONODEPROTO,		/* name */
	K4,K4,											/* size */
	1,mocmossub_m5m6_p,								/* ports */
	3,mocmossub_m5m6_l,								/* layers */
	(NPCONTACT<<NFUNCTIONSH)|NNOTUSED,				/* userbits */
	MULTICUT,K2,K2,K1,K4,0,0,0,0};					/* characteristics */

/* Metal-1-Well Contact */
static TECH_PORTS mocmossub_psub_p[] = {			/* ports */
	{mocmossub_pc_m1a, "metal-1-well", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN6H, BOTIN6H, RIGHTIN6H, TOPIN6H}};
static TECH_POLYGON mocmossub_psub_l[] = {			/* layers */
	{LMETAL1,   0, 4, FILLEDRECT, MINBOX, mocmossub_min5box},
	{LSACTWELL, 0, 4, FILLEDRECT, BOX,    mocmossub_in4box},
	{LWELLP,   -1, 4, FILLEDRECT, BOX,    mocmossub_fullbox},
	{LSELECTP, -1, 4, FILLEDRECT, BOX,    mocmossub_in2box},
	{LACTCUT,   0, 4, FILLEDRECT, BOX,    mocmossub_cutbox}};
static TECH_NODES mocmossub_psub = {
	"Metal-1-Well-Con",NWBUT,NONODEPROTO,			/* name */
	K14,K14,										/* size */
	1,mocmossub_psub_p,								/* ports */
	5,mocmossub_psub_l,								/* layers */
	(NPWELL<<NFUNCTIONSH),							/* userbits */
	MULTICUT,K2,K2,K2,K4,0,0,0,0};					/* characteristics */

/* Metal-1-Substrate Contact */
static TECH_PORTS mocmossub_nsub_p[] = {			/* ports */
	{mocmossub_pc_m1a, "metal-1-substrate", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN6H, BOTIN6H, RIGHTIN6H, TOPIN6H}};
static TECH_POLYGON mocmossub_nsub_l[] = {			/* layers */
	{LMETAL1,   0, 4, FILLEDRECT, MINBOX, mocmossub_min5box},
	{LDACT,     0, 4, FILLEDRECT, BOX,    mocmossub_in4box},
	{LWELLN,   -1, 4, FILLEDRECT, BOX,    mocmossub_fullbox},
	{LSELECTN, -1, 4, FILLEDRECT, BOX,    mocmossub_in2box},
	{LACTCUT,   0, 4, FILLEDRECT, BOX,    mocmossub_cutbox}};
static TECH_NODES mocmossub_nsub = {
	"Metal-1-Substrate-Con",NSBUT,NONODEPROTO,		/* name */
	K14,K14,										/* size */
	1,mocmossub_nsub_p,								/* ports */
	5,mocmossub_nsub_l,								/* layers */
	(NPSUBSTRATE<<NFUNCTIONSH),						/* userbits */
	MULTICUT,K2,K2,K2,K4,0,0,0,0};					/* characteristics */

/* Metal-1-Node */
static TECH_PORTS mocmossub_m1_p[] = {				/* ports */
	{mocmossub_pc_m1, "metal-1", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_m1_l[] = {			/* layers */
	{LMETAL1, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_m1 = {
	"Metal-1-Node",NMETAL1N,NONODEPROTO,			/* name */
	K3,K3,											/* size */
	1,mocmossub_m1_p,								/* ports */
	1,mocmossub_m1_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Metal-2-Node */
static TECH_PORTS mocmossub_m2_p[] = {				/* ports */
	{mocmossub_pc_m2, "metal-2", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_m2_l[] = {			/* layers */
	{LMETAL2, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_m2 = {
	"Metal-2-Node",NMETAL2N,NONODEPROTO,			/* name */
	K3,K3,											/* size */
	1,mocmossub_m2_p,								/* ports */
	1,mocmossub_m2_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Metal-3-Node */
static TECH_PORTS mocmossub_m3_p[] = {				/* ports */
	{mocmossub_pc_m3, "metal-3", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_m3_l[] = {			/* layers */
	{LMETAL3, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_m3 = {
	"Metal-3-Node",NMETAL3N,NONODEPROTO,			/* name */
	K3,K3,											/* size */
	1,mocmossub_m3_p,								/* ports */
	1,mocmossub_m3_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Metal-4-Node */
static TECH_PORTS mocmossub_m4_p[] = {				/* ports */
	{mocmossub_pc_m4, "metal-4", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_m4_l[] = {			/* layers */
	{LMETAL4, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_m4 = {
	"Metal-4-Node",NMETAL4N,NONODEPROTO,			/* name */
	K3,K3,											/* size */
	1,mocmossub_m4_p,								/* ports */
	1,mocmossub_m4_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Metal-5-Node */
static TECH_PORTS mocmossub_m5_p[] = {				/* ports */
	{mocmossub_pc_m5, "metal-5", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_m5_l[] = {			/* layers */
	{LMETAL5, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_m5 = {
	"Metal-5-Node",NMETAL5N,NONODEPROTO,			/* name */
	K3,K3,											/* size */
	1,mocmossub_m5_p,								/* ports */
	1,mocmossub_m5_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE|NNOTUSED,		/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Metal-6-Node */
static TECH_PORTS mocmossub_m6_p[] = {				/* ports */
	{mocmossub_pc_m6, "metal-6", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_m6_l[] = {			/* layers */
	{LMETAL6, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_m6 = {
	"Metal-6-Node",NMETAL6N,NONODEPROTO,			/* name */
	K3,K3,											/* size */
	1,mocmossub_m6_p,								/* ports */
	1,mocmossub_m6_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE|NNOTUSED,		/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Polysilicon-1-Node */
static TECH_PORTS mocmossub_p1_p[] = {				/* ports */
	{mocmossub_pc_p1, "polysilicon-1", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1, BOTIN1, RIGHTIN1, TOPIN1}};
static TECH_POLYGON mocmossub_p1_l[] = {			/* layers */
	{LPOLY1, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_p1 = {
	"Polysilicon-1-Node",NPOLY1N,NONODEPROTO,		/* name */
	K2,K2,											/* size */
	1,mocmossub_p1_p,								/* ports */
	1,mocmossub_p1_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Polysilicon-2-Node */
static TECH_PORTS mocmossub_p2_p[] = {				/* ports */
	{mocmossub_pc_p2, "polysilicon-2", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_p2_l[] = {			/* layers */
	{LPOLY2, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_p2 = {
	"Polysilicon-2-Node",NPOLY2N,NONODEPROTO,		/* name */
	K3,K3,											/* size */
	1,mocmossub_p2_p,								/* ports */
	1,mocmossub_p2_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Active-Node */
static TECH_PORTS mocmossub_a_p[] = {				/* ports */
	{mocmossub_pc_a, "active", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_a_l[] = {				/* layers */
	{LSACT, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_a = {
	"Active-Node",NACTIVEN,NONODEPROTO,				/* name */
	K3,K3,											/* size */
	1,mocmossub_a_p,								/* ports */
	1,mocmossub_a_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* N-Active-Node (or P) */
static TECH_PORTS mocmossub_da_p[] = {				/* ports */
	{mocmossub_pc_a, "active", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN1H, BOTIN1H, RIGHTIN1H, TOPIN1H}};
static TECH_POLYGON mocmossub_da_l[] = {			/* layers */
	{LDACT, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_da = {
	"N-Active-Node",NDACTIVEN,NONODEPROTO,			/* name */
	K3,K3,											/* size */
	1,mocmossub_da_p,								/* ports */
	1,mocmossub_da_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* P-Select-Node */
static TECH_PORTS mocmossub_sp_p[] = {				/* ports */
	{mocmossub_pc_null, "select", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_sp_l[] = {			/* layers */
	{LSELECTP, -1, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_sp = {
	"P-Select-Node",NSELECTPN,NONODEPROTO,			/* name */
	K4,K4,											/* size */
	1,mocmossub_sp_p,								/* ports */
	1,mocmossub_sp_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* N-Select-Node */
static TECH_PORTS mocmossub_sn_p[] = {				/* ports */
	{mocmossub_pc_null, "select", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_sn_l[] = {			/* layers */
	{LSELECTN,  -1, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_sn = {
	"N-Select-Node",NSELECTNN,NONODEPROTO,			/* name */
	K4,K4,											/* size */
	1,mocmossub_sn_p,								/* ports */
	1,mocmossub_sn_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* PolyCut-Node */
static TECH_PORTS mocmossub_gc_p[] = {				/* ports */
	{mocmossub_pc_null, "polycut", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_gc_l[] = {			/* layers */
	{LPOLYCUT, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_gc = {
	"Poly-Cut-Node",NPCUTN,NONODEPROTO,				/* name */
	K2,K2,											/* size */
	1,mocmossub_gc_p,								/* ports */
	1,mocmossub_gc_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* ActiveCut-Node */
static TECH_PORTS mocmossub_ac_p[] = {				/* ports */
	{mocmossub_pc_null, "activecut", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_ac_l[] = {			/* layers */
	{LACTCUT, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_ac = {
	"Active-Cut-Node",NACUTN,NONODEPROTO,			/* name */
	K2,K2,											/* size */
	1,mocmossub_ac_p,								/* ports */
	1,mocmossub_ac_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Via-1-Node */
static TECH_PORTS mocmossub_v1_p[] = {				/* ports */
	{mocmossub_pc_null, "via-1", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_v1_l[] = {			/* layers */
	{LVIA1, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_v1 = {
	"Via-1-Node",NVIA1N,NONODEPROTO,				/* name */
	K2,K2,											/* size */
	1,mocmossub_v1_p,								/* ports */
	1,mocmossub_v1_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Via-2-Node */
static TECH_PORTS mocmossub_v2_p[] = {				/* ports */
	{mocmossub_pc_null, "via-2", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_v2_l[] = {			/* layers */
	{LVIA2, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_v2 = {
	"Via-2-Node",NVIA2N,NONODEPROTO,				/* name */
	K2,K2,											/* size */
	1,mocmossub_v2_p,								/* ports */
	1,mocmossub_v2_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Via-3-Node */
static TECH_PORTS mocmossub_v3_p[] = {				/* ports */
	{mocmossub_pc_null, "via-3", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_v3_l[] = {			/* layers */
	{LVIA3, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_v3 = {
	"Via-3-Node",NVIA3N,NONODEPROTO,				/* name */
	K2,K2,											/* size */
	1,mocmossub_v3_p,								/* ports */
	1,mocmossub_v3_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Via-4-Node */
static TECH_PORTS mocmossub_v4_p[] = {				/* ports */
	{mocmossub_pc_null, "via-4", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_v4_l[] = {			/* layers */
	{LVIA4, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_v4 = {
	"Via-4-Node",NVIA4N,NONODEPROTO,				/* name */
	K2,K2,											/* size */
	1,mocmossub_v4_p,								/* ports */
	1,mocmossub_v4_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE|NNOTUSED,		/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Via-5-Node */
static TECH_PORTS mocmossub_v5_p[] = {				/* ports */
	{mocmossub_pc_null, "via-5", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_v5_l[] = {			/* layers */
	{LVIA5, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_v5 = {
	"Via-5-Node",NVIA5N,NONODEPROTO,				/* name */
	K2,K2,											/* size */
	1,mocmossub_v5_p,								/* ports */
	1,mocmossub_v5_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE|NNOTUSED,		/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* P-Well-Node */
static TECH_PORTS mocmossub_wp_p[] = {				/* ports */
	{mocmossub_pc_pa, "well", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN3, BOTIN3, RIGHTIN3, TOPIN3}};
static TECH_POLYGON mocmossub_wp_l[] = {			/* layers */
	{LWELLP, -1, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_wp = {
	"P-Well-Node",NWELLPN,NONODEPROTO,				/* name */
	K12,K12,										/* size */
	1,mocmossub_wp_p,								/* ports */
	1,mocmossub_wp_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* N-Well-Node */
static TECH_PORTS mocmossub_wn_p[] = {				/* ports */
	{mocmossub_pc_pa, "well", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTIN3, BOTIN3, RIGHTIN3, TOPIN3}};
static TECH_POLYGON mocmossub_wn_l[] = {			/* layers */
	{LWELLN,  -1, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_wn = {
	"N-Well-Node",NWELLNN,NONODEPROTO,				/* name */
	K12,K12,										/* size */
	1,mocmossub_wn_p,								/* ports */
	1,mocmossub_wn_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Passivation-node */
static TECH_PORTS mocmossub_o_p[] = {				/* ports */
	{mocmossub_pc_null, "passivation", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_o_l[] = {				/* layers */
	{LPASS, 0, 4, FILLEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_o = {
	"Passivation-Node",NPASSN,NONODEPROTO,			/* name */
	K8,K8,											/* size */
	1,mocmossub_o_p,								/* ports */
	1,mocmossub_o_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Pad-Frame-node */
static TECH_PORTS mocmossub_pf_p[] = {				/* ports */
	{mocmossub_pc_null, "pad-frame", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_pf_l[] = {			/* layers */
	{LFRAME, 0, 4, CLOSEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pf = {
	"Pad-Frame-Node",NPADFRN,NONODEPROTO,			/* name */
	K8,K8,											/* size */
	1,mocmossub_pf_p,								/* ports */
	1,mocmossub_pf_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* Polysilicon-Capacitor-node */
static TECH_PORTS mocmossub_pc_p[] = {				/* ports */
	{mocmossub_pc_null, "poly-cap", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_pc_l[] = {			/* layers */
	{LPOLYCAP, 0, 4, CLOSEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_pc = {
	"Poly-Cap-Node",NPOLYCAPN,NONODEPROTO,			/* name */
	K8,K8,											/* size */
	1,mocmossub_pc_p,								/* ports */
	1,mocmossub_pc_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

/* P-Active-Well-node */
static TECH_PORTS mocmossub_paw_p[] = {				/* ports */
	{mocmossub_pc_null, "p-active-well", NOPORTPROTO, (180<<PORTARANGESH),
		LEFTEDGE, BOTEDGE, RIGHTEDGE, TOPEDGE}};
static TECH_POLYGON mocmossub_paw_l[] = {			/* layers */
	{LSACTWELL, 0, 4, CLOSEDRECT, BOX, mocmossub_fullbox}};
static TECH_NODES mocmossub_paw = {
	"P-Active-Well-Node",NPACTWELLN,NONODEPROTO,	/* name */
	K8,K8,											/* size */
	1,mocmossub_paw_p,								/* ports */
	1,mocmossub_paw_l,								/* layers */
	(NPNODE<<NFUNCTIONSH)|HOLDSTRACE,				/* userbits */
	POLYGONAL,0,0,0,0,0,0,0,0};						/* characteristics */

TECH_NODES *mocmossub_nodeprotos[NODEPROTOCOUNT+1] = {
	&mocmossub_pm1, &mocmossub_pm2, &mocmossub_pm3,	/* metal 1/2/3 pin */
	&mocmossub_pm4, &mocmossub_pm5, &mocmossub_pm6,	/* metal 4/5/6 pin */
	&mocmossub_pp1, &mocmossub_pp2,					/* polysilicon 1/2 pin */
	&mocmossub_psa, &mocmossub_pda,					/* P/N active pin */
	&mocmossub_pa,									/* active pin */
	&mocmossub_mpa, &mocmossub_mna,					/* metal 1 to P/N active contact */
	&mocmossub_mp1, &mocmossub_mp2,					/* metal 1 to polysilicon 1/2 contact */
	&mocmossub_mp12,								/* poly capacitor */
	&mocmossub_tpa, &mocmossub_tna,					/* P/N transistor */
	&mocmossub_m1m2, &mocmossub_m2m3, &mocmossub_m3m4,	/* via 1/2/3 */
	&mocmossub_m4m5, &mocmossub_m5m6,				/* via 4/5 */
	&mocmossub_psub, &mocmossub_nsub,				/* well / substrate contact */
	&mocmossub_m1, &mocmossub_m2, &mocmossub_m3,	/* metal 1/2/3 node */
	&mocmossub_m4, &mocmossub_m5, &mocmossub_m6,	/* metal 4/5/6 node */
	&mocmossub_p1, &mocmossub_p2,					/* polysilicon 1/2 node */
	&mocmossub_a, &mocmossub_da,					/* active N-Active node */
	&mocmossub_sp, &mocmossub_sn,					/* P/N select node */
	&mocmossub_gc, &mocmossub_ac,					/* poly cut / active cut */
	&mocmossub_v1, &mocmossub_v2, &mocmossub_v3,	/* via 1/2/3 node */
	&mocmossub_v4, &mocmossub_v5,					/* via 4/5 node */
	&mocmossub_wp, &mocmossub_wn,					/* P/N well node */
	&mocmossub_o,									/* overglass node */
	&mocmossub_pf,									/* pad frame node */
	&mocmossub_pc,									/* poly-cap node */
	&mocmossub_paw, ((TECH_NODES *)-1)};			/* p-active-well node */

/* this table must correspond with the above table */
static INTBIG mocmossub_node_widoff[NODEPROTOCOUNT*4] = {
	0,0,0,0, 0,0,0,0, 0,0,0,0,						/* metal 1/2/3 pin */
	0,0,0,0, 0,0,0,0, 0,0,0,0,						/* metal 4/5/6 pin */
	0,0,0,0, 0,0,0,0,								/* polysilicon 1/2 pin */
	K6,K6,K6,K6, K6,K6,K6,K6,						/* P/N active pin */
	0,0,0,0,										/* active pin */
	K6,K6,K6,K6, K6,K6,K6,K6,						/* metal 1 to P/N active contact */
	0,0,0,0, 0,0,0,0,								/* metal 1 to polysilicon 1/2 contact */
	0,0,0,0,										/* poly capacitor */
	K6,K6,K9,K9, K6,K6,K9,K9,						/* P/N transistor */
	0,0,0,0, K1,K1,K1,K1, 0,0,0,0,					/* via 1/2/3 */
	0,0,0,0, 0,0,0,0,								/* via 4/5 */
	K4,K4,K4,K4, K4,K4,K4,K4,						/* well / substrate contact */
	0,0,0,0, 0,0,0,0, 0,0,0,0,						/* metal 1/2/3 node */
	0,0,0,0, 0,0,0,0, 0,0,0,0,						/* metal 4/5/6 node */
	0,0,0,0, 0,0,0,0,								/* polysilicon 1/2 node */
	0,0,0,0, 0,0,0,0,								/* active N-Active node */
	0,0,0,0, 0,0,0,0,								/* P/N select node */
	0,0,0,0, 0,0,0,0,								/* poly cut / active cut */
	0,0,0,0, 0,0,0,0, 0,0,0,0,						/* via 1/2/3 node */
	0,0,0,0, 0,0,0,0,								/* via 4/5 node */
	0,0,0,0, 0,0,0,0,								/* P/N well node */
	0,0,0,0,										/* overglass node */
	0,0,0,0,										/* pad frame node */
	0,0,0,0,										/* poly-cap node */
	0,0,0,0};										/* p-active-well node */

/******************** SIMULATION VARIABLES ********************/

/* for SPICE simulation */
#define	MOCMOSSUB_MIN_RESIST	50.0f	/* minimum resistance consider */
#define	MOCMOSSUB_MIN_CAPAC	     0.04f	/* minimum capacitance consider */
float mocmossub_sim_spice_resistance[MAXLAYERS] = {  /* per square micron */
	0.06f, 0.06f, 0.06f,				/* metal 1/2/3 */
	0.03f, 0.03f, 0.03f,				/* metal 4/5/6 */
	2.5f, 50.0f,						/* poly 1/2 */
	2.5f, 3.0f,							/* P/N active */
	0.0, 0.0,							/* P/N select */
	0.0, 0.0,							/* P/N well */
	2.2f, 2.5f,							/* poly/act cut */
	1.0f, 0.9f, 0.8f, 0.8f, 0.8f,		/* via 1/2/3/4/5 */
	0.0,								/* overglass */
	0.0,								/* transistor */
	0.0,								/* poly cap */
	0.0,								/* P active well */
	0.0, 0.0, 0.0, 0.0, 0.0, 0.0,		/* pseudo metal 1/2/3/4/5/6 */
	0.0, 0.0,							/* pseudo poly 1/2 */
	0.0, 0.0,							/* pseudo P/N active */
	0.0, 0.0,							/* pseudo P/N select */
	0.0, 0.0,							/* pseudo P/N well */
	0.0};								/* pad frame */
float mocmossub_sim_spice_capacitance[MAXLAYERS] = { /* per square micron */
	0.07f, 0.04f, 0.04f,				/* metal 1/2/3 */
	0.04f, 0.04f, 0.04f,				/* metal 4/5/6 */
	0.09f, 1.0f,						/* poly 1/2 */
	0.9f, 0.9f,							/* P/N active */
	0.0, 0.0,							/* P/N select */
	0.0, 0.0,							/* P/N well */
	0.0, 0.0,							/* poly/act cut */
	0.0, 0.0, 0.0, 0.0, 0.0,			/* via 1/2/3/4/5 */
	0.0,								/* overglass */
	0.0,								/* transistor */
	0.0,								/* poly cap */
	0.0,								/* P active well */
	0.0, 0.0, 0.0, 0.0, 0.0, 0.0,		/* pseudo metal 1/2/3/4/5/6 */
	0.0, 0.0,							/* pseudo poly 1/2 */
	0.0, 0.0,							/* pseudo P/N active */
	0.0, 0.0,							/* pseudo P/N select */
	0.0, 0.0,							/* pseudo P/N well */
	0.0};								/* pad frame */
char *mocmossub_sim_spice_header_level1[] = {
	"*CMOS/BULK-NWELL (PRELIMINARY PARAMETERS)",
	".OPTIONS NOMOD DEFL=3UM DEFW=3UM DEFAD=70P DEFAS=70P LIMPTS=1000",
	"+ITL4=1000 ITL5=0 RELTOL=0.01 ABSTOL=500PA VNTOL=500UV LVLTIM=2",
	"+LVLCOD=1",
	".MODEL N NMOS LEVEL=1",
	"+KP=60E-6 VTO=0.7 GAMMA=0.3 LAMBDA=0.05 PHI=0.6",
	"+LD=0.4E-6 TOX=40E-9 CGSO=2.0E-10 CGDO=2.0E-10 CJ=.2MF/M^2",
	".MODEL P PMOS LEVEL=1",
	"+KP=20E-6 VTO=0.7 GAMMA=0.4 LAMBDA=0.05 PHI=0.6",
	"+LD=0.6E-6 TOX=40E-9 CGSO=3.0E-10 CGDO=3.0E-10 CJ=.2MF/M^2",
	".MODEL DIFFCAP D CJO=.2MF/M^2",
	NOSTRING};
char *mocmossub_sim_spice_header_level2[] = {
	"* MOSIS 3u CMOS PARAMS",
	".OPTIONS NOMOD DEFL=2UM DEFW=6UM DEFAD=100P DEFAS=100P",
	"+LIMPTS=1000 ITL4=1000 ITL5=0 ABSTOL=500PA VNTOL=500UV",
	"* Note that ITL5=0 sets ITL5 to infinity",
	".MODEL N NMOS LEVEL=2 LD=0.3943U TOX=502E-10",
	"+NSUB=1.22416E+16 VTO=0.756 KP=4.224E-05 GAMMA=0.9241",
	"+PHI=0.6 UO=623.661 UEXP=8.328627E-02 UCRIT=54015.0",
	"+DELTA=5.218409E-03 VMAX=50072.2 XJ=0.4U LAMBDA=2.975321E-02",
	"+NFS=4.909947E+12 NEFF=1.001E-02 NSS=0.0 TPG=1.0",
	"+RSH=20.37 CGDO=3.1E-10 CGSO=3.1E-10",
	"+CJ=3.205E-04 MJ=0.4579 CJSW=4.62E-10 MJSW=0.2955 PB=0.7",
	".MODEL P PMOS LEVEL=2 LD=0.2875U TOX=502E-10",
	"+NSUB=1.715148E+15 VTO=-0.7045 KP=1.686E-05 GAMMA=0.3459",
	"+PHI=0.6 UO=248.933 UEXP=1.02652 UCRIT=182055.0",
	"+DELTA=1.0E-06 VMAX=100000.0 XJ=0.4U LAMBDA=1.25919E-02",
	"+NFS=1.0E+12 NEFF=1.001E-02 NSS=0.0 TPG=-1.0",
	"+RSH=79.10 CGDO=2.89E-10 CGSO=2.89E-10",
	"+CJ=1.319E-04 MJ=0.4125 CJSW=3.421E-10 MJSW=0.198 PB=0.66",
	".TEMP 25.0",
	NOSTRING};


/******************** VARIABLE AGGREGATION ********************/

TECH_VARIABLES mocmossub_variables[] =
{
	/* set general information about the technology */
	{"TECH_layer_names", (char *)mocmossub_layer_names, 0.0,
		VSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},
	{"TECH_layer_function", (char *)mocmossub_layer_function, 0.0,
		VINTEGER|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},
	{"TECH_arc_width_offset", (char *)mocmossub_arc_widoff, 0.0,
		VFRACT|VDONTSAVE|VISARRAY|(ARCPROTOCOUNT<<VLENGTHSH)},
	{"TECH_node_width_offset", (char *)mocmossub_node_widoff, 0.0,
		VFRACT|VDONTSAVE|VISARRAY|((NODEPROTOCOUNT*4)<<VLENGTHSH)},
	{"TECH_layer_3dthickness", (char *)mocmossub_3dthick_layers, 0.0,
		VINTEGER|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},
	{"TECH_layer_3dheight", (char *)mocmossub_3dheight_layers, 0.0,
		VINTEGER|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},

	/* set information for the USER aid */
	{"USER_color_map", (char *)mocmossub_colmap, 0.0,
		VCHAR|VDONTSAVE|VISARRAY|((sizeof mocmossub_colmap)<<VLENGTHSH)},
	{"USER_layer_letters", (char *)mocmossub_layer_letters, 0.0,
		VSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},

	/* set information for the I/O aid */
	{"IO_cif_layer_names", (char *)mocmossub_cif_layers, 0.0,
		VSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},
	{"IO_gds_layer_numbers", (char *)mocmossub_gds_layers, 0.0,
		VINTEGER|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},
	{"IO_skill_layer_names", (char *)mocmossub_skill_layers, 0.0,
		VSTRING|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},

	/* set information for the DRC aid */
	{"DRC_min_unconnected_distances", (char *)mocmossub_unconnectedtable, 0.0,
		VFRACT|VDONTSAVE|VISARRAY|
			(((sizeof mocmossub_unconnectedtable)/SIZEOFINTBIG)<<VLENGTHSH)},
	{"DRC_min_connected_distances", (char *)mocmossub_connectedtable, 0.0,
		VFRACT|VDONTSAVE|VISARRAY|
			(((sizeof mocmossub_connectedtable)/SIZEOFINTBIG)<<VLENGTHSH)},
	{"DRC_min_width", (char *)mocmossub_minimum_width, 0.0,
		VFRACT|VDONTSAVE|VISARRAY|(MAXLAYERS<<VLENGTHSH)},

	/* set information for the SIM aid (SPICE) */
	{"SIM_spice_min_resistance", 0, MOCMOSSUB_MIN_RESIST, VFLOAT|VDONTSAVE},
	{"SIM_spice_min_capacitance", 0, MOCMOSSUB_MIN_CAPAC, VFLOAT|VDONTSAVE},
	{"SIM_spice_resistance", (char *)mocmossub_sim_spice_resistance, 0.0,
		VFLOAT|VISARRAY|(MAXLAYERS<<VLENGTHSH)|VDONTSAVE},
	{"SIM_spice_capacitance", (char *)mocmossub_sim_spice_capacitance, 0.0,
		VFLOAT|VISARRAY|(MAXLAYERS<<VLENGTHSH)|VDONTSAVE},
	{"SIM_spice_header_level1", (char *)mocmossub_sim_spice_header_level1, 0.0,
		VSTRING|VDONTSAVE|VISARRAY},
	{"SIM_spice_header_level2", (char *)mocmossub_sim_spice_header_level2, 0.0,
		VSTRING|VDONTSAVE|VISARRAY},
	{NULL, NULL, 0.0, 0}
};

/******************** ROUTINES ********************/

INTSML mocmossub_initprocess(TECHNOLOGY *tech, INTSML pass)
{
	/* initialize the technology variable */
	if (pass == 0) mocmossub_tech = tech;
	mocmossub_state = MOCMOSSUB4METAL;
	return(0);
}

INTBIG mocmossub_request(char *command, va_list ap)
{
	if (namesame(command, "get-state") == 0)
	{
		return(mocmossub_state);
	}
	if (namesame(command, "set-state") == 0)
	{
		mocmossub_setstate(va_arg(ap, INTBIG));
		return(0);
	}
	if (namesame(command, "switch-n-and-p") == 0)
	{
		mocmossub_switchnp();
	}
	return(0);
}

void mocmossub_setmode(INTSML count, char *par[])
{
	REGISTER INTSML l;
	REGISTER char *pp;

	l = strlen(pp = par[0]);

	if (namesamen(pp, "full-graphics", l) == 0)
	{
		mocmossub_setstate(mocmossub_state & ~MOCMOSSUBSTICKFIGURE);
		ttyputverbose("MOSIS CMOS Submicron technology displays full graphics");
		return;
	}
	if (namesamen(pp, "stick-display", l) == 0)
	{
		mocmossub_setstate(mocmossub_state | MOCMOSSUBSTICKFIGURE);
		ttyputverbose("MOSIS CMOS Submicron technology displays stick figures");
		return;
	}

	if (namesamen(pp, "2-metal-rules", l) == 0)
	{
		mocmossub_setstate((mocmossub_state & ~MOCMOSSUBMETALS) | MOCMOSSUB2METAL);
		ttyputverbose("MOSIS CMOS Submicron technology uses 2-metal rules");
		return;
	}
	if (namesamen(pp, "3-metal-rules", l) == 0)
	{
		mocmossub_setstate((mocmossub_state & ~MOCMOSSUBMETALS) | MOCMOSSUB3METAL);
		ttyputverbose("MOSIS CMOS Submicron technology uses 3-metal rules");
		return;
	}
	if (namesamen(pp, "4-metal-rules", l) == 0)
	{
		mocmossub_setstate((mocmossub_state & ~MOCMOSSUBMETALS) | MOCMOSSUB4METAL);
		ttyputverbose("MOSIS CMOS Submicron technology uses 4-metal rules");
		return;
	}
	if (namesamen(pp, "5-metal-rules", l) == 0)
	{
		mocmossub_setstate((mocmossub_state & ~MOCMOSSUBMETALS) | MOCMOSSUB5METAL);
		ttyputverbose("MOSIS CMOS Submicron technology uses 5-metal rules");
		return;
	}
	if (namesamen(pp, "6-metal-rules", l) == 0)
	{
		mocmossub_setstate((mocmossub_state & ~MOCMOSSUBMETALS) | MOCMOSSUB6METAL);
		ttyputverbose("MOSIS CMOS Submicron technology uses 6-metal rules");
		return;
	}

	if (namesamen(pp, "switch-n-and-p", l) == 0)
	{
		mocmossub_switchnp();
	}

	ttyputerr("Unknown MOSIS CMOS Submicron option: %s", pp);
}

void mocmossub_setupprimswap(INTBIG index1, INTBIG index2, PRIMSWAP *swap)
{
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *pp;

	swap->np[0] = swap->np[1] = NONODEPROTO;
	for(np = mocmossub_tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		if (np->primindex == index1) swap->np[0] = np;
		if (np->primindex == index2) swap->np[1] = np;
	}
	if (swap->np[0] == NONODEPROTO || swap->np[1] == NONODEPROTO) return;
	swap->portcount = 0;
	for(pp = swap->np[0]->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		swap->pp[0][swap->portcount++] = pp;
	swap->portcount = 0;
	for(pp = swap->np[1]->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		swap->pp[1][swap->portcount++] = pp;
}

void mocmossub_switchnp(void)
{
	REGISTER LIBRARY *lib;
	REGISTER NODEPROTO *np;
	REGISTER NODEINST *ni, *rni;
	REGISTER ARCINST *ai;
	REGISTER ARCPROTO *ap, *app, *apn;
	REGISTER PORTPROTO *pp, *rpp;
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;
	REGISTER INTBIG i, j, k;

	/* find the important node and arc prototypes */
	mocmossub_setupprimswap(NSACTP, NDACTP, &mocmossub_primswap[0]);
	mocmossub_setupprimswap(NMETSACTC, NMETDACTC, &mocmossub_primswap[1]);
	mocmossub_setupprimswap(NSTRANS, NDTRANS, &mocmossub_primswap[2]);
	mocmossub_setupprimswap(NWBUT, NSBUT, &mocmossub_primswap[3]);
	mocmossub_setupprimswap(NACTIVEN, NDACTIVEN, &mocmossub_primswap[4]);
	mocmossub_setupprimswap(NSELECTPN, NSELECTNN, &mocmossub_primswap[5]);
	mocmossub_setupprimswap(NWELLPN, NWELLNN, &mocmossub_primswap[6]);
	app = apn = NOARCPROTO;
	for(ap = mocmossub_tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
	{
		if (namesame(ap->protoname, "P-Active") == 0) app = ap;
		if (namesame(ap->protoname, "N-Active") == 0) apn = ap;
	}

	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
	{
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
			{
				if (ni->proto->primindex == 0) continue;
				if (ni->proto->tech != mocmossub_tech) continue;
				for(i=0; i<7; i++)
				{
					for(k=0; k<2; k++)
					{
						if (ni->proto == mocmossub_primswap[i].np[k])
						{
							ni->proto = mocmossub_primswap[i].np[1-k];
							for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
							{
								for(j=0; j<mocmossub_primswap[i].portcount; j++)
								{
									if (pi->proto == mocmossub_primswap[i].pp[k][j])
									{
										pi->proto = mocmossub_primswap[i].pp[1-k][j];
										break;
									}
								}
							}
							for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
							{
								for(j=0; j<mocmossub_primswap[i].portcount; j++)
								{
									if (pe->proto == mocmossub_primswap[i].pp[k][j])
									{
										pe->proto = mocmossub_primswap[i].pp[1-k][j];
										pe->exportproto->subportproto = pe->proto;
										break;
									}
								}
							}
							break;
						}
					}
				}
			}
			for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
			{
				if (ai->proto->tech != mocmossub_tech) continue;
				if (ai->proto == app)
				{
					ai->proto = apn;
				} else if (ai->proto == apn)
				{
					ai->proto = app;
				}
			}
		}
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
			{
				/* find the primitive at the bottom */
				rpp = pp->subportproto;
				rni = pp->subnodeinst;
				while (rni->proto->primindex == 0)
				{
					rni = rpp->subnodeinst;
					rpp = rpp->subportproto;
				}
				pp->connects = rpp->connects;
			}
		}
	}
	for(i=0; i<7; i++)
	{
		ni = mocmossub_primswap[i].np[0]->firstinst;
		mocmossub_primswap[i].np[0]->firstinst = mocmossub_primswap[i].np[1]->firstinst;
		mocmossub_primswap[i].np[1]->firstinst = ni;
	}
}

void mocmossub_setstate(INTBIG newstate)
{
	static char descript[200];

	if (mocmossub_state == newstate) return;
	mocmossub_state = newstate;

	if ((mocmossub_state&MOCMOSSUBSTICKFIGURE) != 0)
	{
		mocmossub_tech->nodepolys = mocmossub_nodepolys;
		mocmossub_tech->shapenodepoly = mocmossub_shapenodepoly;
		mocmossub_tech->nodesizeoffset = mocmossub_nodesizeoffset;
		mocmossub_tech->arcpolys = mocmossub_arcpolys;
		mocmossub_tech->shapearcpoly = mocmossub_shapearcpoly;
		mocmossub_tech->arcwidthoffset = mocmossub_arcwidthoffset;
	} else
	{
		mocmossub_tech->nodepolys = 0;
		mocmossub_tech->shapenodepoly = 0;
		mocmossub_tech->nodesizeoffset = 0;
		mocmossub_tech->arcpolys = 0;
		mocmossub_tech->shapearcpoly = 0;
		mocmossub_tech->arcwidthoffset = 0;
	}

	/* adjust metal rules according to the number of metal layers */
	switch (mocmossub_state&MOCMOSSUBMETALS)
	{
		case MOCMOSSUB2METAL:							/* 2-metal process: */
			/* metal-2 is 4 apart, 3 wide */
			mocmossub_setlayerrules("Metal-2", K4, K3);
			break;

		case MOCMOSSUB3METAL:							/* 3-metal process: */
			/* metal-2 is 3 apart, 3 wide */
			mocmossub_setlayerrules("Metal-2", K3, K3);

			/* metal-3 is 5 wide, overlaps vias by 2 */
			mocmossub_setarcwidth("Metal-3", K5);
			mocmossub_setnodesize("Metal-3-Pin", K5, H2);
			mocmossub_setmetalonvia("Metal-2-Metal-3-Con", LMETAL3, mocmossub_fullbox, 0);
			mocmossub_m2m3.f3 = K2;
			break;

		case MOCMOSSUB4METAL:							/* 4-metal process: */
			/* metal-2 is 3 apart, 3 wide */
			mocmossub_setlayerrules("Metal-2", K3, K3);

			/* metal-3 is 3 wide, overlaps vias by 1 */
			mocmossub_setarcwidth("Metal-3", K3);
			mocmossub_setnodesize("Metal-3-Pin", K3, H1);
			mocmossub_setmetalonvia("Metal-2-Metal-3-Con", LMETAL3, mocmossub_in1box, K1);
			mocmossub_m2m3.f3 = K1;
			break;

		case MOCMOSSUB5METAL:							/* 5-metal process: */
			/* metal-2 is 3 apart, 3 wide */
			mocmossub_setlayerrules("Metal-2", K3, K3);

			/* metal-3 is 3 wide, overlaps vias by 1 */
			mocmossub_setarcwidth("Metal-3", K3);
			mocmossub_setnodesize("Metal-3-Pin", K3, H1);
			mocmossub_setmetalonvia("Metal-2-Metal-3-Con", LMETAL3, mocmossub_in1box, K1);
			mocmossub_m2m3.f3 = K1;

			/* metal-5 is 4 apart, 4 wide */
			mocmossub_setlayerrules("Metal-5", K4, K4);
			mocmossub_setarcwidth("Metal-5", K4);
			mocmossub_setnodesize("Metal-5-Pin", K4, 0);
			break;

		case MOCMOSSUB6METAL:							/* 6-metal process: */
			/* metal-2 is 3 apart, 3 wide */
			mocmossub_setlayerrules("Metal-2", K3, K3);

			/* metal-3 is 3 wide, overlaps vias by 1 */
			mocmossub_setarcwidth("Metal-3", K3);
			mocmossub_setnodesize("Metal-3-Pin", K3, H1);
			mocmossub_setmetalonvia("Metal-2-Metal-3-Con", LMETAL3, mocmossub_in1box, K1);
			mocmossub_m2m3.f3 = K1;

			/* metal-5 is 3 apart, 3 wide */
			mocmossub_setlayerrules("Metal-5", K3, K3);
			mocmossub_setarcwidth("Metal-5", K3);
			mocmossub_setnodesize("Metal-5-Pin", K3, 0);
			break;
	}

	/* disable Metal-3/4/5/6-Pin, Metal-2/3/4/5-Metal-3/4/5/6-Con, Metal-3/4/5/6-Node, Via-2/3/4/5-Node */
	mocmossub_pm3.creation->userbits |= NNOTUSED;
	mocmossub_pm4.creation->userbits |= NNOTUSED;
	mocmossub_pm5.creation->userbits |= NNOTUSED;
	mocmossub_pm6.creation->userbits |= NNOTUSED;
	mocmossub_m2m3.creation->userbits |= NNOTUSED;
	mocmossub_m3m4.creation->userbits |= NNOTUSED;
	mocmossub_m4m5.creation->userbits |= NNOTUSED;
	mocmossub_m5m6.creation->userbits |= NNOTUSED;
	mocmossub_m3.creation->userbits |= NNOTUSED;
	mocmossub_m4.creation->userbits |= NNOTUSED;
	mocmossub_m5.creation->userbits |= NNOTUSED;
	mocmossub_m6.creation->userbits |= NNOTUSED;
	mocmossub_v2.creation->userbits |= NNOTUSED;
	mocmossub_v3.creation->userbits |= NNOTUSED;
	mocmossub_v4.creation->userbits |= NNOTUSED;
	mocmossub_v5.creation->userbits |= NNOTUSED;

	/* enable the desired nodes */
	switch (mocmossub_state&MOCMOSSUBMETALS)
	{
		case MOCMOSSUB6METAL:
			mocmossub_pm6.creation->userbits &= ~NNOTUSED;
			mocmossub_m5m6.creation->userbits &= ~NNOTUSED;
			mocmossub_m6.creation->userbits &= ~NNOTUSED;
			mocmossub_v5.creation->userbits &= ~NNOTUSED;
		case MOCMOSSUB5METAL:
			mocmossub_pm5.creation->userbits &= ~NNOTUSED;
			mocmossub_m4m5.creation->userbits &= ~NNOTUSED;
			mocmossub_m5.creation->userbits &= ~NNOTUSED;
			mocmossub_v4.creation->userbits &= ~NNOTUSED;
		case MOCMOSSUB4METAL:
			mocmossub_pm4.creation->userbits &= ~NNOTUSED;
			mocmossub_m3m4.creation->userbits &= ~NNOTUSED;
			mocmossub_m4.creation->userbits &= ~NNOTUSED;
			mocmossub_v3.creation->userbits &= ~NNOTUSED;
		case MOCMOSSUB3METAL:
			mocmossub_pm3.creation->userbits &= ~NNOTUSED;
			mocmossub_m2m3.creation->userbits &= ~NNOTUSED;
			mocmossub_m3.creation->userbits &= ~NNOTUSED;
			mocmossub_v2.creation->userbits &= ~NNOTUSED;
			break;
	}

	/* now rewrite the description */
	strcpy(descript, "Complementary MOS (from MOSIS, Submicron");
	switch (mocmossub_state&MOCMOSSUBMETALS)
	{
		case MOCMOSSUB2METAL: strcat(descript, ", double metal");   break;
		case MOCMOSSUB3METAL: strcat(descript, ", triple metal");   break;
		case MOCMOSSUB4METAL: strcat(descript, ", quad metal");     break;
		case MOCMOSSUB5METAL: strcat(descript, ", 5-metal");        break;
		case MOCMOSSUB6METAL: strcat(descript, ", 6-metal");        break;
	}
	strcat(descript, ", double poly");
	if ((mocmossub_state&MOCMOSSUBSTICKFIGURE) != 0) strcat(descript, ", stick-figures"); else
		strcat(descript, ", full-graphics");
	strcat(descript, ")");
	mocmossub_tech->techdescript = descript;
}

/*
 * Routine to change the design rules for layer "layername" layers so that
 * the layers remain "spacing" apart and are at least "minwidth" wide.
 */
void mocmossub_setlayerrules(char *layername, INTBIG spacing, INTBIG minwidth)
{
	REGISTER INTBIG layer, rindex;

	for(layer=0; layer<mocmossub_tech->layercount; layer++)
		if (namesame(mocmossub_layer_names[layer], layername) == 0) break;
	if (layer >= mocmossub_tech->layercount) return;

	rindex = (layer+1) * (layer/2) + (layer&1) * ((layer+1)/2);
	rindex = layer + mocmossub_tech->layercount * layer - rindex;
	nextvarchangequiet();
	setind((INTBIG)mocmossub_tech, VTECHNOLOGY, "DRC_min_unconnected_distances", rindex, spacing);
	nextvarchangequiet();
	setind((INTBIG)mocmossub_tech, VTECHNOLOGY, "DRC_min_connected_distances", rindex, spacing);
	nextvarchangequiet();
	setind((INTBIG)mocmossub_tech, VTECHNOLOGY, "DRC_min_width", layer, minwidth);
}

/*
 * Routine to change the default width of arc "arcname" to "width".
 */
void mocmossub_setarcwidth(char *arcname, INTBIG width)
{
	REGISTER ARCPROTO *ap;

	for(ap = mocmossub_tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
		if (namesame(ap->protoname, arcname) == 0) break;
	if (ap == NOARCPROTO) return;
	ap->nominalwidth = width * mocmossub_tech->deflambda / WHOLE;
}

/*
 * Routine to change the default size of node "nodename" to "size" squared with a
 * port offset of "portoffset" (this is the distance from the edge in to the port).
 */
void mocmossub_setnodesize(char *nodename, INTBIG size, INTSML portoffset)
{
	REGISTER NODEPROTO *np;
	REGISTER INTBIG i;
	REGISTER TECH_NODES *nty;
	REGISTER TECH_PORTS *npp;

	for(i=0; i<mocmossub_tech->nodeprotocount; i++)
	{
		nty = mocmossub_tech->nodeprotos[i];
		np = nty->creation;
		if (namesame(np->primname, nodename) == 0) break;
	}
	if (np == NONODEPROTO) return;

	np->lowx = -size * mocmossub_tech->deflambda / WHOLE / 2;
	np->highx = size * mocmossub_tech->deflambda / WHOLE / 2;
	np->lowy = -size * mocmossub_tech->deflambda / WHOLE / 2;
	np->highy = size * mocmossub_tech->deflambda / WHOLE / 2;
	npp = &nty->portlist[0];
	npp->lowxsum = portoffset;
	npp->lowysum = portoffset;
	npp->highxsum = -portoffset;
	npp->highysum = -portoffset;
}

/*
 * Routine to set the size of metal layer "layer" on via "nodename" so that it is described
 * with "boxdesc".  Also sets the node inset for this node to "nodeoffset".
 */
void mocmossub_setmetalonvia(char *nodename, INTBIG layer, INTBIG *boxdesc, INTBIG nodeoffset)
{
	REGISTER NODEPROTO *np;
	REGISTER INTBIG i, j;
	REGISTER TECH_NODES *nty;
	REGISTER VARIABLE *var;

	for(i=0; i<mocmossub_tech->nodeprotocount; i++)
	{
		nty = mocmossub_tech->nodeprotos[i];
		np = nty->creation;
		if (namesame(np->primname, nodename) == 0) break;
	}
	if (np == NONODEPROTO) return;

	for(j=0; j<nty->layercount; j++)
		if (nty->layerlist[j].layernum == layer) break;
	if (j >= nty->layercount) return;
	nty->layerlist[j].points = boxdesc;

	var = getval((INTBIG)mocmossub_tech, VTECHNOLOGY, VFRACT|VISARRAY, "TECH_node_width_offset");
	if (var != NOVARIABLE)
	{
		((INTBIG *)var->addr)[i*4] = nodeoffset;
		((INTBIG *)var->addr)[i*4+1] = nodeoffset;
		((INTBIG *)var->addr)[i*4+2] = nodeoffset;
		((INTBIG *)var->addr)[i*4+3] = nodeoffset;
	}
}

INTSML mocmossub_nodepolys(NODEINST *ni)
{
	REGISTER INTSML pindex, count;
	TECH_NODES *thistn;
	REGISTER NODEPROTO *np;

	/* special cases for special primitives */
	np = ni->proto;
	pindex = np->primindex;
	thistn = np->tech->nodeprotos[pindex-1];
	count = thistn->layercount;
	switch (pindex)
	{
		case NMETAL1P:
		case NMETAL2P:
		case NMETAL3P:
		case NMETAL4P:
		case NMETAL5P:
		case NMETAL6P:
		case NPOLY1P:
		case NPOLY2P:
		case NSACTP:
		case NDACTP:
		case NACTP:
			/* pins disappear with one or two wires */
			if (tech_pinusecount(ni, NOARCPROTO) != 0) count = 0;
			break;
		case NMETSACTC:
		case NMETDACTC:
		case NMETPOLY1C:
		case NMETPOLY2C:
		case NMETPOLY12C:
		case NVIA1:
		case NVIA2:
		case NVIA3:
		case NVIA4:
		case NVIA5:
		case NWBUT:
		case NSBUT:
			/* contacts draw a box the size of the port */
			count = 1;
			break;
		case NSTRANS:
		case NDTRANS:
			/* prepare for possible serpentine transistor */
			count = tech_inittrans(2, ni);
			break;
	}

	/* add in displayable variables */
	tech_realpolys = count;
	count += tech_displayablenvars(ni);
	return(count);
}

void mocmossub_shapenodepoly(NODEINST *ni, INTSML box, POLYGON *poly)
{
	TECH_POLYGON *lay;
	REGISTER INTSML pindex;
	REGISTER NODEPROTO *np;
	REGISTER INTBIG lambda, cx, cy;
	REGISTER TECH_NODES *thistn;
	REGISTER TECH_PORTS *portdata;
	static GRAPHICS contactdesc = {LAYERO,BLACK, SOLIDC, SOLIDC,
		{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};

	/* handle displayable variables */
	if (box >= tech_realpolys)
	{
		(void)tech_filldisplayablenvar(ni, poly);
		return;
	}

	np = ni->proto;
	pindex = np->primindex;
	thistn = mocmossub_tech->nodeprotos[pindex-1];
	lambda = mocmossub_tech->deflambda;
	switch (pindex)
	{
		case NMETAL1P:
		case NMETAL2P:
		case NMETAL3P:
		case NMETAL4P:
		case NMETAL5P:
		case NMETAL6P:
		case NPOLY1P:
		case NPOLY2P:
		case NSACTP:
		case NDACTP:
		case NACTP:
			/* pins disappear with one or two wires */
			poly->layer = LPOLYCUT;
			if (poly->limit < 2) (void)extendpolygon(poly, 2);
			cx = (ni->lowx + ni->highx) / 2;
			cy = (ni->lowy + ni->highy) / 2;
			poly->xv[0] = cx;   poly->yv[0] = cy;
			poly->xv[1] = cx;   poly->yv[1] = cy + lambda/2;
			poly->count = 2;
			poly->style = DISC;
			poly->desc = mocmossub_tech->layers[poly->layer];
			break;
		case NMETSACTC:
		case NMETDACTC:
		case NMETPOLY1C:
		case NMETPOLY2C:
		case NMETPOLY12C:
		case NVIA1:
		case NVIA2:
		case NVIA3:
		case NVIA4:
		case NVIA5:
		case NWBUT:
		case NSBUT:
			/* contacts draw a box the size of the port */
			poly->layer = LPOLYCUT;
			if (poly->limit < 2) (void)extendpolygon(poly, 2);
			portdata = &thistn->portlist[0];
			subrange(ni->lowx, ni->highx, portdata->lowxmul, portdata->lowxsum,
				portdata->highxmul, portdata->highxsum, &poly->xv[0], &poly->xv[1], lambda);
			subrange(ni->lowy, ni->highy, portdata->lowymul, portdata->lowysum,
				portdata->highymul, portdata->highysum, &poly->yv[0], &poly->yv[1], lambda);
			poly->count = 2;
			poly->style = CLOSEDRECT;
			poly->desc = &contactdesc;
			switch (pindex)
			{
				case NMETSACTC:   contactdesc.bits = LAYERT1|LAYERT3;  contactdesc.col = COLORT1|COLORT3;   break;
				case NMETDACTC:   contactdesc.bits = LAYERT1|LAYERT3;  contactdesc.col = COLORT1|COLORT3;   break;
				case NMETPOLY1C:  contactdesc.bits = LAYERT1|LAYERT2;  contactdesc.col = COLORT1|COLORT2;   break;
				case NMETPOLY2C:  contactdesc.bits = LAYERO;           contactdesc.col = ORANGE;            break;
				case NMETPOLY12C: contactdesc.bits = LAYERO;           contactdesc.col = ORANGE;            break;
				case NVIA1:       contactdesc.bits = LAYERT1|LAYERT4;  contactdesc.col = COLORT1|COLORT4;   break;
				case NVIA2:       contactdesc.bits = LAYERT4|LAYERT5;  contactdesc.col = COLORT4|COLORT5;   break;
				case NVIA3:       contactdesc.bits = LAYERO;           contactdesc.col = LBLUE;             break;
				case NVIA4:       contactdesc.bits = LAYERO;           contactdesc.col = LRED;              break;
				case NVIA5:       contactdesc.bits = LAYERO;           contactdesc.col = CYAN;              break;
				case NWBUT:       contactdesc.bits = LAYERO;           contactdesc.col = BROWN;             break;
				case NSBUT:       contactdesc.bits = LAYERO;           contactdesc.col = YELLOW;            break;
			}
			break;
		case NSTRANS:
		case NDTRANS:
			/* prepare for possible serpentine transistor */
			lay = &thistn->gra[box].basics;
			poly->layer = lay->layernum;
			if (poly->layer == LPOLY1)
			{
				ni->lowy += lambda;
				ni->highy -= lambda;
			} else
			{
				ni->lowx += lambda + lambda/2;
				ni->highx -= lambda + lambda/2;
			}
			tech_filltrans(poly, &lay, thistn->gra, ni, lambda, box, (TECH_PORTS *)0);
			if (poly->layer == LPOLY1)
			{
				ni->lowy -= lambda;
				ni->highy += lambda;
			} else
			{
				ni->lowx -= lambda + lambda/2;
				ni->highx += lambda + lambda/2;
			}
			poly->desc = mocmossub_tech->layers[poly->layer];
			break;
		default:
			lay = &thistn->layerlist[box];
			tech_fillpoly(poly, lay, ni, lambda, FILLED);
			poly->desc = mocmossub_tech->layers[poly->layer];
			break;
	}
}

void mocmossub_nodesizeoffset(NODEINST *ni, INTBIG *lx, INTBIG *ly, INTBIG *hx, INTBIG *hy)
{
	REGISTER INTSML pindex;
	REGISTER NODEPROTO *np;
	REGISTER INTBIG lambda, cx, cy;
	INTBIG bx, by, ux, uy;
	REGISTER TECH_NODES *thistn;
	REGISTER TECH_PORTS *portdata;

	np = ni->proto;
	pindex = np->primindex;
	lambda = mocmossub_tech->deflambda;
	switch (pindex)
	{
		case NMETAL1P:
		case NMETAL2P:
		case NMETAL3P:
		case NMETAL4P:
		case NMETAL5P:
		case NMETAL6P:
		case NPOLY1P:
		case NPOLY2P:
		case NSACTP:
		case NDACTP:
		case NACTP:
			cx = (ni->lowx + ni->highx) / 2;
			cy = (ni->lowy + ni->highy) / 2;
			*lx = (cx - lambda) - ni->lowx;
			*hx = ni->highx - (cx + lambda);
			*ly = (cy - lambda) - ni->lowy;
			*hy = ni->highy - (cy + lambda);
			break;
		case NMETSACTC:
		case NMETDACTC:
		case NMETPOLY1C:
		case NMETPOLY2C:
		case NMETPOLY12C:
		case NVIA1:
		case NVIA2:
		case NVIA3:
		case NVIA4:
		case NVIA5:
		case NWBUT:
		case NSBUT:
			/* contacts draw a box the size of the port */
			thistn = mocmossub_tech->nodeprotos[pindex-1];
			portdata = &thistn->portlist[0];
			subrange(ni->lowx, ni->highx, portdata->lowxmul, portdata->lowxsum,
				portdata->highxmul, portdata->highxsum, &bx, &ux, lambda);
			subrange(ni->lowy, ni->highy, portdata->lowymul, portdata->lowysum,
				portdata->highymul, portdata->highysum, &by, &uy, lambda);
			*lx = bx - ni->lowx;
			*hx = ni->highx - ux;
			*ly = by - ni->lowy;
			*hy = ni->highy - uy;
			break;
		default:
			nodeprotosizeoffset(np, lx, ly, hx, hy);
			if (pindex == NSTRANS || pindex == NDTRANS)
			{
				*lx += lambda + lambda/2;
				*hx += lambda + lambda/2;
				*ly += lambda;
				*hy += lambda;
			}
			break;
	}
}

INTSML mocmossub_arcpolys(ARCINST *ai)
{
	REGISTER INTSML i;

	i = 1;
	mocmossub_arrowbox = -1;
	if ((ai->userbits&ISDIRECTIONAL) != 0) mocmossub_arrowbox = i++;

	/* add in displayable variables */
	tech_realpolys = i;
	i += tech_displayableavars(ai);
	return(i);
}

void mocmossub_shapearcpoly(ARCINST *ai, INTSML box, POLYGON *poly)
{
	REGISTER INTSML aindex, angle;
	REGISTER INTBIG x1,y1, x2,y2, i;
	REGISTER TECH_ARCLAY *thista;
	static GRAPHICS intense = {LAYERO, RED, SOLIDC, SOLIDC,
		{0,0,0,0,0,0,0,0}, NOVARIABLE, 0};

	/* handle displayable variables */
	if (box >= tech_realpolys)
	{
		(void)tech_filldisplayableavar(ai, poly);
		return;
	}

	/* initialize for the arc */
	aindex = ai->proto->arcindex;
	thista = &mocmossub_arcprotos[aindex]->list[box];
	poly->layer = thista->lay;
	switch (ai->proto->arcindex)
	{
		case AMETAL1:
		case AMETAL2:
		case AMETAL3:
		case AMETAL4:
		case AMETAL5:
		case AMETAL6:
			intense.col = BLUE;
			break;
		case APOLY1:
		case APOLY2:
			intense.col = RED;
			break;
		case ASACT:
		case ADACT:
		case AACT:
			intense.col = DGREEN;
			break;
	}
	if (mocmossub_arrowbox < 0 || box == 0)
	{
		/* simple arc */
		poly->desc = mocmossub_tech->layers[poly->layer];
		makearcpoly(ai->length, ai->width-ai->proto->nominalwidth, ai, poly, thista->style);
		return;
	}

	/* prepare special information for directional arcs */
	poly->desc = &intense;
	x1 = ai->end[0].xpos;   y1 = ai->end[0].ypos;
	x2 = ai->end[1].xpos;   y2 = ai->end[1].ypos;
	angle = (INTSML)(((ai->userbits&AANGLE) >> AANGLESH) * 10);
	if ((ai->userbits&REVERSEEND) != 0)
	{
		i = x1;   x1 = x2;   x2 = i;
		i = y1;   y1 = y2;   y2 = i;
		angle = (angle+1800) % 3600;
	}

	/* draw the directional arrow */
	poly->style = VECTORS;
	poly->layer = -1;
	if (poly->limit < 2) (void)extendpolygon(poly, 2);
	poly->count = 0;
	if ((ai->userbits&NOTEND1) == 0) tech_addheadarrow(poly, angle, x2, y2);
}

INTBIG mocmossub_arcwidthoffset(ARCINST *ai)
{
	return(ai->proto->nominalwidth);
}

#endif  /* TECMOCOS - at top */
