;;; math-symbol-lists.el --- Lists of Unicode math symbols and latex commands -*- lexical-binding:t -*-
;;
;; Copyright (C) 2014-2019 Free Software Foundation, Inc.
;; Author: Vitalie Spinu <spinuvit@gmail.com>
;; URL: https://github.com/vspinu/math-symbol-lists
;; Keywords: Unicode, symbols, mathematics
;; Package-Type: simple
;; Version: 1.3
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; This file is part of GNU Emacs.
;;
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 3, or
;; (at your option) any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs.  If not, see <https://www.gnu.org/licenses/>.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;; Commentary:
;;
;; This is a "storage" package used by completion engines such as
;; `company-math.el` and `ac-math.el`.
;;
;; Defined (a)lists are:
;;
;;  `math-symbol-list-basic`
;;  `math-symbol-list-extended`
;;  `math-symbol-list-packages`
;;  `math-symbol-list-latex-commands`
;;  `math-symbol-list-subscripts`
;;  `math-symbol-list-superscripts`
;;
;; Sources:
;;
;;  [1] http://milde.users.sourceforge.net/LUCR/Math/
;;  [2] https://github.com/wspr/unicode-math/blob/master/unicode-math-table.tex
;;  [3] http://milde.users.sourceforge.net/LUCR/Math/data/unimathsymbols.txt
;;  [4] M-x describe-input-method TeX
;;  [5] `LaTeX-math-default' in AucTeX package
;;  [6] LaTeX 2 unicode: https://www.cl.cam.ac.uk/%7Emgk25/ucs/examples/TeX.txt
;;  [7] W3C math WG: https://www.w3.org/2003/entities/2007xml/unicode.xml
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;; Code:


;;; SYMBOLS

(defconst math-symbol-list-latex-commands
  '("address" "addtocounter" "addtolength" "addvspace" "Alph" "alph" "Alph
    example" "alsoname" "and for author" "appendix" "arabic" "arraycolsep"
    "arrayrulewidth" "arraystretch" "baselineskip"
    "baselinestretch" "begin" "bf" "bfseries" "bibitem" "bibliography"
    "bibliographystyle" "bigskip" "bigskipamount" "bmod" "boldmath"
    "bottomfraction" "cal" "caption" "cc" "centering"
    "chapter" "circle" "cite" "cleardoublepage" "clearpage" "cline" "closing"
    "columnsep" "columnseprule" "columnwidth" "contentsline" "copyright"
    "dag" "dashbox" "day" "dblfloatpagefraction"
    "dblfloatsep" "dbltextfloatsep" "dbltopfraction" "ddag" "depth"
    "displaystyle" "documentclass" "dotfill" "doublerulesep" "emph" "encl"
    "enlargethispage" "enumi" "enumii" "enumiii" "enumiv" "evensidemargin"
    "fbox" "fboxrule" "fboxsep" "fill" "floatpagefraction" "floatsep"
    "flushbottom" "fnsymbol" "fontencoding" "fontfamily" "fontseries"
    "fontshape" "fontsize" "footnote" "footnotemark" "footnoterule"
    "footnotesep" "footnotesize" "footnotetext" "footskip" "frame" "framebox"
    "fussy" "gets" "glossary" "glossaryentry"
    "headheight" "headsep" "height" "hfill" "hline" "hrulefill" "hspace"
    "Huge" "huge" "hyphenation" "iff" "include" "includeonly"
    "indent" "index" "indexentry" "input" "intextsep"
    "item" "itemindent" "itemsep" "itshape" "kill"
    "label" "labelenumi" "labelenumii" "labelenumiii" "labelenumiv"
    "labelitemi" "labelitemii" "labelitemiii" "labelitemiv" "labelsep"
    "labelwidth" "land" "LARGE" "Large" "large" "LaTeX" "le" "left" "leadsto"
    "lefteqn" "leftmargin" "leftmargini" "leftmarginii" "leftmarginiii"
    "leftmarginiv" "leftmarginv" "leftmarginvi" "line" "linebreak"
    "linethickness" "linewidth" "listoffigures" "listoftables" "listparindent"
    "lnot" "location" "lor" "lq" "makebox" "makebox " "makeglossary"
    "makeindex" "makelabels" "maketitle" "marginpar" "marginparpush"
    "marginparsep" "marginparwidth" "mathbf" "mathcal" "mathnormal" "mathrm"
    "mathsf" "mathtt" "mathversion" "mbox" "mdseries" "medskip"
    "medskipamount" "month" "multicolumn" "multiput" "name" "newcommand"
    "newcounter" "newenvironment" "newfont" "newlength" "NEWLINE" "newline"
    "newpage" "newsavebox" "newtheorem" "nocite" "nofiles" "noindent"
    "nolinebreak" "nonumber" "nopagebreak" "normalfont" "normalmarginpar"
    "normalsize" "oe" "onecolumn" "opening" "oval" "owns" "P" "pagebreak" "pagenumbering"
    "pageref" "pagestyle" "paragraph" "parbox" "parindent"
    "parsep" "parskip" "parskip example" "part" "partopsep" "pmod" "poptabs"
    "pounds" "protect" "ps" "pushtabs" "put" "raggedbottom" "raggedleft"
    "raggedright" "raisebox" "ref" "refstepcounter" "renewenvironment"
    "restorecr" "reversemarginpar" "right" "rightmargin" "rm" "rmfamily"
    "roman" "rq" "rule" "savebox" "sbox" "sc" "scriptsize" "scshape"
    "section" "seename" "selectfont" "setcounter" "setlength" "settodepth"
    "settoheight" "settowidth" "sf" "sffamily" "shortstack" "signature" "sl"
    "slshape" "small" "smallint" "smallskip" "smallskipamount" "SPACE" "ss"
    "startbreaks" "stepcounter" "stop" "stopbreaks" "subparagraph"
    "subsection" "subsubsection" "symbol" "TAB"
    "tabbingsep" "tabcolsep" "tableofcontents" "telephone" "TeX" "textbf"
    "textfloatsep" "textfraction" "textheight" "textit" "textmd" "textnormal"
    "textrm" "textsc" "textsf" "textsl" "texttt" "textup" "textwidth"
    "thicklines" "thinlines" "thinspace" "thispagestyle" "tiny" "to" "today"
    "topfraction" "topmargin" "topsep" "topskip" "totalheight" "tt" "ttfamily"
    "twocolumn" "typein" "typeout" "u " "unboldmath"
    "unitlength" "upshape" "usebox" "usecounter" "usefont" "usepackage"
    "value" "vector" "verb" "vert" "vfill" "vline" "vspace"
    "width" "year")
  "List of the latex commands.")

(defconst math-symbol-list-basic
  '(("accent"	"\\acute"	#X301)
    ("accent"	"\\bar"	#X304)
    ("accent"	"\\breve"	#X306)
    ("accent"	"\\check"	#X30C)
    ("accent"	"\\ddot"	#X308)
    ("accent"	"\\dot"	#X307)
    ("accent"	"\\grave"	#X300)
    ("accent"	"\\hat"	#X302)
    ("accent"	"\\tilde"	#X303)
    ("accent"	"\\vec"	#X20D7)
    ("arrow"	"\\Downarrow"	#X21D3)
    ("arrow"	"\\Leftarrow"	#X21D0)
    ("arrow"	"\\Leftrightarrow"	#X21D4)
    ("arrow"	"\\Longleftarrow"	#X27F8)
    ("arrow"	"\\Longleftrightarrow"	#X27FA)
    ("arrow"	"\\Longrightarrow"	#X27F9)
    ("arrow"	"\\Rightarrow"	#X21D2)
    ("arrow"	"\\Uparrow"	#X21D1)
    ("arrow"	"\\Updownarrow"	#X21D5)
    ("arrow"	"\\downarrow"	#X2193)
    ("arrow"	"\\hookleftarrow"	#X21A9)
    ("arrow"	"\\hookrightarrow"	#X21AA)
    ("arrow"	"\\leftarrow"	#X2190)
    ("arrow"	"\\leftharpoondown"	#X21BD)
    ("arrow"	"\\leftharpoonup"	#X21BC)
    ("arrow"	"\\leftrightarrow"	#X2194)
    ("arrow"	"\\longleftarrow"	#X27F5)
    ("arrow"	"\\longleftrightarrow"	#X27F7)
    ("arrow"	"\\longmapsto"	#X27FC)
    ("arrow"	"\\longrightarrow"	#X27F6)
    ("arrow"	"\\mapsto"	#X21A6)
    ("arrow"	"\\nearrow"	#X2197)
    ("arrow"	"\\nwarrow"	#X2196)
    ("arrow"	"\\rightarrow"	#X2192)
    ("arrow"	"\\rightharpoondown"	#X21C1)
    ("arrow"	"\\rightharpoonup"	#X21C0)
    ("arrow"	"\\searrow"	#X2198)
    ("arrow"	"\\swarrow"	#X2199)
    ("arrow"	"\\uparrow"	#X2191)
    ("arrow"	"\\updownarrow"	#X2195)
    ("bin"	"\\amalg"	#X2A3F)
    ("bin"	"\\ast"	#X2217)
    ("bin"	"\\bigcirc"	#X25CB)
    ("bin"	"\\bigtriangledown"	#X25BD)
    ("bin"	"\\bigtriangleup"	#X25B3)
    ("bin"	"\\bullet"	#X2219)
    ("bin"	"\\cap"	#X2229)
    ("bin"	"\\cdot"	#X22C5)
    ("bin"	"\\circ"	#X2218)
    ("bin"	"\\cup"	#X222A)
    ("bin"	"\\dagger"	#X2020)
    ("bin"	"\\ddagger"	#X2021)
    ("bin"	"\\diamond"	#X22C4)
    ("bin"	"\\div"	#XF7)
    ("bin"	"\\lhd")
    ("bin"	"\\mp"	#X2213)
    ("bin"	"\\odot"	#X2299)
    ("bin"	"\\ominus"	#X2296)
    ("bin"	"\\oplus"	#X2295)
    ("bin"	"\\oslash"	#X2205)
    ("bin"	"\\otimes"	#X2297)
    ("bin"	"\\pm"	#XB1)
    ("bin"	"\\rhd")
    ("bin"	"\\setminus"	#X2216)
    ("bin"	"\\sqcap"	#X2293)
    ("bin"	"\\star"	#X22C6)
    ("bin"	"\\times"	#XD7)
    ("bin"	"\\triangleleft"	#X25C1)
    ("bin"	"\\triangleright"	#X25B7)
    ("bin"	"\\unlhd")
    ("bin"	"\\unrhd")
    ("bin"	"\\uplus"	#X228E)
    ("bin"	"\\vee"	#X2228)
    ("bin"	"\\wedge"	#X2227)
    ("bin"	"\\wr"	#X2240)
    ("constr"	"\\frac")
    ("constr"	"\\overbrace"	#XFE37)
    ("constr"	"\\overleftarrow"	#X20D6)
    ("constr"	"\\overline")
    ("constr"	"\\overrightarrow")
    ("constr"	"\\sqrt"	#X221A)
    ("constr"	"\\underbrace"	#XFE38)
    ("constr"	"\\underline")
    ("constr"	"\\widehat"	#X302)
    ("constr"	"\\widetilde"	#X303)
    ("delim"	"\\Arrowvert")
    ("delim"	"\\arrowvert")
    ("delim"	"\\backslash"	#X5C)
    ("delim"	"\\bracevert")
    ("delim"	"\\langle"	#X27E8)
    ("delim"	"\\lceil"	#X2308)
    ("delim"	"\\lfloor"	#X230A)
    ("delim"	"\\lgroup")
    ("delim"	"\\lmoustache"	#X23B0)
    ("delim"	"\\rangle"	#X27E9)
    ("delim"	"\\rceil"	#X2309)
    ("delim"	"\\rfloor"	#X230B)
    ("delim"	"\\rgroup")
    ("delim"	"\\rmoustache"	#X23B1)
    ("delim"	"\\{")
    ("delim"	"\\|")
    ("delim"	"\\}")
    ("greek"	"\\alpha"	#X3B1)
    ("greek"	"\\beta"	#X3B2)
    ("greek"	"\\chi"	#X3C7)
    ("greek"	"\\delta"	#X3B4)
    ("greek"	"\\epsilon"	#X3F5)
    ("greek"	"\\eta"	#X3B7)
    ("greek"	"\\gamma"	#X3B3)
    ("greek"	"\\iota"	#X3B9)
    ("greek"	"\\kappa"	#X3BA)
    ("greek"	"\\lambda"	#X3BB)
    ("greek"	"\\mu"	#X3BC)
    ("greek"	"\\nu"	#X3BD)
    ("greek"	"\\omega"	#X3C9)
    ("greek"	"\\phi"	#X3D5)
    ("greek"	"\\pi"	#X3C0)
    ("greek"	"\\psi"	#X3C8)
    ("greek"	"\\rho"	#X3C1)
    ("greek"	"\\sigma"	#X3C3)
    ("greek"	"\\tau"	#X3C4)
    ("greek"	"\\theta"	#X3B8)
    ("greek"	"\\upsilon"	#X3C5)
    ("greek"	"\\varepsilon"	#X3B5)
    ("greek"	"\\varphi"	#X3C6)
    ("greek"	"\\varpi"	#X3D6)
    ("greek"	"\\varrho"	#X3F1)
    ("greek"	"\\varsigma"	#X3C2)
    ("greek"	"\\vartheta"	#X3D1)
    ("greek"	"\\xi"	#X3BE)
    ("greek"	"\\zeta"	#X3B6)
    ("Greek"	"\\Delta"	#X394)
    ("Greek"	"\\Gamma"	#X393)
    ("Greek"	"\\Lambda"	#X39B)
    ("Greek"	"\\Omega"	#X3A9)
    ("Greek"	"\\Phi"	#X3A6)
    ("Greek"	"\\Pi"	#X3A0)
    ("Greek"	"\\Psi"	#X3A8)
    ("Greek"	"\\Sigma"	#X3A3)
    ("Greek"	"\\Theta"	#X398)
    ("Greek"	"\\Upsilon"	#X3D2)
    ("Greek"	"\\Xi"	#X39E)
    ("misc"	"\\Im"	#X2111)
    ("misc"	"\\Re"	#X211C)
    ("misc"	"\\aleph"	#X2135)
    ("misc"	"\\angle"	#X2220)
    ("misc"	"\\bot"	#X22A5)
    ("misc"	"\\clubsuit"	#X2663)
    ("misc"	"\\diamondsuit"	#X2662)
    ("misc"	"\\ell"	#X2113)
    ("misc"	"\\emptyset"	#X2205)
    ("misc"	"\\exists"	#X2203)
    ("misc"	"\\flat"	#X266D)
    ("misc"	"\\forall"	#X2200)
    ("misc"	"\\hbar"	#X210F)
    ("misc"	"\\heartsuit"	#X2661)
    ("misc"	"\\imath"	#X131)
    ("misc"	"\\infty"	#X221E)
    ("misc"	"\\jmath"	#X1D6A5)
    ("misc"	"\\mho"	#X2127)
    ("misc"	"\\nabla"	#X2207)
    ("misc"	"\\natural"	#X266E)
    ("misc"	"\\neg"	#XAC)
    ("misc"	"\\not"	#X338)
    ("misc"	"\\partial"	#X2202)
    ("misc"	"\\prime"	#X2032)
    ("misc"	"\\sharp"	#X266F)
    ("misc"	"\\spadesuit"	#X2660)
    ("misc"	"\\surd"	#X221A)
    ("misc"	"\\top"	#X22A4)
    ("misc"	"\\triangle"	#X25B3)
    ("misc"	"\\wp"	#X2118)
    ("nonsymb"	"\\Box")
    ("nonsymb"	"\\Diamond")
    ("nonsymb"	"\\Pr")
    ("nonsymb"	"\\arccos")
    ("nonsymb"	"\\arcsin")
    ("nonsymb"	"\\arctan")
    ("nonsymb"	"\\arg")
    ("nonsymb"	"\\cos")
    ("nonsymb"	"\\cosh")
    ("nonsymb"	"\\cot")
    ("nonsymb"	"\\coth")
    ("nonsymb"	"\\csc")
    ("nonsymb"	"\\deg")
    ("nonsymb"	"\\det")
    ("nonsymb"	"\\dim")
    ("nonsymb"	"\\exp")
    ("nonsymb"	"\\gcd")
    ("nonsymb"	"\\hom")
    ("nonsymb"	"\\inf")
    ("nonsymb"	"\\ker")
    ("nonsymb"	"\\lg")
    ("nonsymb"	"\\lim")
    ("nonsymb"	"\\liminf")
    ("nonsymb"	"\\limsup")
    ("nonsymb"	"\\ln")
    ("nonsymb"	"\\log")
    ("nonsymb"	"\\max")
    ("nonsymb"	"\\min")
    ("nonsymb"	"\\sec")
    ("nonsymb"	"\\sin")
    ("nonsymb"	"\\sinh")
    ("nonsymb"	"\\sup")
    ("nonsymb"	"\\tan")
    ("nonsymb"	"\\tanh")
    ("punct"	"\\cdots"	#X22EF)
    ("punct"	"\\colon"	#X3A)
    ("punct"	"\\ddots"	#X22F1)
    ("punct"	"\\ldots"	#X2026)
    ("punct"	"\\vdots"	#X22EE)
    ("rel"	"\\Join"	#X2A1D)
    ("rel"	"\\approx"	#X2248)
    ("rel"	"\\asymp"	#X224D)
    ("rel"	"\\bowtie"	#X22C8)
    ("rel"	"\\cong"	#X2245)
    ("rel"	"\\dashv"	#X22A3)
    ("rel"	"\\doteq"	#X2250)
    ("rel"	"\\equiv"	#X2261)
    ("rel"	"\\frown"	#X2322)
    ("rel"	"\\geq"	#X2265)
    ("rel"	"\\gg"	#X226B)
    ("rel"	"\\in"	#X2208)
    ("rel"	"\\leq"	#X2264)
    ("rel"	"\\ll"	#X226A)
    ("rel"	"\\mid"	#X2223)
    ("rel"	"\\models"	#X22A7)
    ("rel"	"\\neq"	#X2260)
    ("rel"	"\\ni"	#X220B)
    ("rel"	"\\parallel"	#X2225)
    ("rel"	"\\perp"	#X27C2)
    ("rel"	"\\prec"	#X227A)
    ("rel"	"\\preceq"	#X2AAF)
    ("rel"	"\\propto"	#X221D)
    ("rel"	"\\qed"	#X220E)
    ("rel"	"\\sim"	#X223C)
    ("rel"	"\\simeq"	#X2243)
    ("rel"	"\\smile"	#X2323)
    ("rel"	"\\sqsubset"	#X228F)
    ("rel"	"\\sqsubseteq"	#X2291)
    ("rel"	"\\sqsupset"	#X2290)
    ("rel"	"\\sqsupseteq"	#X2292)
    ("rel"	"\\subset"	#X2282)
    ("rel"	"\\subseteq"	#X2286)
    ("rel"	"\\succ"	#X227B)
    ("rel"	"\\succeq"	#X2AB0)
    ("rel"	"\\supset"	#X2283)
    ("rel"	"\\supseteq"	#X2287)
    ("rel"	"\\vdash"	#X22A2)
    ("var"	"\\bigcap"	#X22C2)
    ("var"	"\\bigcup"	#X22C3)
    ("var"	"\\bigodot"	#X2A00)
    ("var"	"\\bigoplus"	#X2A01)
    ("var"	"\\bigotimes"	#X2A02)
    ("var"	"\\bigsqcup"	#X2A06)
    ("var"	"\\biguplus"	#X2A04)
    ("var"	"\\bigvee"	#X22C1)
    ("var"	"\\bigwedge"	#X22C0)
    ("var"	"\\coprod"	#X2210)
    ("var"	"\\int"	#X222B)
    ("var"	"\\oint"	#X222E)
    ("var"	"\\prod"	#X220F)
    ("var"	"\\sum"	#X2211)
    ;; extra aliases retrieved from LUCR
    ("bin" "\\land" 8743 "∧")
    ("bin" "\\lor" 8744 "∨")
    ("close" "\\rbrace" 125 "}")
    ("fence" "\\vert" 124 "|")
    ("fence" "\\Vert" 8214 "‖")
    ("open" "\\lbrace" 123 "{")
    ("ord" "\\mathdollar" 36 "$")
    ("ord" "\\lnot" 172 "¬")
    ("rel" "\\gets" 8592 "←")
    ("rel" "\\to" 8594 "→")
    ("rel" "\\owns" 8715 "∋")
    ("rel" "\\ne" 8800 "≠")
    ("rel" "\\le" 8804 "≤")
    ("rel" "\\ge" 8805 "≥")
    ;; manually added
    ("misc" "\\P" 182 "¶")
    ("misc" "\\textpilcrow" 182 "¶"))
  "List of basic LaTeX mathematical symbols.
This list maps standard LaTeX math commands to unicode
characters.  For some symbols in this list the unicode code is
missing.  It is an extension of `LaTeX-math-default' in
AucTeX/latex.el, but it doesn't include AMS symbols. See also
`math-symbol-list-extended' and `math-symbol-list-packages'.")

(defconst math-symbol-list-extended
  '(("mathaccent" "\\LVec" 8406 "⃖")
    ("mathaccent" "\\acute" 769 "́")
    ("mathaccent" "\\annuity" 8423 "⃧")
    ("mathaccent" "\\asteraccent" 8432 "⃰")
    ("mathaccent" "\\bar" 772 "̄")
    ("mathaccent" "\\breve" 774 "̆")
    ("mathaccent" "\\candra" 784 "̐")
    ("mathaccent" "\\check" 780 "̌")
    ("mathaccent" "\\ddddot" 8412 "⃜")
    ("mathaccent" "\\dddot" 8411 "⃛")
    ("mathaccent" "\\ddot" 776 "̈")
    ("mathaccent" "\\dot" 775 "̇")
    ("mathaccent" "\\droang" 794 "̚")
    ("mathaccent" "\\enclosecircle" 8413 "⃝")
    ("mathaccent" "\\enclosediamond" 8415 "⃟")
    ("mathaccent" "\\enclosesquare" 8414 "⃞")
    ("mathaccent" "\\enclosetriangle" 8420 "⃤")
    ("mathaccent" "\\grave" 768 "̀")
    ("mathaccent" "\\hat" 770 "̂")
    ("mathaccent" "\\leftharpoonaccent" 8400 "⃐")
    ("mathaccent" "\\lvec" 8400 "⃐")
    ("mathaccent" "\\mathring" 778 "̊")
    ("mathaccent" "\\not" 824 "̸")
    ("mathaccent" "\\ocirc" 778 "̊")
    ("mathaccent" "\\ocommatopright" 789 "̕")
    ("mathaccent" "\\oturnedcomma" 786 "̒")
    ("mathaccent" "\\overbar" 773 "̅")
    ("mathaccent" "\\overleftarrow" 8406 "⃖")
    ("mathaccent" "\\overleftrightarrow" 8417 "⃡")
    ("mathaccent" "\\overline" 773 "̅")
    ("mathaccent" "\\ovhook" 777 "̉")
    ("mathaccent" "\\rightharpoonaccent" 8401 "⃑")
    ("mathaccent" "\\threeunderdot" 8424 "⃨")
    ("mathaccent" "\\tilde" 771 "̃")
    ("mathaccent" "\\underbar" 817 "̱")
    ("mathaccent" "\\underleftarrow" 8430 "⃮")
    ("mathaccent" "\\underleftharpoondown" 8429 "⃭")
    ("mathaccent" "\\underline" 818 "̲")
    ("mathaccent" "\\underrightarrow" 8431 "⃯")
    ("mathaccent" "\\underrightharpoondown" 8428 "⃬")
    ("mathaccent" "\\utilde" 816 "̰")
    ("mathaccent" "\\vec" 8401 "⃑")
    ("mathaccent" "\\vec" 8407 "⃗")
    ("mathaccent" "\\vertoverlay" 8402 "⃒")
    ("mathaccent" "\\widebridgeabove" 8425 "⃩")
    ("mathaccent" "\\wideutilde" 816 "̰")
    ("mathalpha" "\\Angstroem" 8491 "Å")
    ("mathalpha" "\\Angstrom" 8491 "Å")
    ("mathalpha" "\\BbbA" 120120 "𝔸")
    ("mathalpha" "\\BbbB" 120121 "𝔹")
    ("mathalpha" "\\BbbC" 8450 "ℂ")
    ("mathalpha" "\\BbbD" 120123 "𝔻")
    ("mathalpha" "\\BbbE" 120124 "𝔼")
    ("mathalpha" "\\BbbF" 120125 "𝔽")
    ("mathalpha" "\\BbbG" 120126 "𝔾")
    ("mathalpha" "\\BbbGamma" 8510 "ℾ")
    ("mathalpha" "\\BbbH" 8461 "ℍ")
    ("mathalpha" "\\BbbI" 120128 "𝕀")
    ("mathalpha" "\\BbbJ" 120129 "𝕁")
    ("mathalpha" "\\BbbK" 120130 "𝕂")
    ("mathalpha" "\\BbbL" 120131 "𝕃")
    ("mathalpha" "\\BbbM" 120132 "𝕄")
    ("mathalpha" "\\BbbN" 8469 "ℕ")
    ("mathalpha" "\\BbbO" 120134 "𝕆")
    ("mathalpha" "\\BbbP" 8473 "ℙ")
    ("mathalpha" "\\BbbPi" 8511 "ℿ")
    ("mathalpha" "\\BbbQ" 8474 "ℚ")
    ("mathalpha" "\\BbbR" 8477 "ℝ")
    ("mathalpha" "\\BbbS" 120138 "𝕊")
    ("mathalpha" "\\BbbT" 120139 "𝕋")
    ("mathalpha" "\\BbbU" 120140 "𝕌")
    ("mathalpha" "\\BbbV" 120141 "𝕍")
    ("mathalpha" "\\BbbW" 120142 "𝕎")
    ("mathalpha" "\\BbbX" 120143 "𝕏")
    ("mathalpha" "\\BbbY" 120144 "𝕐")
    ("mathalpha" "\\BbbZ" 8484 "ℤ")
    ("mathalpha" "\\Bbba" 120146 "𝕒")
    ("mathalpha" "\\Bbbb" 120147 "𝕓")
    ("mathalpha" "\\Bbbc" 120148 "𝕔")
    ("mathalpha" "\\Bbbd" 120149 "𝕕")
    ("mathalpha" "\\Bbbe" 120150 "𝕖")
    ("mathalpha" "\\Bbbf" 120151 "𝕗")
    ("mathalpha" "\\Bbbg" 120152 "𝕘")
    ("mathalpha" "\\Bbbgamma" 8509 "ℽ")
    ("mathalpha" "\\Bbbh" 120153 "𝕙")
    ("mathalpha" "\\Bbbi" 120154 "𝕚")
    ("mathalpha" "\\Bbbj" 120155 "𝕛")
    ("mathalpha" "\\Bbbk" 120156 "𝕜")
    ("mathalpha" "\\Bbbl" 120157 "𝕝")
    ("mathalpha" "\\Bbbm" 120158 "𝕞")
    ("mathalpha" "\\Bbbn" 120159 "𝕟")
    ("mathalpha" "\\Bbbo" 120160 "𝕠")
    ("mathalpha" "\\Bbbp" 120161 "𝕡")
    ("mathalpha" "\\Bbbq" 120162 "𝕢")
    ("mathalpha" "\\Bbbr" 120163 "𝕣")
    ("mathalpha" "\\Bbbs" 120164 "𝕤")
    ("mathalpha" "\\Bbbt" 120165 "𝕥")
    ("mathalpha" "\\Bbbu" 120166 "𝕦")
    ("mathalpha" "\\Bbbv" 120167 "𝕧")
    ("mathalpha" "\\Bbbw" 120168 "𝕨")
    ("mathalpha" "\\Bbbx" 120169 "𝕩")
    ("mathalpha" "\\Bbby" 120170 "𝕪")
    ("mathalpha" "\\Bbbz" 120171 "𝕫")
    ("mathalpha" "\\Delta" 916 "Δ")
    ("mathalpha" "\\Delta" 120549 "𝛥")
    ("mathalpha" "\\Digamma" 988 "Ϝ")
    ("mathalpha" "\\Gamma" 915 "Γ")
    ("mathalpha" "\\Gamma" 120548 "𝛤")
    ("mathalpha" "\\Im" 8465 "ℑ")
    ("mathalpha" "\\Koppa" 990 "Ϟ")
    ("mathalpha" "\\Lambda" 923 "Λ")
    ("mathalpha" "\\Lambda" 120556 "𝛬")
    ("mathalpha" "\\Micro" 181 "µ")
    ("mathalpha" "\\Omega" 937 "Ω")
    ("mathalpha" "\\Omega" 120570 "𝛺")
    ("mathalpha" "\\Phi" 934 "Φ")
    ("mathalpha" "\\Phi" 120567 "𝛷")
    ("mathalpha" "\\Pi" 928 "Π")
    ("mathalpha" "\\Pi" 120561 "𝛱")
    ("mathalpha" "\\Psi" 936 "Ψ")
    ("mathalpha" "\\Psi" 120569 "𝛹")
    ("mathalpha" "\\Re" 8476 "ℜ")
    ("mathalpha" "\\Sampi" 992 "Ϡ")
    ("mathalpha" "\\Sigma" 931 "Σ")
    ("mathalpha" "\\Sigma" 120564 "𝛴")
    ("mathalpha" "\\Stigma" 986 "Ϛ")
    ("mathalpha" "\\Theta" 920 "Θ")
    ("mathalpha" "\\Theta" 120553 "𝛩")
    ("mathalpha" "\\Upsilon" 933 "Υ")
    ("mathalpha" "\\Upsilon" 120566 "𝛶")
    ("mathalpha" "\\Xi" 926 "Ξ")
    ("mathalpha" "\\Xi" 120559 "𝛯")
    ("mathalpha" "\\aleph" 8501 "ℵ")
    ("mathalpha" "\\alpha" 945 "α")
    ("mathalpha" "\\alpha" 120572 "𝛼")
    ("mathalpha" "\\beta" 946 "β")
    ("mathalpha" "\\beta" 120573 "𝛽")
    ("mathalpha" "\\beth" 8502 "ℶ")
    ("mathalpha" "\\chi" 967 "χ")
    ("mathalpha" "\\chi" 120594 "𝜒")
    ("mathalpha" "\\daleth" 8504 "ℸ")
    ("mathalpha" "\\delta" 948 "δ")
    ("mathalpha" "\\delta" 120575 "𝛿")
    ("mathalpha" "\\digamma" 989 "ϝ")
    ("mathalpha" "\\ell" 8467 "ℓ")
    ("mathalpha" "\\epsilon" 1013 "ϵ")
    ("mathalpha" "\\epsilon" 120598 "𝜖")
    ("mathalpha" "\\eta" 951 "η")
    ("mathalpha" "\\eta" 120578 "𝜂")
    ("mathalpha" "\\eth" 240 "ð")
    ("mathalpha" "\\gamma" 947 "γ")
    ("mathalpha" "\\gamma" 120574 "𝛾")
    ("mathalpha" "\\gimel" 8503 "ℷ")
    ("mathalpha" "\\hslash" 8463 "ℏ")
    ("mathalpha" "\\imath" 305 "ı")
    ("mathalpha" "\\imath" 120484 "𝚤")
    ("mathalpha" "\\iota" 953 "ι")
    ("mathalpha" "\\iota" 120580 "𝜄")
    ("mathalpha" "\\jmath" 567 "ȷ")
    ("mathalpha" "\\jmath" 120485 "𝚥")
    ("mathalpha" "\\kappa" 954 "κ")
    ("mathalpha" "\\kappa" 120581 "𝜅")
    ("mathalpha" "\\koppa" 991 "ϟ")
    ("mathalpha" "\\lambda" 955 "λ")
    ("mathalpha" "\\lambda" 120582 "𝜆")
    ("mathalpha" "\\matheth" 240 "ð")
    ("mathalpha" "\\mbfA" 119808 "𝐀")
    ("mathalpha" "\\mbfAlpha" 120488 "𝚨")
    ("mathalpha" "\\mbfB" 119809 "𝐁")
    ("mathalpha" "\\mbfBeta" 120489 "𝚩")
    ("mathalpha" "\\mbfC" 119810 "𝐂")
    ("mathalpha" "\\mbfChi" 120510 "𝚾")
    ("mathalpha" "\\mbfD" 119811 "𝐃")
    ("mathalpha" "\\mbfDelta" 120491 "𝚫")
    ("mathalpha" "\\mbfDigamma" 120778 "𝟊")
    ("mathalpha" "\\mbfE" 119812 "𝐄")
    ("mathalpha" "\\mbfEpsilon" 120492 "𝚬")
    ("mathalpha" "\\mbfEta" 120494 "𝚮")
    ("mathalpha" "\\mbfF" 119813 "𝐅")
    ("mathalpha" "\\mbfG" 119814 "𝐆")
    ("mathalpha" "\\mbfGamma" 120490 "𝚪")
    ("mathalpha" "\\mbfH" 119815 "𝐇")
    ("mathalpha" "\\mbfI" 119816 "𝐈")
    ("mathalpha" "\\mbfIota" 120496 "𝚰")
    ("mathalpha" "\\mbfJ" 119817 "𝐉")
    ("mathalpha" "\\mbfK" 119818 "𝐊")
    ("mathalpha" "\\mbfKappa" 120497 "𝚱")
    ("mathalpha" "\\mbfL" 119819 "𝐋")
    ("mathalpha" "\\mbfLambda" 120498 "𝚲")
    ("mathalpha" "\\mbfM" 119820 "𝐌")
    ("mathalpha" "\\mbfMu" 120499 "𝚳")
    ("mathalpha" "\\mbfN" 119821 "𝐍")
    ("mathalpha" "\\mbfNu" 120500 "𝚴")
    ("mathalpha" "\\mbfO" 119822 "𝐎")
    ("mathalpha" "\\mbfOmega" 120512 "𝛀")
    ("mathalpha" "\\mbfOmicron" 120502 "𝚶")
    ("mathalpha" "\\mbfP" 119823 "𝐏")
    ("mathalpha" "\\mbfPhi" 120509 "𝚽")
    ("mathalpha" "\\mbfPi" 120503 "𝚷")
    ("mathalpha" "\\mbfPsi" 120511 "𝚿")
    ("mathalpha" "\\mbfQ" 119824 "𝐐")
    ("mathalpha" "\\mbfR" 119825 "𝐑")
    ("mathalpha" "\\mbfRho" 120504 "𝚸")
    ("mathalpha" "\\mbfS" 119826 "𝐒")
    ("mathalpha" "\\mbfSigma" 120506 "𝚺")
    ("mathalpha" "\\mbfT" 119827 "𝐓")
    ("mathalpha" "\\mbfTau" 120507 "𝚻")
    ("mathalpha" "\\mbfTheta" 120495 "𝚯")
    ("mathalpha" "\\mbfU" 119828 "𝐔")
    ("mathalpha" "\\mbfUpsilon" 120508 "𝚼")
    ("mathalpha" "\\mbfV" 119829 "𝐕")
    ("mathalpha" "\\mbfW" 119830 "𝐖")
    ("mathalpha" "\\mbfX" 119831 "𝐗")
    ("mathalpha" "\\mbfXi" 120501 "𝚵")
    ("mathalpha" "\\mbfY" 119832 "𝐘")
    ("mathalpha" "\\mbfZ" 119833 "𝐙")
    ("mathalpha" "\\mbfZeta" 120493 "𝚭")
    ("mathalpha" "\\mbfa" 119834 "𝐚")
    ("mathalpha" "\\mbfalpha" 120514 "𝛂")
    ("mathalpha" "\\mbfb" 119835 "𝐛")
    ("mathalpha" "\\mbfbeta" 120515 "𝛃")
    ("mathalpha" "\\mbfc" 119836 "𝐜")
    ("mathalpha" "\\mbfchi" 120536 "𝛘")
    ("mathalpha" "\\mbfd" 119837 "𝐝")
    ("mathalpha" "\\mbfdelta" 120517 "𝛅")
    ("mathalpha" "\\mbfdigamma" 120779 "𝟋")
    ("mathalpha" "\\mbfe" 119838 "𝐞")
    ("mathalpha" "\\mbfepsilon" 120518 "𝛆")
    ("mathalpha" "\\mbfeta" 120520 "𝛈")
    ("mathalpha" "\\mbff" 119839 "𝐟")
    ("mathalpha" "\\mbffrakA" 120172 "𝕬")
    ("mathalpha" "\\mbffrakB" 120173 "𝕭")
    ("mathalpha" "\\mbffrakC" 120174 "𝕮")
    ("mathalpha" "\\mbffrakD" 120175 "𝕯")
    ("mathalpha" "\\mbffrakE" 120176 "𝕰")
    ("mathalpha" "\\mbffrakF" 120177 "𝕱")
    ("mathalpha" "\\mbffrakG" 120178 "𝕲")
    ("mathalpha" "\\mbffrakH" 120179 "𝕳")
    ("mathalpha" "\\mbffrakI" 120180 "𝕴")
    ("mathalpha" "\\mbffrakJ" 120181 "𝕵")
    ("mathalpha" "\\mbffrakK" 120182 "𝕶")
    ("mathalpha" "\\mbffrakL" 120183 "𝕷")
    ("mathalpha" "\\mbffrakM" 120184 "𝕸")
    ("mathalpha" "\\mbffrakN" 120185 "𝕹")
    ("mathalpha" "\\mbffrakO" 120186 "𝕺")
    ("mathalpha" "\\mbffrakP" 120187 "𝕻")
    ("mathalpha" "\\mbffrakQ" 120188 "𝕼")
    ("mathalpha" "\\mbffrakR" 120189 "𝕽")
    ("mathalpha" "\\mbffrakS" 120190 "𝕾")
    ("mathalpha" "\\mbffrakT" 120191 "𝕿")
    ("mathalpha" "\\mbffrakU" 120192 "𝖀")
    ("mathalpha" "\\mbffrakV" 120193 "𝖁")
    ("mathalpha" "\\mbffrakW" 120194 "𝖂")
    ("mathalpha" "\\mbffrakX" 120195 "𝖃")
    ("mathalpha" "\\mbffrakY" 120196 "𝖄")
    ("mathalpha" "\\mbffrakZ" 120197 "𝖅")
    ("mathalpha" "\\mbffraka" 120198 "𝖆")
    ("mathalpha" "\\mbffrakb" 120199 "𝖇")
    ("mathalpha" "\\mbffrakc" 120200 "𝖈")
    ("mathalpha" "\\mbffrakd" 120201 "𝖉")
    ("mathalpha" "\\mbffrake" 120202 "𝖊")
    ("mathalpha" "\\mbffrakf" 120203 "𝖋")
    ("mathalpha" "\\mbffrakg" 120204 "𝖌")
    ("mathalpha" "\\mbffrakh" 120205 "𝖍")
    ("mathalpha" "\\mbffraki" 120206 "𝖎")
    ("mathalpha" "\\mbffrakj" 120207 "𝖏")
    ("mathalpha" "\\mbffrakk" 120208 "𝖐")
    ("mathalpha" "\\mbffrakl" 120209 "𝖑")
    ("mathalpha" "\\mbffrakm" 120210 "𝖒")
    ("mathalpha" "\\mbffrakn" 120211 "𝖓")
    ("mathalpha" "\\mbffrako" 120212 "𝖔")
    ("mathalpha" "\\mbffrakp" 120213 "𝖕")
    ("mathalpha" "\\mbffrakq" 120214 "𝖖")
    ("mathalpha" "\\mbffrakr" 120215 "𝖗")
    ("mathalpha" "\\mbffraks" 120216 "𝖘")
    ("mathalpha" "\\mbffrakt" 120217 "𝖙")
    ("mathalpha" "\\mbffraku" 120218 "𝖚")
    ("mathalpha" "\\mbffrakv" 120219 "𝖛")
    ("mathalpha" "\\mbffrakw" 120220 "𝖜")
    ("mathalpha" "\\mbffrakx" 120221 "𝖝")
    ("mathalpha" "\\mbffraky" 120222 "𝖞")
    ("mathalpha" "\\mbffrakz" 120223 "𝖟")
    ("mathalpha" "\\mbfg" 119840 "𝐠")
    ("mathalpha" "\\mbfgamma" 120516 "𝛄")
    ("mathalpha" "\\mbfh" 119841 "𝐡")
    ("mathalpha" "\\mbfi" 119842 "𝐢")
    ("mathalpha" "\\mbfiota" 120522 "𝛊")
    ("mathalpha" "\\mbfitA" 119912 "𝑨")
    ("mathalpha" "\\mbfitAlpha" 120604 "𝜜")
    ("mathalpha" "\\mbfitB" 119913 "𝑩")
    ("mathalpha" "\\mbfitBeta" 120605 "𝜝")
    ("mathalpha" "\\mbfitC" 119914 "𝑪")
    ("mathalpha" "\\mbfitChi" 120626 "𝜲")
    ("mathalpha" "\\mbfitD" 119915 "𝑫")
    ("mathalpha" "\\mbfitDelta" 120607 "𝜟")
    ("mathalpha" "\\mbfitE" 119916 "𝑬")
    ("mathalpha" "\\mbfitEpsilon" 120608 "𝜠")
    ("mathalpha" "\\mbfitEta" 120610 "𝜢")
    ("mathalpha" "\\mbfitF" 119917 "𝑭")
    ("mathalpha" "\\mbfitG" 119918 "𝑮")
    ("mathalpha" "\\mbfitGamma" 120606 "𝜞")
    ("mathalpha" "\\mbfitH" 119919 "𝑯")
    ("mathalpha" "\\mbfitI" 119920 "𝑰")
    ("mathalpha" "\\mbfitIota" 120612 "𝜤")
    ("mathalpha" "\\mbfitJ" 119921 "𝑱")
    ("mathalpha" "\\mbfitK" 119922 "𝑲")
    ("mathalpha" "\\mbfitKappa" 120613 "𝜥")
    ("mathalpha" "\\mbfitL" 119923 "𝑳")
    ("mathalpha" "\\mbfitLambda" 120614 "𝜦")
    ("mathalpha" "\\mbfitM" 119924 "𝑴")
    ("mathalpha" "\\mbfitMu" 120615 "𝜧")
    ("mathalpha" "\\mbfitN" 119925 "𝑵")
    ("mathalpha" "\\mbfitNu" 120616 "𝜨")
    ("mathalpha" "\\mbfitO" 119926 "𝑶")
    ("mathalpha" "\\mbfitOmega" 120628 "𝜴")
    ("mathalpha" "\\mbfitOmicron" 120618 "𝜪")
    ("mathalpha" "\\mbfitP" 119927 "𝑷")
    ("mathalpha" "\\mbfitPhi" 120625 "𝜱")
    ("mathalpha" "\\mbfitPi" 120619 "𝜫")
    ("mathalpha" "\\mbfitPsi" 120627 "𝜳")
    ("mathalpha" "\\mbfitQ" 119928 "𝑸")
    ("mathalpha" "\\mbfitR" 119929 "𝑹")
    ("mathalpha" "\\mbfitRho" 120620 "𝜬")
    ("mathalpha" "\\mbfitS" 119930 "𝑺")
    ("mathalpha" "\\mbfitSigma" 120622 "𝜮")
    ("mathalpha" "\\mbfitT" 119931 "𝑻")
    ("mathalpha" "\\mbfitTau" 120623 "𝜯")
    ("mathalpha" "\\mbfitTheta" 120611 "𝜣")
    ("mathalpha" "\\mbfitU" 119932 "𝑼")
    ("mathalpha" "\\mbfitUpsilon" 120624 "𝜰")
    ("mathalpha" "\\mbfitV" 119933 "𝑽")
    ("mathalpha" "\\mbfitW" 119934 "𝑾")
    ("mathalpha" "\\mbfitX" 119935 "𝑿")
    ("mathalpha" "\\mbfitXi" 120617 "𝜩")
    ("mathalpha" "\\mbfitY" 119936 "𝒀")
    ("mathalpha" "\\mbfitZ" 119937 "𝒁")
    ("mathalpha" "\\mbfitZeta" 120609 "𝜡")
    ("mathalpha" "\\mbfita" 119938 "𝒂")
    ("mathalpha" "\\mbfitalpha" 120630 "𝜶")
    ("mathalpha" "\\mbfitb" 119939 "𝒃")
    ("mathalpha" "\\mbfitbeta" 120631 "𝜷")
    ("mathalpha" "\\mbfitc" 119940 "𝒄")
    ("mathalpha" "\\mbfitchi" 120652 "𝝌")
    ("mathalpha" "\\mbfitd" 119941 "𝒅")
    ("mathalpha" "\\mbfitdelta" 120633 "𝜹")
    ("mathalpha" "\\mbfite" 119942 "𝒆")
    ("mathalpha" "\\mbfitepsilon" 120634 "𝜺")
    ("mathalpha" "\\mbfiteta" 120636 "𝜼")
    ("mathalpha" "\\mbfitf" 119943 "𝒇")
    ("mathalpha" "\\mbfitg" 119944 "𝒈")
    ("mathalpha" "\\mbfitgamma" 120632 "𝜸")
    ("mathalpha" "\\mbfith" 119945 "𝒉")
    ("mathalpha" "\\mbfiti" 119946 "𝒊")
    ("mathalpha" "\\mbfitiota" 120638 "𝜾")
    ("mathalpha" "\\mbfitj" 119947 "𝒋")
    ("mathalpha" "\\mbfitk" 119948 "𝒌")
    ("mathalpha" "\\mbfitkappa" 120639 "𝜿")
    ("mathalpha" "\\mbfitl" 119949 "𝒍")
    ("mathalpha" "\\mbfitlambda" 120640 "𝝀")
    ("mathalpha" "\\mbfitm" 119950 "𝒎")
    ("mathalpha" "\\mbfitmu" 120641 "𝝁")
    ("mathalpha" "\\mbfitn" 119951 "𝒏")
    ("mathalpha" "\\mbfitnu" 120642 "𝝂")
    ("mathalpha" "\\mbfito" 119952 "𝒐")
    ("mathalpha" "\\mbfitomega" 120654 "𝝎")
    ("mathalpha" "\\mbfitomicron" 120644 "𝝄")
    ("mathalpha" "\\mbfitp" 119953 "𝒑")
    ("mathalpha" "\\mbfitphi" 120651 "𝝋")
    ("mathalpha" "\\mbfitpi" 120645 "𝝅")
    ("mathalpha" "\\mbfitpsi" 120653 "𝝍")
    ("mathalpha" "\\mbfitq" 119954 "𝒒")
    ("mathalpha" "\\mbfitr" 119955 "𝒓")
    ("mathalpha" "\\mbfitrho" 120646 "𝝆")
    ("mathalpha" "\\mbfits" 119956 "𝒔")
    ("mathalpha" "\\mbfitsansA" 120380 "𝘼")
    ("mathalpha" "\\mbfitsansAlpha" 120720 "𝞐")
    ("mathalpha" "\\mbfitsansB" 120381 "𝘽")
    ("mathalpha" "\\mbfitsansBeta" 120721 "𝞑")
    ("mathalpha" "\\mbfitsansC" 120382 "𝘾")
    ("mathalpha" "\\mbfitsansChi" 120742 "𝞦")
    ("mathalpha" "\\mbfitsansD" 120383 "𝘿")
    ("mathalpha" "\\mbfitsansDelta" 120723 "𝞓")
    ("mathalpha" "\\mbfitsansE" 120384 "𝙀")
    ("mathalpha" "\\mbfitsansEpsilon" 120724 "𝞔")
    ("mathalpha" "\\mbfitsansEta" 120726 "𝞖")
    ("mathalpha" "\\mbfitsansF" 120385 "𝙁")
    ("mathalpha" "\\mbfitsansG" 120386 "𝙂")
    ("mathalpha" "\\mbfitsansGamma" 120722 "𝞒")
    ("mathalpha" "\\mbfitsansH" 120387 "𝙃")
    ("mathalpha" "\\mbfitsansI" 120388 "𝙄")
    ("mathalpha" "\\mbfitsansIota" 120728 "𝞘")
    ("mathalpha" "\\mbfitsansJ" 120389 "𝙅")
    ("mathalpha" "\\mbfitsansK" 120390 "𝙆")
    ("mathalpha" "\\mbfitsansKappa" 120729 "𝞙")
    ("mathalpha" "\\mbfitsansL" 120391 "𝙇")
    ("mathalpha" "\\mbfitsansLambda" 120730 "𝞚")
    ("mathalpha" "\\mbfitsansM" 120392 "𝙈")
    ("mathalpha" "\\mbfitsansMu" 120731 "𝞛")
    ("mathalpha" "\\mbfitsansN" 120393 "𝙉")
    ("mathalpha" "\\mbfitsansNu" 120732 "𝞜")
    ("mathalpha" "\\mbfitsansO" 120394 "𝙊")
    ("mathalpha" "\\mbfitsansOmega" 120744 "𝞨")
    ("mathalpha" "\\mbfitsansOmicron" 120734 "𝞞")
    ("mathalpha" "\\mbfitsansP" 120395 "𝙋")
    ("mathalpha" "\\mbfitsansPhi" 120741 "𝞥")
    ("mathalpha" "\\mbfitsansPi" 120735 "𝞟")
    ("mathalpha" "\\mbfitsansPsi" 120743 "𝞧")
    ("mathalpha" "\\mbfitsansQ" 120396 "𝙌")
    ("mathalpha" "\\mbfitsansR" 120397 "𝙍")
    ("mathalpha" "\\mbfitsansRho" 120736 "𝞠")
    ("mathalpha" "\\mbfitsansS" 120398 "𝙎")
    ("mathalpha" "\\mbfitsansSigma" 120738 "𝞢")
    ("mathalpha" "\\mbfitsansT" 120399 "𝙏")
    ("mathalpha" "\\mbfitsansTau" 120739 "𝞣")
    ("mathalpha" "\\mbfitsansTheta" 120727 "𝞗")
    ("mathalpha" "\\mbfitsansU" 120400 "𝙐")
    ("mathalpha" "\\mbfitsansUpsilon" 120740 "𝞤")
    ("mathalpha" "\\mbfitsansV" 120401 "𝙑")
    ("mathalpha" "\\mbfitsansW" 120402 "𝙒")
    ("mathalpha" "\\mbfitsansX" 120403 "𝙓")
    ("mathalpha" "\\mbfitsansXi" 120733 "𝞝")
    ("mathalpha" "\\mbfitsansY" 120404 "𝙔")
    ("mathalpha" "\\mbfitsansZ" 120405 "𝙕")
    ("mathalpha" "\\mbfitsansZeta" 120725 "𝞕")
    ("mathalpha" "\\mbfitsansa" 120406 "𝙖")
    ("mathalpha" "\\mbfitsansalpha" 120746 "𝞪")
    ("mathalpha" "\\mbfitsansb" 120407 "𝙗")
    ("mathalpha" "\\mbfitsansbeta" 120747 "𝞫")
    ("mathalpha" "\\mbfitsansc" 120408 "𝙘")
    ("mathalpha" "\\mbfitsanschi" 120768 "𝟀")
    ("mathalpha" "\\mbfitsansd" 120409 "𝙙")
    ("mathalpha" "\\mbfitsansdelta" 120749 "𝞭")
    ("mathalpha" "\\mbfitsanse" 120410 "𝙚")
    ("mathalpha" "\\mbfitsansepsilon" 120750 "𝞮")
    ("mathalpha" "\\mbfitsanseta" 120752 "𝞰")
    ("mathalpha" "\\mbfitsansf" 120411 "𝙛")
    ("mathalpha" "\\mbfitsansg" 120412 "𝙜")
    ("mathalpha" "\\mbfitsansgamma" 120748 "𝞬")
    ("mathalpha" "\\mbfitsansh" 120413 "𝙝")
    ("mathalpha" "\\mbfitsansi" 120414 "𝙞")
    ("mathalpha" "\\mbfitsansiota" 120754 "𝞲")
    ("mathalpha" "\\mbfitsansj" 120415 "𝙟")
    ("mathalpha" "\\mbfitsansk" 120416 "𝙠")
    ("mathalpha" "\\mbfitsanskappa" 120755 "𝞳")
    ("mathalpha" "\\mbfitsansl" 120417 "𝙡")
    ("mathalpha" "\\mbfitsanslambda" 120756 "𝞴")
    ("mathalpha" "\\mbfitsansm" 120418 "𝙢")
    ("mathalpha" "\\mbfitsansmu" 120757 "𝞵")
    ("mathalpha" "\\mbfitsansn" 120419 "𝙣")
    ("mathalpha" "\\mbfitsansnu" 120758 "𝞶")
    ("mathalpha" "\\mbfitsanso" 120420 "𝙤")
    ("mathalpha" "\\mbfitsansomega" 120770 "𝟂")
    ("mathalpha" "\\mbfitsansomicron" 120760 "𝞸")
    ("mathalpha" "\\mbfitsansp" 120421 "𝙥")
    ("mathalpha" "\\mbfitsansphi" 120767 "𝞿")
    ("mathalpha" "\\mbfitsanspi" 120761 "𝞹")
    ("mathalpha" "\\mbfitsanspsi" 120769 "𝟁")
    ("mathalpha" "\\mbfitsansq" 120422 "𝙦")
    ("mathalpha" "\\mbfitsansr" 120423 "𝙧")
    ("mathalpha" "\\mbfitsansrho" 120762 "𝞺")
    ("mathalpha" "\\mbfitsanss" 120424 "𝙨")
    ("mathalpha" "\\mbfitsanssigma" 120764 "𝞼")
    ("mathalpha" "\\mbfitsanst" 120425 "𝙩")
    ("mathalpha" "\\mbfitsanstau" 120765 "𝞽")
    ("mathalpha" "\\mbfitsanstheta" 120753 "𝞱")
    ("mathalpha" "\\mbfitsansu" 120426 "𝙪")
    ("mathalpha" "\\mbfitsansupsilon" 120766 "𝞾")
    ("mathalpha" "\\mbfitsansv" 120427 "𝙫")
    ("mathalpha" "\\mbfitsansvarTheta" 120737 "𝞡")
    ("mathalpha" "\\mbfitsansvarepsilon" 120772 "𝟄")
    ("mathalpha" "\\mbfitsansvarkappa" 120774 "𝟆")
    ("mathalpha" "\\mbfitsansvarphi" 120775 "𝟇")
    ("mathalpha" "\\mbfitsansvarpi" 120777 "𝟉")
    ("mathalpha" "\\mbfitsansvarrho" 120776 "𝟈")
    ("mathalpha" "\\mbfitsansvarsigma" 120763 "𝞻")
    ("mathalpha" "\\mbfitsansvartheta" 120773 "𝟅")
    ("mathalpha" "\\mbfitsansw" 120428 "𝙬")
    ("mathalpha" "\\mbfitsansx" 120429 "𝙭")
    ("mathalpha" "\\mbfitsansxi" 120759 "𝞷")
    ("mathalpha" "\\mbfitsansy" 120430 "𝙮")
    ("mathalpha" "\\mbfitsansz" 120431 "𝙯")
    ("mathalpha" "\\mbfitsanszeta" 120751 "𝞯")
    ("mathalpha" "\\mbfitsigma" 120648 "𝝈")
    ("mathalpha" "\\mbfitt" 119957 "𝒕")
    ("mathalpha" "\\mbfittau" 120649 "𝝉")
    ("mathalpha" "\\mbfittheta" 120637 "𝜽")
    ("mathalpha" "\\mbfitu" 119958 "𝒖")
    ("mathalpha" "\\mbfitupsilon" 120650 "𝝊")
    ("mathalpha" "\\mbfitv" 119959 "𝒗")
    ("mathalpha" "\\mbfitvarTheta" 120621 "𝜭")
    ("mathalpha" "\\mbfitvarepsilon" 120656 "𝝐")
    ("mathalpha" "\\mbfitvarkappa" 120658 "𝝒")
    ("mathalpha" "\\mbfitvarphi" 120659 "𝝓")
    ("mathalpha" "\\mbfitvarpi" 120661 "𝝕")
    ("mathalpha" "\\mbfitvarrho" 120660 "𝝔")
    ("mathalpha" "\\mbfitvarsigma" 120647 "𝝇")
    ("mathalpha" "\\mbfitvartheta" 120657 "𝝑")
    ("mathalpha" "\\mbfitw" 119960 "𝒘")
    ("mathalpha" "\\mbfitx" 119961 "𝒙")
    ("mathalpha" "\\mbfitxi" 120643 "𝝃")
    ("mathalpha" "\\mbfity" 119962 "𝒚")
    ("mathalpha" "\\mbfitz" 119963 "𝒛")
    ("mathalpha" "\\mbfitzeta" 120635 "𝜻")
    ("mathalpha" "\\mbfj" 119843 "𝐣")
    ("mathalpha" "\\mbfk" 119844 "𝐤")
    ("mathalpha" "\\mbfkappa" 120523 "𝛋")
    ("mathalpha" "\\mbfl" 119845 "𝐥")
    ("mathalpha" "\\mbflambda" 120524 "𝛌")
    ("mathalpha" "\\mbfm" 119846 "𝐦")
    ("mathalpha" "\\mbfmu" 120525 "𝛍")
    ("mathalpha" "\\mbfn" 119847 "𝐧")
    ("mathalpha" "\\mbfnu" 120526 "𝛎")
    ("mathalpha" "\\mbfo" 119848 "𝐨")
    ("mathalpha" "\\mbfomega" 120538 "𝛚")
    ("mathalpha" "\\mbfomicron" 120528 "𝛐")
    ("mathalpha" "\\mbfp" 119849 "𝐩")
    ("mathalpha" "\\mbfphi" 120543 "𝛟")
    ("mathalpha" "\\mbfpi" 120529 "𝛑")
    ("mathalpha" "\\mbfpsi" 120537 "𝛙")
    ("mathalpha" "\\mbfq" 119850 "𝐪")
    ("mathalpha" "\\mbfr" 119851 "𝐫")
    ("mathalpha" "\\mbfrho" 120530 "𝛒")
    ("mathalpha" "\\mbfs" 119852 "𝐬")
    ("mathalpha" "\\mbfsansA" 120276 "𝗔")
    ("mathalpha" "\\mbfsansAlpha" 120662 "𝝖")
    ("mathalpha" "\\mbfsansB" 120277 "𝗕")
    ("mathalpha" "\\mbfsansBeta" 120663 "𝝗")
    ("mathalpha" "\\mbfsansC" 120278 "𝗖")
    ("mathalpha" "\\mbfsansChi" 120684 "𝝬")
    ("mathalpha" "\\mbfsansD" 120279 "𝗗")
    ("mathalpha" "\\mbfsansDelta" 120665 "𝝙")
    ("mathalpha" "\\mbfsansE" 120280 "𝗘")
    ("mathalpha" "\\mbfsansEpsilon" 120666 "𝝚")
    ("mathalpha" "\\mbfsansEta" 120668 "𝝜")
    ("mathalpha" "\\mbfsansF" 120281 "𝗙")
    ("mathalpha" "\\mbfsansG" 120282 "𝗚")
    ("mathalpha" "\\mbfsansGamma" 120664 "𝝘")
    ("mathalpha" "\\mbfsansH" 120283 "𝗛")
    ("mathalpha" "\\mbfsansI" 120284 "𝗜")
    ("mathalpha" "\\mbfsansIota" 120670 "𝝞")
    ("mathalpha" "\\mbfsansJ" 120285 "𝗝")
    ("mathalpha" "\\mbfsansK" 120286 "𝗞")
    ("mathalpha" "\\mbfsansKappa" 120671 "𝝟")
    ("mathalpha" "\\mbfsansL" 120287 "𝗟")
    ("mathalpha" "\\mbfsansLambda" 120672 "𝝠")
    ("mathalpha" "\\mbfsansM" 120288 "𝗠")
    ("mathalpha" "\\mbfsansMu" 120673 "𝝡")
    ("mathalpha" "\\mbfsansN" 120289 "𝗡")
    ("mathalpha" "\\mbfsansNu" 120674 "𝝢")
    ("mathalpha" "\\mbfsansO" 120290 "𝗢")
    ("mathalpha" "\\mbfsansOmega" 120686 "𝝮")
    ("mathalpha" "\\mbfsansOmicron" 120676 "𝝤")
    ("mathalpha" "\\mbfsansP" 120291 "𝗣")
    ("mathalpha" "\\mbfsansPhi" 120683 "𝝫")
    ("mathalpha" "\\mbfsansPi" 120677 "𝝥")
    ("mathalpha" "\\mbfsansPsi" 120685 "𝝭")
    ("mathalpha" "\\mbfsansQ" 120292 "𝗤")
    ("mathalpha" "\\mbfsansR" 120293 "𝗥")
    ("mathalpha" "\\mbfsansRho" 120678 "𝝦")
    ("mathalpha" "\\mbfsansS" 120294 "𝗦")
    ("mathalpha" "\\mbfsansSigma" 120680 "𝝨")
    ("mathalpha" "\\mbfsansT" 120295 "𝗧")
    ("mathalpha" "\\mbfsansTau" 120681 "𝝩")
    ("mathalpha" "\\mbfsansTheta" 120669 "𝝝")
    ("mathalpha" "\\mbfsansU" 120296 "𝗨")
    ("mathalpha" "\\mbfsansUpsilon" 120682 "𝝪")
    ("mathalpha" "\\mbfsansV" 120297 "𝗩")
    ("mathalpha" "\\mbfsansW" 120298 "𝗪")
    ("mathalpha" "\\mbfsansX" 120299 "𝗫")
    ("mathalpha" "\\mbfsansXi" 120675 "𝝣")
    ("mathalpha" "\\mbfsansY" 120300 "𝗬")
    ("mathalpha" "\\mbfsansZ" 120301 "𝗭")
    ("mathalpha" "\\mbfsansZeta" 120667 "𝝛")
    ("mathalpha" "\\mbfsansa" 120302 "𝗮")
    ("mathalpha" "\\mbfsansalpha" 120688 "𝝰")
    ("mathalpha" "\\mbfsansb" 120303 "𝗯")
    ("mathalpha" "\\mbfsansbeta" 120689 "𝝱")
    ("mathalpha" "\\mbfsansc" 120304 "𝗰")
    ("mathalpha" "\\mbfsanschi" 120710 "𝞆")
    ("mathalpha" "\\mbfsansd" 120305 "𝗱")
    ("mathalpha" "\\mbfsansdelta" 120691 "𝝳")
    ("mathalpha" "\\mbfsanse" 120306 "𝗲")
    ("mathalpha" "\\mbfsansepsilon" 120692 "𝝴")
    ("mathalpha" "\\mbfsanseta" 120694 "𝝶")
    ("mathalpha" "\\mbfsansf" 120307 "𝗳")
    ("mathalpha" "\\mbfsansg" 120308 "𝗴")
    ("mathalpha" "\\mbfsansgamma" 120690 "𝝲")
    ("mathalpha" "\\mbfsansh" 120309 "𝗵")
    ("mathalpha" "\\mbfsansi" 120310 "𝗶")
    ("mathalpha" "\\mbfsansiota" 120696 "𝝸")
    ("mathalpha" "\\mbfsansj" 120311 "𝗷")
    ("mathalpha" "\\mbfsansk" 120312 "𝗸")
    ("mathalpha" "\\mbfsanskappa" 120697 "𝝹")
    ("mathalpha" "\\mbfsansl" 120313 "𝗹")
    ("mathalpha" "\\mbfsanslambda" 120698 "𝝺")
    ("mathalpha" "\\mbfsansm" 120314 "𝗺")
    ("mathalpha" "\\mbfsansmu" 120699 "𝝻")
    ("mathalpha" "\\mbfsansn" 120315 "𝗻")
    ("mathalpha" "\\mbfsansnu" 120700 "𝝼")
    ("mathalpha" "\\mbfsanso" 120316 "𝗼")
    ("mathalpha" "\\mbfsansomega" 120712 "𝞈")
    ("mathalpha" "\\mbfsansomicron" 120702 "𝝾")
    ("mathalpha" "\\mbfsansp" 120317 "𝗽")
    ("mathalpha" "\\mbfsansphi" 120709 "𝞅")
    ("mathalpha" "\\mbfsanspi" 120703 "𝝿")
    ("mathalpha" "\\mbfsanspsi" 120711 "𝞇")
    ("mathalpha" "\\mbfsansq" 120318 "𝗾")
    ("mathalpha" "\\mbfsansr" 120319 "𝗿")
    ("mathalpha" "\\mbfsansrho" 120704 "𝞀")
    ("mathalpha" "\\mbfsanss" 120320 "𝘀")
    ("mathalpha" "\\mbfsanssigma" 120706 "𝞂")
    ("mathalpha" "\\mbfsanst" 120321 "𝘁")
    ("mathalpha" "\\mbfsanstau" 120707 "𝞃")
    ("mathalpha" "\\mbfsanstheta" 120695 "𝝷")
    ("mathalpha" "\\mbfsansu" 120322 "𝘂")
    ("mathalpha" "\\mbfsansupsilon" 120708 "𝞄")
    ("mathalpha" "\\mbfsansv" 120323 "𝘃")
    ("mathalpha" "\\mbfsansvarTheta" 120679 "𝝧")
    ("mathalpha" "\\mbfsansvarepsilon" 120714 "𝞊")
    ("mathalpha" "\\mbfsansvarkappa" 120716 "𝞌")
    ("mathalpha" "\\mbfsansvarphi" 120717 "𝞍")
    ("mathalpha" "\\mbfsansvarpi" 120719 "𝞏")
    ("mathalpha" "\\mbfsansvarrho" 120718 "𝞎")
    ("mathalpha" "\\mbfsansvarsigma" 120705 "𝞁")
    ("mathalpha" "\\mbfsansvartheta" 120715 "𝞋")
    ("mathalpha" "\\mbfsansw" 120324 "𝘄")
    ("mathalpha" "\\mbfsansx" 120325 "𝘅")
    ("mathalpha" "\\mbfsansxi" 120701 "𝝽")
    ("mathalpha" "\\mbfsansy" 120326 "𝘆")
    ("mathalpha" "\\mbfsansz" 120327 "𝘇")
    ("mathalpha" "\\mbfsanszeta" 120693 "𝝵")
    ("mathalpha" "\\mbfscrA" 120016 "𝓐")
    ("mathalpha" "\\mbfscrB" 120017 "𝓑")
    ("mathalpha" "\\mbfscrC" 120018 "𝓒")
    ("mathalpha" "\\mbfscrD" 120019 "𝓓")
    ("mathalpha" "\\mbfscrE" 120020 "𝓔")
    ("mathalpha" "\\mbfscrF" 120021 "𝓕")
    ("mathalpha" "\\mbfscrG" 120022 "𝓖")
    ("mathalpha" "\\mbfscrH" 120023 "𝓗")
    ("mathalpha" "\\mbfscrI" 120024 "𝓘")
    ("mathalpha" "\\mbfscrJ" 120025 "𝓙")
    ("mathalpha" "\\mbfscrK" 120026 "𝓚")
    ("mathalpha" "\\mbfscrL" 120027 "𝓛")
    ("mathalpha" "\\mbfscrM" 120028 "𝓜")
    ("mathalpha" "\\mbfscrN" 120029 "𝓝")
    ("mathalpha" "\\mbfscrO" 120030 "𝓞")
    ("mathalpha" "\\mbfscrP" 120031 "𝓟")
    ("mathalpha" "\\mbfscrQ" 120032 "𝓠")
    ("mathalpha" "\\mbfscrR" 120033 "𝓡")
    ("mathalpha" "\\mbfscrS" 120034 "𝓢")
    ("mathalpha" "\\mbfscrT" 120035 "𝓣")
    ("mathalpha" "\\mbfscrU" 120036 "𝓤")
    ("mathalpha" "\\mbfscrV" 120037 "𝓥")
    ("mathalpha" "\\mbfscrW" 120038 "𝓦")
    ("mathalpha" "\\mbfscrX" 120039 "𝓧")
    ("mathalpha" "\\mbfscrY" 120040 "𝓨")
    ("mathalpha" "\\mbfscrZ" 120041 "𝓩")
    ("mathalpha" "\\mbfscra" 120042 "𝓪")
    ("mathalpha" "\\mbfscrb" 120043 "𝓫")
    ("mathalpha" "\\mbfscrc" 120044 "𝓬")
    ("mathalpha" "\\mbfscrd" 120045 "𝓭")
    ("mathalpha" "\\mbfscre" 120046 "𝓮")
    ("mathalpha" "\\mbfscrf" 120047 "𝓯")
    ("mathalpha" "\\mbfscrg" 120048 "𝓰")
    ("mathalpha" "\\mbfscrh" 120049 "𝓱")
    ("mathalpha" "\\mbfscri" 120050 "𝓲")
    ("mathalpha" "\\mbfscrj" 120051 "𝓳")
    ("mathalpha" "\\mbfscrk" 120052 "𝓴")
    ("mathalpha" "\\mbfscrl" 120053 "𝓵")
    ("mathalpha" "\\mbfscrm" 120054 "𝓶")
    ("mathalpha" "\\mbfscrn" 120055 "𝓷")
    ("mathalpha" "\\mbfscro" 120056 "𝓸")
    ("mathalpha" "\\mbfscrp" 120057 "𝓹")
    ("mathalpha" "\\mbfscrq" 120058 "𝓺")
    ("mathalpha" "\\mbfscrr" 120059 "𝓻")
    ("mathalpha" "\\mbfscrs" 120060 "𝓼")
    ("mathalpha" "\\mbfscrt" 120061 "𝓽")
    ("mathalpha" "\\mbfscru" 120062 "𝓾")
    ("mathalpha" "\\mbfscrv" 120063 "𝓿")
    ("mathalpha" "\\mbfscrw" 120064 "𝔀")
    ("mathalpha" "\\mbfscrx" 120065 "𝔁")
    ("mathalpha" "\\mbfscry" 120066 "𝔂")
    ("mathalpha" "\\mbfscrz" 120067 "𝔃")
    ("mathalpha" "\\mbfsigma" 120532 "𝛔")
    ("mathalpha" "\\mbft" 119853 "𝐭")
    ("mathalpha" "\\mbftau" 120533 "𝛕")
    ("mathalpha" "\\mbftheta" 120521 "𝛉")
    ("mathalpha" "\\mbfu" 119854 "𝐮")
    ("mathalpha" "\\mbfupsilon" 120534 "𝛖")
    ("mathalpha" "\\mbfv" 119855 "𝐯")
    ("mathalpha" "\\mbfvarTheta" 120505 "𝚹")
    ("mathalpha" "\\mbfvarepsilon" 120540 "𝛜")
    ("mathalpha" "\\mbfvarkappa" 120542 "𝛞")
    ("mathalpha" "\\mbfvarphi" 120535 "𝛗")
    ("mathalpha" "\\mbfvarpi" 120545 "𝛡")
    ("mathalpha" "\\mbfvarrho" 120544 "𝛠")
    ("mathalpha" "\\mbfvarsigma" 120531 "𝛓")
    ("mathalpha" "\\mbfvartheta" 120541 "𝛝")
    ("mathalpha" "\\mbfw" 119856 "𝐰")
    ("mathalpha" "\\mbfx" 119857 "𝐱")
    ("mathalpha" "\\mbfxi" 120527 "𝛏")
    ("mathalpha" "\\mbfy" 119858 "𝐲")
    ("mathalpha" "\\mbfz" 119859 "𝐳")
    ("mathalpha" "\\mbfzeta" 120519 "𝛇")
    ("mathalpha" "\\mfrakA" 120068 "𝔄")
    ("mathalpha" "\\mfrakB" 120069 "𝔅")
    ("mathalpha" "\\mfrakC" 8493 "ℭ")
    ("mathalpha" "\\mfrakD" 120071 "𝔇")
    ("mathalpha" "\\mfrakE" 120072 "𝔈")
    ("mathalpha" "\\mfrakF" 120073 "𝔉")
    ("mathalpha" "\\mfrakG" 120074 "𝔊")
    ("mathalpha" "\\mfrakH" 8460 "ℌ")
    ("mathalpha" "\\mfrakJ" 120077 "𝔍")
    ("mathalpha" "\\mfrakK" 120078 "𝔎")
    ("mathalpha" "\\mfrakL" 120079 "𝔏")
    ("mathalpha" "\\mfrakM" 120080 "𝔐")
    ("mathalpha" "\\mfrakN" 120081 "𝔑")
    ("mathalpha" "\\mfrakO" 120082 "𝔒")
    ("mathalpha" "\\mfrakP" 120083 "𝔓")
    ("mathalpha" "\\mfrakQ" 120084 "𝔔")
    ("mathalpha" "\\mfrakS" 120086 "𝔖")
    ("mathalpha" "\\mfrakT" 120087 "𝔗")
    ("mathalpha" "\\mfrakU" 120088 "𝔘")
    ("mathalpha" "\\mfrakV" 120089 "𝔙")
    ("mathalpha" "\\mfrakW" 120090 "𝔚")
    ("mathalpha" "\\mfrakX" 120091 "𝔛")
    ("mathalpha" "\\mfrakY" 120092 "𝔜")
    ("mathalpha" "\\mfrakZ" 8488 "ℨ")
    ("mathalpha" "\\mfraka" 120094 "𝔞")
    ("mathalpha" "\\mfrakb" 120095 "𝔟")
    ("mathalpha" "\\mfrakc" 120096 "𝔠")
    ("mathalpha" "\\mfrakd" 120097 "𝔡")
    ("mathalpha" "\\mfrake" 120098 "𝔢")
    ("mathalpha" "\\mfrakf" 120099 "𝔣")
    ("mathalpha" "\\mfrakg" 120100 "𝔤")
    ("mathalpha" "\\mfrakh" 120101 "𝔥")
    ("mathalpha" "\\mfraki" 120102 "𝔦")
    ("mathalpha" "\\mfrakj" 120103 "𝔧")
    ("mathalpha" "\\mfrakk" 120104 "𝔨")
    ("mathalpha" "\\mfrakl" 120105 "𝔩")
    ("mathalpha" "\\mfrakm" 120106 "𝔪")
    ("mathalpha" "\\mfrakn" 120107 "𝔫")
    ("mathalpha" "\\mfrako" 120108 "𝔬")
    ("mathalpha" "\\mfrakp" 120109 "𝔭")
    ("mathalpha" "\\mfrakq" 120110 "𝔮")
    ("mathalpha" "\\mfrakr" 120111 "𝔯")
    ("mathalpha" "\\mfraks" 120112 "𝔰")
    ("mathalpha" "\\mfrakt" 120113 "𝔱")
    ("mathalpha" "\\mfraku" 120114 "𝔲")
    ("mathalpha" "\\mfrakv" 120115 "𝔳")
    ("mathalpha" "\\mfrakw" 120116 "𝔴")
    ("mathalpha" "\\mfrakx" 120117 "𝔵")
    ("mathalpha" "\\mfraky" 120118 "𝔶")
    ("mathalpha" "\\mfrakz" 120119 "𝔷")
    ("mathalpha" "\\mitA" 119860 "𝐴")
    ("mathalpha" "\\mitAlpha" 120546 "𝛢")
    ("mathalpha" "\\mitB" 119861 "𝐵")
    ("mathalpha" "\\mitBeta" 120547 "𝛣")
    ("mathalpha" "\\mitC" 119862 "𝐶")
    ("mathalpha" "\\mitChi" 120568 "𝛸")
    ("mathalpha" "\\mitD" 119863 "𝐷")
    ("mathalpha" "\\mitDelta" 120549 "𝛥")
    ("mathalpha" "\\mitE" 119864 "𝐸")
    ("mathalpha" "\\mitEpsilon" 120550 "𝛦")
    ("mathalpha" "\\mitEta" 120552 "𝛨")
    ("mathalpha" "\\mitF" 119865 "𝐹")
    ("mathalpha" "\\mitG" 119866 "𝐺")
    ("mathalpha" "\\mitGamma" 120548 "𝛤")
    ("mathalpha" "\\mitH" 119867 "𝐻")
    ("mathalpha" "\\mitI" 119868 "𝐼")
    ("mathalpha" "\\mitIota" 120554 "𝛪")
    ("mathalpha" "\\mitJ" 119869 "𝐽")
    ("mathalpha" "\\mitK" 119870 "𝐾")
    ("mathalpha" "\\mitKappa" 120555 "𝛫")
    ("mathalpha" "\\mitL" 119871 "𝐿")
    ("mathalpha" "\\mitLambda" 120556 "𝛬")
    ("mathalpha" "\\mitM" 119872 "𝑀")
    ("mathalpha" "\\mitMu" 120557 "𝛭")
    ("mathalpha" "\\mitN" 119873 "𝑁")
    ("mathalpha" "\\mitNu" 120558 "𝛮")
    ("mathalpha" "\\mitO" 119874 "𝑂")
    ("mathalpha" "\\mitOmega" 120570 "𝛺")
    ("mathalpha" "\\mitOmicron" 120560 "𝛰")
    ("mathalpha" "\\mitP" 119875 "𝑃")
    ("mathalpha" "\\mitPhi" 120567 "𝛷")
    ("mathalpha" "\\mitPi" 120561 "𝛱")
    ("mathalpha" "\\mitPsi" 120569 "𝛹")
    ("mathalpha" "\\mitQ" 119876 "𝑄")
    ("mathalpha" "\\mitR" 119877 "𝑅")
    ("mathalpha" "\\mitRho" 120562 "𝛲")
    ("mathalpha" "\\mitS" 119878 "𝑆")
    ("mathalpha" "\\mitSigma" 120564 "𝛴")
    ("mathalpha" "\\mitT" 119879 "𝑇")
    ("mathalpha" "\\mitTau" 120565 "𝛵")
    ("mathalpha" "\\mitTheta" 120553 "𝛩")
    ("mathalpha" "\\mitU" 119880 "𝑈")
    ("mathalpha" "\\mitUpsilon" 120566 "𝛶")
    ("mathalpha" "\\mitV" 119881 "𝑉")
    ("mathalpha" "\\mitW" 119882 "𝑊")
    ("mathalpha" "\\mitX" 119883 "𝑋")
    ("mathalpha" "\\mitXi" 120559 "𝛯")
    ("mathalpha" "\\mitY" 119884 "𝑌")
    ("mathalpha" "\\mitZ" 119885 "𝑍")
    ("mathalpha" "\\mitZeta" 120551 "𝛧")
    ("mathalpha" "\\mita" 119886 "𝑎")
    ("mathalpha" "\\mitalpha" 120572 "𝛼")
    ("mathalpha" "\\mitb" 119887 "𝑏")
    ("mathalpha" "\\mitbeta" 120573 "𝛽")
    ("mathalpha" "\\mitc" 119888 "𝑐")
    ("mathalpha" "\\mitchi" 120594 "𝜒")
    ("mathalpha" "\\mitd" 119889 "𝑑")
    ("mathalpha" "\\mitdelta" 120575 "𝛿")
    ("mathalpha" "\\mite" 119890 "𝑒")
    ("mathalpha" "\\mitepsilon" 120576 "𝜀")
    ("mathalpha" "\\miteta" 120578 "𝜂")
    ("mathalpha" "\\mitf" 119891 "𝑓")
    ("mathalpha" "\\mitg" 119892 "𝑔")
    ("mathalpha" "\\mitgamma" 120574 "𝛾")
    ("mathalpha" "\\miti" 119894 "𝑖")
    ("mathalpha" "\\mitiota" 120580 "𝜄")
    ("mathalpha" "\\mitj" 119895 "𝑗")
    ("mathalpha" "\\mitk" 119896 "𝑘")
    ("mathalpha" "\\mitkappa" 120581 "𝜅")
    ("mathalpha" "\\mitl" 119897 "𝑙")
    ("mathalpha" "\\mitlambda" 120582 "𝜆")
    ("mathalpha" "\\mitm" 119898 "𝑚")
    ("mathalpha" "\\mitmu" 120583 "𝜇")
    ("mathalpha" "\\mitn" 119899 "𝑛")
    ("mathalpha" "\\mitnu" 120584 "𝜈")
    ("mathalpha" "\\mito" 119900 "𝑜")
    ("mathalpha" "\\mitomega" 120596 "𝜔")
    ("mathalpha" "\\mitomicron" 120586 "𝜊")
    ("mathalpha" "\\mitp" 119901 "𝑝")
    ("mathalpha" "\\mitphi" 120593 "𝜑")
    ("mathalpha" "\\mitpi" 120587 "𝜋")
    ("mathalpha" "\\mitpsi" 120595 "𝜓")
    ("mathalpha" "\\mitq" 119902 "𝑞")
    ("mathalpha" "\\mitr" 119903 "𝑟")
    ("mathalpha" "\\mitrho" 120588 "𝜌")
    ("mathalpha" "\\mits" 119904 "𝑠")
    ("mathalpha" "\\mitsansA" 120328 "𝘈")
    ("mathalpha" "\\mitsansB" 120329 "𝘉")
    ("mathalpha" "\\mitsansC" 120330 "𝘊")
    ("mathalpha" "\\mitsansD" 120331 "𝘋")
    ("mathalpha" "\\mitsansE" 120332 "𝘌")
    ("mathalpha" "\\mitsansF" 120333 "𝘍")
    ("mathalpha" "\\mitsansG" 120334 "𝘎")
    ("mathalpha" "\\mitsansH" 120335 "𝘏")
    ("mathalpha" "\\mitsansI" 120336 "𝘐")
    ("mathalpha" "\\mitsansJ" 120337 "𝘑")
    ("mathalpha" "\\mitsansK" 120338 "𝘒")
    ("mathalpha" "\\mitsansL" 120339 "𝘓")
    ("mathalpha" "\\mitsansM" 120340 "𝘔")
    ("mathalpha" "\\mitsansN" 120341 "𝘕")
    ("mathalpha" "\\mitsansO" 120342 "𝘖")
    ("mathalpha" "\\mitsansP" 120343 "𝘗")
    ("mathalpha" "\\mitsansQ" 120344 "𝘘")
    ("mathalpha" "\\mitsansR" 120345 "𝘙")
    ("mathalpha" "\\mitsansS" 120346 "𝘚")
    ("mathalpha" "\\mitsansT" 120347 "𝘛")
    ("mathalpha" "\\mitsansU" 120348 "𝘜")
    ("mathalpha" "\\mitsansV" 120349 "𝘝")
    ("mathalpha" "\\mitsansW" 120350 "𝘞")
    ("mathalpha" "\\mitsansX" 120351 "𝘟")
    ("mathalpha" "\\mitsansY" 120352 "𝘠")
    ("mathalpha" "\\mitsansZ" 120353 "𝘡")
    ("mathalpha" "\\mitsansa" 120354 "𝘢")
    ("mathalpha" "\\mitsansb" 120355 "𝘣")
    ("mathalpha" "\\mitsansc" 120356 "𝘤")
    ("mathalpha" "\\mitsansd" 120357 "𝘥")
    ("mathalpha" "\\mitsanse" 120358 "𝘦")
    ("mathalpha" "\\mitsansf" 120359 "𝘧")
    ("mathalpha" "\\mitsansg" 120360 "𝘨")
    ("mathalpha" "\\mitsansh" 120361 "𝘩")
    ("mathalpha" "\\mitsansi" 120362 "𝘪")
    ("mathalpha" "\\mitsansj" 120363 "𝘫")
    ("mathalpha" "\\mitsansk" 120364 "𝘬")
    ("mathalpha" "\\mitsansl" 120365 "𝘭")
    ("mathalpha" "\\mitsansm" 120366 "𝘮")
    ("mathalpha" "\\mitsansn" 120367 "𝘯")
    ("mathalpha" "\\mitsanso" 120368 "𝘰")
    ("mathalpha" "\\mitsansp" 120369 "𝘱")
    ("mathalpha" "\\mitsansq" 120370 "𝘲")
    ("mathalpha" "\\mitsansr" 120371 "𝘳")
    ("mathalpha" "\\mitsanss" 120372 "𝘴")
    ("mathalpha" "\\mitsanst" 120373 "𝘵")
    ("mathalpha" "\\mitsansu" 120374 "𝘶")
    ("mathalpha" "\\mitsansv" 120375 "𝘷")
    ("mathalpha" "\\mitsansw" 120376 "𝘸")
    ("mathalpha" "\\mitsansx" 120377 "𝘹")
    ("mathalpha" "\\mitsansy" 120378 "𝘺")
    ("mathalpha" "\\mitsansz" 120379 "𝘻")
    ("mathalpha" "\\mitsigma" 120590 "𝜎")
    ("mathalpha" "\\mitt" 119905 "𝑡")
    ("mathalpha" "\\mittau" 120591 "𝜏")
    ("mathalpha" "\\mittheta" 120579 "𝜃")
    ("mathalpha" "\\mitu" 119906 "𝑢")
    ("mathalpha" "\\mitupsilon" 120592 "𝜐")
    ("mathalpha" "\\mitv" 119907 "𝑣")
    ("mathalpha" "\\mitvarTheta" 120563 "𝛳")
    ("mathalpha" "\\mitvarepsilon" 120598 "𝜖")
    ("mathalpha" "\\mitvarkappa" 120600 "𝜘")
    ("mathalpha" "\\mitvarphi" 120601 "𝜙")
    ("mathalpha" "\\mitvarpi" 120603 "𝜛")
    ("mathalpha" "\\mitvarrho" 120602 "𝜚")
    ("mathalpha" "\\mitvarsigma" 120589 "𝜍")
    ("mathalpha" "\\mitvartheta" 120599 "𝜗")
    ("mathalpha" "\\mitw" 119908 "𝑤")
    ("mathalpha" "\\mitx" 119909 "𝑥")
    ("mathalpha" "\\mitxi" 120585 "𝜉")
    ("mathalpha" "\\mity" 119910 "𝑦")
    ("mathalpha" "\\mitz" 119911 "𝑧")
    ("mathalpha" "\\mitzeta" 120577 "𝜁")
    ("mathalpha" "\\msansA" 120224 "𝖠")
    ("mathalpha" "\\msansB" 120225 "𝖡")
    ("mathalpha" "\\msansC" 120226 "𝖢")
    ("mathalpha" "\\msansD" 120227 "𝖣")
    ("mathalpha" "\\msansE" 120228 "𝖤")
    ("mathalpha" "\\msansF" 120229 "𝖥")
    ("mathalpha" "\\msansG" 120230 "𝖦")
    ("mathalpha" "\\msansH" 120231 "𝖧")
    ("mathalpha" "\\msansI" 120232 "𝖨")
    ("mathalpha" "\\msansJ" 120233 "𝖩")
    ("mathalpha" "\\msansK" 120234 "𝖪")
    ("mathalpha" "\\msansL" 120235 "𝖫")
    ("mathalpha" "\\msansM" 120236 "𝖬")
    ("mathalpha" "\\msansN" 120237 "𝖭")
    ("mathalpha" "\\msansO" 120238 "𝖮")
    ("mathalpha" "\\msansP" 120239 "𝖯")
    ("mathalpha" "\\msansQ" 120240 "𝖰")
    ("mathalpha" "\\msansR" 120241 "𝖱")
    ("mathalpha" "\\msansS" 120242 "𝖲")
    ("mathalpha" "\\msansT" 120243 "𝖳")
    ("mathalpha" "\\msansU" 120244 "𝖴")
    ("mathalpha" "\\msansV" 120245 "𝖵")
    ("mathalpha" "\\msansW" 120246 "𝖶")
    ("mathalpha" "\\msansX" 120247 "𝖷")
    ("mathalpha" "\\msansY" 120248 "𝖸")
    ("mathalpha" "\\msansZ" 120249 "𝖹")
    ("mathalpha" "\\msansa" 120250 "𝖺")
    ("mathalpha" "\\msansb" 120251 "𝖻")
    ("mathalpha" "\\msansc" 120252 "𝖼")
    ("mathalpha" "\\msansd" 120253 "𝖽")
    ("mathalpha" "\\msanse" 120254 "𝖾")
    ("mathalpha" "\\msansf" 120255 "𝖿")
    ("mathalpha" "\\msansg" 120256 "𝗀")
    ("mathalpha" "\\msansh" 120257 "𝗁")
    ("mathalpha" "\\msansi" 120258 "𝗂")
    ("mathalpha" "\\msansj" 120259 "𝗃")
    ("mathalpha" "\\msansk" 120260 "𝗄")
    ("mathalpha" "\\msansl" 120261 "𝗅")
    ("mathalpha" "\\msansm" 120262 "𝗆")
    ("mathalpha" "\\msansn" 120263 "𝗇")
    ("mathalpha" "\\msanso" 120264 "𝗈")
    ("mathalpha" "\\msansp" 120265 "𝗉")
    ("mathalpha" "\\msansq" 120266 "𝗊")
    ("mathalpha" "\\msansr" 120267 "𝗋")
    ("mathalpha" "\\msanss" 120268 "𝗌")
    ("mathalpha" "\\msanst" 120269 "𝗍")
    ("mathalpha" "\\msansu" 120270 "𝗎")
    ("mathalpha" "\\msansv" 120271 "𝗏")
    ("mathalpha" "\\msansw" 120272 "𝗐")
    ("mathalpha" "\\msansx" 120273 "𝗑")
    ("mathalpha" "\\msansy" 120274 "𝗒")
    ("mathalpha" "\\msansz" 120275 "𝗓")
    ("mathalpha" "\\mscrA" 119964 "𝒜")
    ("mathalpha" "\\mscrB" 8492 "ℬ")
    ("mathalpha" "\\mscrC" 119966 "𝒞")
    ("mathalpha" "\\mscrD" 119967 "𝒟")
    ("mathalpha" "\\mscrE" 8496 "ℰ")
    ("mathalpha" "\\mscrF" 8497 "ℱ")
    ("mathalpha" "\\mscrG" 119970 "𝒢")
    ("mathalpha" "\\mscrH" 8459 "ℋ")
    ("mathalpha" "\\mscrI" 8464 "ℐ")
    ("mathalpha" "\\mscrJ" 119973 "𝒥")
    ("mathalpha" "\\mscrK" 119974 "𝒦")
    ("mathalpha" "\\mscrL" 8466 "ℒ")
    ("mathalpha" "\\mscrM" 8499 "ℳ")
    ("mathalpha" "\\mscrN" 119977 "𝒩")
    ("mathalpha" "\\mscrO" 119978 "𝒪")
    ("mathalpha" "\\mscrP" 119979 "𝒫")
    ("mathalpha" "\\mscrQ" 119980 "𝒬")
    ("mathalpha" "\\mscrR" 8475 "ℛ")
    ("mathalpha" "\\mscrS" 119982 "𝒮")
    ("mathalpha" "\\mscrT" 119983 "𝒯")
    ("mathalpha" "\\mscrU" 119984 "𝒰")
    ("mathalpha" "\\mscrV" 119985 "𝒱")
    ("mathalpha" "\\mscrW" 119986 "𝒲")
    ("mathalpha" "\\mscrX" 119987 "𝒳")
    ("mathalpha" "\\mscrY" 119988 "𝒴")
    ("mathalpha" "\\mscrZ" 119989 "𝒵")
    ("mathalpha" "\\mscra" 119990 "𝒶")
    ("mathalpha" "\\mscrb" 119991 "𝒷")
    ("mathalpha" "\\mscrc" 119992 "𝒸")
    ("mathalpha" "\\mscrd" 119993 "𝒹")
    ("mathalpha" "\\mscre" 8495 "ℯ")
    ("mathalpha" "\\mscrf" 119995 "𝒻")
    ("mathalpha" "\\mscrg" 8458 "ℊ")
    ("mathalpha" "\\mscrh" 119997 "𝒽")
    ("mathalpha" "\\mscri" 119998 "𝒾")
    ("mathalpha" "\\mscrj" 119999 "𝒿")
    ("mathalpha" "\\mscrk" 120000 "𝓀")
    ("mathalpha" "\\mscrl" 120001 "𝓁")
    ("mathalpha" "\\mscrm" 120002 "𝓂")
    ("mathalpha" "\\mscrn" 120003 "𝓃")
    ("mathalpha" "\\mscro" 8500 "ℴ")
    ("mathalpha" "\\mscrp" 120005 "𝓅")
    ("mathalpha" "\\mscrq" 120006 "𝓆")
    ("mathalpha" "\\mscrr" 120007 "𝓇")
    ("mathalpha" "\\mscrs" 120008 "𝓈")
    ("mathalpha" "\\mscrt" 120009 "𝓉")
    ("mathalpha" "\\mscru" 120010 "𝓊")
    ("mathalpha" "\\mscrv" 120011 "𝓋")
    ("mathalpha" "\\mscrw" 120012 "𝓌")
    ("mathalpha" "\\mscrx" 120013 "𝓍")
    ("mathalpha" "\\mscry" 120014 "𝓎")
    ("mathalpha" "\\mscrz" 120015 "𝓏")
    ("mathalpha" "\\mttA" 120432 "𝙰")
    ("mathalpha" "\\mttB" 120433 "𝙱")
    ("mathalpha" "\\mttC" 120434 "𝙲")
    ("mathalpha" "\\mttD" 120435 "𝙳")
    ("mathalpha" "\\mttE" 120436 "𝙴")
    ("mathalpha" "\\mttF" 120437 "𝙵")
    ("mathalpha" "\\mttG" 120438 "𝙶")
    ("mathalpha" "\\mttH" 120439 "𝙷")
    ("mathalpha" "\\mttI" 120440 "𝙸")
    ("mathalpha" "\\mttJ" 120441 "𝙹")
    ("mathalpha" "\\mttK" 120442 "𝙺")
    ("mathalpha" "\\mttL" 120443 "𝙻")
    ("mathalpha" "\\mttM" 120444 "𝙼")
    ("mathalpha" "\\mttN" 120445 "𝙽")
    ("mathalpha" "\\mttO" 120446 "𝙾")
    ("mathalpha" "\\mttP" 120447 "𝙿")
    ("mathalpha" "\\mttQ" 120448 "𝚀")
    ("mathalpha" "\\mttR" 120449 "𝚁")
    ("mathalpha" "\\mttS" 120450 "𝚂")
    ("mathalpha" "\\mttT" 120451 "𝚃")
    ("mathalpha" "\\mttU" 120452 "𝚄")
    ("mathalpha" "\\mttV" 120453 "𝚅")
    ("mathalpha" "\\mttW" 120454 "𝚆")
    ("mathalpha" "\\mttX" 120455 "𝚇")
    ("mathalpha" "\\mttY" 120456 "𝚈")
    ("mathalpha" "\\mttZ" 120457 "𝚉")
    ("mathalpha" "\\mtta" 120458 "𝚊")
    ("mathalpha" "\\mttb" 120459 "𝚋")
    ("mathalpha" "\\mttc" 120460 "𝚌")
    ("mathalpha" "\\mttd" 120461 "𝚍")
    ("mathalpha" "\\mtte" 120462 "𝚎")
    ("mathalpha" "\\mttf" 120463 "𝚏")
    ("mathalpha" "\\mttg" 120464 "𝚐")
    ("mathalpha" "\\mtth" 120465 "𝚑")
    ("mathalpha" "\\mtti" 120466 "𝚒")
    ("mathalpha" "\\mttj" 120467 "𝚓")
    ("mathalpha" "\\mttk" 120468 "𝚔")
    ("mathalpha" "\\mttl" 120469 "𝚕")
    ("mathalpha" "\\mttm" 120470 "𝚖")
    ("mathalpha" "\\mttn" 120471 "𝚗")
    ("mathalpha" "\\mtto" 120472 "𝚘")
    ("mathalpha" "\\mttp" 120473 "𝚙")
    ("mathalpha" "\\mttq" 120474 "𝚚")
    ("mathalpha" "\\mttr" 120475 "𝚛")
    ("mathalpha" "\\mtts" 120476 "𝚜")
    ("mathalpha" "\\mttt" 120477 "𝚝")
    ("mathalpha" "\\mttu" 120478 "𝚞")
    ("mathalpha" "\\mttv" 120479 "𝚟")
    ("mathalpha" "\\mttw" 120480 "𝚠")
    ("mathalpha" "\\mttx" 120481 "𝚡")
    ("mathalpha" "\\mtty" 120482 "𝚢")
    ("mathalpha" "\\mttz" 120483 "𝚣")
    ("mathalpha" "\\mu" 956 "μ")
    ("mathalpha" "\\mu" 120583 "𝜇")
    ("mathalpha" "\\nu" 957 "ν")
    ("mathalpha" "\\nu" 120584 "𝜈")
    ("mathalpha" "\\omega" 969 "ω")
    ("mathalpha" "\\omega" 120596 "𝜔")
    ("mathalpha" "\\period" 46 ".")
    ("mathalpha" "\\phi" 981 "ϕ")
    ("mathalpha" "\\phi" 120601 "𝜙")
    ("mathalpha" "\\pi" 960 "π")
    ("mathalpha" "\\pi" 120587 "𝜋")
    ("mathalpha" "\\psi" 968 "ψ")
    ("mathalpha" "\\psi" 120595 "𝜓")
    ("mathalpha" "\\rho" 961 "ρ")
    ("mathalpha" "\\rho" 120588 "𝜌")
    ("mathalpha" "\\sampi" 993 "ϡ")
    ("mathalpha" "\\sigma" 963 "σ")
    ("mathalpha" "\\sigma" 120590 "𝜎")
    ("mathalpha" "\\stigma" 987 "ϛ")
    ("mathalpha" "\\tau" 964 "τ")
    ("mathalpha" "\\tau" 120591 "𝜏")
    ("mathalpha" "\\tcohm" 8486 "Ω")
    ("mathalpha" "\\theta" 952 "θ")
    ("mathalpha" "\\theta" 120579 "𝜃")
    ("mathalpha" "\\turnediota" 8489 "℩")
    ("mathalpha" "\\upAlpha" 913 "Α")
    ("mathalpha" "\\upBeta" 914 "Β")
    ("mathalpha" "\\upChi" 935 "Χ")
    ("mathalpha" "\\upDelta" 916 "Δ")
    ("mathalpha" "\\upDigamma" 988 "Ϝ")
    ("mathalpha" "\\upEpsilon" 917 "Ε")
    ("mathalpha" "\\upEta" 919 "Η")
    ("mathalpha" "\\upGamma" 915 "Γ")
    ("mathalpha" "\\upIota" 921 "Ι")
    ("mathalpha" "\\upKappa" 922 "Κ")
    ("mathalpha" "\\upKoppa" 990 "Ϟ")
    ("mathalpha" "\\upLambda" 923 "Λ")
    ("mathalpha" "\\upMu" 924 "Μ")
    ("mathalpha" "\\upNu" 925 "Ν")
    ("mathalpha" "\\upOmega" 937 "Ω")
    ("mathalpha" "\\upOmicron" 927 "Ο")
    ("mathalpha" "\\upPhi" 934 "Φ")
    ("mathalpha" "\\upPi" 928 "Π")
    ("mathalpha" "\\upPsi" 936 "Ψ")
    ("mathalpha" "\\upRho" 929 "Ρ")
    ("mathalpha" "\\upSampi" 992 "Ϡ")
    ("mathalpha" "\\upSigma" 931 "Σ")
    ("mathalpha" "\\upStigma" 986 "Ϛ")
    ("mathalpha" "\\upTau" 932 "Τ")
    ("mathalpha" "\\upTheta" 920 "Θ")
    ("mathalpha" "\\upUpsilon" 933 "Υ")
    ("mathalpha" "\\upUpsilon" 978 "ϒ")
    ("mathalpha" "\\upXi" 926 "Ξ")
    ("mathalpha" "\\upZeta" 918 "Ζ")
    ("mathalpha" "\\upalpha" 945 "α")
    ("mathalpha" "\\upbeta" 946 "β")
    ("mathalpha" "\\upchi" 967 "χ")
    ("mathalpha" "\\updelta" 948 "δ")
    ("mathalpha" "\\updigamma" 989 "ϝ")
    ("mathalpha" "\\upepsilon" 949 "ε")
    ("mathalpha" "\\upeta" 951 "η")
    ("mathalpha" "\\upgamma" 947 "γ")
    ("mathalpha" "\\upiota" 953 "ι")
    ("mathalpha" "\\upkappa" 954 "κ")
    ("mathalpha" "\\upkoppa" 991 "ϟ")
    ("mathalpha" "\\uplambda" 955 "λ")
    ("mathalpha" "\\upmu" 956 "μ")
    ("mathalpha" "\\upnu" 957 "ν")
    ("mathalpha" "\\upomega" 969 "ω")
    ("mathalpha" "\\upomicron" 959 "ο")
    ("mathalpha" "\\upphi" 981 "ϕ")
    ("mathalpha" "\\uppi" 960 "π")
    ("mathalpha" "\\uppsi" 968 "ψ")
    ("mathalpha" "\\uprho" 961 "ρ")
    ("mathalpha" "\\upsampi" 993 "ϡ")
    ("mathalpha" "\\upsigma" 963 "σ")
    ("mathalpha" "\\upsilon" 965 "υ")
    ("mathalpha" "\\upsilon" 120592 "𝜐")
    ("mathalpha" "\\upstigma" 987 "ϛ")
    ("mathalpha" "\\uptau" 964 "τ")
    ("mathalpha" "\\uptheta" 952 "θ")
    ("mathalpha" "\\upupsilon" 965 "υ")
    ("mathalpha" "\\upvarTheta" 1012 "ϴ")
    ("mathalpha" "\\upvarbeta" 976 "ϐ")
    ("mathalpha" "\\upvarepsilon" 1013 "ϵ")
    ("mathalpha" "\\upvarkappa" 1008 "ϰ")
    ("mathalpha" "\\upvarphi" 966 "φ")
    ("mathalpha" "\\upvarpi" 982 "ϖ")
    ("mathalpha" "\\upvarrho" 1009 "ϱ")
    ("mathalpha" "\\upvarsigma" 962 "ς")
    ("mathalpha" "\\upvartheta" 977 "ϑ")
    ("mathalpha" "\\upxi" 958 "ξ")
    ("mathalpha" "\\upzeta" 950 "ζ")
    ("mathalpha" "\\varbeta" 976 "ϐ")
    ("mathalpha" "\\varepsilon" 949 "ε")
    ("mathalpha" "\\varepsilon" 120576 "𝜀")
    ("mathalpha" "\\varkappa" 120600 "𝜘")
    ("mathalpha" "\\varphi" 966 "φ")
    ("mathalpha" "\\varphi" 120593 "𝜑")
    ("mathalpha" "\\varpi" 982 "ϖ")
    ("mathalpha" "\\varpi" 120603 "𝜛")
    ("mathalpha" "\\varrho" 1009 "ϱ")
    ("mathalpha" "\\varrho" 120602 "𝜚")
    ("mathalpha" "\\varsigma" 962 "ς")
    ("mathalpha" "\\varsigma" 120589 "𝜍")
    ("mathalpha" "\\vartheta" 977 "ϑ")
    ("mathalpha" "\\vartheta" 120599 "𝜗")
    ("mathalpha" "\\wp" 8472 "℘")
    ("mathalpha" "\\xi" 958 "ξ")
    ("mathalpha" "\\xi" 120585 "𝜉")
    ("mathalpha" "\\zeta" 950 "ζ")
    ("mathalpha" "\\zeta" 120577 "𝜁")
    ("mathbin" "\\Cap" 8914 "⋒")
    ("mathbin" "\\Circle" 9675 "○")
    ("mathbin" "\\Cup" 8915 "⋓")
    ("mathbin" "\\LHD" 9664 "◀")
    ("mathbin" "\\Otimes" 10807 "⨷")
    ("mathbin" "\\RHD" 9654 "▶")
    ("mathbin" "\\Sqcap" 10830 "⩎")
    ("mathbin" "\\Sqcup" 10831 "⩏")
    ("mathbin" "\\Vee" 10836 "⩔")
    ("mathbin" "\\Wedge" 10835 "⩓")
    ("mathbin" "\\amalg" 10815 "⨿")
    ("mathbin" "\\ast" 8727 "∗")
    ("mathbin" "\\barcap" 10819 "⩃")
    ("mathbin" "\\barcup" 10818 "⩂")
    ("mathbin" "\\barvee" 8893 "⊽")
    ("mathbin" "\\barwedge" 8892 "⊼")
    ("mathbin" "\\bigslopedvee" 10839 "⩗")
    ("mathbin" "\\bigslopedwedge" 10840 "⩘")
    ("mathbin" "\\bigtriangledown" 9661 "▽")
    ("mathbin" "\\bigtriangleup" 9651 "△")
    ("mathbin" "\\blackhourglass" 10711 "⧗")
    ("mathbin" "\\blacklozenge" 10731 "⧫")
    ("mathbin" "\\blacktriangle" 9652 "▴")
    ("mathbin" "\\blacktriangledown" 9662 "▾")
    ("mathbin" "\\blacktriangleleft" 9664 "◀")
    ("mathbin" "\\blacktriangleleft" 9666 "◂")
    ("mathbin" "\\blacktriangleright" 9654 "▶")
    ("mathbin" "\\blacktriangleright" 9656 "▸")
    ("mathbin" "\\blacktriangleup" 9652 "▴")
    ("mathbin" "\\boxast" 10694 "⧆")
    ("mathbin" "\\boxbar" 9707 "◫")
    ("mathbin" "\\boxbox" 10696 "⧈")
    ("mathbin" "\\boxbslash" 10693 "⧅")
    ("mathbin" "\\boxcircle" 10695 "⧇")
    ("mathbin" "\\boxdiag" 10692 "⧄")
    ("mathbin" "\\boxdot" 8865 "⊡")
    ("mathbin" "\\boxminus" 8863 "⊟")
    ("mathbin" "\\boxplus" 8862 "⊞")
    ("mathbin" "\\boxslash" 10692 "⧄")
    ("mathbin" "\\boxtimes" 8864 "⊠")
    ("mathbin" "\\btimes" 10802 "⨲")
    ("mathbin" "\\bullet" 8729 "∙")
    ("mathbin" "\\cap" 8745 "∩")
    ("mathbin" "\\capbarcup" 10825 "⩉")
    ("mathbin" "\\capdot" 10816 "⩀")
    ("mathbin" "\\capovercup" 10823 "⩇")
    ("mathbin" "\\capwedge" 10820 "⩄")
    ("mathbin" "\\cat" 8256 "⁀")
    ("mathbin" "\\cdot" 8901 "⋅")
    ("mathbin" "\\cdotp" 183 "·")
    ("mathbin" "\\circ" 8728 "∘")
    ("mathbin" "\\circledast" 8859 "⊛")
    ("mathbin" "\\circledbslash" 10680 "⦸")
    ("mathbin" "\\circledcirc" 8858 "⊚")
    ("mathbin" "\\circleddash" 8861 "⊝")
    ("mathbin" "\\circledequal" 8860 "⊜")
    ("mathbin" "\\circledgtr" 10689 "⧁")
    ("mathbin" "\\circledless" 10688 "⧀")
    ("mathbin" "\\circledparallel" 10679 "⦷")
    ("mathbin" "\\circledvert" 10678 "⦶")
    ("mathbin" "\\circlehbar" 10677 "⦵")
    ("mathbin" "\\closedvarcap" 10829 "⩍")
    ("mathbin" "\\closedvarcup" 10828 "⩌")
    ("mathbin" "\\closedvarcupsmashprod" 10832 "⩐")
    ("mathbin" "\\commaminus" 10793 "⨩")
    ("mathbin" "\\concavediamond" 10209 "⟡")
    ("mathbin" "\\concavediamondtickleft" 10210 "⟢")
    ("mathbin" "\\concavediamondtickright" 10211 "⟣")
    ("mathbin" "\\cup" 8746 "∪")
    ("mathbin" "\\cupbarcap" 10824 "⩈")
    ("mathbin" "\\cupdot" 8845 "⊍")
    ("mathbin" "\\cupleftarrow" 8844 "⊌")
    ("mathbin" "\\cupovercap" 10822 "⩆")
    ("mathbin" "\\cupvee" 10821 "⩅")
    ("mathbin" "\\curlyvee" 8910 "⋎")
    ("mathbin" "\\curlywedge" 8911 "⋏")
    ("mathbin" "\\dagger" 8224 "†")
    ("mathbin" "\\ddagger" 8225 "‡")
    ("mathbin" "\\diamond" 8900 "⋄")
    ("mathbin" "\\div" 247 "÷")
    ("mathbin" "\\divideontimes" 8903 "⋇")
    ("mathbin" "\\divslash" 8725 "∕")
    ("mathbin" "\\dotminus" 8760 "∸")
    ("mathbin" "\\dotplus" 8724 "∔")
    ("mathbin" "\\dottimes" 10800 "⨰")
    ("mathbin" "\\doublebarvee" 10850 "⩢")
    ("mathbin" "\\doublebarwedge" 10846 "⩞")
    ("mathbin" "\\doubleplus" 10746 "⧺")
    ("mathbin" "\\dsol" 10742 "⧶")
    ("mathbin" "\\dsub" 10852 "⩤")
    ("mathbin" "\\eqqplus" 10865 "⩱")
    ("mathbin" "\\fcmp" 10814 "⨾")
    ("mathbin" "\\fracslash" 8260 "⁄")
    ("mathbin" "\\hourglass" 10710 "⧖")
    ("mathbin" "\\intercal" 8890 "⊺")
    ("mathbin" "\\interleave" 10996 "⫴")
    ("mathbin" "\\intprod" 10812 "⨼")
    ("mathbin" "\\intprodr" 10813 "⨽")
    ("mathbin" "\\invamp" 8523 "⅋")
    ("mathbin" "\\invlazys" 8766 "∾")
    ("mathbin" "\\leftthreetimes" 8907 "⋋")
    ("mathbin" "\\lhd" 9665 "◁")
    ("mathbin" "\\lozengeminus" 10208 "⟠")
    ("mathbin" "\\ltimes" 8905 "⋉")
    ("mathbin" "\\mdlgblklozenge" 10731 "⧫")
    ("mathbin" "\\mdlgwhtcircle" 9675 "○")
    ("mathbin" "\\midbarvee" 10845 "⩝")
    ("mathbin" "\\midbarwedge" 10844 "⩜")
    ("mathbin" "\\minus" 8722 "−")
    ("mathbin" "\\minusdot" 10794 "⨪")
    ("mathbin" "\\minusfdots" 10795 "⨫")
    ("mathbin" "\\minusrdots" 10796 "⨬")
    ("mathbin" "\\mp" 8723 "∓")
    ("mathbin" "\\nhVvert" 10997 "⫵")
    ("mathbin" "\\obar" 9021 "⌽")
    ("mathbin" "\\obslash" 10680 "⦸")
    ("mathbin" "\\odiv" 10808 "⨸")
    ("mathbin" "\\odot" 8857 "⊙")
    ("mathbin" "\\ogreaterthan" 10689 "⧁")
    ("mathbin" "\\olessthan" 10688 "⧀")
    ("mathbin" "\\ominus" 8854 "⊖")
    ("mathbin" "\\operp" 10681 "⦹")
    ("mathbin" "\\oplus" 8853 "⊕")
    ("mathbin" "\\opluslhrim" 10797 "⨭")
    ("mathbin" "\\oplusrhrim" 10798 "⨮")
    ("mathbin" "\\oslash" 8856 "⊘")
    ("mathbin" "\\otimes" 8855 "⊗")
    ("mathbin" "\\otimeshat" 10806 "⨶")
    ("mathbin" "\\otimeslhrim" 10804 "⨴")
    ("mathbin" "\\otimesrhrim" 10805 "⨵")
    ("mathbin" "\\plus" 43 "+")
    ("mathbin" "\\plusdot" 10789 "⨥")
    ("mathbin" "\\pluseqq" 10866 "⩲")
    ("mathbin" "\\plushat" 10787 "⨣")
    ("mathbin" "\\plussim" 10790 "⨦")
    ("mathbin" "\\plussubtwo" 10791 "⨧")
    ("mathbin" "\\plustrif" 10792 "⨨")
    ("mathbin" "\\pm" 177 "±")
    ("mathbin" "\\rhd" 9655 "▷")
    ("mathbin" "\\rightthreetimes" 8908 "⋌")
    ("mathbin" "\\ringplus" 10786 "⨢")
    ("mathbin" "\\rsolbar" 10743 "⧷")
    ("mathbin" "\\rsub" 10853 "⩥")
    ("mathbin" "\\rtimes" 8906 "⋊")
    ("mathbin" "\\setminus" 10741 "⧵")
    ("mathbin" "\\shuffle" 10722 "⧢")
    ("mathbin" "\\simplus" 10788 "⨤")
    ("mathbin" "\\slash" 8725 "∕")
    ("mathbin" "\\smallblacktriangleleft" 9666 "◂")
    ("mathbin" "\\smallblacktriangleright" 9656 "▸")
    ("mathbin" "\\smallsetminus" 8726 "∖")
    ("mathbin" "\\smalltriangledown" 9663 "▿")
    ("mathbin" "\\smalltriangleleft" 9667 "◃")
    ("mathbin" "\\smalltriangleright" 9657 "▹")
    ("mathbin" "\\smalltriangleup" 9653 "▵")
    ("mathbin" "\\smashtimes" 10803 "⨳")
    ("mathbin" "\\smblkcircle" 8226 "•")
    ("mathbin" "\\smwhtdiamond" 8900 "⋄")
    ("mathbin" "\\sqcap" 8851 "⊓")
    ("mathbin" "\\sqcup" 8852 "⊔")
    ("mathbin" "\\sslash" 11005 "⫽")
    ("mathbin" "\\star" 8902 "⋆")
    ("mathbin" "\\talloblong" 11006 "⫾")
    ("mathbin" "\\threedotcolon" 10998 "⫶")
    ("mathbin" "\\tieconcat" 8256 "⁀")
    ("mathbin" "\\times" 215 "×")
    ("mathbin" "\\timesbar" 10801 "⨱")
    ("mathbin" "\\tminus" 10751 "⧿")
    ("mathbin" "\\tplus" 10750 "⧾")
    ("mathbin" "\\triangledown" 9663 "▿")
    ("mathbin" "\\triangleleft" 9665 "◁")
    ("mathbin" "\\triangleminus" 10810 "⨺")
    ("mathbin" "\\triangleplus" 10809 "⨹")
    ("mathbin" "\\triangleright" 9655 "▷")
    ("mathbin" "\\triangleserifs" 10701 "⧍")
    ("mathbin" "\\triangletimes" 10811 "⨻")
    ("mathbin" "\\tripleplus" 10747 "⧻")
    ("mathbin" "\\trslash" 11003 "⫻")
    ("mathbin" "\\twocaps" 10827 "⩋")
    ("mathbin" "\\twocups" 10826 "⩊")
    ("mathbin" "\\typecolon" 10626 "⦂")
    ("mathbin" "\\uminus" 10817 "⩁")
    ("mathbin" "\\upand" 8523 "⅋")
    ("mathbin" "\\uplus" 8846 "⊎")
    ("mathbin" "\\varbarwedge" 8965 "⌅")
    ("mathbin" "\\vardoublebarwedge" 8966 "⌆")
    ("mathbin" "\\vartriangle" 9653 "▵")
    ("mathbin" "\\varveebar" 10849 "⩡")
    ("mathbin" "\\vectimes" 10799 "⨯")
    ("mathbin" "\\vee" 8744 "∨")
    ("mathbin" "\\veebar" 8891 "⊻")
    ("mathbin" "\\veedot" 10183 "⟇")
    ("mathbin" "\\veedoublebar" 10851 "⩣")
    ("mathbin" "\\veemidvert" 10843 "⩛")
    ("mathbin" "\\veeodot" 10834 "⩒")
    ("mathbin" "\\veeonvee" 10838 "⩖")
    ("mathbin" "\\vysmblkcircle" 8729 "∙")
    ("mathbin" "\\vysmwhtcircle" 8728 "∘")
    ("mathbin" "\\wedge" 8743 "∧")
    ("mathbin" "\\wedgebar" 10847 "⩟")
    ("mathbin" "\\wedgedot" 10193 "⟑")
    ("mathbin" "\\wedgedoublebar" 10848 "⩠")
    ("mathbin" "\\wedgemidvert" 10842 "⩚")
    ("mathbin" "\\wedgeodot" 10833 "⩑")
    ("mathbin" "\\wedgeonwedge" 10837 "⩕")
    ("mathbin" "\\whitesquaretickleft" 10212 "⟤")
    ("mathbin" "\\whitesquaretickright" 10213 "⟥")
    ("mathbin" "\\wr" 8768 "≀")
    ("mathclose" "\\Rbag" 10182 "⟆")
    ("mathclose" "\\Rbrbrak" 10221 "⟭")
    ("mathclose" "\\Rbrbrak" 12313 "〙")
    ("mathclose" "\\Rparen" 10630 "⦆")
    ("mathclose" "\\Rparenless" 10646 "⦖")
    ("mathclose" "\\Rvzigzag" 10715 "⧛")
    ("mathclose" "\\lrcorner" 8991 "⌟")
    ("mathclose" "\\rAngle" 10219 "⟫")
    ("mathclose" "\\rBrace" 10628 "⦄")
    ("mathclose" "\\rBrack" 10215 "⟧")
    ("mathclose" "\\rParen" 10630 "⦆")
    ("mathclose" "\\rang" 10219 "⟫")
    ("mathclose" "\\rangle" 10217 "⟩")
    ("mathclose" "\\rangledot" 10642 "⦒")
    ("mathclose" "\\rbag" 10182 "⟆")
    ("mathclose" "\\rblkbrbrak" 10648 "⦘")
    ("mathclose" "\\rblot" 10634 "⦊")
    ("mathclose" "\\rbrace" 125 "}")
    ("mathclose" "\\rbrack" 93 "]")
    ("mathclose" "\\rbracklrtick" 10638 "⦎")
    ("mathclose" "\\rbrackubar" 10636 "⦌")
    ("mathclose" "\\rbrackurtick" 10640 "⦐")
    ("mathclose" "\\rbrbrak" 10099 "❳")
    ("mathclose" "\\rbrbrak" 12309 "〕")
    ("mathclose" "\\rceil" 8969 "⌉")
    ("mathclose" "\\rcurvyangle" 10749 "⧽")
    ("mathclose" "\\rfloor" 8971 "⌋")
    ("mathclose" "\\rgroup" 10223 "⟯")
    ("mathclose" "\\rimg" 10632 "⦈")
    ("mathclose" "\\rparen" 41 ")")
    ("mathclose" "\\rparengtr" 10644 "⦔")
    ("mathclose" "\\rrangle" 10634 "⦊")
    ("mathclose" "\\rrbracket" 10215 "⟧")
    ("mathclose" "\\rrparenthesis" 10632 "⦈")
    ("mathclose" "\\rvzigzag" 10713 "⧙")
    ("mathclose" "\\urcorner" 8989 "⌝")
    ("mathfence" "\\VERT" 10624 "⦀")
    ("mathfence" "\\Vert" 8214 "‖")
    ("mathfence" "\\Vvert" 10624 "⦀")
    ("mathfence" "\\vert" 124 "|")
    ("mathfence" "\\|" 8214 "‖")
    ("mathop" "\\Bbbsum" 8512 "⅀")
    ("mathop" "\\Join" 10781 "⨝")
    ("mathop" "\\awint" 10769 "⨑")
    ("mathop" "\\bigbot" 10200 "⟘")
    ("mathop" "\\bigcap" 8898 "⋂")
    ("mathop" "\\bigcup" 8899 "⋃")
    ("mathop" "\\bigcupdot" 10755 "⨃")
    ("mathop" "\\biginterleave" 11004 "⫼")
    ("mathop" "\\bigodot" 10752 "⨀")
    ("mathop" "\\bigoplus" 10753 "⨁")
    ("mathop" "\\bigotimes" 10754 "⨂")
    ("mathop" "\\bigsqcap" 10757 "⨅")
    ("mathop" "\\bigsqcup" 10758 "⨆")
    ("mathop" "\\bigtalloblong" 11007 "⫿")
    ("mathop" "\\bigtimes" 10761 "⨉")
    ("mathop" "\\bigtop" 10201 "⟙")
    ("mathop" "\\bigtriangleleft" 10782 "⨞")
    ("mathop" "\\biguplus" 10756 "⨄")
    ("mathop" "\\bigvee" 8897 "⋁")
    ("mathop" "\\bigwedge" 8896 "⋀")
    ("mathop" "\\cirfnint" 10768 "⨐")
    ("mathop" "\\conjquant" 10759 "⨇")
    ("mathop" "\\coprod" 8720 "∐")
    ("mathop" "\\disjquant" 10760 "⨈")
    ("mathop" "\\fint" 10767 "⨏")
    ("mathop" "\\fullouterjoin" 10199 "⟗")
    ("mathop" "\\iiiint" 10764 "⨌")
    ("mathop" "\\iiint" 8749 "∭")
    ("mathop" "\\iint" 8748 "∬")
    ("mathop" "\\int" 8747 "∫")
    ("mathop" "\\intBar" 10766 "⨎")
    ("mathop" "\\intbar" 10765 "⨍")
    ("mathop" "\\intcap" 10777 "⨙")
    ("mathop" "\\intclockwise" 8753 "∱")
    ("mathop" "\\intcup" 10778 "⨚")
    ("mathop" "\\intlarhk" 10775 "⨗")
    ("mathop" "\\intx" 10776 "⨘")
    ("mathop" "\\leftouterjoin" 10197 "⟕")
    ("mathop" "\\lowint" 10780 "⨜")
    ("mathop" "\\npolint" 10772 "⨔")
    ("mathop" "\\oiiint" 8752 "∰")
    ("mathop" "\\oiint" 8751 "∯")
    ("mathop" "\\oint" 8750 "∮")
    ("mathop" "\\ointctrclockwise" 8755 "∳")
    ("mathop" "\\pointint" 10773 "⨕")
    ("mathop" "\\prod" 8719 "∏")
    ("mathop" "\\rightouterjoin" 10198 "⟖")
    ("mathop" "\\rppolint" 10770 "⨒")
    ("mathop" "\\scpolint" 10771 "⨓")
    ("mathop" "\\sqint" 10774 "⨖")
    ("mathop" "\\sum" 8721 "∑")
    ("mathop" "\\sumint" 10763 "⨋")
    ("mathop" "\\upint" 10779 "⨛")
    ("mathop" "\\varointclockwise" 8754 "∲")
    ("mathop" "\\varprod" 10761 "⨉")
    ("mathop" "\\xbsol" 10745 "⧹")
    ("mathop" "\\xsol" 10744 "⧸")
    ("mathop" "\\zcmp" 10783 "⨟")
    ("mathop" "\\zhide" 10745 "⧹")
    ("mathop" "\\zpipe" 10784 "⨠")
    ("mathop" "\\zproject" 10785 "⨡")
    ("mathopen" "\\Lbag" 10181 "⟅")
    ("mathopen" "\\Lbrbrak" 10220 "⟬")
    ("mathopen" "\\Lbrbrak" 12312 "〘")
    ("mathopen" "\\Lparen" 10629 "⦅")
    ("mathopen" "\\Lparengtr" 10645 "⦕")
    ("mathopen" "\\Lvzigzag" 10714 "⧚")
    ("mathopen" "\\lAngle" 10218 "⟪")
    ("mathopen" "\\lBrace" 10627 "⦃")
    ("mathopen" "\\lBrack" 10214 "⟦")
    ("mathopen" "\\lParen" 10629 "⦅")
    ("mathopen" "\\lang" 10218 "⟪")
    ("mathopen" "\\langle" 10216 "⟨")
    ("mathopen" "\\langledot" 10641 "⦑")
    ("mathopen" "\\lbag" 10181 "⟅")
    ("mathopen" "\\lblkbrbrak" 10647 "⦗")
    ("mathopen" "\\lblot" 10633 "⦉")
    ("mathopen" "\\lbrace" 123 "{")
    ("mathopen" "\\lbrack" 91 "[")
    ("mathopen" "\\lbracklltick" 10639 "⦏")
    ("mathopen" "\\lbrackubar" 10635 "⦋")
    ("mathopen" "\\lbrackultick" 10637 "⦍")
    ("mathopen" "\\lbrbrak" 10098 "❲")
    ("mathopen" "\\lbrbrak" 12308 "〔")
    ("mathopen" "\\lceil" 8968 "⌈")
    ("mathopen" "\\lcurvyangle" 10748 "⧼")
    ("mathopen" "\\lfloor" 8970 "⌊")
    ("mathopen" "\\lgroup" 10222 "⟮")
    ("mathopen" "\\limg" 10631 "⦇")
    ("mathopen" "\\llangle" 10633 "⦉")
    ("mathopen" "\\llbracket" 10214 "⟦")
    ("mathopen" "\\llcorner" 8990 "⌞")
    ("mathopen" "\\llparenthesis" 10631 "⦇")
    ("mathopen" "\\longdivision" 10188 "⟌")
    ("mathopen" "\\lparen" 40 "(")
    ("mathopen" "\\lparenless" 10643 "⦓")
    ("mathopen" "\\lvzigzag" 10712 "⧘")
    ("mathopen" "\\ulcorner" 8988 "⌜")
    ("mathord" "\\#" 35 "#")
    ("mathord" "\\$" 36 "$")
    ("mathord" "\\%" 37 "%")
    ("mathord" "\\&" 38 "&")
    ("mathord" "\\AC" 8767 "∿")
    ("mathord" "\\APLboxquestion" 9072 "⍰")
    ("mathord" "\\APLboxupcaret" 9043 "⍓")
    ("mathord" "\\APLcomment" 9053 "⍝")
    ("mathord" "\\APLdownarrowbox" 9047 "⍗")
    ("mathord" "\\APLinput" 9054 "⍞")
    ("mathord" "\\APLinv" 9017 "⌹")
    ("mathord" "\\APLleftarrowbox" 9031 "⍇")
    ("mathord" "\\APLlog" 9055 "⍟")
    ("mathord" "\\APLnotbackslash" 9024 "⍀")
    ("mathord" "\\APLrightarrowbox" 9032 "⍈")
    ("mathord" "\\APLuparrowbox" 9040 "⍐")
    ("mathord" "\\Bbbeight" 120800 "𝟠")
    ("mathord" "\\Bbbfive" 120797 "𝟝")
    ("mathord" "\\Bbbfour" 120796 "𝟜")
    ("mathord" "\\Bbbnine" 120801 "𝟡")
    ("mathord" "\\Bbbone" 120793 "𝟙")
    ("mathord" "\\Bbbpi" 8508 "ℼ")
    ("mathord" "\\Bbbseven" 120799 "𝟟")
    ("mathord" "\\Bbbsix" 120798 "𝟞")
    ("mathord" "\\Bbbthree" 120795 "𝟛")
    ("mathord" "\\Bbbtwo" 120794 "𝟚")
    ("mathord" "\\Bbbzero" 120792 "𝟘")
    ("mathord" "\\CIRCLE" 9679 "●")
    ("mathord" "\\CapitalDifferentialD" 8517 "ⅅ")
    ("mathord" "\\CheckedBox" 9745 "☑")
    ("mathord" "\\ComplexI" 8520 "ⅈ")
    ("mathord" "\\ComplexJ" 8521 "ⅉ")
    ("mathord" "\\Diamond" 9671 "◇")
    ("mathord" "\\Diamondblack" 9670 "◆")
    ("mathord" "\\Diamonddot" 10192 "⟐")
    ("mathord" "\\DifferentialD" 8518 "ⅆ")
    ("mathord" "\\Euler" 8455 "ℇ")
    ("mathord" "\\Eulerconst" 8455 "ℇ")
    ("mathord" "\\Exclam" 8252 "‼")
    ("mathord" "\\ExponetialE" 8519 "ⅇ")
    ("mathord" "\\Finv" 8498 "Ⅎ")
    ("mathord" "\\Game" 8513 "⅁")
    ("mathord" "\\Hermaphrodite" 9893 "⚥")
    ("mathord" "\\LEFTCIRCLE" 9686 "◖")
    ("mathord" "\\LEFTcircle" 9680 "◐")
    ("mathord" "\\Planckconst" 8462 "ℎ")
    ("mathord" "\\PropertyLine" 8522 "⅊")
    ("mathord" "\\QED" 8718 "∎")
    ("mathord" "\\Qoppa" 984 "Ϙ")
    ("mathord" "\\Question" 8263 "⁇")
    ("mathord" "\\RIGHTCIRCLE" 9687 "◗")
    ("mathord" "\\RIGHTcircle" 9681 "◑")
    ("mathord" "\\Square" 9744 "☐")
    ("mathord" "\\Sun" 9737 "☉")
    ("mathord" "\\XBox" 9746 "☒")
    ("mathord" "\\Yup" 8516 "⅄")
    ("mathord" "\\Zbar" 437 "Ƶ")
    ("mathord" "\\_" 95 "_")
    ("mathord" "\\accurrent" 9190 "⏦")
    ("mathord" "\\acidfree" 9854 "♾")
    ("mathord" "\\acwopencirclearrow" 8634 "↺")
    ("mathord" "\\ampersand" 38 "&")
    ("mathord" "\\anchor" 9875 "⚓")
    ("mathord" "\\angdnr" 10655 "⦟")
    ("mathord" "\\angle" 8736 "∠")
    ("mathord" "\\angles" 10654 "⦞")
    ("mathord" "\\angleubar" 10660 "⦤")
    ("mathord" "\\aquarius" 9810 "♒")
    ("mathord" "\\aries" 9800 "♈")
    ("mathord" "\\arrowbullet" 10146 "➢")
    ("mathord" "\\astrosun" 9737 "☉")
    ("mathord" "\\atsign" 64 "@")
    ("mathord" "\\backdprime" 8246 "‶")
    ("mathord" "\\backepsilon" 1014 "϶")
    ("mathord" "\\backprime" 8245 "‵")
    ("mathord" "\\backslash" 92 "\\")
    ("mathord" "\\backtrprime" 8247 "‷")
    ("mathord" "\\ballotx" 10007 "✗")
    ("mathord" "\\barleftarrowrightarrowba" 8633 "↹")
    ("mathord" "\\barovernorthwestarrow" 8632 "↸")
    ("mathord" "\\bbrktbrk" 9142 "⎶")
    ("mathord" "\\bdtriplevdash" 9478 "┆")
    ("mathord" "\\because" 8757 "∵")
    ("mathord" "\\benzenr" 9187 "⏣")
    ("mathord" "\\bigblacktriangledown" 9660 "▼")
    ("mathord" "\\bigblacktriangleup" 9650 "▲")
    ("mathord" "\\bigstar" 9733 "★")
    ("mathord" "\\bigwhitestar" 9734 "☆")
    ("mathord" "\\biohazard" 9763 "☣")
    ("mathord" "\\blackcircledownarrow" 10733 "⧭")
    ("mathord" "\\blackcircledrightdot" 9864 "⚈")
    ("mathord" "\\blackcircledtwodots" 9865 "⚉")
    ("mathord" "\\blackcircleulquadwhite" 9685 "◕")
    ("mathord" "\\blackdiamonddownarrow" 10730 "⧪")
    ("mathord" "\\blackinwhitediamond" 9672 "◈")
    ("mathord" "\\blackinwhitesquare" 9635 "▣")
    ("mathord" "\\blacklefthalfcircle" 9686 "◖")
    ("mathord" "\\blackpointerleft" 9668 "◄")
    ("mathord" "\\blackpointerright" 9658 "►")
    ("mathord" "\\blackrighthalfcircle" 9687 "◗")
    ("mathord" "\\blacksmiley" 9787 "☻")
    ("mathord" "\\blacksquare" 9724 "◼")
    ("mathord" "\\blacksquare" 11035 "⬛")
    ("mathord" "\\blkhorzoval" 11052 "⬬")
    ("mathord" "\\blkvertoval" 11054 "⬮")
    ("mathord" "\\blockfull" 9608 "█")
    ("mathord" "\\blockhalfshaded" 9618 "▒")
    ("mathord" "\\blocklefthalf" 9612 "▌")
    ("mathord" "\\blocklowhalf" 9604 "▄")
    ("mathord" "\\blockqtrshaded" 9617 "░")
    ("mathord" "\\blockrighthalf" 9616 "▐")
    ("mathord" "\\blockthreeqtrshaded" 9619 "▓")
    ("mathord" "\\blockuphalf" 9600 "▀")
    ("mathord" "\\bot" 8869 "⊥")
    ("mathord" "\\botsemicircle" 9697 "◡")
    ("mathord" "\\boxonbox" 10697 "⧉")
    ("mathord" "\\bullseye" 9678 "◎")
    ("mathord" "\\cancer" 9803 "♋")
    ("mathord" "\\capricornus" 9809 "♑")
    ("mathord" "\\caretinsert" 8248 "‸")
    ("mathord" "\\carriagereturn" 8629 "↵")
    ("mathord" "\\cdots" 8943 "⋯")
    ("mathord" "\\cent" 162 "¢")
    ("mathord" "\\checkmark" 10003 "✓")
    ("mathord" "\\cirE" 10691 "⧃")
    ("mathord" "\\circlearrowleft" 8634 "↺")
    ("mathord" "\\circlearrowright" 8635 "↻")
    ("mathord" "\\circlebottomhalfblack" 9682 "◒")
    ("mathord" "\\circledR" 174 "®")
    ("mathord" "\\circledbullet" 10687 "⦿")
    ("mathord" "\\circledownarrow" 10732 "⧬")
    ("mathord" "\\circledrightdot" 9862 "⚆")
    ("mathord" "\\circledstar" 10026 "✪")
    ("mathord" "\\circledtwodots" 9863 "⚇")
    ("mathord" "\\circledwhitebullet" 10686 "⦾")
    ("mathord" "\\circlelefthalfblack" 9680 "◐")
    ("mathord" "\\circlellquad" 9717 "◵")
    ("mathord" "\\circlelrquad" 9718 "◶")
    ("mathord" "\\circlerighthalfblack" 9681 "◑")
    ("mathord" "\\circletophalfblack" 9683 "◓")
    ("mathord" "\\circleulquad" 9716 "◴")
    ("mathord" "\\circleurquad" 9719 "◷")
    ("mathord" "\\circleurquadblack" 9684 "◔")
    ("mathord" "\\circlevertfill" 9677 "◍")
    ("mathord" "\\cirscir" 10690 "⧂")
    ("mathord" "\\clubsuit" 9827 "♣")
    ("mathord" "\\complement" 8705 "∁")
    ("mathord" "\\conictaper" 9010 "⌲")
    ("mathord" "\\cwopencirclearrow" 8635 "↻")
    ("mathord" "\\danger" 9761 "☡")
    ("mathord" "\\dashleftarrow" 8672 "⇠")
    ("mathord" "\\dashrightarrow" 8674 "⇢")
    ("mathord" "\\diameter" 8960 "⌀")
    ("mathord" "\\diamondbotblack" 11033 "⬙")
    ("mathord" "\\diamondcdot" 10192 "⟐")
    ("mathord" "\\diamondleftblack" 11030 "⬖")
    ("mathord" "\\diamondrightblack" 11031 "⬗")
    ("mathord" "\\diamondsuit" 9826 "♢")
    ("mathord" "\\diamondtopblack" 11032 "⬘")
    ("mathord" "\\dicei" 9856 "⚀")
    ("mathord" "\\diceii" 9857 "⚁")
    ("mathord" "\\diceiii" 9858 "⚂")
    ("mathord" "\\diceiv" 9859 "⚃")
    ("mathord" "\\dicev" 9860 "⚄")
    ("mathord" "\\dicevi" 9861 "⚅")
    ("mathord" "\\dingasterisk" 10045 "✽")
    ("mathord" "\\dottedcircle" 9676 "◌")
    ("mathord" "\\dottedsquare" 11034 "⬚")
    ("mathord" "\\downdasharrow" 8675 "⇣")
    ("mathord" "\\downrightcurvedarrow" 10549 "⤵")
    ("mathord" "\\downtriangleleftblack" 10728 "⧨")
    ("mathord" "\\downtrianglerightblack" 10729 "⧩")
    ("mathord" "\\downwhitearrow" 8681 "⇩")
    ("mathord" "\\dprime" 8243 "″")
    ("mathord" "\\draftingarrow" 10139 "➛")
    ("mathord" "\\earth" 9793 "♁")
    ("mathord" "\\eighthnote" 9834 "♪")
    ("mathord" "\\elinters" 9191 "⏧")
    ("mathord" "\\emptysetoarr" 10675 "⦳")
    ("mathord" "\\emptysetoarrl" 10676 "⦴")
    ("mathord" "\\emptysetobar" 10673 "⦱")
    ("mathord" "\\emptysetocirc" 10674 "⦲")
    ("mathord" "\\enleadertwodots" 8229 "‥")
    ("mathord" "\\errbarblackcircle" 10739 "⧳")
    ("mathord" "\\errbarblackdiamond" 10737 "⧱")
    ("mathord" "\\errbarblacksquare" 10735 "⧯")
    ("mathord" "\\errbarcircle" 10738 "⧲")
    ("mathord" "\\errbardiamond" 10736 "⧰")
    ("mathord" "\\errbarsquare" 10734 "⧮")
    ("mathord" "\\euro" 8364 "€")
    ("mathord" "\\exists" 8707 "∃")
    ("mathord" "\\fdiagovnearrow" 10543 "⤯")
    ("mathord" "\\fdiagovrdiag" 10540 "⤬")
    ("mathord" "\\female" 9792 "♀")
    ("mathord" "\\fisheye" 9673 "◉")
    ("mathord" "\\flat" 9837 "♭")
    ("mathord" "\\fltns" 9189 "⏥")
    ("mathord" "\\forall" 8704 "∀")
    ("mathord" "\\fourth" 8279 "⁗")
    ("mathord" "\\fourvdots" 10649 "⦙")
    ("mathord" "\\frownie" 9785 "☹")
    ("mathord" "\\gemini" 9802 "♊")
    ("mathord" "\\gtlpar" 10656 "⦠")
    ("mathord" "\\harrowextender" 9135 "⎯")
    ("mathord" "\\heartsuit" 9825 "♡")
    ("mathord" "\\hermitmatrix" 8889 "⊹")
    ("mathord" "\\hexagon" 9108 "⎔")
    ("mathord" "\\hexagonblack" 11043 "⬣")
    ("mathord" "\\horizbar" 8213 "―")
    ("mathord" "\\house" 8962 "⌂")
    ("mathord" "\\hrectangle" 9645 "▭")
    ("mathord" "\\hrectangleblack" 9644 "▬")
    ("mathord" "\\hyphenbullet" 8259 "⁃")
    ("mathord" "\\hzigzag" 12336 "〰")
    ("mathord" "\\iinfin" 10716 "⧜")
    ("mathord" "\\increment" 8710 "∆")
    ("mathord" "\\infty" 8734 "∞")
    ("mathord" "\\intbottom" 8993 "⌡")
    ("mathord" "\\intextender" 9134 "⎮")
    ("mathord" "\\inttop" 8992 "⌠")
    ("mathord" "\\inversebullet" 9688 "◘")
    ("mathord" "\\inversewhitecircle" 9689 "◙")
    ("mathord" "\\invneg" 8976 "⌐")
    ("mathord" "\\invnot" 8976 "⌐")
    ("mathord" "\\invwhitelowerhalfcircle" 9691 "◛")
    ("mathord" "\\invwhiteupperhalfcircle" 9690 "◚")
    ("mathord" "\\jupiter" 9795 "♃")
    ("mathord" "\\laplac" 10720 "⧠")
    ("mathord" "\\lbracelend" 9129 "⎩")
    ("mathord" "\\lbracemid" 9128 "⎨")
    ("mathord" "\\lbraceuend" 9127 "⎧")
    ("mathord" "\\lbrackextender" 9122 "⎢")
    ("mathord" "\\lbracklend" 9123 "⎣")
    ("mathord" "\\lbrackuend" 9121 "⎡")
    ("mathord" "\\ldots" 8230 "…")
    ("mathord" "\\leftdasharrow" 8672 "⇠")
    ("mathord" "\\leftmoon" 9790 "☾")
    ("mathord" "\\leftwhitearrow" 8678 "⇦")
    ("mathord" "\\leo" 9804 "♌")
    ("mathord" "\\lgblkcircle" 11044 "⬤")
    ("mathord" "\\lgblksquare" 11035 "⬛")
    ("mathord" "\\lgwhtcircle" 9711 "◯")
    ("mathord" "\\lgwhtsquare" 11036 "⬜")
    ("mathord" "\\libra" 9806 "♎")
    ("mathord" "\\linefeed" 8628 "↴")
    ("mathord" "\\llarc" 9695 "◟")
    ("mathord" "\\llblacktriangle" 9699 "◣")
    ("mathord" "\\lltriangle" 9722 "◺")
    ("mathord" "\\lmoustache" 9136 "⎰")
    ("mathord" "\\lozenge" 9674 "◊")
    ("mathord" "\\lparenextender" 9116 "⎜")
    ("mathord" "\\lparenlend" 9117 "⎝")
    ("mathord" "\\lparenuend" 9115 "⎛")
    ("mathord" "\\lrarc" 9694 "◞")
    ("mathord" "\\lrblacktriangle" 9698 "◢")
    ("mathord" "\\lrtriangle" 9727 "◿")
    ("mathord" "\\lvboxline" 9144 "⎸")
    ("mathord" "\\male" 9794 "♂")
    ("mathord" "\\maltese" 10016 "✠")
    ("mathord" "\\mathdollar" 36 "$")
    ("mathord" "\\mathslash" 47 "/")
    ("mathord" "\\mbfitnabla" 120629 "𝜵")
    ("mathord" "\\mbfitpartial" 120655 "𝝏")
    ("mathord" "\\mbfitsansnabla" 120745 "𝞩")
    ("mathord" "\\mbfitsanspartial" 120771 "𝟃")
    ("mathord" "\\mbfnabla" 120513 "𝛁")
    ("mathord" "\\mbfpartial" 120539 "𝛛")
    ("mathord" "\\mbfsanseight" 120820 "𝟴")
    ("mathord" "\\mbfsansfive" 120817 "𝟱")
    ("mathord" "\\mbfsansfour" 120816 "𝟰")
    ("mathord" "\\mbfsansnabla" 120687 "𝝯")
    ("mathord" "\\mbfsansnine" 120821 "𝟵")
    ("mathord" "\\mbfsansone" 120813 "𝟭")
    ("mathord" "\\mbfsanspartial" 120713 "𝞉")
    ("mathord" "\\mbfsansseven" 120819 "𝟳")
    ("mathord" "\\mbfsanssix" 120818 "𝟲")
    ("mathord" "\\mbfsansthree" 120815 "𝟯")
    ("mathord" "\\mbfsanstwo" 120814 "𝟮")
    ("mathord" "\\mbfsanszero" 120812 "𝟬")
    ("mathord" "\\mdblkcircle" 9899 "⚫")
    ("mathord" "\\mdblkdiamond" 11045 "⬥")
    ("mathord" "\\mdblklozenge" 11047 "⬧")
    ("mathord" "\\mdblksquare" 9724 "◼")
    ("mathord" "\\mdlgblkcircle" 9679 "●")
    ("mathord" "\\mdlgblkdiamond" 9670 "◆")
    ("mathord" "\\mdlgblksquare" 9632 "■")
    ("mathord" "\\mdlgwhtdiamond" 9671 "◇")
    ("mathord" "\\mdlgwhtlozenge" 9674 "◊")
    ("mathord" "\\mdlgwhtsquare" 9633 "□")
    ("mathord" "\\mdsmblkcircle" 10625 "⦁")
    ("mathord" "\\mdsmblksquare" 9726 "◾")
    ("mathord" "\\mdsmwhtcircle" 9900 "⚬")
    ("mathord" "\\mdsmwhtsquare" 9725 "◽")
    ("mathord" "\\mdwhtcircle" 9898 "⚪")
    ("mathord" "\\mdwhtdiamond" 11046 "⬦")
    ("mathord" "\\mdwhtlozenge" 11048 "⬨")
    ("mathord" "\\mdwhtsquare" 9723 "◻")
    ("mathord" "\\measangledltosw" 10671 "⦯")
    ("mathord" "\\measangledrtose" 10670 "⦮")
    ("mathord" "\\measangleldtosw" 10667 "⦫")
    ("mathord" "\\measanglelutonw" 10665 "⦩")
    ("mathord" "\\measanglerdtose" 10666 "⦪")
    ("mathord" "\\measanglerutone" 10664 "⦨")
    ("mathord" "\\measangleultonw" 10669 "⦭")
    ("mathord" "\\measangleurtone" 10668 "⦬")
    ("mathord" "\\measuredangle" 8737 "∡")
    ("mathord" "\\measuredangleleft" 10651 "⦛")
    ("mathord" "\\measuredrightangle" 8894 "⊾")
    ("mathord" "\\medblackstar" 11089 "⭑")
    ("mathord" "\\medbullet" 9899 "⚫")
    ("mathord" "\\medcirc" 9898 "⚪")
    ("mathord" "\\medwhitestar" 11088 "⭐")
    ("mathord" "\\mercury" 9791 "☿")
    ("mathord" "\\mho" 8487 "℧")
    ("mathord" "\\mitBbbD" 8517 "ⅅ")
    ("mathord" "\\mitBbbd" 8518 "ⅆ")
    ("mathord" "\\mitBbbe" 8519 "ⅇ")
    ("mathord" "\\mitBbbi" 8520 "ⅈ")
    ("mathord" "\\mitBbbj" 8521 "ⅉ")
    ("mathord" "\\mitnabla" 120571 "𝛻")
    ("mathord" "\\mitpartial" 120597 "𝜕")
    ("mathord" "\\modtwosum" 10762 "⨊")
    ("mathord" "\\msanseight" 120810 "𝟪")
    ("mathord" "\\msansfive" 120807 "𝟧")
    ("mathord" "\\msansfour" 120806 "𝟦")
    ("mathord" "\\msansnine" 120811 "𝟫")
    ("mathord" "\\msansone" 120803 "𝟣")
    ("mathord" "\\msansseven" 120809 "𝟩")
    ("mathord" "\\msanssix" 120808 "𝟨")
    ("mathord" "\\msansthree" 120805 "𝟥")
    ("mathord" "\\msanstwo" 120804 "𝟤")
    ("mathord" "\\msanszero" 120802 "𝟢")
    ("mathord" "\\mtteight" 120830 "𝟾")
    ("mathord" "\\mttfive" 120827 "𝟻")
    ("mathord" "\\mttfour" 120826 "𝟺")
    ("mathord" "\\mttnine" 120831 "𝟿")
    ("mathord" "\\mttone" 120823 "𝟷")
    ("mathord" "\\mttseven" 120829 "𝟽")
    ("mathord" "\\mttsix" 120828 "𝟼")
    ("mathord" "\\mttthree" 120825 "𝟹")
    ("mathord" "\\mtttwo" 120824 "𝟸")
    ("mathord" "\\mttzero" 120822 "𝟶")
    ("mathord" "\\nHdownarrow" 8671 "⇟")
    ("mathord" "\\nHuparrow" 8670 "⇞")
    ("mathord" "\\nabla" 8711 "∇")
    ("mathord" "\\natural" 9838 "♮")
    ("mathord" "\\neg" 172 "¬")
    ("mathord" "\\neovnwarrow" 10545 "⤱")
    ("mathord" "\\neovsearrow" 10542 "⤮")
    ("mathord" "\\neptune" 9798 "♆")
    ("mathord" "\\neuter" 9906 "⚲")
    ("mathord" "\\nexists" 8708 "∄")
    ("mathord" "\\notbackslash" 9024 "⍀")
    ("mathord" "\\nvinfty" 10718 "⧞")
    ("mathord" "\\nwovnearrow" 10546 "⤲")
    ("mathord" "\\obot" 10682 "⦺")
    ("mathord" "\\obrbrak" 9184 "⏠")
    ("mathord" "\\octothorpe" 35 "#")
    ("mathord" "\\odotslashdot" 10684 "⦼")
    ("mathord" "\\olcross" 10683 "⦻")
    ("mathord" "\\parallelogram" 9649 "▱")
    ("mathord" "\\parallelogramblack" 9648 "▰")
    ("mathord" "\\partial" 8706 "∂")
    ("mathord" "\\partial" 120597 "𝜕")
    ("mathord" "\\pencil" 9998 "✎")
    ("mathord" "\\pentagon" 11040 "⬠")
    ("mathord" "\\pentagonblack" 11039 "⬟")
    ("mathord" "\\percent" 37 "%")
    ("mathord" "\\perps" 10977 "⫡")
    ("mathord" "\\pisces" 9811 "♓")
    ("mathord" "\\pluto" 9799 "♇")
    ("mathord" "\\pointright" 9758 "☞")
    ("mathord" "\\postalmark" 12306 "〒")
    ("mathord" "\\pounds" 163 "£")
    ("mathord" "\\prime" 8242 "′")
    ("mathord" "\\profline" 8978 "⌒")
    ("mathord" "\\profsurf" 8979 "⌓")
    ("mathord" "\\qoppa" 985 "ϙ")
    ("mathord" "\\qprime" 8279 "⁗")
    ("mathord" "\\quarternote" 9833 "♩")
    ("mathord" "\\question" 63 "?")
    ("mathord" "\\radiation" 9762 "☢")
    ("mathord" "\\rangledownzigzagarrow" 9084 "⍼")
    ("mathord" "\\rbracelend" 9133 "⎭")
    ("mathord" "\\rbracemid" 9132 "⎬")
    ("mathord" "\\rbraceuend" 9131 "⎫")
    ("mathord" "\\rbrackextender" 9125 "⎥")
    ("mathord" "\\rbracklend" 9126 "⎦")
    ("mathord" "\\rbrackuend" 9124 "⎤")
    ("mathord" "\\rdiagovfdiag" 10539 "⤫")
    ("mathord" "\\rdiagovsearrow" 10544 "⤰")
    ("mathord" "\\recycle" 9851 "♻")
    ("mathord" "\\revangle" 10659 "⦣")
    ("mathord" "\\revangleubar" 10661 "⦥")
    ("mathord" "\\revemptyset" 10672 "⦰")
    ("mathord" "\\rightangle" 8735 "∟")
    ("mathord" "\\rightanglemdot" 10653 "⦝")
    ("mathord" "\\rightanglesqr" 10652 "⦜")
    ("mathord" "\\rightdasharrow" 8674 "⇢")
    ("mathord" "\\rightmoon" 9789 "☽")
    ("mathord" "\\rightpentagon" 11092 "⭔")
    ("mathord" "\\rightpentagonblack" 11091 "⭓")
    ("mathord" "\\rightwhitearrow" 8680 "⇨")
    ("mathord" "\\rmoustache" 9137 "⎱")
    ("mathord" "\\rparenextender" 9119 "⎟")
    ("mathord" "\\rparenlend" 9120 "⎠")
    ("mathord" "\\rparenuend" 9118 "⎞")
    ("mathord" "\\rvboxline" 9145 "⎹")
    ("mathord" "\\sagittarius" 9808 "♐")
    ("mathord" "\\sansLmirrored" 8515 "⅃")
    ("mathord" "\\sansLturned" 8514 "⅂")
    ("mathord" "\\saturn" 9796 "♄")
    ("mathord" "\\scorpio" 9807 "♏")
    ("mathord" "\\second" 8243 "″")
    ("mathord" "\\seovnearrow" 10541 "⤭")
    ("mathord" "\\sharp" 9839 "♯")
    ("mathord" "\\sinewave" 8767 "∿")
    ("mathord" "\\sixteenthnote" 9836 "♬")
    ("mathord" "\\skull" 9760 "☠")
    ("mathord" "\\smblkdiamond" 11049 "⬩")
    ("mathord" "\\smblklozenge" 11050 "⬪")
    ("mathord" "\\smblksquare" 9642 "▪")
    ("mathord" "\\smiley" 9786 "☺")
    ("mathord" "\\smwhitestar" 11090 "⭒")
    ("mathord" "\\smwhtcircle" 9702 "◦")
    ("mathord" "\\smwhtlozenge" 11051 "⬫")
    ("mathord" "\\smwhtsquare" 9643 "▫")
    ("mathord" "\\spadesuit" 9824 "♠")
    ("mathord" "\\spddot" 168 "¨")
    ("mathord" "\\sphericalangle" 8738 "∢")
    ("mathord" "\\sphericalangleup" 10657 "⦡")
    ("mathord" "\\spot" 10625 "⦁")
    ("mathord" "\\sptilde" 126 "~")
    ("mathord" "\\sqlozenge" 8977 "⌑")
    ("mathord" "\\sqrtbottom" 9143 "⎷")
    ("mathord" "\\square" 9723 "◻")
    ("mathord" "\\square" 11036 "⬜")
    ("mathord" "\\squarebotblack" 11027 "⬓")
    ("mathord" "\\squarecrossfill" 9641 "▩")
    ("mathord" "\\squarehfill" 9636 "▤")
    ("mathord" "\\squarehvfill" 9638 "▦")
    ("mathord" "\\squareleftblack" 9703 "◧")
    ("mathord" "\\squarellblack" 11029 "⬕")
    ("mathord" "\\squarellquad" 9713 "◱")
    ("mathord" "\\squarelrblack" 9706 "◪")
    ("mathord" "\\squarelrquad" 9714 "◲")
    ("mathord" "\\squareneswfill" 9640 "▨")
    ("mathord" "\\squarenwsefill" 9639 "▧")
    ("mathord" "\\squarerightblack" 9704 "◨")
    ("mathord" "\\squaretopblack" 11026 "⬒")
    ("mathord" "\\squareulblack" 9705 "◩")
    ("mathord" "\\squareulquad" 9712 "◰")
    ("mathord" "\\squareurblack" 11028 "⬔")
    ("mathord" "\\squareurquad" 9715 "◳")
    ("mathord" "\\squarevfill" 9637 "▥")
    ("mathord" "\\squoval" 9634 "▢")
    ("mathord" "\\steaming" 9749 "☕")
    ("mathord" "\\sterling" 163 "£")
    ("mathord" "\\strns" 9188 "⏤")
    ("mathord" "\\subsetcirc" 10179 "⟃")
    ("mathord" "\\sumbottom" 9139 "⎳")
    ("mathord" "\\sumtop" 9138 "⎲")
    ("mathord" "\\sun" 9788 "☼")
    ("mathord" "\\supsetcirc" 10180 "⟄")
    ("mathord" "\\swords" 9876 "⚔")
    ("mathord" "\\taurus" 9801 "♉")
    ("mathord" "\\therefore" 8756 "∴")
    ("mathord" "\\thermod" 10727 "⧧")
    ("mathord" "\\third" 8244 "‴")
    ("mathord" "\\threedangle" 10176 "⟀")
    ("mathord" "\\tieinfty" 10717 "⧝")
    ("mathord" "\\top" 8868 "⊤")
    ("mathord" "\\topbot" 9014 "⌶")
    ("mathord" "\\topcir" 10993 "⫱")
    ("mathord" "\\topsemicircle" 9696 "◠")
    ("mathord" "\\trapezium" 9186 "⏢")
    ("mathord" "\\trianglecdot" 9708 "◬")
    ("mathord" "\\triangleleftblack" 9709 "◭")
    ("mathord" "\\triangleodot" 10698 "⧊")
    ("mathord" "\\trianglerightblack" 9710 "◮")
    ("mathord" "\\triangles" 10700 "⧌")
    ("mathord" "\\triangleubar" 10699 "⧋")
    ("mathord" "\\trprime" 8244 "‴")
    ("mathord" "\\turnangle" 10658 "⦢")
    ("mathord" "\\turnednot" 8985 "⌙")
    ("mathord" "\\twolowline" 8215 "‗")
    ("mathord" "\\twonotes" 9835 "♫")
    ("mathord" "\\ubrbrak" 9185 "⏡")
    ("mathord" "\\ularc" 9692 "◜")
    ("mathord" "\\ulblacktriangle" 9700 "◤")
    ("mathord" "\\ultriangle" 9720 "◸")
    ("mathord" "\\unicodecdots" 8943 "⋯")
    ("mathord" "\\unicodeellipsis" 8230 "…")
    ("mathord" "\\uparrowoncircle" 10685 "⦽")
    ("mathord" "\\upbackepsilon" 1014 "϶")
    ("mathord" "\\updasharrow" 8673 "⇡")
    ("mathord" "\\updownarrowbar" 8616 "↨")
    ("mathord" "\\upoldKoppa" 984 "Ϙ")
    ("mathord" "\\upoldkoppa" 985 "ϙ")
    ("mathord" "\\uprightcurvearrow" 10548 "⤴")
    ("mathord" "\\upwhitearrow" 8679 "⇧")
    ("mathord" "\\uranus" 9797 "♅")
    ("mathord" "\\urarc" 9693 "◝")
    ("mathord" "\\urblacktriangle" 9701 "◥")
    ("mathord" "\\urtriangle" 9721 "◹")
    ("mathord" "\\varcarriagereturn" 9166 "⏎")
    ("mathord" "\\varclubsuit" 9831 "♧")
    ("mathord" "\\vardiamondsuit" 9830 "♦")
    ("mathord" "\\varheartsuit" 9829 "♥")
    ("mathord" "\\varhexagon" 11041 "⬡")
    ("mathord" "\\varhexagonblack" 11042 "⬢")
    ("mathord" "\\varhexagonlrbonds" 9004 "⌬")
    ("mathord" "\\varlrtriangle" 8895 "⊿")
    ("mathord" "\\varnothing" 8709 "∅")
    ("mathord" "\\varspadesuit" 9828 "♤")
    ("mathord" "\\varstar" 10038 "✶")
    ("mathord" "\\vbraceextender" 9130 "⎪")
    ("mathord" "\\viewdata" 8983 "⌗")
    ("mathord" "\\virgo" 9805 "♍")
    ("mathord" "\\vrectangle" 9647 "▯")
    ("mathord" "\\vrectangleblack" 9646 "▮")
    ("mathord" "\\vysmblksquare" 11037 "⬝")
    ("mathord" "\\vysmwhtsquare" 11038 "⬞")
    ("mathord" "\\vzigzag" 10650 "⦚")
    ("mathord" "\\warning" 9888 "⚠")
    ("mathord" "\\wasylozenge" 8977 "⌑")
    ("mathord" "\\whitearrowupfrombar" 8682 "⇪")
    ("mathord" "\\whiteinwhitetriangle" 10177 "⟁")
    ("mathord" "\\whitepointerleft" 9669 "◅")
    ("mathord" "\\whitepointerright" 9659 "▻")
    ("mathord" "\\whthorzoval" 11053 "⬭")
    ("mathord" "\\whtvertoval" 11055 "⬯")
    ("mathord" "\\wideangledown" 10662 "⦦")
    ("mathord" "\\wideangleup" 10663 "⦧")
    ("mathord" "\\yen" 165 "¥")
    ("mathord" "\\yinyang" 9775 "☯")
    ("mathover" "\\overbrace" 9182 "⏞")
    ("mathover" "\\overbracket" 9140 "⎴")
    ("mathover" "\\overparen" 9180 "⏜")
    ("mathpunct" "\\comma" 44 ",")
    ("mathpunct" "\\exclam" 33 "!")
    ("mathpunct" "\\mathcolon" 58 ":")
    ("mathpunct" "\\semicolon" 59 ";")
    ("mathradical" "\\cuberoot" 8731 "∛")
    ("mathradical" "\\fourthroot" 8732 "∜")
    ("mathradical" "\\sqrt" 8730 "√")
    ("mathradical" "\\sqrt[3]" 8731 "∛")
    ("mathradical" "\\sqrt[4]" 8732 "∜")
    ("mathrel" "\\APLnotslash" 9023 "⌿")
    ("mathrel" "\\Barv" 10983 "⫧")
    ("mathrel" "\\Bot" 10987 "⫫")
    ("mathrel" "\\Bumpeq" 8782 "≎")
    ("mathrel" "\\Colon" 8759 "∷")
    ("mathrel" "\\Coloneq" 10868 "⩴")
    ("mathrel" "\\Coloneqq" 10868 "⩴")
    ("mathrel" "\\DDownarrow" 10225 "⟱")
    ("mathrel" "\\DashV" 10981 "⫥")
    ("mathrel" "\\DashVDash" 10202 "⟚")
    ("mathrel" "\\Dashv" 10980 "⫤")
    ("mathrel" "\\Ddownarrow" 10507 "⤋")
    ("mathrel" "\\Doteq" 8785 "≑")
    ("mathrel" "\\DownArrowBar" 10515 "⤓")
    ("mathrel" "\\DownLeftTeeVector" 10590 "⥞")
    ("mathrel" "\\DownLeftVectorBar" 10582 "⥖")
    ("mathrel" "\\DownRightTeeVector" 10591 "⥟")
    ("mathrel" "\\DownRightVectorBar" 10583 "⥗")
    ("mathrel" "\\Downarrow" 8659 "⇓")
    ("mathrel" "\\Equal" 10869 "⩵")
    ("mathrel" "\\Equiv" 8803 "≣")
    ("mathrel" "\\Gt" 10914 "⪢")
    ("mathrel" "\\LLeftarrow" 11077 "⭅")
    ("mathrel" "\\Ldsh" 8626 "↲")
    ("mathrel" "\\LeftArrowBar" 8676 "⇤")
    ("mathrel" "\\LeftDownTeeVector" 10593 "⥡")
    ("mathrel" "\\LeftDownVectorBar" 10585 "⥙")
    ("mathrel" "\\LeftTeeVector" 10586 "⥚")
    ("mathrel" "\\LeftTriangleBar" 10703 "⧏")
    ("mathrel" "\\LeftUpTeeVector" 10592 "⥠")
    ("mathrel" "\\LeftUpVectorBar" 10584 "⥘")
    ("mathrel" "\\LeftVectorBar" 10578 "⥒")
    ("mathrel" "\\Leftarrow" 8656 "⇐")
    ("mathrel" "\\Leftrightarrow" 8660 "⇔")
    ("mathrel" "\\Lleftarrow" 8666 "⇚")
    ("mathrel" "\\Longleftarrow" 10232 "⟸")
    ("mathrel" "\\Longleftrightarrow" 10234 "⟺")
    ("mathrel" "\\Longmapsfrom" 10237 "⟽")
    ("mathrel" "\\Longmapsto" 10238 "⟾")
    ("mathrel" "\\Longrightarrow" 10233 "⟹")
    ("mathrel" "\\Lsh" 8624 "↰")
    ("mathrel" "\\Lt" 10913 "⪡")
    ("mathrel" "\\MapsDown" 8615 "↧")
    ("mathrel" "\\MapsUp" 8613 "↥")
    ("mathrel" "\\Mapsfrom" 10502 "⤆")
    ("mathrel" "\\Mapsto" 10503 "⤇")
    ("mathrel" "\\Nearrow" 8663 "⇗")
    ("mathrel" "\\NestedGreaterGreater" 10914 "⪢")
    ("mathrel" "\\NestedLessLess" 10913 "⪡")
    ("mathrel" "\\Not" 10988 "⫬")
    ("mathrel" "\\NotGreaterLess" 8825 "≹")
    ("mathrel" "\\NotGreaterTilde" 8821 "≵")
    ("mathrel" "\\NotLessTilde" 8820 "≴")
    ("mathrel" "\\Nwarrow" 8662 "⇖")
    ("mathrel" "\\Prec" 10939 "⪻")
    ("mathrel" "\\Proportion" 8759 "∷")
    ("mathrel" "\\RRightarrow" 11078 "⭆")
    ("mathrel" "\\Rdsh" 8627 "↳")
    ("mathrel" "\\RightArrowBar" 8677 "⇥")
    ("mathrel" "\\RightDownTeeVector" 10589 "⥝")
    ("mathrel" "\\RightDownVectorBar" 10581 "⥕")
    ("mathrel" "\\RightTeeVector" 10587 "⥛")
    ("mathrel" "\\RightTriangleBar" 10704 "⧐")
    ("mathrel" "\\RightUpTeeVector" 10588 "⥜")
    ("mathrel" "\\RightUpVectorBar" 10580 "⥔")
    ("mathrel" "\\RightVectorBar" 10579 "⥓")
    ("mathrel" "\\Rightarrow" 8658 "⇒")
    ("mathrel" "\\Rrightarrow" 8667 "⇛")
    ("mathrel" "\\Rsh" 8625 "↱")
    ("mathrel" "\\Same" 10870 "⩶")
    ("mathrel" "\\Searrow" 8664 "⇘")
    ("mathrel" "\\Subset" 8912 "⋐")
    ("mathrel" "\\Succ" 10940 "⪼")
    ("mathrel" "\\Supset" 8913 "⋑")
    ("mathrel" "\\Swarrow" 8665 "⇙")
    ("mathrel" "\\Top" 10986 "⫪")
    ("mathrel" "\\UUparrow" 10224 "⟰")
    ("mathrel" "\\UpArrowBar" 10514 "⤒")
    ("mathrel" "\\Uparrow" 8657 "⇑")
    ("mathrel" "\\Updownarrow" 8661 "⇕")
    ("mathrel" "\\Uuparrow" 10506 "⤊")
    ("mathrel" "\\VDash" 8875 "⊫")
    ("mathrel" "\\Vbar" 10987 "⫫")
    ("mathrel" "\\Vdash" 8873 "⊩")
    ("mathrel" "\\Vvdash" 8874 "⊪")
    ("mathrel" "\\acwcirclearrow" 10560 "⥀")
    ("mathrel" "\\acwgapcirclearrow" 10226 "⟲")
    ("mathrel" "\\acwleftarcarrow" 10553 "⤹")
    ("mathrel" "\\acwoverarcarrow" 10554 "⤺")
    ("mathrel" "\\acwunderarcarrow" 10555 "⤻")
    ("mathrel" "\\adots" 8944 "⋰")
    ("mathrel" "\\approx" 8776 "≈")
    ("mathrel" "\\approxeq" 8778 "≊")
    ("mathrel" "\\approxeqq" 10864 "⩰")
    ("mathrel" "\\approxident" 8779 "≋")
    ("mathrel" "\\arceq" 8792 "≘")
    ("mathrel" "\\assert" 8870 "⊦")
    ("mathrel" "\\asteq" 10862 "⩮")
    ("mathrel" "\\asymp" 8781 "≍")
    ("mathrel" "\\bNot" 10989 "⫭")
    ("mathrel" "\\backcong" 8780 "≌")
    ("mathrel" "\\backsim" 8765 "∽")
    ("mathrel" "\\backsimeq" 8909 "⋍")
    ("mathrel" "\\bagmember" 8959 "⋿")
    ("mathrel" "\\barV" 10986 "⫪")
    ("mathrel" "\\bardownharpoonleft" 10593 "⥡")
    ("mathrel" "\\bardownharpoonright" 10589 "⥝")
    ("mathrel" "\\barin" 8950 "⋶")
    ("mathrel" "\\barleftarrow" 8676 "⇤")
    ("mathrel" "\\barleftharpoon" 10603 "⥫")
    ("mathrel" "\\barleftharpoondown" 10582 "⥖")
    ("mathrel" "\\barleftharpoonup" 10578 "⥒")
    ("mathrel" "\\barrightarrowdiamond" 10528 "⤠")
    ("mathrel" "\\barrightharpoon" 10605 "⥭")
    ("mathrel" "\\barrightharpoondown" 10591 "⥟")
    ("mathrel" "\\barrightharpoonup" 10587 "⥛")
    ("mathrel" "\\baruparrow" 10514 "⤒")
    ("mathrel" "\\barupharpoonleft" 10584 "⥘")
    ("mathrel" "\\barupharpoonright" 10580 "⥔")
    ("mathrel" "\\between" 8812 "≬")
    ("mathrel" "\\bij" 10518 "⤖")
    ("mathrel" "\\bowtie" 8904 "⋈")
    ("mathrel" "\\bsimilarleftarrow" 11073 "⭁")
    ("mathrel" "\\bsimilarrightarrow" 11079 "⭇")
    ("mathrel" "\\bsolhsub" 10184 "⟈")
    ("mathrel" "\\bumpeq" 8783 "≏")
    ("mathrel" "\\bumpeqq" 10926 "⪮")
    ("mathrel" "\\ccwundercurvearrow" 10559 "⤿")
    ("mathrel" "\\cirbot" 10207 "⟟")
    ("mathrel" "\\circeq" 8791 "≗")
    ("mathrel" "\\circleonleftarrow" 11056 "⬰")
    ("mathrel" "\\circleonrightarrow" 8692 "⇴")
    ("mathrel" "\\cirmid" 10991 "⫯")
    ("mathrel" "\\closure" 8272 "⁐")
    ("mathrel" "\\coloneq" 8788 "≔")
    ("mathrel" "\\cong" 8773 "≅")
    ("mathrel" "\\congdot" 10861 "⩭")
    ("mathrel" "\\corresponds" 8793 "≙")
    ("mathrel" "\\csub" 10959 "⫏")
    ("mathrel" "\\csube" 10961 "⫑")
    ("mathrel" "\\csup" 10960 "⫐")
    ("mathrel" "\\csupe" 10962 "⫒")
    ("mathrel" "\\curlyeqprec" 8926 "⋞")
    ("mathrel" "\\curlyeqsucc" 8927 "⋟")
    ("mathrel" "\\curvearrowleft" 8630 "↶")
    ("mathrel" "\\curvearrowleftplus" 10557 "⤽")
    ("mathrel" "\\curvearrowright" 8631 "↷")
    ("mathrel" "\\curvearrowrightminus" 10556 "⤼")
    ("mathrel" "\\cwcirclearrow" 10561 "⥁")
    ("mathrel" "\\cwgapcirclearrow" 10227 "⟳")
    ("mathrel" "\\cwrightarcarrow" 10552 "⤸")
    ("mathrel" "\\cwundercurvearrow" 10558 "⤾")
    ("mathrel" "\\dashV" 10979 "⫣")
    ("mathrel" "\\dashVdash" 10203 "⟛")
    ("mathrel" "\\dashcolon" 8761 "∹")
    ("mathrel" "\\dashleftharpoondown" 10603 "⥫")
    ("mathrel" "\\dashrightharpoondown" 10605 "⥭")
    ("mathrel" "\\dashv" 8867 "⊣")
    ("mathrel" "\\dbkarow" 10511 "⤏")
    ("mathrel" "\\ddots" 8945 "⋱")
    ("mathrel" "\\ddotseq" 10871 "⩷")
    ("mathrel" "\\diamondleftarrow" 10525 "⤝")
    ("mathrel" "\\diamondleftarrowbar" 10527 "⤟")
    ("mathrel" "\\disin" 8946 "⋲")
    ("mathrel" "\\dlsh" 8626 "↲")
    ("mathrel" "\\doteq" 8784 "≐")
    ("mathrel" "\\dotequiv" 10855 "⩧")
    ("mathrel" "\\dotsim" 10858 "⩪")
    ("mathrel" "\\dotsminusdots" 8762 "∺")
    ("mathrel" "\\downarrow" 8595 "↓")
    ("mathrel" "\\downarrowbar" 10515 "⤓")
    ("mathrel" "\\downarrowbarred" 10504 "⤈")
    ("mathrel" "\\downdownarrows" 8650 "⇊")
    ("mathrel" "\\downdownharpoons" 10597 "⥥")
    ("mathrel" "\\downfishtail" 10623 "⥿")
    ("mathrel" "\\downharpoonleft" 8643 "⇃")
    ("mathrel" "\\downharpoonleftbar" 10585 "⥙")
    ("mathrel" "\\downharpoonright" 8642 "⇂")
    ("mathrel" "\\downharpoonrightbar" 10581 "⥕")
    ("mathrel" "\\downharpoonsleftright" 10597 "⥥")
    ("mathrel" "\\downuparrows" 8693 "⇵")
    ("mathrel" "\\downupharpoons" 10607 "⥯")
    ("mathrel" "\\downupharpoonsleftright" 10607 "⥯")
    ("mathrel" "\\downzigzagarrow" 8623 "↯")
    ("mathrel" "\\drbkarow" 10512 "⤐")
    ("mathrel" "\\drsh" 8627 "↳")
    ("mathrel" "\\dualmap" 10719 "⧟")
    ("mathrel" "\\egsdot" 10904 "⪘")
    ("mathrel" "\\elsdot" 10903 "⪗")
    ("mathrel" "\\eparsl" 10723 "⧣")
    ("mathrel" "\\eqcirc" 8790 "≖")
    ("mathrel" "\\eqcolon" 8761 "∹")
    ("mathrel" "\\eqcolon" 8789 "≕")
    ("mathrel" "\\eqdef" 8797 "≝")
    ("mathrel" "\\eqdot" 10854 "⩦")
    ("mathrel" "\\eqeq" 10869 "⩵")
    ("mathrel" "\\eqeqeq" 10870 "⩶")
    ("mathrel" "\\eqgtr" 8925 "⋝")
    ("mathrel" "\\eqless" 8924 "⋜")
    ("mathrel" "\\eqqgtr" 10906 "⪚")
    ("mathrel" "\\eqqless" 10905 "⪙")
    ("mathrel" "\\eqqsim" 10867 "⩳")
    ("mathrel" "\\eqqslantgtr" 10908 "⪜")
    ("mathrel" "\\eqqslantless" 10907 "⪛")
    ("mathrel" "\\eqsim" 8770 "≂")
    ("mathrel" "\\eqslantgtr" 10902 "⪖")
    ("mathrel" "\\eqslantless" 10901 "⪕")
    ("mathrel" "\\equal" 61 "=")
    ("mathrel" "\\equalleftarrow" 11072 "⭀")
    ("mathrel" "\\equalparallel" 8917 "⋕")
    ("mathrel" "\\equalrightarrow" 10609 "⥱")
    ("mathrel" "\\equiv" 8801 "≡")
    ("mathrel" "\\equivDD" 10872 "⩸")
    ("mathrel" "\\equivVert" 10856 "⩨")
    ("mathrel" "\\equivVvert" 10857 "⩩")
    ("mathrel" "\\eqvparsl" 10725 "⧥")
    ("mathrel" "\\fallingdotseq" 8786 "≒")
    ("mathrel" "\\fbowtie" 10707 "⧓")
    ("mathrel" "\\ffun" 8699 "⇻")
    ("mathrel" "\\finj" 10517 "⤕")
    ("mathrel" "\\forks" 10972 "⫝̸")
    ("mathrel" "\\forksnot" 10973 "⫝")
    ("mathrel" "\\forkv" 10969 "⫙")
    ("mathrel" "\\frown" 8994 "⌢")
    ("mathrel" "\\geq" 8805 "≥")
    ("mathrel" "\\geqq" 8807 "≧")
    ("mathrel" "\\geqqslant" 11002 "⫺")
    ("mathrel" "\\geqslant" 10878 "⩾")
    ("mathrel" "\\gescc" 10921 "⪩")
    ("mathrel" "\\gesdot" 10880 "⪀")
    ("mathrel" "\\gesdoto" 10882 "⪂")
    ("mathrel" "\\gesdotol" 10884 "⪄")
    ("mathrel" "\\gesles" 10900 "⪔")
    ("mathrel" "\\gg" 8811 "≫")
    ("mathrel" "\\ggcurly" 10940 "⪼")
    ("mathrel" "\\ggg" 8921 "⋙")
    ("mathrel" "\\gggnest" 11000 "⫸")
    ("mathrel" "\\glE" 10898 "⪒")
    ("mathrel" "\\gla" 10917 "⪥")
    ("mathrel" "\\gleichstark" 10726 "⧦")
    ("mathrel" "\\glj" 10916 "⪤")
    ("mathrel" "\\gnapprox" 10890 "⪊")
    ("mathrel" "\\gneq" 10888 "⪈")
    ("mathrel" "\\gneqq" 8809 "≩")
    ("mathrel" "\\gnsim" 8935 "⋧")
    ("mathrel" "\\greater" 62 ">")
    ("mathrel" "\\gsime" 10894 "⪎")
    ("mathrel" "\\gsiml" 10896 "⪐")
    ("mathrel" "\\gtcc" 10919 "⪧")
    ("mathrel" "\\gtcir" 10874 "⩺")
    ("mathrel" "\\gtquest" 10876 "⩼")
    ("mathrel" "\\gtrapprox" 10886 "⪆")
    ("mathrel" "\\gtrarr" 10616 "⥸")
    ("mathrel" "\\gtrdot" 8919 "⋗")
    ("mathrel" "\\gtreqless" 8923 "⋛")
    ("mathrel" "\\gtreqqless" 10892 "⪌")
    ("mathrel" "\\gtrless" 8823 "≷")
    ("mathrel" "\\gtrsim" 8819 "≳")
    ("mathrel" "\\hash" 8917 "⋕")
    ("mathrel" "\\hatapprox" 10863 "⩯")
    ("mathrel" "\\hknearrow" 10532 "⤤")
    ("mathrel" "\\hknwarrow" 10531 "⤣")
    ("mathrel" "\\hksearow" 10533 "⤥")
    ("mathrel" "\\hkswarow" 10534 "⤦")
    ("mathrel" "\\hookleftarrow" 8617 "↩")
    ("mathrel" "\\hookrightarrow" 8618 "↪")
    ("mathrel" "\\iddots" 8944 "⋰")
    ("mathrel" "\\imageof" 8887 "⊷")
    ("mathrel" "\\in" 8712 "∈")
    ("mathrel" "\\isinE" 8953 "⋹")
    ("mathrel" "\\isindot" 8949 "⋵")
    ("mathrel" "\\isinobar" 8951 "⋷")
    ("mathrel" "\\isins" 8948 "⋴")
    ("mathrel" "\\isinvb" 8952 "⋸")
    ("mathrel" "\\kernelcontraction" 8763 "∻")
    ("mathrel" "\\lat" 10923 "⪫")
    ("mathrel" "\\late" 10925 "⪭")
    ("mathrel" "\\leadsto" 10547 "⤳")
    ("mathrel" "\\leftarrow" 8592 "←")
    ("mathrel" "\\leftarrowapprox" 11082 "⭊")
    ("mathrel" "\\leftarrowbackapprox" 11074 "⭂")
    ("mathrel" "\\leftarrowbsimilar" 11083 "⭋")
    ("mathrel" "\\leftarrowless" 10615 "⥷")
    ("mathrel" "\\leftarrowonoplus" 11058 "⬲")
    ("mathrel" "\\leftarrowplus" 10566 "⥆")
    ("mathrel" "\\leftarrowshortrightarrow" 10563 "⥃")
    ("mathrel" "\\leftarrowsimilar" 10611 "⥳")
    ("mathrel" "\\leftarrowsubset" 10618 "⥺")
    ("mathrel" "\\leftarrowtail" 8610 "↢")
    ("mathrel" "\\leftarrowtriangle" 8701 "⇽")
    ("mathrel" "\\leftarrowx" 11070 "⬾")
    ("mathrel" "\\leftbarharpoon" 10602 "⥪")
    ("mathrel" "\\leftbkarrow" 10508 "⤌")
    ("mathrel" "\\leftcurvedarrow" 11071 "⬿")
    ("mathrel" "\\leftdbkarrow" 10510 "⤎")
    ("mathrel" "\\leftdbltail" 10523 "⤛")
    ("mathrel" "\\leftdotarrow" 11064 "⬸")
    ("mathrel" "\\leftdowncurvedarrow" 10550 "⤶")
    ("mathrel" "\\leftfishtail" 10620 "⥼")
    ("mathrel" "\\leftharpoondown" 8637 "↽")
    ("mathrel" "\\leftharpoondownbar" 10590 "⥞")
    ("mathrel" "\\leftharpoonsupdown" 10594 "⥢")
    ("mathrel" "\\leftharpoonup" 8636 "↼")
    ("mathrel" "\\leftharpoonupbar" 10586 "⥚")
    ("mathrel" "\\leftharpoonupdash" 10602 "⥪")
    ("mathrel" "\\leftleftarrows" 8647 "⇇")
    ("mathrel" "\\leftleftharpoons" 10594 "⥢")
    ("mathrel" "\\leftrightarrow" 8596 "↔")
    ("mathrel" "\\leftrightarrowcircle" 10568 "⥈")
    ("mathrel" "\\leftrightarrows" 8646 "⇆")
    ("mathrel" "\\leftrightarrowtriangle" 8703 "⇿")
    ("mathrel" "\\leftrightharpoon" 10570 "⥊")
    ("mathrel" "\\leftrightharpoondown" 10576 "⥐")
    ("mathrel" "\\leftrightharpoondowndown" 10576 "⥐")
    ("mathrel" "\\leftrightharpoondownup" 10571 "⥋")
    ("mathrel" "\\leftrightharpoons" 8651 "⇋")
    ("mathrel" "\\leftrightharpoonsdown" 10599 "⥧")
    ("mathrel" "\\leftrightharpoonsup" 10598 "⥦")
    ("mathrel" "\\leftrightharpoonup" 10574 "⥎")
    ("mathrel" "\\leftrightharpoonupdown" 10570 "⥊")
    ("mathrel" "\\leftrightharpoonupup" 10574 "⥎")
    ("mathrel" "\\leftrightsquigarrow" 8621 "↭")
    ("mathrel" "\\leftslice" 10918 "⪦")
    ("mathrel" "\\leftsquigarrow" 8668 "⇜")
    ("mathrel" "\\lefttail" 10521 "⤙")
    ("mathrel" "\\leftthreearrows" 11057 "⬱")
    ("mathrel" "\\leftupdownharpoon" 10577 "⥑")
    ("mathrel" "\\leftwavearrow" 8604 "↜")
    ("mathrel" "\\leq" 8804 "≤")
    ("mathrel" "\\leqq" 8806 "≦")
    ("mathrel" "\\leqqslant" 11001 "⫹")
    ("mathrel" "\\leqslant" 10877 "⩽")
    ("mathrel" "\\lescc" 10920 "⪨")
    ("mathrel" "\\lesdot" 10879 "⩿")
    ("mathrel" "\\lesdoto" 10881 "⪁")
    ("mathrel" "\\lesdotor" 10883 "⪃")
    ("mathrel" "\\lesges" 10899 "⪓")
    ("mathrel" "\\less" 60 "<")
    ("mathrel" "\\lessapprox" 10885 "⪅")
    ("mathrel" "\\lessdot" 8918 "⋖")
    ("mathrel" "\\lesseqgtr" 8922 "⋚")
    ("mathrel" "\\lesseqqgtr" 10891 "⪋")
    ("mathrel" "\\lessgtr" 8822 "≶")
    ("mathrel" "\\lesssim" 8818 "≲")
    ("mathrel" "\\lfbowtie" 10705 "⧑")
    ("mathrel" "\\lftimes" 10708 "⧔")
    ("mathrel" "\\lgE" 10897 "⪑")
    ("mathrel" "\\lightning" 8623 "↯")
    ("mathrel" "\\ll" 8810 "≪")
    ("mathrel" "\\llcurly" 10939 "⪻")
    ("mathrel" "\\lll" 8920 "⋘")
    ("mathrel" "\\lllnest" 10999 "⫷")
    ("mathrel" "\\lnapprox" 10889 "⪉")
    ("mathrel" "\\lneq" 10887 "⪇")
    ("mathrel" "\\lneqq" 8808 "≨")
    ("mathrel" "\\lnsim" 8934 "⋦")
    ("mathrel" "\\longdashv" 10206 "⟞")
    ("mathrel" "\\longleftarrow" 10229 "⟵")
    ("mathrel" "\\longleftrightarrow" 10231 "⟷")
    ("mathrel" "\\longleftsquigarrow" 11059 "⬳")
    ("mathrel" "\\longmapsfrom" 10235 "⟻")
    ("mathrel" "\\longmapsto" 10236 "⟼")
    ("mathrel" "\\longrightarrow" 10230 "⟶")
    ("mathrel" "\\longrightsquigarrow" 10239 "⟿")
    ("mathrel" "\\looparrowleft" 8619 "↫")
    ("mathrel" "\\looparrowright" 8620 "↬")
    ("mathrel" "\\lrtriangleeq" 10721 "⧡")
    ("mathrel" "\\lsime" 10893 "⪍")
    ("mathrel" "\\lsimg" 10895 "⪏")
    ("mathrel" "\\lsqhook" 10957 "⫍")
    ("mathrel" "\\ltcc" 10918 "⪦")
    ("mathrel" "\\ltcir" 10873 "⩹")
    ("mathrel" "\\ltlarr" 10614 "⥶")
    ("mathrel" "\\ltquest" 10875 "⩻")
    ("mathrel" "\\ltrivb" 10703 "⧏")
    ("mathrel" "\\mapsdown" 8615 "↧")
    ("mathrel" "\\mapsfrom" 8612 "↤")
    ("mathrel" "\\mapsto" 8614 "↦")
    ("mathrel" "\\mapsup" 8613 "↥")
    ("mathrel" "\\mathratio" 8758 "∶")
    ("mathrel" "\\measeq" 8798 "≞")
    ("mathrel" "\\mid" 8739 "∣")
    ("mathrel" "\\midcir" 10992 "⫰")
    ("mathrel" "\\mlcp" 10971 "⫛")
    ("mathrel" "\\models" 8871 "⊧")
    ("mathrel" "\\multimap" 8888 "⊸")
    ("mathrel" "\\multimapboth" 10719 "⧟")
    ("mathrel" "\\multimapdotbothA" 8886 "⊶")
    ("mathrel" "\\multimapdotbothB" 8887 "⊷")
    ("mathrel" "\\multimapinv" 10204 "⟜")
    ("mathrel" "\\nLeftarrow" 8653 "⇍")
    ("mathrel" "\\nLeftrightarrow" 8654 "⇎")
    ("mathrel" "\\nRightarrow" 8655 "⇏")
    ("mathrel" "\\nVDash" 8879 "⊯")
    ("mathrel" "\\nVdash" 8878 "⊮")
    ("mathrel" "\\nVleftarrow" 8698 "⇺")
    ("mathrel" "\\nVleftarrowtail" 11066 "⬺")
    ("mathrel" "\\nVleftrightarrow" 8700 "⇼")
    ("mathrel" "\\nVrightarrow" 8699 "⇻")
    ("mathrel" "\\nVrightarrowtail" 10517 "⤕")
    ("mathrel" "\\nVtwoheadleftarrow" 11061 "⬵")
    ("mathrel" "\\nVtwoheadleftarrowtail" 11069 "⬽")
    ("mathrel" "\\nVtwoheadrightarrow" 10497 "⤁")
    ("mathrel" "\\nVtwoheadrightarrowtail" 10520 "⤘")
    ("mathrel" "\\napprox" 8777 "≉")
    ("mathrel" "\\nasymp" 8813 "≭")
    ("mathrel" "\\ncong" 8775 "≇")
    ("mathrel" "\\ne" 8800 "≠")
    ("mathrel" "\\nearrow" 8599 "↗")
    ("mathrel" "\\neq" 8800 "≠")
    ("mathrel" "\\nequiv" 8802 "≢")
    ("mathrel" "\\neswarrow" 10530 "⤢")
    ("mathrel" "\\ngeq" 8817 "≱")
    ("mathrel" "\\ngtr" 8815 "≯")
    ("mathrel" "\\ngtrless" 8825 "≹")
    ("mathrel" "\\ngtrsim" 8821 "≵")
    ("mathrel" "\\nhpar" 10994 "⫲")
    ("mathrel" "\\ni" 8715 "∋")
    ("mathrel" "\\niobar" 8958 "⋾")
    ("mathrel" "\\nis" 8956 "⋼")
    ("mathrel" "\\nisd" 8954 "⋺")
    ("mathrel" "\\nleftarrow" 8602 "↚")
    ("mathrel" "\\nleftrightarrow" 8622 "↮")
    ("mathrel" "\\nleq" 8816 "≰")
    ("mathrel" "\\nless" 8814 "≮")
    ("mathrel" "\\nlessgtr" 8824 "≸")
    ("mathrel" "\\nlesssim" 8820 "≴")
    ("mathrel" "\\nmid" 8740 "∤")
    ("mathrel" "\\nni" 8716 "∌")
    ("mathrel" "\\notasymp" 8813 "≭")
    ("mathrel" "\\notin" 8713 "∉")
    ("mathrel" "\\notslash" 9023 "⌿")
    ("mathrel" "\\nparallel" 8742 "∦")
    ("mathrel" "\\nprec" 8832 "⊀")
    ("mathrel" "\\npreccurlyeq" 8928 "⋠")
    ("mathrel" "\\npreceq" 8928 "⋠")
    ("mathrel" "\\nrightarrow" 8603 "↛")
    ("mathrel" "\\nsim" 8769 "≁")
    ("mathrel" "\\nsime" 8772 "≄")
    ("mathrel" "\\nsimeq" 8772 "≄")
    ("mathrel" "\\nsqsubseteq" 8930 "⋢")
    ("mathrel" "\\nsqsupseteq" 8931 "⋣")
    ("mathrel" "\\nsubset" 8836 "⊄")
    ("mathrel" "\\nsubseteq" 8840 "⊈")
    ("mathrel" "\\nsucc" 8833 "⊁")
    ("mathrel" "\\nsucccurlyeq" 8929 "⋡")
    ("mathrel" "\\nsucceq" 8929 "⋡")
    ("mathrel" "\\nsupset" 8837 "⊅")
    ("mathrel" "\\nsupseteq" 8841 "⊉")
    ("mathrel" "\\ntriangleleft" 8938 "⋪")
    ("mathrel" "\\ntrianglelefteq" 8940 "⋬")
    ("mathrel" "\\ntriangleright" 8939 "⋫")
    ("mathrel" "\\ntrianglerighteq" 8941 "⋭")
    ("mathrel" "\\nvDash" 8877 "⊭")
    ("mathrel" "\\nvLeftarrow" 10498 "⤂")
    ("mathrel" "\\nvLeftrightarrow" 10500 "⤄")
    ("mathrel" "\\nvRightarrow" 10499 "⤃")
    ("mathrel" "\\nvdash" 8876 "⊬")
    ("mathrel" "\\nvleftarrow" 8695 "⇷")
    ("mathrel" "\\nvleftarrowtail" 11065 "⬹")
    ("mathrel" "\\nvleftrightarrow" 8697 "⇹")
    ("mathrel" "\\nvrightarrow" 8696 "⇸")
    ("mathrel" "\\nvrightarrowtail" 10516 "⤔")
    ("mathrel" "\\nvtwoheadleftarrow" 11060 "⬴")
    ("mathrel" "\\nvtwoheadleftarrowtail" 11068 "⬼")
    ("mathrel" "\\nvtwoheadrightarrow" 10496 "⤀")
    ("mathrel" "\\nvtwoheadrightarrowtail" 10519 "⤗")
    ("mathrel" "\\nwarrow" 8598 "↖")
    ("mathrel" "\\nwsearrow" 10529 "⤡")
    ("mathrel" "\\origof" 8886 "⊶")
    ("mathrel" "\\parallel" 8741 "∥")
    ("mathrel" "\\parsim" 10995 "⫳")
    ("mathrel" "\\partialmeetcontraction" 10915 "⪣")
    ("mathrel" "\\perp" 10178 "⟂")
    ("mathrel" "\\pfun" 8696 "⇸")
    ("mathrel" "\\pinj" 10516 "⤔")
    ("mathrel" "\\pitchfork" 8916 "⋔")
    ("mathrel" "\\prec" 8826 "≺")
    ("mathrel" "\\precapprox" 10935 "⪷")
    ("mathrel" "\\preccurlyeq" 8828 "≼")
    ("mathrel" "\\preceq" 10927 "⪯")
    ("mathrel" "\\preceqq" 10931 "⪳")
    ("mathrel" "\\precnapprox" 10937 "⪹")
    ("mathrel" "\\precneq" 10929 "⪱")
    ("mathrel" "\\precneqq" 10933 "⪵")
    ("mathrel" "\\precnsim" 8936 "⋨")
    ("mathrel" "\\precsim" 8830 "≾")
    ("mathrel" "\\propto" 8733 "∝")
    ("mathrel" "\\prurel" 8880 "⊰")
    ("mathrel" "\\psur" 10496 "⤀")
    ("mathrel" "\\pullback" 10195 "⟓")
    ("mathrel" "\\pushout" 10196 "⟔")
    ("mathrel" "\\questeq" 8799 "≟")
    ("mathrel" "\\revnmid" 10990 "⫮")
    ("mathrel" "\\rfbowtie" 10706 "⧒")
    ("mathrel" "\\rftimes" 10709 "⧕")
    ("mathrel" "\\rightarrow" 8594 "→")
    ("mathrel" "\\rightarrowapprox" 10613 "⥵")
    ("mathrel" "\\rightarrowbackapprox" 11080 "⭈")
    ("mathrel" "\\rightarrowbar" 8677 "⇥")
    ("mathrel" "\\rightarrowbsimilar" 11084 "⭌")
    ("mathrel" "\\rightarrowdiamond" 10526 "⤞")
    ("mathrel" "\\rightarrowgtr" 11075 "⭃")
    ("mathrel" "\\rightarrowonoplus" 10228 "⟴")
    ("mathrel" "\\rightarrowplus" 10565 "⥅")
    ("mathrel" "\\rightarrowshortleftarrow" 10562 "⥂")
    ("mathrel" "\\rightarrowsimilar" 10612 "⥴")
    ("mathrel" "\\rightarrowsupset" 11076 "⭄")
    ("mathrel" "\\rightarrowtail" 8611 "↣")
    ("mathrel" "\\rightarrowtriangle" 8702 "⇾")
    ("mathrel" "\\rightarrowx" 10567 "⥇")
    ("mathrel" "\\rightbarharpoon" 10604 "⥬")
    ("mathrel" "\\rightbkarrow" 10509 "⤍")
    ("mathrel" "\\rightcurvedarrow" 10547 "⤳")
    ("mathrel" "\\rightdbltail" 10524 "⤜")
    ("mathrel" "\\rightdotarrow" 10513 "⤑")
    ("mathrel" "\\rightdowncurvedarrow" 10551 "⤷")
    ("mathrel" "\\rightfishtail" 10621 "⥽")
    ("mathrel" "\\rightharpoondown" 8641 "⇁")
    ("mathrel" "\\rightharpoondownbar" 10583 "⥗")
    ("mathrel" "\\rightharpoonsupdown" 10596 "⥤")
    ("mathrel" "\\rightharpoonup" 8640 "⇀")
    ("mathrel" "\\rightharpoonupbar" 10579 "⥓")
    ("mathrel" "\\rightharpoonupdash" 10604 "⥬")
    ("mathrel" "\\rightimply" 10608 "⥰")
    ("mathrel" "\\rightleftarrows" 8644 "⇄")
    ("mathrel" "\\rightleftharpoon" 10571 "⥋")
    ("mathrel" "\\rightleftharpoons" 8652 "⇌")
    ("mathrel" "\\rightleftharpoonsdown" 10601 "⥩")
    ("mathrel" "\\rightleftharpoonsup" 10600 "⥨")
    ("mathrel" "\\rightrightarrows" 8649 "⇉")
    ("mathrel" "\\rightrightharpoons" 10596 "⥤")
    ("mathrel" "\\rightslice" 10919 "⪧")
    ("mathrel" "\\rightsquigarrow" 8669 "⇝")
    ("mathrel" "\\righttail" 10522 "⤚")
    ("mathrel" "\\rightthreearrows" 8694 "⇶")
    ("mathrel" "\\rightupdownharpoon" 10575 "⥏")
    ("mathrel" "\\rightwavearrow" 8605 "↝")
    ("mathrel" "\\risingdotseq" 8787 "≓")
    ("mathrel" "\\rsqhook" 10958 "⫎")
    ("mathrel" "\\rtriltri" 10702 "⧎")
    ("mathrel" "\\ruledelayed" 10740 "⧴")
    ("mathrel" "\\scurel" 8881 "⊱")
    ("mathrel" "\\searrow" 8600 "↘")
    ("mathrel" "\\shortdowntack" 10975 "⫟")
    ("mathrel" "\\shortlefttack" 10974 "⫞")
    ("mathrel" "\\shortrightarrowleftarrow" 10564 "⥄")
    ("mathrel" "\\shortuptack" 10976 "⫠")
    ("mathrel" "\\sim" 8764 "∼")
    ("mathrel" "\\simeq" 8771 "≃")
    ("mathrel" "\\simgE" 10912 "⪠")
    ("mathrel" "\\simgtr" 10910 "⪞")
    ("mathrel" "\\similarleftarrow" 11081 "⭉")
    ("mathrel" "\\similarrightarrow" 10610 "⥲")
    ("mathrel" "\\simlE" 10911 "⪟")
    ("mathrel" "\\simless" 10909 "⪝")
    ("mathrel" "\\simminussim" 10860 "⩬")
    ("mathrel" "\\simneqq" 8774 "≆")
    ("mathrel" "\\simrdots" 10859 "⩫")
    ("mathrel" "\\smallin" 8714 "∊")
    ("mathrel" "\\smallni" 8717 "∍")
    ("mathrel" "\\smeparsl" 10724 "⧤")
    ("mathrel" "\\smile" 8995 "⌣")
    ("mathrel" "\\smt" 10922 "⪪")
    ("mathrel" "\\smte" 10924 "⪬")
    ("mathrel" "\\sqsubset" 8847 "⊏")
    ("mathrel" "\\sqsubseteq" 8849 "⊑")
    ("mathrel" "\\sqsubsetneq" 8932 "⋤")
    ("mathrel" "\\sqsupset" 8848 "⊐")
    ("mathrel" "\\sqsupseteq" 8850 "⊒")
    ("mathrel" "\\sqsupsetneq" 8933 "⋥")
    ("mathrel" "\\stareq" 8795 "≛")
    ("mathrel" "\\strictfi" 10620 "⥼")
    ("mathrel" "\\strictif" 10621 "⥽")
    ("mathrel" "\\subedot" 10947 "⫃")
    ("mathrel" "\\submult" 10945 "⫁")
    ("mathrel" "\\subrarr" 10617 "⥹")
    ("mathrel" "\\subset" 8834 "⊂")
    ("mathrel" "\\subsetapprox" 10953 "⫉")
    ("mathrel" "\\subsetdot" 10941 "⪽")
    ("mathrel" "\\subseteq" 8838 "⊆")
    ("mathrel" "\\subseteqq" 10949 "⫅")
    ("mathrel" "\\subsetneq" 8842 "⊊")
    ("mathrel" "\\subsetneqq" 10955 "⫋")
    ("mathrel" "\\subsetplus" 10943 "⪿")
    ("mathrel" "\\subsim" 10951 "⫇")
    ("mathrel" "\\subsub" 10965 "⫕")
    ("mathrel" "\\subsup" 10963 "⫓")
    ("mathrel" "\\succ" 8827 "≻")
    ("mathrel" "\\succapprox" 10936 "⪸")
    ("mathrel" "\\succcurlyeq" 8829 "≽")
    ("mathrel" "\\succeq" 10928 "⪰")
    ("mathrel" "\\succeqq" 10932 "⪴")
    ("mathrel" "\\succnapprox" 10938 "⪺")
    ("mathrel" "\\succneq" 10930 "⪲")
    ("mathrel" "\\succneqq" 10934 "⪶")
    ("mathrel" "\\succnsim" 8937 "⋩")
    ("mathrel" "\\succsim" 8831 "≿")
    ("mathrel" "\\supdsub" 10968 "⫘")
    ("mathrel" "\\supedot" 10948 "⫄")
    ("mathrel" "\\suphsol" 10185 "⟉")
    ("mathrel" "\\suphsub" 10967 "⫗")
    ("mathrel" "\\suplarr" 10619 "⥻")
    ("mathrel" "\\supmult" 10946 "⫂")
    ("mathrel" "\\supset" 8835 "⊃")
    ("mathrel" "\\supsetapprox" 10954 "⫊")
    ("mathrel" "\\supsetdot" 10942 "⪾")
    ("mathrel" "\\supseteq" 8839 "⊇")
    ("mathrel" "\\supseteqq" 10950 "⫆")
    ("mathrel" "\\supsetneq" 8843 "⊋")
    ("mathrel" "\\supsetneqq" 10956 "⫌")
    ("mathrel" "\\supsetplus" 10944 "⫀")
    ("mathrel" "\\supsim" 10952 "⫈")
    ("mathrel" "\\supsub" 10964 "⫔")
    ("mathrel" "\\supsup" 10966 "⫖")
    ("mathrel" "\\swarrow" 8601 "↙")
    ("mathrel" "\\toea" 10536 "⤨")
    ("mathrel" "\\tona" 10535 "⤧")
    ("mathrel" "\\topfork" 10970 "⫚")
    ("mathrel" "\\tosa" 10537 "⤩")
    ("mathrel" "\\towa" 10538 "⤪")
    ("mathrel" "\\trianglelefteq" 8884 "⊴")
    ("mathrel" "\\triangleq" 8796 "≜")
    ("mathrel" "\\trianglerighteq" 8885 "⊵")
    ("mathrel" "\\twoheaddownarrow" 8609 "↡")
    ("mathrel" "\\twoheadleftarrow" 8606 "↞")
    ("mathrel" "\\twoheadleftarrowtail" 11067 "⬻")
    ("mathrel" "\\twoheadleftdbkarrow" 11063 "⬷")
    ("mathrel" "\\twoheadmapsfrom" 11062 "⬶")
    ("mathrel" "\\twoheadmapsto" 10501 "⤅")
    ("mathrel" "\\twoheadrightarrow" 8608 "↠")
    ("mathrel" "\\twoheadrightarrowtail" 10518 "⤖")
    ("mathrel" "\\twoheaduparrow" 8607 "↟")
    ("mathrel" "\\twoheaduparrowcircle" 10569 "⥉")
    ("mathrel" "\\uparrow" 8593 "↑")
    ("mathrel" "\\uparrowbarred" 10505 "⤉")
    ("mathrel" "\\updownarrow" 8597 "↕")
    ("mathrel" "\\updownarrows" 8645 "⇅")
    ("mathrel" "\\updownharpoonleftleft" 10577 "⥑")
    ("mathrel" "\\updownharpoonleftright" 10573 "⥍")
    ("mathrel" "\\updownharpoonrightleft" 10572 "⥌")
    ("mathrel" "\\updownharpoonrightright" 10575 "⥏")
    ("mathrel" "\\updownharpoons" 10606 "⥮")
    ("mathrel" "\\updownharpoonsleftright" 10606 "⥮")
    ("mathrel" "\\upfishtail" 10622 "⥾")
    ("mathrel" "\\upharpoonleft" 8639 "↿")
    ("mathrel" "\\upharpoonleftbar" 10592 "⥠")
    ("mathrel" "\\upharpoonright" 8638 "↾")
    ("mathrel" "\\upharpoonrightbar" 10588 "⥜")
    ("mathrel" "\\upharpoonsleftright" 10595 "⥣")
    ("mathrel" "\\upin" 10194 "⟒")
    ("mathrel" "\\upuparrows" 8648 "⇈")
    ("mathrel" "\\upupharpoons" 10595 "⥣")
    ("mathrel" "\\vBar" 10984 "⫨")
    ("mathrel" "\\vBarv" 10985 "⫩")
    ("mathrel" "\\vDash" 8872 "⊨")
    ("mathrel" "\\vDdash" 10978 "⫢")
    ("mathrel" "\\varVdash" 10982 "⫦")
    ("mathrel" "\\varisinobar" 8950 "⋶")
    ("mathrel" "\\varisins" 8947 "⋳")
    ("mathrel" "\\varniobar" 8957 "⋽")
    ("mathrel" "\\varnis" 8955 "⋻")
    ("mathrel" "\\vartriangleleft" 8882 "⊲")
    ("mathrel" "\\vartriangleright" 8883 "⊳")
    ("mathrel" "\\vbrtri" 10704 "⧐")
    ("mathrel" "\\vdash" 8866 "⊢")
    ("mathrel" "\\vdots" 8942 "⋮")
    ("mathrel" "\\veeeq" 8794 "≚")
    ("mathrel" "\\veeonwedge" 10841 "⩙")
    ("mathrel" "\\vlongdash" 10205 "⟝")
    ("mathrel" "\\wedgeq" 8793 "≙")
    ("mathunder" "\\underbrace" 9183 "⏟")
    ("mathunder" "\\underbracket" 9141 "⎵")
    ("mathunder" "\\underparen" 9181 "⏝"))
  "Extended list of mathematical symbols.
Each element is a list of the form (CLASS COMMAND UNICODE SYMBOL)
where COMMAND is a latex command, UNICODE is the unicode entry
point and SYMBOL is the Emacs with the actual unicode
character. The last two are equivalent and provided for
convenience. This list does not include about 190 of the standard
LaTeX math commands in `math-symbol-list-basic' because of the
conflicting names between latex and unicode-math standard.")

(defconst math-symbol-list-packages
  '(("amsfonts" "mathclose" "\\urcorner" 8989 "⌝")
    ("amsfonts" "mathclose" "\\lrcorner" 8991 "⌟")
    ("amsfonts" "mathopen" "\\ulcorner" 8988 "⌜")
    ("amsfonts" "mathopen" "\\llcorner" 8990 "⌞")
    ("amsfonts" "mathord" "\\yen" 165 "¥")
    ("amsfonts" "mathord" "\\circledR" 174 "®")
    ("amsfonts" "mathord" "\\mho" 8487 "℧")
    ("amsfonts" "mathord" "\\dashleftarrow" 8672 "⇠")
    ("amsfonts" "mathord" "\\dashrightarrow" 8674 "⇢")
    ("amsfonts" "mathord" "\\checkmark" 10003 "✓")
    ("amsfonts" "mathord" "\\maltese" 10016 "✠")
    ("amsfonts" "mathrel" "\\sqsubset" 8847 "⊏")
    ("amsfonts" "mathrel" "\\sqsupset" 8848 "⊐")
    ("amsmath" "mathaccent" "\\dddot" 8411 "⃛")
    ("amsmath" "mathaccent" "\\ddddot" 8412 "⃜")
    ("amsmath" "mathaccent" "\\overleftrightarrow" 8417 "⃡")
    ("amsmath" "mathaccent" "\\underleftarrow" 8430 "⃮")
    ("amsmath" "mathaccent" "\\underrightarrow" 8431 "⃯")
    ("amsmath" "mathop" "\\iint" 8748 "∬")
    ("amsmath" "mathop" "\\iiint" 8749 "∭")
    ("amsmath" "mathop" "\\iiiint" 10764 "⨌")
    ("amssymb" "mathaccent" "\\mathring" 778 "̊")
    ("amssymb" "mathalpha" "\\eth" 240 "ð")
    ("amssymb" "mathalpha" "\\Digamma" 988 "Ϝ" t)
    ("amssymb" "mathalpha" "\\hslash" 8463 "ℏ")
    ("amssymb" "mathalpha" "\\wp" 8472 "℘")
    ("amssymb" "mathalpha" "\\beth" 8502 "ℶ")
    ("amssymb" "mathalpha" "\\gimel" 8503 "ℷ")
    ("amssymb" "mathalpha" "\\daleth" 8504 "ℸ")
    ("amssymb" "mathalpha" "\\varkappa" 120600 "𝜘")
    ("amssymb" "mathbin" "\\dotplus" 8724 "∔")
    ("amssymb" "mathbin" "\\smallsetminus" 8726 "∖")
    ("amssymb" "mathbin" "\\wedge" 8743 "∧")
    ("amssymb" "mathbin" "\\wr" 8768 "≀")
    ("amssymb" "mathbin" "\\circledcirc" 8858 "⊚")
    ("amssymb" "mathbin" "\\circledast" 8859 "⊛")
    ("amssymb" "mathbin" "\\circleddash" 8861 "⊝")
    ("amssymb" "mathbin" "\\boxplus" 8862 "⊞")
    ("amssymb" "mathbin" "\\boxminus" 8863 "⊟")
    ("amssymb" "mathbin" "\\boxtimes" 8864 "⊠")
    ("amssymb" "mathbin" "\\boxdot" 8865 "⊡")
    ("amssymb" "mathbin" "\\intercal" 8890 "⊺")
    ("amssymb" "mathbin" "\\veebar" 8891 "⊻")
    ("amssymb" "mathbin" "\\barwedge" 8892 "⊼")
    ("amssymb" "mathbin" "\\divideontimes" 8903 "⋇")
    ("amssymb" "mathbin" "\\ltimes" 8905 "⋉")
    ("amssymb" "mathbin" "\\rtimes" 8906 "⋊")
    ("amssymb" "mathbin" "\\leftthreetimes" 8907 "⋋")
    ("amssymb" "mathbin" "\\rightthreetimes" 8908 "⋌")
    ("amssymb" "mathbin" "\\curlyvee" 8910 "⋎")
    ("amssymb" "mathbin" "\\curlywedge" 8911 "⋏")
    ("amssymb" "mathbin" "\\Cap" 8914 "⋒")
    ("amssymb" "mathbin" "\\Cup" 8915 "⋓")
    ("amssymb" "mathbin" "\\rhd" 9655 "▷")
    ("amssymb" "mathbin" "\\lhd" 9665 "◁")
    ("amssymb" "mathbin" "\\blacklozenge" 10731 "⧫")
    ("amssymb" "mathbin" "\\doublebarwedge" 10846 "⩞")
    ("amssymb" "mathop" "\\Join" 10781 "⨝")
    ("amssymb" "mathord" "\\backepsilon" 1014 "϶")
    ("amssymb" "mathord" "\\backprime" 8245 "‵")
    ("amssymb" "mathord" "\\Finv" 8498 "Ⅎ")
    ("amssymb" "mathord" "\\circlearrowleft" 8634 "↺")
    ("amssymb" "mathord" "\\circlearrowright" 8635 "↻")
    ("amssymb" "mathord" "\\complement" 8705 "∁")
    ("amssymb" "mathord" "\\nexists" 8708 "∄")
    ("amssymb" "mathord" "\\varnothing" 8709 "∅")
    ("amssymb" "mathord" "\\measuredangle" 8737 "∡")
    ("amssymb" "mathord" "\\sphericalangle" 8738 "∢")
    ("amssymb" "mathord" "\\therefore" 8756 "∴")
    ("amssymb" "mathord" "\\because" 8757 "∵")
    ("amssymb" "mathord" "\\Diamond" 9671 "◇")
    ("amssymb" "mathord" "\\lozenge" 9674 "◊")
    ("amssymb" "mathord" "\\square" 9723 "◻")
    ("amssymb" "mathord" "\\blacksquare" 9724 "◼")
    ("amssymb" "mathord" "\\bigstar" 9733 "★")
    ("amssymb" "mathord" "\\blacksquare" 11035 "⬛" t)
    ("amssymb" "mathord" "\\square" 11036 "⬜" t)
    ("amssymb" "mathrel" "\\nwarrow" 8598 "↖")
    ("amssymb" "mathrel" "\\nleftarrow" 8602 "↚")
    ("amssymb" "mathrel" "\\nrightarrow" 8603 "↛")
    ("amssymb" "mathrel" "\\twoheadleftarrow" 8606 "↞")
    ("amssymb" "mathrel" "\\twoheadrightarrow" 8608 "↠")
    ("amssymb" "mathrel" "\\leftarrowtail" 8610 "↢")
    ("amssymb" "mathrel" "\\rightarrowtail" 8611 "↣")
    ("amssymb" "mathrel" "\\looparrowleft" 8619 "↫")
    ("amssymb" "mathrel" "\\looparrowright" 8620 "↬")
    ("amssymb" "mathrel" "\\leftrightsquigarrow" 8621 "↭")
    ("amssymb" "mathrel" "\\nleftrightarrow" 8622 "↮")
    ("amssymb" "mathrel" "\\Lsh" 8624 "↰")
    ("amssymb" "mathrel" "\\Rsh" 8625 "↱")
    ("amssymb" "mathrel" "\\curvearrowleft" 8630 "↶")
    ("amssymb" "mathrel" "\\curvearrowright" 8631 "↷")
    ("amssymb" "mathrel" "\\upharpoonright" 8638 "↾")
    ("amssymb" "mathrel" "\\upharpoonleft" 8639 "↿")
    ("amssymb" "mathrel" "\\downharpoonright" 8642 "⇂")
    ("amssymb" "mathrel" "\\downharpoonleft" 8643 "⇃")
    ("amssymb" "mathrel" "\\rightleftarrows" 8644 "⇄")
    ("amssymb" "mathrel" "\\leftrightarrows" 8646 "⇆")
    ("amssymb" "mathrel" "\\leftleftarrows" 8647 "⇇")
    ("amssymb" "mathrel" "\\upuparrows" 8648 "⇈")
    ("amssymb" "mathrel" "\\rightrightarrows" 8649 "⇉")
    ("amssymb" "mathrel" "\\downdownarrows" 8650 "⇊")
    ("amssymb" "mathrel" "\\leftrightharpoons" 8651 "⇋")
    ("amssymb" "mathrel" "\\nLeftarrow" 8653 "⇍")
    ("amssymb" "mathrel" "\\nLeftrightarrow" 8654 "⇎")
    ("amssymb" "mathrel" "\\nRightarrow" 8655 "⇏")
    ("amssymb" "mathrel" "\\Lleftarrow" 8666 "⇚")
    ("amssymb" "mathrel" "\\Rrightarrow" 8667 "⇛")
    ("amssymb" "mathrel" "\\rightsquigarrow" 8669 "⇝")
    ("amssymb" "mathrel" "\\nmid" 8740 "∤")
    ("amssymb" "mathrel" "\\nparallel" 8742 "∦")
    ("amssymb" "mathrel" "\\backsim" 8765 "∽")
    ("amssymb" "mathrel" "\\nsim" 8769 "≁")
    ("amssymb" "mathrel" "\\eqsim" 8770 "≂")
    ("amssymb" "mathrel" "\\ncong" 8775 "≇")
    ("amssymb" "mathrel" "\\approxeq" 8778 "≊")
    ("amssymb" "mathrel" "\\Bumpeq" 8782 "≎")
    ("amssymb" "mathrel" "\\bumpeq" 8783 "≏")
    ("amssymb" "mathrel" "\\Doteq" 8785 "≑")
    ("amssymb" "mathrel" "\\fallingdotseq" 8786 "≒")
    ("amssymb" "mathrel" "\\risingdotseq" 8787 "≓")
    ("amssymb" "mathrel" "\\eqcirc" 8790 "≖")
    ("amssymb" "mathrel" "\\circeq" 8791 "≗")
    ("amssymb" "mathrel" "\\triangleq" 8796 "≜")
    ("amssymb" "mathrel" "\\leqq" 8806 "≦")
    ("amssymb" "mathrel" "\\geqq" 8807 "≧")
    ("amssymb" "mathrel" "\\lneqq" 8808 "≨")
    ("amssymb" "mathrel" "\\gneqq" 8809 "≩")
    ("amssymb" "mathrel" "\\between" 8812 "≬")
    ("amssymb" "mathrel" "\\nless" 8814 "≮")
    ("amssymb" "mathrel" "\\ngtr" 8815 "≯")
    ("amssymb" "mathrel" "\\nleq" 8816 "≰")
    ("amssymb" "mathrel" "\\ngeq" 8817 "≱")
    ("amssymb" "mathrel" "\\lesssim" 8818 "≲")
    ("amssymb" "mathrel" "\\gtrsim" 8819 "≳")
    ("amssymb" "mathrel" "\\lessgtr" 8822 "≶")
    ("amssymb" "mathrel" "\\gtrless" 8823 "≷")
    ("amssymb" "mathrel" "\\preccurlyeq" 8828 "≼")
    ("amssymb" "mathrel" "\\succcurlyeq" 8829 "≽")
    ("amssymb" "mathrel" "\\precsim" 8830 "≾")
    ("amssymb" "mathrel" "\\succsim" 8831 "≿")
    ("amssymb" "mathrel" "\\nprec" 8832 "⊀")
    ("amssymb" "mathrel" "\\nsucc" 8833 "⊁")
    ("amssymb" "mathrel" "\\nsubseteq" 8840 "⊈")
    ("amssymb" "mathrel" "\\nsupseteq" 8841 "⊉")
    ("amssymb" "mathrel" "\\subsetneq" 8842 "⊊")
    ("amssymb" "mathrel" "\\supsetneq" 8843 "⊋")
    ("amssymb" "mathrel" "\\dashv" 8867 "⊣")
    ("amssymb" "mathrel" "\\vDash" 8872 "⊨")
    ("amssymb" "mathrel" "\\Vdash" 8873 "⊩")
    ("amssymb" "mathrel" "\\Vvdash" 8874 "⊪")
    ("amssymb" "mathrel" "\\nvdash" 8876 "⊬")
    ("amssymb" "mathrel" "\\nvDash" 8877 "⊭")
    ("amssymb" "mathrel" "\\nVdash" 8878 "⊮")
    ("amssymb" "mathrel" "\\nVDash" 8879 "⊯")
    ("amssymb" "mathrel" "\\vartriangleleft" 8882 "⊲")
    ("amssymb" "mathrel" "\\vartriangleright" 8883 "⊳")
    ("amssymb" "mathrel" "\\trianglelefteq" 8884 "⊴")
    ("amssymb" "mathrel" "\\trianglerighteq" 8885 "⊵")
    ("amssymb" "mathrel" "\\multimap" 8888 "⊸")
    ("amssymb" "mathrel" "\\backsimeq" 8909 "⋍")
    ("amssymb" "mathrel" "\\Subset" 8912 "⋐")
    ("amssymb" "mathrel" "\\Supset" 8913 "⋑")
    ("amssymb" "mathrel" "\\pitchfork" 8916 "⋔")
    ("amssymb" "mathrel" "\\lessdot" 8918 "⋖")
    ("amssymb" "mathrel" "\\gtrdot" 8919 "⋗")
    ("amssymb" "mathrel" "\\lll" 8920 "⋘")
    ("amssymb" "mathrel" "\\ggg" 8921 "⋙")
    ("amssymb" "mathrel" "\\lesseqgtr" 8922 "⋚")
    ("amssymb" "mathrel" "\\gtreqless" 8923 "⋛")
    ("amssymb" "mathrel" "\\curlyeqprec" 8926 "⋞")
    ("amssymb" "mathrel" "\\curlyeqsucc" 8927 "⋟")
    ("amssymb" "mathrel" "\\npreceq" 8928 "⋠")
    ("amssymb" "mathrel" "\\nsucceq" 8929 "⋡")
    ("amssymb" "mathrel" "\\lnsim" 8934 "⋦")
    ("amssymb" "mathrel" "\\gnsim" 8935 "⋧")
    ("amssymb" "mathrel" "\\precnsim" 8936 "⋨")
    ("amssymb" "mathrel" "\\succnsim" 8937 "⋩")
    ("amssymb" "mathrel" "\\ntriangleleft" 8938 "⋪")
    ("amssymb" "mathrel" "\\ntriangleright" 8939 "⋫")
    ("amssymb" "mathrel" "\\ntrianglelefteq" 8940 "⋬")
    ("amssymb" "mathrel" "\\ntrianglerighteq" 8941 "⋭")
    ("amssymb" "mathrel" "\\leqslant" 10877 "⩽")
    ("amssymb" "mathrel" "\\geqslant" 10878 "⩾")
    ("amssymb" "mathrel" "\\lessapprox" 10885 "⪅")
    ("amssymb" "mathrel" "\\gtrapprox" 10886 "⪆")
    ("amssymb" "mathrel" "\\lneq" 10887 "⪇")
    ("amssymb" "mathrel" "\\gneq" 10888 "⪈")
    ("amssymb" "mathrel" "\\lnapprox" 10889 "⪉")
    ("amssymb" "mathrel" "\\gnapprox" 10890 "⪊")
    ("amssymb" "mathrel" "\\lesseqqgtr" 10891 "⪋")
    ("amssymb" "mathrel" "\\gtreqqless" 10892 "⪌")
    ("amssymb" "mathrel" "\\eqslantless" 10901 "⪕")
    ("amssymb" "mathrel" "\\eqslantgtr" 10902 "⪖")
    ("amssymb" "mathrel" "\\precapprox" 10935 "⪷")
    ("amssymb" "mathrel" "\\succapprox" 10936 "⪸")
    ("amssymb" "mathrel" "\\precnapprox" 10937 "⪹")
    ("amssymb" "mathrel" "\\succnapprox" 10938 "⪺")
    ("amssymb" "mathrel" "\\subseteqq" 10949 "⫅")
    ("amssymb" "mathrel" "\\supseteqq" 10950 "⫆")
    ("amssymb" "mathrel" "\\subsetneqq" 10955 "⫋")
    ("amssymb" "mathrel" "\\supsetneqq" 10956 "⫌")
    ("amsxtra" "mathord" "\\sptilde" 126 "~")
    ("amsxtra" "mathord" "\\spddot" 168 "¨")
    ("arevmath" "mathalpha" "\\eth" 240 "ð")
    ("arevmath" "mathalpha" "\\varbeta" 976 "ϐ")
    ("arevmath" "mathalpha" "\\Stigma" 986 "Ϛ")
    ("arevmath" "mathalpha" "\\stigma" 987 "ϛ")
    ("arevmath" "mathalpha" "\\digamma" 989 "ϝ")
    ("arevmath" "mathalpha" "\\Koppa" 990 "Ϟ")
    ("arevmath" "mathalpha" "\\koppa" 991 "ϟ")
    ("arevmath" "mathalpha" "\\Sampi" 992 "Ϡ")
    ("arevmath" "mathalpha" "\\sampi" 993 "ϡ")
    ("arevmath" "mathalpha" "\\hslash" 8463 "ℏ")
    ("arevmath" "mathord" "\\Qoppa" 984 "Ϙ")
    ("arevmath" "mathord" "\\qoppa" 985 "ϙ")
    ("arevmath" "mathord" "\\mho" 8487 "℧")
    ("arevmath" "mathord" "\\steaming" 9749 "☕")
    ("arevmath" "mathord" "\\pointright" 9758 "☞")
    ("arevmath" "mathord" "\\skull" 9760 "☠")
    ("arevmath" "mathord" "\\radiation" 9762 "☢")
    ("arevmath" "mathord" "\\biohazard" 9763 "☣")
    ("arevmath" "mathord" "\\yinyang" 9775 "☯")
    ("arevmath" "mathord" "\\quarternote" 9833 "♩")
    ("arevmath" "mathord" "\\eighthnote" 9834 "♪")
    ("arevmath" "mathord" "\\sixteenthnote" 9836 "♬")
    ("arevmath" "mathord" "\\recycle" 9851 "♻")
    ("arevmath" "mathord" "\\anchor" 9875 "⚓")
    ("arevmath" "mathord" "\\swords" 9876 "⚔")
    ("arevmath" "mathord" "\\warning" 9888 "⚠")
    ("arevmath" "mathord" "\\pencil" 9998 "✎")
    ("arevmath" "mathord" "\\ballotx" 10007 "✗")
    ("arevmath" "mathord" "\\arrowbullet" 10146 "➢")
    ("esint" "mathop" "\\iint" 8748 "∬")
    ("esint" "mathop" "\\iiint" 8749 "∭")
    ("esint" "mathop" "\\oiint" 8751 "∯")
    ("esint" "mathop" "\\varointclockwise" 8754 "∲")
    ("esint" "mathop" "\\ointctrclockwise" 8755 "∳")
    ("esint" "mathop" "\\iiiint" 10764 "⨌")
    ("esint" "mathop" "\\fint" 10767 "⨏")
    ("esint" "mathop" "\\sqint" 10774 "⨖")
    ("fourier" "mathalpha" "\\hslash" 8463 "ℏ")
    ("fourier" "mathbin" "\\smallsetminus" 8726 "∖")
    ("fourier" "mathbin" "\\intercal" 8890 "⊺")
    ("fourier" "mathbin" "\\blacktriangleright" 9656 "▸" t)
    ("fourier" "mathbin" "\\blacktriangleleft" 9666 "◂" t)
    ("fourier" "mathclose" "\\rrbracket" 10215 "⟧")
    ("fourier" "mathfence" "\\VERT" 10624 "⦀")
    ("fourier" "mathop" "\\iint" 8748 "∬")
    ("fourier" "mathop" "\\iiint" 8749 "∭")
    ("fourier" "mathop" "\\oiint" 8751 "∯")
    ("fourier" "mathop" "\\oiiint" 8752 "∰")
    ("fourier" "mathopen" "\\llbracket" 10214 "⟦")
    ("fourier" "mathord" "\\pounds" 163 "£" t)
    ("fourier" "mathord" "\\complement" 8705 "∁")
    ("fourier" "mathord" "\\nexists" 8708 "∄")
    ("fourier" "mathord" "\\square" 9723 "◻" t)
    ("fourier" "mathord" "\\blacksquare" 9724 "◼" t)
    ("fourier" "mathord" "\\blacksquare" 11035 "⬛")
    ("fourier" "mathord" "\\square" 11036 "⬜")
    ("fourier" "mathrel" "\\curvearrowleft" 8630 "↶")
    ("fourier" "mathrel" "\\curvearrowright" 8631 "↷")
    ("fourier" "mathrel" "\\leftleftarrows" 8647 "⇇")
    ("fourier" "mathrel" "\\rightrightarrows" 8649 "⇉")
    ("fourier" "mathrel" "\\nparallel" 8742 "∦")
    ("fourier" "mathrel" "\\vDash" 8872 "⊨")
    ("fourier" "mathrel" "\\nvDash" 8877 "⊭")
    ("fourier" "mathrel" "\\leqslant" 10877 "⩽")
    ("fourier" "mathrel" "\\geqslant" 10878 "⩾")
    ("frenchstyle" "mathalpha" "A" 119860 "𝐴" t)
    ("frenchstyle" "mathalpha" "B" 119861 "𝐵" t)
    ("frenchstyle" "mathalpha" "C" 119862 "𝐶" t)
    ("frenchstyle" "mathalpha" "D" 119863 "𝐷" t)
    ("frenchstyle" "mathalpha" "E" 119864 "𝐸" t)
    ("frenchstyle" "mathalpha" "F" 119865 "𝐹" t)
    ("frenchstyle" "mathalpha" "G" 119866 "𝐺" t)
    ("frenchstyle" "mathalpha" "H" 119867 "𝐻" t)
    ("frenchstyle" "mathalpha" "I" 119868 "𝐼" t)
    ("frenchstyle" "mathalpha" "J" 119869 "𝐽" t)
    ("frenchstyle" "mathalpha" "K" 119870 "𝐾" t)
    ("frenchstyle" "mathalpha" "L" 119871 "𝐿" t)
    ("frenchstyle" "mathalpha" "M" 119872 "𝑀" t)
    ("frenchstyle" "mathalpha" "N" 119873 "𝑁" t)
    ("frenchstyle" "mathalpha" "O" 119874 "𝑂" t)
    ("frenchstyle" "mathalpha" "P" 119875 "𝑃" t)
    ("frenchstyle" "mathalpha" "Q" 119876 "𝑄" t)
    ("frenchstyle" "mathalpha" "R" 119877 "𝑅" t)
    ("frenchstyle" "mathalpha" "S" 119878 "𝑆" t)
    ("frenchstyle" "mathalpha" "T" 119879 "𝑇" t)
    ("frenchstyle" "mathalpha" "U" 119880 "𝑈" t)
    ("frenchstyle" "mathalpha" "V" 119881 "𝑉" t)
    ("frenchstyle" "mathalpha" "W" 119882 "𝑊" t)
    ("frenchstyle" "mathalpha" "X" 119883 "𝑋" t)
    ("frenchstyle" "mathalpha" "Y" 119884 "𝑌" t)
    ("frenchstyle" "mathalpha" "Z" 119885 "𝑍" t)
    ("kpfonts" "mathclose" "\\rrbracket" 10215 "⟧")
    ("kpfonts" "mathopen" "\\llbracket" 10214 "⟦")
    ("literal" "mathalpha" "A" 65 "A" t)
    ("literal" "mathalpha" "B" 66 "B" t)
    ("literal" "mathalpha" "C" 67 "C" t)
    ("literal" "mathalpha" "D" 68 "D" t)
    ("literal" "mathalpha" "E" 69 "E" t)
    ("literal" "mathalpha" "F" 70 "F" t)
    ("literal" "mathalpha" "G" 71 "G" t)
    ("literal" "mathalpha" "H" 72 "H" t)
    ("literal" "mathalpha" "I" 73 "I" t)
    ("literal" "mathalpha" "J" 74 "J" t)
    ("literal" "mathalpha" "K" 75 "K" t)
    ("literal" "mathalpha" "L" 76 "L" t)
    ("literal" "mathalpha" "M" 77 "M" t)
    ("literal" "mathalpha" "N" 78 "N" t)
    ("literal" "mathalpha" "O" 79 "O" t)
    ("literal" "mathalpha" "P" 80 "P" t)
    ("literal" "mathalpha" "Q" 81 "Q" t)
    ("literal" "mathalpha" "R" 82 "R" t)
    ("literal" "mathalpha" "S" 83 "S" t)
    ("literal" "mathalpha" "T" 84 "T" t)
    ("literal" "mathalpha" "U" 85 "U" t)
    ("literal" "mathalpha" "V" 86 "V" t)
    ("literal" "mathalpha" "W" 87 "W" t)
    ("literal" "mathalpha" "X" 88 "X" t)
    ("literal" "mathalpha" "Y" 89 "Y" t)
    ("literal" "mathalpha" "Z" 90 "Z" t)
    ("literal" "mathalpha" "a" 97 "a" t)
    ("literal" "mathalpha" "b" 98 "b" t)
    ("literal" "mathalpha" "c" 99 "c" t)
    ("literal" "mathalpha" "d" 100 "d" t)
    ("literal" "mathalpha" "e" 101 "e" t)
    ("literal" "mathalpha" "f" 102 "f" t)
    ("literal" "mathalpha" "g" 103 "g" t)
    ("literal" "mathalpha" "h" 104 "h" t)
    ("literal" "mathalpha" "i" 105 "i" t)
    ("literal" "mathalpha" "j" 106 "j" t)
    ("literal" "mathalpha" "k" 107 "k" t)
    ("literal" "mathalpha" "l" 108 "l" t)
    ("literal" "mathalpha" "m" 109 "m" t)
    ("literal" "mathalpha" "n" 110 "n" t)
    ("literal" "mathalpha" "o" 111 "o" t)
    ("literal" "mathalpha" "p" 112 "p" t)
    ("literal" "mathalpha" "q" 113 "q" t)
    ("literal" "mathalpha" "r" 114 "r" t)
    ("literal" "mathalpha" "s" 115 "s" t)
    ("literal" "mathalpha" "t" 116 "t" t)
    ("literal" "mathalpha" "u" 117 "u" t)
    ("literal" "mathalpha" "v" 118 "v" t)
    ("literal" "mathalpha" "w" 119 "w" t)
    ("literal" "mathalpha" "x" 120 "x" t)
    ("literal" "mathalpha" "y" 121 "y" t)
    ("literal" "mathalpha" "z" 122 "z" t)
    ("literal" "mathalpha" "\\imath" 305 "ı" t)
    ("literal" "mathalpha" "\\jmath" 567 "ȷ" t)
    ("literal" "mathalpha" "\\Gamma" 915 "Γ" t)
    ("literal" "mathalpha" "\\Delta" 916 "Δ" t)
    ("literal" "mathalpha" "\\Theta" 920 "Θ" t)
    ("literal" "mathalpha" "\\Lambda" 923 "Λ" t)
    ("literal" "mathalpha" "\\Xi" 926 "Ξ" t)
    ("literal" "mathalpha" "\\Pi" 928 "Π" t)
    ("literal" "mathalpha" "\\Sigma" 931 "Σ" t)
    ("literal" "mathalpha" "\\Upsilon" 933 "Υ" t)
    ("literal" "mathalpha" "\\Phi" 934 "Φ" t)
    ("literal" "mathalpha" "\\Psi" 936 "Ψ" t)
    ("literal" "mathalpha" "\\Omega" 937 "Ω" t)
    ("literal" "mathalpha" "\\alpha" 945 "α" t)
    ("literal" "mathalpha" "\\beta" 946 "β" t)
    ("literal" "mathalpha" "\\gamma" 947 "γ" t)
    ("literal" "mathalpha" "\\delta" 948 "δ" t)
    ("literal" "mathalpha" "\\varepsilon" 949 "ε" t)
    ("literal" "mathalpha" "\\zeta" 950 "ζ" t)
    ("literal" "mathalpha" "\\eta" 951 "η" t)
    ("literal" "mathalpha" "\\theta" 952 "θ" t)
    ("literal" "mathalpha" "\\iota" 953 "ι" t)
    ("literal" "mathalpha" "\\kappa" 954 "κ" t)
    ("literal" "mathalpha" "\\lambda" 955 "λ" t)
    ("literal" "mathalpha" "\\mu" 956 "μ" t)
    ("literal" "mathalpha" "\\nu" 957 "ν" t)
    ("literal" "mathalpha" "\\xi" 958 "ξ" t)
    ("literal" "mathalpha" "\\pi" 960 "π" t)
    ("literal" "mathalpha" "\\rho" 961 "ρ" t)
    ("literal" "mathalpha" "\\varsigma" 962 "ς" t)
    ("literal" "mathalpha" "\\sigma" 963 "σ" t)
    ("literal" "mathalpha" "\\tau" 964 "τ" t)
    ("literal" "mathalpha" "\\upsilon" 965 "υ" t)
    ("literal" "mathalpha" "\\varphi" 966 "φ" t)
    ("literal" "mathalpha" "\\chi" 967 "χ" t)
    ("literal" "mathalpha" "\\psi" 968 "ψ" t)
    ("literal" "mathalpha" "\\omega" 969 "ω" t)
    ("literal" "mathalpha" "\\vartheta" 977 "ϑ" t)
    ("literal" "mathalpha" "\\phi" 981 "ϕ" t)
    ("literal" "mathalpha" "\\varpi" 982 "ϖ" t)
    ("literal" "mathalpha" "\\varrho" 1009 "ϱ" t)
    ("literal" "mathalpha" "\\epsilon" 1013 "ϵ" t)
    ("literal" "mathord" "\\partial" 8706 "∂" t)
    ("literal" "mathpunct" ":" 58 ":" t)
    ("marvosym" "mathrel" "\\Rightarrow" 8658 "⇒" t)
    ("mathabx" "mathbin" "\\blacktriangleup" 9652 "▴")
    ("mathabx" "mathbin" "\\smalltriangleup" 9653 "▵")
    ("mathabx" "mathbin" "\\blacktriangleright" 9656 "▸")
    ("mathabx" "mathbin" "\\smalltriangleright" 9657 "▹")
    ("mathabx" "mathbin" "\\blacktriangledown" 9662 "▾")
    ("mathabx" "mathbin" "\\smalltriangledown" 9663 "▿")
    ("mathabx" "mathbin" "\\blacktriangleleft" 9666 "◂")
    ("mathabx" "mathbin" "\\smalltriangleleft" 9667 "◃")
    ("mathabx" "mathord" "\\second" 8243 "″")
    ("mathabx" "mathord" "\\third" 8244 "‴")
    ("mathabx" "mathord" "\\fourth" 8279 "⁗")
    ("mathabx" "mathord" "\\diameter" 8960 "⌀")
    ("mathabx" "mathord" "\\Sun" 9737 "☉")
    ("mathabx" "mathord" "\\rightmoon" 9789 "☽")
    ("mathabx" "mathord" "\\leftmoon" 9790 "☾")
    ("mathabx" "mathrel" "\\dlsh" 8626 "↲")
    ("mathabx" "mathrel" "\\drsh" 8627 "↳")
    ("mathabx" "mathrel" "\\updownarrows" 8645 "⇅")
    ("mathabx" "mathrel" "\\leftsquigarrow" 8668 "⇜")
    ("mathabx" "mathrel" "\\downuparrows" 8693 "⇵")
    ("mathabx" "mathrel" "\\eqcolon" 8761 "∹" t)
    ("mathabx" "mathrel" "\\coloneq" 8788 "≔")
    ("mathabx" "mathrel" "\\eqcolon" 8789 "≕")
    ("mathabx" "mathrel" "\\corresponds" 8793 "≙")
    ("mathabx" "mathrel" "\\notasymp" 8813 "≭")
    ("mathabx" "mathrel" "\\VDash" 8875 "⊫")
    ("mathabx" "mathrel" "\\hash" 8917 "⋕")
    ("mathabx" "mathrel" "\\lll" 8920 "⋘" t)
    ("mathabx" "mathrel" "\\ggg" 8921 "⋙" t)
    ("mathabx" "mathrel" "\\barin" 8950 "⋶")
    ("mathabx" "mathrel" "\\leftrightharpoon" 10570 "⥊")
    ("mathabx" "mathrel" "\\rightleftharpoon" 10571 "⥋")
    ("mathabx" "mathrel" "\\leftleftharpoons" 10594 "⥢")
    ("mathabx" "mathrel" "\\upupharpoons" 10595 "⥣")
    ("mathabx" "mathrel" "\\rightrightharpoons" 10596 "⥤")
    ("mathabx" "mathrel" "\\downdownharpoons" 10597 "⥥")
    ("mathabx" "mathrel" "\\leftbarharpoon" 10602 "⥪")
    ("mathabx" "mathrel" "\\barleftharpoon" 10603 "⥫")
    ("mathabx" "mathrel" "\\rightbarharpoon" 10604 "⥬")
    ("mathabx" "mathrel" "\\barrightharpoon" 10605 "⥭")
    ("mathabx" "mathrel" "\\updownharpoons" 10606 "⥮")
    ("mathabx" "mathrel" "\\downupharpoons" 10607 "⥯")
    ("mathabx" "mathrel" "\\llcurly" 10939 "⪻")
    ("mathabx" "mathrel" "\\ggcurly" 10940 "⪼")
    ("mathbbol" "mathclose" "\\Rparen" 10630 "⦆")
    ("mathbbol" "mathopen" "\\Lparen" 10629 "⦅")
    ("mathcomp" "mathalpha" "\\tcohm" 8486 "Ω")
    ("mathdots" "mathrel" "\\iddots" 8944 "⋰")
    ("omlmathit" "mathord" "\\pounds" 163 "£" t)
    ("oz" "mathaccent" "\\dot" 775 "̇" t)
    ("oz" "mathbin" "\\cat" 8256 "⁀")
    ("oz" "mathbin" "\\fcmp" 10814 "⨾")
    ("oz" "mathbin" "\\dsub" 10852 "⩤")
    ("oz" "mathbin" "\\rsub" 10853 "⩥")
    ("oz" "mathclose" "\\rang" 10219 "⟫")
    ("oz" "mathclose" "\\rimg" 10632 "⦈")
    ("oz" "mathclose" "\\rblot" 10634 "⦊")
    ("oz" "mathop" "\\zhide" 10745 "⧹")
    ("oz" "mathop" "\\zcmp" 10783 "⨟")
    ("oz" "mathop" "\\zpipe" 10784 "⨠")
    ("oz" "mathop" "\\zproject" 10785 "⨡")
    ("oz" "mathopen" "\\lang" 10218 "⟪")
    ("oz" "mathopen" "\\limg" 10631 "⦇")
    ("oz" "mathopen" "\\lblot" 10633 "⦉")
    ("oz" "mathord" "\\#" 35 "#" t)
    ("oz" "mathord" "\\spot" 10625 "⦁")
    ("oz" "mathrel" "\\pfun" 8696 "⇸")
    ("oz" "mathrel" "\\ffun" 8699 "⇻")
    ("oz" "mathrel" "\\psur" 10496 "⤀")
    ("oz" "mathrel" "\\pinj" 10516 "⤔")
    ("oz" "mathrel" "\\finj" 10517 "⤕")
    ("oz" "mathrel" "\\bij" 10518 "⤖")
    ("slantedGreek" "mathalpha" "\\Gamma" 120548 "𝛤")
    ("slantedGreek" "mathalpha" "\\Delta" 120549 "𝛥")
    ("slantedGreek" "mathalpha" "\\Theta" 120553 "𝛩")
    ("slantedGreek" "mathalpha" "\\Lambda" 120556 "𝛬")
    ("slantedGreek" "mathalpha" "\\Xi" 120559 "𝛯")
    ("slantedGreek" "mathalpha" "\\Pi" 120561 "𝛱")
    ("slantedGreek" "mathalpha" "\\Sigma" 120564 "𝛴")
    ("slantedGreek" "mathalpha" "\\Upsilon" 120566 "𝛶")
    ("slantedGreek" "mathalpha" "\\Phi" 120567 "𝛷")
    ("slantedGreek" "mathalpha" "\\Psi" 120569 "𝛹")
    ("slantedGreek" "mathalpha" "\\Omega" 120570 "𝛺")
    ("stmaryrd" "mathbin" "\\boxdot" 8865 "⊡")
    ("stmaryrd" "mathbin" "\\bigtriangleup" 9651 "△" t)
    ("stmaryrd" "mathbin" "\\bigtriangledown" 9661 "▽" t)
    ("stmaryrd" "mathbin" "\\boxbar" 9707 "◫")
    ("stmaryrd" "mathbin" "\\boxslash" 10692 "⧄")
    ("stmaryrd" "mathbin" "\\boxbslash" 10693 "⧅")
    ("stmaryrd" "mathbin" "\\boxast" 10694 "⧆")
    ("stmaryrd" "mathbin" "\\boxcircle" 10695 "⧇")
    ("stmaryrd" "mathbin" "\\boxbox" 10696 "⧈")
    ("stmaryrd" "mathbin" "\\interleave" 10996 "⫴")
    ("stmaryrd" "mathbin" "\\sslash" 11005 "⫽")
    ("stmaryrd" "mathbin" "\\talloblong" 11006 "⫾")
    ("stmaryrd" "mathclose" "\\Rbag" 10182 "⟆")
    ("stmaryrd" "mathclose" "\\rrbracket" 10215 "⟧")
    ("stmaryrd" "mathop" "\\biginterleave" 11004 "⫼")
    ("stmaryrd" "mathopen" "\\Lbag" 10181 "⟅")
    ("stmaryrd" "mathopen" "\\llbracket" 10214 "⟦")
    ("stmaryrd" "mathord" "\\Yup" 8516 "⅄")
    ("stmaryrd" "mathrel" "\\mapsfrom" 8612 "↤")
    ("stmaryrd" "mathrel" "\\lightning" 8623 "↯")
    ("stmaryrd" "mathrel" "\\leftarrowtriangle" 8701 "⇽")
    ("stmaryrd" "mathrel" "\\rightarrowtriangle" 8702 "⇾")
    ("stmaryrd" "mathrel" "\\leftrightarrowtriangle" 8703 "⇿")
    ("stmaryrd" "mathrel" "\\longmapsfrom" 10235 "⟻")
    ("stmaryrd" "mathrel" "\\Longmapsfrom" 10237 "⟽")
    ("stmaryrd" "mathrel" "\\Longmapsto" 10238 "⟾")
    ("stmaryrd" "mathrel" "\\Mapsfrom" 10502 "⤆")
    ("stmaryrd" "mathrel" "\\Mapsto" 10503 "⤇")
    ("stmaryrd" "mathrel" "\\leftslice" 10918 "⪦")
    ("stmaryrd" "mathrel" "\\rightslice" 10919 "⪧")
    ("txfonts" "mathbin" "\\invamp" 8523 "⅋")
    ("txfonts" "mathbin" "\\boxbar" 9707 "◫")
    ("txfonts" "mathbin" "\\circledbslash" 10680 "⦸")
    ("txfonts" "mathbin" "\\circledless" 10688 "⧀")
    ("txfonts" "mathbin" "\\circledgtr" 10689 "⧁")
    ("txfonts" "mathbin" "\\boxslash" 10692 "⧄")
    ("txfonts" "mathbin" "\\boxbslash" 10693 "⧅")
    ("txfonts" "mathbin" "\\boxast" 10694 "⧆")
    ("txfonts" "mathclose" "\\Rbag" 10182 "⟆")
    ("txfonts" "mathop" "\\oiiint" 8752 "∰")
    ("txfonts" "mathop" "\\bigsqcap" 10757 "⨅")
    ("txfonts" "mathop" "\\varprod" 10761 "⨉")
    ("txfonts" "mathopen" "\\Lbag" 10181 "⟅")
    ("txfonts" "mathord" "\\Diamondblack" 9670 "◆")
    ("txfonts" "mathord" "\\varspadesuit" 9828 "♤")
    ("txfonts" "mathord" "\\varheartsuit" 9829 "♥")
    ("txfonts" "mathord" "\\vardiamondsuit" 9830 "♦")
    ("txfonts" "mathord" "\\varclubsuit" 9831 "♧")
    ("txfonts" "mathord" "\\medcirc" 9898 "⚪")
    ("txfonts" "mathord" "\\medbullet" 9899 "⚫")
    ("txfonts" "mathord" "\\Diamonddot" 10192 "⟐")
    ("txfonts" "mathrel" "\\Nwarrow" 8662 "⇖")
    ("txfonts" "mathrel" "\\Nearrow" 8663 "⇗")
    ("txfonts" "mathrel" "\\Searrow" 8664 "⇘")
    ("txfonts" "mathrel" "\\Swarrow" 8665 "⇙")
    ("txfonts" "mathrel" "\\leftsquigarrow" 8668 "⇜")
    ("txfonts" "mathrel" "\\eqcolon" 8761 "∹")
    ("txfonts" "mathrel" "\\nsimeq" 8772 "≄")
    ("txfonts" "mathrel" "\\coloneq" 8788 "≔" t)
    ("txfonts" "mathrel" "\\eqcolon" 8789 "≕" t)
    ("txfonts" "mathrel" "\\VDash" 8875 "⊫")
    ("txfonts" "mathrel" "\\multimapdotbothA" 8886 "⊶")
    ("txfonts" "mathrel" "\\multimapdotbothB" 8887 "⊷")
    ("txfonts" "mathrel" "\\multimapinv" 10204 "⟜")
    ("txfonts" "mathrel" "\\leadsto" 10547 "⤳")
    ("txfonts" "mathrel" "\\strictfi" 10620 "⥼")
    ("txfonts" "mathrel" "\\strictif" 10621 "⥽")
    ("txfonts" "mathrel" "\\multimapboth" 10719 "⧟")
    ("txfonts" "mathrel" "\\Coloneqq" 10868 "⩴")
    ("txfonts" "mathrel" "\\preceqq" 10931 "⪳")
    ("txfonts" "mathrel" "\\succeqq" 10932 "⪴")
    ("txfonts" "mathrel" "\\Top" 10986 "⫪")
    ("txfonts" "mathrel" "\\Bot" 10987 "⫫")
    ("undertilde" "mathaccent" "\\utilde" 816 "̰")
    ("uprightstyle" "mathalpha" "a" 119886 "𝑎" t)
    ("uprightstyle" "mathalpha" "b" 119887 "𝑏" t)
    ("uprightstyle" "mathalpha" "c" 119888 "𝑐" t)
    ("uprightstyle" "mathalpha" "d" 119889 "𝑑" t)
    ("uprightstyle" "mathalpha" "e" 119890 "𝑒" t)
    ("uprightstyle" "mathalpha" "f" 119891 "𝑓" t)
    ("uprightstyle" "mathalpha" "g" 119892 "𝑔" t)
    ("uprightstyle" "mathalpha" "i" 119894 "𝑖" t)
    ("uprightstyle" "mathalpha" "j" 119895 "𝑗" t)
    ("uprightstyle" "mathalpha" "k" 119896 "𝑘" t)
    ("uprightstyle" "mathalpha" "l" 119897 "𝑙" t)
    ("uprightstyle" "mathalpha" "m" 119898 "𝑚" t)
    ("uprightstyle" "mathalpha" "n" 119899 "𝑛" t)
    ("uprightstyle" "mathalpha" "o" 119900 "𝑜" t)
    ("uprightstyle" "mathalpha" "p" 119901 "𝑝" t)
    ("uprightstyle" "mathalpha" "q" 119902 "𝑞" t)
    ("uprightstyle" "mathalpha" "r" 119903 "𝑟" t)
    ("uprightstyle" "mathalpha" "s" 119904 "𝑠" t)
    ("uprightstyle" "mathalpha" "t" 119905 "𝑡" t)
    ("uprightstyle" "mathalpha" "u" 119906 "𝑢" t)
    ("uprightstyle" "mathalpha" "v" 119907 "𝑣" t)
    ("uprightstyle" "mathalpha" "w" 119908 "𝑤" t)
    ("uprightstyle" "mathalpha" "x" 119909 "𝑥" t)
    ("uprightstyle" "mathalpha" "y" 119910 "𝑦" t)
    ("uprightstyle" "mathalpha" "z" 119911 "𝑧" t)
    ("wasysym" "mathbin" "\\RHD" 9654 "▶")
    ("wasysym" "mathbin" "\\rhd" 9655 "▷")
    ("wasysym" "mathbin" "\\LHD" 9664 "◀")
    ("wasysym" "mathbin" "\\lhd" 9665 "◁")
    ("wasysym" "mathbin" "\\Circle" 9675 "○")
    ("wasysym" "mathop" "\\iint" 8748 "∬")
    ("wasysym" "mathop" "\\iiint" 8749 "∭")
    ("wasysym" "mathop" "\\oiint" 8751 "∯")
    ("wasysym" "mathord" "\\cent" 162 "¢")
    ("wasysym" "mathord" "\\AC" 8767 "∿")
    ("wasysym" "mathord" "\\invneg" 8976 "⌐")
    ("wasysym" "mathord" "\\wasylozenge" 8977 "⌑")
    ("wasysym" "mathord" "\\APLinv" 9017 "⌹")
    ("wasysym" "mathord" "\\notbackslash" 9024 "⍀")
    ("wasysym" "mathord" "\\APLleftarrowbox" 9031 "⍇")
    ("wasysym" "mathord" "\\APLrightarrowbox" 9032 "⍈")
    ("wasysym" "mathord" "\\APLuparrowbox" 9040 "⍐")
    ("wasysym" "mathord" "\\APLdownarrowbox" 9047 "⍗")
    ("wasysym" "mathord" "\\APLcomment" 9053 "⍝")
    ("wasysym" "mathord" "\\APLinput" 9054 "⍞")
    ("wasysym" "mathord" "\\APLlog" 9055 "⍟")
    ("wasysym" "mathord" "\\CIRCLE" 9679 "●")
    ("wasysym" "mathord" "\\LEFTcircle" 9680 "◐")
    ("wasysym" "mathord" "\\RIGHTcircle" 9681 "◑")
    ("wasysym" "mathord" "\\LEFTCIRCLE" 9686 "◖")
    ("wasysym" "mathord" "\\RIGHTCIRCLE" 9687 "◗")
    ("wasysym" "mathord" "\\Square" 9744 "☐")
    ("wasysym" "mathord" "\\CheckedBox" 9745 "☑")
    ("wasysym" "mathord" "\\XBox" 9746 "☒")
    ("wasysym" "mathord" "\\frownie" 9785 "☹")
    ("wasysym" "mathord" "\\smiley" 9786 "☺")
    ("wasysym" "mathord" "\\blacksmiley" 9787 "☻")
    ("wasysym" "mathord" "\\sun" 9788 "☼")
    ("wasysym" "mathord" "\\rightmoon" 9789 "☽")
    ("wasysym" "mathord" "\\leftmoon" 9790 "☾")
    ("wasysym" "mathord" "\\mercury" 9791 "☿")
    ("wasysym" "mathord" "\\female" 9792 "♀")
    ("wasysym" "mathord" "\\earth" 9793 "♁")
    ("wasysym" "mathord" "\\male" 9794 "♂")
    ("wasysym" "mathord" "\\jupiter" 9795 "♃")
    ("wasysym" "mathord" "\\saturn" 9796 "♄")
    ("wasysym" "mathord" "\\uranus" 9797 "♅")
    ("wasysym" "mathord" "\\neptune" 9798 "♆")
    ("wasysym" "mathord" "\\pluto" 9799 "♇")
    ("wasysym" "mathord" "\\aries" 9800 "♈")
    ("wasysym" "mathord" "\\taurus" 9801 "♉")
    ("wasysym" "mathord" "\\gemini" 9802 "♊")
    ("wasysym" "mathord" "\\cancer" 9803 "♋")
    ("wasysym" "mathord" "\\leo" 9804 "♌")
    ("wasysym" "mathord" "\\virgo" 9805 "♍")
    ("wasysym" "mathord" "\\libra" 9806 "♎")
    ("wasysym" "mathord" "\\scorpio" 9807 "♏")
    ("wasysym" "mathord" "\\sagittarius" 9808 "♐")
    ("wasysym" "mathord" "\\capricornus" 9809 "♑")
    ("wasysym" "mathord" "\\aquarius" 9810 "♒")
    ("wasysym" "mathord" "\\pisces" 9811 "♓")
    ("wasysym" "mathord" "\\quarternote" 9833 "♩")
    ("wasysym" "mathord" "\\twonotes" 9835 "♫")
    ("wasysym" "mathrel" "\\notslash" 9023 "⌿")
    ("wrisym" "mathaccent" "\\lvec" 8400 "⃐")
    ("wrisym" "mathaccent" "\\vec" 8401 "⃑")
    ("wrisym" "mathaccent" "\\LVec" 8406 "⃖")
    ("wrisym" "mathaccent" "\\vec" 8407 "⃗" t)
    ("wrisym" "mathalpha" "\\Micro" 181 "µ")
    ("wrisym" "mathalpha" "\\Stigma" 986 "Ϛ")
    ("wrisym" "mathalpha" "\\stigma" 987 "ϛ")
    ("wrisym" "mathalpha" "\\Digamma" 988 "Ϝ")
    ("wrisym" "mathalpha" "\\digamma" 989 "ϝ")
    ("wrisym" "mathalpha" "\\Sampi" 992 "Ϡ")
    ("wrisym" "mathalpha" "\\Angstroem" 8491 "Å")
    ("wrisym" "mathalpha" "\\beth" 8502 "ℶ")
    ("wrisym" "mathalpha" "\\gimel" 8503 "ℷ")
    ("wrisym" "mathalpha" "\\daleth" 8504 "ℸ")
    ("wrisym" "mathclose" "\\rrbracket" 10215 "⟧")
    ("wrisym" "mathop" "\\fint" 10767 "⨏")
    ("wrisym" "mathopen" "\\llbracket" 10214 "⟦")
    ("wrisym" "mathord" "\\backepsilon" 1014 "϶")
    ("wrisym" "mathord" "\\Euler" 8455 "ℇ")
    ("wrisym" "mathord" "\\CapitalDifferentialD" 8517 "ⅅ")
    ("wrisym" "mathord" "\\DifferentialD" 8518 "ⅆ")
    ("wrisym" "mathord" "\\ExponetialE" 8519 "ⅇ")
    ("wrisym" "mathord" "\\ComplexI" 8520 "ⅈ")
    ("wrisym" "mathord" "\\ComplexJ" 8521 "ⅉ")
    ("wrisym" "mathord" "\\rightangle" 8735 "∟")
    ("wrisym" "mathord" "\\measuredangle" 8737 "∡")
    ("wrisym" "mathord" "\\sphericalangle" 8738 "∢")
    ("wrisym" "mathord" "\\therefore" 8756 "∴")
    ("wrisym" "mathord" "\\because" 8757 "∵")
    ("wrisym" "mathover" "\\overparen" 9180 "⏜")
    ("wrisym" "mathrel" "\\leftrightarrow" 8596 "↔" t)
    ("wrisym" "mathrel" "\\MapsUp" 8613 "↥")
    ("wrisym" "mathrel" "\\MapsDown" 8615 "↧")
    ("wrisym" "mathrel" "\\LeftArrowBar" 8676 "⇤")
    ("wrisym" "mathrel" "\\RightArrowBar" 8677 "⇥")
    ("wrisym" "mathrel" "\\nni" 8716 "∌")
    ("wrisym" "mathrel" "\\Proportion" 8759 "∷")
    ("wrisym" "mathrel" "\\nsim" 8769 "≁")
    ("wrisym" "mathrel" "\\ncong" 8775 "≇")
    ("wrisym" "mathrel" "\\napprox" 8777 "≉")
    ("wrisym" "mathrel" "\\Bumpeq" 8782 "≎")
    ("wrisym" "mathrel" "\\bumpeq" 8783 "≏")
    ("wrisym" "mathrel" "\\nequiv" 8802 "≢")
    ("wrisym" "mathrel" "\\nleq" 8816 "≰")
    ("wrisym" "mathrel" "\\ngeq" 8817 "≱")
    ("wrisym" "mathrel" "\\NotLessTilde" 8820 "≴")
    ("wrisym" "mathrel" "\\NotGreaterTilde" 8821 "≵")
    ("wrisym" "mathrel" "\\NotGreaterLess" 8825 "≹")
    ("wrisym" "mathrel" "\\nprec" 8832 "⊀")
    ("wrisym" "mathrel" "\\nsucc" 8833 "⊁")
    ("wrisym" "mathrel" "\\nsubset" 8836 "⊄")
    ("wrisym" "mathrel" "\\nsupset" 8837 "⊅")
    ("wrisym" "mathrel" "\\nsubseteq" 8840 "⊈")
    ("wrisym" "mathrel" "\\nsupseteq" 8841 "⊉")
    ("wrisym" "mathrel" "\\npreceq" 8928 "⋠")
    ("wrisym" "mathrel" "\\nsucceq" 8929 "⋡")
    ("wrisym" "mathrel" "\\nsqsubseteq" 8930 "⋢")
    ("wrisym" "mathrel" "\\nsqsupseteq" 8931 "⋣")
    ("wrisym" "mathrel" "\\UpArrowBar" 10514 "⤒")
    ("wrisym" "mathrel" "\\DownArrowBar" 10515 "⤓")
    ("wrisym" "mathrel" "\\leftrightharpoonup" 10574 "⥎")
    ("wrisym" "mathrel" "\\rightupdownharpoon" 10575 "⥏")
    ("wrisym" "mathrel" "\\leftrightharpoondown" 10576 "⥐")
    ("wrisym" "mathrel" "\\leftupdownharpoon" 10577 "⥑")
    ("wrisym" "mathrel" "\\LeftVectorBar" 10578 "⥒")
    ("wrisym" "mathrel" "\\RightVectorBar" 10579 "⥓")
    ("wrisym" "mathrel" "\\RightUpVectorBar" 10580 "⥔")
    ("wrisym" "mathrel" "\\RightDownVectorBar" 10581 "⥕")
    ("wrisym" "mathrel" "\\DownLeftVectorBar" 10582 "⥖")
    ("wrisym" "mathrel" "\\DownRightVectorBar" 10583 "⥗")
    ("wrisym" "mathrel" "\\LeftUpVectorBar" 10584 "⥘")
    ("wrisym" "mathrel" "\\LeftDownVectorBar" 10585 "⥙")
    ("wrisym" "mathrel" "\\LeftTeeVector" 10586 "⥚")
    ("wrisym" "mathrel" "\\RightTeeVector" 10587 "⥛")
    ("wrisym" "mathrel" "\\RightUpTeeVector" 10588 "⥜")
    ("wrisym" "mathrel" "\\RightDownTeeVector" 10589 "⥝")
    ("wrisym" "mathrel" "\\DownLeftTeeVector" 10590 "⥞")
    ("wrisym" "mathrel" "\\DownRightTeeVector" 10591 "⥟")
    ("wrisym" "mathrel" "\\LeftUpTeeVector" 10592 "⥠")
    ("wrisym" "mathrel" "\\LeftDownTeeVector" 10593 "⥡")
    ("wrisym" "mathrel" "\\LeftTriangleBar" 10703 "⧏")
    ("wrisym" "mathrel" "\\RightTriangleBar" 10704 "⧐")
    ("wrisym" "mathrel" "\\Equal" 10869 "⩵")
    ("wrisym" "mathrel" "\\Same" 10870 "⩶")
    ("wrisym" "mathrel" "\\NestedLessLess" 10913 "⪡")
    ("wrisym" "mathrel" "\\NestedGreaterGreater" 10914 "⪢")
    ("wrisym" "mathunder" "\\underparen" 9181 "⏝"))
  "Mathematical symbols from various LaTeX packages.
Each element is of the form

  (PACKAGE CLASS COMMAND UNICODE SYMBOL CONFLICT)

Optional CONFLICT is a Boolean that indicates that this command
conflicts or generates different symbol from math-unicode
package.  Thus the generated LaTeX symbol might be different from
5th element (symbol) in the list above.  See LUCR reference [1]
for more details.  Package 'literal' corresponds to core (La)TeX.

   [1] http://milde.users.sourceforge.net/LUCR/Math/")

(defconst math-symbol-list-subscripts
  '(("subscript" "_0" 8320 "₀")
    ("subscript" "_1" 8321 "₁")
    ("subscript" "_2" 8322 "₂")
    ("subscript" "_3" 8323 "₃")
    ("subscript" "_4" 8324 "₄")
    ("subscript" "_5" 8325 "₅")
    ("subscript" "_6" 8326 "₆")
    ("subscript" "_7" 8327 "₇")
    ("subscript" "_8" 8328 "₈")
    ("subscript" "_9" 8329 "₉")
    ("subscript" "_+" 8330 "₊")
    ("subscript" "_-" 8331 "₋")
    ("subscript" "_=" 8332 "₌")
    ("subscript" "_(" 8333 "₍")
    ("subscript" "_)" 8334 "₎")
    ("subscript" "_a" 8336 "ₐ")
    ("subscript" "_e" 8337 "ₑ")
    ("subscript" "_h" 8341 "ₕ")
    ("subscript" "_i" 7522 "ᵢ")
    ("subscript" "_j" 11388 "ⱼ")
    ("subscript" "_k" 8342 "ₖ")
    ("subscript" "_l" 8343 "ₗ")
    ("subscript" "_m" 8344 "ₘ")
    ("subscript" "_n" 8345 "ₙ")
    ("subscript" "_o" 8338 "ₒ")
    ("subscript" "_p" 8346 "ₚ")
    ("subscript" "_r" 7523 "ᵣ")
    ("subscript" "_s" 8347 "ₛ")
    ("subscript" "_t" 8348 "ₜ")
    ("subscript" "_u" 7524 "ᵤ")
    ("subscript" "_v" 7525 "ᵥ")
    ("subscript" "_x" 8339 "ₓ")
    ("subscript" "_beta" 7526 "ᵦ")
    ("subscript" "_gamma" 7527 "ᵧ")
    ("subscript" "_rho" 7528 "ᵨ")
    ("subscript" "_varphi" 7529 "ᵩ")
    ("subscript" "_chi" 7530 "ᵪ"))
  "List of unicode subscripts.
See `math-symbol-list-extended' for the meaning of the entries.")

(defconst math-symbol-list-superscripts
  '(("superscripts" "^0" 8304 "⁰")
    ("superscripts" "^1" 185 "¹")
    ("superscripts" "^2" 178 "²")
    ("superscripts" "^3" 179 "³")
    ("superscripts" "^4" 8308 "⁴")
    ("superscripts" "^5" 8309 "⁵")
    ("superscripts" "^6" 8310 "⁶")
    ("superscripts" "^7" 8311 "⁷")
    ("superscripts" "^8" 8312 "⁸")
    ("superscripts" "^9" 8313 "⁹")
    ("superscripts" "^+" 8314 "⁺")
    ("superscripts" "^-" 8315 "⁻")
    ("superscripts" "^=" 8316 "⁼")
    ("superscripts" "^(" 8317 "⁽")
    ("superscripts" "^)" 8318 "⁾")
    ("superscripts" "^a" 7491 "ᵃ")
    ("superscripts" "^b" 7495 "ᵇ")
    ("superscripts" "^c" 7580 "ᶜ")
    ("superscripts" "^d" 7496 "ᵈ")
    ("superscripts" "^e" 7497 "ᵉ")
    ("superscripts" "^f" 7584 "ᶠ")
    ("superscripts" "^g" 7501 "ᵍ")
    ("superscripts" "^h" 688 "ʰ")
    ("superscripts" "^i" 8305 "ⁱ")
    ("superscripts" "^j" 690 "ʲ")
    ("superscripts" "^k" 7503 "ᵏ")
    ("superscripts" "^l" 737 "ˡ")
    ("superscripts" "^m" 7504 "ᵐ")
    ("superscripts" "^n" 8319 "ⁿ")
    ("superscripts" "^o" 7506 "ᵒ")
    ("superscripts" "^p" 7510 "ᵖ")
    ("superscripts" "^r" 691 "ʳ")
    ("superscripts" "^s" 738 "ˢ")
    ("superscripts" "^t" 7511 "ᵗ")
    ("superscripts" "^u" 7512 "ᵘ")
    ("superscripts" "^v" 7515 "ᵛ")
    ("superscripts" "^w" 695 "ʷ")
    ("superscripts" "^x" 739 "ˣ")
    ("superscripts" "^y" 696 "ʸ")
    ("superscripts" "^z" 7611 "ᶻ")
    ("superscripts" "^A" 7468 "ᴬ")
    ("superscripts" "^B" 7470 "ᴮ")
    ("superscripts" "^D" 7472 "ᴰ")
    ("superscripts" "^E" 7473 "ᴱ")
    ("superscripts" "^G" 7475 "ᴳ")
    ("superscripts" "^H" 7476 "ᴴ")
    ("superscripts" "^I" 7477 "ᴵ")
    ("superscripts" "^J" 7478 "ᴶ")
    ("superscripts" "^K" 7479 "ᴷ")
    ("superscripts" "^L" 7480 "ᴸ")
    ("superscripts" "^M" 7481 "ᴹ")
    ("superscripts" "^N" 7482 "ᴺ")
    ("superscripts" "^O" 7484 "ᴼ")
    ("superscripts" "^P" 7486 "ᴾ")
    ("superscripts" "^R" 7487 "ᴿ")
    ("superscripts" "^T" 7488 "ᵀ")
    ("superscripts" "^U" 7489 "ᵁ")
    ("superscripts" "^V" 11389 "ⱽ")
    ("superscripts" "^W" 7490 "ᵂ")
    ("superscripts" "^beta" 7517 "ᵝ")
    ("superscripts" "^gamma" 7518 "ᵞ")
    ("superscripts" "^delta" 7519 "ᵟ")
    ("superscripts" "^theta" 7615 "ᶿ")
    ("superscripts" "^iota" 7589 "ᶥ")
    ("superscripts" "^varphi" 7520 "ᵠ")
    ("superscripts" "^chi" 7521 "ᵡ"))
  "List of unicode superscripts.
See `math-symbol-list-extended' for the meaning of each entry.")

(provide 'math-symbol-lists)
;;; math-symbol-lists.el ends here
