/* Copyright (C) 2005 to 2015 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/

#include <c++-gtk-utils/lib_defs.h>

#include <c++-gtk-utils/io_watch.h>
#include <c++-gtk-utils/emitter.h>
#include <c++-gtk-utils/thread.h>

// see comments in cgu_io_watch_check_func() on what this does
#define CGU_IO_WATCH_ERROR_STORM_LIMIT 12


struct WatchSource {
  GSource source;
#ifdef HAVE_UNIX_FD_BACKEND
  void* tag;
  int error_storm_count;
#else
  GPollFD poll_fd;
#endif
  GIOCondition watch_condition;
  void* func;
};

extern "C" {
  static gboolean cgu_io_watch_prepare_func(GSource*, gint*);
  static gboolean cgu_io_watch_check_func(GSource*);

  static gboolean cgu_io_watch_dispatch_func(GSource*, GSourceFunc, void*);
  static gboolean cgu_io_watch_dispatch_tracked_func(GSource*, GSourceFunc, void*);
  static gboolean cgu_io_watch_dispatch_condition_func(GSource*, GSourceFunc, void*);
  static gboolean cgu_io_watch_dispatch_tracked_condition_func(GSource*, GSourceFunc, void*);

  static void cgu_io_watch_finalize_func(GSource*);
  static void cgu_io_watch_finalize_tracked_func(GSource*);
  static void cgu_io_watch_finalize_condition_func(GSource*);
  static void cgu_io_watch_finalize_tracked_condition_func(GSource*);
}

// in the functions below, reinterpret_cast<>()ing to WatchSource* is
// guaranteed to give the correct here as the address of
// WatchSource::source must be the same as the address of the instance
// of the WatchSource struct of which it is the first member as both
// are PODSs

gboolean cgu_io_watch_prepare_func(GSource*, gint* timeout_p) {

  *timeout_p = -1;
  return false; // we want the file descriptor to be polled
}

gboolean cgu_io_watch_check_func(GSource* source) {
  WatchSource* watch_source = reinterpret_cast<WatchSource*>(source);

#ifdef HAVE_UNIX_FD_BACKEND
  // one undocumented behaviour of the unix_fd implementation in glib
  // >= 2.36 means that even if this check function returns false, or
  // the check function pointer were to be set to NULL, then glib
  // still does its own check of descriptors added with
  // g_source_add_unix_fd().  Furthermore, this internal check will
  // regard a descriptor as ready if poll() sets POLLHUP, POLLERR, and
  // POLLNVAL even if they were not requested in the GIOCondition
  // value passed to g_source_add_unix_fd().  That is not especially
  // annoying, but it is different from the GPollFD backend.
  // Therefore, in order not to change the behaviour of programs using
  // this library if they update their version of glib, with glib >=
  // 2.36 we have to check against watch_source->watch_condition in
  // the dispatch function and not in the check function.  In the
  // check function we just want to provide a return value indicating
  // whether revents is set or not.  However, the fact that the
  // dispatch function is always called when revents is set means that
  // a program which is expecting to deal with, say, a closed pipe by
  // calling g_source_remove() directly (rather than dealing with it
  // in the iowatch callback by unsetting keep_source) can get into an
  // error or HUP storm race with the dispatch function which would
  // not occur with the GPollFD backend.  Although having a G_IO_IN
  // watch condition without also checking for G_IO_HUP and G_IO_ERR
  // is usually a mistake, that is not always the case.  We deal with
  // that by applying an error/HUP limit whenever the watch condition
  // does not include G_IO_ERR, G_IO_HUP or G_IO_NVAL, after which the
  // file descriptor is removed from the poll set.
  if (watch_source->error_storm_count < CGU_IO_WATCH_ERROR_STORM_LIMIT) {
    GIOCondition poll_condition = g_source_query_unix_fd(source, watch_source->tag);
    // poll_condition can be false, because this function is called
    // whenever any fd source is ready.  Because this case can be
    // common, we should short-circuit it
    if (!poll_condition) return false;
    GIOCondition watch_condition = watch_source->watch_condition;
    if (((poll_condition & G_IO_ERR)
  	 && !(watch_condition & G_IO_ERR))
  	|| ((poll_condition & G_IO_HUP)
  	    && !(watch_condition & G_IO_HUP))
  	|| ((poll_condition & G_IO_NVAL)
  	    && !(watch_condition & G_IO_NVAL))) {
      ++watch_source->error_storm_count;
      if (watch_source->error_storm_count >= CGU_IO_WATCH_ERROR_STORM_LIMIT) {
  	g_source_remove_unix_fd(source, watch_source->tag);
  	return false; // see below for the false return value
      }
    }
    return true;
  }
  return false; // we need the previous if test because until the
		// source is removed from its main context, this check
		// function continues to be called, and we do not want
		// to call g_source_query_unix_fd() on a fd which has
		// been removed from glib's internal fd list by the
		// call to g_source_remove_unix_fd().  However, glib's
		// internal dispatch check examines its internal file
		// descriptor list so this issue does not arise for
		// the dispatch funcs - once g_source_remove_unix_fd()
		// is called the dispatch functions will no longer
		// fire provided this function returns false
#else
  gushort poll_condition = watch_source->poll_fd.revents;
  gushort watch_condition = watch_source->watch_condition;
  return (poll_condition & watch_condition);
#endif
}

gboolean cgu_io_watch_dispatch_func(GSource* source, GSourceFunc, void*) {
  const Cgu::Callback::CallbackArg<bool&>* cb =
    static_cast<Cgu::Callback::CallbackArg<bool&>*>(reinterpret_cast<WatchSource*>(source)->func);

#ifdef HAVE_UNIX_FD_BACKEND
  // see comments in cgu_io_watch_check_func() above for why this is
  // necessary
  WatchSource* watch_source = reinterpret_cast<WatchSource*>(source);
  GIOCondition poll_condition = g_source_query_unix_fd(source, watch_source->tag);
  GIOCondition watch_condition = watch_source->watch_condition;
  if (!(poll_condition & watch_condition)) return true;
#endif

  // we are not interested in the GSourceFunc argument here as we have never
  // called g_source_set_callback()
  bool keep_source = true;
  // provide a CancelBlock here to make this function NPTL friendly,
  // as we have a catch-all without rethrowing
  Cgu::Thread::CancelBlock b;
  try {
    cb->dispatch(keep_source);
  }
  // we can't propagate exceptions from functions with C linkage.
  catch (...) {
    g_critical("Exception thrown in io_watch_dispatch_func()\n");
    return true;
  }
  return keep_source;
}

gboolean cgu_io_watch_dispatch_tracked_func(GSource* source, GSourceFunc, void*) {
  const Cgu::SafeEmitterArg<bool&>* emitter =
    static_cast<Cgu::SafeEmitterArg<bool&>*>(reinterpret_cast<WatchSource*>(source)->func);

#ifdef HAVE_UNIX_FD_BACKEND
  // see comments in cgu_io_watch_check_func() above for why this is
  // necessary
  WatchSource* watch_source = reinterpret_cast<WatchSource*>(source);
  GIOCondition poll_condition = g_source_query_unix_fd(source, watch_source->tag);
  GIOCondition watch_condition = watch_source->watch_condition;
  if (!(poll_condition & watch_condition)) return true;
#endif

  // we are not interested in the GSourceFunc argument here as we have never
  // called g_source_set_callback()
  bool keep_source = true;
  bool connected;
  // provide a CancelBlock here to make this function NPTL friendly,
  // as we have a catch-all without rethrowing
  Cgu::Thread::CancelBlock b;
  try {
    connected = emitter->test_emit(keep_source);
  }
  // we can't propagate exceptions from functions with C linkage.
  catch (...) {
    g_critical("Exception thrown in io_watch_dispatch_func()\n");
    return true;
  }
  return (keep_source && connected);
}

gboolean cgu_io_watch_dispatch_condition_func(GSource* source, GSourceFunc, void*) {
  WatchSource* watch_source = reinterpret_cast<WatchSource*>(source);
  const Cgu::Callback::CallbackArg<GIOCondition, bool&>* cb =
    static_cast<Cgu::Callback::CallbackArg<GIOCondition, bool&>*>(watch_source->func);

#ifdef HAVE_UNIX_FD_BACKEND
  // see comments in cgu_io_watch_check_func() above for why this is
  // necessary
  GIOCondition poll_condition = g_source_query_unix_fd(source, watch_source->tag);
  GIOCondition watch_condition = watch_source->watch_condition;
  if (!(poll_condition & watch_condition)) return true;
#else
  GIOCondition poll_condition = GIOCondition(watch_source->poll_fd.revents);
#endif

  // we are not interested in the GSourceFunc argument here as we have never
  // called g_source_set_callback()
  bool keep_source = true;
  // provide a CancelBlock here to make this function NPTL friendly,
  // as we have a catch-all without rethrowing
  Cgu::Thread::CancelBlock b;
  try {
    cb->dispatch(poll_condition, keep_source);
  }
  // we can't propagate exceptions from functions with C linkage.
  catch (...) {
    g_critical("Exception thrown in io_watch_dispatch_func()\n");
    return true;
  }
  return keep_source;
}

gboolean cgu_io_watch_dispatch_tracked_condition_func(GSource* source, GSourceFunc, void*) {
  WatchSource* watch_source = reinterpret_cast<WatchSource*>(source);
  const Cgu::SafeEmitterArg<GIOCondition, bool&>* emitter =
    static_cast<Cgu::SafeEmitterArg<GIOCondition, bool&>*>(watch_source->func);

#ifdef HAVE_UNIX_FD_BACKEND
  // see comments in cgu_io_watch_check_func() above for why this is
  // necessary
  GIOCondition poll_condition = g_source_query_unix_fd(source, watch_source->tag);
  GIOCondition watch_condition = watch_source->watch_condition;
  if (!(poll_condition & watch_condition)) return true;
#else
  GIOCondition poll_condition = GIOCondition(watch_source->poll_fd.revents);
#endif

  // we are not interested in the GSourceFunc argument here as we have never
  // called g_source_set_callback()
  bool keep_source = true;
  bool connected;
  // provide a CancelBlock here to make this function NPTL friendly,
  // as we have a catch-all without rethrowing
  Cgu::Thread::CancelBlock b;
  try {
    connected = emitter->test_emit(poll_condition, keep_source);
  }
  // we can't propagate exceptions from functions with C linkage.
  catch (...) {
    g_critical("Exception thrown in io_watch_dispatch_func()\n");
    return true;
  }
  return (keep_source && connected);
}

void cgu_io_watch_finalize_func(GSource* source) {
  WatchSource* watch_source = reinterpret_cast<WatchSource*>(source);
  delete static_cast<Cgu::Callback::CallbackArg<bool&>*>(watch_source->func);
  watch_source->func = 0;
}

void cgu_io_watch_finalize_tracked_func(GSource* source) {
  WatchSource* watch_source = reinterpret_cast<WatchSource*>(source);
  delete static_cast<Cgu::SafeEmitterArg<bool&>*>(watch_source->func);
  watch_source->func = 0;
}

void cgu_io_watch_finalize_condition_func(GSource* source) {
  WatchSource* watch_source = reinterpret_cast<WatchSource*>(source);
  delete static_cast<Cgu::Callback::CallbackArg<GIOCondition, bool&>*>(watch_source->func);
  watch_source->func = 0;
}

void cgu_io_watch_finalize_tracked_condition_func(GSource* source) {
  WatchSource* watch_source = reinterpret_cast<WatchSource*>(source);
  delete static_cast<Cgu::SafeEmitterArg<GIOCondition, bool&>*>(watch_source->func);
  watch_source->func = 0;
}

static GSourceFuncs cgu_io_watch_source_funcs = {
  cgu_io_watch_prepare_func,
  cgu_io_watch_check_func,
  cgu_io_watch_dispatch_func,
  cgu_io_watch_finalize_func
};

static GSourceFuncs cgu_io_watch_source_tracked_funcs = {
  cgu_io_watch_prepare_func,
  cgu_io_watch_check_func,
  cgu_io_watch_dispatch_tracked_func,
  cgu_io_watch_finalize_tracked_func
};

static GSourceFuncs cgu_io_watch_source_condition_funcs = {
  cgu_io_watch_prepare_func,
  cgu_io_watch_check_func,
  cgu_io_watch_dispatch_condition_func,
  cgu_io_watch_finalize_condition_func
};

static GSourceFuncs cgu_io_watch_source_tracked_condition_funcs = {
  cgu_io_watch_prepare_func,
  cgu_io_watch_check_func,
  cgu_io_watch_dispatch_tracked_condition_func,
  cgu_io_watch_finalize_tracked_condition_func
};

namespace Cgu {

guint start_iowatch(int fd, const Callback::CallbackArg<bool&>* cb,
		    GIOCondition io_condition, gint priority,
		    GMainContext* context) {

  // context has a default value of NULL which will create the watch
  // in the default program main context

  GSource* source = g_source_new(&cgu_io_watch_source_funcs, sizeof(WatchSource));
  // reinterpret_cast<>() is guaranteed to give the correct result here as the
  // address of WatchSource::source must be the same as the address of the
  // instance of the WatchSource struct of which it is the first member as
  // both are PODSs
  WatchSource* watch_source = reinterpret_cast<WatchSource*>(source);
  watch_source->watch_condition = io_condition;
  watch_source->func = const_cast<void*>(static_cast<const void*>(cb));
#ifdef HAVE_UNIX_FD_BACKEND
  watch_source->tag = g_source_add_unix_fd(source, fd, io_condition);
  watch_source->error_storm_count = 0;
#else
  watch_source->poll_fd.fd = fd;
  watch_source->poll_fd.events = io_condition;
  watch_source->poll_fd.revents = 0;
  g_source_add_poll(source, &watch_source->poll_fd);
#endif

  g_source_set_priority(source, priority);
  guint id = g_source_attach(source, context);

  // g_source_attach() will add a reference count to the GSource object
  // so we unreference it here so that the callback returning false or
  // calling g_source_remove() on the return value of this function will
  // finalize/destroy the GSource object
  g_source_unref(source);

  return id;
}

guint start_iowatch(int fd, const Callback::CallbackArg<bool&>* cb, Releaser& r,
		    GIOCondition io_condition, gint priority,
		    GMainContext* context) {

  // context has a default value of NULL which will create the watch
  // in the default program main context

  Callback::SafeFunctorArg<bool&> f{cb};   // take ownership
  SafeEmitterArg<bool&>* emitter = 0;
  try {
    emitter = new SafeEmitterArg<bool&>;
    emitter->connect(f, r);
  }
  catch (...) {
    delete emitter; // either NULL or object allocated
    throw;
  }

  GSource* source = g_source_new(&cgu_io_watch_source_tracked_funcs, sizeof(WatchSource));
  // reinterpret_cast<>() is guaranteed to give the correct result here as the
  // address of WatchSource::source must be the same as the address of the
  // instance of the WatchSource struct of which it is the first member as
  // both are PODSs
  WatchSource* watch_source = reinterpret_cast<WatchSource*>(source);
  watch_source->watch_condition = io_condition;
  watch_source->func = emitter;
#ifdef HAVE_UNIX_FD_BACKEND
  watch_source->tag = g_source_add_unix_fd(source, fd, io_condition);
  watch_source->error_storm_count = 0;
#else
  watch_source->poll_fd.fd = fd;
  watch_source->poll_fd.events = io_condition;
  watch_source->poll_fd.revents = 0;
  g_source_add_poll(source, &watch_source->poll_fd);
#endif

  g_source_set_priority(source, priority);
  guint id = g_source_attach(source, context);

  // g_source_attach() will add a reference count to the GSource object
  // so we unreference it here so that the callback returning false or
  // calling g_source_remove() on the return value of this function will
  // finalize/destroy the GSource object
  g_source_unref(source);

  return id;
}

guint start_iowatch(int fd, const Callback::CallbackArg<GIOCondition, bool&>* cb,
		    GIOCondition io_condition, gint priority,
		    GMainContext* context) {

  // context has a default value of NULL which will create the watch
  // in the default program main context

  GSource* source = g_source_new(&cgu_io_watch_source_condition_funcs, sizeof(WatchSource));
  // reinterpret_cast<>() is guaranteed to give the correct result here as the
  // address of WatchSource::source must be the same as the address of the
  // instance of the WatchSource struct of which it is the first member as
  // both are PODSs
  WatchSource* watch_source = reinterpret_cast<WatchSource*>(source);
  watch_source->watch_condition = io_condition;
  watch_source->func = const_cast<void*>(static_cast<const void*>(cb));
#ifdef HAVE_UNIX_FD_BACKEND
  watch_source->tag = g_source_add_unix_fd(source, fd, io_condition);
  watch_source->error_storm_count = 0;
#else
  watch_source->poll_fd.fd = fd;
  watch_source->poll_fd.events = io_condition;
  watch_source->poll_fd.revents = 0;
  g_source_add_poll(source, &watch_source->poll_fd);
#endif

  g_source_set_priority(source, priority);
  guint id = g_source_attach(source, context);

  // g_source_attach() will add a reference count to the GSource object
  // so we unreference it here so that the callback returning false or
  // calling g_source_remove() on the return value of this function will
  // finalize/destroy the GSource object
  g_source_unref(source);

  return id;
}

guint start_iowatch(int fd, const Callback::CallbackArg<GIOCondition, bool&>* cb,
		    Releaser& r, GIOCondition io_condition, gint priority,
		    GMainContext* context) {

  // context has a default value of NULL which will create the watch
  // in the default program main context

  Callback::SafeFunctorArg<GIOCondition, bool&> f{cb};   // take ownership
  SafeEmitterArg<GIOCondition, bool&>* emitter = 0;
  try {
    emitter = new SafeEmitterArg<GIOCondition, bool&>;
    emitter->connect(f, r);
  }
  catch (...) {
    delete emitter; // either NULL or object allocated
    throw;
  }

  GSource* source = g_source_new(&cgu_io_watch_source_tracked_condition_funcs, sizeof(WatchSource));
  // reinterpret_cast<>() is guaranteed to give the correct result here as the
  // address of WatchSource::source must be the same as the address of the
  // instance of the WatchSource struct of which it is the first member as
  // both are PODSs
  WatchSource* watch_source = reinterpret_cast<WatchSource*>(source);
  watch_source->watch_condition = io_condition;
  watch_source->func = emitter;
#ifdef HAVE_UNIX_FD_BACKEND
  watch_source->tag = g_source_add_unix_fd(source, fd, io_condition);
  watch_source->error_storm_count = 0;
#else
  watch_source->poll_fd.fd = fd;
  watch_source->poll_fd.events = io_condition;
  watch_source->poll_fd.revents = 0;
  g_source_add_poll(source, &watch_source->poll_fd);
#endif

  g_source_set_priority(source, priority);
  guint id = g_source_attach(source, context);

  // g_source_attach() will add a reference count to the GSource object
  // so we unreference it here so that the callback returning false or
  // calling g_source_remove() on the return value of this function will
  // finalize/destroy the GSource object
  g_source_unref(source);

  return id;
}

} // namespace Cgu
