/* chroot LD_PRELOAD wrapper to safely chroot any application
 *
 * (C) 2004 Henrik Nordstrom <hno@marasystems.com>
 *
 *  Permission is hereby granted, free of charge, to any person obtaining
 *  a copy of this software and associated documentation files (the
 *  "Software"), to deal in the Software without restriction, including
 *  without limitation the rights to use, copy, modify, merge, publish,
 *  distribute, sublicense, and/or sell copies of the Software, and to
 *  permit persons to whom the Software is furnished to do so, subject
 *  to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 *  CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 *  TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 * Usage:
 *
 *  This application is used as a LD_PRELOAD library to make any shared
 *  executable chroot itself and drop privileges in a safe manner after
 *  any dynamic linking has taken place.
 *
 *  A number of environment variables is used to control how the
 *  application is chrooted:
 *
 *  CHROOT_ROOT   The path to where the application is to be chrooted
 *                (required)
 *
 *  CHROOT_USER   The user to run the application as (defaults to "nobody"
 *                if not defined)
 *
 *  LD_PRELOAD    Must include this application
 *
 * Diagnostics:
 *
 *  If there is any problem chrooting the application an error message is
 *  printed on stderr, and execution of the application is terminated.
 *
 * Compiling:
 *
 *  g++ -fpic -shared -o chroot_safe.so chroot_safe.cpp
 *
 * Suggested use:
 *
 *  #!/bin/sh
 *  if [ $# -lt 3 ]; then
 *    echo "Usage: $0 user root application ..." >&2
 *    exit 1
 *  fi
 *  CHROOT_USER="$1"; export CHROOT_USER
 *  CHROOT_ROOT="$2"; export CHROOT_ROOT
 *  shift ; shift
 *  LD_PRELOAD="/path/to/chroot_safe.so"; export LD_PRELOAD
 *  exec "$@"
 */

#include <sys/types.h>
#include <unistd.h>
#include <grp.h>
#include <pwd.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>

class automatic_chroot {
    private:
	static void failure(char *msg) {
	    char *error = strerror(errno);
	    write(2, msg, strlen(msg));
	    if (errno != 0) {
		write(2, ": ", 2);
		write(2, error, strlen(error));
	    }
	    write(2, "\n", 1);
	    _exit(1);
	}
    public:
	automatic_chroot() {
	    putenv("LD_PRELOAD");
	    char *root = getenv("CHROOT_ROOT");
	    if (root == NULL)
		failure("CHROOT_ROOT not defined");
	    char *user = getenv("CHROOT_USER");
	    if (user == NULL)
		user = "nobody";
	    struct passwd *pwd = getpwnam(user);
	    if (pwd == NULL)
		failure("User not found in /etc/passwd");
	    if (setgid(pwd->pw_gid))
		failure("setgid failure");
	    if (initgroups(user, pwd->pw_gid))
		failure("initgroups failure");
	    if (chdir(root))
		failure("chdir failure");
	    if (chroot("."))
		failure("chroot failure");
	    if (setuid(pwd->pw_uid))
		failure("setresuid failure");
	    putenv("CHROOT_ROOT");
	    putenv("CHROOT_USER");
	}
};

static class automatic_chroot do_chroot;
