/*-
 * Copyright (c) 2001, 2002 Lev Walkin <vlm@lionet.info>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: rshp.c,v 1.19 2004/03/11 09:36:00 vlm Exp $
 */

#include "ipcad.h"
#include "cfgvar.h"
#include "storage.h"
#include "csparse.h"
#include "rw.h"
#include "servers.h"
#include "service.h"
#include "opt.h"
#include "rsh.h"
#include "disp.h"

void rsh_usage(FILE *f);
int process_rsh_command(int sock, char *command, int privlevel);

/*
 * Reset the TCP connection and reset_socket socket.
 */
static void
reset_socket(int sock) {
	struct linger lng = { 1, 0 };
	setsockopt(sock, SOL_SOCKET, SO_LINGER, &lng, sizeof(lng));
	close(sock);
}

static int
getstr(char *buf, size_t bufsize, int sock) {
	char c;
	ssize_t sz;

	*buf = 0;

	do {
		sz = read_timeout(sock, &c, 1, rsh_rw_timeout);

		if(signoff_now)
			return -1;

		if(sz != 1)
			return -1;

		*buf++ = c;

		if (--bufsize == 0)
			return -1;

	} while (c != 0);

	return 0;
}



/*
** remote shell thread:
** [port]\0
** remuser\0
** locuser\0
** command\0
** data
*/

int
process_rsh_request(int client) {
	struct sockaddr_in srem;
	socklen_t srem_len = sizeof(srem);
	char port_s[16];
	char remuser[16];
	char locuser[16];
	char command[256];
	int port;
	int rsock = -1;
	int privlevel;
	int ret;
	char c;

	/*
	 * Figure out peer's address.
	 */
	if(getpeername(client, (struct sockaddr *)&srem, &srem_len)) {
		reset_socket(client);
		return -1;
	}

	if( getstr(port_s, sizeof(port_s), client) ) {
		reset_socket(client);
		return -1;
	}

	if(*port_s) {

		port = atoi(port_s);
		if(port >= 1024 || port < 1) {
			reset_socket(client);
			return -1;
		}

		srem.sin_port = htons(port);

		port = IPPORT_RESERVED - 1;
		if( (rsock = rresvport(&port)) == -1) {
			reset_socket(client);
			return -1;
		}
	
		if( connect(rsock, (struct sockaddr *)&srem,
			sizeof(struct sockaddr_in)) == -1) {
			reset_socket(rsock);
			reset_socket(client);
			return -1;
		}

	} /* if(*port_s) */;

	if( getstr(remuser, sizeof(remuser), client) ) {
		if(rsock != -1)
			reset_socket(rsock);
		reset_socket(client);
		return -1;
	}

	if( getstr(locuser, sizeof(locuser), client) ) {
		if(rsock != -1)
			reset_socket(rsock);
		reset_socket(client);
		return -1;
	}

	if( getstr(command, sizeof(command), client) ) {
		if(rsock != -1)
			reset_socket(rsock);
		reset_socket(client);
		return -1;
	}


	c = '\0';
	safe_write(client, &c, 1);

	if( (privlevel = cfg_check_rsh(remuser, &srem.sin_addr)) == 0 ) {
		char msg[] = "Permission denied\n";
		safe_write(client, msg, sizeof(msg) - 1);
		if(rsock != -1)
			reset_socket(rsock);
		reset_socket(client);
		return -1;
	}

	/*
	 * This function will automatically do reset_socket(client);
	 */
	ret = process_rsh_command(client, command, privlevel);
	if(rsock != -1)
		reset_socket(rsock);
	return ret;
}

static int
check_permission(int privlevel, int operation) {

	/* Priviledge level required by operation */
	operation >>= 4;

	if(privlevel < operation)
		return 0;

	return 1;
}

#define	RETURN_FAIL	do { ret = -1; goto finish; } while(0)
#define	RETURN_OK	do { ret = 0; goto finish; } while(0)

int
process_rsh_command(int sock, char *command, int privlevel) {
	FILE *f;
	char *param = NULL;
	int Operation = 0;
	int ret = -1;	/* Default is failure */

	if(!(f=fdopen(sock, "w"))) {
		char msg[] = "Internal error.\n";
		safe_write(sock, msg, sizeof(msg) - 1);
		close(sock);	/* Close, not reset_socket()! */
		return -1;
	}

	Operation = csparse(command, &param);

	/* Check command against privlevel */

	if(abs(Operation) < CS_HELP) {
		if(param)
			free(param);
		fprintf(f, "Syntax error. Type 'help' for usage.\n");
		RETURN_FAIL;
	}

	if(!check_permission(privlevel, Operation)) {
		fprintf(f, "Permission denied\n");
		if(param)
			free(param);
		RETURN_FAIL;
	}

	switch(Operation) {
	case CS_HELP:
		rsh_usage(f);
		RETURN_OK;
	case CS_STAT:
		show_stats(f);
		RETURN_OK;
	case CS_SVER:
		show_version(f);
		RETURN_OK;
	case CS_DUMP:
		if(!param)
			param = conf->dump_table_file;

		if(param) {
			make_dump(param, f);
			if(param != conf->dump_table_file)
				free(param);
		} else {
			fprintf(f, "Dump file is not defined.\n");
			RETURN_FAIL;
		}
		RETURN_OK;
	case CS_IMPT:
		if(!param)
			param = conf->dump_table_file;

		if(param) {
			import_table(param, f, 0);
			if(param != conf->dump_table_file)
				free(param);
		} else {
			fprintf(f, "Dump file is not defined.\n");
			RETURN_FAIL;
		}
		RETURN_OK;
	case CS_REST:
		if(!param)
			param = conf->dump_table_file;

		if(param) {
			import_table(param, f, 1);
			if(param != conf->dump_table_file)
				free(param);
		} else {
			fprintf(f, "Dump file is not defined.\n");
			RETURN_FAIL;
		}
		RETURN_OK;
	case CS_SIA:	/* show ip accounting */
		display(f, DS_ACTIVE);
		RETURN_OK;
	case CS_SIAC:	/* show ip accounting checkpoint */
		display(f, DS_CHECKPOINT);
		RETURN_OK;
	case CS_SICF:	/* show ip cache flow */
		display(f, DS_NETFLOW);
		RETURN_OK;
	case CS_SIN:	/* show interface xxx */
		if_stat(f, param);
		free(param);
		RETURN_OK;
	case CS_CIAC:	/* CLEAR ... */
		clear_storage(&checkpoint_storage, 0);

		fprintf(f, "IP accounting checkpoint cleared\n");

		RETURN_OK;
	case CS_CIA:	/* clear ip accounting */
		save_checkpoint(&active_storage, &checkpoint_storage);

		fprintf(f, "IP accounting cleared\n");

		RETURN_OK;
	case CS_SHUT:
		fprintf(f, "Shutdown process started\n");
		signoff_now = 1;
		RETURN_OK;
	default:
		fprintf(f, "Invalid request.\n");
		RETURN_FAIL;
	}

finish:

	fflush(f);
	if(0) {
		struct linger lng = { 1, 0 };
		/*
		 * shutdown() has a side effect of flushing network buffers
		 * on some platforms
		 */
		shutdown(sock, SHUT_RDWR);
		/*
		 * Make sure a TCP reset is issued for TIME_WAIT elimination.
		 */
		setsockopt(sock, SOL_SOCKET, SO_LINGER, &lng, sizeof(lng));
	}
	fclose(f);
	return ret;
}

