/*-
 * Copyright (c) 2004 Lev Walkin <vlm@lionet.info>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: psrc-divert.c,v 1.8 2004/07/27 09:49:00 vlm Exp $
 */

#include "ipcad.h"
#include "opt.h"
#include "cfgvar.h"
#include <assert.h>

#ifndef	IPPROTO_DIVERT
int
init_packet_source_divert(packet_source_t *ps) {
	(void)ps;
	fprintf(stderr, "BSD divert(4)/tee source support "
		"is not properly compiled in "
		"or not supported by the target OS, "
		"please contact the ipcad vendor.\n");
	errno = EINVAL;
	return -1;
}

#else	/* IPPROTO_DIVERT */

int
init_packet_source_divert(packet_source_t *ps) {
	struct sockaddr_in sin;
	int divert_fd;

	assert(ps->state != PST_INVALID);	/* Embryonic or Ready */
	assert(ps->iface_type == IFACE_DIVERT
		|| ps->iface_type == IFACE_TEE);	/* Don't cook crap */

	/*
	 * Open divert socket.
	 */
	divert_fd = ps->fd;
	if(divert_fd == -1) {
		divert_fd = socket(PF_INET, SOCK_RAW, IPPROTO_DIVERT);
		if(divert_fd == -1) return -1;
	} else {
		ps->state = PST_EMBRYONIC;
	}
	
	/*
	 * Bind to the divert(4) packet source.
	 */
	memset(&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_port = htons(ps->iface.divert.port);

	if(bind(divert_fd, (struct sockaddr *)&sin, sizeof(sin))) {
		if(ps->fd == -1)
			close(divert_fd);
		return -1;
	}
	
	ps->dlt = DLT_RAW;

	/*
	 * Finish initialization of the structure.
	 */
	ps->fd = divert_fd;
	ps->state = PST_READY;

	return 0;
}

void
print_stats_divert(FILE *f, packet_source_t *ps) {

	assert(ps->iface_type == IFACE_DIVERT
		|| ps->iface_type == IFACE_TEE);

	fprintf(f, "Interface %s: received ??", IFNameBySource(ps));

	fprintf(f, ", %.0f m average %lld bytes/sec, %lld pkts/sec",
		ps->avg_period / 60,
		ps->bps_lp,
		ps->pps_lp
	);
	
	fprintf(f, ", dropped ??"); 
	fprintf(f, "\n");
}

#endif	/* IPPROTO_DIVERT */
