/*
    ifdhandler.c

    Trivial implementation of wrapper functions for the CT-API 
    conforming to MUSCLE PCSC/Lite IFD Handler v2.0 definition
    by David Corcoran.
    
    This file is part of the MUSCLE driver for Towitoko smartcard readers
    Copyright (C) 1998 1999 2000 Carlos Prados (cprados@yahoo.com)

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "pcscdefines.h"
#include "ifdhandler.h"
#include <ctapidefines.h>
#include <ctapi.h>
#include <ctbcs.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif

/*
 * Not exported constants definition
 */

/* Maximum number of readers handled */
#define IFDH_MAX_READERS        4

/* Maximum number of slots per reader handled */
#define IFDH_MAX_SLOTS          2

/*
 * Not exported data types definition
 */

typedef struct
{
  DEVICE_CAPABILITIES device_capabilities;
  ICC_STATE icc_state;
  PROTOCOL_OPTIONS protocol_options;
}
IFDH_Status;

/*
 * Not exported variables definition
 */

/* Matrix that stores status information of all slots and readers */
static IFDH_Status *ifdh_status[IFDH_MAX_READERS][IFDH_MAX_SLOTS] = {
  {NULL, NULL},
  {NULL, NULL},
  {NULL, NULL},
  {NULL, NULL},
};

#ifdef HAVE_PTHREAD_H
static pthread_mutex_t ifdh_status_mutex[IFDH_MAX_READERS] = {
  PTHREAD_MUTEX_INITIALIZER,
  PTHREAD_MUTEX_INITIALIZER,
  PTHREAD_MUTEX_INITIALIZER,
  PTHREAD_MUTEX_INITIALIZER
};
#endif

/* 
 * Exported functions definition
 */

RESPONSECODE IFDHCreateChannel (DWORD Lun, DWORD Channel)
{
  char ret;
  unsigned short ctn, pn, slot;
  RESPONSECODE rv;

  /*
     printf ("Lun %X, Channel %d\n", Lun, Channel);
   */

  ctn = ((unsigned short) (Lun & 0x0000FFFF)) % IFDH_MAX_READERS;
  slot = ((unsigned short) (Lun >> 8)) % IFDH_MAX_SLOTS;

#ifdef HAVE_PTHREAD_H
  pthread_mutex_lock (&ifdh_status_mutex[ctn]);
#endif

  if (ifdh_status[ctn][slot] == NULL)
    {
      /* Conversion of old-style ifd-hanler 1.0 CHANNELID */
      if (Channel == 0x0103F8)
        Channel = 0x000001;
      else if (Channel == 0x0102F8)
        Channel = 0x000002;
      else if (Channel == 0x0103E8)
        Channel = 0x000003;
      else if (Channel == 0x0102E8)
        Channel = 0x000004;

      pn = (unsigned short) Channel - 1;
      ret = CT_init (ctn, pn);

      if (ret == OK)
        {

          for (slot = 0; slot < IFDH_MAX_SLOTS; slot++)
            {
              ifdh_status[ctn][slot] =
                (IFDH_Status *) malloc (sizeof (IFDH_Status));

              if (ifdh_status[ctn][slot] != NULL)
                memset (ifdh_status[ctn][slot], 0, sizeof (IFDH_Status));
            }
          rv = IFD_SUCCESS;
        }

      else
        rv = IFD_COMMUNICATION_ERROR;
    }

  else
    rv = IFD_SUCCESS;

#ifdef HAVE_PTHREAD_H
  pthread_mutex_unlock (&ifdh_status_mutex[ctn]);
#endif

  return rv;
}

RESPONSECODE
IFDHCloseChannel (DWORD Lun)
{
  char ret;
  unsigned short ctn, slot;
  RESPONSECODE rv;

  ctn = ((unsigned short) (Lun & 0x0000FFFF)) % IFDH_MAX_READERS;
  ret = CT_close (ctn);

  if (ret == OK)
    {
#ifdef HAVE_PTHREAD_H
      pthread_mutex_lock (&ifdh_status_mutex[ctn]);
#endif
      for (slot = 0; slot < IFDH_MAX_SLOTS; slot++)
        {
          if (ifdh_status[ctn][slot] != NULL)
            {
              free (ifdh_status[ctn][slot]);
              ifdh_status[ctn][slot] = NULL;
            }
        }
#ifdef HAVE_PTHREAD_H
      pthread_mutex_unlock (&ifdh_status_mutex[ctn]);
#endif
      rv = IFD_SUCCESS;
    }

  else
    rv = IFD_COMMUNICATION_ERROR;

  return rv;
}

RESPONSECODE
IFDHGetCapabilities (DWORD Lun, DWORD Tag, PDWORD Length, PUCHAR Value)
{
  unsigned short ctn, slot;
  RESPONSECODE rv;

  ctn = ((unsigned short) (Lun & 0x0000FFFF)) % IFDH_MAX_READERS;
  slot = ((unsigned short) (Lun >> 8)) % IFDH_MAX_SLOTS;

#ifdef HAVE_PTHREAD_H
  pthread_mutex_lock (&ifdh_status_mutex[ctn]);
#endif

  if (ifdh_status[ctn][slot] != NULL)
    {
      if (Tag == TAG_IFD_ATR)
        {
          (*Length) = ifdh_status[ctn][slot]->icc_state.ATR_Length;
          memcpy (Value, ifdh_status[ctn][slot]->icc_state.ATR, (*Length));

          rv = IFD_SUCCESS;
        }

      else
        rv = IFD_ERROR_TAG;
    }
  else
    rv = IFD_ICC_NOT_PRESENT;

#ifdef HAVE_PTHREAD_H
  pthread_mutex_unlock (&ifdh_status_mutex[ctn]);
#endif

  return rv;
}

RESPONSECODE
IFDHSetCapabilities (DWORD Lun, DWORD Tag, DWORD Length, PUCHAR Value)
{
  return IFD_NOT_SUPPORTED;
}

RESPONSECODE
IFDHSetProtocolParameters (DWORD Lun, DWORD Protocol,
                           UCHAR Flags, UCHAR PTS1, UCHAR PTS2, UCHAR PTS3)
{
  return IFD_NOT_SUPPORTED;
}


RESPONSECODE
IFDHPowerICC (DWORD Lun, DWORD Action, PUCHAR Atr, PDWORD AtrLength)
{
  char ret;
  unsigned short ctn, slot, lc, lr;
  UCHAR cmd[5], rsp[256], sad, dad;
  RESPONSECODE rv;

  ctn = ((unsigned short) (Lun & 0x0000FFFF)) % IFDH_MAX_READERS;
  slot = ((unsigned short) (Lun >> 8)) % IFDH_MAX_SLOTS;

#ifdef HAVE_PTHREAD_H
  pthread_mutex_lock (&ifdh_status_mutex[ctn]);
#endif

  if (ifdh_status[ctn][slot] != NULL)
    {
      if (Action == IFD_POWER_UP)
        {
          cmd[0] = CTBCS_CLA;
          cmd[1] = CTBCS_INS_REQUEST;
          cmd[2] = (UCHAR) (slot + 1);
          cmd[3] = CTBCS_P2_REQUEST_GET_ATR;
          cmd[4] = 0x00;

          dad = 0x01;
          sad = 0x02;
          lr = 256;
          lc = 5;

          ret = CT_data (ctn, &dad, &sad, 5, cmd, &lr, rsp);

          if ((ret == OK) && (lr >= 2))
            {
              ifdh_status[ctn][slot]->icc_state.ATR_Length = (DWORD) lr - 2;
              memcpy (ifdh_status[ctn][slot]->icc_state.ATR, rsp, lr - 2);

              (*AtrLength) = (DWORD) lr - 2;
              memcpy (Atr, rsp, lr - 2);

              rv = IFD_SUCCESS;
            }

          else
            rv = IFD_COMMUNICATION_ERROR;
        }

      else if (Action == IFD_POWER_DOWN)
        {
          cmd[0] = CTBCS_CLA;
          cmd[1] = CTBCS_INS_EJECT;
          cmd[2] = (UCHAR) (slot + 1);
          cmd[3] = 0x00;
          cmd[4] = 0x00;

          dad = 0x01;
          sad = 0x02;
          lr = 256;
          lc = 5;

          ret = CT_data (ctn, &dad, &sad, 5, cmd, &lr, rsp);

          if (ret == OK)
            {
              ifdh_status[ctn][slot]->icc_state.ATR_Length = 0;
              memset (ifdh_status[ctn][slot]->icc_state.ATR, 0, MAX_ATR_SIZE);

              (*AtrLength) = 0;
              rv = IFD_SUCCESS;
            }

          else
            rv = IFD_COMMUNICATION_ERROR;
        }

      else if (Action == IFD_RESET)
        {
          cmd[0] = CTBCS_CLA;
          cmd[1] = CTBCS_INS_RESET;
          cmd[2] = (UCHAR) (slot + 1);
          cmd[3] = CTBCS_P2_RESET_GET_ATR;
          cmd[4] = 0x00;

          dad = 0x01;
          sad = 0x02;
          lr = 256;
          lc = 5;

          ret = CT_data (ctn, &dad, &sad, 5, cmd, &lr, rsp);

          if ((ret == OK) && (lr >= 2))
            {
              ifdh_status[ctn][slot]->icc_state.ATR_Length = (DWORD) lr - 2;
              memcpy (ifdh_status[ctn][slot]->icc_state.ATR, rsp, lr - 2);

              (*AtrLength) = (DWORD) lr - 2;
              memcpy (Atr, rsp, lr - 2);

              rv = IFD_SUCCESS;
            }

          else
            rv = IFD_ERROR_POWER_ACTION;
        }

      else
        rv = IFD_NOT_SUPPORTED;
    }

  else
    rv = IFD_ICC_NOT_PRESENT;

#ifdef HAVE_PTHREAD_H
  pthread_mutex_unlock (&ifdh_status_mutex[ctn]);
#endif

  return rv;
}

RESPONSECODE
IFDHTransmitToICC (DWORD Lun, SCARD_IO_HEADER SendPci,
                   PUCHAR TxBuffer, DWORD TxLength,
                   PUCHAR RxBuffer, PDWORD RxLength, PSCARD_IO_HEADER RecvPci)
{
  char ret;
  unsigned short ctn, slot, lc, lr;
  UCHAR sad, dad;
  RESPONSECODE rv;

  ctn = ((unsigned short) (Lun & 0x0000FFFF)) % IFDH_MAX_READERS;
  slot = ((unsigned short) (Lun >> 8)) % IFDH_MAX_SLOTS;

#ifdef HAVE_PTHREAD_H
  pthread_mutex_lock (&ifdh_status_mutex[ctn]);
#endif

  if (ifdh_status[ctn][slot] != NULL)
    {
#ifdef HAVE_PTHREAD_H
      pthread_mutex_unlock (&ifdh_status_mutex[ctn]);
#endif
      dad = (UCHAR) ((slot == 0) ? 0x00 : slot + 1);
      sad = 0x02;
      lr = (unsigned short) (*RxLength);
      lc = (unsigned short) TxLength;

      ret = CT_data (ctn, &dad, &sad, lc, TxBuffer, &lr, RxBuffer);

      if (ret == OK)
        {
          (*RxLength) = lr;
          rv = IFD_SUCCESS;
        }

      else
        {
          (*RxLength) = 0;
          rv = IFD_COMMUNICATION_ERROR;
        }
    }

  else
    {
#ifdef HAVE_PTHREAD_H
      pthread_mutex_unlock (&ifdh_status_mutex[ctn]);
#endif
      rv = IFD_ICC_NOT_PRESENT;
    }

  return rv;
}

RESPONSECODE
IFDHControl (DWORD Lun, PUCHAR TxBuffer,
             DWORD TxLength, PUCHAR RxBuffer, PDWORD RxLength)
{
  char ret;
  unsigned short ctn, slot, lc, lr;
  UCHAR sad, dad;
  RESPONSECODE rv;

  ctn = ((unsigned short) (Lun & 0x0000FFFF)) % IFDH_MAX_READERS;
  slot = ((unsigned short) (Lun >> 8)) % IFDH_MAX_SLOTS;

#ifdef HAVE_PTHREAD_H
  pthread_mutex_lock (&ifdh_status_mutex[ctn]);
#endif

  if (ifdh_status[ctn][slot] != NULL)
    {
#ifdef HAVE_PTHREAD_H
      pthread_mutex_unlock (&ifdh_status_mutex[ctn]);
#endif
      dad = 0x01;
      sad = 0x02;
      lr = (unsigned short) (*RxLength);
      lc = (unsigned short) TxLength;

      ret = CT_data (ctn, &dad, &sad, lc, TxBuffer, &lr, RxBuffer);

      if (ret == OK)
        {
          (*RxLength) = lr;
          rv = IFD_SUCCESS;
        }
      else
        {
          (*RxLength) = 0;
          rv = IFD_COMMUNICATION_ERROR;
        }
    }

  else
    {
#ifdef HAVE_PTHREAD_H
      pthread_mutex_unlock (&ifdh_status_mutex[ctn]);
#endif
      rv = IFD_ICC_NOT_PRESENT;
    }

  return rv;
}

RESPONSECODE
IFDHICCPresence (DWORD Lun)
{
  char ret;
  unsigned short ctn, slot, lc, lr;
  UCHAR cmd[5], rsp[256], sad, dad;
  RESPONSECODE rv;

  ctn = ((unsigned short) (Lun & 0x0000FFFF)) % IFDH_MAX_READERS;
  slot = ((unsigned short) (Lun >> 8)) % IFDH_MAX_SLOTS;

  cmd[0] = CTBCS_CLA;
  cmd[1] = CTBCS_INS_STATUS;
  cmd[2] = CTBCS_P1_CT_KERNEL;
  cmd[3] = CTBCS_P2_STATUS_ICC;
  cmd[4] = 0x00;

  dad = 0x01;
  sad = 0x02;
  lc = 5;
  lr = 256;

  ret = CT_data (ctn, &dad, &sad, lc, cmd, &lr, rsp);

  if (ret == OK)
    {
      if (slot < lr - 2)
        {
          if (rsp[slot] == CTBCS_DATA_STATUS_NOCARD)
            rv = IFD_ICC_NOT_PRESENT;
          else
            rv = IFD_ICC_PRESENT;
        }
      else
        rv = IFD_ICC_NOT_PRESENT;
    }
  else
    rv = IFD_COMMUNICATION_ERROR;

  return rv;
}
