// Display images inside a terminal
// Copyright (C) 2023  JustKidding
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#ifndef STDOUT_CANVAS_H
#define STDOUT_CANVAS_H

#include "canvas.hpp"
#include "image.hpp"
#include "window.hpp"

#include <spdlog/spdlog.h>

#include <memory>
#include <mutex>
#include <string>
#include <unordered_map>

template <WindowType T>
class StdoutCanvas : public Canvas
{
  public:
    explicit StdoutCanvas(const std::string &output)
    {
        logger = spdlog::get(output);
        logger->info("Canvas created");
    }

    ~StdoutCanvas() override = default;

    void add_image(const std::string &identifier, std::unique_ptr<Image> new_image) override
    {
        logger->info("Displaying image with id {}", identifier);
        images.erase(identifier);
        const auto [entry, success] =
            images.emplace(identifier, std::make_unique<T>(std::move(new_image), &stdout_mutex));
        entry->second->draw();
    }

    void remove_image(const std::string &identifier) override
    {
        logger->info("Removing image with id {}", identifier);
        images.erase(identifier);
    }

  private:
    std::mutex stdout_mutex;
    std::shared_ptr<spdlog::logger> logger;
    std::unordered_map<std::string, std::unique_ptr<T>> images;
};

#endif
