/* awale.c -- plays the african game.
   Copyright (C) 2006 Alain Le Bot and Laurent Le Bot.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
   USA  */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <getopt.h>

#define VERSION "1.6"
#define max(A,B) ( (A) > (B) ? (A):(B))
#define min(A,B) ( (A) > (B) ? (B):(A))
#define who_plays(i) ((i % 12) / 6)   //Return the player (0 ou 1) of hole i.
#define who_waits(i) (1-(i % 12) / 6) //Return the opponent (0 ou 1) of hole i.

struct option longopts[] = {
  {"help", no_argument, NULL, 'h'},
  {"version", no_argument, NULL, 'v'},
  {"rule", no_argument, NULL, 'r'},
  {"board", required_argument, NULL, 'b'},
  {"score", required_argument, NULL, 'o'},
  {"computer", required_argument, NULL, 'c'},
  {"player", required_argument, NULL, 'p'},
  {"depth", required_argument, NULL, 'd'},
  {"level", required_argument, NULL, 'l'},
  {"eat", required_argument, NULL, 'e'},
  {"noeatmore", no_argument, NULL, 'm'},
  {"noeatall", no_argument, NULL, 'a'},
  {"noattackall", no_argument, NULL, 'k'},
  {"single", no_argument, NULL, 's'},
  {"hint", no_argument, NULL, 'i'},
  {"xawale", no_argument, NULL, 'x'},
  {NULL, 0, NULL, 0}
};

static char *progname;
int flagh = 0, flagv = 0, flagr = 0, flagb = 0, flago = 0, flagc = 0;
int flagp = 0, flagd = 0, flagl = 0, flage = 0, flagm = 0, flaga = 0;
int flagk = 0, flags = 0, flagi = 0, flagx = 0, lose = 0;
int depth = 9;                // initial depth of the min-max algorithm
int level = 0;                // level of game
int neh = 4;                  // max number of eaten holes
int computer = 1, human = 0;  // computer plays top line (1) by default
double tmax[3] = {2, 20, 100};// max cpu time of levels 1 to 3

int
main (int argc, char *argv[])
{
  int optc;
  void print_board (int *, int *, int, char *);
  void exit_game (char *);
  int best_move (int *, int *, char *);
  int ask_move (int*, int, char *);
  int distribute (int *, int *, int, char *, int);
  int hole[12] = { 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4 };
  int score[2] = { 0, 0 };
  char label[4] = "BEG";
  int i, j, del[12], val[12], move = -1;
  char response[64];

  
  progname = argv[0];

  while ((optc =
	  getopt_long (argc, argv, "hvrb:o:c:p:d:l:e:maksix", longopts,
		       (int *) 0)) != -1)
    switch (optc)
      {
      case 'h':
	flagh = 1;
	break;
      case 'v':
	flagv = 1;
	break;
      case 'r':
	flagr = 1;
	break;
      case 'b':
	flagb = 1;
	if (sscanf
	    (optarg, "%d-%d-%d-%d-%d-%d-%d-%d-%d-%d-%d-%d", hole, hole + 1,
	     hole + 2, hole + 3, hole + 4, hole + 5, hole + 6, hole + 7,
	     hole + 8, hole + 9, hole + 10, hole + 11) != 12)
	  {
	    fprintf (stderr, "Bad argument for -b option\n");
	    exit (1);
	  }
	break;
      case 'o':
	flago = 1;
	if (sscanf (optarg, "%d-%d", score, score + 1) != 2)
	  {
	    fprintf (stderr, "Bad argument for -o option\n");
	    exit (1);
	  }
	break;
      case 'c':
	flagc = 1;
	if (sscanf (optarg, "%d", &computer) != 1 || computer < 1 || computer > 2)
	  {
	    fprintf (stderr, "Bad argument for -c option\n");
	    exit (1);
	  }
	computer--;
	human = 1 - computer;
	break;
      case 'p':
	flagp = 1;
	if (sscanf (optarg, "%d", &move) != 1 || move < 1 || move > 12)
	  {
	    fprintf (stderr, "Bad argument for -p option\n");
	    exit (1);
	  }
	move--;
	human    = who_plays (move);
	computer = who_waits (move);
	break;
      case 'd':
	flagd = 1;
	if (sscanf (optarg, "%d", &depth) != 1)
	  {
	    fprintf (stderr, "Bad argument for -d option\n");
	    exit (1);
	  }
	break;
      case 'l':
	flagl = 1;
	if (sscanf (optarg, "%d", &level) != 1 || level < 1 || level > 3)
	  {
	    fprintf (stderr, "Bad argument for -l option\n");
	    exit (1);
	  }
	level--;
	break;
      case 'e':
	flage = 1;
	if (sscanf (optarg, "%d", &neh) != 1 || neh < 1 || neh > 6)
	  {
	    fprintf (stderr, "Bad argument for -e option\n");
	    exit (1);
	  }
	level--;
	break;
      case 'm':
	flagm = 1;
	break;
	  case 'k':
    flagk = 1;
    break;
      case 'a':
        flaga = 1;
        break;
      case 's':
	flags = 1;
	break;
      case 'i':
	flagi = 1;
	break;
      case 'x':
	flagx = 1;
	break;
      default:
	lose = 1;
	break;
      }


/* invalid argument or too many arguments */
  if (lose || optind < argc)
    {
      if (optind < argc)
	fprintf (stderr, "Too many arguments\n");
      fprintf (stderr, "Try `%s --help' for more information.\n", progname);
      exit (1);
    }


/* help */
  if (flagh)
    {
      printf ("free awale.\n\n");
      printf ("Usage: %s [OPTION] \n\n", progname);
      printf ("\
  -h, --help      display this help and exit\n\
  -v, --version   display version information and exit\n\
  -r, --rule      display rules of awale\n\n\
  -b BOARD, --board=BOARD\n\
                  start with BOARD. BOARD is a string of type\n\
                  h1-h2- ... h12 where hi is the number of seeds in hole i\n\
  -o SCORE, --score=SCORE\n\
                  start with SCORE. SCORE is a string of type\n\
                  s1-s2 where sj is the score of player j\n\
  -c J, --computer=J\n\
                  computer begins and plays the line J=1 or 2\n\
  -p I, --player=I\n\
                  player begins and first plays the hole I=1...12\n\
  -d N, --depth=N  initial depth N of the algorithm (default N=9)\n\
  -l L, --level=L level L=1, 2 or 3 of game (default L=1)\n\
  -e E, --eat=E   maximum number of holes which can be eaten (default E=4)\n\
  -m, --noeatmore eat nothing when trying to eat more than E holes\n\
  -a, --noeatall  eat nothing when trying to eat all opponent seeds\n\
  -k, --noattackall forbiden to attack all opponent seeds excepted if this is the\n\
                  only possible move\n\
  -s, --single    process a single move\n\
  -i, --hint      print hints\n\
  -x  --xawale    start in engine mode. A several line output is provided at\n\
                  each turn. Each line is, \n\
                      board=BOARD score=SCORE move=I LABEL \n\
                  where I is the last hole played, LABEL may be COM if the turn \n\
                  is still in computing, RUN if the turn is finished but the game  \n\
                  is still running, END if the game is over or ERR if the played \n\
		  hole is invalid.\n\n");
      printf ("Report bugs to <alain.le-bot@ec-lyon.fr>.\n");

      exit (0);
    }


/* version */
  if (flagv)
    {
      printf ("free awale %s\n", VERSION);
      printf ("Written by Alain and Laurent Le Bot, Copyright (C) 2006\n\n\
There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n\
PARTICULAR PURPOSE.  You may redistribute copies of awale under the terms\n\
of the GNU General Public License.\n\
For more information about these matters, see the file named COPYING.\n");

      exit (0);
    }

/* rule */
  if (flagr)
    {
      printf ("Rules of awale.\n\n\
Awale game contains two lines of six holes. Each player plays a single line.\n\n\
Every holes contain four seeds at the beginning of the game.\n\n\
One plays alternately. One never plays two turns, one cannot pass his turn.\n\n\
When playing, take all the seeds of a hole in your line and distribute it\n\
in successive holes turning counterclock wise, one seed per hole. Do not\n\
put a seed in the starting hole (this case arrives when the starting hole\n\
contains more than twelve seeds.)\n\n\
The seeds of the arrival hole are eaten when the arrival hole lies in the\n\
opponent line and contains two or three seeds (after all seeds are distri-\n\
buted). All contiguous previous holes which meet this condition are also\n\
eaten. It is possible to eat a maximum of four holes, when five holes are\n\
eatable, only the last four ones are eaten.\n\n\
Players have to play such that the opponent always has at least one seed.\n\
In case it is impossible, the game stops.\n\n\
The winner is the player who has eaten the most seeds.\n\n\
VARIANTS:\n\
It exists many variants in Africa. Most of them can be played by configuring\n\
the rule with the four following parameters.\n\
- the maximum number E of holes that can be eaten can be modified (E=1 to 6).\n\
- instead of eating the last E holes when attacking more, you eat nothing.\n\
- you can forbid any move attacking all opponent seeds excepted if this is the\n\
  only possible move.\n\
- in case all opponent seeds are attacked, nothing is eaten.\n");

      exit (0);
    }



/* incompatible options -c and -p */
  if (flagc && flagp)
    {
      fprintf (stderr, "%s: Incompatible flags: -c and -p\n", progname);
      exit (1);
    }

/*cccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! Starting.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccc*/    

  if (!flagx && !flagl)
     {
      do
        {
          printf("Game level <1-3> ");
          scanf ("%s", response);
	}
      while (sscanf (response, "%d", &level) != 1 || level < 1 || level > 3);
      level--;                            // level ranges from 0 to 2
     }

    
  if (!flagx && !flagp && !flagc)
      do
        {
          printf("Begin <1> or no <0> ");
          scanf ("%s", response);
	}
      while (sscanf (response, "%d", &flagp) != 1 || flagp < 0 || flagp > 1);

    
/*cccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! Main loop.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccc*/

  if (!flagx)
    print_board (hole, score, 0, label);

  do
    {
    
      if (!flagp || strcmp(label, "BEG"))
        {
         move = best_move (hole, score, label);
         distribute (hole, score, move, label, 1);
         print_board (hole, score, move, label);
         if (!strcmp (label, "END") || flags) exit_game (label);
        }
        
      do
        {
          move = ask_move (hole, move, label);
          if (strcmp(label, "ERR"))
            distribute (hole, score, move, label, 1);
          print_board (hole, score, move, label);
         }
       while (!strcmp(label, "ERR") && !flags);
       if (!strcmp (label, "END") || flags) exit_game (label);

    }
  while (1);

}


/*cccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! Exit message.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccc*/

void
exit_game (char * label)
{
  if (!flagx)
    {
      printf ("Bye\n");
    }
  exit (0);
}


/*cccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! Ask for a move and return the move.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccc*/

int
ask_move (int hole[], int move, char* label)
{
  int valid (int *, int);
  void exit_game (char *);
  char r[64];

  if (!strcmp(label,"BEG") && flagp && move != -1)
    {
      if (!flagx)
        printf ("Your move is: %d\n", move+1);
    }
  else
    {
      do
        {
          if (!flagx)
            printf ("Your move is: ");
          scanf ("%s", r);
          if (!strcmp (r, "quit") || !strcmp (r, "exit") || !strcmp (r, "stop") || !strcmp (r, "q"))
    	    exit_game (label);
        }
      while (sscanf (r, "%d", &move) != 1 || move < (human * 6 + 1) || move > (human * 6 + 6));
      move--;
    }
     
  if (!valid (hole, move)) 
    {
     strcpy(label, "ERR");
     if (!flagx) printf("Invalid move\n");
    }
  else
    {
     strcpy(label, "RUN");
    }
  
  return move;

}


/*cccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! Compute and return the best move.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccc*/

int
best_move (int hole[], int score[], char *label)
{
  int valid (int *, int);
  int delay (int *, int *, int);
  int value (int *, int *, int, int);
  int distribute (int *, int *, int, char *, int);
  int move;
  int supv = -100, supr = -100;
  int val[12];
  int del[12];
  int j, h;
  clock_t t1, t2;
  double duree;

  if (hole[0] == 4 && hole[1] == 4 && hole[2] == 4 && hole[3] == 4
      && hole[4] == 4 && hole[5] == 4 && hole[6] == 4 && hole[7] == 4
      && hole[8] == 4 && hole[9] == 4 && hole[10] == 4 && hole[11] == 4)
    {
      srand ((int) time (NULL));
      h = rand () % 100;
      if (h >= 00 && h < 20)
	move = 5 + computer * 6;
      if (h >= 20 && h < 40)
	move = 4 + computer * 6;
      if (h >= 40 && h < 60)
	move = 3 + computer * 6;
      if (h >= 60 && h < 80)
	move = 2 + computer * 6;
      if (h >= 80 && h < 95)
	move = 1 + computer * 6;
      if (h >= 95 && h < 100)
	move = 0 + computer * 6;
    }
  else
    {
      t1 = clock ();

      for (j = computer * 6; j <= computer * 6 + 5; j++)
	if (valid (hole, j))
	  {
	    val[j] = value (hole, score, j, 0);
	    supv = max (val[j], supv);
	  }

      for (j = computer * 6; j <= computer * 6 + 5; j++)
	if (valid (hole, j) && val[j] == supv)
	  {
	    del[j] = delay (hole, score, j);
	    if (del[j] > supr)
	      {
		supr = del[j];
		move = j;
	      }
	  }
	  
      t2 = clock ();
      duree = (t2 - t1) * 1.0 / CLOCKS_PER_SEC;
      if (duree > tmax[level])
	depth--;
      if (duree < tmax[level] / 6)
	depth++;
	
    }

  if (!flagx)
    printf ("My move is: %d (depth %d in %3.1f s)\n", move + 1, depth, duree);

  return move;

}


/*cccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! Output board and score.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccc*/

void
print_board (int hole[], int score[], int move, char *label)
{
  void outputline(int *, int *, int , char *);
  int delay (int *, int *, int);
  int value (int *, int *, int, int);
  int valid (int *, int);
  int i;
  int del[12] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
  int val[12] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

  if (flagx)
      outputline(hole, score, move, label);
  else if (strcmp(label,"ERR"))
    {
      if (flagi)
	{
	  for (i = 0; i <= 11; i++)
	    if (valid (hole, i))
	      {
		del[i] = delay (hole, score, i);
		val[i] = value (hole, score, i, 0);
	      }
	  for (i = 11; i >= 6; i--)
	    printf ("%2d ", val[i]);
	  printf ("value\n");
	  for (i = 11; i >= 6; i--)
	    printf ("%2d ", del[i]);
	  printf ("delay\n");
	}

      for (i = 11; i >= 6; i--)
	printf ("%2d ", hole[i]);
      printf ("score %2d\n", score[1]);

      for (i = 0; i <= 5; i++)
	printf ("%2d ", hole[i]);
      printf ("score %2d\n", score[0]);

      if (flagi)
	{
	  for (i = 0; i <= 5; i++)
	    printf ("%2d ", del[i]);
	  printf ("delay\n");
	  for (i = 0; i <= 5; i++)
	    printf ("%2d ", val[i]);
	  printf ("value\n");
	}

      if (!strcmp (label, "END"))
	{
	  if (score[human] > score[computer])
	    printf ("You win\n");
	  else if (score[computer] > score[human])
	    printf ("You lose\n");
	  else
	    printf ("It's a draw\n");
	}
    }
}


/*cccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! Generate output line for -x option.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccc*/

void
outputline (int hole[], int score[], int move, char *label)
{
  printf ("board=%d-%d-%d-%d-%d-%d-%d-%d-%d-%d-%d-%d score=%d-%d move=%d %s\n",\
    hole[0],hole[1],hole[2],hole[3],hole[4],hole[5],hole[6],hole[7],hole[8],hole[9],hole[10],hole[11],\
    score[0],score[1],move+1,label);
}


/*cccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! Distribution of seeds.
! Return 1 if the game is running, 0 if the game is over.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccc*/

int
distribute (int hole[], int score[], int i, char *label, int verbose)
{
  int end_game (int *, int *, int);
  void outputline(int *, int *, int , char *);
  int player, other;
  int n, i0, i1, j, j0, j1, na, ne;

  player = who_plays (i);
  other = who_waits (i);

/* distribution of seeds */

  n = hole[i];
  hole[i] = 0;
  j = i;
  while (n > 0)
    {
      j = (j+1) % 12;
      if (j != i)
        {
         hole[j]++;
	 n--;
         if (verbose&&flagx) outputline(hole, score, i, "COM");
        }
    }
  j1=j;                 // j1 is the last hole reached
  
/* attacked holes */

  na = 0;
  while (j >= 0 && j / 6 == other && (hole[j] == 2 || hole[j] == 3))
    {
      j--;
      na++; 
    }                   // na is the number of attacked holes

/* score calculation */

  if (flagm == 0 || na <= neh)
    {

     j0=j1-min(na,neh); // eaten holes run from j0+1 to j1

     if (flaga) 
       {
        for (ne=0, i0=other*6, i1=i0+6, j=i0; j<i1; j++)
           if (hole[j]!=0) ne++; // ne is the number of non-empty holes    
       }

     if (!flaga || min(na,neh)!=ne)
       {
        for (j=j1; j>j0; j--)
          {
           score[player] = score[player] + hole[j];
           hole[j] = 0;
          }             // score is updated
       }
    }

/* update label */

  if (end_game (hole, score, i))
    {
      strcpy (label, "END");
      return 0;
    }
  else
    {
      strcpy (label, "RUN");
      return 1;
    }

}

/*cccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! Return 1 if the game is ended after moving the hole i
! and 0 if the game is still running.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccc*/

int
end_game (int hole[], int score[], int i)
{
  int player, other;
  int j, i0, i1, n;

  player = who_plays (i);
  other = who_waits (i);

/* the opponent has no seed */

  i0 = player * 6;
  i1 = other * 6;

  if (hole[i1] == 0 && hole[i1 + 1] == 0 && hole[i1 + 2] == 0
      && hole[i1 + 3] == 0 && hole[i1 + 4] == 0 && hole[i1 + 5] == 0)
    {
      for (j = i0; j <= i0 + 5; j++)
	score[player] += hole[j];
      return 1;
    }

/* the player has no seed and the opponent doesn't provide seed */

  n = (2 - player) * 6;

  if (hole[i0] == 0 && hole[i0 + 1] == 0 && hole[i0 + 2] == 0
      && hole[i0 + 3] == 0 && hole[i0 + 4] == 0 && hole[i0 + 5] == 0
      && (i1 + hole[i1]) < n && (i1 + 1 + hole[i1 + 1]) < n
      && (i1 + 2 + hole[i1 + 2]) < n && (i1 + 3 + hole[i1 + 3]) < n
      && (i1 + 4 + hole[i1 + 4]) < n && (i1 + 5 + hole[i1 + 5]) < n)
    {
      for (j = i1; j <= i1 + 5; j++)
	score[other] += hole[j];
      return 1;
    }

/* cyclic end game */

  if (hole[0] == 0 && hole[1] == 0 && hole[2] == 0 && hole[3] == 0
      && hole[4] == 0 && hole[6] == 0 && hole[7] == 0 && hole[8] == 0
      && hole[9] == 0 && hole[10] == 0 && hole[5] == hole[11] && hole[5] < 6)
    {
      score[player] += hole[5];
      score[other] += hole[5];
      return 1;
    }

  return 0;

}


/*cccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! Return 1 if the move of hole i is valid and 0 otherwise.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccc*/

int
valid (int hole[], int i)
{
  int all_is_attacked (int *, int);
  int i0, i1;

/* no seed in hole i */

  if (hole[i] == 0)
    return 0;

/* the opponent has no seed and the player doesn't provide seed */

  i0 = who_plays (i) * 6;
  i1 = who_waits (i) * 6;

  if (hole[i1] == 0 && hole[i1 + 1] == 0 && hole[i1 + 2] == 0
      && hole[i1 + 3] == 0 && hole[i1 + 4] == 0 && hole[i1 + 5] == 0
      && (i + hole[i]) <= i0 + 5)
    return 0;

/* the player attacks all opponent seeds although it is not necessary */

  if (flagk && all_is_attacked(hole,i   ) && 
     (!all_is_attacked(hole,i0  ) || !all_is_attacked(hole,i0+1) || 
      !all_is_attacked(hole,i0+2) || !all_is_attacked(hole,i0+3) || 
      !all_is_attacked(hole,i0+4) || !all_is_attacked(hole,i0+5)))
    {
     return 0;
    }

/* the move is valid */

  return 1;

}

/*cccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! Return 1 if all opponent seeds are attacked by move i.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccc*/

int
all_is_attacked (int hole2[], int i)
{
  int hole[12];
  int player, other;
  int n, i0, i1, j, na, ne;

  if (hole2[i] == 0) return 1; // return 1 if the hole i is empty
  
  for (j = 0; j <= 11; j++)
    hole[j] = hole2[j];
    
//  player = who_plays (i);
  other = who_waits (i);
  
/* Alternative algorithm for distribution
  n = i+hole[i]+hole[i]/12;
  for (j = i+1; j <= n; j++)
     hole[j%12]++;
  hole[i] = 0;
  j1=n%12;             // j1 is the last seed reached
  j=j1; */
  
  n = hole[i];         // distribution of seeds
  hole[i] = 0;
  j = i;
  while (n > 0)
   {
    j = (j+1) % 12;
    if (j != i)
      {
       hole[j]++;
       n--;
      }
    }                 // j is the last hole reached

  na = 0;
  while (j >= 0 && j / 6 == other && (hole[j] == 2 || hole[j] == 3))
    {
     j--;
     na++;            // na is the number of attacked holes 
    }
    
  if (na >= 0 && na <= neh)
    {
     for (ne=0, i0=other*6, i1=i0+6, j=i0; j<i1; j++)
        if (hole[j]!=0) ne++;
     if (na == ne)
        return 1;     // ne is the number of non-empty holes
    }
    
  return 0;
    
}

/*cccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! Return the best val=score[player]-score[other] by the
! min-max algorithm with depth depth-p and beginning by the
! move i.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccc*/

int
value (int hole2[], int score2[], int i, int p)
{
  int valid (int *, int);
  int distribute (int *, int *, int, char *, int);

  int hole[12];
  int score[2];
  char label[4];
  int player, other, i1;
  int val, j, q;

  q = p + 1;
  for (j = 0; j <= 11; j++)
    hole[j] = hole2[j];
  for (j = 0; j <= 1; j++)
    score[j] = score2[j];

  player = who_plays (i);
  other = who_waits (i);
  i1 = other * 6;

  if (distribute (hole, score, i, label, 0) && q < depth)
    {
      val = -100;
      for (j = i1; j <= i1 + 5; j++)
	if (valid (hole, j))
	  val = max (val, value (hole, score, j, q));
      val = -val;
    }
  else
    val = score[player] - score[other];

  return val;

}

/*cccccccccccccccccccccccccccccccccccccccccccccccccccccccc
! Return the largest delay before providing seeds
! beginning with move i.
!ccccccccccccccccccccccccccccccccccccccccccccccccccccccc*/

int
delay (int hole2[], int score2[], int i)
{
  int distribute (int *, int *, int, char *, int);
  int hole[12];
  int score[2];
  char label[4] = "RUN";
  int del, player, i0, tmp, j;

  for (j = 0; j <= 11; j++)
    hole[j] = hole2[j];
  for (j = 0; j <= 1; j++)
    score[j] = score2[j];

  player = who_plays (i);
  i0 = player * 6;

  if (i + hole[i] > i0 + 5)
    del = 0;
  else
    {
      distribute (hole, score, i, label, 0);
      del = 0;
      for (j = i0; j <= i0 + 5; j++)
	if (hole[j] != 0)
	  del = max (del, delay (hole, score, j));
      del++;
    }

  return del;

}

