'use strict';

var _ = require('lodash');
var path = require('path');
var async = require('async');
var webpackDevMiddleware = require('webpack-dev-middleware');
var webpack = require('webpack');
var SingleEntryDependency = require('webpack/lib/dependencies/SingleEntryDependency');

var blocked = [];
var isBlocked = false;

function Plugin(
/* config.webpack */webpackOptions,
/* config.webpackServer */webpackServerOptions,
/* config.webpackMiddleware */webpackMiddlewareOptions,
/* config.basePath */basePath,
/* config.files */files,
/* config.frameworks */frameworks, customFileHandlers, emitter) {
  webpackOptions = _.clone(webpackOptions) || {};
  webpackMiddlewareOptions = _.clone(webpackMiddlewareOptions || webpackServerOptions) || {};

  var applyOptions = Array.isArray(webpackOptions) ? webpackOptions : [webpackOptions];
  var includeIndex = applyOptions.length > 1;

  applyOptions.forEach(function (webpackOptions, index) {
    // The webpack tier owns the watch behavior so we want to force it in the config
    webpackOptions.watch = true;

    // Webpack 2.1.0-beta.7+ will throw in error if both entry and plugins are not specified in options
    // https://github.com/webpack/webpack/commit/b3bc5427969e15fd3663d9a1c57dbd1eb2c94805
    if (!webpackOptions.entry) {
      webpackOptions.entry = function () {
        return {};
      };
    };

    if (!webpackOptions.output) {
      webpackOptions.output = {};
    };

    // When using an array, even of length 1, we want to include the index value for the build.
    // This is due to the way that the dev server exposes commonPath for build output.
    var indexPath = includeIndex ? index + '/' : '';
    var publicPath = indexPath !== '' ? indexPath + '/' : '';

    // Must have the common _karma_webpack_ prefix on path here to avoid
    // https://github.com/webpack/webpack/issues/645
    webpackOptions.output.path = '/_karma_webpack_/' + indexPath;
    webpackOptions.output.publicPath = '/_karma_webpack_/' + publicPath;
    webpackOptions.output.filename = '[name]';
    if (includeIndex) {
      webpackOptions.output.jsonpFunction = 'webpackJsonp' + index;
    }
    webpackOptions.output.chunkFilename = '[id].bundle.js';
  });

  this.emitter = emitter;
  this.wrapMocha = frameworks.indexOf('mocha') >= 0 && includeIndex;
  this.optionsCount = applyOptions.length;
  this.files = [];
  this.basePath = basePath;
  this.waiting = [];

  var compiler;
  try {
    compiler = webpack(webpackOptions);
  } catch (e) {
    console.error(e.stack || e);
    if (e.details) {
      console.error(e.details);
    }
    throw e;
  }

  var applyPlugins = compiler.compilers || [compiler];

  applyPlugins.forEach(function (compiler) {
    compiler.plugin('this-compilation', function (compilation, params) {
      compilation.dependencyFactories.set(SingleEntryDependency, params.normalModuleFactory);
    });
    compiler.plugin('make', this.make.bind(this));
  }, this);

  ['invalid', 'watch-run', 'run'].forEach(function (name) {
    compiler.plugin(name, function (_, callback) {
      isBlocked = true;

      if (typeof callback === 'function') {
        callback();
      }
    });
  });

  compiler.plugin('done', function (stats) {
    var applyStats = Array.isArray(stats.stats) ? stats.stats : [stats];
    var assets = [];
    var noAssets = false;

    applyStats.forEach(function (stats) {
      stats = stats.toJson();

      assets.push.apply(assets, stats.assets);
      if (stats.assets.length === 0) {
        noAssets = true;
      }
    });

    if (!this.waiting || this.waiting.length === 0) {
      this.notifyKarmaAboutChanges();
    }

    if (this.waiting && !noAssets) {
      var w = this.waiting;

      this.waiting = null;
      w.forEach(function (cb) {
        cb();
      });
    }

    isBlocked = false;
    for (var i = 0; i < blocked.length; i++) {
      blocked[i]();
    }
    blocked = [];
  }.bind(this));
  compiler.plugin('invalid', function () {
    if (!this.waiting) {
      this.waiting = [];
    }
  }.bind(this));

  webpackMiddlewareOptions.publicPath = '/_karma_webpack_/';
  var middleware = this.middleware = new webpackDevMiddleware(compiler, webpackMiddlewareOptions);

  customFileHandlers.push({
    urlRegex: /^\/_karma_webpack_\/.*/,
    handler: function handler(req, res) {
      middleware(req, res, function () {
        res.statusCode = 404;
        res.end('Not found');
      });
    }
  });

  emitter.on('exit', function (done) {
    middleware.close();
    done();
  });
}

Plugin.prototype.notifyKarmaAboutChanges = function () {
  // Force a rebuild
  this.emitter.refreshFiles();
};

Plugin.prototype.addFile = function (entry) {
  if (this.files.indexOf(entry) >= 0) {
    return;
  }
  this.files.push(entry);

  return true;
};

Plugin.prototype.make = function (compilation, callback) {
  async.forEach(this.files.slice(), function (file, callback) {
    var entry = file;

    if (this.wrapMocha) {
      entry = require.resolve('./mocha-env-loader') + '!' + entry;
    }

    var dep = new SingleEntryDependency(entry);

    compilation.addEntry('', dep, path.relative(this.basePath, file).replace(/\\/g, '/'), function (err) {
      // If the module fails because of an File not found error, remove the test file
      if (dep.module && dep.module.error && dep.module.error.error && dep.module.error.error.code === 'ENOENT') {
        this.files = this.files.filter(function (f) {
          return file !== f;
        });
        this.middleware.invalidate();
      }
      callback(err);
    }.bind(this));
  }.bind(this), callback);
};

Plugin.prototype.readFile = function (file, callback) {
  var middleware = this.middleware;
  var optionsCount = this.optionsCount;

  var doRead = function () {
    if (optionsCount > 1) {
      async.times(optionsCount, function (idx, callback) {
        middleware.fileSystem.readFile('/_karma_webpack_/' + idx + '/' + file.replace(/\\/g, '/'), callback);
      }, function (err, contents) {
        if (err) {
          return callback(err);
        };
        contents = contents.reduce(function (arr, x) {
          if (!arr) {
            return [x];
          };
          arr.push(new Buffer('\n'), x);

          return arr;
        }, null);
        callback(null, Buffer.concat(contents));
      });
    } else {
      try {
        var fileContents = middleware.fileSystem.readFileSync('/_karma_webpack_/' + file.replace(/\\/g, '/'));

        callback(undefined, fileContents);
      } catch (e) {
        // If this is an error from `readFileSync` method, wait for the next tick.
        // Credit #69 @mewdriller
        if (e.code === 'ENOENT') {
          // eslint-disable-line quotes
          this.waiting = [process.nextTick.bind(process, this.readFile.bind(this, file, callback))];

          // throw otherwise
        } else {
          callback(e);
        }
      }
    }
  }.bind(this);

  if (!this.waiting) {
    doRead();
  } else {
    // Retry to read once a build is finished
    // do it on process.nextTick to catch changes while building
    this.waiting.push(process.nextTick.bind(process, this.readFile.bind(this, file, callback)));
  }
};

function createPreprocesor( /* config.basePath */basePath, webpackPlugin) {
  return function (content, file, done) {
    if (webpackPlugin.addFile(file.originalPath)) {
      // recompile as we have an asset that we have not seen before
      webpackPlugin.middleware.invalidate();
    }

    // read blocks until bundle is done
    webpackPlugin.readFile(path.relative(basePath, file.originalPath), function (err, content) {
      if (err) {
        throw err;
      }

      done(err, content && content.toString());
    });
  };
}

function createWebpackBlocker() {
  return function (request, response, next) {
    if (isBlocked) {
      blocked.push(next);
    } else {
      next();
    }
  };
}

module.exports = {
  webpackPlugin: ['type', Plugin],
  'preprocessor:webpack': ['factory', createPreprocesor],
  'middleware:webpackBlocker': ['factory', createWebpackBlocker]
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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