/* pdvspec.c
	$Id: pdvspec.c,v 1.4 2001/09/30 01:46:22 gwiley Exp $
	Glen Wiley <gwiley@ieee.org>

Copyright (c)1999,2000,2001 Glen Wiley

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sub-license, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

	these functions provide the interface to a pdv spec file for both the
	command line and X11 utilities, a pdv spec file is a way of capturing
	the metadata for a payload so that you can more easily recreate it.  The
	goal is to avoid really long command lines for the utilities.
*/

#if HAVE_CONFIG_H
 #include "config.h"
#endif

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRING_H
 #include <string.h>
#else
 #include <strings.h>
#endif
#include "pdv.h"

/* the longest line expected in the spec - this should be more than
   plenty */
static const int MAXBUFFER = 2048;

struct attval_st
{
	char *att;
	char **val;
	int  req;
};

/*---------------------------------------- writespec 
	overwrites any existing file
	returns 0 on success, errno on error
*/
int
writespec(struct pdvspec_st *spec)
{
	FILE *fh;
	int  retval = 0;
	int  res;

	fh = fopen(spec->fn_spec, "w+");
	if(fh)
	{
		res = fprintf(fh,
		 "HELPMSGFILE %s\n"
		 "AGRMSGFILE %s\n"
		 "OUTPUTFILE %s\n"
		 "PAYLDFILE %s\n"
		 "PDVSTUB %s\n"
		 "EXEC %s\n"
		 "PKGFILE %s\n"
		 "COMPRESS %s\n"
		 "TAR %s\n"
		 , (spec->fn_helpmsg ? spec->fn_helpmsg : "")
		 , (spec->fn_agrmsg  ? spec->fn_agrmsg : "")
		 , (spec->fn_output  ? spec->fn_output : "")
		 , (spec->fn_payld   ? spec->fn_payld : "")
		 , (spec->fn_pdvstub ? spec->fn_pdvstub : "")
		 , (spec->fn_exec    ? spec->fn_exec : "")
		 , (spec->fn_pkg     ? spec->fn_pkg : "")
		 , (spec->compress   ? spec->compress : "")
		 , (spec->tar        ? spec->tar : "")
		 );

		if(res <= 0)
			retval = errno;

		if(fclose(fh) != 0)
			retval = errno;
	}
	else
		retval = errno;

	return retval;
} /* writespec */

/*---------------------------------------- readspec
	read file specified in fn and populate spec (spec must have
	been allocated by the caller)

	badfield will be assigned a string indicating the name of the
	first field to fail, or NULL if no known field failed - DO NOT FREE
	THE STORAGE FOR THIS RETURNED STRING

	returns 0 on success, errno on error
	returns ENOMSG if any arguments invalid or missing
*/
int
readspec(const char *fn, struct pdvspec_st *spec, char **badfield)
{
	const int natts = 9;
	FILE *fh;
	char *buf;
	char *att;
	char *val;
	char *p;
	int  i;
	int  l;
	int  retval    = 0;
	struct attval_st atts[natts];

	atts[0].att = "HELPMSGFILE";
	atts[0].val = &(spec->fn_helpmsg);
	atts[0].req = 0;
	atts[1].att = "OUTPUTFILE";
	atts[1].val = &(spec->fn_output);
	atts[1].req = 1;
	atts[2].att = "PAYLDFILE";
	atts[2].val = &(spec->fn_payld);
	atts[2].req = 1;
	atts[3].att = "PDVSTUB";
	atts[3].val = &(spec->fn_pdvstub);
	atts[3].req = 0;
	atts[4].att = "EXEC";
	atts[4].val = &(spec->fn_exec);
	atts[4].req = 0;
	atts[5].att = "PKGFILE";
	atts[5].val = &(spec->fn_pkg);
	atts[5].req = 1;
	atts[6].att = "COMPRESS";
	atts[6].val = &(spec->compress);
	atts[6].req = 1;
	atts[7].att = "TAR";
	atts[7].val = &(spec->tar);
	atts[7].req = 1;
	atts[8].att = "AGRMSGFILE";
	atts[8].val = &(spec->fn_agrmsg);
	atts[8].req = 0;

	*badfield = NULL;

	spec->fn_spec = strdup(fn);
	if(spec->fn_spec == NULL)
		return errno;

	buf = (char *) malloc(MAXBUFFER);
	if(buf == NULL)
		return errno;

	for(i=0; i<natts-1; i++)
		*(atts[i].val) = NULL;

	fh = fopen(fn, "r");
	if(fh != NULL)
	{
		/* each attribute ends up on a line by itself */

		while(fgets(buf, MAXBUFFER, fh) != NULL)
		{
			/* the attribute name is the first field, assume it begins at col 0
			   the value is the next field following a variable number of
				spaces */

			att = buf;
			val = strchr(buf, ' ');
			if(val && *val != '\0')
			{
				*val = '\0';
				val++;
				val = val + strspn(val, " \t");

				/* remove the EOLN char */

				p = strchr(val, '\n');
				if(p && *p != '\0')
					*p = '\0';
			}

			/* find the attrib tag in our array and store the value */

			for(i=0; i<natts; i++)
			{
				if(strcmp(atts[i].att, att) == 0)
				{
					*(atts[i].val) = strdup(val);
					if(*(atts[i].val) == NULL)
					{
						retval = errno;
						break;
					}
				}
			}
		} /* while(fgets(buf, buflen, fh) != NULL) */

		fclose(fh);
	}
	else
		retval = errno;

	if(buf)
		free(buf);

	/* if we saw all of the required elements then we are ok */

	for(i=0; i<natts; i++)
	{
		if(atts[i].req && (*(atts[i].val) == NULL 
		 || strlen(*(atts[i].val)) == 0))
		{
			retval = ENOMSG;
			if(*badfield == NULL)
				*badfield = atts[i].att;
		}
	}
		
	return retval;
} /* readspec */

/*---------------------------------------- freespecmembers */
void
freespecmembers(struct pdvspec_st *spec)
{
	if(spec->fn_spec)
		free(spec->fn_spec);
	spec->fn_spec = NULL;

	if(spec->fn_helpmsg)
		free(spec->fn_helpmsg);
	spec->fn_helpmsg = NULL;

	if(spec->fn_agrmsg)
		free(spec->fn_agrmsg);
	spec->fn_agrmsg = NULL;

	if(spec->fn_output)
		free(spec->fn_output);
	spec->fn_output = NULL;

	if(spec->fn_payld)
		free(spec->fn_payld);
	spec->fn_payld = NULL;

	if(spec->fn_pdvstub)
		free(spec->fn_pdvstub);
	spec->fn_pdvstub = NULL;

	if(spec->fn_exec)
		free(spec->fn_exec);
	spec->fn_exec = NULL;

	if(spec->fn_pkg)
		free(spec->fn_pkg);
	spec->fn_pkg = NULL;

	if(spec->compress)
		free(spec->compress);
	spec->compress = NULL;

	if(spec->tar)
		free(spec->tar);
	spec->tar = NULL;


	return;
} /* freespecmembers */

/*---------------------------------------- cmpspecelem
	compare a single element in a spec
	return 0 if they match
*/
static int
cmpspecelem(char *elem1, char *elem2)
{
	int retval = 0;

	if(elem1 != elem2)
	{
		if(elem1 == NULL || elem2 == NULL)
			retval = 1;
		else
			retval = strcmp(elem1, elem2);
	}

	return retval;
} /* cmpspecelem */

/*---------------------------------------- cmpspecs
	compare two specs (ignore the fn_spec element)
	return 0 if the two specs are identical
*/
int
cmpspec(struct pdvspec_st *spec1, struct pdvspec_st *spec2)
{
	int retval = 0;

	retval = cmpspecelem(spec1->fn_helpmsg, spec2->fn_helpmsg);
	if(retval == 0)
		retval = cmpspecelem(spec1->fn_agrmsg, spec2->fn_agrmsg);
	if(retval == 0)
		retval = cmpspecelem(spec1->fn_output, spec2->fn_output);
	if(retval == 0)
		retval = cmpspecelem(spec1->fn_payld, spec2->fn_payld);
	if(retval == 0)
		retval = cmpspecelem(spec1->fn_pdvstub, spec2->fn_pdvstub);
	if(retval == 0)
		retval = cmpspecelem(spec1->fn_exec, spec2->fn_exec);
	if(retval == 0)
		retval = cmpspecelem(spec1->fn_pkg, spec2->fn_pkg);
	if(retval == 0)
		retval = cmpspecelem(spec1->compress, spec2->compress);
	if(retval == 0)
		retval = cmpspecelem(spec1->tar, spec2->tar);

	return retval;
} /* cmpspec */

/*---------------------------------------- spectopayload
	convert a spec structure to a payload structure
	NOTE: helpmsg, agrmsg, payloadstart and metadatastart are not populated

	return 0 on success, if non-zero return then the pld
	structure is not in a reliable state
*/
int
spectopayload(struct pdvspec_st *spec, struct payload_st *pld)
{
	int retval = 0;

	if(spec->fn_output)
	{
		pld->fn_out = strdup(spec->fn_output);
		if(pld->fn_out == NULL)
			retval = errno;
	}

	if(spec->fn_exec)
	{
		pld->cmd = strdup(spec->fn_exec);
		if(pld->cmd == NULL)
			retval = errno;
	}

	if(spec->compress && spec->compress[0] == 'y')
		pld->iscompressed = 1;
	else
		pld->iscompressed = 0;

	if(spec->tar && spec->tar[0] == 'y')
		pld->isatar = 1;
	else
		pld->isatar = 0;

	pld->iszipped = 0;

	return retval;
} /* spectopayload */

/* pdvspec.c */
