#include "config.h"
#include <unistd.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <dynhash.h>
#include <stdlib.h> /* malloc + free */
#include <errno.h>
#include <sys/stat.h>
#include "relaydb.h"
#include "uostr.h"
#include "crc32ansi.h"
#include "uoio.h"

static void *hash;

/*
 * supported lines in relaydb: 
 * relaydb:         
 *  host.domain       allows to accept messages for *@host.domain
 *  !host.domain      denies to accept messages for *@host.domain
 *  .domain           allows to accept messages for *@*.domain
 *  !.domain          denies to accept messages for *@*.domain
 *  !user@host.domain denies access to this email address for anybody
 */
int 
relaydb_init(const char *p)
{
	int fd;
	struct stat st;
	uoio_t i;
	void *hash_handle=NULL;
	fd=open(p,O_RDONLY);
	if (fd==-1 && errno!=ENOENT) return -1;
	if (fd==-1) return 0; /* don't relay */
	if (-1==fstat(fd,&st)) {int e=errno;close(fd);errno=e;return -1;}
	uoio_assign_r(&i,fd,read,0);
	while (1) {
		ssize_t len;
		char *s;
		DYN_ENTRY n,*f;
		len=uoio_getdelim_zc(&i,&s,'\n');
		if (len==-1) {int e=errno;uoio_destroy(&i);close(fd);errno=e;return -1;}
		if (len==0) break;
		if (s[len-1]=='\n') len--;
		while (len && (*s==' ' || *s=='\t')) {len--;s++;} /* skip WS */
		if (!len) continue; /* empty line */
		if (*s=='#') continue; /* allow comments */
		if (!hash_handle) hash_handle=hhcreate(st.st_size/8+10);
		if (!hash_handle) {uoio_destroy(&i);close(fd);errno=ENOMEM;return -1;}
		while (len && (s[len-1]==' ' || s[len-1]=='\t')) len--; /* get rid of trailing spaces */
		if (*s=='!') {n.u.nr=0; s++; len--; if (!len) continue;} 
		else n.u.nr=1;
		n.crc=crc32ansi(s,len);
		f=hhsearch(hash_handle,&n,DYN_ENTER);
		if (!f) {
			f=hhsearch(hash_handle,&n,DYN_FIND);
			if (!f) {hhdestroy(hash_handle);uoio_destroy(&i);close(fd);errno=ENOMEM;return -1;}
		}
	}
	hash=hash_handle;
	uoio_destroy(&i);
	close(fd);
	return 0;
}

int
relaydb_lookup_host(const char *host)
{
	DYN_ENTRY e,*f;
	size_t l;
	if (!hash) {errno=0;return 0;} /* default: _deny_ */
	/* look up host.domain */
	l=strlen(host);
	e.crc=crc32ansi(host,l);
	f=hhsearch(hash,&e,DYN_FIND);
	if (f) { 
		if (f->u.nr) return 1; 
		else return 0; 
	}
	while (1) { 
		while (*host && *host!='.') {l--;host++;}
		if (!*host) { return 0;}
		/* look for .domain */
		e.crc=crc32ansi(host,l);
		f=hhsearch(hash,&e,DYN_FIND);
		if (f) {
			if (f->u.nr) return 1; 
			else return 0; 
		}
		host++;
	}
}

int
relaydb_lookup_address(const char *address)
{
	DYN_ENTRY e,*f;
	size_t l;
	if (!hash) {errno=0;return 1;} /* default: allow */
	l=strlen(address);
	e.crc=crc32ansi(address,l);
	f=hhsearch(hash,&e,DYN_FIND);
	if (f) { 
		if (f->u.nr) return 1; 
		else return 0; 
	}
	return 1;
}
